# =============================================================================
#
# File:		cb_Help.tcl
# Project:	cb_tools
# Requires:     Tcl 7.4, Tk 4.0, [incr tcl] 1.5
#
# Started:	19.04.95
# Changed:	10.07.95
# Author:       Christian Bolik (zzhibol@rrzn-user.uni-hannover.de)
#
# Description:	Provides a class for (quite) simple help windows.
#               Needs the egrep utility to find the headings.
#
# =============================================================================

#
# =============================================================================
#
# Class:	cb_Help
# Desc:		Implements a class for rather sophisticated online help.
#               This creates a toplevel window that contains a listbox that
#               lists all the "headers" of the help file (which is simple
#               ascii), a text widget that displays the help file and an
#               entry for searching.
#               The help file is a simple ascii file that may be structured
#               by numbered or any other kind of headings. Cross references
#               are enclosed within `` and ''. If the first word inside the
#               cross reference is one of @file, @shell or @eval the reference
#               will be interpreted like this:
#
#                   ``@file <file> ?<regexp>?'' loads <file> into a new window.
#                            <file> may also be of the form "|<command>" and
#                            <filename>#<keyword>.
#                   ``@shell <command>'' executes <command> when clicked.
#                   ``@eval <script>'' evaluates the tcl-script <script> at
#                            global level.
#
# Interface:    None of the methods should be used from outside the class
#               since the functionality is provided at a higher level through
#               the following class procs:
#
#               cb_Help :: show <file> ?<regexp>? ?<indent>?
#                   If <file> has not yet been loaded this creates a new help
#                   window that displays <file>, else the respective window
#                   is made visible. <regexp> is used to define the "headings"
#                   of <file> and may be any egrep-style regular expression or
#                   one of: numbered, howto, subjects, man, c, tcl. If <regexp>
#                   is omitted the heading listbox will not be displayed.
#                   <indent> is a boolean value used to signal the headings
#                   should be indented depending on their respective level.
#                   This is only used if <regexp> is a true regular expression.
#
#               cb_Help :: bind_for_context <widgets> <refs> ?<fallback>?
#                   Binds all widgets in <widgets> so that F1 invokes the
#                   "show" proc on the value of the respective array-variable
#                   of <refs>. <refs> is a global array whose indices are the
#                   widget names for whom there is help available.
#                   <fallback> is a list like {<file> ?<regexp>?} and will
#                   be passed to cb_Help :: show when there is no item for
#                   the current widget.
#
#               cb_Help :: setfont <font>
#                   Makes <font> the standard font for help windows.
#

itcl_class cb_Help {

    constructor {config} {
	global [set this] cb_tools cb_tools_help tk_strictMotif

	#
	# Create a toplevel with this object's name
	# (later accessible as $this-top):
	#
        set class [$this info class]
        ::rename $this $this-tmp-
        ::toplevel $this -class $class
	wm withdraw $this
	catch "rename $this-top {}"
        ::rename $this $this-top
        ::rename $this-tmp- $this

	if {[winfo depth .] > 1} {set is_color 1} {set is_color 0}

	# the entry and button bar:
	frame $this.fb -bd 1 -relief raised
	pack $this.fb -fill x -side bottom

	label $this.ls -text "Search (regexp):"
	pack $this.ls -in $this.fb -side left -padx $pad

	entry $this.es -bd 2 -relief sunken -width 10 -font $font
	#if {$tbgcolor != ""} {
	 #   $this.es config -bg $tbgcolor
	#}
	pack $this.es -in $this.fb -side left -pady $pad -ipady 2 \
		-fill x -expand yes

	frame $this.fsep -width 16
	pack $this.fsep -in $this.fb -side left
	
	button $this.bClose -text "  Close  " -command "$this delete" \
		-underline 2
	button $this.bBack -text "  Back  " -command "$this _scroll back" \
		-underline 2

	set ipad [expr $tk_strictMotif ? 2 : 0]
	pack $this.bClose $this.bBack -in $this.fb -side right \
		-ipady $ipad -padx $pad -pady $pad

	if {$regexp != "" && $fmode != "context"} {
	    # the listbox:
	    frame $this.fflb -bd 1 -relief raised
	    pack $this.fflb -fill x -side top
	    
	    cb_listbox $this.flb -vscroll 1 -hscroll 0 -lborder 1 -uborder 0 \
		    -rbuttons 1 -width 10 -height 8 \
		    -bd 2 -relief sunken
	    $this.flb.lbox config -selectmode browse
	    pack $this.flb -in $this.fflb -fill both -expand yes -pady $pad

	    bind $this.flb.lbox <1> "
		%W select clear 0 end
		%W select set \[%W nearest %y\]
		$this _scroll to \
			\[lindex \[lindex \[$this headings\] \
			\[%W nearest %y\]\] 0\]; break
	    "

	    bind $this.es <Tab> "$this _scroll next; break"
	    bind $this.es <Shift-Tab> "$this _scroll previous; break"
	    bind $this.es <Control-Tab> "$this _scroll bottom; break"
	    bind $this.es <Control-Shift-Tab> "$this _scroll top; break"
	}

	# the textwidget:
	frame $this.fft -bd 1 -relief raised
	pack $this.fft -fill both -expand yes -side top
	
	cb_text $this.ft -vscroll 1 -lborder 1 -width 20 -height 5 \
		-bd 2 -relief sunken -state disabled -setgrid 0 -wrap word \
		-font $font -cursor top_left_arrow -highlightthickness 0
	if {$tbgcolor != ""} {
	    $this.ft.text config -bg $tbgcolor
	}
	if [info exists filemode($fmode,font,body)] {
	    $this.ft.text config -font $filemode($fmode,font,body)
	}
	pack $this.ft -in $this.fft -fill both -expand yes \
		-pady $pad

	# create heading tags:
	if [info exists filemode($fmode,font,body)] {
	    $this.ft.text tag configure subject \
		    -font $filemode($fmode,font,subject) \
		    -underline 1
	} else {
	    $this.ft.text tag configure subject -font $headfonts(subject) \
		    -underline 1
	}
	for {set i 1} {$i <= 6} {incr i} {
	    $this.ft.text tag configure level$i -font $headfonts(l$i) \
		-underline 1
	}

	set tag_bg [lindex [$this.ft.text config -bg] 4]
	if {$tag_bg == ""} {set tag_bg [lindex [$this.ft.text config -bg] 4]}

	if {[lindex [winfo name .] 0] == "xtv"} {
	    global ttv xtv_help
	    set xtv_help($this) \
		    "\{$ttv(docs)/Applications#xtv's Help System\} howto"
	    cb_Help :: bind_for_context "$this $this.es" xtv_help
	} else {
	    set cb_tools_help($this) \
		    "$cb_tools(path)/cb_Help.cbh#Window context"
	    cb_Help :: bind_for_context "$this $this.es" cb_tools_help
	}

	#bind $this <Enter> "focus $this.es"
	#bind $this <Leave> {focus .}
	bind $this.es <Return> "$this do_search \[%W get\]; break"
	bind $this.es <Down> "$this _scroll down line; break"
	bind $this.es <Up> "$this _scroll up line; break"
	bind $this.es <Next> "$this _scroll down page; break"
	bind $this.es <Prior> "$this _scroll up page; break"
	bind $this.es <Control-Home> "$this.ft.text yview 0; break"
	bind $this.es <Control-End> "$this _scroll end; break"	
	bind $this.es <Escape> "$this delete; break"	
	bind $this.es <Alt-c> "$this delete; break"	
	bind $this.es <Alt-b> "$this _scroll back; break"	
	bind $this.es <Alt-r> "
	    $this _scroll mark
	    $this load
	    $this _scroll back; break
	"

	# fill widgets:
	load
	if ![winfo exists $this] return

	#
	# Window manager settings
	#
	wm minsize $this 440 266
	if {$fmode != "context"} {
	    wm geometry $this 610x454
	} else {
	    wm geometry $this 556x292
	}
	wm protocol $this WM_DELETE_WINDOW "$this delete"
	wm iconbitmap $this @$cb_tools(path)/bitmaps/help.xbm
	wm deiconify $this
	update
	focus $this.es
    }

    destructor {
        ::rename $this-top {}	;# delete this name
        catch {destroy $this}		;# destroy associated window
    }

    #
    # ----- Methods and Procs -------------------------------------------------
    #

    method configure {config} {
    }

    method config {config} {
    }

    method load {} {
	global argv0
	
	_cursor busy
	set tw $this.ft.text

	set fref ""
	if {[string first "#" $file] > -1} {
	    set flist [split $file "#"]
	    set file [lindex $flist 0]
	    set fref [lindex $flist 1]
	}

	if {[string index $file 0] == "|"} {
	    # $file is a pipe
	    set err [catch "set fd \[open \{$file\}\]"]
	} else {
	    #foreach fake_ext {"" .gz .z .Z} {
	        set ext [file extension $file];#$fake_ext
		if {$ext == ".gz" || $ext == ".z"} {
		    set err [catch  "set fd \[open \"|gzip -cd $file\"\]"]
		} elseif {$ext == ".Z"} {
		    set err [catch  "set fd \[open \"|zcat $file\"\]"]
		} else {
		    set err [catch {set fd [open $file]}]
		}
		#if !$err break
	    #}
	}
	if $err {
	    catch "cb_error \"Error: Couldn't open $file for reading!\""
	    catch "$this delete"
	    _cursor lazy
	    return
	}
	set help_text [read $fd]
	catch "close $fd"
	
	$tw config -state normal
	$tw delete 1.0 end
	$tw insert end $help_text
	$tw config -state disabled

	set headings ""
	if {$regexp != ""} {
	    set lw $this.flb.lbox
	    catch {$lw delete 0 end}
	    set err [catch "set hs \[exec egrep -n \$regexp << \$help_text\]" \
		    msg]
	    if !$err {
		set hs [split $hs \n]
		foreach h $hs {
		    set colon [string first : $h]
		    set l [string range $h 0 [expr $colon - 1]]
		    set t [string trimleft \
			    [string range $h [expr $colon + 1] 1000] " "]
		    if $indent {
			set num [split [string trimright [lindex $t 0] .] .]
			set numl [llength $num]
			for {set i 1} {$i < $numl} {incr i} {
			    set t "    $t"
			}
		    }
		    catch {$lw insert end $t}
		    lappend headings [list [expr $l - 1] $t]
		}
	    } else {
		# puts "$msg ($regexp)"
	    }
	}

	# tag the headings with a different font:
	$tw config -state normal
	foreach head $headings {
	    set line [expr [lindex $head 0] + 1]
	    set idx1 ${line}.0
	    set idx2 [expr $line + 1].0
	    if $indent {
		set level [llength [split [string trimright \
			[lindex [lindex $head 1] 0] .] .]]

		# replace tabs with spaces
		set hb [$tw get $idx1 $idx2]
		if [regsub -all {	} $hb { } nhb] {
		    set hb $nhb
		}
		$tw delete $idx1 $idx2
		$tw insert $idx1 $hb

		# determine number of leading blanks:
		set h [string trimleft $hb " "]
		set a [expr [string length $hb] - [string length $h]]
		$tw tag add level$level "$idx1 + $a chars" $idx2
	    } elseif $tag_headings {
		$tw tag add subject $idx1 $idx2
	    }
	}
	$tw config -state disabled

	# get and bind references to headings (``...'')
	# and to shell commands (``@Shell ...''):
	set stidx 1.0
	set what {``[^`]*''}
	set success [regexp -indices $what [$tw get $stidx end] mrange]
	while {$success != 0} {
	    set mstart [expr [lindex $mrange 0] + 2]
	    set mend [expr [lindex $mrange 1] - 1]
	    
	    set head [$tw get "$stidx + $mstart chars" "$stidx + $mend chars"]
	    if [string match {@[Ss][Hh][Ee][Ll][Ll]*} $head] {
		set hidx 1000
	    } elseif [string match {@[Ee][Vv][Aa][Ll]*} $head] {
		set hidx 1001
	    } elseif [string match {@[Ff][Ii][Ll][Ee]*} $head] {
		set hidx 1002
	    } elseif [string match {@[Ll][Yy][Nn][Xx]*} $head] {
		set hidx 1003
	    } else {
		set hidx 0
	    }

	    if {$hidx > -1} {
		set reftag ref[incr refcount]

		set rfont [lindex [$this.ft.text config -font] 4]
		#if [regsub medium $rfont bold tmpfont] {
		#    set rfont $tmpfont
		#}
		if $is_color {
		    if {$hidx < 1000} {
			$tw tag configure $reftag \
				-foreground blue -borderwidth 0 \
				-relief flat -background $tag_bg -font $rfont
		    } else {
			$tw tag configure $reftag \
				-foreground $tagcolor($hidx) -borderwidth 0 \
				-relief flat -background $tag_bg -font $rfont
		    }
		} else {
		    $tw tag configure $reftag -underline 1 -font $rfont
		}

		if {$hidx == 0} {
		    $tw tag bind $reftag <B1-ButtonRelease> \
			    "$this _headref \{$head\}; break"
		} elseif {$hidx == 1000} {
		    # $head contains a shell command
		    set cmd [lrange $head 1 100]
		    $tw tag bind $reftag <B1-ButtonRelease> \
			    "$this _shellref \{$cmd\}; break"
		} elseif {$hidx == 1001} {
		    # $head contains a tcl command to evaluate
		    set script [lrange $head 1 100]
		    $tw tag bind $reftag <B1-ButtonRelease> \
			    "$this _evalref \{$script\}; break"
		} elseif {$hidx == 1002} {
		    # $head contains a filename and perhaps regexp/filemode
		    set fname [lindex $head 1]
		    set fmode [lindex $head 2]
		    $tw tag bind $reftag <B1-ButtonRelease> \
			    "$this _fileref \{$fname\} \{$fmode\}; break"
		} elseif {$hidx == 1003} {
		    # $head contains an URL to get via lynx
		    set url [lindex $head 1]
		    $tw tag bind $reftag <B1-ButtonRelease> \
			    "$this _lynxref \{$url\}; break"
		}

		$tw tag bind $reftag <Enter> "$this _tag_entered; break"
		$tw tag bind $reftag <Leave> "$this _tag_left; break"
		$tw tag add $reftag "$stidx + $mstart chars" \
			"$stidx + $mend chars"
	    } else {
		#puts "Couldn't find a heading containing \"$head\","
	    }
	    
	    set stidx [$tw index "$stidx + $mend chars"]
	    set success [regexp -indices $what [$tw get $stidx end] mrange]
	}
	    
	wm title $this "[file tail $argv0]: [file tail $file]"
	wm iconname $this "[file tail $file]"
	_cursor lazy

	if {$fref != ""} {
	    _headref $fref
	}
    }

    method do_search {what} {
	_cursor busy
	set tw $this.ft.text
	catch "$tw tag remove sel 1.0 end"
	set stidx "insert"
	if {$what != $last_what} {
	    $tw mark set insert 1.0
	    set last_matched 0
	}
	set last_what $what
	
	set success [regexp -nocase -indices $what [$tw get insert end] mrange]
	if !$success {
	    $tw mark set insert 1.0
	    $this.es delete 0 end
	    if $last_matched {
		$this.es insert end "No more matches."
	    } else {
		$this.es insert end "No match."
	    }
	    _cursor lazy
	    after 1000 
	    $this.es delete 0 end
	    $this.es insert end $what
	    
	} else {
	    set last_matched 1
	    set mstart [lindex $mrange 0]
	    set mend [expr [lindex $mrange 1] + 1]
	    lappend match_range [$tw index "$stidx + $mstart chars"]
	    lappend match_range [$tw index "$stidx + $mend chars"]
	    set last_ypos [lindex [cb_sbGet $this.ft.vscroll] 2]
	    $tw tag add sel "$stidx + $mstart chars" "$stidx + $mend chars"
	    $tw yview -pickplace "$stidx + $mstart chars"
	    $tw mark set insert "$stidx + $mend chars"
	    _cursor lazy
	}
    }

    method headings {} {
	return $headings
    }

    method _scroll {where {amount ""}} {
	set sbset [$this.ft.vscroll get]
	set fr1 [lindex $sbset 0]
	set fr2 [lindex $sbset 1]
	set tunits [expr [$this.ft.text index end] - 1]
	set wunits [expr ($fr2 - $fr1) * $tunits]
	set pos1 [expr round($fr1 * $tunits)]
	set pos2 [expr round($fr2 * $tunits)]
	#puts "$tunits $wunits $pos1 $pos2, $where $amount"
	set lb $this.flb.lbox
	set tw $this.ft.text
	switch $where {
	    end {
		$this.ft.text yview [expr round($tunits - $wunits)]
	    }
	    down {
		if {$pos1 + $wunits >= $tunits} return
		if {$amount == "line"} {
		    $this.ft.text yview [expr $pos1 + 1]
		} else {
		    set np [expr $pos2 - 1]
		    if {$np + $wunits >= $tunits} {
			set np [expr round($tunits - $wunits)]
		    }
		    $this.ft.text yview $np	
		}
	    }
	    up {
		if {$pos1 <= 0} return
		if {$amount == "line"} {
		    $this.ft.text yview [expr $pos1 - 1]
		} else {
		    set np [expr round($pos1 - $wunits) + 1]
		    if {$np < 0} {set np 0}
		    $this.ft.text yview $np
		}
	    }
	    back {
		set lpos $pos1
		$this.ft.text yview $last_ypos
		set last_ypos $lpos
	    }
	    mark {
		set last_ypos $pos1
	    }
	    to {
		set last_ypos $pos1
		$this.ft.text yview $amount
	    }
	    next {
		set csel [$lb curselection]
		if {$csel == [llength $headings] - 1} return
		if {$csel == ""} {set csel 0} {incr csel}
		$lb select clear 0 end
		$lb select set $csel
		if {$csel > [lindex [cb_sbGet $this.flb.vscroll] 3]} {
		    $lb yview $csel
		}
		_scroll to [lindex [lindex $headings $csel] 0]
	    }
	    previous {
		set csel [$lb curselection]
		if {$csel == 0} return
		if {$csel == ""} {
		    set csel [expr [llength $headings] - 1]
		} else {
		    incr csel -1
		}
		$lb select clear 0 end
		$lb select set $csel
		if {$csel < [lindex [cb_sbGet $this.flb.vscroll] 2]} {
		    set lbh [lindex [cb_sbGet $this.flb.vscroll] 1]
		    set nyv [expr $csel - $lbh + 1]
		    if {$nyv < 0} {set nyv 0}
		    $lb yview $nyv
		}
		_scroll to [lindex [lindex $headings $csel] 0]
	    }
	    top {
		set csel 0
		$lb select clear 0 end
		$lb select set $csel
		$lb yview 0
		_scroll to [lindex [lindex $headings $csel] 0]
	    }
	    bottom {
		set csel [expr [llength $headings] - 1]
		$lb select clear 0 end
		$lb select set $csel
		set sbg [cb_sbGet $this.flb.vscroll]
		if {$csel > [lindex $sbg 3]} {
		    $lb yview [expr [lindex $sbg 0] - [lindex $sbg 1]]
		}
		_scroll to [lindex [lindex $headings $csel] 0]
	    }
	}
    }

    method _headref {head} {
	global cb_Text
	set cb_Text(skip_binding) 1
	
	set i 0
	set hidx -1
	foreach h $headings {
	    if [string match "$head*" [string trimleft [lindex $h 1] " "]] {
		set hidx $i
		break
	    }
	    incr i
	}
	if {$hidx == -1} {
	    set i 0
	    foreach h $headings {
		if [string match "*$head*" [lindex $h 1]] {
		    set hidx $i
		    break
		}
		incr i
	    }
	}

	if {$hidx > -1} {
	    $this.ft.text tag remove sel 1.0 end
	    _scroll to  [lindex [lindex $headings $hidx] 0]
	} else {
	    $this.es delete 0 end
	    $this.es insert end $head
	    set last_what ""
	    do_search $head
	}
	
    }

    method _shellref {cmd} {
	_cursor busy
	set op [pwd]
	cd [file dirname $file]
	exec sh -c "$cmd" &
	cd $op
	_cursor lazy
    }

    method _evalref {script} {
	set op [pwd]
	cd [file dirname $file]
	uplevel #0 eval $script
	cd $op
    }

    method _fileref {fname freg} {
	_cursor busy
	set fc [string index $fname 0]
	if {$fc != "/" && $fc != "|" && $fc != "~"} {
	    set fname [file dirname $file]/$fname
	}
	
	$this.ft.text tag remove sel 1.0 end
	cb_Help :: show $fname $freg
	_cursor lazy
    }

    method _lynxref {url} {
	_cursor busy
	cb_Help :: show "|lynx -dump $url" lynx
	_cursor lazy
    }

    method _cursor {cur} {
	global cb_tools

	set gw [grab current]
	set busy_cur "@$cb_tools(path)/bitmaps/timer.xbm \
			$cb_tools(path)/bitmaps/timer.mask.xbm black white"
	
	switch $cur {
	    busy {
		if {$gw == ""} {
		    cb_BusyCursor
		    $this.ft.text config -cursor $busy_cur
		    $this.es config -cursor $busy_cur
		} else {
		    $gw config -cursor $busy_cur
		}
	    }
	    lazy {
		if {$gw == ""} {
		    cb_NormalCursor
		    $this.ft.text config -cursor top_left_arrow
		    $this.es config -cursor xterm
		} else {
		    $gw config -cursor arrow
		}
	    }
	}
	
	update idletasks
    }

    method _tag_entered {} {
	set tag_entered 1
        if {$tag_left} {
	    set tag_left 0
	    $this.ft.text config -cursor hand2
	}
    }
    
    method _tag_left {} {
	set tag_entered 0
	if {!$tag_aftered} {
	    set tag_aftered 1
	    after 50 "$this _tag_really_left"
	}
    }

    method _tag_really_left {} {
	if {!$tag_entered && !$tag_left} {
	    set tag_left 1
	    $this.ft.text config -cursor top_left_arrow
	}
	set tag_aftered 0
    }

    # ----------------------------------------------------
    # Procs

    proc show {fname {regexp ""} {indent ""}} {
	global cb_tools

	set fref ""
	set ofname $fname
	if {[string first "#" $fname] > -1} {
	    set flist [split $fname "#"]
	    set fname [lindex $flist 0]
	    set fref [lindex $flist 1]
	}
	
	foreach obj [itcl_info objects -class cb_Help] {
	    if {$fname == [$obj info public file -value]} {
		# $fname has already been loaded
		wm deiconify $obj
		raise $obj
		if {$fref != ""} {
		    $obj _headref $fref
		}
		return $obj
	    }
	}

	if {$indent != ""} {
	    set obj [cb_Help .cb_Help[incr object_id] \
		    -file $ofname -regexp $regexp -indent $indent]
	} else {
	    set obj [cb_Help .cb_Help[incr object_id] \
		    -file $ofname -regexp $regexp]
	}
	return $obj
    }

    proc context {key refs {fallback ""}} {
	global $refs cb_tools

	if $cb_tools(in_development) {
	    puts stderr $key
	}

	set tkey $key
	set ocur ""
	
	if [winfo exists $key] {
	    catch {set ocur [lindex [$key config -cursor] 4]}
	    catch {$key config -cursor watch}

	    if {[winfo class $key] == "Menu"} {
		# support for "Help on Menu Entry"
		set active ""
		catch {set active [$key index active]}
		if {$active != ""} {
		    if $cb_tools(in_development) {
			puts stderr "active: $active"
		    }
		    if [info exists [set refs](${key}:$active)] {
			set tkey ${key}:$active
		    }
		}
	    }
	}
	
	set fname ""
	set ex 0

	while {!$ex && $tkey != ""} {
	    if [info exists [set refs]($tkey)] {
		set ref [set [set refs]($tkey)]
		set fname [lindex $ref 0]
		set freg [lindex $ref 1]
		set ex 1
		break
	    } elseif [winfo exists $tkey] {
		# if there's no entry for widget $key, try its parent:
		set tkey [winfo parent $tkey]
	    } else {
		set tkey ""
	    }
	}
	
	if {$ex == 0 && $fallback != ""} {
	    set fname [lindex $fallback 0]
	    set freg [lindex $fallback 1]
	}

	if {$fname != ""} {
	    show $fname $freg
	    set rc 1
	} else {
	    set rc 0
	}

	catch {$key config -cursor $ocur}
	return $rc
    }

    proc bind_for_context {widget_list ref_array {fallback ""}} {
	foreach widget $widget_list {
	    if 0 {
		if {[winfo class $widget] != "Menu"} {
		    bind $widget <Enter> {
			# avoid conflicts with menu traversal:
			if [winfo exists [focus]] {
			    if {[winfo class [focus]] != "Menu"} {
				focus %W
			    }
			} else {
			    focus %W
			}
			break
		    }
		    bind $widget <Leave> {
			focus [winfo toplevel %W]; break
		    }
		}
	    }
	    bind $widget <Key-F1> "
		cb_Help :: context \[winfo containing %X %Y\] \
			$ref_array $fallback; break
	    "
	    bind $widget <Key-Help> "
		cb_Help :: context \[winfo containing %X %Y\] \
			$ref_array $fallback; break
	    "
	}
    }

    proc id {{cmd ""}} {
	if {$cmd == ""} {
	    set i $object_id
	    incr object_id
	    return $i
	} elseif {$cmd == "reset"} {
	    set object_id 0
	}
    }

    proc setfont {newfont} {
	#
	# Sets the common variables font_xxx.
	#
	set font $newfont
	set headfonts(subject) $font
	if [regsub medium $headfonts(subject) bold nf] {
	    set headfonts(subject) $nf
	}
	if [regsub 12 $headfonts(subject) 14 nf] {
	    set headfonts(subject) $nf
	}
	foreach obj [itcl_info objects -class cb_Help] {
	    $obj.ft.text config -font $font
	    $obj.es config -font $font
	}
    }

    proc textbg {color} {
	set tbgcolor $color
	foreach obj [itcl_info objects -class cb_Help] {
	    $obj.ft.text config -bg $tbgcolor
	    #$obj.es config -bg $tbgcolor
	}
    }

    #
    # ----- Variables ---------------------------------------------------------
    #

    # complete filename of the helpfile:
    public file "" {
	if [winfo exists $this] {
	    load
	}
    }

    # regular expression for getting the "headings":
    public regexp {} {
	if [info exists filemode($regexp,regexp)] {
	    set fmode $regexp
	    set indent $filemode($regexp,indent)
	    set tag_headings $filemode($regexp,tag)
	    set regexp $filemode($regexp,regexp)
	}
	if [winfo exists $this] {
	    load $file
	}
    }

    # indent headings in listbox?
    public indent 0

    public pad 4

    protected help_text
    protected headings
    protected last_what ""
    protected last_matched 0
    protected is_color
    protected refcount 0
    protected tag_bg
    protected last_ypos 0
    protected tag_headings 1
    protected fmode ""
    protected tag_entered 0
    protected tag_left 1
    protected tag_aftered 0
    
    common font -*-courier-medium-r-*-*-12-*
    common object_id 0

    # With "@File" references the second argument can be any of these
    # keywords: man, numbered, howto, subjects, c, tcl, context, lynx
    common filemode
    set filemode(man,regexp) {^[A-Z][-A-Z1-90 ]*$}
    set filemode(man,indent) 0
    set filemode(man,tag) 1
    set filemode(numbered,regexp) {^ *([1-9][1-90]*\.)+ }
    set filemode(numbered,indent) 1
    set filemode(numbered,tag) 1
    set filemode(howto,regexp) {^  ([1-9][1-90]*\.)+(  |	)}
    set filemode(howto,indent) 1
    set filemode(howto,tag) 1
    set filemode(subjects,regexp) {^Subject: }
    set filemode(subjects,indent) 0
    set filemode(subjects,tag) 1
    set filemode(c,regexp) {^[a-zA-Z].*[,)]$|^typedef}
    set filemode(c,indent) 0
    set filemode(c,tag) 0
    set filemode(tcl,regexp) {^proc|^itcl_class|^ *method}
    set filemode(tcl,indent) 0
    set filemode(tcl,tag) 0
    set filemode(context,regexp) {^ ?[A-Za-z]}
    set filemode(context,indent) 0
    set filemode(context,tag) 1
    set filemode(context,font,body) -*-times-medium-r-*-*-14-*
    set filemode(context,font,subject) -*-times-bold-r-*-*-14-*
    set filemode(lynx,regexp) {^ *[A-Z1-90][A-Z1-90 -:]+$|^[^ 	]}
    set filemode(lynx,indent) 0
    set filemode(lynx,tag) 0
    set filemode(khoros,regexp) {^[A-Z]\.}
    set filemode(khoros,indent) 1
    set filemode(khoros,tag) 1

    # different colors for different types of tags (depends on hidx
    # in method load above), in-file references are colored blue
    common tagcolor
    set tagcolor(1000) red
    set tagcolor(1001) purple
    set tagcolor(1002) forestgreen
    set tagcolor(1003) darkorchid

    common headfonts
    set headfonts(subject) -*-courier-bold-r-*-*-14-*
    set headfonts(l1) -*-times-bold-r-*-*-24-*
    set headfonts(l2) -*-times-bold-r-*-*-18-*
    set headfonts(l3) -*-times-bold-r-*-*-14-*
    set headfonts(l4) -*-times-bold-r-*-*-14-*
    set headfonts(l5) -*-times-bold-r-*-*-12-*
    set headfonts(l6) -*-times-bold-r-*-*-12-*

    common tbgcolor ""
}

