/*
 * Copyright (c) 1993 Ulrich Pegelow <pegelow@moorea.uni-muenster.de>
 * Copyright (c) 1993 Branko Lankester <branko@hacktic.nl>
 * Copyright (c) 1993, 1994 Rick Sladkey <jrs@world.std.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Paul Kranenburg,
 *      Branko Lankester and Rick Sladkey.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *	ipc.c,v 2.19 1994/03/12 23:11:45 jrs Exp
 */

#include "defs.h"

#ifdef LINUX

#define __KERNEL__
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/msg.h>
#include <sys/shm.h>
#undef __KERNEL__

static struct xlat msgctl_flags[] = {
	{ IPC_RMID,	"IPC_RMID"	},
	{ IPC_SET,	"IPC_SET"	},
	{ IPC_STAT,	"IPC_STAT"	},
	{ IPC_INFO,	"IPC_INFO"	},
	{ MSG_STAT,	"MSG_STAT"	},
	{ MSG_INFO,	"MSG_INFO"	},
	{ 0,		NULL		},
};

static struct xlat semctl_flags[] = {
	{ IPC_RMID,	"IPC_RMID"	},
	{ IPC_SET,	"IPC_SET"	},
	{ IPC_STAT,	"IPC_STAT"	},
	{ IPC_INFO,	"IPC_INFO"	},
	{ SEM_STAT,	"SEM_STAT"	},
	{ SEM_INFO,	"SEM_INFO"	},
	{ GETPID,	"GETPID"	},
	{ GETVAL,	"GETVAL"	},
	{ GETALL,	"GETALL"	},
	{ GETNCNT,	"GETNCNT"	},
	{ GETZCNT,	"GETZCNT"	},
	{ SETVAL,	"SETVAL"	},
	{ SETALL,	"SETALL"	},
	{ 0,		NULL		},
};

static struct xlat shmctl_flags[] = {
	{ IPC_RMID,	"IPC_RMID"	},
	{ IPC_SET,	"IPC_SET"	},
	{ IPC_STAT,	"IPC_STAT"	},
	{ IPC_INFO,	"IPC_INFO"	},
	{ SHM_STAT,	"SHM_STAT"	},
	{ SHM_INFO,	"SHM_INFO"	},
	{ SHM_LOCK,	"SHM_LOCK"	},
	{ SHM_UNLOCK,	"SHM_UNLOCK"	},
	{ 0,		NULL		},
};

static struct xlat resource_flags[] = {
	{ IPC_CREAT,	"IPC_CREAT"	},
	{ IPC_EXCL,	"IPC_EXCL"	},
	{ IPC_NOWAIT,	"IPC_NOWAIT"	},
	{ 0,		NULL		},
};

static struct xlat shm_flags[] = {
	{ SHM_REMAP,	"SHM_REMAP"	},
	{ SHM_RDONLY,	"SHM_RDONLY"	},
	{ SHM_RND,	"SHM_RND"	},
	{ 0,		NULL		},
};

static struct xlat msg_flags[] = {
	{ MSG_NOERROR,	"MSG_NOERROR"	},
	{ MSG_EXCEPT,	"MSG_EXCEPT"	},
	{ IPC_NOWAIT,	"IPC_NOWAIT"	},
	{ 0,		NULL		},
};

int sys_msgget(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_arg[0])
			tprintf("%u", tcp->u_arg[0]);
		else
			tprintf("IPC_PRIVATE");
		tprintf(", ");
		if (printflags(resource_flags, tcp->u_arg[1]) != 0)
			tprintf("|");
		tprintf("%#o", tcp->u_arg[1] & 0666);
	}
	return 0;
}

int sys_msgctl(tcp)
struct tcb *tcp;
{
	char *cmd = xlookup(msgctl_flags, tcp->u_arg[1]);

	if (entering(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		tprintf(", %s", cmd == NULL ? "MSG_???" : cmd);
		tprintf(", %#x", tcp->u_arg[3]);
	}
	return 0;
}

int sys_msgsnd(tcp)
struct tcb *tcp;
{
	long mtype;

	if (entering(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		umove(tcp, tcp->u_arg[3], &mtype);
		tprintf(", {%lu, ", mtype);
		printstr(tcp, tcp->u_arg[3] + sizeof(long),
			tcp->u_arg[1]);
		tprintf("}, %u", tcp->u_arg[1]);
		tprintf(", ");
		if (printflags(msg_flags, tcp->u_arg[2]) == 0)
			tprintf("0");
	}
	return 0;
}

int sys_msgrcv(tcp)
struct tcb *tcp;
{
	long mtype;
	struct ipc_kludge tmp;

	if (exiting(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		umove(tcp, tcp->u_arg[3], &tmp);
		umove(tcp, (int) tmp.msgp, &mtype);
		tprintf(", {%lu, ", mtype);
		printstr(tcp, (int) (tmp.msgp) + sizeof(long),
			tcp->u_arg[1]);
		tprintf("}, %u", tcp->u_arg[1]);
		tprintf(", %ld", tmp.msgtyp);
		tprintf(", ");
		if (printflags(msg_flags, tcp->u_arg[2]) == 0)
			tprintf("0");
	}
	return 0;
}

int sys_semop(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		tprintf(", %#x", tcp->u_arg[3]);
		tprintf(", %u", tcp->u_arg[1]);
	}
	return 0;
}

int sys_semget(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_arg[0])
			tprintf("%u", tcp->u_arg[0]);
		else
			tprintf("IPC_PRIVATE");
		tprintf(", %u", tcp->u_arg[1]);
		tprintf(", ");
		if (printflags(resource_flags, tcp->u_arg[2]) != 0)
			tprintf("|");
		tprintf("%#o", tcp->u_arg[2] & 0666);
	}
	return 0;
}

int sys_semctl(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		tprintf(", %u, ", tcp->u_arg[1]);
		printxval(semctl_flags, tcp->u_arg[2], "SEM_???");
		tprintf(", %#x", tcp->u_arg[3]);
	}
	return 0;
}

int sys_shmget(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_arg[0])
			tprintf("%u", tcp->u_arg[0]);
		else
			tprintf("IPC_PRIVATE");
		tprintf(", %u", tcp->u_arg[1]);
		tprintf(", ");
		if (printflags(resource_flags, tcp->u_arg[2]) != 0)
			tprintf("|");
		tprintf("%#o", tcp->u_arg[2] & 0666);
	}
	return 0;
}

int sys_shmctl(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		tprintf("%u, ", tcp->u_arg[0]);
		printxval(shmctl_flags, tcp->u_arg[1], "SHM_???");
		tprintf(", %#x", tcp->u_arg[3]);
	}
	return 0;
}

int sys_shmat(tcp)
struct tcb *tcp;
{
	unsigned long raddr;

	if (exiting(tcp)) {
		tprintf("%u", tcp->u_arg[0]);
		tprintf(", %#x", tcp->u_arg[3]);
		tprintf(", ");
		if (printflags(shm_flags, tcp->u_arg[1]) == 0)
			tprintf("0");
		if (syserror(tcp))
			return 0;
		if (umove(tcp, tcp->u_arg[2], &raddr) < 0)
			return RVAL_NONE;
		tcp->u_rval = raddr;
		return RVAL_HEX;
	}
	return 0;
}

int sys_shmdt(tcp)
struct tcb *tcp;
{
	if (entering(tcp))
		tprintf("%#x", tcp->u_arg[3]);
	return 0;
}

#endif /* LINUX */
