/* This module, and the entire GoodStuff program, and the concept for
 * interfacing this module to the Window Manager, are all original work
 * by Robert Nation
 *
 * Copyright 1993, Robert Nation. No guarantees or warantees or anything
 * are provided or implied in any way whatsoever. Use this program at your
 * own risk. Permission to use this program for any purpose is given,
 * as long as the copyright is kept intact. */
/***********************************************************************
 *
 * Derived from fvwm icon code
 *
 ***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>

#ifdef NeXT
#include <fcntl.h>
#endif

#include "filemgr.h"
#include "debug.h"

#include <X11/extensions/shape.h>

extern char *iconPath, *pixmapPath;
extern Display *dpy;
extern Window Root;
extern int d_depth;
extern GC NormalGC;
extern Pixel hilite_pix, back_pix, shadow_pix, fore_pix;
extern GC  NormalGC,ShadowGC,ReliefGC, rvGC;

XFontStruct *font;


/****************************************************************************
 *
 * Loads an icon file into a pixmap
 *
 ****************************************************************************/
void LoadIcon(struct mapping *map)
{
  /* First, check for a monochrome bitmap */
  if(map->pixmap_file != NULL)
    GetBitmapFile(map);

  /* Next, check for a color pixmap */
  if((map->pixmap_file != NULL)&&
     (map->icon_w == 0)&&(map->icon_h == 0))
    GetXPMFile(map);
}

/****************************************************************************
 *
 * Creates an Icon Window
 *
 ****************************************************************************/
void CreateIconWindow(Window main_win, struct filelist *list)
{
  unsigned long valuemask;		/* mask for create windows */
  XSetWindowAttributes attributes;	/* attributes for create windows */
  int w,h;

  attributes.background_pixel = back_pix;
  attributes.event_mask = ExposureMask;
  valuemask =  CWEventMask | CWBackPixel;

  list->LabelW = XTextWidth(font,list->filename,strlen(list->filename))+
    2*HORIZ_PAD;

  list->LabelWin = 
    XCreateWindow(dpy, main_win, 0, 0, list->LabelW,
		  font->ascent+font->descent+4, 0, CopyFromParent,
		  CopyFromParent,CopyFromParent,valuemask,&attributes);

  if((list->icon_w == 0)||(list->icon_h == 0))
    {
      w = 10;
      h = 10;
    }
  else
    {
      w = list->icon_w;
      h = list->icon_h;
    }

  attributes.event_mask = 0;
  valuemask =  CWEventMask | CWBackPixel;

  list->HoldingWin = 
    XCreateWindow(dpy, main_win, 0,0, w,h, 0, CopyFromParent,
		  CopyFromParent,CopyFromParent,valuemask,&attributes);

  list->IconWin = 
    XCreateWindow(dpy, list->HoldingWin,0,0,w,h, 0, 
		  CopyFromParent, CopyFromParent,CopyFromParent,
		  valuemask,&attributes);

  if (list->icon_mask != None)
    {
      XShapeCombineMask(dpy,list->IconWin, ShapeBounding, 0, 0,
			list->icon_mask, ShapeSet);
    }

  if(list->icon != None)
    XSetWindowBackgroundPixmap(dpy, list->IconWin,list->icon);

  XMapWindow(dpy,list->IconWin);
  XMapWindow(dpy,list->HoldingWin);
  XMapWindow(dpy,list->LabelWin);
  list->icon_x = 0;
  list->icon_y = 0;
  return;
}


/****************************************************************************
 *
 * Creates an Icon Window
 *
 ****************************************************************************/
void ResizeIconWin(struct filelist *list, char *newname,int newwidth)
{
  if(list->CheckedOut)
    return;

  newwidth += 2*HORIZ_PAD;
  if(newwidth != list->LabelW)
    list->LabelW = newwidth;
  else
    return;

  if(list->LabelW == 0)
    list->LabelW = 1;
  
  list->label_x = list->actual_x + (list->icon_w - list->LabelW)/2;
  XMoveResizeWindow(dpy,list->LabelWin,
		    list->label_x,list->label_y,
		    list->LabelW,font->ascent+font->descent+4);
}

/****************************************************************************
 *
 * Combines icon shape masks after a resize
 *
 ****************************************************************************/
void ConfigureIconWindow(struct filelist *list,int min_width, int min_height)
{
  if(list->CheckedOut)
    return;
  
  list->actual_x = list->icon_x + (min_width - list->icon_w)/2;
  list->actual_y = list->icon_y + 
    (min_height - list->icon_h)/2;
  XMoveWindow(dpy, list->HoldingWin, list->actual_x,list->actual_y);

  list->label_x = list->icon_x + (min_width - list->LabelW)/2;
  list->label_y = list->icon_y + min_height;
  XMoveWindow(dpy, list->LabelWin, list->label_x, list->label_y);
}


/***************************************************************************
 *
 * Find the specified icon file somewhere along the given path.
 *
 * There is a possible race condition here:  We check the file and later
 * do something with it.  By then, the file might not be accessible.
 * Oh well.
 *
 **************************************************************************/
char *findIconFile(char *icon, char *pathlist)
{
#ifndef NO_ICONS
  char *path;
  char *dir_end;
  int l1,l2;

  if(icon != NULL)
    l1 = strlen(icon);
  else 
    l1 = 0;

  if(pathlist != NULL)
    l2 = strlen(pathlist);
  else
    l2 = 0;

  path = safemalloc(l1 + l2 + 10,"FindIconFile");
  *path = '\0';
  if (*icon == '/') 
    {
      /* No search if icon begins with a slash */
      strcpy(path, icon);
      return path;
    }
     
  if ((pathlist == NULL) || (*pathlist == '\0')) 
    {
      /* No search if pathlist is empty */
      strcpy(path, icon);
      return path;
    }
 
  /* Search each element of the pathlist for the icon file */
  while ((pathlist)&&(*pathlist))
    { 
      dir_end = strchr(pathlist, ':');
      if (dir_end != NULL)
	{
	  strncpy(path, pathlist, dir_end - pathlist);
	  path[dir_end - pathlist] = 0;
	}
      else 
	strcpy(path, pathlist);
      strcat(path, "/");
      strcat(path, icon);
      if (access(path, R_OK) == 0) 
	return path;
      strcat(path, ".gz");
      if (access(path, R_OK) == 0) 
	return path;
 
      /* Point to next element of the path */
      if(dir_end == NULL)
	pathlist = NULL;
      else
	pathlist = dir_end + 1;
    }
  /* Hmm, couldn't find the file.  Return NULL */
  safefree(path,"path","FindIconFile");
#endif
  return NULL;
}
 

/***************************************************************************
 *
 * Looks for a monochrome icon bitmap file
 *
 **************************************************************************/
void GetBitmapFile(struct mapping *map)
{
#ifndef NO_ICONS
  char *path = NULL;
  int HotX,HotY;
  Pixmap temp;

  path = findIconFile(map->pixmap_file, iconPath);
  if(path == NULL)return;

  if(XReadBitmapFile (dpy, Root,path,(unsigned int *)&map->icon_w, 
		      (unsigned int *)&map->icon_h,
		      &map->icon, (int *)&HotX, (int *)&HotY) != BitmapSuccess)
    {
      map->icon_depth = -1;
      map->icon_w = 0;
      map->icon_h = 0;
      map->icon = None;
    }
  else
    {
      temp = map->icon;
      map->icon = XCreatePixmap(dpy,Root, map->icon_w, map->icon_h,d_depth);
      map->icon_depth = d_depth;
      XCopyPlane(dpy,temp,map->icon,NormalGC,
		 0,0,map->icon_w,map->icon_h, 0,0,1);
    }
  map->icon_mask = None;
  safefree(path,"path","GetBitmapFile");
  return;
#endif
}


/****************************************************************************
 *
 * Looks for a color XPM icon file
 *
 ****************************************************************************/
void GetXPMFile(struct mapping *map)
{
  XpmAttributes xpm_attributes;
  char *path = NULL;
  static int got_attr = 0;
  static XWindowAttributes root_attr;

  if(!got_attr)
    {
      XGetWindowAttributes(dpy,Root,&root_attr);
      got_attr = 1;
    }

  path = findIconFile(map->pixmap_file, pixmapPath);
  if(path == NULL)return; 
  xpm_attributes.colormap = root_attr.colormap;
  xpm_attributes.valuemask = XpmSize | XpmReturnPixels|XpmColormap;
  if(XpmReadFileToPixmap(dpy, Root, path,
			 &map->icon,
			 &map->icon_mask, 
			 &xpm_attributes) == XpmSuccess) 
    { 
      map->icon_w = xpm_attributes.width;
      map->icon_h = xpm_attributes.height;
      map->icon_depth = d_depth;
    } 
  else
    {
      map->icon_depth = -1;
      map->icon_w = 0;
      map->icon_h = 0;
      map->icon = None;
    }
  safefree(path,"path","GetXPMFile");
}

/***************************************************************************
 *
 * Redraw the icon which represents a single file 
 *
 **************************************************************************/
void RedrawIcon(struct filelist *list,char *temp_name)
{
  int w;
  char *name;

  if((!list)||(list->CheckedOut))
    return;

  if(temp_name != NULL)
    name = temp_name;
  else
    name = list->filename;
  
  w=XTextWidth(font,name,strlen(name));

  ResizeIconWin(list,name,w);

  if(list->selected)
    {
      if(!list->drawn_selected)
	{
	  XSetWindowBackground(dpy,list->HoldingWin,
			       fore_pix);
	  XSetWindowBackground(dpy,list->LabelWin,
			       fore_pix);
	  XClearWindow(dpy,list->HoldingWin);
	  XClearWindow(dpy,list->LabelWin);
	  list->drawn_selected = 1;
	}
      XDrawString(dpy,list->LabelWin,rvGC,
		  HORIZ_PAD,font->ascent+1, name, strlen(name));
    }
  else
    {
      if(list->drawn_selected)
	{
	  XSetWindowBackground(dpy,list->HoldingWin,
			       back_pix);
	  XSetWindowBackground(dpy,list->LabelWin,
			       back_pix);
	  XClearWindow(dpy,list->HoldingWin);
	  XClearWindow(dpy,list->LabelWin);
	  list->drawn_selected = 0;
	}
      XDrawString(dpy,list->LabelWin,NormalGC,
		  HORIZ_PAD,font->ascent+1, name, strlen(name));
    }

}
