/* session-properties.c - Edit session properties.

   Copyright 1999 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. 

   Authors: Felix Bellaby */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include "capplet-widget.h"
#include "gsm-client-list.h"
#include "gsm-protocol.h"

/* time waited before restarting the last session saved/selected */
static gint chooser_delay = 10000;

/* common widgets */
static GtkObject *protocol;
static GtkWidget *client_list;

/* capplet widgets */
static GtkWidget *capplet;
static GtkWidget *entry;
static GtkWidget *add_button;
static GtkWidget *remove_button;
static GtkWidget *client_editor;
static GtkWidget *scrolled_window;

/* CORBA callbacks and intialization */
static void capplet_build (void);

/* capplet callback prototypes */
static void try (void);
static void revert (void);
static void ok (void);
static void cancel (void);
static void help (void);

/* chooser widgets */
static GtkWidget *dialog;
static GtkWidget *session_list;
static GtkWidget *gnome_foot;
static GtkWidget *left_scrolled;
static GtkWidget *right_scrolled;

/* other widget callback prototypes */
static void entry_changed_cb (GtkWidget *widget);
static void add_cb (GtkWidget *widget);
static void remove_cb (GtkWidget *widget);
static void select_cb (GtkCList *clist);
static void unselect_cb (GtkCList *clist);
static void dirty_cb (GtkWidget *widget);
static void initialized_cb (GtkWidget *widget);

/* chooser intialization */
static void chooser_build (void);

/* session list callbacks */
static void sess_select_row_cb (GtkWidget *widget, gint row);
static void last_session_cb (GtkWidget *w, gchar* name);
static void saved_sessions_cb (GtkWidget *w, GSList* session_names);

/* other widget callback prototypes */
static gint start_timeout (gpointer data);
static void start_cb (GtkWidget *widget);
static void cancel_cb (GtkWidget *widget);
static void right_initialized_cb (GtkWidget *widget);

static void
capplet_build (void)
{
  GtkWidget *table;

  /* capplet */
  capplet = capplet_widget_new ();
  gtk_signal_connect (GTK_OBJECT (capplet), "try",
		      GTK_SIGNAL_FUNC (try), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "revert",
		      GTK_SIGNAL_FUNC (revert), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "cancel",
		      GTK_SIGNAL_FUNC (cancel), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "ok",
		      GTK_SIGNAL_FUNC (ok), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "help",
		      GTK_SIGNAL_FUNC (help), NULL);

  /* program entry box */
  entry = gtk_entry_new ();
  gtk_signal_connect (GTK_OBJECT (entry), "activate", 
		      GTK_SIGNAL_FUNC (add_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "changed", 
		      GTK_SIGNAL_FUNC (entry_changed_cb), NULL);

  /* add/remove buttons */
  add_button = gnome_stock_button (GNOME_STOCK_PIXMAP_ADD);
  gtk_widget_set_sensitive (GTK_WIDGET (add_button), FALSE);
  gtk_signal_connect (GTK_OBJECT (add_button), "clicked",
		      GTK_SIGNAL_FUNC (add_cb), NULL);
  remove_button = gnome_stock_button (GNOME_STOCK_PIXMAP_REMOVE);
  gtk_widget_set_sensitive (GTK_WIDGET (remove_button), FALSE);
  gtk_signal_connect(GTK_OBJECT (remove_button), "clicked",
		     GTK_SIGNAL_FUNC (remove_cb), NULL);
  /* client list */
  client_list = gsm_client_list_new ();
  gsm_client_list_live_session (GSM_CLIENT_LIST (client_list));
  gtk_signal_connect(GTK_OBJECT(client_list), "select_row",
		     GTK_SIGNAL_FUNC (select_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(client_list), "unselect_row",
		     GTK_SIGNAL_FUNC (unselect_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(client_list), "dirty",
		     GTK_SIGNAL_FUNC (dirty_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(client_list), "initialized",
		     GTK_SIGNAL_FUNC (initialized_cb), NULL);

  client_editor = gsm_client_list_get_editor (GSM_CLIENT_LIST (client_list));

  /* startup program list */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_container_add (GTK_CONTAINER (scrolled_window), client_list);
  gtk_signal_connect(GTK_OBJECT(client_list), "select_row",
		     GTK_SIGNAL_FUNC (select_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(client_list), "unselect_row",
		     GTK_SIGNAL_FUNC (unselect_cb), NULL);

  /* table */
  table = gtk_table_new (3, 5, FALSE);
  gtk_table_attach (GTK_TABLE (table), gtk_label_new (_("Program:")),
		    0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), gtk_hbox_new (FALSE, 0), 
		    1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), add_button, 
		    2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0); 
  gtk_table_attach (GTK_TABLE (table), remove_button, 
		    3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), entry, 
		    0, 4, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), client_editor, 
		    0, 4, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), scrolled_window,
		    0, 4, 3, 4, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND,0, 0);
  gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD);
  gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
  gtk_container_add (GTK_CONTAINER(capplet), table);
}

static void
chooser_build (void)
{
  GtkWidget *paned;
  gchar* foot_file = gnome_pixmap_file ("gnome-logo-large.png");
  gchar* title = _("Session");
  GtkRequisition req;

  /* gnome foot */
  gnome_foot = gnome_pixmap_new_from_file (foot_file);
  gtk_widget_size_request (gnome_foot, &req);

  /* session list */
  session_list = gtk_clist_new_with_titles(1, &title);
  gtk_widget_set_usize (session_list, req.width + GNOME_PAD, req.height);
  gtk_signal_connect (GTK_OBJECT (session_list), "select_row",
		      sess_select_row_cb, NULL);

  /* left scrolled window */
  left_scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (left_scrolled),
				  GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_container_add (GTK_CONTAINER (left_scrolled), session_list);

  /* client list */
  client_list = gsm_client_list_new ();

  /* right scrolled window */
  right_scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (right_scrolled),
				  GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_container_add (GTK_CONTAINER (right_scrolled), client_list);

  /* paned window */
  paned = gtk_hpaned_new();
  gtk_paned_handle_size (GTK_PANED (paned), 10);
  gtk_paned_gutter_size (GTK_PANED (paned), 10);
  gtk_paned_add1 (GTK_PANED (paned), left_scrolled);
  gtk_paned_add2 (GTK_PANED (paned), right_scrolled);

  /* dialog */
  dialog = gnome_dialog_new (_("Session Chooser"), NULL);
  gtk_container_add (GTK_CONTAINER (GNOME_DIALOG (dialog)->vbox), paned);
  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
					  _("Start Session"),
					  GNOME_STOCK_BUTTON_OK);
  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
					  _("Cancel Login"),
					  GNOME_STOCK_BUTTON_CANCEL);
  gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
  gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
			       GTK_SIGNAL_FUNC (start_cb), NULL);
  gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
			       GTK_SIGNAL_FUNC (cancel_cb), NULL);

  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  gtk_signal_connect (GTK_OBJECT (protocol), "last_session", 
		      GTK_SIGNAL_FUNC (last_session_cb), NULL);
  gsm_protocol_get_last_session (GSM_PROTOCOL (protocol));

  gtk_signal_connect (GTK_OBJECT (protocol), "saved_sessions", 
		      GTK_SIGNAL_FUNC (saved_sessions_cb), NULL);
  gsm_protocol_get_saved_sessions (GSM_PROTOCOL (protocol));
}

static gboolean warner = FALSE;

static struct poptOption options[] = {
  {"warner", '\0', POPT_ARG_NONE, &warner, 0, N_("Only display warnings."), NULL},
  {NULL, '\0', 0, NULL, 0}
};

int
main (int argc, char *argv[])
{
  gchar* name;
  gint init_result;

  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  init_result = gnome_capplet_init("session-properties", VERSION, argc, argv,
				   options, 0, NULL);

  gtk_signal_connect (GTK_OBJECT (gnome_master_client ()), "die",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gnome_client_set_restart_style (gnome_master_client(), GNOME_RESTART_NEVER);
  switch(init_result) 
    {
    case 0:
    case 1:
      break;

    default:
      g_error ("an initialization error occurred while "
	       "starting 'session-properties-capplet'.");
    }

  protocol = gsm_protocol_new (gnome_master_client());
  if (!protocol)
    {
      g_warning ("Could not connect to gnome-session.");
      exit (1);
    }

  switch(init_result) 
    {
    case 0:
      capplet_build ();
      capplet_gtk_main ();
      break;

    case 1:
      if (warner)
	gsm_session_live (gsm_client_new, NULL) ;
      else
	chooser_build ();
      gtk_main ();
      break;
    }
  return 0;
}

/* CAPPLET CALLBACKS */
static void
try (void)
{
  gsm_client_list_commit_changes (GSM_CLIENT_LIST (client_list));
  capplet_widget_state_changed (CAPPLET_WIDGET (capplet), FALSE);
}

static void
revert (void)
{
  gsm_client_list_revert_changes (GSM_CLIENT_LIST (client_list));
}

static void
ok (void)
{
  try();
  gtk_main_quit();
}

static void
cancel (void)
{
  revert();
  gtk_main_quit();  
}

static void
help (void)
{
  gchar* file = gnome_help_file_find_file(program_invocation_short_name, 
					  "index.html");
  if (file)
    gnome_help_goto (NULL, file);
}

/* This is called when user has changed the entry  */
static void
entry_changed_cb (GtkWidget *widget)
{
  gchar *value = gtk_entry_get_text (GTK_ENTRY (entry));
  gtk_widget_set_sensitive (GTK_WIDGET (add_button), (value && *value));
}

/* Add a new client. */
static void
add_cb (GtkWidget *widget)
{
  gchar *command = gtk_entry_get_text (GTK_ENTRY (entry));

  if (gsm_client_list_add_program (GSM_CLIENT_LIST (client_list), command))
    {
      gtk_entry_set_text (GTK_ENTRY (entry), "");
      gtk_widget_set_sensitive (GTK_WIDGET (add_button), FALSE);
    }
}

/* Remove the selected clients. */
static void
remove_cb (GtkWidget *widget)
{
  gsm_client_list_remove_selection (GSM_CLIENT_LIST (client_list));
}

/* This is called when a client is selected.  */
static void
select_cb (GtkCList *clist)
{
  gtk_widget_set_sensitive (GTK_WIDGET (remove_button), TRUE);
}

/* This is called when no clients are selected.  */
static void
unselect_cb (GtkCList *clist)
{
  if (! clist->selection)
    gtk_widget_set_sensitive (GTK_WIDGET (remove_button), FALSE);
}

/* This is called when an change is made in the client list.  */
static void
dirty_cb (GtkWidget *widget)
{
  capplet_widget_state_changed (CAPPLET_WIDGET (capplet), TRUE);
}

/* This is called when the client_list has been filled by gnome-session */
static void
initialized_cb (GtkWidget *widget)
{
  gtk_widget_show_all (GTK_WIDGET (capplet));
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
}

/* SESSION LIST CALLBACKS */
static gint chooser_timeout = -1;

static void
start (void)
{
  GtkWidget *viewport;

  gtk_signal_connect(GTK_OBJECT(client_list), "started",
		     GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gsm_client_list_start_session (GSM_CLIENT_LIST (client_list));
  gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
  gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 1, FALSE);
  gtk_signal_disconnect_by_func (GTK_OBJECT (session_list), 
				 sess_select_row_cb, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (left_scrolled),
				  GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_container_remove (GTK_CONTAINER (left_scrolled), session_list);
  viewport = gtk_viewport_new (gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (left_scrolled)),
			       gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (left_scrolled)));
  gtk_container_add (GTK_CONTAINER (viewport), gnome_foot);
  gtk_widget_show_all (viewport);
  gtk_container_add (GTK_CONTAINER (left_scrolled), viewport);
}

static void
start_cb (GtkWidget *widget)
{
  if (chooser_timeout != -1)
    gtk_timeout_remove (chooser_timeout);

  start ();
}

static gint
start_timeout (gpointer data)
{
  start ();
  return 0;
}


static void
sess_select_row_cb (GtkWidget* widget, gint row)
{
  gchar* name;

  gtk_clist_get_text (GTK_CLIST (widget), row, 0, &name);
  gsm_client_list_saved_session (GSM_CLIENT_LIST (client_list), name);
  if (chooser_timeout == -1)
    {
      gtk_signal_connect(GTK_OBJECT (client_list), "initialized",
			 GTK_SIGNAL_FUNC (right_initialized_cb), NULL);
    }
  else
    {
      gtk_timeout_remove (chooser_timeout);
      chooser_timeout = gtk_timeout_add (chooser_delay, start_timeout, NULL);
    }
}

static void
cancel_cb (GtkWidget *widget)
{
  gnome_client_request_save (gnome_master_client(), GNOME_SAVE_BOTH, 1, 
			     GNOME_INTERACT_ANY, 0, 1);
}

static gchar* last_session = NULL;

static void
last_session_cb (GtkWidget *widget, gchar* name)
{
  last_session = g_strdup (name);
}

/* gnome-session is responsible for ensuring that there is always
 * at least one session to choose. */  
static void
saved_sessions_cb (GtkWidget *widget, GSList* session_names)
{
  GSList *list;

  gint selected_row = 0;
  
  for (list = session_names; list; list = list->next)
    {
      gint row;
      gchar* name = (gchar*)list->data;
      row = gtk_clist_append (GTK_CLIST (session_list), &name);
      if (! strcmp (last_session, name))
	selected_row = row;
    }
  gtk_clist_select_row (GTK_CLIST (session_list), selected_row, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (left_scrolled),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
}

static void
right_initialized_cb (GtkWidget *widget)
{
  gboolean wait = (GTK_CLIST (session_list)->rows > 1);
  /* We are normally started BEFORE the WM: */
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  if (!wait)
    start();
  gtk_widget_show_all (dialog);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (right_scrolled),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  if (wait)
    chooser_timeout = gtk_timeout_add (chooser_delay, start_timeout, NULL); 
  gtk_signal_disconnect_by_func (GTK_OBJECT (client_list), 
				 right_initialized_cb, NULL);
}
