/* GDM - The Gnome Display Manager
 * Copyright (C) 1998, 1999 Martin Kasper Petersen <mkp@SunSITE.auc.dk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <syslog.h>
#include <glib.h>
#include <crypt.h>
#include <config.h>
#include <confdefs.h>
#include <gnome.h>

#ifdef HAVE_PAM
#include <security/pam_appl.h>

extern pam_handle_t *pamh;

#endif /* HAVE_PAM */

#ifdef HAVE_SHADOW
#include <shadow.h>
#endif /* HAVE_SHADOW */

#include "gdm.h"

static const gchar RCSid[]="$Id: gdmverify.c,v 1.11 1999/02/24 22:25:49 mkp Exp $";

extern void gdm_abort(const char*, ...);

gint gdm_verify_user (gchar *login, gchar *pwd, gchar *display);
void gdm_verify_cleanup (void);
extern void gdm_debug(const char*, ...);


#ifdef HAVE_PAM
gchar *passwd;

static gint 
gdm_verify_pam_conv (int num_msg, const struct pam_message **msg,
		      struct pam_response **resp,
		      void *appdata_ptr)
{
    gint replies = 0;
    struct pam_response *reply = NULL;
    
    reply = g_malloc(num_msg*sizeof(struct pam_response));
    if (!reply) return PAM_CONV_ERR;
    
    for (replies = 0; replies < num_msg; replies++) {
	switch (msg[replies]->msg_style) {
	case PAM_PROMPT_ECHO_OFF:
	    reply[replies].resp_retcode = PAM_SUCCESS;
	    reply[replies].resp = g_strdup(passwd);
	    break;
	case PAM_TEXT_INFO:
	    reply[replies].resp_retcode = PAM_SUCCESS;
	    reply[replies].resp=g_strdup("XXXX");
	    break;
	case PAM_PROMPT_ECHO_ON:
	default:
	    g_free (reply);
	    return PAM_CONV_ERR;
	}
    }
    *resp = reply;
    return(PAM_SUCCESS);
}


static struct pam_conv pamc = {
    &gdm_verify_pam_conv,
    NULL
};

#endif /* HAVE_PAM */


gint 
gdm_verify_user (gchar *login, gchar *pwd, gchar *display) 
{
#ifdef HAVE_PAM
    gint pamerr;

    passwd=pwd;
    
    if((pamerr=pam_start("gdm", login, &pamc, &pamh)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Can't find /etc/pam.d/gdm!"));
	return(0);
    }
    
    if((pamerr=pam_set_item(pamh, PAM_TTY, display)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Can't set PAM_TTY=%s"), display);
	return(0);
    }
    
    if((pamerr=pam_authenticate(pamh, 0)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Couldn't authenticate %s"), login);
	return(0);
    }

    if((pamerr=pam_acct_mgmt(pamh, 0)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Couldn't set acct. mgmt for %s"), login);
	return(0);
    }

    if((pamerr=pam_setcred(pamh, 0)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Couldn't set credentials for %s"), login);
	return(0);
    }

    if((pamerr=pam_open_session(pamh, 0)) != PAM_SUCCESS) {
	pam_end(pamh, pamerr);
	pamh=NULL;
	syslog(LOG_ERR, _("Couldn't open session for %s"), login);
	return(0);
    }

    gdm_debug("gdm_verify_user: PAM session %d initiated", pamh);

    /* Workaround to avoid gdm messages being logged as PAM_pwdb */
    closelog();
    openlog("gdm", LOG_PID, LOG_DAEMON);

#else /* HAVE_PAM */

    gchar *passwd;
    struct passwd *pwent;

#ifdef HAVE_SHADOW
    struct spwd *sp;
#endif

    pwent = getpwnam(login);

    passwd = !pwent ? NULL : pwent->pw_passwd;

#ifdef HAVE_SHADOW
    sp = getspnam(login);
    if (sp != NULL)
        passwd = sp->sp_pwdp;
    endspent();
#endif /* HAVE_SHADOW */

    if (!passwd || strcmp(crypt(pwd, passwd), passwd))
    {
	syslog(LOG_ERR, _("Couldn't authenticate %s"), login);
	return(0);
    }

#endif /* HAVE_PAM */

    return(1);
}


void 
gdm_verify_cleanup (void)
{
#ifdef HAVE_PAM
    gdm_debug("gdm_verify_cleanup: Closing session %d", pamh);

    pam_close_session(pamh, 0);
    pam_end(pamh, PAM_SUCCESS);
    pamh=NULL;

    /* Workaround to avoid gdm messages being logged as PAM_pwdb */
    closelog();
    openlog("gdm", LOG_PID, LOG_DAEMON);
#endif /* HAVE_PAM */
}

/* EOF */
