#include "maillist.h"
#include <objc/objc-api.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <regex.h>

BOOL MLProcessMessage(id aMessage)
{
  void *iter;
  char *akey, *avalue;
  char *laddr, *ltype;
  const char *tmp;
  id mylist = nil;
  BOOL retval = FALSE;
  GString *tmpstr;

  tmpstr = g_string_new(NULL);
  g_string_sprintf(tmpstr, "/%s/lists", APPNAME);
  iter = gnome_config_init_iterator_sections(tmpstr->str);

  tmp = [aMessage getHeaderValue:"Resent-From"];
  if(tmp == NULL)
    {
      g_warning("No Resent-From header\n");
      return FALSE;
    }

  while((iter = gnome_config_iterator_next(iter, &akey, &avalue))
	&& mylist == nil)
    {
      g_string_sprintf(tmpstr, "/%s/lists/%s/listaddr",
		       APPNAME, akey);
      laddr = gnome_config_get_string(tmpstr->str);
      
      if(!strcmp(laddr, tmp))
	{
	  Class t;
	  g_string_sprintf(tmpstr, "/%s/lists/%s/mltype",
			   APPNAME, akey);
	  ltype = gnome_config_get_string(tmpstr->str);

	  if(!ltype)
	    goto out;

	  t = objc_lookup_class(ltype);
	  if(t != Nil)
	    {
	      g_string_sprintf(tmpstr, "/%s/lists/%s",
			       APPNAME, akey);
	      mylist = [[t alloc] initFromConfig:tmpstr->str];
	    }
	  g_free(ltype);
	}

    out:
      g_free(laddr);
      g_free(akey); g_free(avalue);
    }
  [mylist handleMessage:aMessage];

  g_string_free(tmpstr, TRUE);
  return retval;
}

GList *MLGetListTypes(void)
{
  static GList *ltypes = NULL;
  if(!ltypes)
    {
      Class aclass = [MailList class], t;
      for(t = aclass->subclass_list; t; t = t->sibling_class)
	{
	  ltypes = g_list_prepend(ltypes, (gpointer)t->name);
	}
    }
  return ltypes;
}

GList *
MLGetListObjects(void)
{
  void *iter;
  char *akey = NULL, *ltype = NULL;
  static GString *iterpath = NULL;
  id mylist = nil, newobj;
  Class aclass;
  static GList *retval = NULL;

  if(retval) return retval;
  if(!iterpath) iterpath = g_string_new(NULL);

  gnome_config_sync();
  g_string_sprintf(iterpath, "/%s/lists", APPNAME);

  iter = gnome_config_init_iterator_sections(iterpath->str);

  while((iter = gnome_config_iterator_next(iter, &akey, NULL))
	&& mylist == nil)
    {
      g_string_sprintf(iterpath, "/%s/lists/%s/mltype",
		       APPNAME, akey);
      ltype = gnome_config_get_string(iterpath->str);
      
      if(ltype == NULL)
	{
	  g_warning("Couldn't get %s key\n", iterpath->str);
	  g_free(akey); akey = NULL;
	  continue;
	}

      aclass = objc_get_class(ltype);
      if(aclass != Nil)
	{
	  g_string_sprintf(iterpath, "/%s/lists/%s",
			   APPNAME, akey);
	  newobj = [[aclass alloc] initFromConfig:iterpath->str];
	  retval = g_list_append(retval, (gpointer) newobj);
	}
      g_free(ltype); ltype = NULL;
      g_free(akey); akey = NULL;
    }
  return retval;
}

@implementation Persona
- initFromConfig:(char *) config_path
{
  GString *fullpath;

  self = [self init];

  fullpath = g_string_new(config_path);

  g_string_append(fullpath, "/fullname");
  gecos = gnome_config_get_string(fullpath->str);

  g_string_sprintf(fullpath, "%s/email", config_path);
  emailaddr = gnome_config_get_string(fullpath->str);

  g_string_free(fullpath, TRUE);

  return self;
}

- saveToConfig:(char *) config_path
{
  GString *fullpath;

  fullpath = g_string_new(NULL);

  g_string_sprintf(fullpath, "%s/fullname", config_path);
  gnome_config_set_string(fullpath->str, gecos);

  g_string_sprintf(fullpath, "%s/email", config_path);
  gnome_config_set_string(fullpath->str, emailaddr);

  gnome_config_sync();

  g_string_free(fullpath, TRUE);

  return self;
}

- initWithPersonaInfo:(char *)fullName
		     :(char *)emailAddress
{
  self = [self init];
  gecos = g_strdup(fullName);
  emailaddr = g_strdup(emailAddress);
  return self;
}

- (const char *)getFullName
{
  return (const char *)gecos;
}

- (const char *)getEmailAddress
{
  return (const char *)emailaddr;
}

- free
{
  g_free(gecos); gecos = NULL;
  g_free(emailaddr); emailaddr = NULL;
  return [super free];
}
@end

@implementation MailList
- (BOOL) sendControlMessage:(id) persona
			   :(const char *) ctlmsg
			   :(const char *) data
{
  [self error:"sendControlMessage unimplemented in parent\n"];
  return FALSE;
}

- (BOOL) handleMessage:(id) aMessage
{
  [self error:"handleMessage unimplemented in parent\n"];
  return FALSE;
}

+ newFromStream:(FILE *) astream
{
  id newobj = [self alloc];
  return [newobj initFromStream:astream];
}

- init
{
  self = [super init];
  listaddr = NULL; ctladdr = NULL; name = NULL; desc = NULL;
  status = LIST_UNSUBSCRIBED;
  return self;
}

- initFromStream:(FILE *) astream
{
  char abuf[512], *str, *tmp;
  int i;

  self = [self init];

  while(fgets(abuf, 512, astream) != NULL) {
    i = strlen(abuf) - 1;
    while(isspace(abuf[i])) abuf[i--] = '\0';
    for(str = abuf; isspace(*str); str++)
      /* */ ;

    tmp = strtok(str, "=");
    if(!strcmp(tmp, "listaddr"))
      { g_free(listaddr); listaddr = g_strdup(strtok(NULL, "=")); }
    else if(!strcmp(tmp, "ctladdr"))
      { g_free(ctladdr); ctladdr = g_strdup(strtok(NULL, "=")); }
    else if(!strcmp(tmp, "name"))
      { g_free(name); name = g_strdup(strtok(NULL, "=")); }
    else if(!strcmp(tmp, "desc"))
      { g_free(desc); desc = g_strdup(strtok(NULL, "=")); }
    else if(!strcmp(tmp, "mltype"))
      { g_free(mltype); mltype = g_strdup(strtok(NULL, "=")); }
  }

  if(listaddr == NULL || ctladdr == NULL)
    /* We need exceptions here :) */
    return [self free];
  if(name == NULL) {
    g_snprintf(abuf, sizeof(abuf), "%s", listaddr);
    tmp = strtok(abuf, "@");
    if(tmp)
      name = g_strdup(tmp);
    else
      name = g_strdup(listaddr);
  }
  return self;
}

- initWithMaillistInfo:(const char *) inlistaddr
		      :(const char *) inctladdr
		      :(const char *) inname
		      :(const char *) indesc
		      :(const char *) inmltype
		      :(SubscriptionStatus) instatus
{
  self = [self init];
  listaddr = g_strdup(inlistaddr);
  ctladdr = g_strdup(inctladdr);
  name = g_strdup(inname);
  desc = g_strdup(indesc);
  mltype = g_strdup(inmltype);
  status = instatus;

  return self;
}

- initFromConfig:(char *) config_path
{
  GString *abuf;

  self = [self init];

  abuf = g_string_new(NULL);

#define INITPARAM(T) \
  g_string_sprintf(abuf, "%s/%s", config_path, #T); \
  T = gnome_config_get_string(abuf->str);

  INITPARAM(listaddr);
  INITPARAM(ctladdr);
  INITPARAM(desc);
  INITPARAM(mltype);
  INITPARAM(name);

#undef INITPARAM

  g_string_sprintf(abuf, "%s/status", config_path);
  status = gnome_config_get_int(abuf->str);
  g_string_free(abuf, TRUE);

  return self;
}

- saveToConfig:(char *) config_path
{
  GString *abuf;

  abuf = g_string_new(NULL);
#define INITPARAM(T) \
  g_string_sprintf(abuf, "%s/%s", config_path, #T); \
  gnome_config_set_string(abuf->str, T);

  INITPARAM(name);
  INITPARAM(listaddr);
  INITPARAM(ctladdr);
  INITPARAM(desc);
  INITPARAM(mltype);
#undef INITPARAM

  g_string_sprintf(abuf, "%s/status", config_path);
  gnome_config_set_int(abuf->str, status);

  gnome_config_sync();

  g_string_free(abuf, TRUE);

  return self;
}

- (BOOL)subscribe:(id) persona
{
  BOOL retval = 
    [self sendControlMessage:persona
	  :"subscribe"
	  :[persona getEmailAddress]];
  status = LIST_SUBSCRIBED;
  return retval;
}

- (BOOL)unsubscribe:(id) persona
{
  BOOL retval = 
    [self sendControlMessage:persona
	  :"unsubscribe"
	  :[persona getEmailAddress]];
  status = LIST_UNSUBSCRIBED;
  return retval;
}

- (const char *)getName
{
  return (const char *)name;
}

- (const char *)getListAddress
{
  return (const char *)listaddr;
}

- (const char *)getControlAddress
{
  return (const char *)ctladdr;
}

- (const char *)getDescription
{
  return (const char *)desc;
}

- (SubscriptionStatus) getStatus
{
  return status;
}

- (FILE *) sendMail:(id) fromPersona
		   :(const char *) toAddress
		   :(const char *) subjectLine
{
  FILE *retval;
  char timebuf[128];
  time_t foo = time(NULL);
  retval = popen("/usr/sbin/sendmail -t", "w");
  if(retval != NULL) {
    strftime(timebuf, sizeof(timebuf), "%a, %d %b %Y %H:%M:%S (%Z)",
	     localtime(&foo));
    fprintf(retval, "From: <%s>\nTo: <%s>\nSubject: %s\n\n",
	    [fromPersona getEmailAddress],
	    toAddress,
	    subjectLine);
  }
  return retval;
}

- free
{
  g_free(listaddr); g_free(ctladdr); g_free(name); g_free(desc);
  return [super free];
}

@end

static void
writeHeader(gpointer key,
	    gpointer value,
	    gpointer user_data)
{
  fprintf((FILE *)user_data, "%s: %s\n", (char *)key, (char *)value);
}

static void
freeHeader(gpointer key,
	   gpointer value,
	   gpointer user_data)
{
  g_free(key);
  g_free(value);
}

@implementation MailMessage
- writeToStream:(FILE *) astream
{
  g_hash_table_foreach(headers, writeHeader, (gpointer) astream);
  fprintf(astream, "\n%s", body->str);
  return self;
}

- init
{
  self = [super init];
  headers = g_hash_table_new((GHashFunc)g_str_hash,
			     (GCompareFunc)g_str_equal);
  body = g_string_new(NULL);
  return self;
}

- initFromStream:(FILE *) astream
{
  char aline[512], *key, *value, *colonloc;
  gboolean inheaders = TRUE;
  gint i;

  self = [self init];
  while(fgets(aline, sizeof(aline), astream)) {
    if(inheaders == FALSE) {
      body = g_string_append(body, aline);
    } else {      
      i = strlen(aline) - 1;
      while(isspace(aline[i])) aline[i--] = '\0';
      if(aline[0] == '\0')
	inheaders = FALSE;
      else {
	colonloc = strchr(aline, ':'); value = colonloc + 1;
	while(*value && isspace(*value)) value++;
	value = g_strdup(value);
	key = g_malloc(colonloc - aline);
	sprintf(key, "%*.*s",
		colonloc - aline - 1,
		colonloc - aline - 1,
		aline);
	g_hash_table_insert(headers, key, value);
      }
    }
  }
  
  return self;
}

- (const char *) getHeaderValue:(const char *) headerName
{
  return (const char *)g_hash_table_lookup(headers,
					   (const gpointer)headerName);
}

- (const char *) getBody
{
  return (const char *) body;
}

- free
{
  g_hash_table_foreach(headers, freeHeader, NULL);
  g_hash_table_destroy(headers);
  g_free(body);
  return [super free];
}
@end
