/* gnome-ppp - The GNOME PPP Dialer
 * Copyright (C) 1997 Jay Painter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 *(at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <config.h>
#include <gnome.h>
#include "gnome-ppp.h"
#include "account-window.h"
#include "dial-window.h"
#include "global.h"
#include "misc.h"

/* for zvtterm */
#include <zvt/zvtterm.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>
#include <gdk/gdkkeysyms.h>

#define FONT "-misc-fixed-medium-r-normal--12-200-75-75-c-100-iso8859-1"
#define SCROLLBACK_LINES 50


enum
{
  _DISCONNECTED,
  _IN_PROGRESS,
  _CONNECTED
};


/* dial application context structure */
typedef struct _Dial
{
  GtkWidget *app;
  GtkWidget *account_clist;
  GtkWidget *appbar;
  GtkWidget *connect_button_label;
  GtkWidget *connect;
  GtkWidget *debug_terminal_check_menu_item;

  GtkWidget *term_app;
  GtkWidget *term;

  GtkWidget *input_dialog;

  Account *account;
  gint state;
  gint timer_id;
} Dial;


static void refresh_dial_windows();
static void refresh_dial_app(Dial *dial);
static void refresh_dial_account_list(Dial *dial);
static gint close_dial_app(GtkWidget *widget, Dial *dial);


/* app callbacks */
static void destroy_dial_app_cb(GtkWidget *widget);
static void connect_button_cb(GtkWidget *widget);


/* dial engine callbacks */
static gint timeout_cb(Dial *dial);
static void connect_message_cb(Account         *account, 
			       PPPMessageType   message,
			       gchar           *text, 
			       gpointer         data);


/* menu callbacks */
static void menu_file_new_dialer_cb(GtkWidget *widget, Dial *dial);
static void menu_file_exit_cb(GtkWidget *widget, Dial *dial);
static void menu_view_debug_terminal_cb(GtkWidget *widget, Dial *dial);
static void menu_account_new_cb(GtkWidget *widget, Dial *dial);
static void menu_account_duplicate_cb(GtkWidget *widget, Dial *dial);
static void menu_account_edit_cb(GtkWidget *widget, Dial *dial);
static void menu_account_delete_cb(GtkWidget *widget, Dial *dial);
static void menu_help_about_cb(GtkWidget *widget, Dial *dial);


/* dial application context */
static Dial *malloc_dial();
static void free_dial(Dial *dial);
static Dial *get_dial_context(GtkObject *object);
static void set_dial_context(GtkObject *object, Dial *dial);
static Account *get_selected_account(Dial *dial);


/* terminal stuff */
static void new_terminal_app(Dial *dial);
static void terminal_show(Dial *dial);
static void terminal_hide(Dial *dial);
static void terminal_write(Dial *dial, gchar *buff);

static void terminal_size_allocate_cb(GtkWidget *widget);
static gint terminal_delete_event_cb(GtkWidget *widget);
static void terminal_menu_close_cb(GtkWidget *widget, Dial *dial);


/* input dialog */
static void input_dialog_destroy_cb(GtkWidget *widget);
static void input_dialog_cb(gchar *string, gpointer data);


/* menus */
static GnomeUIInfo file_menu[] = 
{
  GNOMEUIINFO_MENU_NEW_ITEM(N_("New Dialer"), NULL, menu_file_new_dialer_cb, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_CLOSE_ITEM(menu_file_exit_cb, NULL),
  GNOMEUIINFO_MENU_EXIT_ITEM(menu_file_exit_cb, NULL),
  GNOMEUIINFO_END
};


static GnomeUIInfo view_menu[] =
{
  GNOMEUIINFO_TOGGLEITEM(N_("_Debug Terminal"), NULL, menu_view_debug_terminal_cb, NULL),
  GNOMEUIINFO_END
};


static GnomeUIInfo account_menu[] = 
{
  GNOMEUIINFO_ITEM_NONE(N_("New..."), NULL, menu_account_new_cb),
  GNOMEUIINFO_ITEM_NONE(N_("Duplicate..."), NULL, menu_account_duplicate_cb),
  GNOMEUIINFO_ITEM_NONE(N_("Edit..."), NULL, menu_account_edit_cb),
  GNOMEUIINFO_ITEM_NONE(N_("Delete..."), NULL, menu_account_delete_cb),
  GNOMEUIINFO_END
};


static GnomeUIInfo help_menu[] =
{
  GNOMEUIINFO_MENU_ABOUT_ITEM(menu_help_about_cb, NULL),
  GNOMEUIINFO_END
};


/* The menu definitions: File/Exit and Help/About are mandatory */
static GnomeUIInfo main_menu[] =
{
  GNOMEUIINFO_MENU_FILE_TREE(file_menu),
  GNOMEUIINFO_MENU_VIEW_TREE(view_menu),

  { GNOME_APP_UI_SUBTREE, N_("_Account"), NULL, account_menu,
    NULL, NULL, (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0,
    NULL },

  GNOMEUIINFO_MENU_HELP_TREE(help_menu),

  GNOMEUIINFO_END
};


/* keep track of the number of dial applications */
static GList *__dial_app_list = NULL;


void
new_dial_app()
{
  static gboolean watcher_init = FALSE;
  Dial *dial;
  GtkWidget *vbox, *hbox;

  /* initalize account watcher to update the clist of all dial
   * application windows
   */
  if (!watcher_init)
    {
      watcher_init = TRUE;
      account_add_watcher(refresh_dial_windows);
    }

  dial = malloc_dial();

  /* set up the main application */
  dial->app = gnome_app_new(GNOME_PPP_NAME, _("gnome-ppp"));
  set_dial_context(GTK_OBJECT(dial->app), dial);

  /* menus & status bar are imbedded into the gnome-app */
  gnome_app_create_menus_with_data(GNOME_APP(dial->app), main_menu, dial);
  dial->debug_terminal_check_menu_item = view_menu[0].widget;

  dial->appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
  gnome_app_set_statusbar(GNOME_APP(dial->app), GTK_WIDGET(dial->appbar));

  gtk_signal_connect(
      GTK_OBJECT(dial->app),
      "destroy",
      GTK_SIGNAL_FUNC(destroy_dial_app_cb),
      NULL);

  gtk_signal_connect(
      GTK_OBJECT(dial->app),
      "delete_event",
      GTK_SIGNAL_FUNC(close_dial_app),
      NULL);

  /* hbox */
  hbox =  gtk_hbox_new(FALSE, 0);
  gnome_app_set_contents(GNOME_APP(dial->app), hbox);
  gtk_container_border_width(GTK_CONTAINER(hbox), 3);
  gtk_widget_show(hbox);

  /* boxes for widget layout */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_box_set_spacing(GTK_BOX(vbox), 5);
  gtk_widget_show(vbox);

  /* account clist */
  dial->account_clist = gtk_clist_new(1);
  set_dial_context(GTK_OBJECT(dial->account_clist), dial);
  gtk_box_pack_start(GTK_BOX(vbox), dial->account_clist, TRUE, TRUE, 0);

  gtk_clist_column_titles_passive(GTK_CLIST(dial->account_clist));
  gtk_clist_set_selection_mode(GTK_CLIST(dial->account_clist), GTK_SELECTION_BROWSE);
  gtk_widget_show(dial->account_clist);

  /* connect button */
  dial->connect_button_label = gtk_label_new("");
  set_dial_context(GTK_OBJECT(dial->connect_button_label), dial);
  gtk_widget_show(dial->connect_button_label);

  dial->connect = gtk_button_new();
  set_dial_context(GTK_OBJECT(dial->connect), dial);
  gtk_box_pack_start(GTK_BOX(vbox), dial->connect, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(dial->connect), dial->connect_button_label);
  gtk_widget_set_usize(dial->connect, 100, BUTTON_HEIGHT);

  gtk_signal_connect(
      GTK_OBJECT(dial->connect),
      "clicked",
      GTK_SIGNAL_FUNC(connect_button_cb),
      NULL);

  gtk_widget_show(dial->connect);

  /* create terminal */
  new_terminal_app(dial);

  /* set data in the dial application */
  refresh_dial_account_list(dial);
  refresh_dial_app(dial);

  /* Show it */
  gtk_widget_show(dial->app);
}


void
refresh_dial_windows()
{
  GList *list;
  Dial *dial;

  /* refresh all the dial applications */
  list = __dial_app_list;
  while (list)
    {
      dial = (Dial *) list->data;
      list = list->next;
      refresh_dial_account_list(dial);
    } 
}


static void
refresh_dial_account_list(Dial *dial)
{
  GList *list;
  Account *saved_account, *account;

  /* save selected account */
  saved_account = get_selected_account(dial);

  /* refresh the accounts clist */
  gtk_clist_freeze(GTK_CLIST(dial->account_clist));
  gtk_clist_clear(GTK_CLIST(dial->account_clist));

  list = account_list();
  while (list)
    {
      gint row;
      gchar *text[1];

      account = (Account *) list->data;
      list = list->next;

      text[0] = account->name->str;
      row = gtk_clist_append(GTK_CLIST(dial->account_clist), text);
      if (row < 0)
	{
	  g_error(_("refresh_dial_app(): gtk_clist_append failed."));
	}
      gtk_clist_set_row_data(GTK_CLIST(dial->account_clist), row, account);

      /* select this row if it is the default account */
      if (account == saved_account)
	{
	  gtk_clist_select_row(GTK_CLIST(dial->account_clist), row, 0);
	}
    }
  gtk_clist_thaw(GTK_CLIST(dial->account_clist));
}


static void
refresh_dial_app(Dial *dial)
{
  /* set the active state of the connection button and the
   * account-selection option menu based on the connection state
   */
  if (account_list())
    {
      switch (dial->state)
	{
	case _DISCONNECTED:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), _("Connect"));
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  break;

	case _IN_PROGRESS:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), _("Cancel"));
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  break;

	case _CONNECTED:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), _("Disconnect"));
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  break;
	}
    }
  else
    {
      gtk_label_set_text(GTK_LABEL(dial->connect_button_label), _("Connect"));
      gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), FALSE);
    }
}


static gint
close_dial_app(GtkWidget *widget, Dial *dial)
{
  dial = get_dial_context(GTK_OBJECT(widget));

  /* don't close if the connection is active */
  if (dial->state != _DISCONNECTED)
    {
      gnome_error_dialog(_("You must disconnect before closing."));
      return TRUE;
    }

  return FALSE;
}


static void
destroy_dial_app_cb(GtkWidget *widget)
{ 
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));

  gtk_widget_destroy(dial->term_app);
  free_dial(dial);

  /* quit gnome-ppp when there are no moer dial applications */
  if (__dial_app_list == NULL)
    {
      account_remove_watcher(refresh_dial_windows);
      gtk_main_quit();
    }
}


static void
connect_button_cb(GtkWidget *widget)
{
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));

  switch (dial->state)
    {
    case _DISCONNECTED:
      g_assert(dial->account == NULL);
      g_assert(GTK_CLIST(dial->account_clist)->selection);

      /* get selected account */
      if (!(dial->account = get_selected_account(dial)))
	{
	  g_error(_("connect_button_cb(): no selected account"));
	}

      /* locked accounts are already in use */
      if (!account_is_locked(dial->account))
	{
	  account_lock(dial->account);

	  /* start the connection; if the start is successful, then
	   * add the timeout callback to run the state engine
	   */
	  if (connect_start(dial->account, connect_message_cb, dial))
	    {
	      dial->timer_id = 
		gtk_timeout_add(500, (GtkFunction) timeout_cb, dial);
	    }
	}
      else
	{
	  gnome_error_dialog(_("This account is in use."));
	  dial->account = NULL; 
	}
    break;

    case _IN_PROGRESS:
    case _CONNECTED:
      g_assert(dial->account);
      connect_stop(dial->account);
    break;
    }
}


static gint
timeout_cb(Dial *dial)
{
  if (connect_engine_iteration(dial->account))
    {
      return 1;
    }
  else
    {
      /* blank out the timeout id and return 0 shutting
       * down the gtk_timeout
       */
      dial->timer_id = -1;
      return 0;
    }
}


/* message callback from connect */
static void
connect_message_cb(Account         *account, 
		   PPPMessageType   message, 
		   gchar           *text, 
		   gpointer         data)
{
  gboolean update_statusbar;
  gint saved_state;
  Dial *dial;

  dial = (Dial *) data;
  update_statusbar = FALSE;
  saved_state = dial->state;

  /* sanity checks */
  if (dial->account != account)
    {
      g_error("connect_message_cb(): account != dial->account");
    }

  /* map messages from the connect engine to internal
   * states of the dial window
   */
  switch (message)
    {
    case PPP_DISCONNECTED:
      dial->state = _DISCONNECTED;
      update_statusbar = TRUE;
      break;

    case PPP_IN_PROGRESS:
      dial->state = _IN_PROGRESS;
      update_statusbar = TRUE;
      break;

    case PPP_CONNECTED:
      dial->state = _CONNECTED;
      update_statusbar = TRUE;
      break;

    case PPP_ERROR:
      gnome_error_dialog(_(text));
      break;

    case PPP_DEBUG:
      terminal_write(dial, _(text));
      return;
      break;

    case PPP_PROMPT_INPUT:
      if (dial->input_dialog)
	{
	  g_error("connect_message_cb(): dial->user_input_dialog != NULL");
	}

      dial->input_dialog = gnome_request_dialog(
	                       TRUE,
			       _(text),
			       "",
			       100,
			       (GnomeStringCallback) input_dialog_cb,
			       dial,
			       NULL);
      
      set_dial_context(GTK_OBJECT(dial->input_dialog), dial);

      gtk_signal_connect(
          GTK_OBJECT(dial->input_dialog),
	  "destroy",
	  GTK_SIGNAL_FUNC(input_dialog_destroy_cb),
	  NULL);

      break;

    default:
      g_error("connect_message_cb(): unknown callback message");
      break;
    }

  /* display a message if there is one */
  if (update_statusbar)
    {
      if (text)
	{
	  gnome_appbar_set_status(GNOME_APPBAR(dial->appbar), _(text));
	}
      else
	{
	  gnome_appbar_set_status(GNOME_APPBAR(dial->appbar), "");
	}
    }

  /* if the state is disconnected, then unlock the account */
  if (dial->state == _DISCONNECTED)
    {
      /* get rid of any remaining input dialog */
      if (dial->input_dialog)
	{
	  gtk_widget_destroy(dial->input_dialog);
	}

      account_unlock(account);
      dial->account = NULL;
    }


  /* refresh the dial window if the state changed */
  if (dial->state != saved_state)
    {
      refresh_dial_app(dial);
    }
}


/* menu callbacks need to be different */
static void
menu_file_new_dialer_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);
  new_dial_app();
}


static void
menu_file_exit_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);

  if (close_dial_app(dial->app, dial) == FALSE)
    {
      gtk_widget_destroy(dial->app);
    }
}


static void
menu_view_debug_terminal_cb(GtkWidget *widget, Dial *dial)
{
  if (GTK_CHECK_MENU_ITEM(widget)->active)
    {
      terminal_show(dial);
    }
  else
    {
      terminal_hide(dial);
    }
}


static void
menu_account_new_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);
  open_account_window(NULL, FALSE);
}


static void
menu_account_duplicate_cb(GtkWidget *widget, Dial *dial)
{
  Account *account;

  g_assert(dial != NULL);

  if (!(account = get_selected_account(dial)))
    {
      return;
    }
  open_account_window(account, TRUE);
}


static void
menu_account_edit_cb(GtkWidget *widget, Dial *dial)
{
  Account *account;

  g_assert(dial != NULL);

  if (!(account = get_selected_account(dial)))
    {
      return;
    }
  open_account_window(account, FALSE);
}


static void
menu_account_delete_cb(GtkWidget *widget, Dial *dial)
{
  Account *account;

  g_assert(dial != NULL);

  if (!(account = get_selected_account(dial)))
    {
      return;
    }

  /* can't delete locked (in-use) accounts */
  if (account_is_locked(account) || !account_delete(account))
    {
      gnome_error_dialog(_("The account is in use and cannot be removed."));
      return;
    }

  account_save();
}


static void
menu_help_about_cb(GtkWidget *widget, Dial *dial)
{
  GtkWidget *about;
  const gchar *author[] = 
  {
    GNOME_PPP_AUTHOR,
    NULL
  };
	
  about = gnome_about_new(
      _(GNOME_PPP_NAME), 
      GNOME_PPP_VERSION,
      _(GNOME_COPYRIGHT),
      author,
      NULL,
      NULL);

  gtk_widget_show(about);
}


/* dial application context */
static Dial*
malloc_dial()
{
  Dial *dial = g_malloc(sizeof(Dial));
  dial->account = NULL;
  dial->state = _DISCONNECTED;
  dial->timer_id = -1;

  dial->term_app = NULL;
  dial->term = NULL;

  dial->input_dialog = NULL;

  __dial_app_list = g_list_append(__dial_app_list, dial);
  
  return dial;
}


static void
free_dial(Dial *dial)
{
  __dial_app_list = g_list_remove(__dial_app_list, dial);
  g_free(dial);
}


static Dial* 
get_dial_context(GtkObject *object)
{
  return gtk_object_get_data(object, "dial_context");
}


static void
set_dial_context(GtkObject *object, Dial *dial)
{
  gtk_object_set_data(object, "dial_context", dial);
}

static Account *
get_selected_account(Dial *dial)
{
  Account *account;

  if (GTK_CLIST(dial->account_clist)->selection)
    {
      return (Account *)
	gtk_clist_get_row_data(
	    GTK_CLIST(dial->account_clist),
	    GPOINTER_TO_INT(GTK_CLIST(dial->account_clist)->selection->data));
    }
  
  return NULL;
}


/*** terminal ***/
static void
new_terminal_app(Dial *dial)
{
  GtkWidget *hbox, *scrollbar;

  dial->term_app = gnome_app_new(GNOME_PPP_NAME, "gnome-ppp");
  set_dial_context(GTK_OBJECT(dial->term_app), dial);

  gtk_signal_connect(
      GTK_OBJECT(dial->term_app),
      "delete_event",
      GTK_SIGNAL_FUNC(terminal_delete_event_cb),
      NULL);

  /* create hbox */
  hbox = gtk_hbox_new(FALSE, 0);
  gnome_app_set_contents(GNOME_APP(dial->term_app), hbox);
  gtk_box_set_spacing(GTK_BOX(hbox), 2);
  gtk_container_border_width(GTK_CONTAINER(hbox), 2);
  gtk_widget_show(hbox);

  /* create terminal */
  dial->term = zvt_term_new();
  set_dial_context(GTK_OBJECT(dial->term), dial);
  gtk_box_pack_start(GTK_BOX(hbox), dial->term, 1, 1, 0);
  zvt_term_set_shadow_type(ZVT_TERM(dial->term), GTK_SHADOW_IN);
  zvt_term_set_size(ZVT_TERM(dial->term), 80, 25);
  zvt_term_set_font_name(ZVT_TERM(dial->term), FONT);
  zvt_term_set_blink(ZVT_TERM(dial->term), FALSE);
  zvt_term_set_bell(ZVT_TERM(dial->term), TRUE);
  zvt_term_set_scrollback(ZVT_TERM(dial->term), SCROLLBACK_LINES);
  zvt_term_set_scroll_on_keystroke(ZVT_TERM(dial->term), TRUE);
  zvt_term_set_scroll_on_output(ZVT_TERM(dial->term), FALSE);
  zvt_term_set_background(ZVT_TERM(dial->term), NULL, 0, 0);

  gtk_signal_connect(
      GTK_OBJECT(dial->term),
      "key_press_event",
      GTK_SIGNAL_FUNC(NULL),
      NULL);

  gtk_signal_connect_after(
      GTK_OBJECT(dial->term),
      "size_allocate",
      GTK_SIGNAL_FUNC(terminal_size_allocate_cb),
      NULL);

  gtk_widget_show(dial->term);

  /* scrollbar */
  scrollbar = 
    gtk_vscrollbar_new(GTK_ADJUSTMENT(ZVT_TERM(dial->term)->adjustment));
  GTK_WIDGET_UNSET_FLAGS(scrollbar, GTK_CAN_FOCUS);
  gtk_box_pack_start(GTK_BOX(hbox), scrollbar, FALSE, TRUE, 0);
  gtk_widget_show(scrollbar);

  /* realize the GNOME terminal app */
  gtk_widget_realize(dial->term_app);
}


static void
terminal_show(Dial *dial)
{
  gtk_widget_show(dial->term_app);
}


static void
terminal_hide(Dial *dial)
{
  gtk_widget_hide(dial->term_app);
}


static void
terminal_write(Dial *dial, gchar *buff)
{
  zvt_term_feed(ZVT_TERM(dial->term), buff, strlen(buff));
#if 0
  int state;

  state = 0;
  while (*buff != '\0')
    {
      switch (state)
	{
	case 0:
	  if (*buff == '\r')
	    {
	      state = 1;
	      zvt_term_feed(ZVT_TERM(dial->term), "\r\n", 2);
	    }
	  else
	    {
	      zvt_term_feed(ZVT_TERM(dial->term), buff, 1);
	    }
	  break;

	case 1:
	  if (*buff == '\r')
	    {
	      zvt_term_feed(ZVT_TERM(dial->term), "\r\n", 2);
	    }
	  else if (*buff == '\n')
	    {
	      state = 0;
	    }
	  else
	    {
	      state = 0;
	      zvt_term_feed(ZVT_TERM(dial->term), buff, 1);
	    }
	  break;
	}

      buff++;
    }
#endif
}


static void
terminal_size_allocate_cb(GtkWidget *widget)
{
  ZvtTerm *term;
  XSizeHints sizehints;
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));
  term = ZVT_TERM(widget);
  
  sizehints.base_width = 
    (GTK_WIDGET(dial->term_app)->allocation.width) +
    (GTK_WIDGET(term)->style->klass->xthickness * 2) -
    (GTK_WIDGET(term)->allocation.width);
  
  sizehints.base_height =
    (GTK_WIDGET(dial->term_app)->allocation.height) +
    (GTK_WIDGET(term)->style->klass->ythickness * 2) -
    (GTK_WIDGET(term)->allocation.height);
  
  sizehints.width_inc = term->charwidth;
  sizehints.height_inc = term->charheight;
  sizehints.min_width = sizehints.base_width + sizehints.width_inc;
  sizehints.min_height = sizehints.base_height + sizehints.height_inc;
  
  sizehints.flags = (PBaseSize|PMinSize|PResizeInc);
  
  XSetWMNormalHints(GDK_DISPLAY(),
		    GDK_WINDOW_XWINDOW (GTK_WIDGET(dial->term_app)->window),
		    &sizehints);
  gdk_flush();
}


static gint
terminal_delete_event_cb(GtkWidget *widget)
{
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));
  terminal_hide(dial);

  /* update the view/debug window check menu item */
  gtk_check_menu_item_set_active(
      GTK_CHECK_MENU_ITEM(dial->debug_terminal_check_menu_item),
      FALSE);

  return TRUE;
}


static void
terminal_menu_close_cb(GtkWidget *widget, Dial *dial)
{
  terminal_hide(dial);
}


/*** input dialog ***/
static void
input_dialog_destroy_cb(GtkWidget *widget)
{
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));
  dial->input_dialog = NULL;
}



static void 
input_dialog_cb(gchar *string, gpointer data)
{
  Dial *dial;

  dial = (Dial *) data;
  g_assert(dial->account != NULL);

  connect_send(dial->account, string, TRUE);
}
