/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include <netdb.h>
#include <arpa/inet.h>
#include "gtkutil.h"
#ifndef USE_GNOME
#ifdef USE_IMLIB
#include <gdk_imlib.h>
#endif
#endif

GdkFont *font_normal = 0;
GdkFont *font_bold = 0;

GdkFont *dialog_font_normal = 0;
GdkFont *dialog_font_bold = 0;

GSList *sess_list = 0;
GSList *serv_list = 0;
GSList *dcc_list = 0;
GSList *dialog_list = 0;
GSList *url_list = 0;

GtkStyle *channelwin_style = 0;
GtkStyle *dialogwin_style = 0;

gint notify_tag = -1;

extern GSList *ctcp_list;
extern GSList *popup_list;
extern GSList *button_list;
extern GSList *command_list;

#ifdef USE_PERL
extern struct session *perl_sess;
#endif
extern struct session *current_tab;
extern GtkWidget *main_window;
extern struct session *lastlog_last_sess;
extern struct session *menu_sess;
extern GdkColor colors[];
struct xchatprefs prefs;

void xchat_cleanup(void);
void kill_session(struct session *sess);
struct session *new_session(struct server *serv);

/* maingui.c */

extern void create_window(struct session *sess);

/* rawlog.c */

extern void add_rawlog(struct server *serv, char *text);
extern void close_rawlog(GtkWidget *wid, struct server *serv);

/* outbound.c */

extern void handle_inputgad(GtkWidget *igad, struct session *sess);

/* inbound.c */

extern void new_topic(char *buf);
extern void names_list(char *buf);
extern void you_joined(struct session *sess, char *chan);
extern void user_joined(char *chan, char *user, char *ip);
extern void new_nick(struct session *sess, char *newnick);
extern void process_line(struct session *sess, struct server *serv, char *buf);
extern void module_setup();

/* dcc.c */

extern void dcc_check_timeouts(void);
extern void free_dccs(void);
extern void dcc_notify_kill(struct server *serv);

/* dialog.c */

extern struct dialog *new_dialog(char *nick);
extern void dialog_notify_kill(struct server *serv);

/* server.c */

extern void connecting_fin(struct session *);
extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern void disconnect_server(struct session *sess, int sendquit);

/* serverlist.c */

extern void serverlist_autoconnect(struct session *sess, int aut);
extern void open_server_list(GtkWidget *widd, struct session *sess);

/* config.c */

extern void save_config(void);
extern void load_config(void);

/* userlist.c */

extern void init_userlist_xpm(struct session *sess);
extern struct user *find_name(struct session *sess, char *name);

/* notify.c */

extern void notify_load(void);
extern void notify_save(void);
extern void notify_checklist(void);

/* lastlog.c */

extern void hide_lastlog(GtkWidget *wid, gpointer *data);

#ifdef USE_PERL
/* perl.c */

void perl_init(struct session *, int);
void perl_end(void);
#endif

/* editlist.c */

extern void list_loadconf(char *, GSList **, char *);
extern void list_free(GSList **);

/* text.c */

extern void PrintText(struct session *sess, unsigned char *text);
extern unsigned char *strip_color(unsigned char *text);
extern void end_logging (int fd);

/* util.c */

extern int is_data_avail(int sok);



struct session *find_session_from_nick(char *nick, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;

   if(find_name(serv->front_session, nick)) return serv->front_session;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(find_name(sess, nick)) return sess;
      }
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_channel(char *chan, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(!strcasecmp(chan, sess->channel))
      {
	 if(!serv) return sess;
	 if(serv == sess->server) return sess;
      }
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_waitchannel(char *chan, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(!strcasecmp(chan, sess->waitchannel)) return sess;
      }
      list = list->next;
   }
   return 0;
}

void show_generic_channel(struct server *serv, char *chan, char *msg)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess) PrintText(sess, msg);
}

void read_data(struct server *serv, gint sok)
{
   int len;
   char lbuf[2050];

   do
   {
      len = recv(sok, lbuf, sizeof lbuf-2, 0);
      if(len < 1)
      {
	 if(prefs.autoreconnect)
	 {
	    struct session *s;
	    GSList *list = sess_list; /* make sure auto rejoin can work */
	    while(list)
	    {
	       s = (struct session *)list->data;
	       strcpy(s->waitchannel, s->channel);
	       strcpy(s->willjoinchannel, s->channel);
	       list = list->next;
	    }
	    disconnect_server(serv->front_session, FALSE);
	    connect_server(serv->front_session, serv->servername, serv->port, FALSE);
	 } else
	   disconnect_server(serv->front_session, FALSE);
	 return;
      } else {
	 int i = 0;
	 lbuf[len] = 0;

	 while(i < len)
	 {
	    switch(lbuf[i])
	    {
	     case '\r':
	       break;
	     case '\n':
	       serv->linebuf[serv->pos] = 0;
	       if(prefs.stripcolor)
	       {
		  char *temp = strip_color(serv->linebuf);
		  process_line(serv->front_session, serv, temp);
		  free(temp);
	       } else
		 process_line(serv->front_session, serv, serv->linebuf);
	       serv->pos = 0;
	       break;
	     default:
	       serv->linebuf[serv->pos] = lbuf[i];
	       serv->pos++;
	       if(serv->pos == 2047) serv->pos = 2046;
	    }
	    i++; 
	 }
      }
   }
   while(is_data_avail(sok));
}

gint kill_session_callback(GtkWidget *win, struct session *sess)
{
   char tbuf[256];
   int willquit = TRUE;
   struct server *serv = sess->server;
   struct session *killsess = sess;
   struct session *s;
   GSList *list = sess_list;

   if(!sess->quitreason) sess->quitreason = prefs.quitreason;

   while(list)
   {
      s = (struct session *)list->data;
      if(s->server == killsess->server && s != killsess)
      {
	 willquit = FALSE;
	 list = 0;
      } else
	 list = list->next;
   }

#ifdef USE_PERL
   if(perl_sess == sess) perl_sess = 0;
#endif
   if(current_tab == sess) current_tab = 0;
   if(menu_sess == sess) menu_sess = (struct session *)sess_list->data;
   if(sess->server->front_session == sess) sess->server->front_session = 0;
   if(sess->bar) connecting_fin(sess);
   
   if(sess->server && sess->server->connected && willquit)
   {
      sprintf(tbuf, "QUIT :%s\r\n", sess->quitreason);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
   } else {
      if(sess->server && sess->server->connected && sess->channel[0] && 
	((sess->channel[0] == '#') || (sess->channel[0] == '&')))
      {
	 sprintf(tbuf, "PART %s\r\n", sess->channel);
	 send(sess->server->sok, tbuf, strlen(tbuf), 0);
      }
   }

   if(lastlog_last_sess == sess) hide_lastlog(0, 0);

   if (sess->logfd != -1)
   	end_logging(sess->logfd);
   sess_list = g_slist_remove(sess_list, sess);
   free(sess);

   if(sess_list)
   {
         list = sess_list;
	 while(list)
	 {
	    sess = (struct session *)list->data;
	    if(sess->server == serv) return TRUE;
	    list = list->next;
	 }
	 if(serv->connected)
	 {
/*	    sprintf(tbuf, "QUIT :%s\r\n", sess->quitreason);
	    send(serv->sok, tbuf, strlen(tbuf), 0);*/
	    gdk_input_remove(serv->iotag);
	    shutdown(serv->sok, 2);
	    close(serv->sok);
	 }
         if(serv->connecting)
	 {
	    gdk_input_remove(serv->iotag);
	    close(serv->childread);
	    close(serv->sok);
	 }
         dialog_notify_kill(serv);
	 if(serv->rawlog_window) close_rawlog(0, serv);
         serv_list = g_slist_remove(serv_list, serv);
         dcc_notify_kill(serv);
         free(serv);
	 return TRUE;
   }
   if(prefs.tabchannels && main_window) gtk_widget_destroy(main_window);
   
   xchat_cleanup();
   return TRUE;
}

struct server *new_server(void)
{
   struct server *serv = malloc(sizeof(struct server));
   if(!serv) return 0;
   memset(serv, 0, sizeof(struct server));
   serv->sok = -1;
   strcpy(serv->nick, prefs.nick1);
   serv_list = g_slist_append(serv_list, serv);
   return serv;
}

GdkFont *my_font_load(char *fontname)
{
   GdkFont *font = gdk_font_load(fontname);
   if(!font)
   {
      char temp[256];
      sprintf(temp, "Cannot open font:\n\n%s", fontname);
      gtkutil_simpledialog(temp);
      strcpy(fontname, "fixed");
      font = gdk_font_load(fontname);
      if(!font)
      {
	 g_error("gdk_font_load failed");
	 gtk_exit(0);
      }
   }
   return font;
}

struct session *new_session(struct server *serv)
{
   struct session *sess;

   sess = malloc(sizeof(struct session));
   if(!sess) return 0;

   memset(sess, 0, sizeof(struct session));
   sess->server = serv;
   sess->logfd = -1;

   create_window(sess);

   sess_list = g_slist_append(sess_list, sess);

   return(sess);
}

void free_sessions(void)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if (sess->logfd != -1)
      	end_logging(sess->logfd);
      history_free(&sess->history);
      kill_session_callback(0, sess);
      free(sess);
      list = list->next;
   }
}

#define defaultconf_ctcp "NAME TIME\nCMD /nctcp %s TIME %t\n\nNAME PING\nCMD /nctcp %s PING %d\n\nNAME USERINFO\nCMD /notice %s What are you doing Dave?\n"
#define defaultconf_popup "NAME CTCP Version\nCMD /ctcp %s VERSION\n\nNAME CTCP Userinfo\nCMD /ctcp %s USERINFO\n\nNAME CTCP Ping\nCMD /ping %s\n\nSEP\n\nNAME Give Voice\nCMD /voice %s\n\nNAME Take Voice\nCMD /devoice %s\n"
#define defaultconf_buttons "NAME Op\nCMD /op %s\n\nNAME DeOp\nCMD /deop %s\n\nNAME Ban\nCMD /ban %s\n\nNAME Kick\nCMD /kick %s\n\nNAME Send\nCMD /dcc send %s\n\nNAME Chat\nCMD /chat %s\n\nNAME Dialog\nCMD /query %s\n\nNAME Whois\nCMD /whois %s\n"
char *defaultconf_commands =
"NAME BANLIST\nCMD /quote MODE %c +b\n\n"
"NAME CHAT\nCMD /dcc chat %2\n\n"
"NAME CONNECT\nCMD /quote CONNECT &2\n\n"
"NAME DIALOG\nCMD /query %2\n\n"
"NAME DRAW\nCMD /dcc draw %2\n\n"
"NAME EXIT\nCMD /quit\n\n"
"NAME INFO\nCMD /quote INFO\n\n"
"NAME J\nCMD /join &2\n\n"
"NAME KILL\nCMD /quote KILL %2 :&3\n\n"
"NAME LEAVE\nCMD /part &2\n\n"
"NAME LINKS\nCMD /quote LINKS\n\n"
"NAME LIST\nCMD /quote LIST\n\n"
"NAME LUSERS\nCMD /quote LUSERS\n\n"
"NAME M\nCMD /msg &2\n\n"
"NAME MAP\nCMD /quote MAP\n\n"
"NAME MOTD\nCMD /quote MOTD\n\n"
"NAME OPER\nCMD /quote OPER %2 %3\n\n"
"NAME RAW\nCMD /quote &2\n\n"
"NAME REHASH\nCMD /quote REHASH\n\n"
"NAME SERVHELP\nCMD /quote HELP\n\n"
"NAME TIME\nCMD /quote TIME\n\n"
"NAME SQUIT\nCMD /quote SQUIT &2\n\n"
"NAME STATS\nCMD /quote STATS &2\n\n"
"NAME UPTIME\nCMD /quote STATS u\n\n"
"NAME VER\nCMD /ctcp %2 VERSION\n\n"
"NAME VERSION\nCMD /ctcp %2 VERSION\n\n"
"NAME WII\nCMD /quote WHOIS %2 %2\n\n"
"NAME WHO\nCMD /quote WHO &2\n\n"
"NAME WHOIS\nCMD /quote WHOIS &2\n\n"
"NAME WHOWAS\nCMD /quote WHOWAS &2\n\n"
;

void xchat_init(int autoconnect)
{
   struct session *sess;
   struct server *serv;

#ifndef USE_GNOME
#ifdef USE_IMLIB
   gdk_imlib_init();
#endif
#endif

   load_config();
   list_loadconf("popup.conf", &popup_list, defaultconf_popup);
   list_loadconf("ctcpreply.conf", &ctcp_list, defaultconf_ctcp);
   list_loadconf("buttons.conf", &button_list, defaultconf_buttons);
   list_loadconf("commands.conf", &command_list, defaultconf_commands);
   notify_load();

   font_normal = my_font_load(prefs.font_normal);
   font_bold = my_font_load(prefs.font_bold);
   dialog_font_normal = my_font_load(prefs.dialog_font_normal);
   dialog_font_bold = my_font_load(prefs.dialog_font_bold);

   serv = new_server();
   if(serv)
   {
      sess = new_session(serv);
      init_userlist_xpm(sess);
#ifdef USE_PERL
      perl_init(sess, TRUE);
#endif
#ifdef USE_PLUGIN
      module_setup();
#endif

      if(!prefs.skipserverlist || autoconnect) {
	 open_server_list(0, sess);
      }

      if(autoconnect) serverlist_autoconnect(sess, autoconnect);
      if(prefs.notify_timeout)
	notify_tag = gtk_timeout_add(prefs.notify_timeout, (GtkFunction) notify_checklist, 0);

      gtk_timeout_add(5000, (GtkFunction) dcc_check_timeouts, 0);
   }
}

void xchat_cleanup(void)
{
   if(prefs.autosave) save_config();
   notify_save();
   free_sessions();
   list_free(&dialog_list);
   list_free(&serv_list);
   list_free(&dcc_list);
   list_free(&ctcp_list);
   list_free(&popup_list);
   list_free(&button_list);
   list_free(&command_list);
#ifdef USE_PERL
   perl_end();
#endif
   gtk_exit(0);
}

int main(int argc, gchar *argv[])
{
   int autoconnect = 0;
   char *a[] = { "X-Chat", 0 };
   char **av = a;
   int ac = 1;

   if(argc > 1)
   {
      if(!strcasecmp(argv[1], "-v") || !strcasecmp(argv[1], "--version"))
      {
	 puts("X-Chat "VERSION"");
         return 0;
      }
      if(!strcasecmp(argv[1], "-h") || !strcasecmp(argv[1], "--help"))
      {
	 puts("X-Chat "VERSION" Options:\n\n"
	      "       -c <n>    : auto connect to server entry <n>\n"
	      "       -v        : show version information\n"
	      );
         return 0;
      }
   }

   if(argc > 2)
   {
      if(!strcasecmp(argv[1], "-c")) autoconnect = atoi(argv[2]);
   }

#ifdef USE_GNOME
   gnome_init("X-Chat", VERSION, ac, av);
#else
   gtk_init(&ac, &av);
#endif

   xchat_init(autoconnect);

   gtk_main();

   xchat_cleanup();

   return 0;
}
