/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 *
 * Window list
 */

#include "icewm.h"

YListBox::YListBox(YScrollBar *vert, YWindow *aParent, Window win): YWindow(aParent, win) {
    fVerticalScroll = vert;
    if (vert)
        vert->setListener(this);
    fOffsetY = 0;
    fFirst = fLast = 0;
    fFocusedItem = 0;
    fSelectStart = fSelectEnd = -1;
    fSelecting = -1;
}

YListBox::~YListBox() {
}

int YListBox::addItem(YListItem *item) {
    YFrameClient *c = item->frame()->client();

    if (c->owner() && c->owner()->frame()) {
        YListItem *w = c->owner()->frame()->winListItem();

        item->setNext(w->getNext());
        item->setPrev(w);
        w->setNext(item);
        if (w == fLast)
            fLast = item;
    } else {
        item->setNext(0);
        item->setPrev(fLast);
        if (fLast)
            fLast->setNext(item);
        else
            fFirst = item;
        fLast = item;
    }
    repaint();
    return 1;
}

YListItem *YListBox::findItem(int mouseY, int &no) {
    int y = 0;
    int lh, fh = windowListFont->height();
    YListItem *a = fFirst;
    
    no = 0;
    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;
    while (a) {
        if (mouseY >= y - fOffsetY && mouseY < y - fOffsetY + lh)
            return a;
        y += lh;
        a = a->getNext();
        no++;
    }
    no = -1;
    return 0;
}

void YListBox::selectItem(YListItem *i) {
    if (i == 0) {
        fFocusedItem = 0;
        fSelectStart = 0;
        fSelectEnd = 0;
    } else {
        int n = 0;
        YListItem *f = fFirst;
        for(; f ; f = f->getNext(), n++)
            if (i == f) {
                fFocusedItem = n;
                fSelectStart = fSelectEnd = n;
            }
    }
    setSelection();
    repaint();
}

void YListBox::updateSelection(int apply) {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int s = i->getSelected();
            int t = (n >= beg) && (n <= end);

            if (fSelecting) {
                s &= 1;
                if (t)
                    s |= apply ? 1 : 2;
            } else {
                s &= 1;
                if (t) {
                    if (s & 1)
                        if (apply)
                            s = 0;
                        else
                            s |= 2;
                }
            }
            if (apply)
                s &= 1;
            i->setSelected(s);
        }
    }
}

void YListBox::setSelection() {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int t = (n >= beg) && (n <= end);
            i->setSelected(t);
        }
    }
}

int YListBox::itemCount() {
    YListItem *i;
    int n;
    
    for (i = fFirst, n = 0; i; i = i->getNext(), n++)
        ;
    return n;
}

YListItem *YListBox::item(int no) {
    YListItem *a = fFirst;
    int n;
    
    for (n = 0; a; a = a->getNext(), n++)
        if (n == no)
            return a;
    return 0;
}

void YListBox::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        int k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        int m = KEY_MODMASK(key.state);

        int SelPos, OldPos = fFocusedItem, count = itemCount();

        if (m & ShiftMask) {
            SelPos = fFocusedItem;
        } else {
            SelPos = -1;
        }
        
        switch (k) {
        case XK_Escape:
            windowList->frame()->wmHide();
            return;
        case XK_Return:
            {
                YListItem *i = item(fFocusedItem);
                YFrameWindow *f = 0;
                if (i)
                    f = i->frame();
                if (f) {
                    f->activate();
                    windowList->frame()->wmHide();
                }
            }
            break;
        case XK_Up:
            if (fFocusedItem > 0)
                fFocusedItem--;
            break;
        case XK_Down:
            if (fFocusedItem < count - 1)
                fFocusedItem++;
            break;
        case XK_Prior:
            //case XK_Page_Up:
            break;
        case XK_Next:
        //case XK_Page_Down:
            break;
        case XK_Home:
            fFocusedItem = 0;
            break;
        case XK_End:
            fFocusedItem = count - 1;
            break;
        case XK_F10:
            if (key.state & ShiftMask) {
                windowListPopup->popup(0, this,
                                       key.x_root, key.y_root, -1, -1,
                                       YPopupWindow::pfCanFlipVertical |
                                       YPopupWindow::pfCanFlipHorizontal |
                                       YPopupWindow::pfPopupMenu);
            }
            break;
        case XK_Delete:
            {
                handleCommand(cmdClose, 0);
            }
            break;
        }
        if (fFocusedItem != OldPos) {
            if (SelPos == -1) {
                fSelectStart = fFocusedItem;
                fSelectEnd = fFocusedItem;
                fSelecting = 1;
            } else {
                if (fSelectStart == OldPos)
                    fSelectStart = fFocusedItem;
                else if (fSelectEnd == OldPos)
                    fSelectEnd = fFocusedItem;
                else
                    fSelectStart = fSelectEnd = fFocusedItem;
                if (fSelectStart == -1)
                    fSelectStart = fSelectEnd;
                if (fSelectEnd == -1)
                    fSelectEnd = fSelectStart;

                fSelecting = 1;
            }
            setSelection();
            repaint();
        }
    }
}

void YListBox::handleButton(const XButtonEvent &button) {
    if (button.button == 1) {
        int no;
        YListItem *i = findItem(button.y, no);
        fFocusedItem = no;
        
        if (button.type == ButtonPress) {
            if (!(button.state & ControlMask))
                for (YListItem *j = fFirst; j; j = j->getNext())
                    j->setSelected(0);
            if (i) {
                fSelectStart = no;
                fSelectEnd = no;
                fFocusedItem = no;
                fSelecting = i->getSelected() ? 0 : 1;
                updateSelection(0);
                repaint();
            } else {
                fSelectStart = -1;
                fSelectEnd = -1;
                fSelecting = 1;
                repaint();
            }
        } else if (button.type == ButtonRelease) {
            if (no != -1 && no != fSelectEnd) {
                fSelectEnd = no;
                updateSelection(0);
            }
            updateSelection(1);
            fSelecting = -1;
            repaint();
        }
    }
    YWindow::handleButton(button);
}

void YListBox::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int count) {
    if (up.button == 3 && count == 1) {
        int no;
        YListItem *i = findItem(up.y, no);

        if (i) {
            if (i->getSelected() != 1) {
                fSelectStart = no;
                fSelectEnd = no;
                fSelecting = -1;
                fFocusedItem = no;
                setSelection();
                fSelectStart = -1;
                fSelectEnd = -1;
            } else {
                fFocusedItem = -1;
            }
            repaint();
            windowListPopup->popup(0, this,
                                   up.x_root, up.y_root, -1, -1,
                                   YPopupWindow::pfCanFlipVertical |
                                   YPopupWindow::pfCanFlipHorizontal |
                                   YPopupWindow::pfPopupMenu);
        }
    } else if (up.button == 1 && count == 2) {
        int no;
        YListItem *i = findItem(up.y, no);
        YFrameWindow *f = 0;
        if (i)
            f = i->frame();
        if (f) {
            f->activate();
            windowList->frame()->wmHide();
        }
    }
}

void YListBox::handleMotion(const XMotionEvent &motion) {
    if (motion.state & Button1Mask) {
        int no;

        findItem(motion.y, no);

        fFocusedItem = no;
        if (no != -1 && fSelectEnd != no) {
            if (fSelectStart == -1)
                fSelectStart = no;
            fSelectEnd = no;
            updateSelection(0);
            repaint();
        }
    }
    YWindow::handleMotion(motion);
}

void YListBox::removeItem(YListItem *item) {
    if (item->getPrev())
        item->getPrev()->setNext(item->getNext());
    else
        fFirst = item->getNext();

    if (item->getNext())
        item->getNext()->setPrev(item->getPrev());
    else
        fLast = item->getPrev();
    repaint();
}

void YListBox::scroll(YScrollBar *scroll, int delta) {
    if (scroll == fVerticalScroll) {
        fOffsetY += delta;
        repaint();
    }
}

void YListBox::move(YScrollBar *scroll, int pos) {
    if (scroll == fVerticalScroll) {
        fOffsetY = pos;
        repaint();
    }
}

void YListBox::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    int y = 0, x = 3;
    int lh, fh = windowListFont->height();
    //YFrameWindow *w = app->top(0);
    YListItem *a = fFirst;
    int n = 0;

    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;

    g.setColor(listBoxBg);
    g.fillRect(0, 0, width(), height());
    
    while (a) {
        if (a->frame() && a->frame()->client()) {
            int xpos = 0;
            unsigned char *title = a->frame()->client()->windowTitle();
            int yPos = y + lh - (lh - fh) / 2 - windowListFont->descent();
            
            YFrameClient *c = a->frame()->client();
            
            while (c->owner()) {
                xpos += 20;
                c = c->owner();
            }
            
            int s = a->getSelected();
            if (fSelecting == -1)
                s = (s == 1) ? 1 : 0;
            if (fSelecting == 0 && (s & 2))
                s = 0;

            if (s)
                g.setColor(listBoxSelBg);
            else
                g.setColor(listBoxBg);
            g.fillRect(0, y - fOffsetY, width(), lh);
            if (fFocusedItem == n) {
                g.setColor(black);
                g.drawRect(0, y - fOffsetY, width() - 1, lh - 1);
            }
            if (a->frame()->clientIcon() && a->frame()->clientIcon()->small())
                g.drawMaskPixmap(a->frame()->clientIcon()->small(),
                                 xpos + x, y - fOffsetY + 1);
            if (s)
                g.setColor(listBoxSelFg);
            else
                g.setColor(listBoxFg);
            g.setFont(windowListFont);
            if (title) {
                g.drawChars((char *)title, 0, strlen((char *)title),
                            xpos + x + 20, yPos - fOffsetY);
            }
        }
        y += lh;
        a = a->getNext();
        n++;
    }
    fVerticalScroll->setValues(fOffsetY, height(), 0, y);
}

void YListBox::setPopupActive(YPopupWindow */*popup*/) {
}

void YListBox::handleCommand(WMCommand command, void *context) {
    YListItem *i;

    switch (command) {
    case cmdCascade:
    case cmdTileVertical:
    case cmdTileHorizontal:
        break;
    default:
        if (fSelectStart != -1 && fSelectEnd != -1) {
            for (i = fFirst; i; i = i->getNext()) {
                if (i->getSelected()) {
                    if (command == cmdHide)
                        if (i->frame()->hidden())
                            continue;
                    i->frame()->handleCommand(command, context);
                }
            }
        }
    }
}

WindowList::WindowList(YWindow *aParent, Window win): YFrameClient(aParent, 0, win) {
    scrollVert = new YScrollBar(YScrollBar::Vertical, this);
    scrollVert->show();
    list = new YListBox(scrollVert, this);
    list->show();

    int w = app->root()->width();
    int h = app->root()->height();
    
    setGeometry(3 * w / 8, 2 * h / 6, w / 4, h / 3);

    windowList = this;
    setWindowTitle((unsigned char *)"Window List");
    setIconTitle((unsigned char *)"Window List");
}

WindowList::~WindowList() {
}

void WindowList::handleFocus(const XFocusChangeEvent &focus) {
    if (focus.type == FocusIn) {
        list->setFocus();
    } else if (focus.type == FocusOut) {
    }
}

void WindowList::relayout() {
    list->repaint();
}

YListItem *WindowList::addWindow(YFrameWindow *frame) {
    if (frame->client() == windowList)
        return 0;
    YListItem *item = new YListItem(frame);
    if (item)
        list->addItem(item);
    return item;
}

void WindowList::removeWindow(YListItem *item) {
    list->removeItem(item);
}

void WindowList::configure(int x, int y, unsigned int width, unsigned int height) {
    YFrameClient::configure(x, y, width, height);

    scrollVert->setGeometry(width - SCROLLBAR_SIZE, 0,
                            SCROLLBAR_SIZE, height);
    list->setGeometry(0, 0, width - SCROLLBAR_SIZE, height);
}

void WindowList::handleClose() {
    frame()->wmHide();
}

void WindowList::showFocused() {
    YFrameWindow *f = app->focus();

    if (f != frame()) {
        if (f)
            list->selectItem(f->winListItem());
        else
            list->selectItem(0);
    }
    setAllWorkspaces(1);
    setWorkspaces(ALLWORKSPACES);
    show();
    raise();
}
