/*
 * Copyright (C) 2000 Avaya Labs, Avaya Inc.
 * Copyright (C) 1999 Bell Labs, Lucent Technologies.
 * Copyright (C) Arash Baratloo, Timothy Tsai, and Navjot Singh.
 *
 * This file is part of the Libsafe library.
 * Libsafe version 2.x: protecting against stack smashing attacks.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, 
 *   visit http://www.research.avayalabs.com/project/libsafe/index.html
 *   or email libsafe@research.avayalabs.com
 */
/* ------------------------------------------------- */




#ifndef _LIBSAFE_DETECT_OVERFLOW
#define _LIBSAFE_DETECT_OVERFLOW
#include "util.h"
#endif

/*
 * Use _libsafe_level to determine when to bypass the libsafe code.  This is
 * needed because libsafe code might try to call *printf(), which would could
 * infinite recursion.  By setting _libsafe_level>0 and only calling libsafe
 * code when _libsafe_level==0, the recursion is avoided.
 * -- tkt
 */
static int _libsafe_level = 0;

/*
 * NOTE: This is NOT thread-safe!!!
 * -- tkt
 */
#define DEBUG(format, args...) \
    if (_libsafe_level == 0) { \
	_libsafe_level++; \
	printf(format, ##args); \
	_libsafe_level--; \
    }



/*
 * Since we don't have direct access to the _nl_current_LC_NUMERIC pointer
 * defined in libc.so, I call localeconv directly to get the values for the
 * decimal point and the thousands separator.  The first time that we need this
 * information, we call localeconv() and store the info in decimal_point and
 * thousands_sep.  Thereafter, we use the values directly.
 * -- tkt
 */

static int dolocaleconv = 0;	    // Have we called localeconv yet?
static struct lconv *conv;
static char *thousands;

/*****************************************************************************/
/*
 * The following are needed to get this file to compile.  This file and the
 * code I added below are taken from libc-2.1.3-91.src.rpm.
 * -- tkt
 */

#define USE_IN_LIBIO

#include <stdio.h>
#include <sys/types.h>

#define internal_function   /* empty */

/*
 * The original code did a check for the magic number.  I omit this check,
 * since I didn't want to deal with the incompatabilities between the new and
 * old stdio.
 * -- tkt
 */
#define	__validfp(stream)						      \
  (stream != NULL)

#if 1
# define COERCE_FILE(FILE) /* Nothing */
#else
/* This is part of the kludge for binary compatibility with old stdio. */
# define COERCE_FILE(FILE) \
  (((FILE)->_IO_file_flags & _IO_MAGIC_MASK) == _OLD_MAGIC_MASK \
    && (FILE) = *(FILE**)&((int*)fp)[1])
#endif
# define CHECK_FILE(FILE, RET) COERCE_FILE (FILE)

# define __set_errno(Val) errno = (Val)

#ifdef EINVAL
# define MAYBE_SET_EINVAL __set_errno (EINVAL)
#else
# define MAYBE_SET_EINVAL /* nothing */
#endif

#ifdef _G_USING_THUNKS
# define JUMP_FIELD(TYPE, NAME) TYPE NAME
# define JUMP0(FUNC, THIS) _IO_JUMPS_FUNC(THIS)->FUNC (THIS)
# define JUMP1(FUNC, THIS, X1) _IO_JUMPS_FUNC(THIS)->FUNC (THIS, X1)
# define JUMP2(FUNC, THIS, X1, X2) _IO_JUMPS_FUNC(THIS)->FUNC (THIS, X1, X2)
# define JUMP3(FUNC, THIS, X1,X2,X3) _IO_JUMPS_FUNC(THIS)->FUNC (THIS, X1,X2, X3)
# define JUMP_INIT(NAME, VALUE) VALUE
# define JUMP_INIT_DUMMY JUMP_INIT(dummy, 0), JUMP_INIT (dummy2, 0)
#else
/* These macros will change when we re-implement vtables to use "thunks"! */
# define JUMP_FIELD(TYPE, NAME) struct { short delta1, delta2; TYPE pfn; } NAME
# define JUMP0(FUNC, THIS) _IO_JUMPS_FUNC(THIS)->FUNC.pfn (THIS)
# define JUMP1(FUNC, THIS, X1) _IO_JUMPS_FUNC(THIS)->FUNC.pfn (THIS, X1)
# define JUMP2(FUNC, THIS, X1, X2) _IO_JUMPS_FUNC(THIS)->FUNC.pfn (THIS, X1, X2)
# define JUMP3(FUNC, THIS, X1,X2,X3) _IO_JUMPS_FUNC(THIS)->FUNC.pfn (THIS, X1,X2,X3)
# define JUMP_INIT(NAME, VALUE) {0, 0, VALUE}
# define JUMP_INIT_DUMMY JUMP_INIT(dummy, 0)
#endif

/* The 'finish' function does any final cleaning up of an _IO_FILE object.
   It does not delete (free) it, but does everything else to finalize it/
   It matches the streambuf::~streambuf virtual destructor.  */
typedef void (*_IO_finish_t) __PMT ((_IO_FILE *, int)); /* finalize */
#define _IO_FINISH(FP) JUMP1 (__finish, FP, 0)

/* The 'overflow' hook flushes the buffer.
   The second argument is a character, or EOF.
   It matches the streambuf::overflow virtual function. */
typedef int (*_IO_overflow_t) __PMT ((_IO_FILE *, int));
#define _IO_OVERFLOW(FP, CH) JUMP1 (__overflow, FP, CH)

/* The 'underflow' hook tries to fills the get buffer.
   It returns the next character (as an unsigned char) or EOF.  The next
   character remains in the get buffer, and the get position is not changed.
   It matches the streambuf::underflow virtual function. */
typedef int (*_IO_underflow_t) __PMT ((_IO_FILE *));
#define _IO_UNDERFLOW(FP) JUMP0 (__underflow, FP)

/* The 'pbackfail' hook handles backing up.
   It matches the streambuf::pbackfail virtual function. */
typedef int (*_IO_pbackfail_t) __PMT ((_IO_FILE *, int));
#define _IO_PBACKFAIL(FP, CH) JUMP1 (__pbackfail, FP, CH)

/* The 'xsputn' hook writes upto N characters from buffer DATA.
   Returns the number of character actually written.
   It matches the streambuf::xsputn virtual function. */
typedef _IO_size_t (*_IO_xsputn_t) __PMT ((_IO_FILE *FP, const void *DATA,
					   _IO_size_t N));
#define _IO_XSPUTN(FP, DATA, N) JUMP2 (__xsputn, FP, DATA, N)
#define _IO_sputn(__fp, __s, __n) _IO_XSPUTN (__fp, __s, __n)

/* The 'xsgetn' hook reads upto N characters into buffer DATA.
   Returns the number of character actually read.
   It matches the streambuf::xsgetn virtual function. */
typedef _IO_size_t (*_IO_xsgetn_t) __PMT ((_IO_FILE *FP, void *DATA,
					   _IO_size_t N));
#define _IO_XSGETN(FP, DATA, N) JUMP2 (__xsgetn, FP, DATA, N)

/*
 * The following definition of _IO_fpos64_t is for GLIBC2.1.  GLIBC2.2 defines
 * _IO_fpos64_t as a struct, but that is incompatible with the vfprintf code
 * from libc-2.1.3-91.src.rpm.  So we use the GLIBC2.1 definition and make sure
 * that all references to our definition are local to libsafe.
 * -- tkt
 */
#ifdef _G_fpos64_t
#undef _G_fpos64_t
#define _G_fpos64_t _G_off64_t
#else
#define _G_fpos64_t _G_off64_t
#endif

/* The 'seekoff' hook moves the stream position to a new position
   relative to the start of the file (if DIR==0), the current position
   (MODE==1), or the end of the file (MODE==2).
   It matches the streambuf::seekoff virtual function.
   It is also used for the ANSI fseek function. */
typedef _IO_fpos64_t (*_IO_seekoff_t) __PMT ((_IO_FILE *FP, _IO_off64_t OFF,
					      int DIR, int MODE));
#define _IO_SEEKOFF(FP, OFF, DIR, MODE) JUMP3 (__seekoff, FP, OFF, DIR, MODE)

/* The 'seekpos' hook also moves the stream position,
   but to an absolute position given by a fpos64_t (seekpos).
   It matches the streambuf::seekpos virtual function.
   It is also used for the ANSI fgetpos and fsetpos functions.  */
/* The _IO_seek_cur and _IO_seek_end options are not allowed. */
typedef _IO_fpos64_t (*_IO_seekpos_t) __PMT ((_IO_FILE *, _IO_fpos64_t, int));
#define _IO_SEEKPOS(FP, POS, FLAGS) JUMP2 (__seekpos, FP, POS, FLAGS)

/* The 'setbuf' hook gives a buffer to the file.
   It matches the streambuf::setbuf virtual function. */
typedef _IO_FILE* (*_IO_setbuf_t) __PMT ((_IO_FILE *, char *, _IO_ssize_t));
#define _IO_SETBUF(FP, BUFFER, LENGTH) JUMP2 (__setbuf, FP, BUFFER, LENGTH)

/* The 'sync' hook attempts to synchronize the internal data structures
   of the file with the external state.
   It matches the streambuf::sync virtual function. */
typedef int (*_IO_sync_t) __PMT ((_IO_FILE *));
#define _IO_SYNC(FP) JUMP0 (__sync, FP)

/* The 'doallocate' hook is used to tell the file to allocate a buffer.
   It matches the streambuf::doallocate virtual function, which is not
   in the ANSI/ISO C++ standard, but is part traditional implementations. */
typedef int (*_IO_doallocate_t) __PMT ((_IO_FILE *));
#define _IO_DOALLOCATE(FP) JUMP0 (__doallocate, FP)

/* The following four hooks (sysread, syswrite, sysclose, sysseek, and
   sysstat) are low-level hooks specific to this implementation.
   There is no correspondence in the ANSI/ISO C++ standard library.
   The hooks basically correspond to the Unix system functions
   (read, write, close, lseek, and stat) except that a _IO_FILE*
   parameter is used instead of a integer file descriptor;  the default
   implementation used for normal files just calls those functions.
   The advantage of overriding these functions instead of the higher-level
   ones (underflow, overflow etc) is that you can leave all the buffering
   higher-level functions.  */

/* The 'sysread' hook is used to read data from the external file into
   an existing buffer.  It generalizes the Unix read(2) function.
   It matches the streambuf::sys_read virtual function, which is
   specific to this implementation. */
typedef _IO_ssize_t (*_IO_read_t) __PMT ((_IO_FILE *, void *, _IO_ssize_t));
#define _IO_SYSREAD(FP, DATA, LEN) JUMP2 (__read, FP, DATA, LEN)

/* The 'syswrite' hook is used to write data from an existing buffer
   to an external file.  It generalizes the Unix write(2) function.
   It matches the streambuf::sys_write virtual function, which is
   specific to this implementation. */
typedef _IO_ssize_t (*_IO_write_t) __PMT ((_IO_FILE *, const void *,
					   _IO_ssize_t));
#define _IO_SYSWRITE(FP, DATA, LEN) JUMP2 (__write, FP, DATA, LEN)

/* The 'sysseek' hook is used to re-position an external file.
   It generalizes the Unix lseek(2) function.
   It matches the streambuf::sys_seek virtual function, which is
   specific to this implementation. */
typedef _IO_fpos64_t (*_IO_seek_t) __PMT ((_IO_FILE *, _IO_off64_t, int));
#define _IO_SYSSEEK(FP, OFFSET, MODE) JUMP2 (__seek, FP, OFFSET, MODE)

/* The 'sysclose' hook is used to finalize (close, finish up) an
   external file.  It generalizes the Unix close(2) function.
   It matches the streambuf::sys_close virtual function, which is
   specific to this implementation. */
typedef int (*_IO_close_t) __PMT ((_IO_FILE *)); /* finalize */
#define _IO_SYSCLOSE(FP) JUMP0 (__close, FP)

/* The 'sysstat' hook is used to get information about an external file
   into a struct stat buffer.  It generalizes the Unix fstat(2) call.
   It matches the streambuf::sys_stat virtual function, which is
   specific to this implementation. */
typedef int (*_IO_stat_t) __PMT ((_IO_FILE *, void *));
#define _IO_SYSSTAT(FP, BUF) JUMP1 (__stat, FP, BUF)

/* The 'showmany' hook can be used to get an image how much input is
   available.  In many cases the answer will be 0 which means unknown
   but some cases one can provide real information.  */
typedef int (*_IO_showmanyc_t) __PMT ((_IO_FILE *));
#define _IO_SHOWMANYC(FP) JUMP0 (__showmanyc, FP)

/* The 'imbue' hook is used to get information about the currently
   installed locales.  */
typedef void (*_IO_imbue_t) __PMT ((_IO_FILE *, void *));
#define _IO_IMBUE(FP, LOCALE) JUMP1 (__imbue, FP, LOCALE)

struct _IO_jump_t
{
    JUMP_FIELD(_G_size_t, __dummy);
#ifdef _G_USING_THUNKS
    JUMP_FIELD(_G_size_t, __dummy2);
#endif
    JUMP_FIELD(_IO_finish_t, __finish);
    JUMP_FIELD(_IO_overflow_t, __overflow);
    JUMP_FIELD(_IO_underflow_t, __underflow);
    JUMP_FIELD(_IO_underflow_t, __uflow);
    JUMP_FIELD(_IO_pbackfail_t, __pbackfail);
    /* showmany */
    JUMP_FIELD(_IO_xsputn_t, __xsputn);
    JUMP_FIELD(_IO_xsgetn_t, __xsgetn);
    JUMP_FIELD(_IO_seekoff_t, __seekoff);
    JUMP_FIELD(_IO_seekpos_t, __seekpos);
    JUMP_FIELD(_IO_setbuf_t, __setbuf);
    JUMP_FIELD(_IO_sync_t, __sync);
    JUMP_FIELD(_IO_doallocate_t, __doallocate);
    JUMP_FIELD(_IO_read_t, __read);
    JUMP_FIELD(_IO_write_t, __write);
    JUMP_FIELD(_IO_seek_t, __seek);
    JUMP_FIELD(_IO_close_t, __close);
    JUMP_FIELD(_IO_stat_t, __stat);
    JUMP_FIELD(_IO_showmanyc_t, __showmanyc);
    JUMP_FIELD(_IO_imbue_t, __imbue);
#if 0
    get_column;
    set_column;
#endif
};

/* We always allocate an extra word following an _IO_FILE.
   This contains a pointer to the function jump table used.
   This is for compatibility with C++ streambuf; the word can
   be used to smash to a pointer to a virtual function table. */
struct _IO_FILE_plus
{
  _IO_FILE file;
  const struct _IO_jump_t *vtable;
};

#define _IO_JUMPS(THIS) ((struct _IO_FILE_plus *) (THIS))->vtable
#if _IO_JUMPS_OFFSET
# define _IO_JUMPS_FUNC(THIS) \
 (*(struct _IO_jump_t **) ((void *) &((struct _IO_FILE_plus *) (THIS))->vtable\
			   + (THIS)->_vtable_offset))
#else
# define _IO_JUMPS_FUNC(THIS) _IO_JUMPS(THIS)
#endif

#include <wchar.h>
extern size_t __mbrtowc __P ((wchar_t *__restrict __pwc,
			      __const char *__restrict __s, size_t __n,
			      mbstate_t *__restrict __p));

/*
 * __mbrtowc() is defined as a local function in libc.so, and since we don't
 * have access to it here, we just use the mbrtowc alias instead.
 */
#define __mbrtowc mbrtowc


#include <wchar.h>
size_t __wcrtomb (char *s, wchar_t wc, mbstate_t *ps);
size_t __wcsrtombs (char *dst, const wchar_t **src, size_t len, mbstate_t *ps);




/*------------------------------------------------------------------*/
/*
 * _itoa() code
 */

/*-------------------------------------------------------------*/
/* 
 * From sysdep/i386/gmp-mparam.h
 */

/* gmp-mparam.h -- Compiler/machine parameter header file.
 *
 * Copyright (C) 1991, 1993, 1994, 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU MP Library.
 *
 * The GNU MP Library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * The GNU MP Library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with the GNU MP Library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA. */

#if defined(i386)

#define BITS_PER_MP_LIMB 32
#define BYTES_PER_MP_LIMB 4
#define BITS_PER_LONGINT 32
#define BITS_PER_INT 32
#define BITS_PER_SHORTINT 16
#define BITS_PER_CHAR 8

#define IEEE_DOUBLE_BIG_ENDIAN 0

#endif

typedef unsigned int          mp_limb_t;

#define W_TYPE_SIZE BITS_PER_MP_LIMB

#if defined (__GNUC__)
/* Define stuff for longlong.h.  */
typedef unsigned int UQItype    __attribute__ ((mode (QI)));
typedef          int SItype     __attribute__ ((mode (SI)));
typedef unsigned int USItype    __attribute__ ((mode (SI)));
typedef          int DItype     __attribute__ ((mode (DI)));
typedef unsigned int UDItype    __attribute__ ((mode (DI)));
#else
typedef unsigned char UQItype;
typedef          long SItype;
typedef unsigned long USItype;
#endif

/*-------------------------------------------------------------*/
/* 
 * From stdlib/longlong.h
 */

#if (defined (__i386__) || defined (__i486__)) && W_TYPE_SIZE == 32
#define add_ssaaaa(sh, sl, ah, al, bh, bl) \
  __asm__ ("addl %5,%1
	adcl %3,%0"							\
	   : "=r" ((USItype)(sh)),					\
	     "=&r" ((USItype)(sl))					\
	   : "%0" ((USItype)(ah)),					\
	     "g" ((USItype)(bh)),					\
	     "%1" ((USItype)(al)),					\
	     "g" ((USItype)(bl)))
#define sub_ddmmss(sh, sl, ah, al, bh, bl) \
  __asm__ ("subl %5,%1
	sbbl %3,%0"							\
	   : "=r" ((USItype)(sh)),					\
	     "=&r" ((USItype)(sl))					\
	   : "0" ((USItype)(ah)),					\
	     "g" ((USItype)(bh)),					\
	     "1" ((USItype)(al)),					\
	     "g" ((USItype)(bl)))
#define umul_ppmm(w1, w0, u, v) \
  __asm__ ("mull %3"							\
	   : "=a" ((USItype)(w0)),					\
	     "=d" ((USItype)(w1))					\
	   : "%0" ((USItype)(u)),					\
	     "rm" ((USItype)(v)))
#define udiv_qrnnd(q, r, n1, n0, d) \
  __asm__ ("divl %4"							\
	   : "=a" ((USItype)(q)),					\
	     "=d" ((USItype)(r))					\
	   : "0" ((USItype)(n0)),					\
	     "1" ((USItype)(n1)),					\
	     "rm" ((USItype)(d)))
#define count_leading_zeros(count, x) \
  do {									\
    USItype __cbtmp;							\
    __asm__ ("bsrl %1,%0"						\
	     : "=r" (__cbtmp) : "rm" ((USItype)(x)));			\
    (count) = __cbtmp ^ 31;						\
  } while (0)
#define count_trailing_zeros(count, x) \
  __asm__ ("bsfl %1,%0" : "=r" (count) : "rm" ((USItype)(x)))
#ifndef UMUL_TIME
#define UMUL_TIME 40
#endif
#ifndef UDIV_TIME
#define UDIV_TIME 40
#endif
#endif /* 80x86 */

/*-------------------------------------------------------------*/

/* Control memory layout.  */
#ifdef PACK
# undef PACK
# define PACK __attribute__ ((packed))
#else
# define PACK
#endif

/* Declare local types.  */
struct base_table_t
{
#if (UDIV_TIME > 2 * UMUL_TIME)
  mp_limb_t base_multiplier;
#endif
  char flag;
  char post_shift;
#if BITS_PER_MP_LIMB == 32
  struct
    {
      char normalization_steps;
      char ndigits;
      mp_limb_t base PACK;
#if UDIV_TIME > 2 * UMUL_TIME
      mp_limb_t base_ninv PACK;
#endif
    } big;
#endif
};

/* To reduce the memory needed we include some fields of the tables
   only conditionally.  */
#if UDIV_TIME > 2 * UMUL_TIME
# define SEL1(X) X,
# define SEL2(X) ,X
#else
# define SEL1(X)
# define SEL2(X)
#endif

/* Local variables.  */
static const struct base_table_t base_table[] =
{
#if BITS_PER_MP_LIMB == 64
  /*  2 */ {SEL1(0ul) 1, 1},
  /*  3 */ {SEL1(0xaaaaaaaaaaaaaaabul) 0, 1},
  /*  4 */ {SEL1(0ul) 1, 2},
  /*  5 */ {SEL1(0xcccccccccccccccdul) 0, 2},
  /*  6 */ {SEL1(0xaaaaaaaaaaaaaaabul) 0, 2},
  /*  7 */ {SEL1(0x2492492492492493ul) 1, 3},
  /*  8 */ {SEL1(0ul) 1, 3},
  /*  9 */ {SEL1(0xe38e38e38e38e38ful) 0, 3},
  /* 10 */ {SEL1(0xcccccccccccccccdul) 0, 3},
  /* 11 */ {SEL1(0x2e8ba2e8ba2e8ba3ul) 0, 1},
  /* 12 */ {SEL1(0xaaaaaaaaaaaaaaabul) 0, 3},
  /* 13 */ {SEL1(0x4ec4ec4ec4ec4ec5ul) 0, 2},
  /* 14 */ {SEL1(0x2492492492492493ul) 1, 4},
  /* 15 */ {SEL1(0x8888888888888889ul) 0, 3},
  /* 16 */ {SEL1(0ul) 1, 4},
  /* 17 */ {SEL1(0xf0f0f0f0f0f0f0f1ul) 0, 4},
  /* 18 */ {SEL1(0xe38e38e38e38e38ful) 0, 4},
  /* 19 */ {SEL1(0xd79435e50d79435ful) 0, 4},
  /* 20 */ {SEL1(0xcccccccccccccccdul) 0, 4},
  /* 21 */ {SEL1(0x8618618618618619ul) 1, 5},
  /* 22 */ {SEL1(0x2e8ba2e8ba2e8ba3ul) 0, 2},
  /* 23 */ {SEL1(0x642c8590b21642c9ul) 1, 5},
  /* 24 */ {SEL1(0xaaaaaaaaaaaaaaabul) 0, 4},
  /* 25 */ {SEL1(0x47ae147ae147ae15ul) 1, 5},
  /* 26 */ {SEL1(0x4ec4ec4ec4ec4ec5ul) 0, 3},
  /* 27 */ {SEL1(0x97b425ed097b425ful) 0, 4},
  /* 28 */ {SEL1(0x2492492492492493ul) 1, 5},
  /* 29 */ {SEL1(0x1a7b9611a7b9611bul) 1, 5},
  /* 30 */ {SEL1(0x8888888888888889ul) 0, 4},
  /* 31 */ {SEL1(0x0842108421084211ul) 1, 5},
  /* 32 */ {SEL1(0ul) 1, 5},
  /* 33 */ {SEL1(0x0f83e0f83e0f83e1ul) 0, 1},
  /* 34 */ {SEL1(0xf0f0f0f0f0f0f0f1ul) 0, 5},
  /* 35 */ {SEL1(0xea0ea0ea0ea0ea0ful) 0, 5},
  /* 36 */ {SEL1(0xe38e38e38e38e38ful) 0, 5}
#endif
#if BITS_PER_MP_LIMB == 32
  /*  2 */ {SEL1(0ul) 1, 1, {0, 31, 0x80000000ul SEL2(0xfffffffful)}},
  /*  3 */ {SEL1(0xaaaaaaabul) 0, 1, {0, 20, 0xcfd41b91ul SEL2(0x3b563c24ul)}},
  /*  4 */ {SEL1(0ul) 1, 2, {1, 15, 0x40000000ul SEL2(0xfffffffful)}},
  /*  5 */ {SEL1(0xcccccccdul) 0, 2, {1, 13, 0x48c27395ul SEL2(0xc25c2684ul)}},
  /*  6 */ {SEL1(0xaaaaaaabul) 0, 2, {0, 12, 0x81bf1000ul SEL2(0xf91bd1b6ul)}},
  /*  7 */ {SEL1(0x24924925ul) 1, 3, {1, 11, 0x75db9c97ul SEL2(0x1607a2cbul)}},
  /*  8 */ {SEL1(0ul) 1, 3, {1, 10, 0x40000000ul SEL2(0xfffffffful)}},
  /*  9 */ {SEL1(0x38e38e39ul) 0, 1, {0, 10, 0xcfd41b91ul SEL2(0x3b563c24ul)}},
  /* 10 */ {SEL1(0xcccccccdul) 0, 3, {2, 9, 0x3b9aca00ul SEL2(0x12e0be82ul)}},
  /* 11 */ {SEL1(0xba2e8ba3ul) 0, 3, {0, 9, 0x8c8b6d2bul SEL2(0xd24cde04ul)}},
  /* 12 */ {SEL1(0xaaaaaaabul) 0, 3, {3, 8, 0x19a10000ul SEL2(0x3fa39ab5ul)}},
  /* 13 */ {SEL1(0x4ec4ec4ful) 0, 2, {2, 8, 0x309f1021ul SEL2(0x50f8ac5ful)}},
  /* 14 */ {SEL1(0x24924925ul) 1, 4, {1, 8, 0x57f6c100ul SEL2(0x74843b1eul)}},
  /* 15 */ {SEL1(0x88888889ul) 0, 3, {0, 8, 0x98c29b81ul SEL2(0xad0326c2ul)}},
  /* 16 */ {SEL1(0ul) 1, 4, {3, 7, 0x10000000ul SEL2(0xfffffffful)}},
  /* 17 */ {SEL1(0xf0f0f0f1ul) 0, 4, {3, 7, 0x18754571ul SEL2(0x4ef0b6bdul)}},
  /* 18 */ {SEL1(0x38e38e39ul) 0, 2, {2, 7, 0x247dbc80ul SEL2(0xc0fc48a1ul)}},
  /* 19 */ {SEL1(0xaf286bcbul) 1, 5, {2, 7, 0x3547667bul SEL2(0x33838942ul)}},
  /* 20 */ {SEL1(0xcccccccdul) 0, 4, {1, 7, 0x4c4b4000ul SEL2(0xad7f29abul)}},
  /* 21 */ {SEL1(0x86186187ul) 1, 5, {1, 7, 0x6b5a6e1dul SEL2(0x313c3d15ul)}},
  /* 22 */ {SEL1(0xba2e8ba3ul) 0, 4, {0, 7, 0x94ace180ul SEL2(0xb8cca9e0ul)}},
  /* 23 */ {SEL1(0xb21642c9ul) 0, 4, {0, 7, 0xcaf18367ul SEL2(0x42ed6de9ul)}},
  /* 24 */ {SEL1(0xaaaaaaabul) 0, 4, {4, 6, 0x0b640000ul SEL2(0x67980e0bul)}},
  /* 25 */ {SEL1(0x51eb851ful) 0, 3, {4, 6, 0x0e8d4a51ul SEL2(0x19799812ul)}},
  /* 26 */ {SEL1(0x4ec4ec4ful) 0, 3, {3, 6, 0x1269ae40ul SEL2(0xbce85396ul)}},
  /* 27 */ {SEL1(0x2f684bdbul) 1, 5, {3, 6, 0x17179149ul SEL2(0x62c103a9ul)}},
  /* 28 */ {SEL1(0x24924925ul) 1, 5, {3, 6, 0x1cb91000ul SEL2(0x1d353d43ul)}},
  /* 29 */ {SEL1(0x8d3dcb09ul) 0, 4, {2, 6, 0x23744899ul SEL2(0xce1deceaul)}},
  /* 30 */ {SEL1(0x88888889ul) 0, 4, {2, 6, 0x2b73a840ul SEL2(0x790fc511ul)}},
  /* 31 */ {SEL1(0x08421085ul) 1, 5, {2, 6, 0x34e63b41ul SEL2(0x35b865a0ul)}},
  /* 32 */ {SEL1(0ul) 1, 5, {1, 6, 0x40000000ul SEL2(0xfffffffful)}},
  /* 33 */ {SEL1(0x3e0f83e1ul) 0, 3, {1, 6, 0x4cfa3cc1ul SEL2(0xa9aed1b3ul)}},
  /* 34 */ {SEL1(0xf0f0f0f1ul) 0, 5, {1, 6, 0x5c13d840ul SEL2(0x63dfc229ul)}},
  /* 35 */ {SEL1(0xd41d41d5ul) 1, 6, {1, 6, 0x6d91b519ul SEL2(0x2b0fee30ul)}},
  /* 36 */ {SEL1(0x38e38e39ul) 0, 3, {0, 6, 0x81bf1000ul SEL2(0xf91bd1b6ul)}}
#endif
};

/* Lower-case digits.  */
extern const char _itoa_lower_digits[];
/* Upper-case digits.  */
extern const char _itoa_upper_digits[];

char *
_itoa (value, buflim, base, upper_case)
     unsigned long long int value;
     char *buflim;
     unsigned int base;
     int upper_case;
{
  const char *digits = upper_case ? _itoa_upper_digits : _itoa_lower_digits;
  char *bp = buflim;
  const struct base_table_t *brec = &base_table[base - 2];

  switch (base)
    {
#define RUN_2N(BITS) \
      do								      \
        {								      \
	  /* `unsigned long long int' always has 64 bits.  */		      \
	  mp_limb_t work_hi = value >> (64 - BITS_PER_MP_LIMB);		      \
									      \
	  if (BITS_PER_MP_LIMB == 32)					      \
	    {								      \
	      if (work_hi != 0)						      \
		{							      \
		  mp_limb_t work_lo;					      \
		  int cnt;						      \
									      \
		  work_lo = value & 0xfffffffful;			      \
		  for (cnt = BITS_PER_MP_LIMB / BITS; cnt > 0; --cnt)	      \
		    {							      \
		      *--bp = digits[work_lo & ((1ul << BITS) - 1)];	      \
		      work_lo >>= BITS;					      \
		    }							      \
		  if (BITS_PER_MP_LIMB % BITS != 0)			      \
		    {							      \
		      work_lo						      \
			|= ((work_hi					      \
			     & ((1 << (BITS - BITS_PER_MP_LIMB%BITS))	      \
				- 1))					      \
			    << BITS_PER_MP_LIMB % BITS);		      \
		      work_hi >>= BITS - BITS_PER_MP_LIMB % BITS;	      \
		      if (work_hi == 0)					      \
			work_hi = work_lo;				      \
		      else						      \
			*--bp = digits[work_lo];			      \
		    }							      \
		}							      \
	      else							      \
		work_hi = value & 0xfffffffful;				      \
	    }								      \
	  do								      \
	    {								      \
	      *--bp = digits[work_hi & ((1 << BITS) - 1)];		      \
	      work_hi >>= BITS;						      \
	    }								      \
	  while (work_hi != 0);						      \
	}								      \
      while (0)
    case 8:
      RUN_2N (3);
      break;

    case 16:
      RUN_2N (4);
      break;

    default:
      {
#if BITS_PER_MP_LIMB == 64
	mp_limb_t base_multiplier = brec->base_multiplier;
	if (brec->flag)
	  while (value != 0)
	    {
	      mp_limb_t quo, rem, x, dummy;

	      umul_ppmm (x, dummy, value, base_multiplier);
	      quo = (x + ((value - x) >> 1)) >> (brec->post_shift - 1);
	      rem = value - quo * base;
	      *--bp = digits[rem];
	      value = quo;
	    }
	else
	  while (value != 0)
	    {
	      mp_limb_t quo, rem, x, dummy;

	      umul_ppmm (x, dummy, value, base_multiplier);
	      quo = x >> brec->post_shift;
	      rem = value - quo * base;
	      *--bp = digits[rem];
	      value = quo;
	    }
#endif
#if BITS_PER_MP_LIMB == 32
	mp_limb_t t[3];
	int n;

	/* First convert x0 to 1-3 words in base s->big.base.
	   Optimize for frequent cases of 32 bit numbers.  */
	if ((mp_limb_t) (value >> 32) >= 1)
	  {
#if UDIV_TIME > 2 * UMUL_TIME || UDIV_NEEDS_NORMALIZATION
	    int big_normalization_steps = brec->big.normalization_steps;
	    mp_limb_t big_base_norm
	      = brec->big.base << big_normalization_steps;
#endif
	    if ((mp_limb_t) (value >> 32) >= brec->big.base)
	      {
		mp_limb_t x1hi, x1lo, r;
		/* If you want to optimize this, take advantage of
		   that the quotient in the first udiv_qrnnd will
		   always be very small.  It might be faster just to
		   subtract in a tight loop.  */

#if UDIV_TIME > 2 * UMUL_TIME
		mp_limb_t x, xh, xl;

		if (big_normalization_steps == 0)
		  xh = 0;
		else
		  xh = (mp_limb_t) (value >> (64 - big_normalization_steps));
		xl = (mp_limb_t) (value >> (32 - big_normalization_steps));
		udiv_qrnnd_preinv (x1hi, r, xh, xl, big_base_norm,
				   brec->big.base_ninv);

		xl = ((mp_limb_t) value) << big_normalization_steps;
		udiv_qrnnd_preinv (x1lo, x, r, xl, big_base_norm,
				   brec->big.base_ninv);
		t[2] = x >> big_normalization_steps;

		if (big_normalization_steps == 0)
		  xh = x1hi;
		else
		  xh = ((x1hi << big_normalization_steps)
			| (x1lo >> (32 - big_normalization_steps)));
		xl = x1lo << big_normalization_steps;
		udiv_qrnnd_preinv (t[0], x, xh, xl, big_base_norm,
				   brec->big.base_ninv);
		t[1] = x >> big_normalization_steps;
#elif UDIV_NEEDS_NORMALIZATION
		mp_limb_t x, xh, xl;

		if (big_normalization_steps == 0)
		  xh = 0;
		else
		  xh = (mp_limb_t) (value >> 64 - big_normalization_steps);
		xl = (mp_limb_t) (value >> 32 - big_normalization_steps);
		udiv_qrnnd (x1hi, r, xh, xl, big_base_norm);

		xl = ((mp_limb_t) value) << big_normalization_steps;
		udiv_qrnnd (x1lo, x, r, xl, big_base_norm);
		t[2] = x >> big_normalization_steps;

		if (big_normalization_steps == 0)
		  xh = x1hi;
		else
		  xh = ((x1hi << big_normalization_steps)
			| (x1lo >> 32 - big_normalization_steps));
		xl = x1lo << big_normalization_steps;
		udiv_qrnnd (t[0], x, xh, xl, big_base_norm);
		t[1] = x >> big_normalization_steps;
#else
		udiv_qrnnd (x1hi, r, 0, (mp_limb_t) (value >> 32),
			    brec->big.base);
		udiv_qrnnd (x1lo, t[2], r, (mp_limb_t) value, brec->big.base);
		udiv_qrnnd (t[0], t[1], x1hi, x1lo, brec->big.base);
#endif
		n = 3;
	      }
	    else
	      {
#if (UDIV_TIME > 2 * UMUL_TIME)
		mp_limb_t x;

		value <<= brec->big.normalization_steps;
		udiv_qrnnd_preinv (t[0], x, (mp_limb_t) (value >> 32),
				   (mp_limb_t) value, big_base_norm,
				   brec->big.base_ninv);
		t[1] = x >> brec->big.normalization_steps;
#elif UDIV_NEEDS_NORMALIZATION
		mp_limb_t x;

		value <<= big_normalization_steps;
		udiv_qrnnd (t[0], x, (mp_limb_t) (value >> 32),
			    (mp_limb_t) value, big_base_norm);
		t[1] = x >> big_normalization_steps;
#else
		udiv_qrnnd (t[0], t[1], (mp_limb_t) (value >> 32),
			    (mp_limb_t) value, brec->big.base);
#endif
		n = 2;
	      }
	  }
	else
	  {
	    t[0] = value;
	    n = 1;
	  }

	/* Convert the 1-3 words in t[], word by word, to ASCII.  */
	do
	  {
	    mp_limb_t ti = t[--n];
	    int ndig_for_this_limb = 0;

#if UDIV_TIME > 2 * UMUL_TIME
	    mp_limb_t base_multiplier = brec->base_multiplier;
	    if (brec->flag)
	      while (ti != 0)
		{
		  mp_limb_t quo, rem, x, dummy;

		  umul_ppmm (x, dummy, ti, base_multiplier);
		  quo = (x + ((ti - x) >> 1)) >> (brec->post_shift - 1);
		  rem = ti - quo * base;
		  *--bp = digits[rem];
		  ti = quo;
		  ++ndig_for_this_limb;
		}
	    else
	      while (ti != 0)
		{
		  mp_limb_t quo, rem, x, dummy;

		  umul_ppmm (x, dummy, ti, base_multiplier);
		  quo = x >> brec->post_shift;
		  rem = ti - quo * base;
		  *--bp = digits[rem];
		  ti = quo;
		  ++ndig_for_this_limb;
		}
#else
	    while (ti != 0)
	      {
		mp_limb_t quo, rem;

		quo = ti / base;
		rem = ti % base;
		*--bp = digits[rem];
		ti = quo;
		++ndig_for_this_limb;
	      }
#endif
	    /* If this wasn't the most significant word, pad with zeros.  */
	    if (n != 0)
	      while (ndig_for_this_limb < brec->big.ndigits)
		{
		  *--bp = '0';
		  ++ndig_for_this_limb;
		}
	  }
	while (n != 0);
#endif
      }
      break;
    }

  return bp;
}


static inline char * __attribute__ ((unused))
_itoa_word (unsigned long value, char *buflim,
	    unsigned int base, int upper_case)
{
  extern const char _itoa_upper_digits[], _itoa_lower_digits[];
  const char *digits = upper_case ? _itoa_upper_digits : _itoa_lower_digits;
  char *bp = buflim;

  switch (base)
    {
#define SPECIAL(Base)                                                         \
    case Base:                                                                \
      do                                                                      \
	*--bp = digits[value % Base];                                         \
      while ((value /= Base) != 0);                                           \
      break

      SPECIAL (10);
      SPECIAL (16);
      SPECIAL (8);
    default:
      do
	*--bp = digits[value % base];
      while ((value /= base) != 0);
    }
  return bp;
}

/*------------------------------------------------------------------*/

/*
 * The following are defined as static just for libsafe:
 *	_IO_default_seekoff
 *	_IO_default_seekpos
 *	_IO_default_seek
 * -- tkt
 */

extern void _IO_default_finish __P ((_IO_FILE *, int));

extern int _IO_default_underflow __P ((_IO_FILE *));
extern int _IO_default_uflow __P ((_IO_FILE *));
extern int _IO_default_pbackfail __P ((_IO_FILE *, int)); 
extern _IO_size_t _IO_default_xsputn __P ((_IO_FILE *, const void *,
	                                               _IO_size_t));
extern _IO_size_t _IO_default_xsgetn __P ((_IO_FILE *, void *, _IO_size_t));
static _IO_fpos64_t _IO_default_seekoff __P ((_IO_FILE *,
	                                                  _IO_off64_t, int,
							  int)); 
static _IO_fpos64_t _IO_default_seekpos __P ((_IO_FILE *,
	                                                  _IO_fpos64_t, int));
extern _IO_FILE* _IO_default_setbuf __P ((_IO_FILE *, char *, _IO_ssize_t));
extern int _IO_default_sync __P ((_IO_FILE *));
extern int _IO_default_doallocate __P ((_IO_FILE *));
extern _IO_ssize_t _IO_default_write __P ((_IO_FILE *, const void *,
	                                               _IO_ssize_t));
extern _IO_ssize_t _IO_default_read __P ((_IO_FILE *, void *, _IO_ssize_t));
extern int _IO_default_stat __P ((_IO_FILE *, void *));
static _IO_fpos64_t _IO_default_seek __P ((_IO_FILE *, _IO_off64_t, int));
#define _IO_default_close ((_IO_close_t) _IO_default_sync)

int
_IO_default_underflow (fp)
     _IO_FILE *fp;
{
  return EOF;
}

#ifndef _IO_pos_BAD
#define _IO_pos_BAD ((_IO_fpos64_t)(-1))
#endif

/*
 * Added static.
 * -- tkt
 */
static _IO_fpos64_t
_IO_default_seekoff (fp, offset, dir, mode)
     _IO_FILE *fp;
     _IO_off64_t offset;
     int dir;
     int mode;
{
    return _IO_pos_BAD;
}

/* _IO_pos_as_off converts an _IO_fpos64_t value to an _IO_off64_t value. */
#ifndef _IO_pos_as_off
# define _IO_pos_as_off(__pos) ((_IO_off64_t) (__pos))
#endif

/*
 * Added static.
 * -- tkt
 */
static _IO_fpos64_t
_IO_default_seekpos (fp, pos, mode)
     _IO_FILE *fp;
     _IO_fpos64_t pos;
     int mode;
{
  return (_IO_fpos64_t)_IO_SEEKOFF (fp, _IO_pos_as_off (pos), 0, mode);
}

#include <sys/mman.h>
# define ROUND_TO_PAGE(_S) \
       (((_S) + EXEC_PAGESIZE - 1) & ~(EXEC_PAGESIZE - 1))
# define FREE_BUF(_B, _S) \
       munmap ((_B), ROUND_TO_PAGE (_S))
#define _IO_blen(fp) ((fp)->_IO_buf_end - (fp)->_IO_buf_base)

void
_IO_setb (f, b, eb, a)
     _IO_FILE *f;
      char *b;
     char *eb;
     int a;
{
  if (f->_IO_buf_base && !(f->_flags & _IO_USER_BUF))
    FREE_BUF (f->_IO_buf_base, _IO_blen (f));
  f->_IO_buf_base = b;
  f->_IO_buf_end = eb;
  if (a)
    f->_flags &= ~_IO_USER_BUF;
  else
    f->_flags |= _IO_USER_BUF;
}

_IO_FILE *
_IO_default_setbuf (fp, p, len)
     _IO_FILE *fp;
     char *p;
     _IO_ssize_t len;
{
    if (_IO_SYNC (fp) == EOF)
	return NULL;
    if (p == NULL || len == 0)
      {
	fp->_flags |= _IO_UNBUFFERED;
	_IO_setb (fp, fp->_shortbuf, fp->_shortbuf+1, 0);
      }
    else
      {
	fp->_flags &= ~_IO_UNBUFFERED;
	_IO_setb (fp, p, p+len, 0);
      }
    fp->_IO_write_base = fp->_IO_write_ptr = fp->_IO_write_end = 0;
    fp->_IO_read_base = fp->_IO_read_ptr = fp->_IO_read_end = 0;
    return fp;
}

int
_IO_default_sync (fp)
     _IO_FILE *fp;
{
  return 0;
}

_IO_ssize_t
_IO_default_read (fp, data, n)
     _IO_FILE* fp;
     void *data;
     _IO_ssize_t n;
{
  return -1;
}

_IO_ssize_t
_IO_default_write (fp, data, n)
     _IO_FILE *fp;
     const void *data;
     _IO_ssize_t n;
{
  return 0;
}

/*
 * Added static.
 * -- tkt
 */
static _IO_fpos64_t
_IO_default_seek (fp, offset, dir)
     _IO_FILE *fp;
     _IO_off64_t offset;
     int dir;
{
  return _IO_pos_BAD;
}

int
_IO_default_stat (fp, st)
     _IO_FILE *fp;
     void* st;
{
  return EOF;
}





/* We need this to define pthread_cleanup_push_defer() */
#define __USE_GNU

/*****************************************************************************/













/* Copyright (C) 1991,92,93,94,95,96,97,98,99 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <ctype.h>
#include <limits.h>
#include <printf.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdlib.h>
#include <errno.h>
#include <wchar.h>
#include <bits/libc-lock.h>
#include <sys/param.h>
//#include "_itoa.h"
//#include <locale/localeinfo.h>
#include <locale.h>

/* This code is shared between the standard stdio implementation found
   in GNU C library and the libio implementation originally found in
   GNU libg++.

   Beside this it is also shared between the normal and wide character
   implementation as defined in ISO/IEC 9899:1990/Amendment 1:1995.  */

#ifndef COMPILE_WPRINTF
# define CHAR_T		char
# define UCHAR_T	unsigned char
# define INT_T		int
# define L_(Str)	Str
# define ISDIGIT(Ch)	isdigit (Ch)

# ifdef USE_IN_LIBIO
#  define PUT(F, S, N)	_IO_sputn ((F), (S), (N))
#  define PAD(Padchar)							      \
  if (width > 0)							      \
    done += _IO_padn (s, (Padchar), width)
# else
#  define PUTC(C, F)	putc (C, F)
ssize_t __printf_pad __P ((FILE *, char pad, size_t n));
# define PAD(Padchar)							      \
  if (width > 0)							      \
    { ssize_t __res = __printf_pad (s, (Padchar), width);		      \
      if (__res == -1)							      \
	{								      \
	  done = -1;							      \
	  goto all_done;						      \
	}								      \
      done += __res; }
# endif
#else
# define vfprintf	vfwprintf
# define CHAR_T		wchar_t
# define UCHAR_T	uwchar_t
# define INT_T		wint_t
# define L_(Str)	L##Str
# define ISDIGIT(Ch)	iswdigit (Ch)

# ifdef USE_IN_LIBIO
#  define PUT(F, S, N)	_IO_sputn ((F), (S), (N))
#  define PAD(Padchar)							      \
  if (width > 0)							      \
    done += _IO_wpadn (s, (Padchar), width)
# else
#  define PUTC(C, F)	wputc (C, F)
ssize_t __wprintf_pad __P ((FILE *, wchar_t pad, size_t n));
#  define PAD(Padchar)							      \
  if (width > 0)							      \
    { ssize_t __res = __wprintf_pad (s, (Padchar), width);		      \
      if (__res == -1)							      \
	{								      \
	  done = -1;							      \
	  goto all_done;						      \
	}								      \
      done += __res; }
# endif
#endif

/* Include the shared code for parsing the format string.  */
#include "printf-parse.h"


#ifdef USE_IN_LIBIO
/* This code is for use in libio.  */
//# include <libioP.h>
# define PUTC(C, F)	_IO_putc_unlocked (C, F)
# define vfprintf	_IO_vfprintf
# define FILE		_IO_FILE
# undef va_list
# define va_list	_IO_va_list
# undef	BUFSIZ
# define BUFSIZ		_IO_BUFSIZ
# define ARGCHECK(S, Format)						      \
  do									      \
    {									      \
      /* Check file argument for consistence.  */			      \
      CHECK_FILE (S, -1);						      \
      if (S->_flags & _IO_NO_WRITES)					      \
	{								      \
	  __set_errno (EBADF);						      \
	  return -1;							      \
	}								      \
      if (Format == NULL)						      \
	{								      \
	  MAYBE_SET_EINVAL;						      \
	  return -1;							      \
	}								      \
    } while (0)
# define UNBUFFERED_P(S) ((S)->_IO_file_flags & _IO_UNBUFFERED)
#else /* ! USE_IN_LIBIO */
/* This code is for use in the GNU C library.  */
# include <stdio.h>
# define PUT(F, S, N)	fwrite (S, 1, N, F)
# define ARGCHECK(S, Format)						      \
  do									      \
    {									      \
      /* Check file argument for consistence.  */			      \
      if (!__validfp (S) || !S->__mode.__write)				      \
	{								      \
	  __set_errno (EBADF);						      \
	  return -1;							      \
	}								      \
      if (Format == NULL)						      \
	{								      \
	  __set_errno (EINVAL);						      \
	  return -1;							      \
	}								      \
      if (!S->__seen)							      \
	{								      \
	  if (__flshfp (S, EOF) == EOF)					      \
	    return -1;							      \
	}								      \
    }									      \
   while (0)
# define UNBUFFERED_P(s) ((s)->__buffer == NULL)

/* XXX These declarations should go as soon as the stdio header files
   have these prototypes.   */
extern void __flockfile (FILE *);
extern void __funlockfile (FILE *);
#endif /* USE_IN_LIBIO */


#define	outchar(Ch)							      \
  do									      \
    {									      \
      register const int outc = (Ch);					      \
      if (PUTC (outc, s) == EOF)					      \
	{								      \
	  done = -1;							      \
	  goto all_done;						      \
	}								      \
      else								      \
	++done;								      \
    }									      \
  while (0)

#define outstring(String, Len)						      \
  do									      \
    {									      \
      if ((size_t) PUT (s, (String), (Len)) != (size_t) (Len))		      \
	{								      \
	  done = -1;							      \
	  goto all_done;						      \
	}								      \
      done += (Len);							      \
    }									      \
  while (0)

/* For handling long_double and longlong we use the same flag.  */
#ifndef is_longlong
# define is_longlong is_long_double
#endif


/* Global variables.  */
static const char null[] = "(null)";


/* Helper function to provide temporary buffering for unbuffered streams.  */
static int buffered_vfprintf __P ((FILE *stream, const CHAR_T *fmt, va_list))
     internal_function;

/* Handle unknown format specifier.  */
static int printf_unknown __P ((FILE *, const struct printf_info *,
				const void *const *));

/* Group digits of number string.  */
static char *group_number __P ((CHAR_T *, CHAR_T *, const CHAR_T *, wchar_t))
     internal_function;


/* The function itself.  */
int
vfprintf (FILE *s, const CHAR_T *format, va_list ap)
{
  /* The character used as thousands separator.  */
  wchar_t thousands_sep;

  /* The string describing the size of groups of digits.  */
  const char *grouping;

  /* Place to accumulate the result.  */
  int done;

  /* Current character in format string.  */
  const UCHAR_T *f;

  /* End of leading constant string.  */
  const UCHAR_T *lead_str_end;

  /* Points to next format specifier.  */
  const UCHAR_T *end_of_spec;

  /* Buffer intermediate results.  */
  char work_buffer[1000];
  char *workend;

  /* State for restartable multibyte character handling functions.  */
  mbstate_t mbstate;

  /* We have to save the original argument pointer.  */
  va_list ap_save;

  /* Count number of specifiers we already processed.  */
  int nspecs_done;

  /* For the %m format we may need the current `errno' value.  */
  int save_errno = errno;


  /* This table maps a character into a number representing a
     class.  In each step there is a destination label for each
     class.  */
  static const int jump_table[] =
  {
    /* ' ' */  1,            0,            0, /* '#' */  4,
	       0, /* '%' */ 14,            0, /* '\''*/  6,
	       0,            0, /* '*' */  7, /* '+' */  2,
	       0, /* '-' */  3, /* '.' */  9,            0,
    /* '0' */  5, /* '1' */  8, /* '2' */  8, /* '3' */  8,
    /* '4' */  8, /* '5' */  8, /* '6' */  8, /* '7' */  8,
    /* '8' */  8, /* '9' */  8,            0,            0,
	       0,            0,            0,            0,
	       0, /* 'A' */ 26,            0, /* 'C' */ 25,
	       0, /* 'E' */ 19,            0, /* 'G' */ 19,
	       0,            0,            0,            0,
    /* 'L' */ 12,            0,            0,            0,
	       0,            0,            0, /* 'S' */ 21,
	       0,            0,            0,            0,
    /* 'X' */ 18,            0, /* 'Z' */ 13,            0,
	       0,            0,            0,            0,
	       0, /* 'a' */ 26,            0, /* 'c' */ 20,
    /* 'd' */ 15, /* 'e' */ 19, /* 'f' */ 19, /* 'g' */ 19,
    /* 'h' */ 10, /* 'i' */ 15, /* 'j' */ 28,            0,
    /* 'l' */ 11, /* 'm' */ 24, /* 'n' */ 23, /* 'o' */ 17,
    /* 'p' */ 22, /* 'q' */ 12,            0, /* 's' */ 21,
    /* 't' */ 27, /* 'u' */ 16,            0,            0,
    /* 'x' */ 18,            0, /* 'z' */ 13
  };

#define NOT_IN_JUMP_RANGE(Ch) ((Ch) < ' ' || (Ch) > 'z')
#define CHAR_CLASS(Ch) (jump_table[(int) (Ch) - ' '])
#define JUMP(ChExpr, table)						      \
      do								      \
	{								      \
	  const void *ptr;						      \
	  spec = (ChExpr);						      \
	  ptr = NOT_IN_JUMP_RANGE (spec) ? REF (form_unknown)		      \
	    : table[CHAR_CLASS (spec)];					      \
	  goto *ptr;							      \
	}								      \
      while (0)

#define STEP0_3_TABLE							      \
    /* Step 0: at the beginning.  */					      \
    static const void *step0_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (flag_space),		/* for ' ' */				      \
      REF (flag_plus),		/* for '+' */				      \
      REF (flag_minus),		/* for '-' */				      \
      REF (flag_hash),		/* for '<hash>' */			      \
      REF (flag_zero),		/* for '0' */				      \
      REF (flag_quote),		/* for '\'' */				      \
      REF (width_asterics),	/* for '*' */				      \
      REF (width),		/* for '1'...'9' */			      \
      REF (precision),		/* for '.' */				      \
      REF (mod_half),		/* for 'h' */				      \
      REF (mod_long),		/* for 'l' */				      \
      REF (mod_longlong),	/* for 'L', 'q' */			      \
      REF (mod_size_t),		/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_float),		/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_character),	/* for 'c' */				      \
      REF (form_string),	/* for 's', 'S' */			      \
      REF (form_pointer),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_strerror),	/* for 'm' */				      \
      REF (form_wcharacter),	/* for 'C' */				      \
      REF (form_floathex),	/* for 'A', 'a' */			      \
      REF (mod_ptrdiff_t),      /* for 't' */				      \
      REF (mod_intmax_t),       /* for 'j' */				      \
    };									      \
    /* Step 1: after processing width.  */				      \
    static const void *step1_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (form_unknown),	/* for ' ' */				      \
      REF (form_unknown),	/* for '+' */				      \
      REF (form_unknown),	/* for '-' */				      \
      REF (form_unknown),	/* for '<hash>' */			      \
      REF (form_unknown),	/* for '0' */				      \
      REF (form_unknown),	/* for '\'' */				      \
      REF (form_unknown),	/* for '*' */				      \
      REF (form_unknown),	/* for '1'...'9' */			      \
      REF (precision),		/* for '.' */				      \
      REF (mod_half),		/* for 'h' */				      \
      REF (mod_long),		/* for 'l' */				      \
      REF (mod_longlong),	/* for 'L', 'q' */			      \
      REF (mod_size_t),		/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_float),		/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_character),	/* for 'c' */				      \
      REF (form_string),	/* for 's', 'S' */			      \
      REF (form_pointer),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_strerror),	/* for 'm' */				      \
      REF (form_wcharacter),	/* for 'C' */				      \
      REF (form_floathex),	/* for 'A', 'a' */			      \
      REF (mod_ptrdiff_t),      /* for 't' */				      \
      REF (mod_intmax_t)        /* for 'j' */				      \
    };									      \
    /* Step 2: after processing precision.  */				      \
    static const void *step2_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (form_unknown),	/* for ' ' */				      \
      REF (form_unknown),	/* for '+' */				      \
      REF (form_unknown),	/* for '-' */				      \
      REF (form_unknown),	/* for '<hash>' */			      \
      REF (form_unknown),	/* for '0' */				      \
      REF (form_unknown),	/* for '\'' */				      \
      REF (form_unknown),	/* for '*' */				      \
      REF (form_unknown),	/* for '1'...'9' */			      \
      REF (form_unknown),	/* for '.' */				      \
      REF (mod_half),		/* for 'h' */				      \
      REF (mod_long),		/* for 'l' */				      \
      REF (mod_longlong),	/* for 'L', 'q' */			      \
      REF (mod_size_t),		/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_float),		/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_character),	/* for 'c' */				      \
      REF (form_string),	/* for 's', 'S' */			      \
      REF (form_pointer),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_strerror),	/* for 'm' */				      \
      REF (form_wcharacter),	/* for 'C' */				      \
      REF (form_floathex),	/* for 'A', 'a' */			      \
      REF (mod_ptrdiff_t),      /* for 't' */				      \
      REF (mod_intmax_t)        /* for 'j' */				      \
    };									      \
    /* Step 3a: after processing first 'h' modifier.  */		      \
    static const void *step3a_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (form_unknown),	/* for ' ' */				      \
      REF (form_unknown),	/* for '+' */				      \
      REF (form_unknown),	/* for '-' */				      \
      REF (form_unknown),	/* for '<hash>' */			      \
      REF (form_unknown),	/* for '0' */				      \
      REF (form_unknown),	/* for '\'' */				      \
      REF (form_unknown),	/* for '*' */				      \
      REF (form_unknown),	/* for '1'...'9' */			      \
      REF (form_unknown),	/* for '.' */				      \
      REF (mod_halfhalf),	/* for 'h' */				      \
      REF (form_unknown),	/* for 'l' */				      \
      REF (form_unknown),	/* for 'L', 'q' */			      \
      REF (form_unknown),	/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_unknown),	/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_unknown),	/* for 'c' */				      \
      REF (form_unknown),	/* for 's', 'S' */			      \
      REF (form_unknown),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_unknown),	/* for 'm' */				      \
      REF (form_unknown),	/* for 'C' */				      \
      REF (form_unknown),	/* for 'A', 'a' */			      \
      REF (form_unknown),       /* for 't' */				      \
      REF (form_unknown)        /* for 'j' */				      \
    };									      \
    /* Step 3b: after processing first 'l' modifier.  */		      \
    static const void *step3b_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (form_unknown),	/* for ' ' */				      \
      REF (form_unknown),	/* for '+' */				      \
      REF (form_unknown),	/* for '-' */				      \
      REF (form_unknown),	/* for '<hash>' */			      \
      REF (form_unknown),	/* for '0' */				      \
      REF (form_unknown),	/* for '\'' */				      \
      REF (form_unknown),	/* for '*' */				      \
      REF (form_unknown),	/* for '1'...'9' */			      \
      REF (form_unknown),	/* for '.' */				      \
      REF (form_unknown),	/* for 'h' */				      \
      REF (mod_longlong),	/* for 'l' */				      \
      REF (form_unknown),	/* for 'L', 'q' */			      \
      REF (form_unknown),	/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_float),		/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_character),	/* for 'c' */				      \
      REF (form_string),	/* for 's', 'S' */			      \
      REF (form_pointer),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_strerror),	/* for 'm' */				      \
      REF (form_wcharacter),	/* for 'C' */				      \
      REF (form_floathex),	/* for 'A', 'a' */			      \
      REF (form_unknown),       /* for 't' */				      \
      REF (form_unknown)        /* for 'j' */				      \
    }

#define STEP4_TABLE							      \
    /* Step 4: processing format specifier.  */				      \
    static const void *step4_jumps[29] =				      \
    {									      \
      REF (form_unknown),						      \
      REF (form_unknown),	/* for ' ' */				      \
      REF (form_unknown),	/* for '+' */				      \
      REF (form_unknown),	/* for '-' */				      \
      REF (form_unknown),	/* for '<hash>' */			      \
      REF (form_unknown),	/* for '0' */				      \
      REF (form_unknown),	/* for '\'' */				      \
      REF (form_unknown),	/* for '*' */				      \
      REF (form_unknown),	/* for '1'...'9' */			      \
      REF (form_unknown),	/* for '.' */				      \
      REF (form_unknown),	/* for 'h' */				      \
      REF (form_unknown),	/* for 'l' */				      \
      REF (form_unknown),	/* for 'L', 'q' */			      \
      REF (form_unknown),	/* for 'z', 'Z' */			      \
      REF (form_percent),	/* for '%' */				      \
      REF (form_integer),	/* for 'd', 'i' */			      \
      REF (form_unsigned),	/* for 'u' */				      \
      REF (form_octal),		/* for 'o' */				      \
      REF (form_hexa),		/* for 'X', 'x' */			      \
      REF (form_float),		/* for 'E', 'e', 'f', 'G', 'g' */	      \
      REF (form_character),	/* for 'c' */				      \
      REF (form_string),	/* for 's', 'S' */			      \
      REF (form_pointer),	/* for 'p' */				      \
      REF (form_number),	/* for 'n' */				      \
      REF (form_strerror),	/* for 'm' */				      \
      REF (form_wcharacter),	/* for 'C' */				      \
      REF (form_floathex),	/* for 'A', 'a' */			      \
      REF (form_unknown),       /* for 't' */				      \
      REF (form_unknown)        /* for 'j' */				      \
    }


#define process_arg(fspec)						      \
      /* Start real work.  We know about all flags and modifiers and	      \
	 now process the wanted format specifier.  */			      \
    LABEL (form_percent):						      \
      /* Write a literal "%".  */					      \
      outchar ('%');							      \
      break;								      \
									      \
    LABEL (form_integer):						      \
      /* Signed decimal integer.  */					      \
      base = 10;							      \
									      \
      if (is_longlong)							      \
	{								      \
	  long long int signed_number;					      \
									      \
	  if (fspec == NULL)						      \
	    signed_number = va_arg (ap, long long int);			      \
	  else								      \
	    signed_number = args_value[fspec->data_arg].pa_long_long_int;     \
									      \
	  is_negative = signed_number < 0;				      \
	  number.longlong = is_negative ? (- signed_number) : signed_number;  \
									      \
	  goto LABEL (longlong_number);					      \
	}								      \
      else								      \
	{								      \
	  long int signed_number;					      \
									      \
	  if (fspec == NULL)						      \
	    {								      \
	      if (is_long)						      \
		signed_number = va_arg (ap, long int);			      \
	      else  /* `char' and `short int' will be promoted to `int'.  */  \
		signed_number = va_arg (ap, int);			      \
	    }								      \
	  else								      \
	    if (is_long)						      \
	      signed_number = args_value[fspec->data_arg].pa_long_int;	      \
	    else							      \
	      signed_number = args_value[fspec->data_arg].pa_int;	      \
									      \
	  is_negative = signed_number < 0;				      \
	  number.word = is_negative ? (- signed_number) : signed_number;      \
									      \
	  goto LABEL (number);						      \
	}								      \
      /* NOTREACHED */							      \
									      \
    LABEL (form_unsigned):						      \
      /* Unsigned decimal integer.  */					      \
      base = 10;							      \
      goto LABEL (unsigned_number);					      \
      /* NOTREACHED */							      \
									      \
    LABEL (form_octal):							      \
      /* Unsigned octal integer.  */					      \
      base = 8;								      \
      goto LABEL (unsigned_number);					      \
      /* NOTREACHED */							      \
									      \
    LABEL (form_hexa):							      \
      /* Unsigned hexadecimal integer.  */				      \
      base = 16;							      \
									      \
    LABEL (unsigned_number):	  /* Unsigned number of base BASE.  */	      \
									      \
      /* ISO specifies the `+' and ` ' flags only for signed		      \
	 conversions.  */						      \
      is_negative = 0;							      \
      showsign = 0;							      \
      space = 0;							      \
									      \
      if (is_longlong)							      \
	{								      \
	  if (fspec == NULL)						      \
	    number.longlong = va_arg (ap, unsigned long long int);	      \
	  else								      \
	    number.longlong = args_value[fspec->data_arg].pa_u_long_long_int; \
									      \
	LABEL (longlong_number):					      \
	  if (prec < 0)							      \
	    /* Supply a default precision if none was given.  */	      \
	    prec = 1;							      \
	  else								      \
	    /* We have to take care for the '0' flag.  If a precision	      \
	       is given it must be ignored.  */				      \
	    pad = ' ';							      \
									      \
	  /* If the precision is 0 and the number is 0 nothing has to	      \
	     be written for the number, except for the 'o' format in	      \
	     alternate form.  */					      \
	  if (prec == 0 && number.longlong == 0)			      \
	    {								      \
	      string = workend;						      \
	      if (base == 8 && alt)					      \
		*string-- = '0';					      \
	    }								      \
	  else								      \
	    {								      \
	      /* Put the number in WORK.  */				      \
	      string = _itoa (number.longlong, workend + 1, base,	      \
			      spec == 'X');				      \
	      string -= 1;						      \
	      if (group && grouping)					      \
		string = group_number (string, workend, grouping,	      \
				       thousands_sep);			      \
	    }								      \
	  /* Simplify further test for num != 0.  */			      \
	  number.word = number.longlong != 0;				      \
	}								      \
      else								      \
	{								      \
	  if (fspec == NULL)						      \
	    {								      \
	      if (is_long)						      \
		number.word = va_arg (ap, unsigned long int);		      \
	      else if (!is_short)					      \
		number.word = va_arg (ap, unsigned int);		      \
	      else							      \
		number.word = (unsigned short int) va_arg (ap, unsigned int); \
	    }								      \
	  else								      \
	    if (is_long)						      \
	      number.word = args_value[fspec->data_arg].pa_u_long_int;	      \
	    else if (is_char)						      \
	      number.word = (unsigned char)				      \
		args_value[fspec->data_arg].pa_char;			      \
	    else if (!is_short)						      \
	      number.word = args_value[fspec->data_arg].pa_u_int;	      \
	    else							      \
	      number.word = (unsigned short int)			      \
		args_value[fspec->data_arg].pa_u_short_int;		      \
									      \
	LABEL (number):							      \
	  if (prec < 0)							      \
	    /* Supply a default precision if none was given.  */	      \
	    prec = 1;							      \
	  else								      \
	    /* We have to take care for the '0' flag.  If a precision	      \
	       is given it must be ignored.  */				      \
	    pad = ' ';							      \
									      \
	  /* If the precision is 0 and the number is 0 nothing has to	      \
	     be written for the number, except for the 'o' format in	      \
	     alternate form.  */					      \
	  if (prec == 0 && number.word == 0)				      \
	    {								      \
	      string = workend;						      \
	      if (base == 8 && alt)					      \
		*string-- = '0';					      \
	    }								      \
	  else								      \
	    {								      \
	      /* Put the number in WORK.  */				      \
	      string = _itoa_word (number.word, workend + 1, base,	      \
				   spec == 'X');			      \
	      string -= 1;						      \
	      if (group && grouping)					      \
		string = group_number (string, workend, grouping,	      \
				       thousands_sep);			      \
	    }								      \
	}								      \
									      \
      if (prec <= workend - string && number.word != 0 && alt && base == 8)   \
	/* Add octal marker.  */					      \
	*string-- = '0';						      \
									      \
      prec = MAX (0, prec - (workend - string));			      \
									      \
      if (!left)							      \
	{								      \
	  width -= workend - string + prec;				      \
									      \
	  if (number.word != 0 && alt && base == 16)			      \
	    /* Account for 0X hex marker.  */				      \
	    width -= 2;							      \
									      \
	  if (is_negative || showsign || space)				      \
	    --width;							      \
									      \
	  if (pad == ' ')						      \
	    {								      \
	      PAD (L_(' '));						      \
	      width = 0;						      \
	    }								      \
									      \
	  if (is_negative)						      \
	    outchar (L_('-'));						      \
	  else if (showsign)						      \
	    outchar (L_('+'));						      \
	  else if (space)						      \
	    outchar (L_(' '));						      \
									      \
	  if (number.word != 0 && alt && base == 16)			      \
	    {								      \
	      outchar (L_('0'));					      \
	      outchar (spec);						      \
	    }								      \
									      \
	  width += prec;						      \
	  PAD (L_('0'));						      \
									      \
	  outstring (string + 1, workend - string);			      \
									      \
	  break;							      \
	}								      \
      else								      \
	{								      \
	  if (is_negative)						      \
	    {								      \
	      outchar (L_('-'));					      \
	      --width;							      \
	    }								      \
	  else if (showsign)						      \
	    {								      \
	      outchar (L_('+'));					      \
	      --width;							      \
	    }								      \
	  else if (space)						      \
	    {								      \
	      outchar (L_(' '));					      \
	      --width;							      \
	    }								      \
									      \
	  if (number.word != 0 && alt && base == 16)			      \
	    {								      \
	      outchar (L_('0'));					      \
	      outchar (spec);						      \
	      width -= 2;						      \
	    }								      \
									      \
	  width -= workend - string + prec;				      \
									      \
	  if (prec > 0)							      \
	    {								      \
	      int temp = width;						      \
	      width = prec;						      \
	      PAD (L_('0'));						      \
	      width = temp;						      \
	    }								      \
									      \
	  outstring (string + 1, workend - string);			      \
									      \
	  PAD (' ');							      \
	  break;							      \
	}								      \
									      \
    LABEL (form_float):							      \
      {									      \
	/* Floating-point number.  This is handled by printf_fp.c.  */	      \
	extern int __printf_fp __P ((FILE *, const struct printf_info *,      \
				     const void **const));		      \
	const void *ptr;						      \
	int function_done;						      \
									      \
	if (fspec == NULL)						      \
	  {								      \
	    struct printf_info info = { prec: prec,			      \
					width: width,			      \
					spec: spec,			      \
					is_long_double: is_long_double,	      \
					is_short: is_short,		      \
					is_long: is_long,		      \
					alt: alt,			      \
					space: space,			      \
					left: left,			      \
					showsign: showsign,		      \
					group: group,			      \
					pad: pad,			      \
					extra: 0 };			      \
									      \
	    if (is_long_double)						      \
	      the_arg.pa_long_double = va_arg (ap, long double);	      \
	    else							      \
	      the_arg.pa_double = va_arg (ap, double);			      \
	    ptr = (const void *) &the_arg;				      \
									      \
	    function_done = __printf_fp (s, &info, &ptr);		      \
	  }								      \
	else								      \
	  {								      \
	    ptr = (const void *) &args_value[fspec->data_arg];		      \
									      \
	    function_done = __printf_fp (s, &fspec->info, &ptr);	      \
	  }								      \
									      \
	if (function_done < 0)						      \
	  {								      \
	    /* Error in print handler.  */				      \
	    done = -1;							      \
	    goto all_done;						      \
	  }								      \
									      \
	done += function_done;						      \
      }									      \
      break;								      \
									      \
    LABEL (form_floathex):						      \
      {									      \
        /* FLoating point number printed as hexadecimal number.  */	      \
	extern int __printf_fphex __P ((FILE *, const struct printf_info *,   \
					const void **const));		      \
	const void *ptr;						      \
	int function_done;						      \
									      \
	if (fspec == NULL)						      \
	  {								      \
	    struct printf_info info = { prec: prec,			      \
					width: width,			      \
					spec: spec,			      \
					is_long_double: is_long_double,	      \
					is_short: is_short,		      \
					is_long: is_long,		      \
					alt: alt,			      \
					space: space,			      \
					left: left,			      \
					showsign: showsign,		      \
					group: group,			      \
					pad: pad,			      \
					extra: 0 };			      \
									      \
	    if (is_long_double)						      \
	      the_arg.pa_long_double = va_arg (ap, long double);	      \
	    else							      \
	      the_arg.pa_double = va_arg (ap, double);			      \
	    ptr = (const void *) &the_arg;				      \
									      \
	    function_done = __printf_fphex (s, &info, &ptr);		      \
	  }								      \
	else								      \
	  {								      \
	    ptr = (const void *) &args_value[fspec->data_arg];		      \
									      \
	    function_done = __printf_fphex (s, &fspec->info, &ptr);	      \
	  }								      \
									      \
	if (function_done < 0)						      \
	  {								      \
	    /* Error in print handler.  */				      \
	    done = -1;							      \
	    goto all_done;						      \
	  }								      \
									      \
	done += function_done;						      \
      }									      \
      break;								      \
									      \
    LABEL (form_character):						      \
      /* Character.  */							      \
      if (is_long)							      \
	goto LABEL (form_wcharacter);					      \
      --width;	/* Account for the character itself.  */		      \
      if (!left)							      \
	PAD (' ');							      \
      if (fspec == NULL)						      \
	outchar ((unsigned char) va_arg (ap, int)); /* Promoted.  */	      \
      else								      \
	outchar ((unsigned char) args_value[fspec->data_arg].pa_char);	      \
      if (left)								      \
	PAD (' ');							      \
      break;								      \
									      \
    LABEL (form_wcharacter):						      \
      {									      \
	/* Wide character.  */						      \
	char buf[MB_CUR_MAX];						      \
	mbstate_t mbstate;						      \
	size_t len;							      \
									      \
	memset (&mbstate, '\0', sizeof (mbstate_t));			      \
	len = __wcrtomb (buf, (fspec == NULL ? va_arg (ap, wint_t)	      \
			       : args_value[fspec->data_arg].pa_wchar),	      \
			 &mbstate);					      \
	width -= len;							      \
	if (!left)							      \
	  PAD (' ');							      \
	outstring (buf, len);						      \
	if (left)							      \
	  PAD (' ');							      \
      }									      \
      break;								      \
									      \
    LABEL (form_string):						      \
      {									      \
	size_t len;							      \
									      \
	/* The string argument could in fact be `char *' or `wchar_t *'.      \
	   But this should not make a difference here.  */		      \
	if (fspec == NULL)						      \
	  string = (char *) va_arg (ap, const char *);			      \
	else								      \
	  string = (char *) args_value[fspec->data_arg].pa_string;	      \
									      \
	/* Entry point for printing other strings.  */			      \
      LABEL (print_string):						      \
									      \
	if (string == NULL)						      \
	  {								      \
	    /* Write "(null)" if there's space.  */			      \
	    if (prec == -1 || prec >= (int) sizeof (null) - 1)		      \
	      {								      \
		string = (char *) null;					      \
		len = sizeof (null) - 1;				      \
	      }								      \
	    else							      \
	      {								      \
		string = (char *) "";					      \
		len = 0;						      \
	      }								      \
	  }								      \
	else if (!is_long && spec != L_('S'))				      \
	  {								      \
	    if (prec != -1)						      \
	      /* Search for the end of the string, but don't search past      \
		 the length specified by the precision.  */		      \
	      len = strnlen (string, prec);				      \
	    else							      \
	      len = strlen (string);					      \
	  }								      \
	else								      \
	  {								      \
	    const wchar_t *s2 = (const wchar_t *) string;		      \
	    mbstate_t mbstate;						      \
									      \
	    memset (&mbstate, '\0', sizeof (mbstate_t));		      \
									      \
	    if (prec > 0)						      \
	      {								      \
		/* The string `s2' might not be NUL terminated.  */	      \
		string = (char *) alloca (prec);			      \
		len = __wcsrtombs (string, &s2, prec, &mbstate);	      \
	      }								      \
	    else							      \
	      {								      \
		len = __wcsrtombs (NULL, &s2, 0, &mbstate);		      \
		if (len != (size_t) -1)					      \
		  {							      \
		    assert (__mbsinit (&mbstate));			      \
		    s2 = (const wchar_t *) string;			      \
		    string = (char *) alloca (len + 1);			      \
		    (void) __wcsrtombs (string, &s2, len + 1, &mbstate);      \
		  }							      \
	      }								      \
									      \
	    if (len == (size_t) -1)					      \
	      {								      \
	        /* Illegal wide-character string.  */			      \
		done = -1;						      \
		goto all_done;						      \
	      }								      \
	  }								      \
									      \
	if ((width -= len) < 0)						      \
	  {								      \
	    outstring (string, len);					      \
	    break;							      \
	  }								      \
									      \
	if (!left)							      \
	  PAD (' ');							      \
	outstring (string, len);					      \
	if (left)							      \
	  PAD (' ');							      \
      }									      \
      break;								      \
									      \
    LABEL (form_pointer):						      \
      /* Generic pointer.  */						      \
      {									      \
	const void *ptr;						      \
	if (fspec == NULL)						      \
	  ptr = va_arg (ap, void *);					      \
	else								      \
	  ptr = args_value[fspec->data_arg].pa_pointer;			      \
	if (ptr != NULL)						      \
	  {								      \
	    /* If the pointer is not NULL, write it as a %#x spec.  */	      \
	    base = 16;							      \
	    number.word = (unsigned long int) ptr;			      \
	    is_negative = 0;						      \
	    alt = 1;							      \
	    group = 0;							      \
	    spec = 'x';							      \
	    goto LABEL (number);					      \
	  }								      \
	else								      \
	  {								      \
	    /* Write "(nil)" for a nil pointer.  */			      \
	    string = (char *) "(nil)";					      \
	    /* Make sure the full string "(nil)" is printed.  */	      \
	    if (prec < 5)						      \
	      prec = 5;							      \
	    is_long = 0;	/* This is no wide-char string.  */	      \
	    goto LABEL (print_string);					      \
	  }								      \
      }									      \
      /* NOTREACHED */							      \
									      \
    LABEL (form_number):						      \
      /* Answer the count of characters written.  */			      \
      if (fspec == NULL)						      \
	{								      \
/*-------------------------------------------------------------------------*/ \
/* Start libsafe additions						   */ \
	  if (_libsafe_level == 0) {					      \
	    /* Disable additional checks if *printf() is called by libsafe */ \
	    _libsafe_level++;						      \
									      \
	    /* Check to see if the current %n pointer points to a return */   \
	    /* address or frame pointer on the stack?  We only perform   */   \
	    /* libsafe warnings without dying because these violations   */   \
	    /* are mostly due to bad input instead of source that relies */   \
	    /* %n actually overwriting a return address or frame pointer */   \
	    /* -- tkt					                 */   \
	    if (_libsafe_raVariableP((void *)*((uint *)ap))) {		      \
	      _libsafe_warn("printf(\"%%n\") -- WARNING ONLY!!!");	      \
	    }								      \
	    else {							      \
	    /*-------------------------------------------------------------*/ \
	    /* ----- Original code -----				   */ \
	      if (is_longlong)						      \
		*(long long int *) va_arg (ap, void *) = done;		      \
	      else if (is_long)						      \
		*(long int *) va_arg (ap, void *) = done;		      \
	      else if (!is_short)					      \
		*(int *) va_arg (ap, void *) = done;			      \
	      else							      \
		*(short int *) va_arg (ap, void *) = done;		      \
	    /*-------------------------------------------------------------*/ \
	    }								      \
									      \
	    _libsafe_level--;						      \
	  }								      \
/* End libsafe additions						   */ \
/*-------------------------------------------------------------------------*/ \
	}								      \
      else								      \
	{								      \
/*-------------------------------------------------------------------------*/ \
/* Start libsafe additions						   */ \
	  if (_libsafe_level == 0) {					      \
	    /* Disable additional checks if *printf() is called by libsafe */ \
	    _libsafe_level++;						      \
									      \
	    /* Check to see if the current %n pointer points to a return */   \
	    /* address or frame pointer on the stack?  We only perform   */   \
	    /* libsafe warnings without dying because these violations   */   \
	    /* are mostly due to bad input instead of source that relies */   \
	    /* %n actually overwriting a return address or frame pointer */   \
	    /* -- tkt					                 */   \
	    if (_libsafe_raVariableP((void *)*((uint *)args_value[fspec->data_arg].pa_pointer))) { \
	      _libsafe_warn("printf(\"%%n\") -- WARNING ONLY!!!");	      \
	    }								      \
	    else {							      \
	    /*-------------------------------------------------------------*/ \
	    /* ----- Original code -----				   */ \
	      if (is_longlong)						      \
	        *(long long int *) args_value[fspec->data_arg].pa_pointer     \
		    = done;						      \
	      else if (is_long)						      \
	        *(long int *) args_value[fspec->data_arg].pa_pointer = done;  \
	      else if (!is_short)					      \
	        *(int *) args_value[fspec->data_arg].pa_pointer = done;	      \
	      else							      \
	        *(short int *) args_value[fspec->data_arg].pa_pointer = done; \
	    /*-------------------------------------------------------------*/ \
	    }								      \
									      \
	    _libsafe_level--;						      \
/* End libsafe additions						   */ \
/*-------------------------------------------------------------------------*/ \
	  }								      \
	}								      \
      break;								      \
									      \
    LABEL (form_strerror):						      \
      /* Print description of error ERRNO.  */				      \
      string =								      \
	(char *) __strerror_r (save_errno, work_buffer, sizeof work_buffer);  \
      is_long = 0;		/* This is no wide-char string.  */	      \
      goto LABEL (print_string)

  /* Sanity check of arguments.  */
  ARGCHECK (s, format);

  if (UNBUFFERED_P (s))
    /* Use a helper function which will allocate a local temporary buffer
       for the stream and then call us again.  */
    return buffered_vfprintf (s, format, ap);

  /* Initialize local variables.  */
  done = 0;
  grouping = (const char *) -1;
#ifdef __va_copy
  /* This macro will be available soon in gcc's <stdarg.h>.  We need it
     since on some systems `va_list' is not an integral type.  */
  __va_copy (ap_save, ap);
#else
  ap_save = ap;
#endif
  nspecs_done = 0;

  /* Put state for processing format string in initial state.  */
  memset (&mbstate, '\0', sizeof (mbstate_t));

  /* Find the first format specifier.  */
  f = lead_str_end = find_spec (format, &mbstate);

  /* Lock stream.  */
#ifdef USE_IN_LIBIO
  __libc_cleanup_region_start ((void (*) (void *)) &_IO_funlockfile, s);
  _IO_flockfile (s);
#else
  __libc_cleanup_region_start ((void (*) (void *)) &__funlockfile, s);
  __flockfile (s);
#endif

  /* Write the literal text before the first format.  */
  outstring ((const UCHAR_T *) format,
	     lead_str_end - (const UCHAR_T *) format);

  /* If we only have to print a simple string, return now.  */
  if (*f == L_('\0'))
    goto all_done;

  /* Process whole format string.  */
  do
    {
#define REF(Name) &&do_##Name
#define LABEL(Name) do_##Name
      STEP0_3_TABLE;
      STEP4_TABLE;

      union printf_arg *args_value;	/* This is not used here but ... */
      int is_negative;	/* Flag for negative number.  */
      union
      {
	unsigned long long int longlong;
	unsigned long int word;
      } number;
      int base;
      union printf_arg the_arg;
      char *string;	/* Pointer to argument string.  */
      int alt = 0;	/* Alternate format.  */
      int space = 0;	/* Use space prefix if no sign is needed.  */
      int left = 0;	/* Left-justify output.  */
      int showsign = 0;	/* Always begin with plus or minus sign.  */
      int group = 0;	/* Print numbers according grouping rules.  */
      int is_long_double = 0; /* Argument is long double/ long long int.  */
      int is_short = 0;	/* Argument is long int.  */
      int is_long = 0;	/* Argument is short int.  */
      int is_char = 0;	/* Argument is promoted (unsigned) char.  */
      int width = 0;	/* Width of output; 0 means none specified.  */
      int prec = -1;	/* Precision of output; -1 means none specified.  */
      char pad = ' ';	/* Padding character.  */
      CHAR_T spec;

      workend = &work_buffer[sizeof (work_buffer) - 1];

      /* Get current character in format string.  */
      JUMP (*++f, step0_jumps);

      /* ' ' flag.  */
    LABEL (flag_space):
      space = 1;
      JUMP (*++f, step0_jumps);

      /* '+' flag.  */
    LABEL (flag_plus):
      showsign = 1;
      JUMP (*++f, step0_jumps);

      /* The '-' flag.  */
    LABEL (flag_minus):
      left = 1;
      pad = L_(' ');
      JUMP (*++f, step0_jumps);

      /* The '#' flag.  */
    LABEL (flag_hash):
      alt = 1;
      JUMP (*++f, step0_jumps);

      /* The '0' flag.  */
    LABEL (flag_zero):
      if (!left)
	pad = L_('0');
      JUMP (*++f, step0_jumps);

      /* The '\'' flag.  */
    LABEL (flag_quote):
      group = 1;

      /* XXX Completely wrong.  Use wctob.  */
      if (grouping == (const char *) -1)
	{
	  mbstate_t mbstate;

#ifdef OLD_CODE_DONT_EXECUTE_THIS
	  /* Figure out the thousands separator character.  */
	  memset (&mbstate, '\0', sizeof (mbstate));
	  if (__mbrtowc (&thousands_sep,
			 _NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP),
			 strlen (_NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP)),
			 &mbstate) <= 0)
	    thousands_sep = (wchar_t)
	      *_NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP);
	  grouping = _NL_CURRENT (LC_NUMERIC, GROUPING);
	  if (*grouping == '\0' || *grouping == CHAR_MAX
	      || thousands_sep == L'\0')
	    grouping = NULL;
#else
	  /*******************************************************************/
	  /*
	   * I added the following as a way to get the locale info for
	   * LC_NUMERIC -- tkt
	   */
	  if (!dolocaleconv) {
	      dolocaleconv = 1;
	      conv = localeconv();
	      thousands = conv->thousands_sep;
	      grouping = conv->grouping;
	  }
	  /*******************************************************************/

	  /* Figure out the thousands separator character.  */
	  memset (&mbstate, '\0', sizeof (mbstate));
	  if (__mbrtowc (&thousands_sep, thousands, strlen (thousands),
			 &mbstate) <= 0)
	    thousands_sep = *thousands;
	  if (*grouping == '\0' || *grouping == CHAR_MAX
	      || thousands == L'\0')
	    grouping = NULL;
#endif
	}
      JUMP (*++f, step0_jumps);

      /* Get width from argument.  */
    LABEL (width_asterics):
      {
	const UCHAR_T *tmp;	/* Temporary value.  */

	tmp = ++f;
	if (ISDIGIT (*tmp) && read_int (&tmp) && *tmp == L_('$'))
	  /* The width comes from a positional parameter.  */
	  goto do_positional;

	width = va_arg (ap, int);

	/* Negative width means left justified.  */
	if (width < 0)
	  {
	    width = -width;
	    pad = L_(' ');
	    left = 1;
	  }

	if (width + 32 >= sizeof (work_buffer))
	  /* We have to use a special buffer.  The "32" is just a safe
	     bet for all the output which is not counted in the width.  */
	  workend = alloca (width + 32) + (width + 31);
      }
      JUMP (*f, step1_jumps);

      /* Given width in format string.  */
    LABEL (width):
      width = read_int (&f);

      if (width + 32 >= sizeof (work_buffer))
	/* We have to use a special buffer.  The "32" is just a safe
	   bet for all the output which is not counted in the width.  */
	workend = alloca (width + 32) + (width + 31);
      if (*f == L_('$'))
	/* Oh, oh.  The argument comes from a positional parameter.  */
	goto do_positional;
      JUMP (*f, step1_jumps);

    LABEL (precision):
      ++f;
      if (*f == L_('*'))
	{
	  const UCHAR_T *tmp;	/* Temporary value.  */

	  tmp = ++f;
	  if (ISDIGIT (*tmp) && read_int (&tmp) > 0 && *tmp == L_('$'))
	    /* The precision comes from a positional parameter.  */
	    goto do_positional;

	  prec = va_arg (ap, int);

	  /* If the precision is negative the precision is omitted.  */
	  if (prec < 0)
	    prec = -1;
	}
      else if (ISDIGIT (*f))
	prec = read_int (&f);
      else
	prec = 0;
      if (prec > width && prec + 32 > sizeof (work_buffer))
	workend = alloca (spec + 32) + (spec + 31);
      JUMP (*f, step2_jumps);

      /* Process 'h' modifier.  There might another 'h' following.  */
    LABEL (mod_half):
      is_short = 1;
      JUMP (*++f, step3a_jumps);

      /* Process 'hh' modifier.  */
    LABEL (mod_halfhalf):
      is_short = 0;
      is_char = 1;
      JUMP (*++f, step4_jumps);

      /* Process 'l' modifier.  There might another 'l' following.  */
    LABEL (mod_long):
      is_long = 1;
      JUMP (*++f, step3b_jumps);

      /* Process 'L', 'q', or 'll' modifier.  No other modifier is
	 allowed to follow.  */
    LABEL (mod_longlong):
      is_long_double = 1;
      JUMP (*++f, step4_jumps);

    LABEL (mod_size_t):
      is_longlong = sizeof (size_t) > sizeof (unsigned long int);
      is_long = sizeof (size_t) > sizeof (unsigned int);
      JUMP (*++f, step4_jumps);

    LABEL (mod_ptrdiff_t):
      is_longlong = sizeof (ptrdiff_t) > sizeof (unsigned long int);
      is_long = sizeof (ptrdiff_t) > sizeof (unsigned int);
      JUMP (*++f, step4_jumps);

    LABEL (mod_intmax_t):
      is_longlong = sizeof (intmax_t) > sizeof (unsigned long int);
      is_long = sizeof (intmax_t) > sizeof (unsigned int);
      JUMP (*++f, step4_jumps);

      /* Process current format.  */
      while (1)
	{
	  process_arg (((struct printf_spec *) NULL));

	LABEL (form_unknown):
	  if (spec == L_('\0'))
	    {
	      /* The format string ended before the specifier is complete.  */
	      done = -1;
	      goto all_done;
	    }

	  /* If we are in the fast loop force entering the complicated
	     one.  */
	  goto do_positional;
	}

      /* The format is correctly handled.  */
      ++nspecs_done;

      /* Look for next format specifier.  */
      f = find_spec ((end_of_spec = ++f), &mbstate);

      /* Write the following constant string.  */
      outstring (end_of_spec, f - end_of_spec);
    }
  while (*f != L_('\0'));

  /* Unlock stream and return.  */
  goto all_done;

  /* Here starts the more complex loop to handle positional parameters.  */
do_positional:
  {
    /* Array with information about the needed arguments.  This has to
       be dynamically extensible.  */
    size_t nspecs = 0;
    size_t nspecs_max = 32;	/* A more or less arbitrary start value.  */
    struct printf_spec *specs
      = alloca (nspecs_max * sizeof (struct printf_spec));

    /* The number of arguments the format string requests.  This will
       determine the size of the array needed to store the argument
       attributes.  */
    size_t nargs = 0;
    int *args_type;
    union printf_arg *args_value;

    /* Positional parameters refer to arguments directly.  This could
       also determine the maximum number of arguments.  Track the
       maximum number.  */
    size_t max_ref_arg = 0;

    /* Just a counter.  */
    size_t cnt;


    if (grouping == (const char *) -1)
      {
	mbstate_t mbstate;

#ifdef OLD_CODE_DONT_EXECUTE_THIS
	/* Figure out the thousands separator character.  */
	memset (&mbstate, '\0', sizeof (mbstate));
	if (__mbrtowc (&thousands_sep,
		       _NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP),
		       strlen (_NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP)),
		       &mbstate) <= 0)
	  thousands_sep = (wchar_t) *_NL_CURRENT (LC_NUMERIC, THOUSANDS_SEP);
	grouping = _NL_CURRENT (LC_NUMERIC, GROUPING);
	if (*grouping == '\0' || *grouping == CHAR_MAX
	    || thousands_sep == L'\0')
	  grouping = NULL;
#else
	/*******************************************************************/
	/*
	 * I added the following as a way to get the locale info for
	 * LC_NUMERIC -- tkt
	 */
	if (!dolocaleconv) {
	    dolocaleconv = 1;
	    conv = localeconv();
	    thousands = conv->thousands_sep;
	    grouping = conv->grouping;
	}
	/*******************************************************************/

	/* Figure out the thousands separator character.  */
	memset (&mbstate, '\0', sizeof (mbstate));
	if (__mbrtowc (&thousands_sep, thousands, strlen (thousands),
	      	 &mbstate) <= 0)
	  thousands_sep = *thousands;
	if (*grouping == '\0' || *grouping == CHAR_MAX
	    || thousands == L'\0')
	  grouping = NULL;
#endif
      }

    for (f = lead_str_end; *f != '\0'; f = specs[nspecs++].next_fmt)
      {
	if (nspecs >= nspecs_max)
	  {
	    /* Extend the array of format specifiers.  */
	    struct printf_spec *old = specs;

	    nspecs_max *= 2;
	    specs = alloca (nspecs_max * sizeof (struct printf_spec));

	    if (specs == &old[nspecs])
	      /* Stack grows up, OLD was the last thing allocated;
		 extend it.  */
	      nspecs_max += nspecs_max / 2;
	    else
	      {
		/* Copy the old array's elements to the new space.  */
		memcpy (specs, old, nspecs * sizeof (struct printf_spec));
		if (old == &specs[nspecs])
		  /* Stack grows down, OLD was just below the new
		     SPECS.  We can use that space when the new space
		     runs out.  */
		  nspecs_max += nspecs_max / 2;
	      }
	  }

	/* Parse the format specifier.  */
	nargs += parse_one_spec (f, nargs, &specs[nspecs], &max_ref_arg,
				 &mbstate);
      }

    /* Determine the number of arguments the format string consumes.  */
    nargs = MAX (nargs, max_ref_arg);

    /* Allocate memory for the argument descriptions.  */
    args_type = alloca (nargs * sizeof (int));
    memset (args_type, 0, nargs * sizeof (int));
    args_value = alloca (nargs * sizeof (union printf_arg));

    /* XXX Could do sanity check here: If any element in ARGS_TYPE is
       still zero after this loop, format is invalid.  For now we
       simply use 0 as the value.  */

    /* Fill in the types of all the arguments.  */
    for (cnt = 0; cnt < nspecs; ++cnt)
      {
	/* If the width is determined by an argument this is an int.  */
	if (specs[cnt].width_arg != -1)
	  args_type[specs[cnt].width_arg] = PA_INT;

	/* If the precision is determined by an argument this is an int.  */
	if (specs[cnt].prec_arg != -1)
	  args_type[specs[cnt].prec_arg] = PA_INT;

	switch (specs[cnt].ndata_args)
	  {
	  case 0:		/* No arguments.  */
	    break;
	  case 1:		/* One argument; we already have the type.  */
	    args_type[specs[cnt].data_arg] = specs[cnt].data_arg_type;
	    break;
	  default:
	    /* We have more than one argument for this format spec.
	       We must call the arginfo function again to determine
	       all the types.  */
	    (void) (*__printf_arginfo_table[specs[cnt].info.spec])
	      (&specs[cnt].info,
	       specs[cnt].ndata_args, &args_type[specs[cnt].data_arg]);
	    break;
	  }
      }

    /* Now we know all the types and the order.  Fill in the argument
       values.  */
    for (cnt = 0; cnt < nargs; ++cnt)
      switch (args_type[cnt])
	{
#define T(tag, mem, type)						      \
	case tag:							      \
	  args_value[cnt].mem = va_arg (ap_save, type);			      \
	  break

	T (PA_CHAR, pa_char, int); /* Promoted.  */
	T (PA_WCHAR, pa_wchar, wint_t);
	T (PA_INT|PA_FLAG_SHORT, pa_short_int, int); /* Promoted.  */
	T (PA_INT, pa_int, int);
	T (PA_INT|PA_FLAG_LONG, pa_long_int, long int);
	T (PA_INT|PA_FLAG_LONG_LONG, pa_long_long_int, long long int);
	T (PA_FLOAT, pa_float, double);	/* Promoted.  */
	T (PA_DOUBLE, pa_double, double);
	T (PA_DOUBLE|PA_FLAG_LONG_DOUBLE, pa_long_double, long double);
	T (PA_STRING, pa_string, const char *);
	T (PA_WSTRING, pa_wstring, const wchar_t *);
	T (PA_POINTER, pa_pointer, void *);
#undef T
	default:
	  if ((args_type[cnt] & PA_FLAG_PTR) != 0)
	    args_value[cnt].pa_pointer = va_arg (ap_save, void *);
	  else
	    args_value[cnt].pa_long_double = 0.0;
	  break;
	}

    /* Now walk through all format specifiers and process them.  */
    for (; (size_t) nspecs_done < nspecs; ++nspecs_done)
      {
#undef REF
#define REF(Name) &&do2_##Name
#undef LABEL
#define LABEL(Name) do2_##Name
	STEP4_TABLE;

	int is_negative;
	union
	{
	  unsigned long long int longlong;
	  unsigned long int word;
	} number;
	int base;
	union printf_arg the_arg;
	char *string;	/* Pointer to argument string.  */

	/* Fill variables from values in struct.  */
	int alt = specs[nspecs_done].info.alt;
	int space = specs[nspecs_done].info.space;
	int left = specs[nspecs_done].info.left;
	int showsign = specs[nspecs_done].info.showsign;
	int group = specs[nspecs_done].info.group;
	int is_long_double = specs[nspecs_done].info.is_long_double;
	int is_short = specs[nspecs_done].info.is_short;
	int is_char = specs[nspecs_done].info.is_char;
	int is_long = specs[nspecs_done].info.is_long;
	int width = specs[nspecs_done].info.width;
	int prec = specs[nspecs_done].info.prec;
	char pad = specs[nspecs_done].info.pad;
	CHAR_T spec = specs[nspecs_done].info.spec;

	/* Fill in last information.  */
	if (specs[nspecs_done].width_arg != -1)
	  {
	    /* Extract the field width from an argument.  */
	    specs[nspecs_done].info.width =
	      args_value[specs[nspecs_done].width_arg].pa_int;

	    if (specs[nspecs_done].info.width < 0)
	      /* If the width value is negative left justification is
		 selected and the value is taken as being positive.  */
	      {
		specs[nspecs_done].info.width *= -1;
		left = specs[nspecs_done].info.left = 1;
	      }
	    width = specs[nspecs_done].info.width;
	  }

	if (specs[nspecs_done].prec_arg != -1)
	  {
	    /* Extract the precision from an argument.  */
	    specs[nspecs_done].info.prec =
	      args_value[specs[nspecs_done].prec_arg].pa_int;

	    if (specs[nspecs_done].info.prec < 0)
	      /* If the precision is negative the precision is
		 omitted.  */
	      specs[nspecs_done].info.prec = -1;

	    prec = specs[nspecs_done].info.prec;
	  }

	/* Maybe the buffer is too small.  */
	if (MAX (prec, width) + 32 > sizeof (work_buffer))
	  workend = alloca (MAX (prec, width) + 32) + (MAX (prec, width) + 31);

	/* Process format specifiers.  */
	while (1)
	  {
	    JUMP (spec, step4_jumps);

	    process_arg ((&specs[nspecs_done]));

	  LABEL (form_unknown):
	    {
	      extern printf_function **__printf_function_table;
	      int function_done;
	      printf_function *function;
	      unsigned int i;
	      const void **ptr;

	      function =
		(__printf_function_table == NULL ? NULL :
		 __printf_function_table[specs[nspecs_done].info.spec]);

	      if (function == NULL)
		function = &printf_unknown;

	      ptr = alloca (specs[nspecs_done].ndata_args
			    * sizeof (const void *));

	      /* Fill in an array of pointers to the argument values.  */
	      for (i = 0; i < specs[nspecs_done].ndata_args; ++i)
		ptr[i] = &args_value[specs[nspecs_done].data_arg + i];

	      /* Call the function.  */
	      function_done = (*function) (s, &specs[nspecs_done].info, ptr);

	      /* If an error occurred we don't have information about #
		 of chars.  */
	      if (function_done < 0)
		{
		  done = -1;
		  goto all_done;
		}

	      done += function_done;
	    }
	    break;
	  }

	/* Write the following constant string.  */
	outstring (specs[nspecs_done].end_of_fmt,
		   specs[nspecs_done].next_fmt
		   - specs[nspecs_done].end_of_fmt);
      }
  }

all_done:

#ifdef _LIBSAFE_DETECT_OVERFLOW
/*-------------------------------------------------------------------------*/
/* Start libsafe additions						   */
  /* Disable additional checks if *printf() is called by libsafe */
  if (_libsafe_level == 0) {
    _libsafe_level++;
    if (_libsafe_span_stack_frames(ap_save, (void *)((uint)ap-1))) {
      _libsafe_die("printf() args span stack frames ");
    }
    _libsafe_level--;
  }
/* End libsafe additions						   */
/*-------------------------------------------------------------------------*/
#endif /* _LIBSAFE_DETECT_OVERFLOW */

  /* Unlock the stream.  */
#ifdef USE_IN_LIBIO
  _IO_funlockfile (s);
#else
  __funlockfile (s);
#endif
  __libc_cleanup_region_end (0);

  return done;
}

#ifdef USE_IN_LIBIO
# undef vfprintf
# ifdef strong_alias
/* This is for glibc.  */
strong_alias (_IO_vfprintf, vfprintf);
# else
#  if defined __ELF__ || defined __GNU_LIBRARY__
//#   include <gnu-stabs.h>
#   ifdef weak_alias
weak_alias (_IO_vfprintf, vfprintf);
#   endif
#  endif
# endif
#endif

/* Handle an unknown format specifier.  This prints out a canonicalized
   representation of the format spec itself.  */
static int
printf_unknown (FILE *s, const struct printf_info *info,
		const void *const *args)

{
  int done = 0;
  char work_buffer[MAX (info->width, info->spec) + 32];
  char *const workend = &work_buffer[sizeof (work_buffer) - 1];
  register char *w;

  outchar ('%');

  if (info->alt)
    outchar ('#');
  if (info->group)
    outchar ('\'');
  if (info->showsign)
    outchar ('+');
  else if (info->space)
    outchar (' ');
  if (info->left)
    outchar ('-');
  if (info->pad == '0')
    outchar ('0');

  if (info->width != 0)
    {
      w = _itoa_word (info->width, workend + 1, 10, 0);
      while (w <= workend)
	outchar (*w++);
    }

  if (info->prec != -1)
    {
      outchar ('.');
      w = _itoa_word (info->prec, workend + 1, 10, 0);
      while (w <= workend)
	outchar (*w++);
    }

  if (info->spec != '\0')
    outchar (info->spec);

 all_done:
  return done;
}

/* Group the digits according to the grouping rules of the current locale.
   The interpretation of GROUPING is as in `struct lconv' from <locale.h>.  */
static char *
internal_function
group_number (CHAR_T *w, CHAR_T *rear_ptr, const CHAR_T *grouping,
	      wchar_t thousands_sep)
{
  int len;
  char *src, *s;

  /* We treat all negative values like CHAR_MAX.  */

  if (*grouping == CHAR_MAX || *grouping <= 0)
    /* No grouping should be done.  */
    return w;

  len = *grouping;

  /* Copy existing string so that nothing gets overwritten.  */
  src = (char *) alloca (rear_ptr - w);
  s = (char *) __mempcpy (src, w + 1, rear_ptr - w) - 1;
  w = rear_ptr;

  /* Process all characters in the string.  */
  while (s >= src)
    {
      *w-- = *s--;

      if (--len == 0 && s >= src)
	{
	  /* A new group begins.  */
	  *w-- = thousands_sep;

	  len = *grouping++;
	  if (*grouping == '\0')
	    /* The previous grouping repeats ad infinitum.  */
	    --grouping;
	  else if (*grouping == CHAR_MAX
#if CHAR_MIN < 0
		   || *grouping < 0
#endif
		   )
	    {
	      /* No further grouping to be done.
		 Copy the rest of the number.  */
	      do
		*w-- = *s--;
	      while (s >= src);
	      break;
	    }
	}
    }
  return w;
}


/*------------------------------------------------------------------*/

/*
 * NOTE:
 * _IO_vfprintf() intercepts
 *	sprintf(), snprintf(), vsprintf(), vsnprintf()
 *
 *  The following function is needed to intercept
 *	printf(), fprintf(), vprintf(), vfprintf() 
 *
 * -- tkt
 */

#undef vfprintf
int vfprintf (FILE *s, const CHAR_T *format, va_list ap)
{
    return _IO_vfprintf(s, format, ap);
}

/*------------------------------------------------------------------*/





#ifdef USE_IN_LIBIO
/* Helper "class" for `fprintf to unbuffered': creates a temporary buffer.  */
struct helper_file
  {
    struct _IO_FILE_plus _f;
    _IO_FILE *_put_stream;
#ifdef _IO_MTSAFE_IO
    _IO_lock_t lock;
#endif
  };

static int
_IO_helper_overflow (_IO_FILE *s, int c)
{
  _IO_FILE *target = ((struct helper_file*) s)->_put_stream;
  int used = s->_IO_write_ptr - s->_IO_write_base;
  if (used)
    {
      _IO_size_t written = _IO_sputn (target, s->_IO_write_base, used);
      s->_IO_write_ptr -= written;
    }
  return PUTC (c, s);
}

extern void _IO_default_finish __P ((_IO_FILE *, int));

extern int _IO_default_underflow __P ((_IO_FILE *));
extern int _IO_default_uflow __P ((_IO_FILE *));
extern int _IO_default_pbackfail __P ((_IO_FILE *, int)); 
extern _IO_size_t _IO_default_xsputn __P ((_IO_FILE *, const void *,
	                                               _IO_size_t));
extern _IO_size_t _IO_default_xsgetn __P ((_IO_FILE *, void *, _IO_size_t));
extern _IO_fpos64_t _IO_default_seekoff __P ((_IO_FILE *,
	                                                  _IO_off64_t, int,
							  int)); 
extern _IO_fpos64_t _IO_default_seekpos __P ((_IO_FILE *,
	                                                  _IO_fpos64_t, int));
extern _IO_FILE* _IO_default_setbuf __P ((_IO_FILE *, char *, _IO_ssize_t));
extern int _IO_default_sync __P ((_IO_FILE *));
extern int _IO_default_doallocate __P ((_IO_FILE *));
extern _IO_ssize_t _IO_default_write __P ((_IO_FILE *, const void *,
	                                               _IO_ssize_t));
extern _IO_ssize_t _IO_default_read __P ((_IO_FILE *, void *, _IO_ssize_t));
extern int _IO_default_stat __P ((_IO_FILE *, void *));
extern _IO_fpos64_t _IO_default_seek __P ((_IO_FILE *, _IO_off64_t, int));
#define _IO_default_close ((_IO_close_t) _IO_default_sync)
static const struct _IO_jump_t _IO_helper_jumps =
{
  JUMP_INIT_DUMMY,
  JUMP_INIT (finish, _IO_default_finish),
  JUMP_INIT (overflow, _IO_helper_overflow),
  JUMP_INIT (underflow, _IO_default_underflow),
  JUMP_INIT (uflow, _IO_default_uflow),
  JUMP_INIT (pbackfail, _IO_default_pbackfail),
  JUMP_INIT (xsputn, _IO_default_xsputn),
  JUMP_INIT (xsgetn, _IO_default_xsgetn),
  JUMP_INIT (seekoff, _IO_default_seekoff),
  JUMP_INIT (seekpos, _IO_default_seekpos),
  JUMP_INIT (setbuf, _IO_default_setbuf),
  JUMP_INIT (sync, _IO_default_sync),
  JUMP_INIT (doallocate, _IO_default_doallocate),
  JUMP_INIT (read, _IO_default_read),
  JUMP_INIT (write, _IO_default_write),
  JUMP_INIT (seek, _IO_default_seek),
  JUMP_INIT (close, _IO_default_close),
  JUMP_INIT (stat, _IO_default_stat)
};

static int
internal_function
buffered_vfprintf (register _IO_FILE *s, const CHAR_T *format,
		   _IO_va_list args)
{
  char buf[_IO_BUFSIZ];
  struct helper_file helper;
  register _IO_FILE *hp = (_IO_FILE *) &helper;
  int result, to_flush;

  /* Initialize helper.  */
  helper._put_stream = s;
  hp->_IO_write_base = buf;
  hp->_IO_write_ptr = buf;
  hp->_IO_write_end = buf + sizeof buf;
  hp->_IO_file_flags = _IO_MAGIC|_IO_NO_READS;
#if _IO_JUMPS_OFFSET
  hp->_vtable_offset = 0;
#endif
#ifdef _IO_MTSAFE_IO
  hp->_lock = &helper.lock;
  __libc_lock_init (*hp->_lock);
#endif
  _IO_JUMPS (hp) = (struct _IO_jump_t *) &_IO_helper_jumps;

  /* Now print to helper instead.  */
  result = _IO_vfprintf (hp, format, args);

  /* Lock stream.  */
  __libc_cleanup_region_start ((void (*) (void *)) &_IO_funlockfile, s);
  _IO_flockfile (s);

  /* Now flush anything from the helper to the S. */
  if ((to_flush = hp->_IO_write_ptr - hp->_IO_write_base) > 0)
    {
      if ((int) _IO_sputn (s, hp->_IO_write_base, to_flush) != to_flush)
	result = -1;
    }

  /* Unlock the stream.  */
  _IO_funlockfile (s);
  __libc_cleanup_region_end (0);

  return result;
}

#else /* !USE_IN_LIBIO */

static int
internal_function
buffered_vfprintf (register FILE *s, const CHAR_T *format, va_list args)
{
  char buf[BUFSIZ];
  int result;

  s->__bufp = s->__buffer = buf;
  s->__bufsize = sizeof buf;
  s->__put_limit = s->__buffer + s->__bufsize;
  s->__get_limit = s->__buffer;

  /* Now use buffer to print.  */
  result = vfprintf (s, format, args);

  if (fflush (s) == EOF)
    result = -1;
  s->__buffer = s->__bufp = s->__get_limit = s->__put_limit = NULL;
  s->__bufsize = 0;

  return result;
}

/* Pads string with given number of a specified character.
   This code is taken from iopadn.c of the GNU I/O library.  */
#define PADSIZE 16
static const CHAR_T blanks[PADSIZE] =
{ L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' '),
  L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' '), L_(' ') };
static const CHAR_T zeroes[PADSIZE] =
{ L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0'),
  L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0'), L_('0') };

ssize_t
#ifndef COMPILE_WPRINTF
__printf_pad (FILE *s, char pad, size_t count)
#else
__wprintf_pad (FILE *s, wchar_t pad, size_t count)
#endif
{
  const CHAR_T *padptr;
  register size_t i;

  padptr = pad == L_(' ') ? blanks : zeroes;

  for (i = count; i >= PADSIZE; i -= PADSIZE)
    if (PUT (s, padptr, PADSIZE) != PADSIZE)
      return -1;
  if (i > 0)
    if (PUT (s, padptr, i) != i)
      return -1;

  return count;
}
#undef PADSIZE
#endif /* USE_IN_LIBIO */
