/*
 * Copyright (c) 1980, 1990 Regents of the University of California. All
 * rights reserved.
 * 
 * This code is derived from software contributed to Berkeley by Robert Elz at
 * The University of Melbourne.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution. 3. All advertising
 * materials mentioning features or use of this software must display the
 * following acknowledgement: This product includes software developed by the
 * University of California, Berkeley and its contributors. 4. Neither the
 * name of the University nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1980, 1990 Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char rcsid[] = "$Id: quotaops.c,v 1.4 2000/09/05 18:47:15 mvw Exp mvw $";
#endif /* not lint */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/quota.h>
#include <errno.h>
#include <mntent.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <paths.h>
#include <quotaops.h>
#if defined(RPC)
#include <rquota.h>
#endif

#include "pot.h"

#include "bylabel.h"

#define   FOUND   0x01

static char *qfextension[] = INITQFNAMES;

/*
 * Convert seconds to ASCII times.
 */
static char *cvtstoa(time_t time)
{
   static char buf[20];

   if (time % (24 * 60 * 60) == 0) 
   {
      time /= 24 * 60 * 60;
      if( time == 1 )
          sprintf(buf, _("day"));
	else
	  sprintf(buf, _("%d days"), time );
	  
   } else if (time % (60 * 60) == 0) {
      time /= 60 * 60;
      if( time == 1 )
          sprintf(buf, _("hour") );
	else
	  sprintf(buf, _("%d hours"), time );

   } else if (time % 60 == 0) {
      time /= 60;
      if( time == 1 )
          sprintf(buf, _("minute") );
	else 
	  sprintf(buf, _("%d minutes"), time );
	  
   } else
   {       
       if( time == 1 )
           sprintf(buf, _("second") );
	 else 
	   sprintf(buf, _("%d seconds"), time );
   }
   return (buf);
}

/*
 * Convert ASCII input times to seconds.
 */
static int cvtatos(time_t time, char *units, time_t *seconds)
{
   if (bcmp(units, "second", 6) == 0)
      *seconds = time;
   else if (bcmp(units, "minute", 6) == 0)
      *seconds = time * 60;
   else if (bcmp(units, "hour", 4) == 0)
      *seconds = time * 60 * 60;
   else if (bcmp(units, "day", 3) == 0)
      *seconds = time * 24 * 60 * 60;
   else {
      printf(_("%s: bad units, specify:\n %s, %s, %s, or %s"), units,
             "days", "hours", "minutes", "seconds" );
      return (0);
   }
   return (1);
}

/*
 * This routine converts a name for a particular quota type to an identifier.
 */
static int getentry(char *name, int quotatype)
{
   struct passwd  *pw;
   struct group   *gr;
   
   switch (quotatype) {
      case USRQUOTA:
         if (pw = getpwnam(name))
            return (pw->pw_uid);
         fprintf(stderr, _("%s: no such user\n"), name);
         break;
      case GRPQUOTA:
         if (gr = getgrnam(name))
            return (gr->gr_gid);
         fprintf(stderr, _("%s: no such group\n"), name);
         break;
      default:
         fprintf(stderr, _("%d: unknown quota type\n"), quotatype);
         break;
   }
   return (-1);
}

/*
 * Collect the requested quota information.
 */
struct quotause *getprivs(long id, int quotatype, char local_only)
{
   struct mntent *mnt;
   struct quotause *qup, *quptail;
   FILE *fp;
   struct quotause *quphead;
   int qcmd, qupsize;
   char *qfpathname;
   static int warned = 0;
   extern int errno;
   const char *mnt_fsname;

   fp = setmntent(MOUNTED, "r");
   quphead = (struct quotause *) 0;
   qcmd = QCMD(Q_GETQUOTA, quotatype);
   while ((mnt = getmntent(fp)) != (struct mntent *) 0) {
      if (strcmp(mnt->mnt_type, MNTTYPE_NFS)) {
         if (hasmntopt(mnt, MNTOPT_NOQUOTA))
            continue;

         qfpathname = NULL;
         if (!hasquota(mnt, quotatype, &qfpathname))
            continue;

         qupsize = sizeof(*qup);
         if (qfpathname)
            qupsize += strlen(qfpathname);

         if ((qup = (struct quotause *) malloc(qupsize)) == (struct quotause *)NULL) {
            fprintf(stderr, _("edquota: out of memory\n"));
            exit(2);
         }
         memset((caddr_t)qup, 0, qupsize);

         mnt_fsname = get_device_name(mnt->mnt_fsname);
         if (quotactl(qcmd, mnt_fsname, id, (caddr_t) &qup->dqblk) != 0) {
            if ((errno == EOPNOTSUPP || errno == ENOSYS) && !warned) {
               warned++;
               fprintf(stderr, _("Warning: Quotas are not compiled into this kernel\n"));
               sleep(3);
            }

            if (read_quota_from_file(qfpathname, id, &qup->dqblk) != 0) {
               free(qup);
               continue;
            }
         }
      } else {
#if defined(RPC)
         if (local_only)
            continue;

         qupsize = sizeof(*qup);
         if ((qup = (struct quotause *) malloc(qupsize)) == (struct quotause *)NULL) {
            fprintf(stderr, _("edquota: out of memory\n"));
            exit(2);
         }
         memset((caddr_t)qup, 0, qupsize);

         mnt_fsname = get_device_name(mnt->mnt_fsname);
         rpc_rquota_get(mnt_fsname, id, quotatype, (struct dqblk *)&qup->dqblk);
#else
         continue;
#endif
      }
      
      if (qfpathname)
        strcpy(qup->qfname, qfpathname);
      strcpy(qup->fsname, mnt_fsname);
      if (quphead == NULL)
         quphead = qup;
      else
         quptail->next = qup;
      quptail = qup;
      qup->next = 0;
   }
   endmntent(fp);
   return (quphead);
}

/*
 * Store the requested quota information.
 */
int putprivs(long id, int quotatype, struct quotause *quplist)
{
   struct quotause *qup;
   struct mntent *mnt;
   int qcmd;
   FILE *fp;
   
   qcmd = QCMD(Q_SETQLIM, quotatype);
   for (qup = quplist; qup; qup = qup->next) {
      fp = setmntent(MOUNTED, "r");
      while ((mnt = getmntent(fp)) != (struct mntent *) 0) {
         const char * mnt_fsname = get_device_name(mnt->mnt_fsname);

         if (strcmp(qup->fsname, mnt_fsname) == 0){
            break;
         }
      }

      if (mnt == (struct mntent *) 0){
         continue;
      }

      if (strcmp(mnt->mnt_type, MNTTYPE_NFS)) {
         endmntent(fp);

         if (quotactl(qcmd, qup->fsname, id, (caddr_t) &qup->dqblk) == 0)
            continue;

         write_quota_to_file(qup->qfname, id, &qup->dqblk);
      } else {
         endmntent(fp);
#if defined(RPC) && defined(RPC_SETQUOTA)
         rpc_rquota_set(qcmd, qup->fsname, id, (caddr_t)&qup->dqblk, quotatype);
#endif
      }
   }
}

/*
 * Take a list of priviledges and get it edited.
 */
int editprivs(char *tmpfile)
{
   long omask;
   int pid, stat;
   extern char *getenv();

   omask = sigblock(sigmask(SIGINT) | sigmask(SIGQUIT) | sigmask(SIGHUP));
   if ((pid = fork()) < 0) {
      extern errno;

      perror("fork");
      return (0);
   }
   if (pid == 0) {
      char  *ed;

      sigsetmask(omask);
      setgid(getgid());
      setuid(getuid());
      if ((ed = getenv("VISUAL")) == (char *) 0)
         if ((ed = getenv("EDITOR")) == (char *) 0)
            ed = _PATH_VI;
      execlp(ed, ed, tmpfile, 0);
      perror(ed);
      exit(1);
   }
   waitpid(pid, &stat, 0);
   sigsetmask(omask);

   return (1);
}

/*
 * Convert a quotause list to an ASCII file.
 */
int writeprivs(struct quotause *quplist, int outfd, char *name, int quotatype)
{
   struct quotause *qup;
   FILE *fd;

   ftruncate(outfd, 0);
   lseek(outfd, 0, L_SET);
   if ((fd = fdopen(dup(outfd), "w")) == NULL) {
      exit(1);
   }

#if defined(ALT_FORMAT)
   fprintf(fd, _("Disk quotas for %s %s (%cid %d):\n"), 
           qfextension[quotatype], name,
           *qfextension[quotatype], getentry(name, quotatype));

   fprintf(fd, _("  Filesystem                   blocks       soft       hard     inodes     soft     hard\n"));

   for (qup = quplist; qup; qup = qup->next) {
      fprintf(fd, "  %-24s %10d %10d %10d %10d %8d %8d\n",
              qup->fsname,
              qup->dqblk.dqb_curblocks,
              qup->dqblk.dqb_bsoftlimit,
              qup->dqblk.dqb_bhardlimit,
              qup->dqblk.dqb_curinodes,
              qup->dqblk.dqb_isoftlimit, 
	      qup->dqblk.dqb_ihardlimit);
   }
#else
   fprintf(fd, _("Quotas for %s %s:\n"), qfextension[quotatype], name);
   for (qup = quplist; qup; qup = qup->next) {
      fprintf(fd, _("%s: %s %d, limits (soft = %d, hard = %d)\n"),
              qup->fsname, _("blocks in use:"),
              qup->dqblk.dqb_curblocks,
              qup->dqblk.dqb_bsoftlimit,
              qup->dqblk.dqb_bhardlimit);
      fprintf(fd, _("%s %d, limits (soft = %d, hard = %d)\n"),
              _("\tinodes in use:"), qup->dqblk.dqb_curinodes,
              qup->dqblk.dqb_isoftlimit, qup->dqblk.dqb_ihardlimit);
   }
#endif
   fclose(fd);
   return (1);
}

/*
 * Merge changes to an ASCII file into a quotause list.
 */
int readprivs(struct quotause *quplist, int infd)
{
   struct quotause *qup;
   FILE *fd;
   int cnt;
   char *cp;
   struct dqblk dqblk;
#if defined(ALT_FORMAT)
   char fsp[BUFSIZ], line[BUFSIZ];
#else
   char *fsp, line1[BUFSIZ], line2[BUFSIZ];
#endif

   lseek(infd, 0, L_SET);
   fd = fdopen(dup(infd), "r");
   if (fd == NULL) {
      fprintf(stderr, _("Can't re-read temp file!!\n"));
      return (0);
   }

#if defined(ALT_FORMAT)
   /*
    * Discard title lines, then read lines to process.
    */
   (void) fgets(line, sizeof(line), fd);
   (void) fgets(line, sizeof(line), fd);
	
   while (fgets(line, sizeof(line), fd)) {
      cnt = sscanf(line, "%s %d %d %d %d %d %d",
	           fsp, &dqblk.dqb_curblocks, &dqblk.dqb_bsoftlimit,
                   &dqblk.dqb_bhardlimit, &dqblk.dqb_curinodes,
                   &dqblk.dqb_isoftlimit, &dqblk.dqb_ihardlimit);

      if (cnt != 7) {
         fprintf(stderr, _("bad format:\n%s\n"), line);
         return (0);
      }
	   
      dqblk.dqb_curblocks  = dqblk.dqb_curblocks;
      dqblk.dqb_bsoftlimit = dqblk.dqb_bsoftlimit;
      dqblk.dqb_bhardlimit = dqblk.dqb_bhardlimit;

      for (qup = quplist; qup; qup = qup->next) {
         if (strcmp(fsp, qup->fsname))
            continue;
         /*
          * Cause time limit to be reset when the quota is
          * next used if previously had no soft limit or were
          * under it, but now have a soft limit and are over
          * it.
          */
         if (dqblk.dqb_bsoftlimit &&
            (qup->dqblk.dqb_curblocks >= dqblk.dqb_bsoftlimit) &&
            (qup->dqblk.dqb_bsoftlimit == 0 ||
             qup->dqblk.dqb_curblocks < qup->dqblk.dqb_bsoftlimit))
            qup->dqblk.dqb_btime = 0;

         if (dqblk.dqb_isoftlimit &&
            (qup->dqblk.dqb_curinodes >= dqblk.dqb_isoftlimit) &&
            (qup->dqblk.dqb_isoftlimit == 0 ||
             qup->dqblk.dqb_curinodes < qup->dqblk.dqb_isoftlimit))
            qup->dqblk.dqb_itime = 0;

         qup->dqblk.dqb_bsoftlimit = dqblk.dqb_bsoftlimit;
         qup->dqblk.dqb_bhardlimit = dqblk.dqb_bhardlimit;
         qup->dqblk.dqb_isoftlimit = dqblk.dqb_isoftlimit;
         qup->dqblk.dqb_ihardlimit = dqblk.dqb_ihardlimit;
         qup->flags |= FOUND;

         if (dqblk.dqb_curblocks != qup->dqblk.dqb_curblocks)
            fprintf(stderr, _("%s: cannot change current block allocation\n"), fsp);
	 if (dqblk.dqb_curinodes != qup->dqblk.dqb_curinodes)
            fprintf(stderr, _("%s: cannot change current inode allocation\n"), fsp);
      }
   }
#else
   /*
    * Discard title line, then read pairs of lines to process.
    */
   (void) fgets(line1, sizeof(line1), fd);
   while (fgets(line1, sizeof(line1), fd) != NULL &&
          fgets(line2, sizeof(line2), fd) != NULL) {
      if ((fsp = strtok(line1, " \t:")) == NULL) {
         fprintf(stderr, _("%s: bad format\n"), line1);
         return (0);
      }
      if ((cp = strtok((char *) 0, "\n")) == NULL) {
         fprintf(stderr, _("%s: %s: bad format\n"), fsp,
            &fsp[strlen(fsp) + 1]);
         return (0);
      }

      cnt = sscanf(cp, _(" blocks in use: %d, limits (soft = %d, hard = %d)"),
                   &dqblk.dqb_curblocks, &dqblk.dqb_bsoftlimit, &dqblk.dqb_bhardlimit);
      if (cnt != 3) {
         fprintf(stderr, _("%s:%s: bad format\n"), fsp, cp);
         return (0);
      }

      dqblk.dqb_curblocks = dqblk.dqb_curblocks;
      dqblk.dqb_bsoftlimit = dqblk.dqb_bsoftlimit;
      dqblk.dqb_bhardlimit = dqblk.dqb_bhardlimit;

      if ((cp = strtok(line2, "\n")) == NULL) {
         fprintf(stderr, _("%s: %s: bad format\n"), fsp, line2);
         return (0);
      }

      cnt = sscanf(cp, _("\tinodes in use: %d, limits (soft = %d, hard = %d)"),
                   &dqblk.dqb_curinodes, &dqblk.dqb_isoftlimit, &dqblk.dqb_ihardlimit);
      if (cnt != 3) {
         fprintf(stderr, _("%s: %s: bad format\n"), fsp, line2);
         return (0);
      }

      for (qup = quplist; qup; qup = qup->next) {
         if (strcmp(fsp, qup->fsname))
            continue;
         /*
          * Cause time limit to be reset when the quota is
          * next used if previously had no soft limit or were
          * under it, but now have a soft limit and are over
          * it.
          */
         if (dqblk.dqb_bsoftlimit &&
             qup->dqblk.dqb_curblocks >= dqblk.dqb_bsoftlimit &&
             (qup->dqblk.dqb_bsoftlimit == 0 ||
              qup->dqblk.dqb_curblocks <
              qup->dqblk.dqb_bsoftlimit))
            qup->dqblk.dqb_btime = 0;

         if (dqblk.dqb_isoftlimit &&
             qup->dqblk.dqb_curinodes >= dqblk.dqb_isoftlimit &&
             (qup->dqblk.dqb_isoftlimit == 0 ||
              qup->dqblk.dqb_curinodes <
              qup->dqblk.dqb_isoftlimit))
            qup->dqblk.dqb_itime = 0;

         qup->dqblk.dqb_bsoftlimit = dqblk.dqb_bsoftlimit;
         qup->dqblk.dqb_bhardlimit = dqblk.dqb_bhardlimit;
         qup->dqblk.dqb_isoftlimit = dqblk.dqb_isoftlimit;
         qup->dqblk.dqb_ihardlimit = dqblk.dqb_ihardlimit;
         qup->flags |= FOUND;

         if (dqblk.dqb_curblocks == qup->dqblk.dqb_curblocks &&
             dqblk.dqb_curinodes == qup->dqblk.dqb_curinodes)
            break;
         fprintf(stderr,
              _("%s: cannot change current allocation\n"), fsp);
         break;
      }
   }
#endif
   fclose(fd);

   /*
    * Disable quotas for any filesystems that have not been found.
    */
   for (qup = quplist; qup; qup = qup->next) {
      if (qup->flags & FOUND) {
         qup->flags &= ~FOUND;
         continue;
      }
      qup->dqblk.dqb_bsoftlimit = 0;
      qup->dqblk.dqb_bhardlimit = 0;
      qup->dqblk.dqb_isoftlimit = 0;
      qup->dqblk.dqb_ihardlimit = 0;
   }
   return (1);
}

/*
 * Convert a quotause list to an ASCII file of grace times.
 */
int writetimes(struct quotause *quplist, int outfd, int quotatype)
{
   struct quotause *qup;
   FILE *fd;

   ftruncate(outfd, 0);
   lseek(outfd, 0, L_SET);
   if ((fd = fdopen(dup(outfd), "w")) == NULL) {
      exit(1);
   }

#if defined(ALT_FORMAT)
   fprintf(fd, _("Grace period before enforcing soft limits for %ss:\n"), qfextension[quotatype]);
   fprintf(fd, _("Time units may be: days, hours, minutes, or seconds\n"));
   fprintf(fd, _("  Filesystem             Block grace period     Inode grace period\n"));

   for (qup = quplist; qup; qup = qup->next) {
      fprintf(fd, "  %-12s %22s ", qup->fsname, cvtstoa(qup->dqblk.dqb_btime));
      fprintf(fd, "%22s\n", cvtstoa(qup->dqblk.dqb_itime));
   }
#else
   fprintf(fd, _("Time units may be: days, hours, minutes, or seconds\n"));
   fprintf(fd, _("Grace period before enforcing soft limits for %ss:\n"),
      qfextension[quotatype]);
   for (qup = quplist; qup; qup = qup->next) {
      fprintf(fd, _("%s: block grace period: %s, "),
         qup->fsname, cvtstoa(qup->dqblk.dqb_btime));
      fprintf(fd, _("file grace period: %s\n"),
         cvtstoa(qup->dqblk.dqb_itime));
   }
#endif

   fclose(fd);
   return (1);
}

/*
 * Merge changes of grace times in an ASCII file into a quotause list.
 */
int readtimes(struct quotause *quplist, int infd)
{
   struct quotause *qup;
   FILE *fd;
   int cnt;
   char  *cp;
   time_t itime, btime, iseconds, bseconds;
#if defined(ALT_FORMAT)
   char fsp[BUFSIZ], bunits[10], iunits[10], line[BUFSIZ];
#else
   char *fsp, bunits[10], iunits[10], line1[BUFSIZ];
#endif

   lseek(infd, 0, L_SET);
   fd = fdopen(dup(infd), "r");
   if (fd == NULL) {
      fprintf(stderr, _("Can't re-read temp file!!\n"));
      return (0);
   }

#if defined(ALT_FORMAT)
   /*
    * Discard three title lines, then read lines to process.
    */
   (void) fgets(line, sizeof(line), fd);
   (void) fgets(line, sizeof(line), fd);
   (void) fgets(line, sizeof(line), fd);

   while (fgets(line, sizeof(line), fd) != NULL) {
      cnt = sscanf(line, "%s %d %s %d %s",
                   fsp, &btime, bunits, &itime, iunits);
      if (cnt != 5) {
         fprintf(stderr, _("bad format:\n%s\n"), line);
         return (0);
      }
#else
   /*
    * Discard two title lines, then read lines to process.
    */
   (void) fgets(line1, sizeof(line1), fd);
   (void) fgets(line1, sizeof(line1), fd);

   while (fgets(line1, sizeof(line1), fd) != NULL) {
      if ((fsp = strtok(line1, " \t:")) == NULL) {
         fprintf(stderr, _("%s: bad format\n"), line1);
         return (0);
      }
      if ((cp = strtok((char *) 0, "\n")) == NULL) {
         fprintf(stderr, _("%s: %s: bad format\n"), fsp,
            &fsp[strlen(fsp) + 1]);
         return (0);
      }
      cnt = sscanf(cp,
            _(" block grace period: %d %s file grace period: %d %s"),
              &btime, bunits, &itime, iunits);
      if (cnt != 4) {
         fprintf(stderr, _("%s:%s: bad format\n"), fsp, cp);
         return (0);
      }
#endif
      if (cvtatos(btime, bunits, &bseconds) == 0)
         return (0);
      if (cvtatos(itime, iunits, &iseconds) == 0)
         return (0);
      for (qup = quplist; qup; qup = qup->next) {
         if (strcmp(fsp, qup->fsname))
            continue;
         qup->dqblk.dqb_btime = bseconds;
         qup->dqblk.dqb_itime = iseconds;
         qup->flags |= FOUND;
         break;
      }
   }
   fclose(fd);

   /*
    * Reset default grace periods for any filesystems that have not been found.
    */
   for (qup = quplist; qup; qup = qup->next) {
      if (qup->flags & FOUND) {
         qup->flags &= ~FOUND;
         continue;
      }
      qup->dqblk.dqb_btime = 0;
      qup->dqblk.dqb_itime = 0;
   }
   return (1);
}

/*
 * Free a list of quotause structures.
 */
int freeprivs(struct quotause *quplist)
{
   struct quotause *qup, *nextqup;

   for (qup = quplist; qup; qup = nextqup) {
      nextqup = qup->next;
      free(qup);
   }
}
