/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include "gtm.h"
#include "file-data.h"
#include "dialogs.h"
#include "main-window.h"
#include "download-dirs.h"


/* Variables used in the selection of the toolbar style */
#define NUM_TOOLBAR_TYPES 3

static guint toolbar_type[NUM_TOOLBAR_TYPES] = {
    GTK_TOOLBAR_TEXT,
    GTK_TOOLBAR_ICONS,
    GTK_TOOLBAR_BOTH
};

static gchar *toolbar_type_label[NUM_TOOLBAR_TYPES] = {
    N_ ("Text"),
    N_ ("Icons"),
    N_ ("Both")
};

/* Variables used in the selection of the presume protocol */
#define NUM_PRESUME_PROTOCOLS 3

static Protocol presume_protocol[NUM_PRESUME_PROTOCOLS] = {
    PROTOCOL_NONE,
    PROTOCOL_HTTP,
    PROTOCOL_FTP
};

static gchar *presume_protocol_label[NUM_PRESUME_PROTOCOLS] = {
    N_ ("None"),
    N_ ("HTTP"),
    N_ ("FTP")
};


/* Structure that keeps the widgets containing the current preferences */
typedef struct 
{
    GtkWidget *pbox;                            /* GNOME Property Box */
    GtkWidget *use_proxy;                       /* Toggle button */
    GtkWidget *proxy_user;                      /* Entry */
    GtkWidget *proxy_pass;                      /* Entry */
    GtkWidget *proxy_http;                      /* Entry */
    GtkWidget *proxy_ftp;                       /* Entry */
    GtkWidget *interface;                       /* Entry */
    GtkWidget *chk_interf_time;                 /* Spin */
    GtkWidget *simul;                           /* Spin */
    GtkWidget *def_download_dir;                /* Entry */
    GtkWidget *file_list_refresh;               /* Spin */
    GtkWidget *unlim_retries;                   /* Toggle button */
    GtkWidget *num_retries;                     /* Spin */
    GtkWidget *use_passive_ftp;                 /* Toggle button */
    GtkWidget *toolbar_type[NUM_TOOLBAR_TYPES]; /* Toggle button */
    GtkWidget *addnew_changes_def_dl_dir;       /* Toggle button */
    GtkWidget *show_list_stat;                  /* Toggle button */
    GtkWidget *show_page_stat;                  /* Toggle button */
    GtkWidget *confirm_remove_incomplete;       /* Toggle button */
    GtkWidget *confirm_remove_completed;        /* Toggle button */
    GtkWidget *confirm_restart;                 /* Toggle button */
    GtkWidget *manual_stop_disables_auto_dl;    /* Toggle button */
    GtkWidget *confirm_dnd_url;                 /* Toggle button */
    GtkWidget *use_sound;                       /* Toggle button */
    GtkWidget *use_beep;                        /* Toggle button */
    GtkWidget *dir_list;                        /* CList */
    GtkWidget *stop_dl_on_link_loss;            /* Toggle button */
    GtkWidget *presume_protocol[NUM_PRESUME_PROTOCOLS]; /* Toggle button */
    GtkWidget *load_pause_auto_dl;              /* Toggle button */
    GtkWidget *auto_hide_dl_info_page;          /* Toggle button */
    GtkWidget *auto_hide_dl_info_page_timeout;  /* Spin */
} PrefDialogData;


/* Callback functions used to update the current preferences */
static void
apply_prefs_proxy (PrefDialogData *pref)
{
    /* Use proxy */
    gtm_pref.use_proxy = gtk_toggle_button_get_active (
        GTK_TOGGLE_BUTTON (pref->use_proxy));
    gnome_config_private_set_bool ("use_proxy", gtm_pref.use_proxy);
    gtm_pref.proxy_user = 
        g_strdup (gtk_entry_get_text (GTK_ENTRY (pref->proxy_user)));
    gnome_config_private_set_string ("proxy_user", gtm_pref.proxy_user);
    gtm_pref.proxy_pass = 
        g_strdup (gtk_entry_get_text (GTK_ENTRY (pref->proxy_pass)));
    gnome_config_private_set_string ("proxy_pass", gtm_pref.proxy_pass);
    gtm_pref.proxy_http = 
        g_strdup (gtk_entry_get_text (GTK_ENTRY (pref->proxy_http)));
    gnome_config_private_set_string ("proxy_http", gtm_pref.proxy_http);
    gtm_pref.proxy_ftp = 
        g_strdup (gtk_entry_get_text (GTK_ENTRY (pref->proxy_ftp)));
    gnome_config_private_set_string ("proxy_ftp", gtm_pref.proxy_ftp);
}

static void
apply_prefs_auto (PrefDialogData *pref)
{
    /* Interface to use */
    gtm_pref.interface = 
        g_strdup(gtk_entry_get_text (GTK_ENTRY (pref->interface)));
    gnome_config_set_string("interface", gtm_pref.interface);

    /* Check interface time */
    update_check_interface_timeout (gtk_spin_button_get_value_as_int (
        GTK_SPIN_BUTTON (pref->chk_interf_time)));

    /* Number of autodownload */
    gtm_pref.num_of_auto_download =
        gtk_spin_button_get_value_as_int (
            GTK_SPIN_BUTTON (pref->simul));
    gnome_config_set_int ("num_of_autodownload", 
                          gtm_pref.num_of_auto_download);

    /* Manual stop disables auto download */
    gtm_pref.manual_stop_disables_auto_dl =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->manual_stop_disables_auto_dl));
    gnome_config_set_bool ("manual_stop_disables_auto_dl",
                           gtm_pref.manual_stop_disables_auto_dl);

    /* Stop downloads when interface goes down */
    gtm_pref.stop_dl_on_link_loss =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->stop_dl_on_link_loss));
    gnome_config_set_bool ("stop_dl_on_link_loss",
                           gtm_pref.stop_dl_on_link_loss);

    /* Load pause auto download */
    gtm_pref.load_pause_auto_dl =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->load_pause_auto_dl));
    gnome_config_set_bool ("load_pause_auto_dl",
                           gtm_pref.load_pause_auto_dl);
}

static void
apply_prefs_download (PrefDialogData *pref)
{
    guint i;

    /* Default download directory */
    gtm_pref.def_download_dir = 
        g_strdup (gtk_entry_get_text (GTK_ENTRY (
            pref->def_download_dir)));
    gnome_config_set_string ("def_download_dir", 
                             gtm_pref.def_download_dir);

    /* Number of retries */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
        pref->unlim_retries)))
        gtm_pref.num_retries = 0;
    else
        gtm_pref.num_retries = gtk_spin_button_get_value_as_int (
            GTK_SPIN_BUTTON (pref->num_retries));
    gnome_config_set_int ("num_retries", gtm_pref.num_retries);

    /* Use passive FTP */
    gtm_pref.use_passive_ftp = gtk_toggle_button_get_active (
        GTK_TOGGLE_BUTTON (pref->use_passive_ftp));
    gnome_config_set_bool ("use_passive_ftp",
                           gtm_pref.use_passive_ftp);

    /* Presume protocol */
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        if (gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->presume_protocol[i]))) {
            gtm_pref.presume_protocol = presume_protocol[i];
            break;
        }
    }
    gnome_config_set_int ("presume_protocol", gtm_pref.presume_protocol);
}

static void
apply_prefs_confirmations (PrefDialogData *pref)
{
    /* New changes default download dir */
    gtm_pref.addnew_changes_def_dl_dir =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->addnew_changes_def_dl_dir));
    gnome_config_set_bool ("addnew_changes_def_dl_dir",
                           gtm_pref.addnew_changes_def_dl_dir);

    /* Confirm remove (not completed) */
    gtm_pref.confirm_remove_incomplete =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->confirm_remove_incomplete));
    gnome_config_set_bool ("confirm_remove_incomplete",
                           gtm_pref.confirm_remove_incomplete);


    /* Confirm remove (completed) */
    gtm_pref.confirm_remove_completed =
      gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->confirm_remove_completed));
    gnome_config_set_bool ("confirm_remove_completed", 
                           gtm_pref.confirm_remove_completed);

    /* Confirm restart */
    gtm_pref.confirm_restart =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->confirm_restart));
    gnome_config_set_bool ("confirm_restart",
                           gtm_pref.confirm_restart);

    /* Confirm drag-n-drop URL */
    gtm_pref.confirm_dnd_url =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->confirm_dnd_url));
    gnome_config_set_bool ("confirm_dnd_url",
                           gtm_pref.confirm_dnd_url);
}

static void
apply_prefs_directories (PrefDialogData *pref) {
    download_dirs_apply (pref->dir_list);
}

static void
apply_prefs_display (PrefDialogData *pref)
{
    guint i;

    /* Toolbar style */
    for (i = 0; i < NUM_TOOLBAR_TYPES; i++) {
        if (gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->toolbar_type[i]))) {
            gtm_pref.toolbar_style = toolbar_type[i];
            toolbar_update_style ();
            break;
        }
    }
    gnome_config_set_int ("toolbar_style", gtm_pref.toolbar_style);

    /* File list refresh interval */
    gtm_pref.file_list_refresh = 
        gtk_spin_button_get_value_as_int (
            GTK_SPIN_BUTTON (pref->file_list_refresh));
    gnome_config_set_int ("file_list_refresh", 
                          gtm_pref.file_list_refresh);

    /* Show list statistics */
    gtm_pref.show_list_stat = gtk_toggle_button_get_active (
        GTK_TOGGLE_BUTTON (pref->show_list_stat));
    gnome_config_set_bool ("show_list_stat", gtm_pref.show_list_stat);
    
    /* Show page statistics */
    gtm_pref.show_page_stat = gtk_toggle_button_get_active (
        GTK_TOGGLE_BUTTON (pref->show_page_stat));
    gnome_config_set_bool ("show_page_stat", gtm_pref.show_page_stat);

    /* Auto hide download info page */
    main_window_update_auto_hide_dl_info_page (
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (pref->auto_hide_dl_info_page)),
        gtk_spin_button_get_value_as_int (
            GTK_SPIN_BUTTON (pref->auto_hide_dl_info_page_timeout)));
}

static void
apply_prefs_sound (PrefDialogData *pref)
{
    /* Use sound */
    gtm_pref.use_sound =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->use_sound));
    gnome_config_set_bool ("use_sound", gtm_pref.use_sound);

    /* Use beep */
    gtm_pref.use_beep =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->use_beep));
    gnome_config_set_bool ("use_beep", gtm_pref.use_beep);
}

static void
apply_prefs (GtkWidget *widget, gint page_num, PrefDialogData *pref)
{
    /* If it's not an absolute and we don't use that */
    if (page_num == -1)
        return;

    /* Set configuration location */
    gnome_config_push_prefix ("/gtm/Preferences/");

    /* This function is called for each page changed so we need to apply prefs
     * for each page individually */
    switch (page_num) {
        case 0: /* Proxy */
            apply_prefs_proxy (pref);
            break;

        case 1: /* Auto */
            apply_prefs_auto (pref);
            break;
            
        case 2: /* Download */
            apply_prefs_download (pref);
            break;

        case 3: /* Directories */
            apply_prefs_directories (pref);
            
        case 4: /* Confirmations */
            apply_prefs_confirmations (pref);
            break;
            
        case 5: /* Display */
            apply_prefs_display (pref);
            break;

        case 6: /* Sound */
            apply_prefs_sound (pref);
            break;
    }    

    /* Unset the config location and sync the new configuration */
    gnome_config_pop_prefix ();
    gnome_config_sync ();
}

static void
destroy_prefbox (GtkWidget *widget, PrefDialogData **pref)
{
    gtk_widget_destroy ((*pref)->pbox);
    g_free (*pref);
    *pref = NULL;
}

static void
pref_modified_cb (GtkWidget *widget, PrefDialogData *pref)
{
    /* Change the sensitivity of the spin containing the number of retries */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->unlim_retries)))
        gtk_widget_set_sensitive (pref->num_retries, FALSE);
    else
        gtk_widget_set_sensitive (pref->num_retries, TRUE);

    /* Change the sensitivity of the proxy preferences */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->use_proxy))) {
        gtk_widget_set_sensitive (pref->proxy_user, TRUE);
        gtk_widget_set_sensitive (pref->proxy_pass, TRUE);
        gtk_widget_set_sensitive (pref->proxy_http, TRUE);
        gtk_widget_set_sensitive (pref->proxy_ftp, TRUE);
    } else {
        gtk_widget_set_sensitive (pref->proxy_user, FALSE);
        gtk_widget_set_sensitive (pref->proxy_pass, FALSE);
        gtk_widget_set_sensitive (pref->proxy_http, FALSE);
        gtk_widget_set_sensitive (pref->proxy_ftp, FALSE);
    }

    /* Change the sensitivity of the auto hide download info page timeout */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
        pref->auto_hide_dl_info_page)))
        gtk_widget_set_sensitive (pref->auto_hide_dl_info_page_timeout, TRUE);
    else
        gtk_widget_set_sensitive (pref->auto_hide_dl_info_page_timeout, FALSE);
    

    /* Tell to preferty box that data changed */
    gnome_property_box_changed (GNOME_PROPERTY_BOX (pref->pbox));
}


/* Functions to create notebook page and set it's contents to the program
 * current preferences */
static GtkWidget *
create_proxy_page (PrefDialogData *pref)
{
    GtkWidget *vbox, *hbox, *label;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Use proxy */
    pref->use_proxy = gtk_check_button_new_with_label (_ ("Use proxy"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->use_proxy, 
                        FALSE, FALSE, 0);

    /* Proxy username */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Username:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    pref->proxy_user = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), pref->proxy_user, FALSE, FALSE, 0);

    /* Proxy password */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Password:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    pref->proxy_pass = gtk_entry_new ();
    gtk_entry_set_visibility (GTK_ENTRY (pref->proxy_pass), FALSE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->proxy_pass, FALSE, FALSE, 0);

    /* Proxy for HTTP */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("HTTP:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    pref->proxy_http = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), pref->proxy_http, TRUE, TRUE, 0);

    /* Proxy for FTP */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("FTP:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    pref->proxy_ftp = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), pref->proxy_ftp, TRUE, TRUE, 0);

    /* Set the current option */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->use_proxy),
                                  gtm_pref.use_proxy);
    if (gtm_pref.use_proxy == FALSE) {
        gtk_widget_set_sensitive (pref->proxy_user, FALSE);
        gtk_widget_set_sensitive (pref->proxy_pass, FALSE);
        gtk_widget_set_sensitive (pref->proxy_http, FALSE);
        gtk_widget_set_sensitive (pref->proxy_ftp, FALSE);
    }
    gtk_entry_set_text (GTK_ENTRY (pref->proxy_user), 
                        gtm_pref.proxy_user);
    gtk_entry_set_text (GTK_ENTRY (pref->proxy_pass), 
                        gtm_pref.proxy_pass);
    gtk_entry_set_text (GTK_ENTRY (pref->proxy_http), 
                        gtm_pref.proxy_http);
    gtk_entry_set_text (GTK_ENTRY (pref->proxy_ftp), 
                        gtm_pref.proxy_ftp);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (pref->use_proxy), "toggled", 
                        GTK_SIGNAL_FUNC (pref_modified_cb), 
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->proxy_user), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->proxy_pass), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->proxy_http), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->proxy_ftp), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);

    return vbox;
}

static GtkWidget *
create_auto_page (PrefDialogData *pref)
{
    GtkWidget *vbox, *hbox, *label;
    GtkObject *adjustment;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Interface to connect */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Interface to connect:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    pref->interface = gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (hbox), pref->interface, FALSE, FALSE, 0);

    /* Check interface time */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Check interface time (ms):"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    adjustment = gtk_adjustment_new (1000, 100, 10000, 1, 100, 100);
    pref->chk_interf_time = 
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->chk_interf_time), 
                                 TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->chk_interf_time, TRUE, TRUE, 
                        0);
    
    /* Simultaneos downloads */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Simultaneous:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    adjustment = gtk_adjustment_new (1, 0, INT_MAX, 1, 100, 100);
    pref->simul = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->simul), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->simul, TRUE, TRUE, 0);

    /* Manual stop disables auto download */
    pref->manual_stop_disables_auto_dl = 
        gtk_check_button_new_with_label (_ (
            "Manual stop disables auto download"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->manual_stop_disables_auto_dl,
                        FALSE, FALSE, 0);

    /* Stop auto downloads on link loss */
    pref->stop_dl_on_link_loss =
        gtk_check_button_new_with_label (_ ("Stop download on link loss"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->stop_dl_on_link_loss,
                        FALSE, FALSE, 0);

    /* Load pause auto download */
    pref->load_pause_auto_dl =
        gtk_check_button_new_with_label (_ ("Load pause auto download state"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->load_pause_auto_dl,
                        FALSE, FALSE, 0);

    /* Set current preferences */
    gtk_entry_set_text(GTK_ENTRY(pref->interface), gtm_pref.interface);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->chk_interf_time), 
                               (gfloat) gtm_pref.chk_interf_time);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->simul), 
                               (gfloat) gtm_pref.num_of_auto_download);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (pref->manual_stop_disables_auto_dl),
        gtm_pref.manual_stop_disables_auto_dl);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (pref->stop_dl_on_link_loss),
        gtm_pref.stop_dl_on_link_loss);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (pref->load_pause_auto_dl),
        gtm_pref.load_pause_auto_dl);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (pref->interface), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->simul), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->chk_interf_time), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->manual_stop_disables_auto_dl),
                        "toggled", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->stop_dl_on_link_loss),
                        "toggled", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->load_pause_auto_dl),
                        "toggled", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);

    return vbox;
}

static GtkWidget *
create_download_page (PrefDialogData *pref)
{
    GtkWidget *vbox, *hbox, *label, *radio;
    GtkWidget *def_download_dir;
    GtkObject *adjustment;
    GSList *group;
    guint i;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
    
    /* Default download directory */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Default download dir:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    def_download_dir = gnome_file_entry_new ("DefDownloadDir", 
                                             _("Def Download Dir"));
    pref->def_download_dir = gnome_file_entry_gtk_entry (
        GNOME_FILE_ENTRY (def_download_dir));
    gnome_file_entry_set_directory (GNOME_FILE_ENTRY (def_download_dir), 1);
    gtk_box_pack_start (GTK_BOX (hbox), def_download_dir, TRUE, TRUE, 0);

    /* Number of retries */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Number retries:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    group = NULL;
    pref->unlim_retries = 
        gtk_radio_button_new_with_label (group, _ ("Unlimited"));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (pref->unlim_retries));
    gtk_box_pack_start (GTK_BOX (hbox), pref->unlim_retries, FALSE, FALSE, 0);
    radio = gtk_radio_button_new_with_label (group, _ ("Limited"));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (radio));
    gtk_box_pack_start (GTK_BOX (hbox), radio, FALSE, FALSE, 0);
    adjustment = gtk_adjustment_new (1, 1, INT_MAX, 1, 100, 100);
    pref->num_retries = 
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->num_retries), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->num_retries, TRUE, TRUE, 0);

    /* Use passive ftp */
    pref->use_passive_ftp = 
        gtk_check_button_new_with_label (_ ("Use passive FTP"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->use_passive_ftp, FALSE, FALSE,
                        0);

    /* Toolbar style */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Presume Protocol:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    group = NULL;
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        pref->presume_protocol[i] = 
            gtk_radio_button_new_with_label (group,
                                             _ (presume_protocol_label[i]));
        gtk_box_pack_start (GTK_BOX (hbox), pref->presume_protocol[i], FALSE, 
                            FALSE, 0);
        group = 
            gtk_radio_button_group (GTK_RADIO_BUTTON (
                pref->presume_protocol[i]));
    }

    /* Set current preferences */
    gtk_entry_set_text (GTK_ENTRY (pref->def_download_dir), 
                        gtm_pref.def_download_dir);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->use_passive_ftp),
                                  gtm_pref.use_passive_ftp);
    if (gtm_pref.num_retries == 0) {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->unlim_retries),
                                      TRUE);
        gtk_widget_set_sensitive (pref->num_retries, FALSE);
    } else {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->num_retries),
                                   (gfloat) gtm_pref.num_retries);
    }
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        if (gtm_pref.presume_protocol == presume_protocol[i]) {
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (pref->presume_protocol[i]), TRUE);
            break;
        }
    }

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (pref->def_download_dir), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->use_passive_ftp), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->unlim_retries), "clicked",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (radio), "clicked",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->num_retries), "changed",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++)
        gtk_signal_connect (GTK_OBJECT (pref->presume_protocol[i]), "clicked",
                            GTK_SIGNAL_FUNC (pref_modified_cb),
                            (gpointer) pref);

    return vbox;
}


static GtkWidget *
create_directories_page (PrefDialogData *pref)
{
    GtkWidget* widget;

    widget = download_dirs_list_create (&(pref->dir_list));
    gtk_signal_connect (GTK_OBJECT (pref->dir_list), "grab-focus",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    
    return widget;
}


static GtkWidget *
create_confirmations_page (PrefDialogData *pref)
{
    GtkWidget *vbox;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* New changes default download dir */
    pref->addnew_changes_def_dl_dir = 
        gtk_check_button_new_with_label (_ (
            "Dialog new or dialog add changes default download dir"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->addnew_changes_def_dl_dir, FALSE,
                        FALSE, 0);

    /* Confirm remove (not completed) */
    pref->confirm_remove_incomplete = 
        gtk_check_button_new_with_label (_ (
            "Confirm remove when transfer incomplete"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->confirm_remove_incomplete, FALSE,
                        FALSE, 0);

    /* Confirm remove (completed) */
    pref->confirm_remove_completed =
      gtk_check_button_new_with_label (_ (
            "Confirm remove completed transfers"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->confirm_remove_completed, FALSE,
                        FALSE, 0);
    
    /* Confirm restart */
    pref->confirm_restart = 
        gtk_check_button_new_with_label (_ (
            "Confirm restart"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->confirm_restart, FALSE,
                        FALSE, 0);

    /* Confirm drag-n-drop URL */
    pref->confirm_dnd_url = 
        gtk_check_button_new_with_label (_ (
            "Confirm drag and drop URL"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->confirm_dnd_url, FALSE,
                        FALSE, 0);

    /* Set current preferences */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->addnew_changes_def_dl_dir),
                                  gtm_pref.addnew_changes_def_dl_dir);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->confirm_remove_incomplete), gtm_pref.confirm_remove_incomplete);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->confirm_remove_completed), gtm_pref.confirm_remove_completed);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->confirm_restart), gtm_pref.confirm_restart);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->confirm_dnd_url), gtm_pref.confirm_dnd_url);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (pref->addnew_changes_def_dl_dir),
                        "toggled", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->confirm_remove_incomplete),
                        "toggled", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->confirm_remove_completed), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->confirm_restart), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->confirm_dnd_url), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);

    return vbox;
}

static GtkWidget *
create_display_page (PrefDialogData *pref)
{
    GtkWidget *vbox, *hbox, *label;
    GtkObject *adjustment;
    GSList *group;
    guint i;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
    
    /* Toolbar style */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Toolbar style:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    group = NULL;
    for (i = 0; i < NUM_TOOLBAR_TYPES; i++) {
        pref->toolbar_type[i] = 
            gtk_radio_button_new_with_label (group, _ (toolbar_type_label[i]));
        gtk_box_pack_start (GTK_BOX (hbox), pref->toolbar_type[i], FALSE, 
                            FALSE, 0);
        group = 
            gtk_radio_button_group (GTK_RADIO_BUTTON (pref->toolbar_type[i]));
    }

    /* File list refresh interval */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("File list refresh rate (ms):"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    adjustment = gtk_adjustment_new (1000, 100, 10000, 1, 100, 100);
    pref->file_list_refresh = 
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pref->file_list_refresh), 
                                 TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->file_list_refresh, TRUE, TRUE,
                        0);

    /* Show list statistics */
    pref->show_list_stat = 
        gtk_check_button_new_with_label (_ ("Show list statistics"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->show_list_stat, FALSE, FALSE, 0);

    /* Show list statistics */
    pref->show_page_stat = 
        gtk_check_button_new_with_label (_ ("Show page statistics"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->show_page_stat, FALSE, FALSE, 0);

    /* Auto hide download info page */
    pref->auto_hide_dl_info_page = 
        gtk_check_button_new_with_label (_ ("Auto hide download info page"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->auto_hide_dl_info_page, FALSE,
                        FALSE, 0);
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Auto hide download info page timeout (ms):"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    adjustment = gtk_adjustment_new (1000, 1000, 50000, 1000, 1000, 1000);
    pref->auto_hide_dl_info_page_timeout = 
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (
        pref->auto_hide_dl_info_page_timeout), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), pref->auto_hide_dl_info_page_timeout,
                        TRUE, TRUE, 0);

    /* Set current preferences */
    for (i = 0; i < NUM_TOOLBAR_TYPES; i++) {
        if (gtm_pref.toolbar_style == toolbar_type[i]) {
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (pref->toolbar_type[i]), TRUE);
            break;
        }
    }
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->file_list_refresh),
                               (gfloat) gtm_pref.file_list_refresh);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->show_list_stat),
                                  gtm_pref.show_list_stat);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->show_page_stat),
                                  gtm_pref.show_page_stat);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->auto_hide_dl_info_page), gtm_pref.auto_hide_dl_info_page);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (
        pref->auto_hide_dl_info_page_timeout),
                               (gfloat) 
                               gtm_pref.auto_hide_dl_info_page_timeout);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    for (i = 0; i < NUM_TOOLBAR_TYPES; i++)
        gtk_signal_connect (GTK_OBJECT (pref->toolbar_type[i]), "clicked",
                            GTK_SIGNAL_FUNC (pref_modified_cb),
                            (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->file_list_refresh), 
                        "changed", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->show_list_stat), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->show_page_stat), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->auto_hide_dl_info_page), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->auto_hide_dl_info_page_timeout), 
                        "changed", GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);

    return vbox;
}

static GtkWidget *
create_sound_page (PrefDialogData *pref)
{
    GtkWidget *vbox;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Use sound */
    pref->use_sound = 
        gtk_check_button_new_with_label (_ ("Use sound"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->use_sound, FALSE, FALSE, 0);

    /* Use beep */
    pref->use_beep = 
        gtk_check_button_new_with_label (_ ("Use beep"));
    gtk_box_pack_start (GTK_BOX (vbox), pref->use_beep, FALSE, FALSE, 0);

    /* Set current preferences */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->use_sound), gtm_pref.use_sound);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        pref->use_beep), gtm_pref.use_beep);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (pref->use_sound), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);
    gtk_signal_connect (GTK_OBJECT (pref->use_beep), "toggled",
                        GTK_SIGNAL_FUNC (pref_modified_cb),
                        (gpointer) pref);

    return vbox;
}


/* Function to create and run the preferences diaload box */
void
dialog_prefs (void)
{
    GtkWidget *label;
    static GnomeHelpMenuEntry help_entry = { "gtm", "index.html#PREF" };
    static PrefDialogData *pref_dialog_data;

    /* Only one preferences dialog box is allowed */
    if (pref_dialog_data != NULL) {
        gdk_window_raise (GTK_WIDGET (
            GNOME_DIALOG (pref_dialog_data->pbox))->window);
        return;
    }

    /* Create the gnome property box and set the common signals */
    pref_dialog_data = g_new (PrefDialogData, 1);
    pref_dialog_data->pbox = gnome_property_box_new ();
    gtk_window_set_title (GTK_WINDOW (pref_dialog_data->pbox),
                          _ ("Preferences"));
    gnome_dialog_set_parent (GNOME_DIALOG (pref_dialog_data->pbox),
                             GTK_WINDOW (gtm_app));
    gtk_signal_connect (GTK_OBJECT (pref_dialog_data->pbox), "destroy",
                        GTK_SIGNAL_FUNC (destroy_prefbox), 
                        (gpointer) &pref_dialog_data);
    gtk_signal_connect (GTK_OBJECT (pref_dialog_data->pbox), "apply",
                        GTK_SIGNAL_FUNC (apply_prefs), 
                        (gpointer) pref_dialog_data);
    gtk_signal_connect (GTK_OBJECT (pref_dialog_data->pbox), "help",
                        GTK_SIGNAL_FUNC (gnome_help_pbox_goto), &help_entry);

    /* Create the notebook pages */
    label = gtk_label_new (_ ("Proxy"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_proxy_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Auto"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_auto_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Download"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_download_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Directories"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_directories_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Confirmations"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_confirmations_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Display"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_display_page (pref_dialog_data), label);
    label = gtk_label_new (_ ("Sound"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (pref_dialog_data->pbox)->notebook), 
        create_sound_page (pref_dialog_data), label);

    /* We are ready to show it all */
    gtk_widget_show_all (pref_dialog_data->pbox);
}
