/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qpixmap.h>
#include <qbitmap.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qvaluelist.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>

#include <kdrawutil.h>
#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <kapp.h>
#include <kstyle.h>
#include <kconfig.h>
#include <kwindowlistmenu.h>
#include <kiconloader.h>
#include <kstddirs.h>

#include "taskbarapplet.h"
#include "taskbarapplet.moc"
#include "taskmanager.h"

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

extern "C"
{
    KPanelApplet* init(QWidget *parent, const QString& configFile)
    {
        KGlobal::locale()->insertCatalogue("ktaskbarapplet");
        TaskbarApplet *taskbar = new TaskbarApplet(configFile, KPanelApplet::Stretch,
                                                   KPanelApplet::Preferences, parent, "ktaskbarapplet");
        taskbar->setFrameStyle(QFrame::NoFrame);
        return taskbar;
    }
}

const int knArrowPreviewSize = 12;
const int knBtnSpacing = 1;

const int knDefaultScrolStep = 12;
const int knScrollBtnSize = 12;

TaskManager* task_manager = 0;

TaskbarApplet::TaskbarApplet(const QString& configFile, Type type, int actions,
                             QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name),
      m_pBoxLayout(0), m_pWindowListMenu(0), m_pWindowListBtn(0)
{
    // read settings
    KConfig c("ktaskbarrc", false, false);
    c.setGroup("General");

    // set font
    setFont(c.readFontEntry("taskbarFont"));

    bool bShowArrowButton = c.readBoolEntry("ShowWindowListBtn", true);
    if (bShowArrowButton) {
        m_pWindowListBtn = new ArrowPreviewButton(this);
        m_pWindowListBtn->setFocusPolicy(NoFocus);
        connect (m_pWindowListBtn, SIGNAL(clicked()), SLOT(slotExecWindowListMenu()));
        QToolTip::add(m_pWindowListBtn, i18n("Show Windows List"));
    }
    m_pScrollableArea = new ScrollableTaskbarArea(this);
}

TaskbarApplet::~TaskbarApplet()
{
}

void TaskbarApplet::popupDirectionChange( Direction d)
{
    if(!m_pWindowListBtn) return;

    ArrowType at = UpArrow;

    switch(d) {
        case Up:
            at = UpArrow;
            break;
        case Down:
            at = DownArrow;
            break;
        case Left:
            at = LeftArrow;
            break;
        case Right:
            at = RightArrow;
            break;
    }
    m_pWindowListBtn->setArrowType(at);
}

void TaskbarApplet::resizeEvent(QResizeEvent*)
{
    if (m_Orientation != orientation() || !m_pBoxLayout) {
        if (m_pBoxLayout)
        {
            delete m_pBoxLayout;
        }

        m_Orientation = orientation();
        m_pBoxLayout = new QBoxLayout(this, m_Orientation == Horizontal
                                      ? QBoxLayout::LeftToRight : QBoxLayout::TopToBottom);

        if (m_pWindowListBtn)
        {
            if (m_Orientation == Horizontal)
            {
                m_pWindowListBtn->setMinimumSize(knArrowPreviewSize, 1);
                m_pWindowListBtn->setMaximumSize(knArrowPreviewSize, 1024);
            }
            else
            {
                m_pWindowListBtn->setMinimumSize(1, knArrowPreviewSize);
                m_pWindowListBtn->setMaximumSize(1024, knArrowPreviewSize);
            }

            m_pBoxLayout->addWidget(m_pWindowListBtn);
            m_pBoxLayout->addSpacing(knBtnSpacing);
        }
        m_pBoxLayout->addWidget(m_pScrollableArea);

        m_pScrollableArea->setOrientation(orientation());

        m_pBoxLayout->activate();
        updateGeometry();
    }
}

void TaskbarApplet::slotExecWindowListMenu()
{
    if (!m_pWindowListMenu) {
        m_pWindowListMenu = new KWindowListMenu;
        connect(m_pWindowListMenu, SIGNAL(aboutToHide()), SLOT(slotWindowListMenuAboutToHide()));
    }

    m_pWindowListMenu->init();

    // calc popup menu position
    QPoint pos(m_pWindowListBtn->mapToGlobal(QPoint(0,0)));
    QSize sz(m_pWindowListBtn->size());
    if (orientation() == Horizontal) {
        if (popupDirection() == Down)
            pos.setY(pos.y() + sz.height());
        else
        {
            int y = pos.y() - m_pWindowListMenu->sizeHint().height();
            pos.setY(y < 0 ? 0 : y);
        }
    }
    else {
        if (popupDirection() == Right)
            pos.setX(pos.x() + sz.width());
        else
        {
            int x = pos.x() - m_pWindowListMenu->sizeHint().width();
            pos.setX(x);
        }
    }
    m_pWindowListMenu->exec(pos);
}

void TaskbarApplet::slotWindowListMenuAboutToHide()
{
    m_pWindowListBtn->setDown(false);
}

int TaskbarApplet::widthForHeight(int) const
{
    return 180; // a default size (we are a stretch applet so this does not really matter)
}

int TaskbarApplet::heightForWidth(int) const
{
    return 180; // a default size (we are a stretch applet so this does not really matter)
}

void TaskbarApplet::preferences()
{
    kapp->startServiceByDesktopName("kcmtaskbar");
}

void TaskbarApplet::slotPreferences()
{
    preferences();
}

TaskBar::TaskBar(QWidget *parent, const char *name)
    : QWidget(parent, name), _layout(0), _showall(false)
{
    // read settings
    KConfig c("ktaskbarrc", false, false);
    c.setGroup("General");
    _showall = c.readBoolEntry("ShowAllWindows", false);

    // setup and connect manager
    task_manager = new TaskManager(this);
    connect(task_manager, SIGNAL(taskAdded(Task*)), SLOT(taskAdded(Task*)));
    connect(task_manager, SIGNAL(taskRemoved(Task*)), SLOT(taskRemoved(Task*)));
    connect(task_manager, SIGNAL(startupAdded(Startup*)), SLOT(startupAdded(Startup*)));
    connect(task_manager, SIGNAL(startupRemoved(Startup*)), SLOT(startupRemoved(Startup*)));
    connect(task_manager, SIGNAL(desktopChanged(int)), SLOT(desktopChanged(int)));

    // register existant tasks
    QList<Task> tasks = task_manager->tasks();

    for (Task* t = tasks.first(); t!=0; t = tasks.next()) {
        TaskButton* b = new TaskButton(t, this);
        _taskButtons.append(b);
    }

    // register existant startups
    QList<Startup> startups = task_manager->startups();

    for (Startup* s = startups.first(); s!=0; s = startups.next()) {
        StartupButton* b = new StartupButton(s, this);
        _startupButtons.append(b);
    }

    // publish timer
    connect(&_publishTimer, SIGNAL(timeout()), this, SLOT(publishIconGeometry()));
}

TaskBar::~TaskBar()
{
}

void TaskBar::taskAdded(Task* task)
{
    if(!task) return;

    TaskButton* b = new TaskButton(task, this);
    _taskButtons.append(b);
    emit buttonCountChanged();
    resizeEvent(0L);
}

void TaskBar::taskRemoved(Task* task)
{
    if(!task) return;

    TaskButton* b;
    for (b = _taskButtons.first(); b != 0; b = _taskButtons.next()) {
        if (b->task() == task)
            break;
    }

    if (!b) return;

    _taskButtons.removeRef(b);
    delete(b);
    emit buttonCountChanged();
    resizeEvent(0L);
}

void TaskBar::startupAdded(Startup* startup)
{
    if(!startup) return;

    StartupButton* b = new StartupButton(startup, this);
    _startupButtons.append(b);
    emit buttonCountChanged();
    resizeEvent(0L);
}

void TaskBar::startupRemoved(Startup* startup)
{
    if(!startup) return;

    StartupButton* b;
    for (b = _startupButtons.first(); b != 0; b = _startupButtons.next()) {
        if (b->startup() == startup)
            break;
    }

    if (!b) return;

    _startupButtons.removeRef(b);
    delete(b);
    emit buttonCountChanged();
    resizeEvent(0L);
}

void TaskBar::desktopChanged(int)
{
    if(!_showall)
        resizeEvent(0L);
}

void TaskBar::resizeEvent(QResizeEvent*)
{
    if(_layout) {
        delete _layout;
	_layout = 0;
    }

    bool horiz = orientation() == Horizontal;
    int items, i, col, row, mod = 0;

    TaskButton *b;
    for(items=0, b = _taskButtons.first(); b; b = _taskButtons.next())
        items += (b->task()->onCurrentDesktop() || _showall)?1:0;

    items += _startupButtons.count();

    if(items){
	mod = (horiz && height() > 32) ? 2 : 1;
	_layout = new QGridLayout(this, mod ,1 );
    }

    for(i=0, col=0, row=0, b = _taskButtons.first(); b; b = _taskButtons.next()){
        if(b->task()->onCurrentDesktop() || _showall){
            b->show();
            if(horiz){
                b->setMinimumSize(24, 8);
                b->setMaximumSize(180, 180);
                _layout->addWidget(b, row, col);
		_layout->setColStretch(col, 100);
            }
            else{
                b->setMinimumSize(8, 24);
                _layout->addWidget(b, row, 0);
            }
            ++i;
            if ( horiz && ((i%mod) == 0) ) {
		row = 0;
		++col;
            }
            else
		++row;
        }
        else{
            b->move(0, 0);
            b->hide();
        }
    }

    for (StartupButton* b = _startupButtons.first(); b!=0; b = _startupButtons.next()) {

	b->show();

	if (horiz) {
	    b->setMinimumSize(24, 8);
	    b->setMaximumSize(180, 180);
	    _layout->addWidget(b, row, col);
	    _layout->setColStretch(col, 100);
	} else {
	    b->setMinimumSize(8, 24);
	    _layout->addWidget(b, row, 0);
	}

	++i;
	if ( horiz && ((i%mod) == 0) ) {
	    row = 0;
	    ++col;
	}
	else
	    ++row;
    }

    if(!_layout) return;

    if(horiz)
    	_layout->setColStretch(++col, 1);
    else
    	_layout->setRowStretch(++row, 1);
    _layout->activate();
    updateGeometry();
    _publishTimer.start(100, true);
}

void TaskBar::publishIconGeometry()
{
    QPoint p = mapToGlobal( QPoint(0,0) ); // roundtrip, don't do that too often
    for (TaskButton* b = _taskButtons.first(); b != 0; b = _taskButtons.next())
        b->publishIconGeometry( p );
}

void TaskBar::mousePressEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}
void TaskBar::mouseReleaseEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}
void TaskBar::mouseDoubleClickEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}
void TaskBar::mouseMoveEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}

void TaskBar::propagateMouseEvent(QMouseEvent* e)
{
    if ( !isTopLevel()  ) {
	QMouseEvent me(e->type(), mapTo( topLevelWidget(), e->pos() ),
		       e->globalPos(), e->button(), e->state() );
	QApplication::sendEvent( topLevelWidget(), &me );
    }
}

ArrowPreviewButton::ArrowPreviewButton(QWidget* parent, const char* name)
	: QToolButton(parent, name)
{
    setAutoRaise(true);
}

void ArrowPreviewButton::mousePressEvent( QMouseEvent * e)
{
    if ( e->button() != RightButton )
        QToolButton::mousePressEvent( e );
    else
        emit showMenu( e->globalPos() );
}

void ArrowPreviewButton::drawButtonLabel(QPainter * pPainter)
{
    int nX = 0, nY = 0;
    if (m_ArrowType == DownArrow)
    {
        nY = height() - 6;
        nX = (width() - 4) / 2 + 1;
    }
    else if (m_ArrowType == UpArrow)
    {
        nY = 6;
        nX = (width() - 4) / 2 + 1;
    }
    else if (m_ArrowType == RightArrow)
    {
        nX = width() - 6;
        nY = (height() - 4) / 2 + 1;
    }
    else // left arrow
    {
        nX = 6;
        nY = (height() - 4) / 2 + 1;
    }

    if ((isDown() || isOn()) && style().guiStyle() == Qt::WindowsStyle)
    {
        nX++; nY++;
    }


    QColorGroup colgrp = QColorGroup( colorGroup().text(),
                                      backgroundColor(),
                                      white, black, black,
                                      black, white );
    qDrawArrow( pPainter, m_ArrowType, Qt::WindowsStyle, false,
                nX, nY,
                0, 0,
                colgrp, false);
}

ScrollableTaskbarArea::ScrollableTaskbarArea(QWidget *parent, const char* name)
    : QWidget(parent, name)
{
    scroll_buttons_displayed = false;
    m_bArrowPixmapsCreated = false;
    m_nButtonsAreaSize = 0;
    scroll_area_min_size = scroll_area_max_size = 0;
    m_Orientation = Horizontal;

    m_pScroller = new FittsLawScrollView(this);
    m_pScroller->setResizePolicy(QScrollView::Manual);
    m_pScroller->setVScrollBarMode(QScrollView::AlwaysOff);
    m_pScroller->setHScrollBarMode(QScrollView::AlwaysOff);
    m_pScroller->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    m_pScroller->setLineWidth(1);
    m_pScroller->resize(0,0); //   initialize with nulls

    m_pTaskbarArea = new TaskBar(m_pScroller->viewport());
    connect(m_pTaskbarArea, SIGNAL(buttonCountChanged()), SLOT(taskbarButtonCountChanged()) );

    m_pScroller->addChild(m_pTaskbarArea);
    m_pTaskbarArea->move(0,0);
    // use same background for viewport
    m_pScroller->viewport()->setBackgroundColor(m_pTaskbarArea->backgroundColor());

    top_left_scroll_button = new QPushButton( i18n("Scroll Left"), this);
    top_left_scroll_button->hide();
    top_left_scroll_button->setAutoRepeat(true);
    top_left_scroll_button->setFocusPolicy(NoFocus);
    top_left_scroll_button->setMouseTracking(true);
    connect( top_left_scroll_button, SIGNAL(pressed()), SLOT(scrollTopLeft()) );

    bottom_right_scroll_button = new QPushButton( i18n("Scroll Right"), this);
    bottom_right_scroll_button->hide();
    bottom_right_scroll_button->setAutoRepeat(true);
    bottom_right_scroll_button->setFocusPolicy(NoFocus);
    bottom_right_scroll_button->setMouseTracking(true);
    connect( bottom_right_scroll_button, SIGNAL(pressed()), SLOT(scrollBottomRight()) );
}

void ScrollableTaskbarArea::setOrientation(Orientation orientation)
{
    m_Orientation = orientation;
    m_pTaskbarArea->setOrientation(orientation);

    if (m_bArrowPixmapsCreated)
        m_bArrowPixmapsCreated = false;
}

void ScrollableTaskbarArea::taskbarButtonCountChanged()
{
    layoutComponents();
}

void ScrollableTaskbarArea::resizeEvent(QResizeEvent *)
{
    layoutComponents();
}

void ScrollableTaskbarArea::layoutComponents()
{
    if (!scroll_buttons_displayed) {
        top_left_scroll_button->hide();
        bottom_right_scroll_button->hide();
    }

    QSize szTaskbarMin(0, 0);
    if (m_pTaskbarArea->layout())
        szTaskbarMin = m_pTaskbarArea->layout()->minimumSize();

    if (m_Orientation == Vertical) {
        top_left_scroll_button->setGeometry(0, height() - knScrollBtnSize*2, width(), knScrollBtnSize);
        bottom_right_scroll_button->setGeometry(0, height() - knScrollBtnSize, width(), knScrollBtnSize);

        scroll_area_max_size = height();
        scroll_area_min_size = scroll_area_max_size - knScrollBtnSize * 2;

        int nHg = szTaskbarMin.height();
        if (nHg < scroll_area_max_size)
        {
            nHg = scroll_area_max_size;
        }

        m_pScroller->resizeContents(width()-2, nHg);
        m_pTaskbarArea->resize(width()-2, nHg);
        m_nButtonsAreaSize = m_pTaskbarArea->height();

        if (m_nButtonsAreaSize <= scroll_area_max_size)
            m_pScroller->setGeometry(0, 0, width(), scroll_area_max_size);
        else
            m_pScroller->setGeometry(0, 0, width(), scroll_area_min_size);
    }
    else {
        top_left_scroll_button->setGeometry(width() - knScrollBtnSize*2, 0, knScrollBtnSize, height());
        bottom_right_scroll_button->setGeometry(width() - knScrollBtnSize, 0, knScrollBtnSize, height());

        scroll_area_max_size = width();
        scroll_area_min_size = scroll_area_max_size - knScrollBtnSize * 2;

        int nWd = szTaskbarMin.width()+2;	// need this correction to fit all taskbar
        if (nWd < scroll_area_max_size)
        {
            nWd = scroll_area_max_size;
        }

        m_pScroller->resizeContents(nWd, height()-2);
        m_pTaskbarArea->resize(nWd-2, height()-2);
        m_nButtonsAreaSize = nWd;//m_pTaskbarArea->width();

        if (m_nButtonsAreaSize <= scroll_area_max_size)
            m_pScroller->setGeometry(0, 0, scroll_area_max_size, height());
        else
            m_pScroller->setGeometry(0, 0, scroll_area_min_size, height());
    }

    buttonsAreaResized();
}

// This function shows or hides the scrolling buttons
// depending on the need for them.
void ScrollableTaskbarArea::showHideScrollButtons()
{
    if (m_Orientation == Vertical) {
        if (scroll_buttons_displayed)
        {
            // Hide scrolling buttons if the buttons_area fits
            // entirely in the buttons_scroll_area
            if (m_nButtonsAreaSize <= scroll_area_max_size)
            {
                if (m_pScroller->height() != scroll_area_max_size)
                {
                    m_pScroller->resize(m_pScroller->width(), scroll_area_max_size);
                }
                top_left_scroll_button->hide();
                bottom_right_scroll_button->hide();
                scroll_buttons_displayed = false;
            }
        }
        else
        {
            // Display scrolling buttons if the buttons_area doesn't fit
            // entirely in the buttons_scroll_area
            if (m_nButtonsAreaSize > scroll_area_max_size)
            {
                if (m_pScroller->height() != scroll_area_min_size)
                {
                    m_pScroller->resize(m_pScroller->width(), scroll_area_min_size);
                }
                top_left_scroll_button->show();
                bottom_right_scroll_button->show();
                scroll_buttons_displayed = true;
            }
        }
    }
    else {
        if (scroll_buttons_displayed)
        {
            // Hide scrolling buttons if the buttons_area fits
            // entirely in the buttons_scroll_area
            if (m_nButtonsAreaSize <= scroll_area_max_size)
            {
                if (m_pScroller->width() != scroll_area_max_size)
                {
                    m_pScroller->resize(scroll_area_max_size, m_pScroller->height());
                }
                top_left_scroll_button->hide();
                bottom_right_scroll_button->hide();
                scroll_buttons_displayed = false;
            }
        }
        else
        {
            // Display scrolling buttons if the buttons_area doesn't fit
            // entirely in the buttons_scroll_area
            if (m_nButtonsAreaSize > scroll_area_max_size)
            {
                if (m_pScroller->width() != scroll_area_min_size)
                {
                    m_pScroller->resize(scroll_area_min_size, m_pScroller->height());
                }
                top_left_scroll_button->show();
                bottom_right_scroll_button->show();
                scroll_buttons_displayed = true;
            }
        }
    }


    if (!m_bArrowPixmapsCreated)
    {
        m_bArrowPixmapsCreated = true;
        createArrowPixmaps();
    }
}

// This function enables/disables the scrolling buttons depending on the
// buttons area location in the scrolling area.
void ScrollableTaskbarArea::enableDisableScrollButtons()
{
    if(!scroll_buttons_displayed) return;

    if(m_Orientation == Vertical) {
        if(m_pScroller->contentsY() > 0)
            top_left_scroll_button->setEnabled(true);
        else
            top_left_scroll_button->setEnabled(false);

        if(m_pTaskbarArea->height() > m_pScroller->contentsY() + m_pScroller->height())
            bottom_right_scroll_button->setEnabled(true);
        else
            bottom_right_scroll_button->setEnabled(false);
    }
    else {
        if(m_pScroller->contentsX() > 0)
            top_left_scroll_button->setEnabled(true);
        else
            top_left_scroll_button->setEnabled(false);

        if(m_pTaskbarArea->width() >
           m_pScroller->contentsX() + m_pScroller->width())
            bottom_right_scroll_button->setEnabled(true);
        else
            bottom_right_scroll_button->setEnabled(false);
    }
}

// This function is called when the button area has been resized.
// It shows or hides the scrolling buttons and resizes the scrolling
// area to show the button area the best way.
void ScrollableTaskbarArea::buttonsAreaResized()
{
    showHideScrollButtons();
    enableDisableScrollButtons();
}

void ScrollableTaskbarArea::scrollTopLeft()
{
    scrollTopLeft(0);
}

void ScrollableTaskbarArea::scrollTopLeft(int step)
{
    if (step <= 0)
        step = knDefaultScrolStep;

    if (m_Orientation == Vertical)
        m_pScroller->scrollBy(0, -step);
    else
        m_pScroller->scrollBy(-step, 0);

    enableDisableScrollButtons();
}

void ScrollableTaskbarArea::scrollBottomRight()
{
    scrollBottomRight(0);
}

void ScrollableTaskbarArea::scrollBottomRight(int step)
{
    if (step <= 0)
        step = knDefaultScrolStep;

    if (m_Orientation == Vertical)
        m_pScroller->scrollBy(0, step);
    else
        m_pScroller->scrollBy(step, 0);

    enableDisableScrollButtons();
}

void arrow_on_pixmap(QWidget *w, QPixmap* pm, Qt::ArrowType rt)
{
    QColorGroup colgrp = QColorGroup( w->colorGroup().text(),
                                      w->backgroundColor(),
                                      Qt::white, Qt::black, Qt::black,
                                      Qt::black, Qt::white );
    QColorGroup colgrp2 = QColorGroup( Qt::color1, Qt::color0, Qt::color1, Qt::color1,
                                       Qt::color1, Qt::color1, Qt::color1 );
    QPainter paint;
    QPainter paint2;
    paint.begin(pm);
    paint2.begin(pm->mask());
    qDrawArrow( &paint, rt, Qt::WindowsStyle, false,
                pm->width()/2, pm->height()/2,
                0, 0,
                colgrp, false);
    qDrawArrow( &paint2, rt, Qt::WindowsStyle, false,
                pm->width()/2, pm->height()/2,
                0, 0,
                colgrp2, false);
    paint.end();
    paint2.end();
}

void ScrollableTaskbarArea::createArrowPixmaps()
{
    {
        // create the arrow pixmap for the scroll left button
        QPixmap pm;
        QBitmap bm;
        pm.resize(9, (m_Orientation ==
                      Horizontal)?top_left_scroll_button->height():top_left_scroll_button->width());
        bm.resize(pm.width(), pm.height());
        bm.fill(color0);
        pm.setMask(bm);
        QPixmap pm2 = pm;
        arrow_on_pixmap(this, &pm, LeftArrow);

        QWMatrix m;
        m.rotate((m_Orientation == Horizontal)?0:90);
        top_left_scroll_button->setPixmap(pm.xForm(m));
    }

    {
        // create the arrow pixmap for the scroll right button
        QPixmap pm;
        QBitmap bm;
        pm.resize(9, (m_Orientation == Horizontal)?bottom_right_scroll_button->height():bottom_right_scroll_button->width());
        bm.resize(pm.width(), pm.height());
        bm.fill(color0);
        pm.setMask(bm);
        arrow_on_pixmap(this, &pm, RightArrow);

        QWMatrix m;
        m.rotate((m_Orientation == Horizontal)?0:90);
        bottom_right_scroll_button->setPixmap(pm.xForm(m));
    }
}

TaskButton::TaskButton( Task* task, QWidget* parent, const char *name )
    : QPushButton(parent, name), _task(task), _popup(0), _deskpopup(0)
{
    setToggleButton(true);
    setAcceptDrops(true);

    connect(this, SIGNAL(clicked()), SLOT(toggled()));
    connect(_task, SIGNAL(changed()), SLOT(refresh()));

    // drag switch timer
    _dragSwitchTimer = new QTimer(this);
    connect(_dragSwitchTimer, SIGNAL(timeout()), SLOT(dragSwitchTimeout()));

    connect(this, SIGNAL(clicked()), SLOT(toggled()));

    refresh();
}

void TaskButton::dragEnterEvent(QDragEnterEvent*)
{
    // if a dragitem is held for over a taskbutton for two seconds,
    // activate corresponding window
    if(!_task->active())
	_dragSwitchTimer->start(2000, TRUE);
}

void TaskButton::dragLeaveEvent(QDragLeaveEvent*)
{
    _dragSwitchTimer->stop();
}

void TaskButton::dragSwitchTimeout()
{
    _task->activate();
}

void TaskButton::refresh()
{
    QString name = _task->visibleName();

    if(name != text()) {
        setText(name);
        QToolTip::add(this, name);
    }
    setOn(_task->active());
    update();
}

QSize TaskButton::sizeHint() const
{
    return (minimumSize());
}

void TaskButton::drawButton(QPainter *p)
{
    bool sunken = isOn() || isDown();

    if(kapp->kstyle()){
        kapp->kstyle()->drawKickerTaskButton(p, 0, 0, width(), height(),
                                             colorGroup(), QString::null,
                                             sunken, NULL);
    }
    else{
        style().drawPanel (p, 0, 0, width(), height(), colorGroup(), sunken,
                           1, &colorGroup().brush(QColorGroup::Background));
    }
    int pxWidth = 20;
    int textPos = 0;

    QRect br(style().buttonRect(0, 0, width(), height()));

    if (sunken)
        p->translate(1,1);

    // draw icon
    QPixmap pixmap = _task->pixmap();

    if (!pixmap.isNull()) {

        int dx = (pxWidth   - pixmap.width())   / 2;
        int dy = (height()  - pixmap.height())  / 2;

        p->drawPixmap(br.x() + dx, dy, pixmap);
        textPos += pxWidth;
    }

    QString s(text());

    QString parens( "(%1)" );
    if (_task->iconified())
        s = parens.arg(s);

    // modified icon
    static QString modStr = QString::fromUtf8("[") + i18n("modified") + QString::fromUtf8("]");
    int modStrPos = s.find(modStr);

    if (-1 != modStrPos) {

        // +1 because we include a space after the closing brace.
        s.remove(modStrPos, modStr.length()+1);

        QPixmap modPixmap = SmallIcon("modified");

        int dx = (pxWidth   - modPixmap.width())  / 2;
        int dy = (height()  - modPixmap.height()) / 2;

        p->drawPixmap(br.x() + textPos + dx, dy, modPixmap);
        textPos += pxWidth;
    }

    // draw text
    if (!s.isEmpty()) {

        if (p->fontMetrics().width(s) > width() - br.x() *2 - textPos) {

            int maxLen = width() - br.x() * 2 - textPos - p->fontMetrics().width("...");
            while ((!s.isEmpty()) && (p->fontMetrics().width(s) > maxLen))
                s.truncate(s.length() - 1);

            s.append("...");
        }

        if (_task->iconified())
            p->setPen(colorGroup().dark());
        else
            p->setPen(colorGroup().text());

        p->drawText(br.x() + textPos, -1, width() - textPos, height(), AlignLeft | AlignVCenter, s);
    }
}

void TaskButton::toggled()
{
    if (isOn())
        _task->activate();
    else if (!task_manager->isOnTop(_task))
        _task->raise();
    else
	_task->iconify();
}

void TaskButton::clientPopupAboutToShow()
{
    if (!_popup) return;

    _popup->setItemEnabled(RestoreOp, _task->iconified() || _task->maximized());
    _popup->setItemEnabled(IconifyOp, !_task->iconified());
    _popup->setItemEnabled(MaximizeOp, !_task->maximized());
    _popup->setItemEnabled(ToCurrentOp, !_task->onCurrentDesktop());
    _popup->setItemChecked(StayOnTopOp, _task->staysOnTop());
    _popup->setItemChecked(ShadeOp, _task->shaded());
}

void TaskButton::desktopPopupAboutToShow()
{
    if (!_deskpopup) return;

    _deskpopup->clear();
    _deskpopup->insertItem( i18n("&All Desktops"), 0 );
    _deskpopup->insertSeparator();

    if (_task->onAllDesktops())
        _deskpopup->setItemChecked(0, true);

    int id;
    for (int i = 1; i <= task_manager->numberOfDesktops(); i++) {
        id = _deskpopup->insertItem( QString("&%1 %2").arg(i).arg(task_manager->desktopName(i)), i );
        if ( !_task->onAllDesktops() && _task->desktop() == i )
            _deskpopup->setItemChecked( id, TRUE );
    }
}

void TaskButton::clientPopupActivated( int id )
{
    switch (id) {
        case MaximizeOp:
            _task->maximize();
            break;
        case IconifyOp:
            _task->iconify();
            break;
        case RestoreOp:
            _task->restore();
            break;
        case CloseOp:
            _task->close();
            break;
        case ToCurrentOp:
            _task->toCurrentDesktop();
            break;
        case StayOnTopOp:
            _task->stayOnTop(!_task->staysOnTop());
            break;
        case ShadeOp:
            _task->shade(!_task->shaded());
            break;
        default:
            break;
    }
}

void TaskButton::publishIconGeometry(const QPoint& global)
{
    QPoint p = global + geometry().topLeft();
    _task->publishIconGeometry(QRect(p.x(), p.y(), width(), height()));
}

void TaskButton::sendToDesktop(int desk)
{
    _task->toDesktop(desk);
}

void TaskButton::mousePressEvent(QMouseEvent *e)
{
    if ( e->button() == QMouseEvent::RightButton )
    {
        if (!_popup)
        {
            _popup = new QPopupMenu;
            _popup->setCheckable(true );
            connect(_popup, SIGNAL(aboutToShow()), this, SLOT(clientPopupAboutToShow()));
            connect(_popup, SIGNAL(activated(int)), this, SLOT(clientPopupActivated(int)));

            if (!_deskpopup) _deskpopup = new QPopupMenu(_popup);
            _deskpopup->setCheckable(true);
            connect(_deskpopup, SIGNAL(aboutToShow()), this, SLOT(desktopPopupAboutToShow()));
            connect(_deskpopup, SIGNAL(activated(int)), this, SLOT(sendToDesktop(int)));

            _popup->insertItem(i18n("Mi&nimize"), IconifyOp);
            _popup->insertItem(i18n("Ma&ximize"), MaximizeOp);
            _popup->insertItem(i18n("&Restore"), RestoreOp);

            _popup->insertSeparator();

            _popup->insertItem(i18n("&Shade"), ShadeOp);
            _popup->insertItem(SmallIcon("attach"), i18n("&Always On Top"), StayOnTopOp);

            _popup->insertSeparator();

            _popup->insertItem(SmallIcon("remove"), i18n("&Close"), CloseOp);

            _popup->insertSeparator();

            _popup->insertItem(i18n("To &Desktop"), _deskpopup);
            _popup->insertItem(i18n("&To Current Desktop"), ToCurrentOp);

        }
        _popup->popup(e->globalPos());
        return;
    }
    QPushButton::mousePressEvent(e);
}

StartupButton::StartupButton(Startup* startup, QWidget* parent, const char *name)
    : QPushButton(parent, name), _startup(startup), _frame(0)
{
    setText(_startup->text());
    QToolTip::add(this, i18n("Starting %1").arg(_startup->text()));

    // setup animation frames
    _anim.setAutoDelete(true);
    for (int i = 1; i < 9; i++)
	_anim.append(new QPixmap(locate("data", "kicker/pics/disk" + QString::number(i) + ".png")));

    // setup animation timer
    connect(&_animTimer, SIGNAL(timeout()), this, SLOT(animTimerFired()));
    _animTimer.start(100);
}

void StartupButton::animTimerFired()
{
    QPainter p(this);
    QPixmap *pix = _anim.at(_frame);

    p.drawPixmap(4, (height() - pix->height())/2, *pix);

    if (_frame == 7)
	_frame = 0;
    else
	_frame++;
}

QSize StartupButton::sizeHint() const
{
    return (minimumSize());
}

void StartupButton::drawButton(QPainter * p)
{
    bool sunken = isOn() || isDown();
    if(kapp->kstyle()){
        kapp->kstyle()->drawKickerTaskButton(p, 0, 0, width(), height(),
                                             colorGroup(), QString::null,
                                             sunken, NULL);
    }
    else
        style().drawPanel (p, 0, 0, width(), height(), colorGroup(), sunken,
                           1, &colorGroup().brush(QColorGroup::Background));

    int pxWidth = 20;

    QRect br(style().buttonRect(0, 0, width(), height()));

    if (sunken)
        p->translate(1,1);

    // draw text
    QString s(text());
    if (!s.isEmpty()){
	if (p->fontMetrics().width(s) > width() - br.x() * 2 - pxWidth) {

	    int maxLen = width() - br.x() * 2 - pxWidth - p->fontMetrics().width("...");

	    while ((!s.isEmpty()) && (p->fontMetrics().width(s) > maxLen))
		s.truncate(s.length() - 1);

	    s.append("...");
	}

        p->setPen(colorGroup().text());
	p->drawText(br.x() + pxWidth, 0, width() - pxWidth, height(), AlignLeft | AlignVCenter, s);
    }
}

void StartupButton::mousePressEvent(QMouseEvent *)
{
    //Don't call default handler ... we literally want to do nothing!
}

FittsLawScrollView::FittsLawScrollView(QWidget *parent, const char *name )
    : QScrollView(parent, name), _activeWidget(0) {}

bool FittsLawScrollView::x11Event(XEvent* e)
{
    // "Catch" XEvents which occur on the panel's edges, and redirect them
    // to the apropriate applets in the appletarea. This way the panel obeys
    // Fitt's law.
    switch ( e->type ) {

        case ButtonPress:
	{
	    // Only ButtonPress events which occur on the widget's frame need to
	    // be handled.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ) ||
		!rect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    // Determine the difference between the catched event's position and
	    // the position of the new event that we will contruct. The new
	    // position may not be on the frame, but has to be in
	    // contentsRect(). Using a difference is easier because it can be
	    // applied it to both the local and global positions.
	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // The widget which will be the destination of the new event.
	    QWidget* destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
                                                         e->xbutton.y_root + dy, true);

	    // If there is no such widget, we leave the event to Qt's event
	    // handler. If destWidget is equal to this widget pass it to Qt's
	    // event handler too to avoid nasty loops.
	    if (!destWidget || destWidget == this)
		return false;

	    // Now construct the new event.
	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child; // Not used
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    // Pretty obvious... Send the event.
	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Make the receiver our active widget. It will receive all events
	    // until the mouse button is released.
	    _activeWidget = destWidget;

	    // We're done with this event.
	    return true;
	}

	// The rest of the cases are more or less a duplication of the first
	// one with off course some minor differences. ButtonRelease is almost
	// the same as MotionNotify, but there's xbutton and xmotion.
        case ButtonRelease:
	{
	    // Handle events outside the widget's rectangle too, since the mouse
	    // can be grabbed.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // If there is a widget active it should receive the new event.
	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
                                                    e->xbutton.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    // The event's position can be outside the widget as well, so
	    // there's no need to adjust the position.
	    if (!rect().contains( QPoint(e->xbutton.x, e->xbutton.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Turn off the active widget.
	    _activeWidget = 0;

	    return true;
	}

        case MotionNotify:
	{
	    if (contentsRect().contains( QPoint(e->xmotion.x, e->xmotion.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xmotion.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xmotion.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xmotion.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xmotion.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xmotion.x_root + dx,
                                                    e->xmotion.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xmotion.x, e->xmotion.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xmotion.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xmotion.x + dx, e->xmotion.y + dy,
				  &ne.xmotion.x, &ne.xmotion.y, &child);
	    ne.xmotion.x_root = e->xmotion.x_root + dx;
	    ne.xmotion.y_root = e->xmotion.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}

        case EnterNotify:
        case LeaveNotify:
	{
	    if (contentsRect().contains(
		QPoint(e->xcrossing.x, e->xcrossing.y) ) ||
		!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xcrossing.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xcrossing.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xcrossing.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xcrossing.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xcrossing.x_root + dx,
                                                    e->xcrossing.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xcrossing.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xcrossing.x + dx, e->xcrossing.y + dy,
				  &ne.xcrossing.x, &ne.xcrossing.y, &child);
	    ne.xcrossing.x_root = e->xcrossing.x_root + dx;
	    ne.xcrossing.y_root = e->xcrossing.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}
        default:
            break;
    }

    return false;
}
