/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef KBABEL_H
#define KBABEL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kapp.h>
#include <kmainwindow.h>
#include <qstringlist.h>
#include <qlist.h>

#include "kbabelview.h"
#include "settings.h"

class KAccel;
class KAction;
class KLed;
class KProgress;
class QHBox;
class QLabel;
class QTimer;

class KBCatalog;
class KBabelPreferences;
class CatalogManager;
class DictionaryMenu;

/**
 * This class serves as the main window for KBabel.  It handles the
 * menus, toolbars, and status bars.
 *
 * @short Main window class
 * @author Matthias Kiefer <matthias.kiefer@gmx.de>
 * @version 0.1
 */
class KBabel : public KMainWindow
{
    Q_OBJECT
public:
    /**
     * Default Constructor
     */
    KBabel();

    /** use this contructor, if you just want to create a new view of an existing catalog*/
    KBabel(KBCatalog* catalog);

    /**
     * Default Destructor
     */
    virtual ~KBabel();

    void open(const KURL& url,bool newWindow);
    void openTemplate(const KURL& openURL,const KURL& saveURL,bool newWindow=false);

    void setSettings(SearchSettings,EditorSettings,SaveSettings,IdentitySettings);
    void setSettings(SearchSettings,EditorSettings);

protected:
    /**
     * Overridden virtuals for Qt drag 'n drop (XDND)
     */
    virtual void dragEnterEvent(QDragEnterEvent *event);
    virtual void dropEvent(QDropEvent *event);

    /**
     * This function is called when it is time for the app to save its
     * properties for session management purposes.
     */
    virtual void saveProperties(KConfig *);

    /**
     * This function is called when this app is restored.  The KConfig
     * object points to the session management config file that was saved
     * with @ref saveProperties
     */
    virtual void readProperties(KConfig *);

    virtual bool queryExit();
    virtual bool queryClose();

private slots:
    void quit();

    void open(const KURL& url);
    void fileOpen();
    void fileSave();
    void fileSaveAs();
    void fileNewView();
    KBabel* fileNewWindow();

    void addToRecentFiles(KURL url);
    
    void toggleEditMode();
    void optionsShowToolbar(bool on);
    void optionsShowNavBar(bool on);
    void optionsShowStatusbar(bool on);
    void optionsEditToolbars();
    void checkMenuItemTools();
    void checkMenuItemComments();
    void optionsConfigure();
    void optionsPreferences();

    /** opens the gettext info page */
    void gettextHelp();

    void firstEntryDisplayed(bool);
    void lastEntryDisplayed(bool);
    void fuzzyDisplayed(bool);
    void untranslatedDisplayed(bool);
    void faultyDisplayed(bool);
    void displayedEntryChanged(uint index);
    void setNumberOfTotal(uint number);
    void setNumberOfFuzzies(uint number);
    void setNumberOfUntranslated(uint number);
    void hasFuzzyAfterwards(bool);
    void hasFuzzyInFront(bool);
    void hasUntranslatedAfterwards(bool);
    void hasUntranslatedInFront(bool);
    void hasErrorAfterwards(bool);
    void hasErrorInFront(bool);

    void enableBackHistory(bool);
    void enableForwardHistory(bool);

    void enableUndo(bool);
    void enableRedo(bool);
    void enableStop(bool);

    void openCatalogManager();

    /**
    * prepare the window and the progressbar for showing
    * activity. message is displayed left to the progressbar
    * and max is the maximum number for the progressbar
    */
    void prepareProgressBar(QString message,int max);
    /**
    * resets the progressBar and enables the window
    */
    void clearProgressBar();

    void changeStatusbar(const QString& text);
    void clearStatusbarMsg();
    void changeCaption(const QString& text);
    void showModified(bool);

    /**
    * enables menu- and toolbar items that are always enabled when a cat is opened
    */
    void enableDefaults(bool readOnly);

    /**
    * updates the settings for the catalog manager in all preferences dialogs
    */
    void updateCatManSettings(CatManSettings);

    /**
    * sets the settings for the catalog manager and calls @ref updateCatManSettings
    */
    void setCatManSettings(CatManSettings);

    void setLedColor(const QColor& color);


    void buildDictMenus();

    /**
    * used when creating standard toggle actions, because I prefer
    * using signal toggled(bool)
    */
    void dummySlot(){}

signals:
    void catalogManagerSettingsChanged();
    
private:
    void init(KBCatalog* catalog);
    void setupActions();
    void setupStatusBar();
    void saveSettings();
    void restoreSettings();

private:
    KBabelView *m_view;

    KProgress* _progressBar;
    QLabel* _progressLabel;
    KLed* _fuzzyLed;
    KLed* _untransLed;
    KLed* _errorLed;

    QTimer *_statusbarTimer;

    KBabelPreferences* _prefDialog;

    static CatalogManager* _catalogManager;
    /**
    * holds settings for the catalogmanager
    * necessary, because the Catalogmanager is only build when
    * it is called
    */
    static CatManSettings _catManSettings;

    static QStringList _recentFiles;

    /**
    * used for updating preferences, that are common in
    * the whole application
    */
    static QList<KBabelPreferences> prefDialogs;


    // frequently used actions
    KAction* a_unsetFuzzy;
    KAction* a_prevFoU;
    KAction* a_nextFoU;
    KAction* a_prevFuzzy;
    KAction* a_nextFuzzy;
    KAction* a_prevUntrans;
    KAction* a_nextUntrans;

    KAccel *m_accel;

    DictionaryMenu *dictMenu;
    DictionaryMenu *selectionDictMenu;
    DictionaryMenu *configDictMenu;
    DictionaryMenu *editDictMenu;
    DictionaryMenu *aboutDictMenu;
};

#endif // KBABEL_H
