/*
 * kcmlisa.cpp
 *
 * Copyright (c) 2000 Alexander Neundorf <neundorf@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "kcmlisa.h"

#include <qlabel.h>
#include <klocale.h>
#include <qtooltip.h>

#include <qhbox.h>

LisaSettings::LisaSettings(const QString& config, QWidget *parent)
:QVBox(parent)
,m_config(config,false,true)
{
   QLabel *comment=new QLabel(i18n("If you have problems running or configuring LISa,\nread the documentation or contact Alexander Neundorf <neundorf@kde.org>"),this);
   setStretchFactor(comment,0);
   m_useNmblookup=new QCheckBox(i18n("Use nmblookup for searching"),this);
   QToolTip::add(m_useNmblookup,i18n("only hosts running smb servers will answer"));
   setStretchFactor(m_useNmblookup,0);

   QHBox *hbox=new QHBox(this);
   hbox->setSpacing(10);
   QLabel *label=new QLabel(i18n("Ping addresses"),hbox);
   QToolTip::add(label,i18n("enter ranges or whole subnets to ping, see README"));
   m_pingAddresses=new KRestrictedLine(hbox,"a","0123456789.-/;");
   QToolTip::add(m_pingAddresses,i18n("enter ranges or whole subnets to ping, see README"));
   setStretchFactor(hbox,0);

   m_pingNames=new KEditListBox(i18n("Check hosts"),this,"a",false);
   QToolTip::add(m_pingNames,i18n("The names of the hosts you want to check"));
   setStretchFactor(m_pingNames,1);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Allowed addresses"),hbox);
   QToolTip::add(label,i18n("usually your network address/subnet mask"));
   m_allowedAddresses=new KRestrictedLine(hbox,"a","0123456789./;");
   QToolTip::add(m_allowedAddresses,i18n("usually your network address/subnet mask"));
   setStretchFactor(hbox,0);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Broadcast network"),hbox);
   QToolTip::add(label,i18n("your network address/subnet mask"));
   m_broadcastNetwork=new KRestrictedLine(hbox,"a","0123456789./");
   QToolTip::add(m_broadcastNetwork,i18n("your network address/subnet mask"));
   setStretchFactor(hbox,0);

   label=new QLabel(i18n("The following settings are not that important, usually you can keep them."),this);
   setStretchFactor(label,0);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Wait (in hundredth seconds)"),hbox);
   QToolTip::add(label,i18n("how long to wait for answers from hosts"));
   m_firstWait=new QSpinBox(1,99,5,hbox);
   QToolTip::add(m_firstWait,i18n("how long to wait for answers from hosts"));
   setStretchFactor(hbox,0);

   m_secondScan=new QCheckBox(i18n("Enable second scan"),this);
   QToolTip::add(m_secondScan,i18n("check the hosts twice"));
   setStretchFactor(m_secondScan,0);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Wait on second scan (in hundredth seconds)"),hbox);
   QToolTip::add(label,i18n("how long to wait for answers from hosts"));
   m_secondWait=new QSpinBox(0,99,5,hbox);
   QToolTip::add(m_secondWait,i18n("how long to wait for answers from hosts"));
   setStretchFactor(hbox,0);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Update period (in seconds)"),hbox);
   QToolTip::add(label,i18n("search hosts after this number of seconds"));
   m_updatePeriod=new QSpinBox(30,1800,10,hbox);
   QToolTip::add(m_updatePeriod,i18n("search hosts after this number of seconds"));
   setStretchFactor(hbox,0);

   m_deliverUnnamedHosts=new QCheckBox(i18n("Report unnamed hosts"),this);
   QToolTip::add(m_deliverUnnamedHosts,i18n("report hosts without DNS names"));
   setStretchFactor(m_deliverUnnamedHosts,0);

   hbox=new QHBox(this);
   hbox->setSpacing(10);
   label=new QLabel(i18n("Send pings at once"),hbox);
   QToolTip::add(label,i18n("send this number of ping packets at once"));
   m_maxPingsAtOnce=new QSpinBox(8,1024,5,hbox);
   QToolTip::add(m_maxPingsAtOnce,i18n("send this number of ping packets at once"));
   setStretchFactor(hbox,0);

   connect(m_secondScan,SIGNAL(toggled(bool)),m_secondWait,SLOT(setEnabled(bool)));

   connect(m_pingAddresses,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));
   connect(m_allowedAddresses,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));
   connect(m_broadcastNetwork,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));

   connect(m_pingAddresses,SIGNAL(returnPressed()),this,SIGNAL(changed()));
   connect(m_allowedAddresses,SIGNAL(returnPressed()),this,SIGNAL(changed()));
   connect(m_broadcastNetwork,SIGNAL(returnPressed()),this,SIGNAL(changed()));

   connect(m_firstWait,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_secondWait,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_maxPingsAtOnce,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_secondScan,SIGNAL(toggled(bool)),this,SIGNAL(changed()));
   connect(m_updatePeriod,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_pingNames,SIGNAL(changed()),this,SIGNAL(changed()));
   connect(m_useNmblookup,SIGNAL(toggled(bool)),this,SLOT(changed()));

   setMargin(10);
   setSpacing(15);
}

void LisaSettings::load()
{
   int secondWait=m_config.readNumEntry("SecondWait",-1);
   if (secondWait<0)
   {
      m_secondWait->setValue(0);
      m_secondScan->setChecked(FALSE);
      m_secondWait->setEnabled(FALSE);
   }
   else
   {
      m_secondWait->setValue(secondWait);
      m_secondScan->setChecked(TRUE);
      m_secondWait->setEnabled(TRUE);
   };
   m_firstWait->setValue(m_config.readNumEntry("FirstWait",30));
   m_maxPingsAtOnce->setValue(m_config.readNumEntry("MaxPingsAtOnce",256));
   m_updatePeriod->setValue(m_config.readNumEntry("UpdatePeriod",300));
   m_pingAddresses->setText(m_config.readEntry("PingAddresses","192.168.0.0/255.255.255.0;192.168.100.0-192.168.100.254"));
   m_allowedAddresses->setText(m_config.readEntry("AllowedAddresses","192.168.0.0/255.255.0.0"));
   m_broadcastNetwork->setText(m_config.readEntry("BroadcastNetwork","192.168.0.0/255.255.0.0"));
   m_pingNames->insertStringList(m_config.readListEntry("PingNames",';'));
   int i=m_config.readNumEntry("SearchUsingNmblookup",1);
   m_useNmblookup->setChecked(i!=0);
};

void LisaSettings::save()
{
   if (m_secondScan->isChecked())
      m_config.writeEntry("SecondWait",m_secondWait->value());
   else
      m_config.writeEntry("SecondWait",-1);

   if (m_useNmblookup->isChecked())
      m_config.writeEntry("SearchUsingNmblookup",1);
   else
      m_config.writeEntry("SearchUsingNmblookup",0);

   m_config.writeEntry("FirstWait",m_firstWait->value());
   m_config.writeEntry("MaxPingsAtOnce",m_maxPingsAtOnce->value());
   m_config.writeEntry("UpdatePeriod",m_updatePeriod->value());
   m_config.writeEntry("PingAddresses",m_pingAddresses->text());
   m_config.writeEntry("AllowedAddresses",m_allowedAddresses->text());
   m_config.writeEntry("BroadcastNetwork",m_broadcastNetwork->text());
   QStringList writeStuff;
   for (int i=0; i<m_pingNames->count(); i++)
      writeStuff.append(m_pingNames->text(i));
   m_config.writeEntry("PingNames",writeStuff,';');

   m_config.sync();
};

#include "kcmlisa.moc"

