/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *  Copyright (C) 2000-2001 Matthew Tuck
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm.h"
#include "file-data.h"
#include "download-dirs.h"
#include "dialog-common-elements.h"

static GtkTargetEntry private_target_table[] = {
    { "text/uri-list", 0, 0 },
    { "text/plain", 0, 0 },
    { "x-url/http", 0, 0 },
    { "x-url/ftp", 0, 0 },
#ifdef ENABLE_SSL
    { "x-url/https", 0, 0 },
#endif
    { "_NETSCAPE_URL", 0, 0 }
};

/* Call back function used when the override download dir is changed to
 * change the sensitivity of download dir text box */
static void
override_dd_cb (GtkWidget *widget, DlPolicyWidgets *widgets)
{
    gtk_widget_set_sensitive (widgets->download_dir,
                              gtk_toggle_button_get_active (
                                  GTK_TOGGLE_BUTTON (widgets->override_dd)));
}

void
add_download_policy_widget (GtkWidget *dlg, DlPolicyWidgets *widgets)
{
    GtkWidget *hbox, *gtk_entry;
    
    /* Download directory override */
    widgets->override_dd =
        gtk_check_button_new_with_label (_ ("Override Download dir"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
                        widgets->override_dd, FALSE, FALSE, 0);

    /* Download directory */
    widgets->download_dir = gnome_file_entry_new ("NewDownloadDir", 
                                                  _ ("New Download Dir"));
    gtk_entry =
        gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (widgets->download_dir));
    gnome_file_entry_set_directory (GNOME_FILE_ENTRY (widgets->download_dir),
                                    1);
    gtk_entry_set_text (GTK_ENTRY (gtk_entry), gnome_util_user_home ());
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
                        widgets->download_dir, FALSE, FALSE, 0);
    gtk_widget_set_sensitive (widgets->download_dir, FALSE);

    /* Download policy checkboxes */
    hbox = gtk_hbox_new (FALSE, FALSE);

    widgets->use_proxy = 
	gtk_check_button_new_with_label (_ ("Use proxy"));
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widgets->use_proxy),
                                  gtm_pref.use_proxy);
    gtk_box_pack_start (GTK_BOX (hbox), widgets->use_proxy,
                        FALSE, FALSE, 0);

    widgets->use_auto_dl =
        gtk_check_button_new_with_label (_ ("Use auto download"));
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widgets->use_auto_dl),
                                  TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), widgets->use_auto_dl,
                        FALSE, FALSE, 0);

    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
                        hbox, FALSE, FALSE, 0);

    /* Set the signal handlers */
    gtk_signal_connect (GTK_OBJECT (widgets->override_dd), "toggled", 
                        GTK_SIGNAL_FUNC (override_dd_cb),
                        (gpointer) widgets);
}

void
get_dl_policy_data (DlPolicyData *data, DlPolicyWidgets *widgets)
{
    GtkWidget *gtk_entry;

    /* Get download directory override */
    data->override_dd =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
            widgets->override_dd));

    /* Get download directory data */
    gtk_entry =
        gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (widgets->download_dir));
    data->download_dir =
        gtk_entry_get_text (GTK_ENTRY (gtk_entry));

    /* Get the use proxy option data */
    data->use_proxy =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
            widgets->use_proxy));

    /* Get the use auto download option data */
    data->use_auto_dl = 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
	    widgets->use_auto_dl));
}

FileData *
dl_policy_file_data_create (gchar *url, DlPolicyData *data)
{
    if (data->override_dd)
        return file_data_create (url, data->download_dir, data->use_proxy,
                                 data->use_auto_dl);
    else {
	gchar *dir = NULL;

        download_dirs_suggest_dir (url, &dir);
	
	if (dir == NULL)
            dir = gtm_pref.def_download_dir;

        return file_data_create (url, dir, data->use_proxy, data->use_auto_dl);
    }
}

gchar *
download_dir_canonical_dup (gchar *download_dir)
{
    gint length;

    length = strlen (download_dir);
    if (length != 0 && download_dir[length - 1] != '/')
        return g_strconcat (download_dir, "/", NULL);
    else
        return g_strdup (download_dir);
}

/* If download dialog changes default download dir update the def dl dir */
void
change_default_download_dir (DlPolicyData *data)
{
    if (gtm_pref.addnew_changes_def_dl_dir && data->override_dd) {
        g_free (gtm_pref.def_download_dir);
	gtm_pref.def_download_dir =
            download_dir_canonical_dup (data->download_dir);
    }
}

/* Get the filename from a URL */
gchar *
get_filename (gchar *url)
{
    gchar *filename;

    filename = &url[strlen (url)];
    while (*filename != '/' && filename != url)
	filename--;
    filename++;

    return filename;
}

GtkTargetEntry *
target_table (void)
{
    return private_target_table;
}

gint
target_table_count (void)
{
    return sizeof(private_target_table) / sizeof (private_target_table[0]);
}

void
configure_ok_cancel_dialog (GnomeDialog *dlg, 
                            GtkSignalFunc ok_callback,
                            GtkSignalFunc cancel_callback,
                            gpointer data)
{
    gnome_dialog_set_close (dlg, FALSE);
    gnome_dialog_set_default (dlg, GNOME_OK);

    gnome_dialog_button_connect (dlg, GNOME_OK, ok_callback, data);
    gnome_dialog_button_connect (dlg, GNOME_CANCEL, cancel_callback, data);
}

gboolean
download_policy_check (DlPolicyData *data, GtkWidget *dlg)
{
    GtkWidget *error;

    g_return_val_if_fail (data != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);

    /* TODO: Check directory is valid instead of non-null */
    if (data->override_dd && strlen (data->download_dir) == 0) {
        error = gnome_error_dialog_parented (_ ("No override download "
                                                "directory entered."),
                                             GTK_WINDOW (dlg));
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
        return FALSE;
    } else {
	return TRUE;
    }
}

gboolean
download_not_dir_check (gchar *url, GtkWidget *dlg)
{
    GtkWidget *error;
    gchar *filename;

    g_return_val_if_fail (url != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);

    filename = get_filename (url);

    /* TODO: If filename == NULL instead of reminding the user that it is not
     * possible yet to do multiple downloads, when the group stuff is 
     * implemented this should enter a new dialog like dialog-multiple and 
     * ask there for download depth and get the available files to download. 
     * Then the user can remove what he doesn't want. */
    if (strlen (filename) == 0) {
        error = gnome_error_dialog_parented (_ ("Multiple download not "
                                             "implemented yet"),
                                             GTK_WINDOW (dlg));
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
        return FALSE;
    } else {
	return TRUE;
    }
}
