#include "ai.h"
#include "ai.moc"

#include <qlabel.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qgrid.h>

#include <kglobal.h>
#include <klocale.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kseparator.h>

#include "board.h"
#include "base/piece.h"
#include "base/defines.h"


//-----------------------------------------------------------------------------
AIPiece::AIPiece()
    : _p(0)
{}

AIPiece::~AIPiece()
{
    delete _p;
}

void AIPiece::init(const Piece *p, Board *b)
{
    _piece = p;
    _board = b;
    nbRot = p->nbConfigurations() - 1;
    if ( _p==0 )_p = new Piece;
    reset();
}

void AIPiece::reset()
{
	curPos = 0;
	curRot = 0;
    _p->copy(_piece);
    nbPos = _board->matrix().width() - _piece->width() + 1;
}

bool AIPiece::increment()
{
	curPos++;
	if ( curPos==nbPos ) {
		if ( curRot==nbRot ) {
            reset();
            return false;
        }
        _p->rotate(true, 0, 0);
        nbPos = _board->matrix().width() - _p->width() + 1;
        curRot++;
        curPos = 0;
	}
	return true;
}

bool AIPiece::place()
{
	if ( curRot==3 ) {
        if ( !_board->rotateRight() ) return false;
	} else for (uint i=0; i<curRot; i++)
        if ( !_board->rotateLeft() ) return false;
	curDec = curPos - _board->currentCol() - _p->minX();
    if ( curDec!=0 && _board->moveRight(curDec)!=(uint)kAbs(curDec) )
        return false;
	_board->dropDown();
    return !_board->isGameOver();
}

//-----------------------------------------------------------------------------
AI::AI(uint tTime, uint oTime)
    : timer(this), thinkTime(tTime), orderTime(oTime), stopped(false),
      board(0)
{
    elements.setAutoDelete(true);
    resizePieces(AISettingWidget::readThinkingDepth());
	connect(&timer, SIGNAL(timeout()), SLOT(timeout()));
}

void AI::resizePieces(uint size)
{
    uint oldSize = pieces.size();
    for (uint i=size; i<oldSize; i++) delete pieces[i];
    pieces.resize(size);
    for (uint i=oldSize; i<size; i++) pieces[i] = new AIPiece;
}

AI::~AI()
{
	delete board;
    resizePieces(0);
}

void AI::addElement(AIElement *e)
{
    uint s = elements.size();
    elements.resize(s+1);
    elements.insert(s, e);
    AISettingWidget::readElement(e);
}

void AI::initThink()
{
	board->copy(*main);
}

void AI::launch(Board *m)
{
    main = m;
	if ( board==0 )
        board = static_cast<Board *>(Factory::self()->createBoard(false, 0));

	pieces[0]->init(main->currentPiece(), board);
    if ( pieces.size()==2 )
        pieces[1]->init(main->nextPiece(), board);

    state = Thinking;
    hasBestPoints = false;
	startTimer();
}

void AI::stop()
{
    timer.stop();
    stopped = true;
}

void AI::start()
{
	if (stopped) {
		startTimer();
		stopped = false;
	}
}

void AI::startTimer()
{
	switch (state) {
	case Thinking:     timer.start(thinkTime, true); break;
	case GivingOrders: timer.start(orderTime, true); break;
	}
}

void AI::timeout()
{
	switch (state) {
	case Thinking:
		if ( think() ) state = GivingOrders;
		break;
	case GivingOrders:
		if ( emitOrder() ) return;
		break;
	}

	startTimer();
}

bool AI::emitOrder()
{
    if ( bestRot==3 ) {
		bestRot = 0;
		main->pRotateRight();
	} else if (bestRot) {
		bestRot--;
        main->pRotateLeft();
	} else if ( bestDec>0 ) {
		bestDec--;
        main->pMoveRight();
	} else if ( bestDec<0 ) {
		bestDec++;
        main->pMoveLeft();
	} else {
        main->pDropDown();
        return true;
    }
    return false;
}

bool AI::think()
{
    initThink();
    bool moveOk = true;
    for (uint i=0; i<pieces.size(); i++)
        if ( !pieces[i]->place() ) {
            moveOk = false;
            break;
        }

	if (moveOk) {
        double p = points();
        if ( !hasBestPoints || p>bestPoints
             || (p==hasBestPoints && random.getBool()) ) {
            hasBestPoints = true;
            bestPoints = p;
            bestDec = pieces[0]->dec();
            bestRot = pieces[0]->rot();
        }
    }

    for (uint i=pieces.size(); i>0; i--)
        if ( pieces[i-1]->increment() ) return false;
    return true;
}

double AI::points() const
{
	double pts = 0;
    for (uint i=0; i<elements.size(); i++)
        pts += elements.at(i)->points(*main, *board);
	return pts;
}

KSettingWidget *AI::createSettingWidget()
{
    KSettingWidget *sw = new AISettingWidget(elements);
    connect(sw->settings(), SIGNAL(hasBeenSaved()), SLOT(settingsChanged()));
    return sw;
}

void AI::settingsChanged()
{
    resizePieces(AISettingWidget::readThinkingDepth());
    for (uint i=0; i<elements.size(); i++)
        AISettingWidget::readElement(elements[i]);
    if ( timer.isActive() ) launch(main);
}

int AI::nbOccupiedLines(const Board &, const Board &current)
{
	return current.matrix().height() - current.nbClearLines();
}

int AI::nbHoles(const Board &, const Board &current)
{
	uint nb = 0;
	for (uint i=0; i<current.matrix().width(); i++) {
		for (int j=current.firstColumnBlock(i)-1; j>=0; j--) {
            Coord c(i, j);
			if ( current.matrix()[c]==0 ) nb++;
        }
	}
	return nb;
}

int AI::peakToPeak(const Board &, const Board &current)
{
	int min = current.matrix().height()-1;
	for (uint i=0; i<current.matrix().width(); i++)
		min = kMin(min, current.firstColumnBlock(i));
	return (int)current.firstClearLine()-1 - min;
}

int AI::mean(const Board &, const Board &current)
{
	uint mean = 0;
	for (uint i=0; i<current.matrix().width(); i++)
		mean += current.firstColumnBlock(i);
	return mean / current.matrix().width();
}

int AI::nbSpaces(const Board &main, const Board &current)
{
	int nb = 0;
	int m = mean(main, current);
	for (uint i=0; i<current.matrix().width(); i++) {
		int j = current.firstColumnBlock(i);
		if ( j<m ) nb += m - j;
	}
	return nb;
}

int AI::nbRemoved(const Board &main, const Board &current)
{
	return current.nbRemoved() - main.nbRemoved();
}

//-----------------------------------------------------------------------------
const char *AI_GRP = "AI";

AIElement::AIElement(const char *key, const char *description,
                     double cdef, double cmin, double cmax,
                     int tdef, int tmin, int tmax,
                     int (*function)(const Board &, const Board &))
    : _description(description), _tdef(tdef), _tmin(tmin), _tmax(tmax),
      _cdef(cdef), _cmin(cmin), _cmax(cmax), _key(key), _function(function)
{
	Q_ASSERT( tmax>=tmin && tdef<=tmax && tdef>=tmin );
	Q_ASSERT( cmax>=cmin && cdef<=cmax && cdef>=cmin );
    Q_ASSERT( function );
}

QString AIElement::triggerConfigKey() const
{
    return QString("%1 %2").arg(_key).arg("TRIG");
}

QString AIElement::coeffConfigKey() const
{
    return QString("%1 %2").arg(_key).arg("COEF");
}

bool AIElement::triggered() const
{
    return ( _tmin!=0 && _tmax!=0 );
}

double AIElement::points(const Board &main, const Board &current) const
{
	if ( _coeff==0 ) return 0;
	int v = _function(main, current);
	if ( triggered() && v<_trigger ) return 0;
	return v * _coeff;
}

//-----------------------------------------------------------------------------
KIntNumInput *createTrigger(KSettingCollection *col,
                            QWidget *parent, const AIElement *element)
{
    KIntNumInput *trig = new KIntNumInput(parent);
    trig->setRange(element->_tmin, element->_tmax);
    col->plug(trig, AI_GRP, element->triggerConfigKey(), element->_tdef);
    return trig;
}

KDoubleNumInput *createCoeff(KSettingCollection *col,
                             QWidget *parent, const AIElement *element)
{
    KDoubleNumInput *coeff = new KDoubleNumInput(parent);
    coeff->setRange(element->_cmin, element->_cmax, 1);
    col->plug(coeff, AI_GRP, element->coeffConfigKey(), element->_cdef);
    return coeff;
}

KIntNumInput *createThinkingDepth(KSettingWidget *sw)
{
    KIntNumInput *td = new KIntNumInput(sw);
    td->setRange(1, 2);
    sw->settings()->plug(td, AI_GRP, "thinking depth", 2);
    return td;
}

AISettingWidget::AISettingWidget(const QPtrVector<AIElement> &elements)
    : KSettingWidget(i18n("A.I."), "A.I.")
{
    QGridLayout *top = new QGridLayout(this, 3, 2, KDialog::marginHint(),
                                       KDialog::spacingHint());

    QLabel *label = new QLabel(i18n("Thinking depth"), this);
    top->addWidget(label, 0, 0);
    KIntNumInput *td = createThinkingDepth(this);
    top->addWidget(td, 0, 1);

    top->addRowSpacing(1, KDialog::spacingHint());

    QGrid *grid = new QGrid(2, this);
    top->addMultiCellWidget(grid, 2, 2, 0, 1);
    for (uint i=0; i<elements.size(); i++)
        addElement(elements.at(i), grid);
}

void AISettingWidget::readElement(AIElement *element)
{
    KSettingWidget sw;

    KIntNumInput *trig = createTrigger(sw.settings(), &sw, element);
    element->_trigger = sw.settings()->readValue(trig).toInt();

    KDoubleNumInput *coeff = createCoeff(sw.settings(), &sw, element);
    element->_coeff = sw.settings()->readValue(coeff).toDouble();
}

void AISettingWidget::addElement(const AIElement *element, QGrid *grid)
{
    QLabel *lab = new QLabel(i18n(element->_description), grid);
	lab->setFrameStyle(QFrame::Panel | QFrame::Plain);

	QVBox *vb = new QVBox(grid);
	vb->setMargin(KDialog::spacingHint());
	vb->setSpacing(KDialog::spacingHint());
	vb->setFrameStyle(QFrame::Panel | QFrame::Plain);
	if ( element->triggered() ) createTrigger(settings(), vb, element);
	createCoeff(settings(), vb, element);
}

uint AISettingWidget::readThinkingDepth()
{
    KSettingWidget sw;
    KIntNumInput *td = createThinkingDepth(&sw);
    return sw.settings()->readValue(td).toUInt();
}
