/*
    This file is part of KBugBuster.
    Copyright (c) 2002 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include "domprocessor.h"

#include "bugserver.h"
#include "packageimpl.h"
#include "bugimpl.h"
#include "bugdetailsimpl.h"

DomProcessor::DomProcessor( BugServer *server )
  : mServer( server )
{
}

DomProcessor::~DomProcessor()
{
}

Package::List DomProcessor::parsePackageList( const QDomElement &element )
{
  Package::List packages;

  QDomNode p;
  for ( p = element.firstChild(); !p.isNull(); p = p.nextSibling() ) {
    QDomElement bug = p.toElement();

    if ( bug.tagName() != "product" ) continue;

    QString pkgName = bug.attribute( "name" );
    uint bugCount = 999;
    Person maintainer;
    QString description;
    QStringList components;

    QDomNode n;
    for( n = bug.firstChild(); !n.isNull(); n = n.nextSibling() ) {
      QDomElement e = n.toElement();
      if ( e.tagName() == "descr" ) description= e.text().stripWhiteSpace();
      if ( e.tagName() == "component" ) components += e.text().stripWhiteSpace();
    }

    Package pkg( new PackageImpl( pkgName, description, bugCount, maintainer, components ) );

    if ( !pkg.isNull() ) {
        packages.append( pkg );
    }
  }

  return packages;
}

Bug::List DomProcessor::parseBugList( const QDomElement &element )
{
  Bug::List bugs;

  QDomNode p;
  for ( p = element.firstChild(); !p.isNull(); p = p.nextSibling() ) {
    QDomElement hit = p.toElement();
    if ( hit.tagName() != "hit" ) continue;

    QString title;
    Person submitter;
    QString bugNr;
    uint age = 0;
    Bug::Status status = Bug::StatusUndefined;
    Bug::Severity severity = Bug::SeverityUndefined;
    Person developerTodo;
    Bug::BugMergeList mergedList;

    QDomNode n;
    for( n = hit.firstChild(); !n.isNull(); n = n.nextSibling() ) {
      QDomElement e = n.toElement();

      if ( e.tagName() == "bugid" ) {
        bugNr = e.text();
      } else if ( e.tagName() == "status" ) {
        status = server()->bugStatus( e.text() );
      } else if ( e.tagName() == "descr" ) {
        title = e.text();
      }
    }

    Bug bug( new BugImpl( title, submitter, bugNr, age, severity,
                          developerTodo, status, mergedList ) );

    if ( !bug.isNull() ) {
      bugs.append( bug );
    }
  }
  
  return bugs;
}

BugDetails DomProcessor::parseBugDetails( const QDomElement &element )
{
  if ( element.tagName() != "bug" ) return BugDetails();

  BugDetailsPart::List parts;

  QString version;
  QString source;
  QString compiler;
  QString os;

  QDomNode n;
  for( n = element.firstChild(); !n.isNull(); n = n.nextSibling() ) {
    QDomElement e = n.toElement();
    if ( e.tagName() == "version" ) version = e.text().stripWhiteSpace();
    if ( e.tagName() == "op_sys" ) os = e.text().stripWhiteSpace();
    
    if ( e.tagName() == "long_desc" ) {

      Person sender;
      QString date;
      QString text;

      QDomNode n2;
      for( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
        QDomElement e2 = n2.toElement();
        if ( e2.tagName() == "who" ) {
          sender = Person::parseFromString( e2.text() );
        } else if ( e2.tagName() == "bug_when" ) {
          date = e2.text().stripWhiteSpace();
        } else if ( e2.tagName() == "thetext" ) {
          text = "<pre>" + wrapLines( e2.text() ) + "\n</pre>";
        }
      }

      parts.prepend( BugDetailsPart( sender, date, text ) );      
    }
  }

  BugDetails bugDetails( new BugDetailsImpl( version, source, compiler, os,
                                             parts ) );

  return bugDetails;
}

void DomProcessor::setPackageListQuery( KURL &url )
{
  url.setFileName( "xml.cgi" );
  url.setQuery( "?data=versiontable" );
}

void DomProcessor::setBugListQuery( KURL &url, const Package &product, const QString &component )
{
  url.setFileName( "xmlquery.cgi" );
  if ( component.isNull() )
      url.setQuery( "?user=freitag@suse.de&product=" + product.name() );
  else
      url.setQuery( "?user=freitag@suse.de&product=" + product.name() + "&component=" + component );
}

void DomProcessor::setBugDetailsQuery( KURL &url, const Bug &bug )
{
  url.setFileName( "xml.cgi" );
  url.setQuery( "?id=" + bug.number() );
}

QString DomProcessor::wrapLines( const QString &text )
{
  // Bug reporters don't wrap at 80... let's do it ourselves then
  QStringList lines = QStringList::split( '\n', text );
  //kdDebug() << lines.count() << " lines." << endl;
  for ( QStringList::Iterator it = lines.begin() ; it != lines.end() ; ++it )
  {
      QString line = *it;
      //kdDebug() << "BugDetailsJob::processNode IN line='" << line << "'" << endl;
      QString wrappedLine;
      while ( line.length() > 80 )
      {
          int breakPoint = line.findRev( ' ', 80 );
          //kdDebug() << "Breaking at " << breakPoint << endl;
          if( breakPoint == -1 ) {
              wrappedLine += line.left( 80 ) + '\n';
              line = line.mid( 80 );
          } else {
              wrappedLine += line.left( breakPoint ) + '\n';
              line = line.mid( breakPoint + 1 );
          }
      }
      wrappedLine += line; // the remainder
      //kdDebug() << "BugDetailsJob::processNode OUT wrappedLine='" << wrappedLine << "'" << endl;
      (*it) = wrappedLine;
  }

  return lines.join( "\n" );
}

/*
 * vim:sw=4:ts=4:et
 */
