/*****************************************************************************\
  djgenericvip.cpp : Implimentation for the generic VIP class

  Copyright (c) 2001-2002, Hewlett-Packard Co.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of Hewlett-Packard nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
  NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
  TO, PATENT INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
\*****************************************************************************/


#if defined(APDK_DJGENERICVIP) && defined (APDK_DJ9xxVIP)

#include "header.h"
#include "dj9xxvip.h"
#include "djgenericvip.h"

#ifdef APDK_PROTO
extern BOOL argPhotoTray;
#endif

APDK_BEGIN_NAMESPACE

/*
 *  All VIP printers that are released after the APDK release.
 *  This subclass is mainly there to allow any combination of
 *  installed pens.
 */

DJGenericVIP::DJGenericVIP (SystemServices* pSS, BOOL proto)
    : DJ9xxVIP (pSS, TRUE)
{

    if (!proto && IOMode.bDevID)
    {
        bCheckForCancelButton = TRUE;
        constructor_error = VerifyPenInfo ();
        CERRCHECK;
    }
	else
		ePen = BOTH_PENS;

/*
 *  For this printer class, allow any print mode to be compatible with
 *  any installed pen set. Printer is expected to do the right thing.
 *  Also, language has to be PCL3GUI because some printers may not work
 *  properly in plain PCL3 mode. ex. PhotoSmart 1xx family. This means
 *  no device text support.
 */

    for (int i = 0; i < (int) ModeCount; i++)
    {
        pMode[i]->bFontCapable = FALSE;
        for (int j = BLACK_PEN; j < MAX_COMPATIBLE_PENS; j++)
        {
            pMode[i]->CompatiblePens[j] = (PEN_TYPE) j;
        }
    }
}

BOOL DJGenericVIP::UseGUIMode (PrintMode* pPrintMode)
{
    return TRUE;
}

/*
 *  We don't really know beforehand the largest papersize the attached printer
 *  can support. The first reserved nibble after the VIP flag nibble contains
 *  this information. Only firmware version 4 or greater support this feature.
 */

PAPER_SIZE DJGenericVIP::MandatoryPaperSize ()
{
    BYTE    sDevIdStr[DevIDBuffSize];
    char    *pStr;

/*
 *  Try to get the DevID. Advance the pointer to the beginning of the status field.
 *  Currently, only 2 lower bits are used for reporting paper size. Meaning of these
 *  bit-values is as follows:
 *      00 - Allow any size
 *      01 - A6
 *      10 - Letter
 *      11 - B4
 */

    if (IOMode.bDevID && ((pSS->GetDeviceID (sDevIdStr, DevIDBuffSize, TRUE)) == NO_ERROR))
    {
        if ((pStr = strstr ((char *) sDevIdStr, ";S:")) && (pSS->GetVIPVersion ()) >= 3)
        {
            PAPER_SIZE  PaperSizes[4] = {UNSUPPORTED_SIZE, A6, LETTER, UNSUPPORTED_SIZE};
            return (PaperSizes[(pStr[17] & 0x03)]);
        }
    }
    return UNSUPPORTED_SIZE;
} //MandantoryPaperSize

BOOL DJGenericVIP::FullBleedCapable (PAPER_SIZE ps, float *xOverSpray, float *yOverSpray,
                                     float *fLeftOverSpray, float *fTopOverSpray)
{
    switch (ps)
    {
        case PHOTO_SIZE:
        case A6:
        case CARD_4x6:
        case OUFUKU:
        case HAGAKI:
        case A6_WITH_TEAR_OFF_TAB:
        {
            *xOverSpray = (float) 0.12;
            *yOverSpray = (float) 0.06;

            if (fLeftOverSpray)
                *fLeftOverSpray = (float) 0.05;
            if (fTopOverSpray)
                *fTopOverSpray  = (float) 0.03;
            return TRUE;
        }
        default:
            break;
    }

    *xOverSpray = (float) 0.18;
    *yOverSpray = (float) 0.12;
    if (fLeftOverSpray)
        *fLeftOverSpray = (float) 0.08;
    if (fTopOverSpray)
        *fTopOverSpray  = (float) 0.06;

    return TRUE;
}

PHOTOTRAY_STATE DJGenericVIP::PhotoTrayEngaged (BOOL bQueryPrinter)
{
#ifdef APDK_PROTO
    if (!IOMode.bDevID)
    {
        return (PHOTOTRAY_STATE) argPhotoTray;
    }
#endif
/*
 *	Malibu and its derivatives do not have a phototray sensor, so they don't
 *	report the current state of phototray engagement correctly. Always return
 *	UNKNOWN here.
    if (((PhotoTrayStatus (bQueryPrinter) & 9) == 9))
	{
		return TRUE;
	}
	return FALSE;
 */
	return UNKNOWN;
}

DRIVER_ERROR DJGenericVIP::VerifyPenInfo ()
{

    DRIVER_ERROR err = NO_ERROR;

    if (IOMode.bDevID == FALSE)
        return err;

	ePen = NO_PEN;

    err = ParsePenInfo (ePen);

    if(err == UNSUPPORTED_PEN) // probably Power Off - pens couldn't be read
    {

        // have to delay for Broadway or the POWER ON will be ignored
        if (pSS->BusyWait ((DWORD) 2000) == JOB_CANCELED)
        {
            return JOB_CANCELED;
        }

        DWORD length = sizeof (Venice_Power_On);
        err = pSS->ToDevice (Venice_Power_On, &length);
        ERRCHECK;

        err = pSS->FlushIO ();
        ERRCHECK;

        // give the printer some time to power up
        if (pSS->BusyWait ((DWORD) 2500) == JOB_CANCELED)
        {
            return JOB_CANCELED;
        }

        err = ParsePenInfo (ePen);
    }

    ERRCHECK;

/*
 *	If one or more pens are not installed, check the device id for ReserveMode
 *	capability for this printer. This info in in the flags nibble
 *	;S:<XX><toplid><supplylic><duplexer><phototray><in-1><in-2><banner><flags>
 *	flags : x1xx - reserve mode supported, x0xx - not supported
 */

    BYTE    sDevIdStr[DevIDBuffSize];
    char    *pStr;
	BOOL	bNeedAllPens = FALSE;

    if (iNumMissingPens > 0 && ((pSS->GetDeviceID (sDevIdStr, DevIDBuffSize, FALSE)) == NO_ERROR))
    {
        if ((pStr = strstr ((char *) sDevIdStr, ";S:")))
        {
			bNeedAllPens = !(*(pStr+12) & 0x04);
        }
	}

    while (ePen == NO_PEN || (bNeedAllPens && iNumMissingPens > 0))
    {
		if (ePen == NO_PEN || iNumMissingPens > 0)
		{
			pSS->DisplayPrinterStatus (DISPLAY_NO_PENS);

			if (pSS->BusyWait (500) == JOB_CANCELED)
			{
				return JOB_CANCELED;
			}
		}
        err =  ParsePenInfo (ePen);
        ERRCHECK;
    }

    pSS->DisplayPrinterStatus(DISPLAY_PRINTING);

    return NO_ERROR;
} //ParasePenInfo

void DJGenericVIP::AdjustModeSettings (BOOL bDoFullBleed, MEDIATYPE ReqMedia,
									   MediaType *medium, Quality *quality)
{

/*
 *	Malibu platform printers have a defect that prevents these printers from
 *	performing full-bleed and media detect functions correctly. Check for this
 *	case, i.e., full-bleed and media detect flags and explicitly set the
 *	media type and quality as requested by the application.
 */

	if (bDoFullBleed && *medium == mediaAuto)
	{

		if (ReqMedia == MEDIA_PHOTO)
		{
			*medium  = mediaGlossy;
			*quality = qualityPresentation;

		}
		else
		{
			*medium	 = mediaPlain;
			*quality = qualityNormal;

		}

	}

} // AdjustModeSettings
APDK_END_NAMESPACE

#endif // defined(APDK_DJGENERICVIP) && defined (APDK_DJ9xxVIP)
