/*
 *	$Id: cdeclient.cpp,v 1.11 2002/08/29 08:59:31 lunakl Exp $
 *	
 *	CDE KWin client - emulates the look and feel
 *	of dtwm, the CDE window manager.
 *	
 *	Copyright (c) 2000-2001, 2002
 *		Chris Lee       <lee@azsites.com>
 *		Lennart Kudling <kudling@kde.org>
 *      	Fredrik Hglund <fredrik@kde.org>
 *
 *	Originally based on the KStep client.
 *	
 *	Distributed under the terms of the BSD license.	
 */

#include <qdatetime.h>
#include <qlayout.h>
#include <qbutton.h>
#include <qdrawutil.h>
#include <qpainter.h>
#include <klocale.h>
#include <kconfig.h>
#include <kwin/workspace.h>
#include <kwin/options.h>

#include "cdeclient.h"

using namespace KWinInternal;

namespace CDE {

static int s_frameWidth = 5;
static bool titlebarButtonMode = true;
static bool coloredFrame = true;
static Qt::AlignmentFlags textAlignment = Qt::AlignHCenter;
static const int s_buttonSize = 19;

// These are the line segments for the X on the close button
static const QCOORD closeLLines[] =
    { 14,3, 12,3,  12,3, 9,6,  5,3, 3,3,  3,3, 3,5,
      3,5, 6,8, 6,9, 3,12,  3,12, 3,14 };

static const QCOORD closeDLines[] =
    { 6,4, 8,6,  14,4, 14,5,  14,5, 11,8,  11,9, 14,12,  14,12, 14,14,
      14,14, 12,14,  12,14, 9,11,  8,11, 5,14,  5,14, 4,14 };

// These are the line segments for the ? on the help button
static const QCOORD helpLLines[] =
    { 4,6, 4,5,  4,5, 6,3,  6,3, 9,3,  10,3, 11,4,
      9,7, 7,9,  7,9, 7,10,  7,14, 7,13,  8,12, 9,12  };

static const QCOORD helpDLines[] =
    {  5,7, 8,6,  12,5, 12,8,  12,8, 10,10,  10,10, 10,11,
       10,11, 8,11,  10,14, 10,13,  9,15, 8,15 };


// This question mark is taller than the one above and
// is positioned one pixel higher on the button
/*
static const QCOORD helpLLines[] =
    { 4,5, 4,4,  4,4, 6,2,  6,2, 9,2,  10,2, 11,3,
      9,6, 7,8,  7,9, 7,10,  7,13, 8,12,  8,12, 9,12  };

static const QCOORD helpDLines[] =
    {  5,6, 8,5,  12,4, 12,7,  12,7, 10,9,  10,10, 10,11,
       10,11, 8,11,  10,13, 9,14,  9,14, 8,14 };
*/
// Same as the one above but with a larger dot under
// the question mark
/*
static const QCOORD helpLLines[] =
    { 4,5, 4,4,  4,4, 6,2,  6,2, 9,2,  10,2, 11,3,
      9,6, 7,8,  7,9, 7,10,  7,14, 7,13,  8,12, 9,12  };

static const QCOORD helpDLines[] =
    {  5,6, 8,5,  12,4, 12,7,  12,7, 10,9,  10,10, 10,11,
       10,11, 8,11,  10,13, 10,14,  9,15, 8,15 };
*/

static void fixColorGroup(QColorGroup & colorGroup)
{
    QColor light = colorGroup.light();

    int hue, saturation, value;

    light.hsv(&hue, &saturation, &value);

    if (value < 128)
    {
      light.setHsv(hue, saturation, 128);
      colorGroup.setColor(QColorGroup::Light, light);
    }

    QColor dark = colorGroup.light();

    dark.hsv(&hue, &saturation, &value);

    if (value < 84)
    {
      dark.setHsv(hue, saturation, 84);
      colorGroup.setColor(QColorGroup::Dark, dark);
    }
}

CdeButton::CdeButton( CdeClient* parent, const char* name, int btnType, const QString& tip )
    : KWinButton( parent, name, tip ), m_btnType(btnType)
{
    setBackgroundMode( QWidget::NoBackground );
    setFixedSize( s_buttonSize, s_buttonSize );
    resize( s_buttonSize, s_buttonSize );
    m_parent = parent;
}

void CdeButton::reset()
{
    repaint( false );
}

void CdeButton::drawButton( QPainter* p )
{
    p->setBrush( options->color( Options::TitleBar, m_parent->isActive() ) );
    p->drawRect( 0, 0, s_buttonSize, s_buttonSize );

    QColorGroup colorGroup =
      options->colorGroup( Options::TitleBar, m_parent->isActive() );

    fixColorGroup(colorGroup);

    qDrawShadePanel( p, 0, 0, s_buttonSize, s_buttonSize,
    colorGroup, isDown() );

    switch ( m_btnType )
    {
        case (BtnMenu):
            qDrawShadePanel( p, 4, 7, 11, 4, colorGroup );
	break;
	case (BtnHelp):
	    p->setPen( colorGroup.light() );
	    p->drawLineSegments( QPointArray(16, helpLLines) );
	    p->setPen( colorGroup.dark() );
	    p->drawLineSegments( QPointArray(14, helpDLines) );
	break;
	case (BtnIconify):
	    qDrawShadePanel( p, 7, 7, 5, 5, colorGroup );
	break;
	case (BtnMax):
	    qDrawShadePanel( p, 4, 4, 11, 11, colorGroup, m_parent->isMaximized() );
	break;
	case (BtnClose):
	    p->setPen( colorGroup.light() );
	    p->drawLineSegments( QPointArray(15, closeLLines) );
	    p->setPen( colorGroup.dark() );
	    p->drawLineSegments( QPointArray(18, closeDLines) );
	break;
    }
}


CdeClient::CdeClient( Workspace* ws, WId w, QWidget* parent, const char* name )
    : Client( ws, w, parent, name, WStaticContents | WResizeNoErase
	| WRepaintNoErase )
{
    setBackgroundMode( QWidget::NoBackground );

    mainLayout = new QVBoxLayout( this );
    QBoxLayout* windowLayout = new QBoxLayout(0, QBoxLayout::LeftToRight, 0, 0, 0);
    titleLayout = new QBoxLayout(0, QBoxLayout::LeftToRight, 0, 0, 0);

    if ( s_frameWidth > 1 )
    {
       // the style normally draws a black frame around the window, so we
       // need 1 line of space for that in addition to the normal window frame
    	mainLayout->setMargin( s_frameWidth+1 );
    }
    else
    {
	// but if the frame is set to just 1 pixel we just draw the black frame instead of
	// the normal window frame, so no extra space is needed. if its 0 we don't draw anything.
    	mainLayout->setMargin( s_frameWidth );
    }
    
    mainLayout->addLayout( titleLayout );
    mainLayout->addLayout( windowLayout, 1 );

    windowLayout->addWidget( windowWrapper(), 1 );

    for ( int i=0; i < BtnCount; i++ )
        button[i] = NULL;

    addClientButtons( options->titleButtonsLeft() );

    titlebar = new QSpacerItem( 10, 16, QSizePolicy::Expanding, QSizePolicy::Minimum );
    titleLayout->addItem( titlebar );

    addClientButtons( options->titleButtonsRight() );

    titlebarPressed = false;
}

void CdeClient::addClientButtons( const QString& s )
{
    if ( s.length() > 0 )
        for ( unsigned int i = 0; i < s.length(); i++ )
        {
            switch( s[i].latin1() )
            {
                // Menu button
                case 'M':
                    if ( ! button[BtnMenu] )
                    {
                        button[BtnMenu] = new CdeButton( this, "menu", BtnMenu, i18n("Menu") );
                        connect( button[BtnMenu], SIGNAL(pressed()), SLOT(menuButtonPressed()) );
                        titleLayout->addWidget( button[BtnMenu] );
                    }
                    break;

                //Help button
                case 'H':
                    if ( providesContextHelp() && (! button[BtnHelp] ) )
                    {
                        button[BtnHelp] = new CdeButton( this, "help", BtnHelp, i18n("Help") );
                        connect( button[BtnHelp], SIGNAL( clicked() ), SLOT(contextHelp()) );
                        titleLayout->addWidget( button[BtnHelp] );
                    }
                    break;

		//Minimize button
                case 'I':
                    if ( (! button[BtnIconify] ) && isMinimizable() )
                    {
                        button[BtnIconify] = new CdeButton( this, "iconify", BtnIconify, i18n("Minimize") );
                        connect( button[BtnIconify], SIGNAL( clicked()), SLOT(iconify()) );
                        titleLayout->addWidget( button[BtnIconify] );
                    }
                    break;

                // Maximize button
                case 'A':
                    if ( (! button[BtnMax] ) && isMaximizable() )
                    {
                        button[BtnMax] = new CdeButton(this, "maximize", BtnMax, i18n("Maximize"));
                        connect( button[BtnMax], SIGNAL( clicked()), SLOT(maximize()) );
                        titleLayout->addWidget( button[BtnMax] );
                    }
                    break;

                // Close button
                case 'X':
                    if ( !button[BtnClose] && isCloseable())
                    {
                        button[BtnClose] = new CdeButton(this, "close", BtnClose, i18n("Close"));
                        connect( button[BtnClose], SIGNAL( clicked()), SLOT(closeWindow()) );
                        titleLayout->addWidget( button[BtnClose] );
                    }
            }
        }

}

void CdeClient::captionChange( const QString& )
{
    repaint( titlebar->geometry(), false );
}

void CdeClient::activeChange( bool )
{
    for ( int i=0; i < BtnCount; i++ )
    if ( button[i] ) button[i]->reset();

    repaint( false );
}

void CdeClient::maximizeChange(bool m)
{
    if ( button[BtnMax] ) {
	button[BtnMax]->setTipText(m ? i18n("Restore") : i18n("Maximize"));
	button[BtnMax]->repaint();	
    }
}

void CdeClient::menuButtonPressed()
{
    static QTime* t = NULL;
    static CdeClient* lastClient = NULL;
    if( t == NULL )
	t = new QTime;
    bool dbl = ( lastClient == this && t->elapsed() <= QApplication::doubleClickInterval());
    lastClient = this;
    t->start();
    if( !dbl )
    {
        QPoint pos = button[BtnMenu]->mapToGlobal(
    	    button[BtnMenu]->rect().bottomLeft() );
	workspace()->showWindowMenu( pos.x(), pos.y(), this );
	button[BtnMenu]->setDown(false);
    }
    else
	closeWindow();
}

void CdeClient::resizeEvent( QResizeEvent* e)
{
    Client::resizeEvent( e );

    if ( isVisibleToTLW() )
    {
        update( rect() );
        int dx = 0;
        int dy = 0;

	    if ( e->oldSize().width() != width() )
	       dx = 32 + QABS( e->oldSize().width() -  width() );

	    if ( e->oldSize().height() != height() )
	       dy = 8 + QABS( e->oldSize().height() -  height() );

	    if ( dy )
	       update( 0, height() - dy + 1, width(), dy );

        if ( dx )
        {
	    update( width() - dx + 1, 0, dx, height() );
	    update( QRect( QPoint(4,4),
	    titlebar->geometry().bottomLeft() - QPoint(1,0) ) );
	    update( QRect( titlebar->geometry().topRight(), QPoint( width() - 4, titlebar->geometry().bottom() ) ) );

	    // Titlebar needs no paint event
	    QApplication::postEvent( this, new QPaintEvent( titlebar->geometry(), false ) );
	}
    }
}

void CdeClient::paintEvent( QPaintEvent* )
{
    QPainter p( this );

    QColorGroup colorGroup;
    
    if ( coloredFrame )
	colorGroup = options->colorGroup( Options::TitleBar, isActive() );
    else
	colorGroup = options->colorGroup( Options::Frame, isActive() );

    fixColorGroup( colorGroup );

    QRect trect = titlebar->geometry();
    QRect wrect = windowWrapper()->geometry();
    QRect mrect = rect();

    if ( s_frameWidth > 0 )
    {
	// draw black frame:
	p.setPen( Qt::black );
	p.drawRect( mrect );
    }

    p.setPen( Qt::NoPen );
    p.setBrush( colorGroup.background() );

    if ( s_frameWidth > 1 )
    {
	bool shaded = isShade();

	// draw frame-background:
	p.drawRect( 1, 1,
    		mrect.width() - 2, s_frameWidth );
	p.drawRect( 1, mrect.height() - s_frameWidth - 1,
		mrect.width() - 2, s_frameWidth );
	p.drawRect( 1, s_frameWidth + 1, 
		s_frameWidth, mrect.height() - 2*s_frameWidth - 2 );
	p.drawRect( mrect.width() - s_frameWidth - 1, s_frameWidth + 1,
		s_frameWidth, mrect.height() - 2*s_frameWidth - 2 );

	if ( ! shaded )
	{
	    // draw left frame:
	    qDrawShadePanel( &p, 1, wrect.y(),
		s_frameWidth, wrect.height() - s_buttonSize,
		colorGroup );

	    // draw right frame:
	    qDrawShadePanel( &p, mrect.width() - s_frameWidth - 1, wrect.y(),
		s_frameWidth, wrect.height() - s_buttonSize,
		colorGroup );
	}
	
	// draw top frame:
	qDrawShadePanel( &p, s_frameWidth + s_buttonSize + 1, 1, 
	    wrect.width() - 2*s_buttonSize, s_frameWidth,
    	    colorGroup );
    
	// draw bottom frame:
	qDrawShadePanel( &p, s_frameWidth + s_buttonSize + 1, mrect.height() - s_frameWidth - 1,
	    wrect.width() - 2*s_buttonSize, s_frameWidth,
	    colorGroup );

	// draw light corner parts:
	p.setPen( colorGroup.light() );

	// tl corner:
	p.drawLine( 1, 1, s_frameWidth + s_buttonSize - 1, 1 );
	p.drawLine( 1, 1, 1, s_frameWidth + s_buttonSize - 1 );
	
	// tr corner:
	p.drawLine( mrect.width() - 3, 1, mrect.width() - s_frameWidth - s_buttonSize - 1, 1 );
	p.drawLine( mrect.width() - s_frameWidth - s_buttonSize - 1, 1,
	    mrect.width() - s_frameWidth - s_buttonSize - 1, s_frameWidth - 1 );
	p.drawLine( mrect.width() - s_frameWidth - 1, s_frameWidth, 
	    mrect.width() - s_frameWidth - 1, s_frameWidth + s_buttonSize - 1 );

	// br corner:
	if ( !shaded )
	{
	    p.drawLine( mrect.width() - 3, mrect.height() - s_frameWidth - s_buttonSize - 1,
		mrect.width() - s_frameWidth - 1, mrect.height() - s_frameWidth - s_buttonSize - 1 );
	}
	p.drawLine( mrect.width() - s_frameWidth - 1, mrect.height() - s_frameWidth - s_buttonSize,
	    mrect.width() - s_frameWidth - 1, mrect.height() - s_frameWidth - 1 );
	p.drawLine( mrect.width() - s_frameWidth - 2, mrect.height() - s_frameWidth - 1,
	    mrect.width() - s_frameWidth - s_buttonSize - 1, mrect.height() - s_frameWidth - 1 );
	p.drawLine( mrect.width() - s_frameWidth - s_buttonSize - 1, mrect.height() - s_frameWidth,
	    mrect.width() - s_frameWidth - s_buttonSize - 1, mrect.height() - 2 );
	
	// bl corner:
	if ( !shaded )
	{
	    p.drawLine( s_frameWidth-1, mrect.height() - s_frameWidth - s_buttonSize - 1, 
		2, mrect.height() - s_frameWidth - s_buttonSize - 1 );
	}
	p.drawLine( 1, mrect.height() - s_frameWidth - s_buttonSize - 1,
	    1, mrect.height() - 3 );
	p.drawLine( s_frameWidth + s_buttonSize - 1, mrect.height() - s_frameWidth - 1,
	    s_frameWidth + 1, mrect.height() - s_frameWidth - 1 );

	// draw dark corner parts:
	p.setPen( colorGroup.dark() );

	// tl corner:
	if ( !shaded )
	    p.drawLine( 1, s_frameWidth + s_buttonSize, s_frameWidth, s_frameWidth + s_buttonSize );
	p.drawLine( s_frameWidth, s_frameWidth + s_buttonSize - 1, s_frameWidth, s_frameWidth );
	p.drawLine( s_frameWidth + 1, s_frameWidth, s_frameWidth + s_buttonSize, s_frameWidth );
	p.drawLine( s_frameWidth + s_buttonSize, s_frameWidth, s_frameWidth + s_buttonSize, 1 );
	
	// tr corner:
	p.drawLine( mrect.width() - s_frameWidth - s_buttonSize - 1, s_frameWidth,
	    mrect.width() - s_frameWidth - 2, s_frameWidth );
	if ( !shaded )
	{
	    p.drawLine( mrect.width() - s_frameWidth - 1, s_frameWidth + s_buttonSize,
		mrect.width() - 2, s_frameWidth + s_buttonSize );
	}
	p.drawLine( mrect.width() - 2, s_frameWidth + s_buttonSize, mrect.width() - 2, 1 );

	// br corner:
	p.drawLine( mrect.width() - s_frameWidth - s_buttonSize - 1, mrect.height() - 2,
	    mrect.width() - 3, mrect.height() - 2 );
	p.drawLine( mrect.width() - 2, mrect.height() - 2,
	    mrect.width() - 2, mrect.height() - s_frameWidth - s_buttonSize - 2 );
	
	// bl corner:
	p.drawLine(
	1,
	mrect.height() - 2,
	s_frameWidth + s_buttonSize,
	mrect.height() - 2 );
	p.drawLine(
	s_frameWidth + s_buttonSize,
	mrect.height() - 3,
	s_frameWidth + s_buttonSize,
	mrect.height() - s_frameWidth - 1 );
	p.drawLine(
	s_frameWidth,
	mrect.height() - s_frameWidth - 1,
	s_frameWidth,
	mrect.height() - s_frameWidth - s_buttonSize - 1 );
    }

    p.setPen( Qt::NoPen );
    
    if ( !coloredFrame )
    {
	colorGroup = options->colorGroup( Options::TitleBar, isActive() );
	fixColorGroup( colorGroup );
	p.setBrush( colorGroup.background() );
    }
    
    // draw titlebar:
    p.drawRect( trect );
    qDrawShadePanel( &p, trect,
	colorGroup, titlebarPressed );

    // draw caption:	
    if ( titlebarPressed ) // move the caption right and down if the titlebar is pressed
	trect.moveBy( 1,1 ); // Note: the real Mwm doesn't actually do this
    
    p.setFont( options->font( true ) );
    p.setPen( options->color( Options::Font, isActive() ) );
    if ( p.fontMetrics().width( caption() ) > trect.width() - 6 )
    {
	// left align the text if its too wide to fit in the titlebar
	p.drawText( trect.x() + 3, trect.y(),
	    trect.width() - 6, trect.height(),
	    AlignLeft | AlignVCenter, caption() );
    }
    else
    {  
	// otherwise we'll draw it according to the user settings
	p.drawText( trect.x() + 3, trect.y(),
	    trect.width() - 6, trect.height(),
	    textAlignment | AlignVCenter, caption() );
    }

    // Draw a line behind the wrapped window to prevent having
    // unpainted areas when we're shaded.
    p.setPen( colorGroup.dark() );
    p.drawLine(s_frameWidth + 1, mrect.height() - s_frameWidth - 2, 
               mrect.width() - s_frameWidth - 2, mrect.height() - s_frameWidth - 2);
    
}

Client::MousePosition CdeClient::mousePosition( const QPoint& p ) const
{
    const int range = s_frameWidth + s_buttonSize;
    const int border = s_frameWidth + 1;

    MousePosition m = Nowhere;

    if ( ( p.x() > border && p.x() < width() - border )
         && ( p.y() > border && p.y() < height() - border ) )
        return Center;

    if ( p.y() < range && p.x() <= range)
        m = TopLeft;
    else if ( p.y() >= height()-range && p.x() >= width()-range)
        m = BottomRight;
    else if ( p.y() >= height()-range && p.x() <= range)
        m = BottomLeft;
    else if ( p.y() < range && p.x() >= width()-range)
        m = TopRight;
    else if ( p.y() < border )
        m = Top;
    else if ( p.y() >= height()-border )
        m = Bottom;
    else if ( p.x() <= border )
        m = Left;
    else if ( p.x() >= width()-border )
        m = Right;
    else
        m = Center;
    return m;
}

void CdeClient::mouseDoubleClickEvent( QMouseEvent * e )
{
    if ( titlebar->geometry().contains( e->pos() ) )
        workspace()->performWindowOperation( this, options->operationTitlebarDblClick() );
}

void CdeClient::mousePressEvent( QMouseEvent * e )
{
    if ( e->button() == LeftButton && titlebar->geometry().contains( e->pos() ) )
    {
	if ( titlebarButtonMode )
	{
	    titlebarPressed = true;
	    repaint( titlebar->geometry(), false );
	}
    }
    Client::mousePressEvent( e );
}

void CdeClient::mouseReleaseEvent( QMouseEvent * e )
{
    if ( e->button() == LeftButton && titlebarPressed )
    {
	titlebarPressed = false;
	repaint( titlebar->geometry(), false );
    }
    Client::mouseReleaseEvent( e );
}

static void readConfig()
{
    KConfig conf( "kwincderc" );
    
    conf.setGroup("General");
    coloredFrame = conf.readBoolEntry( "UseTitleBarBorderColors", true );
    titlebarButtonMode = conf.readBoolEntry( "TitlebarButtonMode", true );
    
    QString value = conf.readEntry( "TextAlignment", "AlignHCenter" );
    if ( value == "AlignLeft" )
	textAlignment = Qt::AlignLeft;
    else if ( value == "AlignHCenter" )
	textAlignment = Qt::AlignHCenter;
    else if ( value == "AlignRight" )
	textAlignment = Qt::AlignRight;
    
    s_frameWidth = conf.readUnsignedNumEntry( "FrameWidth", s_frameWidth );

    // Do not allow malicious users or corrupt config files to
    // go past the domain of the valid border sizes.
    if (s_frameWidth < 0)  s_frameWidth = 0;
    if (s_frameWidth > 10) s_frameWidth = 10;
}

};

extern "C"
{
    Client* allocate( Workspace* ws, WId w )
    {
        return ( new CDE::CdeClient(ws, w) );
    }
    void init()
    {
        CDE::readConfig();
    }
    void reset()
    {
        CDE::readConfig();
	
	// Tell kwin to reset all the clients
        Workspace::self()->slotResetAllClientsDelayed();
    }
    void deinit()
    {
    }
}

#include "cdeclient.moc"
