#!/usr/bin/perl -Tw

# NAME
#  mon.cgi
#
#
# DESCRIPTION
#  Web interface for the Mon resource monitoring system. mon.cgi
#  implements a significant subset of the Perl interface to Mon, which
#  allows administrators to quickly view the status of their network
#  and perform many common Mon tasks with a simple web client.
#
# Requires mon 0.38-16 and Mon::Client 0.7 for proper operation.
#
#
# AUTHORS
#  Originally by:
#   Arthur K. Chan <artchan@althem.com>
#  Based on the Mon program by Jim Trocki <trockij@transmeta.com>. 
#   http://www.kernel.org/software/mon/
#  Rewritten to support Mon::Client, mod_perl, taint mode,
#  authentication, the strict pragma, and other visual/functional 
#  enhancements by Andrew Ryan <andrewr@nam-shub.com>.
#  Downtime logging contributed by Martha H Greenberg <marthag@mit.edu>
#
# ----------------------------------------------------------------------
# $Id: mon.cgi,v 1.38 2000/02/19 02:29:43 andrewr Exp $
# ----------------------------------------------------------------------
#
#
# INSTRUCTIONS
# Install this cgi script to wherever your cgi-bin directory sits
# on your mon server. If you don't have a web server installed, try
# http://www.apache.org. This script hasn't been tested with any
# web server, although there is no reason it wouldn't work under
# any other web server as a CGI script.
#
# This script now runs cleanly under mod_perl (tested under apache 1.3.9,
# mod_perl 1.21), if you're running that. Global variables have not
# been eliminated but at least we're being careful now.
#
# This script also runs cleanly under taint mode, which is activated
# by using the -T switch for CGI scripts, and by using the directive
# "PerlTaintCheck On" in your httpd.conf file if you are running
# under mod_perl.
#
# If you want mon.cgi to carp less to your Apache error logs, run
# mon.cgi without the -w (warn) flag. It has been left in primarily
# for debugging purposes since this script is very much
# still in development.
#
# Modify the "Configurable Parameters" section below to customize it
# to your site's settings. 
#
# If you want to do a lot of the things that this script lets you do,
# and you don't want any authorization to be necessary, then
# you need to open up your auth.cf file to allow anyone to perform
# actions that you would like mon.cgi to perform. 
#
# No authentication might work in an environment where there are very
# few Mon users and they can all be trusted equally, or if you want
# to use mon.cgi in a sort of "read-only" capacity, where all users
# can list, for example, but no web users can enable/disable 
# monitoring and/or control the server in any way.
#
# Alternatively, if you want to use authentication, you need to have
# a working authentication setup with Mon from the command line
# before attempting to make authentication work with mon.cgi.
#
# Authentication is very flexible, and is trivial to implement in 
# mon.cgi, assuming you already have authentication working from 
# the command line. Just un-comment out the "$must_login line, change
# $app_secret to be something unique (VERY IMPORTANT!) and 
# mon will start requiring authentication for ALL commands.
#
# Authentication users should change their app secret on a regular
# basis if security is a concern. Actually, if security is a concern,
# don't run mon.cgi, because unless you use SSL, AND your monhost is
# on the same server as your web server, AND you use a short timeout
# on cookies, AND you change your app secret often and keep it
# VERY secure, you don't have a secure web system. But this simple 
# authentication mechanism is enough to keep most people happy.
#
#
# This script will require the CGI perl module. Available at any
# perl CPAN site. See http://www.perl.org for details. Oh, and don't
# forget Mon::Client, but the assumption is you are already running
# mon in some fashion and so you know this already.
#
# In addition, if you want to use the authentication piece of mon.cgi,
# you need to install the Crypt::TripleDES module, also available
# (tested w/ Crypt::TripleDES v0.24), and your browser needs to allow
# cookies (or else you need to be prepared to type in your username
# and password an awful lot!).
#
#
# BUGS
#  Probably many.
#  Send bugs/comments about this software to 
#  Andrew Ryan <andrewr@nam-shub.com>
#  Please include any output from your web server's error log that the
#  script might have output, this will help immensely in solving problems.
#  Also please include the versions of mon.cgi, Mon and Mon::Client you 
#  are using.
#

BEGIN {
    # Auto-detect if we are running under mod_perl or CGI.
    $USE_MOD_PERL = exists $ENV{'MOD_PERL'}
    ? 1 : 0;
    if ($USE_MOD_PERL) {
	# Use the cgi module and compile all methods at 
	# the beginning but only once
	use CGI qw (-compile :standard) ;				       
    } else {
	# Use the cgi module and compile all methods only 
	# when they are invoked via the autoloader.
	use CGI qw (:standard) ;
    }
    $CGI::POST_MAX=1024 * 100;  # max 100K posts
    $CGI::DISABLE_UPLOADS = 1;  # no uploads
}


# Configurable Parameters ----------------------------------------------
# Basic global vars
use Mon::Client;			       # mon client interface
use strict;			               # because strict is GOOD
use vars qw($organization $monadmin $logo $reload_time  $monhost $url $login_expire_time $cookie_name $cookie_path);
# Formatting-related global vars
use vars qw($BGCOLOR $BUTTONCOLOR $TEXTCOLOR $LINKCOLOR $VLINKCOLOR 
	    $greenlight_color $redlight_color $unchecked_color $disabled_color);
# Security-related global vars
use vars qw($must_login $app_secret %loginhash $des $has_prompted_for_auth $destroy_auth_cookie $default_username $default_password);
undef $must_login;                    #this must always be undef'd for mod_perl
undef $has_prompted_for_auth;        #this must always be undef'd for mod_perl
undef $destroy_auth_cookie;        #this must always be undef'd for mod_perl


$organization = "";	   # Organization name.
$monadmin = "BOFH\@your.domain";		   # Your e-mail address.
                                                   # note: must escape @ signs!
$logo = "";      # Company or mon logo.
$reload_time=180;				      # Seconds for page reload.
$monhost="localhost";				# Mon server hostname.

#$must_login = "yes";                 # Uncomment this out if you want 
                                     # authentication to be mandatory
                                     # for all connections to the mon server.
#!!! WARNING!!!!! You must change $app_secret to something unique to your site!
$app_secret = 'U78HlI%9@$j GJ 3ujfdluk-=+~3$%hJ6sWEe2'; # something unguessable
$default_username = "readonly" ;       # default username to try when
                                       # authenticating (should be a
                                       # low-privilege account)
$default_password = "public" ;         # default password to try when
                                       # authenticating (should be a
                                       # low-privilege account)
$login_expire_time = 900 ;            # Idle time, in seconds, until login
                                     # cookie is invalidated. Note that if
                                     # $login_expire_time < $reload_time,
                                     # you will not be able to "idle" and
                                     # authentication will be required much
                                     # more often.
$cookie_name = "mon-cookie";           #name of cookie given to browser for auth
$cookie_path = "/";                  # path for auth cookie
                                     # Set this to "" for auto-path set

$BGCOLOR = "black";				# Background color
$BUTTONCOLOR = "blue";				# Button bar color
$TEXTCOLOR = "#D8D8BF";			        # Text color
$LINKCOLOR = "yellow";			        # Link color
$VLINKCOLOR = "#00FFFF";			# Visited link color

$greenlight_color="#009900";                # color of "good" status events
$redlight_color="red";                      # color of "bad" status events
$unchecked_color="white";                   # color of "unknown" status events
$disabled_color="#999999";                  # color of "disabled" items

###############################################################
# You shouldn't need to change anything below this !!!!!!!!
###############################################################
$url = CGI::script_name();			# URL of this script.


# General Declarations -------------------------------------------------
use vars qw($c $webpage $time $localtime @year_months @days_of_week );
@year_months = ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec');
@days_of_week = ('Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday');

$c = new Mon::Client (
			 host => $monhost,
			 );


%loginhash = ("username","", "password","");


$webpage = new CGI;			       # Initialize web page CGI object

$time = time;				       # This will be used for 
$localtime = localtime(time);		       # the time on the page.


use vars  qw($command $args $rt $rtargs);
# These global vars all need to be zeroed out at the beginning of each
# instance so that they don't confuse mod_perl. If you're running under CGI,
# you don't need to do this, but it won't hurt either.
$command = "";
$args = "";
$rt="";
$rtargs="";

# Initialize a TripleDES global if login is required
if ($must_login) {
    eval "use Crypt::TripleDES";
    $des = new Crypt::TripleDES;
}


###############################################################
# Function definitions begin below
###############################################################

###############################################################
# General functions, not specific to Mon or mon.cgi
###############################################################
sub backwards {
    # Reverse sort
    my ($a, $b);
    $b cmp $a;
}


sub pp_sec {
    # This routine converts a number of seconds into a text string
    # suitable for (pretty) printing. The dtlog from Mon reports downtime
    # in seconds, and we want to present the user with more meaningful
    # data than "the service has been down for 13638 seconds"
    #
    # By Martha Greenberg <marthag@mit.edu> w/ pedantic plural 
    # modifications by Andrew.
    use integer;
    my $n = $_[0];
    my ($days, $hrs, $min, $sec) = ($n / 86400, $n % 86400 / 3600,
				    $n % 3600 / 60, $n % 60);
    my $s = $sec . " second";
    $s .= "s" if $sec != 1;   #because 0 is plural too :)
    if ($min > 0) {
	if ($min == 1) {
	    $s = $min . " minute, " . $s;
	} else {
	    $s = $min . " minutes, " . $s;
	}
    }
    if ($hrs > 0) {
	if ($hrs == 1) {
	    $s = $hrs . " hour, " . $s;
	} else {
	    $s = $hrs . " hours, " . $s;
	}
    }
    if ($days > 0) {
	if ($min == 1) {
	    $s = $days . " day, " . $s;
	} else {
	    $s = $days . " days, " . $s;
	}
    }
    return $s;
}


sub arithmetic_mean {
    # Given an array of numbers, this function returns the arithmetic mean
    return 0 if scalar(@_) == 0 ;      #don't waste our time
    my $sum = 0;
    foreach (@_) {
	$sum += $_;
    }
    return $sum/scalar(@_);
}


sub std_dev {
    # Given an array of numbers, this function returns their 
    # standard deviation.
    return 0 if scalar(@_) < 2 ;      #don't waste our time
    my $sum = 0;
    my $mean = &arithmetic_mean(@_);
    foreach (@_) {
	$sum += ( $_ - $mean )**2 ;
    }
   return ( $sum/(scalar(@_) - 1) )**0.5 ;
}


sub validate_name {
    # Return untainted host or group name if safe, undef otherwise.
    # Because you can never scrub your input too well.
    return $_[0] =~ /^([\w.\-_]+)$/ ? $1 : undef;
}


sub gen_ciphertext {
    # This function takes as its input a piece of plaintext and
    # returns a piece of ASCII, 3DES-encoded ciphertext, or undef if 3DES fails
    # for some reason. The key used is the global value "$app_secret".
    my ($plaintext) = (@_);
    my $ciphertext ;

    if ($ciphertext = $des->encrypt3 ("$plaintext", "$app_secret" )) {
	# convert key to hex
	$ciphertext = unpack("H*", $ciphertext) ;
#	print "ciphertext is $ciphertext<br>\n";
	return $ciphertext;
    } else {
	return undef;
    }
}


sub gen_cleartext {
    # This function takes as its input a piece of ASCII, 3DES-encoded 
    # ciphertext and returns a piece of  plaintext, 
    # or undef if 3DES fails for some reason. The key used is the 
    # global value "$app_secret".
    my ($ciphertext) = (@_);
    my $plaintext ;

    #convert key to format decrypt3() will understand
    $ciphertext = pack("H*", $ciphertext) ;    

    if ($plaintext = $des->decrypt3 ("$ciphertext", "$app_secret" )) {
#	print "plaintext is $plaintext<br>\n";
	return $plaintext;
    } else {
	return undef;
    }
}



###############################################################
# Presentation functions. These all have the common feature
# that they format a bunch of information and present it in 
# a nice(?) way to the user.
###############################################################
sub setup_page {
    # Setup the html doc headers and such
    # Also, get/set username/password cookie if $must_login is in effect

    my ($title) = (@_);
    my (@time, $qtime, $ttime);
    my $title_color = "$TEXTCOLOR"; 
    my $page_title = "$organization : " if $organization ne "";
    $page_title = "${page_title}MON - $title";
    my $time_now = time;
    my @expires_time = gmtime($time_now + $login_expire_time);
    # Put the cookie date format in the standard cookie format
    my $expires = sprintf ("%s, %.2d-%s-%d %.2d:%.2d:%.2d GMT", @days_of_week[$expires_time[6]], $expires_time[3], @year_months[$expires_time[4]], $expires_time[5] + 1900, @expires_time[2,1,0]);
    my ($encrypted_password, $cookie, $cookie_value);


    if ($must_login) {
	if ($loginhash{'username'} ne "") {
	    # Don't get the username and password from the cookie 
	    # if the user just submitted it via the login form.
	    # Encrypt the password for cookie storage.
	    $encrypted_password = &gen_ciphertext($loginhash{'password'}) ;
	} else {
	    # Get the existing cookie and parse it
	    $cookie_value = $webpage->cookie(-name=>"$cookie_name",
						);
	    ($loginhash{'username'},$loginhash{'password'}) = split(':', $cookie_value) ;
	    $encrypted_password = $loginhash{'password'} ;
	    # Decrypt the password string (if any) for use by the app,
	    # unless the user just submitted it in cleartext.
	    $loginhash{'password'} = &gen_cleartext($loginhash{'password'}) ;
	    # for some reason (bug?) I get a space at the end of the password
	    # that is returned here, so for now let's take it out, since
	    # spaces are illegal in passwords anyway.
	    $loginhash{'password'} =~ s/\s+//g ;
	}

	# Set up the new cookie (re-issue a new cookie with every access)
	if ($destroy_auth_cookie) {
	    $cookie_value = "" ;
	} else {
	    $cookie_value = "$loginhash{'username'}:$encrypted_password";
	}
	$cookie = $webpage->cookie(-name=>"$cookie_name",
				   -value=>"$cookie_value",
				   -expires=>"$expires",
				   -path=>"$cookie_path",
				   );	
	print $webpage->header(
			       -cookie=>$cookie,
			       );
    } else {
	# Plain & simple, no cookie, no passsword
	$encrypted_password = "" ;
	print $webpage->header();
    }


    print $webpage->start_html(-title=>"$page_title",
			       -BGCOLOR=>$BGCOLOR,
			       -TEXT=>$TEXTCOLOR,
			       -LINK=>$LINKCOLOR,
			       -VLINK=>$VLINKCOLOR,
			       );

# Useful for debugging username/password/cookie issues
#    print "cookie value is $cookie_value<br>\n";
#    print "encrypt passwd is $encrypted_password<br>\n";
#    print "username is &quot;$loginhash{'username'}&quot;<br>\n";
#    print "decrypt passwd is &quot;$loginhash{'password'}&quot;<br>\n";

    if ($logo) {
	$webpage->print("<table border=0><tr><td bgcolor=$title_color>\n");
	$webpage->print("\n<img src=\"$logo\" alt=\"[$organization logo]\"></td><td ><h1><font color=$title_color>MON: $title</font></h1>\n");
	$webpage->print("</td></tr></table>\n");
	$webpage->print("<hr>\n");
    } else {
	#just print the generic page
	$webpage->print("<h1><font color=$title_color>MON: $title</font></h1>\n");     
    }
    &print_bar;


    $qtime = time;
    @time = localtime($qtime);
    $ttime = sprintf ("%.2d:%.2d:%.2d on %s, %.2d-%s-%d", @time[2,1,0], @days_of_week[$time[6]], $time[3], @year_months[$time[4]], $time[5] + 1900 );

    $webpage->print("<center>");
    $webpage->print
	("\nThis information was presented at $ttime");
    $webpage->print
	(" to user <i>$loginhash{'username'}</i> (<a href=$url?command=moncgi_logout>log off user <i>$loginhash{'username'}</i></a>)") if $loginhash{'username'};
    $webpage->print(".</center>");

} 



sub print_bar {
    # Print the command bar
    #
    my $button = "INPUT TYPE=\"submit\" NAME=\"command\"";
    my $table_width = "100%";
    my $face="Helvetica, Arial";

    $webpage->print("<table width=\"$table_width\" border=1 align=center>\n");
    $webpage->print("<tr>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=query_opstatus>Show Operational Status (summary)</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=query_opstatus_full>Show Operational Status (full)</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=list_alerthist>Show Alert History</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a
href=$url?command=list_dtlog>Show Downtime Log</a></font></td>\n") ;
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_list_disabled>List Disabled Hosts/ Watches/ Svcs</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_pids>List Mon PIDs</a></font></td>\n");

    $webpage->print("</tr>\n");
    $webpage->print("</table>");
    $webpage->print("<table width=\"$table_width\" border=1 align=center>\n");
    $webpage->print("<tr>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_reset>Reset Mon (Reset sched. state)</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_reset&args=keepstate>Reset Mon (Keep sched. state)</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_reload&args=auth>Reload auth file</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_schedctl&args=start>Start scheduler</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_schedctl&args=stop>Stop scheduler</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_savestate&args=>Save scheduler state</a></font></td>\n");
    $webpage->print("\t<td width=3 align=center><font FACE=\"$face\"><a href=$url?command=mon_loadstate&args=>Load scheduler state</a></font></td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("</table>");
}


# query the server operational status ----------------------------------
sub query_opstatus {
    my ($detail_level) = (@_);
    my ($retval);            # some variables for failures
    my (%op_success, %op_failure);
    my @scheduler_status = &mon_list_sched_state ;
    my $opstatus_table_width = "80%";

    print $webpage->center
	("This page will reload every $reload_time seconds.<br>\n");

    if ($scheduler_status[0] != 0) {
	print $webpage->center
	    ("The scheduler is currently <font color=$greenlight_color>running</font>.<br><br>\n");
    } else {
	my @sched_down_time = localtime ($scheduler_status[1]);
	my $pretty_sched_down_time = sprintf ("%.2d:%.2d:%.2d, %s-%s-%s\n", @sched_down_time[2, 1, 0, 3], @year_months[$sched_down_time[4]], $sched_down_time[5]+1900);
	print $webpage->center
	    ("The scheduler has been <font color=$redlight_color>stopped</font> since $pretty_sched_down_time.<br><br>\n");
    }

    $webpage->print("\n<META HTTP-EQUIV=\"Refresh\" ");
    if ($detail_level eq "full") {
	$webpage->print("CONTENT=\"$reload_time\", URL=\"$url?command=query_opstatus_full\">\n");
    } else {
	$webpage->print("CONTENT=\"$reload_time\", URL=\"$url?command=query_opstatus\">\n");
    }

    %op_success = &mon_list_successes;
    %op_failure = &mon_list_failures ;


    $webpage->print("<table align=center border=1 width=35%>");
    $webpage->print
	("<tr><td><font color=$TEXTCOLOR>Service color legend:</font></td>\n");
    $webpage->print("<td><font color=$unchecked_color>Unchecked</font></td>\n");
    $webpage->print("<td><font color=$greenlight_color>Good</font></td>\n ");
    $webpage->print("<td><font color=$redlight_color>Error</font></td>\n");
    $webpage->print("<td><font color=$disabled_color>Disabled</font></td>\n");
    $webpage->print("</tr></table>");    

    $webpage->print
	("<table align=center width=\"$opstatus_table_width\" border=1>\n");
    $webpage->print
	("<tr><th width=25%><font size=+2>Host Group</font></th><th><font size=+2>Service</font></th>\n");
    $webpage->print
	("<th><font size=+2>Last Checked</font></th><th><font size=+2>Est. Next Check</font></th></tr>\n");

    &list_status($detail_level, %op_failure);
    &list_status($detail_level, %op_success);

    $webpage->print("</table>\n");    

}



sub list_status {
    # This function lists the status of all hosts and services. It is 
    # kind of a mess, but this is the function that 90% of the time
    # you will be viewing, and it has to do a lot. It could still be
    # cleaned up considerably though.
    #
    my ($detail_level, %op) = (@_);
    my (%group_list, $group, $service, $s, $g, $h);
    my (@time, $qtime);
    my $bg_fail = $redlight_color ; 
    my $bg_ok   = $greenlight_color;
    my $td_bg_color;
    my $face="Arial, Helvetica";
    
    my %d = &mon_list_disabled ;
    my (%ONDS, %ONDS_lastcheck, %ONDS_nextcheck) ;    #special hashes of arrays for "OK, Non-Disabled Services"
    my %OPSTAT = %Mon::Client::OPSTAT;

    my $service_disabled_string ;
    my $host_disabled_string ;
    my $watch_disabled_string ;
    my %saw ;      #used for sorting
    my @disabled_hosts;
    $qtime = time;


    foreach $group (sort keys %op) {
	if ($detail_level eq "full") {
	    # get a list of members of the group if we haven't already
	    $group_list{$group} = [ &mon_list_group($group) ] unless defined (@{$group_list{$group}});
	}
	foreach $service (sort keys %{$op{$group}}) {
	    $s = \%{$op{$group}->{$service}};
	    $service_disabled_string = "";
	    $host_disabled_string = "";
	    $watch_disabled_string = "";
	    undef %saw;
	    undef @disabled_hosts;
	    $service_disabled_string = "(DISABLED)" if ${d{"services"}{$group}{$service} };
	
	    foreach $g (keys %{$d{"hosts"}}) {
		foreach $h (keys %{$d{"hosts"}{$group}}) {
		    push(@disabled_hosts , $h);
		}
	    
	    }
	    # uniq and sort the returned array of disabled hosts
	    @saw{@disabled_hosts} = ();
	    @disabled_hosts = sort keys %saw;
	    $host_disabled_string = join(" " , @disabled_hosts) if scalar(@disabled_hosts) > 0 ;
	    $host_disabled_string = "<br>($host_disabled_string DISABLED)\n" if ($host_disabled_string ne "");
	    $watch_disabled_string = "(DISABLED)" if (${d{"watches"}{$group}});
            $td_bg_color = ( ($watch_disabled_string ne "") || ($host_disabled_string ne "") ) ? $disabled_color : $BGCOLOR ;

            # Don't print the service individually if we are in brief mode
            # mode and the service is an ONDS.
	    next if ( ($s->{"opstatus"} == $OPSTAT{"ok"}) && ($service_disabled_string eq "") && ($detail_level ne "full") );

            # Now print the first column (the group and its status)
            $webpage->print("<tr><td align=left bgcolor=\"$td_bg_color\">\n");
            # check to see if full display was requested
	    if ($detail_level eq "full") {
		$webpage->print("$watch_disabled_string<a href=\"$url?command=query_group&args=$group\">");
		$webpage->print("<font FACE=\"$face\" size=+1>$group</font></a><br>\n(");
		$webpage->print(join(", ",@{$group_list{$group}}));
		$webpage->print(")$host_disabled_string</td>\n");
	    } else {
		$webpage->print("<font FACE=\"$face\" size=+1>\n");
		$webpage->print("$watch_disabled_string<a href=\"$url?command=query_group&args=$group\">");
		$webpage->print("$group</a></font>$host_disabled_string</td>\n");
	    }

	    
            # Now print the second column (the service and its status)
	    if ($s->{"opstatus"} == $OPSTAT{"untested"})  {
		# for untested, don't use a bg in table cell and change
		# font color instead.
		$td_bg_color = ($service_disabled_string eq "") ?  $unchecked_color : $disabled_color ;
		$webpage->print("<td align=left>\n");
		$webpage->print("$service_disabled_string<a href=\"$url?command=alert_details&args=$group,$service\">");
		$webpage->print("<b>$service</b>\n");
		$webpage->print("<br><font color=\"$td_bg_color\">(UNCHECKED)</font></a>\n");
		$webpage->print("</td>\n");

	    } elsif ($s->{"opstatus"} == $OPSTAT{"fail"}) {
		$td_bg_color = ($service_disabled_string eq "") ? $bg_fail : $disabled_color ;
		$webpage->print("<td align=left bgcolor=\"$td_bg_color\">");
		$webpage->print("$service_disabled_string<a href=\"$url?command=alert_details&args=$group,$service\">");
		$webpage->print("<font size=+1><b>$service</b></font></a> : \n");
		$webpage->print("<font size=+1>$s->{last_summary}</font>\n");
		$webpage->print("<br>(FAILED)");
		$webpage->print("</td>\n");
	    
	    } elsif ($s->{"opstatus"} == $OPSTAT{"ok"}) {
		$td_bg_color = ($service_disabled_string eq "") ?  $bg_ok : $disabled_color ;
		$webpage->print("<td bgcolor=\"$td_bg_color\"><font size=+0>");
		$webpage->print("$service_disabled_string<a href=\"$url?command=alert_details&args=$group,$service\">");
		$webpage->print("$service</font></a></td>\n");
		
	    } else {
		my $txt = "";
		for (keys %OPSTAT) {
		    $txt = $_ if ($s->{"opstatus"} == $OPSTAT{$_});
		}
		$webpage->print("<td><font>");
		$webpage->print("${service_disabled_string}${service} (details: $txt)</font></td>\n");
	    }
    
            if ($s->{"opstatus"} == $OPSTAT{"untested"}) {
		$webpage->print("<font COLOR=$unchecked_color>");
		$webpage->print("<td>-</td>");
		$webpage->print("</font>\n");
		
	    } else {
		@time = localtime ($s->{"last_check"});
		$webpage->print("<td><font size=+1>\n");
		printf("%.2d:%.2d:%.2d\n", @time[2, 1, 0] );
		$webpage->print("</font></td>\n");
	    }
	    
	    	    
            @time = localtime ($qtime + $s->{"timer"});
	    $webpage->print("<td><font size=+1>\n");
	    printf("%.2d:%.2d:%.2d\n", @time[2, 1, 0] );
            $webpage->print("</font></td>\n");
            $webpage->print("</tr>");
	    
        } # end $service loop
#
# NEW: print a "compressed" version of OK, Non-Disabled Services (ONDS)
# (the assumption being that ONDS's are not all that interesting,
# let's not use up a lot of screen real estate discussing them)
#
#  The whole thing is contingent upon us being in "brief" mode
        if ($detail_level ne "full") {
	    # Build the array of ONDS's
            foreach $service (sort keys %{ $op{$group} }) {
	        $s = \%{ $op{$group}->{$service} };
	        next if ( ${d{"services"}{$group}{$service}} ) ;
        	if ($s->{"opstatus"} == $OPSTAT{"ok"}) {
	            push (@{ $ONDS{$group} }, "<a href=\"$url?command=alert_details&args=$group,$service\">$service</a>");
		    @time = localtime ($s->{"last_check"});
		    push (@{ $ONDS_lastcheck{$group} }, sprintf("%.2d:%.2d:%.2d\n", @time[2, 1, 0]) );
		    @time = localtime ($qtime + $s->{"timer"});
		    push (@{ $ONDS_nextcheck{$group} }, sprintf("%.2d:%.2d:%.2d\n", @time[2, 1, 0]) );
		}
            }
            # print the OK, no disabled services for this host if any exist
            if ( defined(@{ $ONDS{$group} }) ) {
		$td_bg_color = ( ($watch_disabled_string ne "") || ($host_disabled_string ne "") ) ? $disabled_color : $BGCOLOR ;
		$webpage->print("<tr>\n");
		$webpage->print("<td align=left bgcolor=\"$td_bg_color\">");
		$webpage->print("$watch_disabled_string<font FACE=\"$face\" size=+1><a href=\"$url?command=query_group&args=$group\">$group</font></a>$host_disabled_string");
		$webpage->print("</td>\n");
		$webpage->print("<td align=left bgcolor=\"$bg_ok\"><font size=-0>");
		print join(", ", @{$ONDS{$group}});
		$webpage->print("</font></td>\n");
		#Now print out the time(s)
		$webpage->print("<td align=left>");
		print join(", ", @{$ONDS_lastcheck{$group}});
		$webpage->print("</td>\n");
		$webpage->print("<td align=left>");
		print join(", ", @{$ONDS_nextcheck{$group}});
		$webpage->print("</td>\n");
		$webpage->print("</tr>\n");
	    }
	}  # end of ONDS printing
    }   # end of $group loop
}


sub query_group {
    # Print out info about the hosts in a particular hostgroup
    my ($args) = (@_);
    my $group = &validate_name ($args);
    my (%hosts, $host, $retval, $c, $e) ; 
    #my $td_bg;

    my %d = &mon_list_disabled ;
#    $watch_disabled_string = "(DISABLED)" if ( ${d{"watches"}{$group}});

    if (!defined $group) {
	$webpage->print("Invalid host group\n");
	
    } else {
	@_ = &mon_list_group ;
	# turn the array into a hash, which is what we really want
	foreach (@_) {
	    $hosts{$_} = "";
	}

	$webpage->print("<br><table align=center width=50% border=1>");
	if ( ${d{"watches"}{$group}} ) {
	    # hostgroup is disabled, offer to enable
	    $webpage->print("<th bgcolor=\"$disabled_color\">Members of group \"<em>$group</em>\"</th>") ;
	    $webpage->print( "<th bgcolor=\"$disabled_color\"><a href=\"$url?command=mon_enable&rt=query_group&rtargs=$group&args=watch,$group\">(ENABLE all monitoring for group <em>$group</em>)</a></th>\n") ;
	} else {
	    # hostgroup is enabled, offer to disable
	    $webpage->print("<th>Members of group \"<em>$group</em>\"</th>") ;
	    $webpage->print( "<th><a href=\"$url?command=mon_disable&rt=query_group&rtargs=$group&args=watch,$group\">(DISABLE all monitoring for group <em>$group</em>)</a></th>") ;
	}

	foreach $host (keys %hosts) {
	    if ($host =~ /^\*/) {
		$host =~ s/^\*// ;    #strip the * or else mon dies
		$hosts{$host} = "disabled";
	    } else {
		$hosts{$host} = "enabled";
	    }		
	}
	foreach $host (sort keys %hosts) {
	    next if ($host =~ /^\*/);      #ignore disabled hosts
	    $webpage->print("<tr>");
	    if ($hosts{$host} eq "disabled") {    #check to see if the host is disabled
		# the host is currently disabled
		$host =~ s/^\*// ;    #strip the * or else mon dies
		$webpage->print("<td bgcolor=\"$disabled_color\">$host (DISABLED)</td>");
		$webpage->print("<td bgcolor=\"$disabled_color\"><a href=\"$url?command=mon_enable&rt=query_group&rtargs=$group&args=host,$host\">(ENABLE monitoring on host <em>$host</em>)</td>");
	    } else {    #host is not currently disabled
		$webpage->print("<td>$host</td>");
		$webpage->print( "<td><a href=\"$url?command=mon_disable&rt=query_group&rtargs=$group&args=host,$host\">(DISABLE monitoring on host <em>$host</em>)</td>");
	    }
	    $webpage->print( "</tr>") ;
	}
        $webpage->print("</table>");
    }
}


sub end_page {
    # End the document with a footer and contact info
    &print_bar;
    if ($monadmin ne "") {
	print $webpage->center("For questions about this server, contact ");
	print $webpage->center
	    ("<a href=\"mailto:$monadmin\">$monadmin</a><br>");
    }
    print $webpage->end_html;
}

sub list_alerthist {
    # This function lists the alert history formatted in a table    
    my @l = &mon_list_alerthist ;
    my ($line, $localtime);
    my $table_width = "80%";

    $webpage->print("<table border=1 width=\"$table_width\" align=center>\n");

    $webpage->print("<tr><th>Group</th><th>Service</th>\n");
    $webpage->print("<th>Type</th><th>Time</th><th>Alert</th>\n");
    $webpage->print("<th>Args</th><th>Summary</th>\n");
    $webpage->print("</tr>\n");	

    foreach $line (reverse sort {$a->{"time"} <=> $b->{"time"}} (@l)) {
	$localtime = localtime ($line->{"time"});

	$webpage->print("<tr><td><a href=\"$url?command=query_group&");
	$webpage->print("args=$line->{group}\">$line->{group}</a></td>");

        $webpage->print("<td>$line->{service}</td>\n");
	$webpage->print("<td>$line->{type}</td>\n");
	$webpage->print("<td>$localtime</td>\n");

	$line->{"alert"} =~ s{^.*\/([^/]*)$}{$1};

	$webpage->print("<td>$line->{alert}</td>\n");

	my $args = "-";
	if ($line->{"args"} !~ /^\s*$/) {
	    $args = $line->{"args"};
	}

	$webpage->print("<td>$args</td>\n");
	$webpage->print("<td>$line->{summary}</td>");

	$webpage->print("</tr>\n");
    }

    $webpage->print("</table>\n");	
    print $webpage->hr;    
}



sub alert_details {
    # Lists details about a particular alert's status, regardless of
    # whether the alert is successful or not.
    my ($arg) = (@_);
    my ($group, $service) = split (/\,/, $arg);
    my $status;
    my $retval;
    my (%op, $s, $g, $var, @time);
    my $table_width = "75%";            #let's give both tables the same width

    my %d = &mon_list_disabled;


    # Determine whether the service is failing or not
    # We'll be optimistic and assume it's NOT failing
    %op = &mon_list_successes;
    if ($op{$group}{$service}) {
	$status = "ok";
    } else {
	$status = "fail";
	%op = &mon_list_failures;
    }

    print $webpage->hr;
    $webpage->print("<center>");
    if ($status eq "ok") {
	$webpage->print
	    ("<font size=+2>Success detail for group <font color=$greenlight_color>$group</font>\n");
	$webpage->print
	    ("and service test <font size=+2 color=$greenlight_color><i>$service</i></font>:</font>&nbsp;");
	
    } else {
	$webpage->print
	    ("<font size=+2>Failure detail for group <font color=$redlight_color>$group</font> ");
	$webpage->print
	    ("and service test <font color=$redlight_color><i>$service</i></font>:</font>&nbsp;\n");
    }
    $webpage->print("</center>");
    print $webpage->br;

    $webpage->print("<table width=\"$table_width\" align=center border=1><tr>");
    $webpage->print
	("<td><a><font size=+1><a href=\"$url?command=mon_test_service&args=$group,$service\">Test service <em>$service</em> on group <em>$group</em> immediately</font></a></td>\n");

    if (${d{"services"}{$group}{$service}}) {
        #service is disabled, offer to enable
        $webpage->print("<td><font size=+1><a href=\"$url?command=mon_enable&args=service,$group,$service&rt=alert_details\">(ENABLE service <em>$service</em> in group <em>$group</em>)</a></font></td>\n");
    } else {
        #service is enabled, offer to disable
        $webpage->print("<td><font size=+1><a href=\"$url?command=mon_disable&args=service,$group,$service&rt=alert_details\">(DISABLE service <em>$service</em> in group <em>$group</em>)</a></font></td>\n");
    }
    $webpage->print
	("<td><a><font size=+1><a href=\"$url?command=list_dtlog&args=$group,$service\">List downtime log for service <em>$service</em> and group <em>$group</em></font></a></td>\n");

    $webpage->print("</table></tr>");
    print $webpage->br;
#    $webpage->print("<br><br><br>\n");


    $webpage->print("<table width=\"$table_width\" border=1 align=center cellpadding=2 cellspacing=2>\n");
    $webpage->print("<tr><td align=left><font size=+2><b>Variable</b></font></td>\n");
    $webpage->print("<td align=left><font size=+2><b>Value</b></font></td></tr>\n");
    foreach $g (keys %op) {
	next if ($g ne $group);
    	foreach $s (keys %{$op{$g}}) {
	    next if ($s ne $service);
	    foreach $var (keys %{$op{$g}{$s}}) {
		# special case where we have a time formatted in secs 
		# since 1970, we'll make it look purty
		if (($var eq "last_check") || ($var eq "last_opstatus") || ($var eq "last_failure")|| ($var eq "last_trap") || ($var eq "last_success") || ($var eq "ack"))  {
		    if ($op{$g}->{$s}->{$var} == 0) {
			$op{$g}->{$s}->{$var} = "Never" ;
		    } else {
			@time = localtime ($op{$g}->{$s}->{$var});
			$op{$g}->{$s}->{$var} = sprintf ("%.2d:%.2d:%.2d, %s-%s-%s\n", @time[2, 1, 0, 3], @year_months[$time[4]], $time[5]+1900);
		    }
		}
		if ($op{$g}->{$s}->{$var} eq "") {
		    # special case where value of $var is empty 
		    #(i.e. mon has never seen the service fail)
		    $op{$g}->{$s}->{$var} = "-";
		}
		$op{$g}->{$s}->{$var} =~ s/\n/<BR>/g;
		$webpage->print("<tr><td><font size=+1><b>$var</b></font></td><td><font size=+1>$op{$g}->{$s}->{$var}</font></td></tr>\n");
	    }
	}
    }
    $webpage->print("</table>\n");

    print $webpage->hr;
}



sub query_disabled {
    # This function lists all the disabled watches, services, and hosts
    # and returns the result as pretty(?) HTML
    my %d;
    my ($group, $service, $host, $watch);

    %d = &mon_list_disabled;

    print "<hr>\n";

    print "<h2>Disabled watches:</h2>\n";
    print "<table border=1><tr>\n";
    print "<th><font size=+1>Watch</font></th>\n";
    print "</tr>\n";
    for (keys %{$d{"watches"}}) {
	print "<tr>\n";
	print "<td><font size=+1>$_</font>\n";
	print "<a href=\"$url?command=mon_enable&rt=query_disabled&args=watch,$_,\">(ENABLE watch <em>$_</em>)</a>\n";
	print "</td>\n";
	print "</tr>\n";
    }
    print "</table>\n";

    print "<hr>\n";

    print "<h2>Disabled services:</h2>\n";
    print "<table border=1><tr>\n";
    print "<th><font size=+1>Watch</font></th>\n";
    print "<th><font size=+1>Service</font></th>\n";
    print "</tr>\n";
    foreach $watch (keys %{$d{"services"}}) {
	foreach $service (keys %{$d{"services"}{$watch}}) {
	    print "<tr>\n";
	    print "<td><font size=+1>$watch</font></td>\n";
	    print "<td><font size=+1>$service</font>\n";
	    print "(<a href=\"$url?command=mon_enable&rt=query_disabled&args=service,$watch,$service\">ENABLE service <em>$service</em> on watch <em>$watch</em></a>)</td>\n";
	    print "</tr>\n";
	}
    }
    print "</table>\n";

    print "<hr>\n";

    print "<h2>Disabled hosts:</h2>\n";
    print "<table border=1><tr>\n";
    print "<th><font size=+1>Group</font></th>\n";
    print "<th><font size=+1>Host</font></th>\n";
    print "</tr>\n";
    foreach $group (keys %{$d{"hosts"}}) {
	foreach $host (keys %{$d{"hosts"}{$group}}) {
	    print "<tr>\n";
	    print "<td><font size=+1>$group</font></td>\n";
	    print "<td><font size=+1>$host</font>\n";
	    print "(<a href=\"$url?command=mon_enable&rt=query_disabled&args=host,$host\">ENABLE host <em>$host</em></a>)</td>\n";
	    print "</tr>\n";

	}
    }
    print "</table>\n";

}



sub list_dtlog {
    # Accepts as optional arguments a group, a service, and a column
    # to sort by. All arguments are optional. Default sort key is
    # failtime. If {service,group} is blank,
    # shows detail about all failures for the given {group,service}.
    # No arguments means show all service failures for all groups, 
    # sorted by failtime.
    #
    # Someday it would be nice to take args like time ranges, etc., but
    # that capability should really be built into Mon itself since it
    # is such a useful feature and something which could leverage a lot
    # of mon's timeperiod work as well.
    #
    # Original patch by Martha H Greenberg <marthag@mit.edu>
    #
    my ($arg) = (@_);
    my ($group, $service,$sortby) = split (/\,/, $arg);
    my $face="Helvetica, Arial";
    my $summary_table_width = "80%";
    my $dt_table_width = "100%";

    print $webpage->hr;
    print $webpage->h2("<center>Downtime Log</center>");
    my ($line, $localtimeup, $localfailtime, $ppdowntime, $ppinterval);

    my ($first_failure_time, $total_failures, $mtbf, $mean_recovery_time, $std_dev_recovery_time, $min_recovery_time, $max_recovery_time, @l) = &mon_list_dtlog($group, $service) ;
    my ($ppmtbf, $ppmean_recovery_time, $ppmin_recovery_time, $ppmax_recovery_time, $ppstd_dev_recovery_time);

    # Estimated uptime calculation
    my $time_now = time;
    my $approx_uptime_pct = ( ($time_now - $first_failure_time + $mtbf ) > 0 ) ? sprintf("%.2f%", ( ( ($time_now - $first_failure_time + $mtbf) - (scalar(@l) * $mean_recovery_time) ) / ($time_now - $first_failure_time + $mtbf) ) * 100 ) : "undefined";

    $webpage->print("<table border=1 align=center width=\"$summary_table_width\">\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td colspan=2 align=center><font size=+1 face=\"$face\">\n");
    $webpage->print("Downtime Summary For Hostgroup ");
    if ($group eq "") {
	$webpage->print("<b>&lt;any&gt;</b>");
    } else {
	$webpage->print("<b>&quot;$group&quot;</b>");
    }
    $webpage->print(" and Service ");
    if ($service eq "") {
	$webpage->print("<b>&lt;any&gt;</b>");
    } else {
	$webpage->print("<b>&quot;$service&quot;</b>");
    }
    $webpage->print("</font></td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td>Total observed service failures:</td>\n<td>$total_failures</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<td>Mean time between service failures:</td>\n");
    $ppmtbf = &pp_sec($mtbf);
    $webpage->print("<td>$ppmtbf</td>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td>Mean observed service failure time:</td>\n");
    $ppmean_recovery_time = &pp_sec($mean_recovery_time);
    $webpage->print("<td>$ppmean_recovery_time</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td>Standard deviation of observed service failure times:</td>\n");
    $ppstd_dev_recovery_time = &pp_sec($std_dev_recovery_time);
    $webpage->print("<td>$ppstd_dev_recovery_time</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td>Minimum observed service failure time:</td>\n");
    $ppmin_recovery_time = &pp_sec($min_recovery_time);
    $webpage->print("<td>$ppmin_recovery_time</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td>Maximum observed service failure time:</td>\n");
    $ppmax_recovery_time = &pp_sec($max_recovery_time);
    $webpage->print("<td>$ppmax_recovery_time</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td><i>Approximate</i> percentage of time in failure-free operation:</td>\n");
    $webpage->print("<td>$approx_uptime_pct</td>\n");
    $webpage->print("</tr>\n");
    $webpage->print("</table>\n");

    $webpage->print("<br><br>\n");

    $webpage->print("<table border=1 width=\"$dt_table_width\" align=center>\n");
    $webpage->print("<tr><th><a href=\"$url?command=list_dtlog&args=$group,$service,group\">Group</a></th>\n");
    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,service\">Service</a></th>\n");
    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,failtime\">Service Failure Begin Time</a></th>\n");
    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,timeup\">Service Failure End Time</a></th>\n");

    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,downtime\">Total Observed Service Failure Time</a></th>\n");
    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,interval\">Testing Interval</a></th>\n");
    $webpage->print("<th><a href=\"$url?command=list_dtlog&args=$group,$service,summary\">Summary</a></th>\n");
    $webpage->print("</tr>\n");

    if ($sortby ne "") {
	# do a forward-alphanumeric or reverse-numeric sort, 
	# depending on the sortby parameter
	if ( ($sortby eq "group") || ($sortby eq "service") || ($sortby eq "summary") ) {
	    @l = (sort {$a->{"$sortby"} cmp $b->{"$sortby"}}(@l));
	} else {
	    @l = (reverse sort {$a->{"$sortby"} <=> $b->{"$sortby"}}(@l));
	}
    }

    foreach $line (@l) {
	$webpage->print("<tr><td><a href=\"$url?command=list_dtlog&args=$line->{\"group\"}\">");
	$webpage->print("<font face=\"$face\">$line->{\"group\"}</font></a></td>");
        $webpage->print("<td><a href=\"$url?command=list_dtlog&args=$group,$line->{\"service\"}\">");
	$webpage->print("<font face=\"$face\">$line->{\"service\"}</font></a></td>\n");
	
	$localfailtime = localtime ($line->{"failtime"});
	$webpage->print("<td>$localfailtime</td>\n");

	$localtimeup = localtime ($line->{"timeup"});
	$webpage->print("<td>$localtimeup</td>\n");
	
	$ppdowntime = &pp_sec ($line->{"downtime"});
	$webpage->print("<td>$ppdowntime</td>\n");
        $ppinterval = &pp_sec ($line->{"interval"});
	$webpage->print("<td>$ppinterval</td>\n");
	$webpage->print("<td>$line->{\"summary\"}</td>");
	
	$webpage->print("</tr>\n");
    }
    
    $webpage->print("</table>\n");
    print $webpage->hr;
}



###############################################################
# Mon-specific functions. These all have the common feature
# that they connect to a Mon server and retrieve some data.
# Generally these functions are called by the Presentation functions,
# or if they are called directly they do no special output
# formatting.
###############################################################

sub mon_connect {
    # Performs the basic connection, and if necessary, authentication,
    # to a mon server.
#    # If successful, returns a Mon::Client object.
    # If successful, returns a 1
    # If unsuccessful, returns 0
    my $retval;

#    my $c = new Mon::Client (
#    	host => $monhost,
#    );

    #$c->connect();
    # If we're not connected, we need to connect and possibly authenticate
    if ($c->connected() == 0) {
	$c->connect();
	if ($must_login) {
	    #print "<pre>Login is &quot;$loginhash{\"username\"}&quot; and passwordis &quot;$loginhash{\"password\"}&quot;\n</pre>";
	    if (($loginhash{"username"} eq "") && ($loginhash{"password"} eq "")) {
		# Login is required but no username/password was given, so
		# try the login and password to the default account
		$loginhash{"username"} = $default_username ;
		$loginhash{"password"} = $default_password ;
	    }	
	    $c->login(%loginhash);
	}
    }

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not contact mon server on server &quot;$monhost&quot;: $retval </pre>\n";
	if (($must_login) && (($retval =~ /530 login unsuccessful/) || ($retval =~ /no password/) || ($retval =~ /no username/))) {
	    # Login was required and unsuccessful, present the authform
	    # if it hasn't already been presented. Since some presentation
	    # functions call multiple methods you could very easily
	    # end up with multiple prompts, which is confusing to the 
	    # user.
	    &moncgi_authform ($command,"$args") unless $has_prompted_for_auth;
	    $has_prompted_for_auth = 1;
	}
	$c->disconnect();
	return 0;
    }
    
    #return $c;
    return 1;
}


sub mon_list_group {
    # List all the hosts in a given group. Returns an array of hosts
    # if successful, or undef if failure.
    my ($group) = (@_);
    my (@hosts, $retval);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;
    
    @hosts = $c->list_group ($group);
    
    unless ($c->error) {
	return @hosts ;
    } else {
	$retval = $c->error;
	$webpage->print ("<pre>Could not list groups on mon server &quot;$monhost&quot;: $retval</pre>");
	return undef;
    }
#    $c->disconnect();
}




sub mon_list_failures {
    # This function returns a hash of failures.
    my (%op, $retval);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    %op = $c->list_failures();

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not execute list failures command mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	#print "<pre>list_failures command executed successfully</pre>\n";
    }

#    $c->disconnect();
    return %op;
}


sub mon_list_successes {
    # This function returns a hash of successes
    my (%op, $retval);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    %op = $c->list_successes();

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not execute list successes command mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	#print "<pre>list_successes command executed successfully</pre>\n";
    }

#    $c->disconnect();
    return %op;
}


sub mon_list_disabled {
    # This function lists all the disabled watches, services, and hosts
    # and returns the result as a hash
    my (%d, $retval);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    %d = $c->list_disabled();

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not execute list disabled command mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	#print "<pre>list_disabled command executed successfully</pre>\n";
    }
    
#    $c->disconnect();
    
    return %d;

}


sub mon_reload {
    # Reload mon config file.
    # Right now the only option supported is to reload the auth.cf file.
    #
    my ($what) = (@_);
    print $webpage->hr;
    print $webpage->h2("Reloading mon...");
    my $retval;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    $retval = $c->reload($what);

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not reload &quot;$what&quot; mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>mon server on &quot;$monhost&quot; successfully reloaded.</pre>\n";
    }


#    $c->disconnect();
}



sub mon_loadstate {
    # A simple wrapper function that calls mon_loadstate_savestate with
    # the proper arguments.
    my ($state) = (@_);
    $state = "disabled" if $state eq "";
    print $webpage->hr;
    print $webpage->h2("Loading saved state for $state...");
    &mon_loadstate_savestate("load",$state);
}


sub mon_savestate {
    # A simple wrapper function that calls mon_loadstate_savestate with
    # the proper arguments.
    my ($state) = (@_);
    $state = "disabled" if $state eq "";
    print $webpage->hr;
    print $webpage->h2("Saving current state for $state...");
    &mon_loadstate_savestate("save",$state);
}


sub mon_loadstate_savestate {
    # Loads or saves state of a mon object specifed by $target. Currently
    # the only object supported by mon for loading/saving state is the
    # state of the scheduler, so using this function with $target="" will
    # load or save the state of the scheduler.
    #
    # The load/save action is specified by the variable $action
    my ($action, $state) = (@_);
    my $retval;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    if ($action eq "save") {
	($retval = $c->savestate($state)) || ($retval = $c->error);
    }
    elsif ($action eq "load") {
	($retval = $c->loadstate($state)) || ($retval = $c->error);
    }

    if ($c->error ne "") {
	print "<pre>Could not $action state mon server for state &quot;$state&quot; on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>$action state succeded on mon server &quot;$monhost&quot;.</pre>\n";
    }

#    $c->disconnect();
}


# Stop or start scheduler  ---------------------------------------------
sub mon_schedctl {
    # Either stops or starts the scheduler, depending on how it was called,
    # either with the "stop" argument or the "start" argument.
    # No return value.
    my ($action) = (@_);
    print $webpage->hr;
    print $webpage->h2("MON: $action scheduler...");
    my $retval;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    if ($action eq "stop") {
	$retval = $c->stop();
    } elsif ($action eq "start") {
	$retval = $c->start();
    }
    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not $action mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>$action scheduler on mon server on &quot;$monhost&quot; succeeded.</pre>\n";
    }

#    $c->disconnect();
}



sub mon_list_pids {
    print $webpage->hr;
    print $webpage->h2("List of mon PID's:");
    my $retval;
    my @pids;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    @pids = $c->list_pids;
    my $server = shift @pids;

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not list pids on mon server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>\n";
	print "Server PID is $server\n";
	print "\n";
	print "PID's of currently active monitors:\n";
	for (@pids) {
	    print join (" ",
			$_->{"watch"},
			$_->{"service"},
			$_->{"pid"},
			"\n",
			);
	}
	print "\n(If the above list is empty, it means that no monitors\n";
	print "are running at this time)\n";
	print "</pre>\n";
    }

#    $c->disconnect();
}


# Enable a disabled host/watch/service ----------------------------------
sub mon_enable {
    my ($arg) = (@_);
    my ($type, $arg1, $arg2) = split (/\,/, $arg);
    print $webpage->h2("Enabling service...");
    my $retval;
    
    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    if ($type eq "service") {
	$retval = $c->enable_service($arg1, $arg2);
    } elsif ($type eq "host") {
	$retval = $c->enable_host($arg1);
    } elsif ($type eq "watch") {
	$retval = $c->enable_watch($arg1);
    }

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not successfully execute command enable_${type} with arguments &quot;$arg1&quot; and &quot;$arg2&quot; on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>enable_${type} succeeded for ";
	    if ($type eq "service") {
		print "watch $arg1, service $arg2";
	    } elsif ($type eq "host") {
		print "host $arg1";
	    } elsif ($type eq "watch") {
		print "watch $arg1";
	    }
	print "</pre>\n";
    }

#    $c->disconnect();

}

# Disable an enabled service ----------------------------------------------
sub mon_disable {
    my ($arg) = (@_);
    my ($type, $arg1, $arg2) = split (/\,/, $arg);
    print $webpage->h2("Disabling service...");
    my $retval;
    
    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    if ($type eq "service") {
	$retval = $c->disable_service($arg1, $arg2);
    } elsif ($type eq "host") {
	$retval = $c->disable_host($arg1);
    } elsif ($type eq "watch") {
	$retval = $c->disable_watch($arg1);
    }

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not successfully execute command disable_${type} with arguments &quot;$arg1&quot; and &quot;$arg2&quot; on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>disable_${type} succeeded for ";
	    if ($type eq "service") {
		print "watch $arg1, service $arg2";
	    } elsif ($type eq "host") {
		print "host $arg1";
	    } elsif ($type eq "watch") {
		print "watch $arg1";
	    }
	print "</pre>\n";
    }

#    $c->disconnect();

}


sub mon_test_service {
    # Test a service immediately.
    # Accepts as arguments a group and a service, and optionally an
    # test argument, which can be either alert, upalert, or startupalert.
    # Default test is "alert"
    my ($arg) = (@_);
    my ($group, $service,$test) = split (/\,/, $arg);
    $test = "monitor" if $test eq "";
    print $webpage->h2("Performing $test test on service $service in hostgroup $group...");
    my $retval;
    
    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    
    $retval = $c->test($test, $group, $service);

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not successfully execute command &quot;$test&quot; test service &quot;$service&quot; on hostgroup &quot;$group&quot; on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	print "<pre>test $test completed for service $service and hostgroup $group:\n";
	print "\n $retval\n</pre>\n";
    }

#    $c->disconnect();

}


# Reset  mon  ----------------------------------------------------
sub mon_reset {
    ($args) = (@_);
    print $webpage->hr;
    print $webpage->h2("Reset mon...");
    my $retval;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    if ( $args eq "keepstate" ) {
	# specify we want to keep the scheduler state
	$retval = $c->reset($args);
    } else {
	# reset scheduler state, don't give reset any arguments
	$retval = $c->reset();
    }

    if ($c->error) {
	$retval = $c->error;
    	$webpage->print 
	    ("<pre>Could not reset mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n");
	&moncgi_switch_user($retval);
    } else {
	$webpage->print ("<pre>mon server on &quot;$monhost&quot; successfully reset.\n\n");
	if ( $args eq "keepstate" ) {
	    $webpage->print ("Scheduler state was NOT reset.\n\nAll previously disabled hosts/groups/services are still disabled."); 
	} else {
	    $webpage->print ("Scheduler state was reset.\n\nAll hosts/groups/services are now enabled."); 
	}
	$webpage->print ("</pre>\n");
    }

#    $c->disconnect();
}



# List alert history --------------------------------------------------
sub mon_list_alerthist {
    print $webpage->hr;
    print $webpage->h2("Alert History:");
    my $retval ;

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    my @l = $c->list_alerthist();

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not list alert history on mon server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	#print "<pre>alert history on on &quot;$monhost&quot; successfully retrieved.</pre>\n";
    }

#    $c->disconnect();
    return @l;
}


sub mon_list_sched_state {
    # This function returns an array, @scheduler_state, which 
    # contains the state of the scheduler. This  is not exactly 
    # documented, but @scheduler_state[0]==0 seems to indicate 
    # that the scheduler is stopped and @scheduler_state[1]
    # seems to hold the time (in epoch seconds) since the scheduler was
    # stopped.
    my (@scheduler_state, $retval);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0;

    @scheduler_state = $c->list_state();

    if ($c->error) {
	$retval = $c->error;
    	print "<pre>Could not execute list state command mon server on server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n";
	&moncgi_switch_user($retval);
    } else {
	#print "<pre>list state command executed successfully</pre>\n";
    }

#    $c->disconnect();
    return @scheduler_state;

}


sub mon_list_dtlog {
    # Lists the downtime log (all of it) and returns the results as
    # an array of hash references.
    my ($group, $service) = (@_);
    my $retval ;
    my (@ltmp, @l, $line);
    my (@recovery_times);
    my ($first_failure_time, $mtbf, $mean_recovery_time, $std_dev_recovery_time, $max_recovery_time, $min_recovery_time);

    my $conn = &mon_connect ; #test
    return 0 if $conn == 0 ; #test

    @ltmp = $c->list_dtlog();

    if ($c->error) {
      $retval = $c->error;
      $webpage->print("<pre>Could not list downtime log on mon server &quot;$monhost&quot;: $retval (perhaps you don't have permissions in auth.cf?)</pre>\n");
      &moncgi_switch_user($retval);
    }

#    $c->disconnect();

    my $time_now = time;
    $max_recovery_time = -1;                     # initialize this to something really small
    $min_recovery_time = 9999999999999;          # initialize this to something really big
    $first_failure_time = $time_now;
    foreach $line (reverse sort {$a->{"failtime"} <=> $b->{"failtime"}}(@ltmp)){
	if ($line->{"failtime"} < $first_failure_time) {
	    # since this list is already sorted, this will only be true
	    # the very first time we go thru this loop
	    $first_failure_time = $line->{"failtime"} if $line->{"failtime"} < $first_failure_time ;
	}
	next if ( ($group ne "") && ($group ne $line->{"group"}) );
	next if ( ($service ne "") && ($service ne $line->{"service"}) );
	push(@l, $line);
	push(@recovery_times, $line->{"downtime"});
	# set min and max
	$min_recovery_time = $line->{"downtime"} if $line->{"downtime"} < $min_recovery_time;
	$max_recovery_time = $line->{"downtime"} if $line->{"downtime"} > $max_recovery_time;
    }

    # recovery time is easy, just a straight mean
    $mean_recovery_time = &arithmetic_mean(@recovery_times);
    # calculate the mean time between failures as:
    # (total elapsed time since first failure + E(time until first failure))/(total # of failures)
    $mtbf = (@recovery_times == 0) ? 0 : ($time_now - $first_failure_time + (($time_now - $first_failure_time) / scalar(@recovery_times))) / scalar(@recovery_times);
    $std_dev_recovery_time = &std_dev(@recovery_times);
    
    return $first_failure_time, scalar(@recovery_times), $mtbf, $mean_recovery_time, $std_dev_recovery_time, $min_recovery_time, $max_recovery_time,  @l ;
}



###############################################################
# mon.cgi-specific functions
# These fuctions generally do not manipulate Mon in any way
# or present any Mon output to the user.
###############################################################

# Get the params from the form -----------------------------------------
sub moncgi_get_params {
    $command = $webpage->param('command');
    $args = $webpage->param('args');
    $rt = $webpage->param('rt');   # return to value for pages 
                                   # which need to keep state info about
                                   # which page called them.
    $rtargs = $webpage->param('rtargs');   #args for $rt
    # For the login form, grab username and password
    $loginhash{'username'} = $webpage->param('username');
    $loginhash{'password'} = $webpage->param('password');
}


sub moncgi_logout {
    # This subroutine provides the verbiage evidence to the user
    # that they have been logged out of the mon server. The actual
    # logging out is done in the setup_page() routine, when the
    # auth cookie is destroyed in accordance to the value of the 
    # global variable $destroy_auth_cookie
    print $webpage->hr;
    print $webpage->h3("User <i>$loginhash{'username'}</i> has been logged off.<br>");
    print $webpage->h3("You will need to re-authenticate to perform further privileged actions.");
}


sub moncgi_authform {
    my ($command, $args) = (@_);
#    print $webpage->hr;
    print $webpage->startform(-method=>'POST',
			      );
    print $webpage->h3("You must authenticate to perform this command.<br>");
    $webpage->print("<table border=0>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td><b>MON username:</b></td>\n");
    $webpage->print("<td>");
    print $webpage->textfield(-name=>'username',
				   -size=>8,
				   -maxlength=>100,
				   );
    $webpage->print("</td>");
    $webpage->print("</tr>\n");
    $webpage->print("<tr>\n");
    $webpage->print("<td><b>MON password:</b></td>\n");
    $webpage->print("<td>");
    print $webpage->password_field(-name=>'password',
				   -size=>8,
				   -maxlength=>8,
				   );
    $webpage->print("</td>");
    $webpage->print("</tr>\n");
    $webpage->print("</table>\n");

    print $webpage->br;
    # if this works correctly, the command will be re-executed with 
    # the credentials the user just entered
    print $webpage->hidden(-name=>'command',
			   -value=>$command,
			   );
    print $webpage->hidden(-name=>'args',
			   -value=>"$args",
			   );
#    print $webpage->br;
    print $webpage->submit(-name=>'Login',
			   );
    print $webpage->end_form;

}    


# Generic button function ---------------------------------------------
# Not strictly necessary, but could be useful if you wished to disable
# certain features of the client or add new ones in a test capacity.
sub moncgi_generic_button {
    my ($title, $command) = (@_);

    print $webpage->hr;
    print $webpage->h2("$title");
    $webpage->print ("(command $command not implemented in this client)\n");
    print $webpage->hr;
}


sub moncgi_switch_user {
    # This subroutine is called after a command fails because a user
    # is authenticated as a user without sufficient permission to perform
    # the requested command, so we give the user a chance to re-authenticate
    # as someone of sufficient privilege.
    my ($retval) = (@_) ;
    if ( ($must_login) && ($retval =~ /520 command could not be executed/) ) {
        # User doesn't have permission to perform command
        &moncgi_authform ($command,"$args") unless $has_prompted_for_auth;
        $has_prompted_for_auth = 1;
    }
}


###############################################################
# Main program
###############################################################

&moncgi_get_params;				       # Read the args.

if ($command =~ "query_group" ){		       # Expand hostgroup.
    &setup_page("Group Expansion");
    &query_group($args);
}
elsif ($command =~ "list_alerthist"){	       # Alert history button.
    &setup_page("List the alert history");
    &list_alerthist;
}
elsif ($command =~ "alert"){		       # View alert details.
    &setup_page("Alert Details");
    &alert_details($args);
}
elsif ($command =~ "mon_list_disabled"){		       # List disabled hosts button.
    &setup_page("List disabled hosts");
    &query_disabled;
}
elsif ($command =~ "mon_test_service"){		 # Test a service immediately
    &setup_page("Test Service");
    &mon_test_service($args);
    &alert_details($args);
}
elsif ($command =~ "mon_schedctl"){	       # Stop/start the scheduler
    &setup_page("$args scheduler");
    &mon_schedctl ($args);
    &query_opstatus("summary");
}
elsif ($command =~ "list_dtlog"){             # List the downtime log
    &setup_page("List Downtime Log");
    &list_dtlog($args);
}
elsif ($command =~ "mon_disable"){		       # Disable a host/group/service
    &setup_page("Disable alert for host, group. or service");
    &mon_disable($args);
    if ($rt eq "query_group") {
	&query_group($rtargs);
    } else {
	&query_opstatus("summary");
    }
}
elsif ($command =~ "mon_enable"){		       # Enable a host/group/service
    &setup_page("Enable alert for host, group, or service");
    &mon_enable($args);
    if ($rt eq "query_group") {
	&query_group($rtargs);
    } elsif ($rt eq "alert") {
	&query_opstatus("summary");
    } else {
	&query_disabled;
    }
}
elsif ($command =~ "mon_pids"){		       # View pid button.
    &setup_page("List pids of server, alerts and monitors.");
    &mon_list_pids;
}
elsif ($command =~ "mon_reset"){		       # Reset mon button.
    &setup_page("Reset mon");
    &mon_reset($args);
}
elsif ($command =~ "mon_reload"){		       # Reload mon button.
    &setup_page("Reload mon");
    &mon_reload($args);
}
elsif ($command =~ "mon_loadstate"){	       # load mon state
    &setup_page("Load state ($args)");
    # right now we expect $args to be empty, since loadstate doesn't take
    # any arguments, but someday it might, so we prepare.
    &mon_loadstate($args);
}
elsif ($command =~ "mon_savestate"){	       # save mon state
    &setup_page("Load state ($args)");
    # right now we expect $args to be empty, since loadstate doesn't take
    # any arguments, but someday it might, so we prepare.
    &mon_savestate($args);
}
elsif ($command =~ "moncgi_logout"){		       # Log out as auth user.
    $destroy_auth_cookie = "yes";
    &setup_page("Logging out");
    &moncgi_logout;
}
elsif ($command =~ "query_opstatus_full"){		       # Full operations status
    &setup_page("Operation Status: Full");
    &query_opstatus("full");
# Selection "mon_opstatus" will fall through to else.
}
else {					       # All else.
    &setup_page("Operation Status: Summary");
    &query_opstatus("summary");
}

$webpage->print("<hr>");

&end_page;
$c->disconnect();
