(ns com.puppetlabs.puppetdb.http
  (:require [ring.util.response :as rr])
  (:use [com.puppetlabs.http :only [json-response]]))

(def header-map
  "Maps the legal keys from a puppetdb query response object to the
  corresponding HTTP header names."
  {:count "X-Records"})

(defn header-for-key
  "Given a key from a PuppetDB query response, returns the HTTP header that
  should be used in the HTTP response."
  [k]
  {:pre [(contains? header-map k)]
   :post [(string? %)]}
  (header-map k))

(defn add-headers
  "Given a Ring response and a PuppetDB query result map, returns
  an updated Ring response with the headers added."
  [response query-result]
  {:pre  [(map? query-result)]
   :post [(rr/response? %)]}
  (reduce
    (fn [r [k v]] (rr/header r (header-for-key k) v))
    response
    query-result))

(defn query-result-response
  "Given a PuppetDB query result map (as returned by `query/execute-query`),
  returns a Ring HTTP response object."
  [query-result]
  {:pre [(map? query-result)
         (contains? query-result :result)]
   :post [(rr/response? %)]}
  (->
    (json-response (:result query-result))
    (add-headers (dissoc query-result :result))))

(defn remove-environment
  "dissocs the :environment key when the version is :v4"
  [result-map version]
  (if-not (= :v4 version)
    (dissoc result-map :environment)
    result-map))

(defn remove-all-environments
  "Removes environment from a seq of results"
  [version rows]
  (map #(remove-environment % version) rows))

(defn v4?
  "Returns a function that always returns true if `version` is :v4"
  [version]
  (constantly (= :v4 version)))
