/*--------------------------------------------------------------------------+
$Id: LineSplitter.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.string;

/**
 * This class is used to split a string in lines.
 * <p>
 * <b>Note:</b> According to tests I performed this is the fastest method to
 * split a string. It is about nine times faster than the regex-bases split
 * with:
 * 
 * <pre>
 * Pattern pattern = Pattern.compile(&quot;\r\n|\r|\n&quot;);
 * pattern.split(content);
 * </pre>
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Revision: 26268 $
 * @levd.rating GREEN Hash: F99C8B9E8F156988EBFA29796D5D1AEF
 */
public class LineSplitter {

	/** Character array. */
	private char[] characters;

	/** Starting index. */
	private int startIndex;

	/** Length of the line to be returned from {@link #getNextLine()}. */
	private int length;

	/**
	 * Set the string to split.
	 * 
	 * @param content
	 *            the string to split. if string is <code>null</code> or the
	 *            empty string, {@link #getNextLine()} will return
	 *            <code>null</code>
	 * 
	 */
	public void setContent(String content) {
		if (content == null) {
			characters = null;
		} else {
			characters = content.toCharArray();
		}
		startIndex = 0;
		length = 0;
	}

	/**
	 * Obtain next identified line.
	 * 
	 * @return <code>null</code> if all lines were returned. On returning the
	 *         last line all references to the input string are deleted. So it
	 *         is free for garbage collection.
	 */
	public String getNextLine() {

		if (characters == null) {
			return null;
		}

		startIndex = startIndex + length;

		if (startIndex >= characters.length) {

			// delete reference to array to allow garbage collection
			characters = null;
			return null;
		}

		// length to skip may vary due to the length of the line separator (\r,
		// \n or \r\n)
		int skip = 0;

		int i = startIndex;

		while (skip == 0 && i < characters.length) {
			char c = characters[i];

			i++;

			// Skip newlines.
			if (c == '\n') {
				skip = 1;
			}

			// Skip newlines.
			if (c == '\r') {
				skip = 1;
				if (i < characters.length) {
					if (characters[i] == '\n') {
						skip = 2;
						i++;
					}
				}
			}

		}

		length = i - startIndex;

		return new String(characters, startIndex, length - skip);
	}
}