/*--------------------------------------------------------------------------+
$Id: MaxWeightMatchingTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.algo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import junit.framework.TestCase;
import edu.tum.cs.commons.algo.MaxWeightMatching.IWeightProvider;
import edu.tum.cs.commons.collections.PairList;

/**
 * Tests for the {@link MaxWeightMatching} class. This only tests correctness,
 * but not correct implementation in terms of efficiency. The
 * {@link #testPerformance()} method is only to show how much better we are
 * compared to the exponential brute-force method, but this is just a manual
 * check and does not really show whether it is as efficient as promised.
 * 
 * @author hummelb
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: F1933AC26B78495540467249A02B8C05
 */
public class MaxWeightMatchingTest extends TestCase {

	/** Trivial test. */
	public void testTrivial() {
		assertSolution(new double[][] { { 1 } }, 1);
	}

	/** Simple test. */
	public void testSimple() {
		assertSolution(new double[][] { { 1, 2 }, { 2, 1 } }, 4);
	}

	/** Simple test requiring augmentation. */
	public void testSimpleAugmentation() {
		assertSolution(new double[][] { { 1, 2 }, { 1, 4 } }, 5);
	}

	/**
	 * Makes sure that the matching implied by the given weight matrix returns
	 * the given solution.
	 */
	private void assertSolution(double[][] weight, int solution) {
		List<Integer> nodes1 = new ArrayList<Integer>();
		List<Integer> nodes2 = new ArrayList<Integer>();
		for (int i = 0; i < weight.length; ++i) {
			nodes1.add(i);
		}
		for (int i = 0; i < weight[0].length; ++i) {
			nodes2.add(i);
		}

		double bf = new BruteForceSubSolver<Integer, Integer>(nodes1, nodes2,
				new ArrayWeightProvider(weight),
				new PairList<Integer, Integer>()).solve();
		double ap = new MaxWeightMatching<Integer, Integer>()
				.calculateMatching(nodes1, nodes2, new ArrayWeightProvider(
						weight), new PairList<Integer, Integer>());
		assertTrue("BF failed (expected: " + solution + ", received: " + bf
				+ ")", Math.abs(solution - bf) < 1e-15);
		assertTrue("AP failed (expected: " + solution + ", received: " + ap
				+ ")", Math.abs(solution - ap) < 1e-15);
	}

	/** Perform some randomized tests. */
	public void testRandom() {
		Random r = new Random(42);
		final int numRuns = 50;
		MaxWeightMatching<Integer, Integer> maxWeightMatching = new MaxWeightMatching<Integer, Integer>();

		for (int run = 0; run < numRuns; ++run) {
			int i = 1 + r.nextInt(7);
			int j = 1 + r.nextInt(i);
			double[][] weight = new double[j][i];
			for (int a = 0; a < j; ++a) {
				for (int b = 0; b < i; ++b) {
					weight[a][b] = r.nextDouble();
				}
			}

			List<Integer> nodes1 = new ArrayList<Integer>();
			List<Integer> nodes2 = new ArrayList<Integer>();
			for (int k = 0; k < weight.length; ++k) {
				nodes1.add(k);
			}
			for (int k = 0; k < weight[0].length; ++k) {
				nodes2.add(k);
			}

			double bf = new BruteForceSubSolver<Integer, Integer>(nodes1,
					nodes2, new ArrayWeightProvider(weight),
					new PairList<Integer, Integer>()).solve();
			double ap = maxWeightMatching.calculateMatching(nodes1, nodes2,
					new ArrayWeightProvider(weight),
					new PairList<Integer, Integer>());
			assertTrue(Math.abs(bf - ap) < 1e-15);
		}
	}

	/** Simple benchmarking test. */
	public void testPerformance() {
		Random r = new Random(42);
		final int numRuns = 1000;

		for (int i = 1; i <= 7; ++i) {
			for (int j = 1; j <= i; ++j) {
				double[][] weight = new double[j][i];
				for (int a = 0; a < j; ++a) {
					for (int b = 0; b < i; ++b) {
						weight[a][b] = r.nextDouble();
					}
				}

				List<Integer> nodes1 = new ArrayList<Integer>();
				List<Integer> nodes2 = new ArrayList<Integer>();
				for (int k = 0; k < weight.length; ++k) {
					nodes1.add(k);
				}
				for (int k = 0; k < weight[0].length; ++k) {
					nodes2.add(k);
				}

				double bf = 0, ap = 0;
				long ticksStart = System.currentTimeMillis();
				for (int k = 0; k < numRuns; ++k) {
					bf = new BruteForceSubSolver<Integer, Integer>(nodes1,
							nodes2, new ArrayWeightProvider(weight),
							new PairList<Integer, Integer>()).solve();
				}
				long bfTime = System.currentTimeMillis() - ticksStart;

				ticksStart = System.currentTimeMillis();
				MaxWeightMatching<Integer, Integer> m = new MaxWeightMatching<Integer, Integer>();
				for (int k = 0; k < numRuns; ++k) {
					ap = m.calculateMatching(nodes1, nodes2,
							new ArrayWeightProvider(weight),
							new PairList<Integer, Integer>());
				}
				long apTime = System.currentTimeMillis() - ticksStart;

				System.err.println("Time for " + j + "/" + i + " [bf/ap]: "
						+ bfTime + " " + apTime);
				assertTrue(Math.abs(ap - bf) < 1e-15);
			}
		}
	}

	/** Weight provider based on an array of weights. */
	private class ArrayWeightProvider implements
			MaxWeightMatching.IWeightProvider<Integer, Integer> {

		/** Weight used. */
		private final double[][] weight;

		/** Constructor. */
		public ArrayWeightProvider(double[][] weight) {
			this.weight = weight;
		}

		/** {@inheritDoc} */
		public double getConnectionWeight(Integer node1, Integer node2) {
			return weight[node1][node2];
		}
	}

	/** A solver which uses brute force and can be used for comparison purposes. */
	public static class BruteForceSubSolver<N1, N2> {

		/** Store the size of the smaller node set. */
		private final int size;

		/** The first node set. */
		private final List<N1> nodes1;

		/** The second node set. */
		private final List<N2> nodes2;

		/** The result to write into. */
		private final PairList<N1, N2> result;

		/** The weight provider used. */
		private final IWeightProvider<N1, N2> weightProvider;

		/** The best solution found so far. */
		private double bestSol = Double.NEGATIVE_INFINITY;

		/** Constructor. */
		public BruteForceSubSolver(List<N1> nodes1, List<N2> nodes2,
				IWeightProvider<N1, N2> weightProvider, PairList<N1, N2> result) {
			if (nodes1.size() > nodes2.size()) {
				throw new IllegalArgumentException();
			}
			this.size = nodes1.size();
			this.nodes1 = new ArrayList<N1>(nodes1);
			this.nodes2 = new ArrayList<N2>(nodes2);
			this.weightProvider = weightProvider;
			this.result = result;
		}

		/** Solve the problem. */
		public double solve() {
			result.clear();
			bestSol = Double.NEGATIVE_INFINITY;
			return solve(0, 0);
		}

		/** Recursive solver routine. */
		private double solve(int currentIndex, double currentSol) {
			if (currentIndex == size) {
				if (currentSol > bestSol) {
					result.clear();
					for (int i = 0; i < size; ++i) {
						result.add(nodes1.get(i), nodes2.get(i));
					}
					bestSol = currentSol;
				}
			} else {
				for (int i = currentIndex; i < nodes2.size(); ++i) {
					Collections.swap(nodes2, i, currentIndex);
					bestSol = solve(currentIndex + 1, currentSol
							+ weightProvider.getConnectionWeight(nodes1
									.get(currentIndex), nodes2
									.get(currentIndex)));
					Collections.swap(nodes2, i, currentIndex);
				}
			}

			return bestSol;
		}
	}
}