/*--------------------------------------------------------------------------+
$Id: FileSystemUtilsTest.java 29696 2010-08-13 10:31:31Z deissenb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.filesystem;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.GZIPOutputStream;

import edu.tum.cs.commons.collections.CollectionUtils;
import edu.tum.cs.commons.string.StringUtils;
import edu.tum.cs.commons.test.CCSMTestCaseBase;

/**
 * JUnit test for <code>FileSystemUtils</code> class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: deissenb $
 * @version $Rev: 29696 $
 * @levd.rating GREEN Hash: 498E4FD5E3E12E0FBA21A70CED66ECEB
 */
public class FileSystemUtilsTest extends CCSMTestCaseBase {

	/** Content of test files. */
	private static final String FILE_CONTENT = "test_file";

	/** Filenames used in {@link #createDirectoryStructure()} */
	private static final String[] filenames = { "file-1.txt",
			"directory-1/file-1.txt", "directory-1/file-2.txt",
			"directory-1/directory-1/file1.txt",
			"directory-1/directory-1/file2.bak",
			"directory-1/directory-2/file1.txt",
			"directory-1/directory-2/file2.bak",
			"directory-1/directory-2/directory-1/file1.txt",
			"directory-1/directory-2/directory-1/file2.txt",
			"directory-1/directory-2/directory-1/file3.bak" };

	/**
	 * Test method {@link FileSystemUtils#copyFile(File, File)}
	 */
	public void testCopyFile() throws IOException {
		String content = "Test\nString\n";

		File sourceFile = createTmpFile("test.txt", content);

		File targetFile = new File(getTmpDirectory(), "test2.txt");

		FileSystemUtils.copyFile(sourceFile, targetFile);

		String readContent = FileSystemUtils.readFile(targetFile);
		assertEquals(content, readContent);
	}

	/**
	 * Test method for
	 * {@link FileSystemUtils#copyFiles(File, File, java.io.FileFilter)}
	 * 
	 * @throws IOException
	 */
	public void testCopyFiles() throws IOException {
		File source = createDirectoryStructure();
		File target = new File(getTmpDirectory(), "target");

		int fileCount = FileSystemUtils.copyFiles(source, target,
				new FileOnlyFilter());

		assertEquals(filenames.length, fileCount);

		for (String filename : filenames) {
			File originalFile = new File(source, filename);
			File copiedFile = new File(target, filename);
			assertEquals(originalFile.length(), copiedFile.length());
		}
	}

	/**
	 * Test method for
	 * {@link FileSystemUtils#copy(java.io.InputStream, java.io.OutputStream)}
	 * 
	 * @throws IOException
	 */
	public void testCopyStreams() throws IOException {
		String content = StringUtils.randomString(2000);
		File sourceFile = createTmpFile("stream_source.txt", content);
		InputStream in = new FileInputStream(sourceFile);
		File targetFile = new File(getTmpDirectory(), "stream_target.txt");
		OutputStream out = new FileOutputStream(targetFile);

		FileSystemUtils.copy(in, out);

		in.close();
		out.close();

		assertEquals(content, FileSystemUtils.readFile(targetFile));
	}

	/**
	 * Test
	 * {@link FileSystemUtils#createJARFile(File, File, java.io.FileFilter)}
	 */
	public void testCreateJARFile() throws IOException {
		File source = createDirectoryStructure();
		File jarFile = new File("test.jar");

		int fileCount = FileSystemUtils.createJARFile(jarFile, source,
				new FileOnlyFilter());

		assertEquals(filenames.length, fileCount);

		HashSet<String> files = CollectionUtils.asHashSet(filenames);

		JarFile jar = new JarFile(jarFile);
		Enumeration<JarEntry> entries = jar.entries();
		while (entries.hasMoreElements()) {
			JarEntry entry = entries.nextElement();
			assertTrue(files.contains(entry.getName()));
		}
		jar.close();
		jarFile.delete();
	}

	/**
	 * Test method for {@link FileSystemUtils#unjar(File, File)}
	 */
	public void testUnjar() throws IOException {
		File source = createDirectoryStructure();
		File jarFile = new File("test.jar");
		FileSystemUtils.createJARFile(jarFile, source, new FileOnlyFilter());

		File targetDirectory = new File(getTmpDirectory(), "target");

		FileSystemUtils.unjar(jarFile, targetDirectory);

		List<File> files = FileSystemUtils.listFilesRecursively(
				targetDirectory, new FileOnlyFilter());

		assertEquals(filenames.length, files.size());

		for (File file : files) {
			assertEquals(FILE_CONTENT.length(), file.length());
		}

		jarFile.delete();

	}

	/**
	 * Test for {@link FileSystemUtils#createRelativePath(File, File)}.
	 */
	public void testCreateRelativePath() throws IOException {
		createDirectoryStructure();
		File dir1 = new File(getTmpDirectory(), "directory-1");
		File dir1dir1 = new File(getTmpDirectory(), "directory-1/directory-1");
		File dir1dir2 = new File(getTmpDirectory(), "directory-1/directory-2");
		File dir1dir2dir1 = new File(getTmpDirectory(),
				"directory-1/directory-2/directory-1");

		String path = FileSystemUtils.createRelativePath(dir1, dir1);
		assertEquals("", path);

		path = FileSystemUtils.createRelativePath(dir1, dir1dir1);
		assertEquals("../", path);

		path = FileSystemUtils.createRelativePath(dir1dir1, dir1);
		assertEquals("directory-1/", path);

		path = FileSystemUtils.createRelativePath(dir1dir2dir1, dir1);
		assertEquals("directory-2/directory-1/", path);

		path = FileSystemUtils.createRelativePath(dir1, dir1dir2dir1);
		assertEquals("../../", path);

		path = FileSystemUtils.createRelativePath(dir1dir1, dir1dir2);
		assertEquals("../directory-1/", path);

		path = FileSystemUtils.createRelativePath(dir1dir2, dir1dir1);
		assertEquals("../directory-2/", path);

		path = FileSystemUtils.createRelativePath(dir1dir1, dir1dir2dir1);
		assertEquals("../../directory-1/", path);

		path = FileSystemUtils.createRelativePath(dir1dir2dir1, dir1dir1);
		assertEquals("../directory-2/directory-1/", path);
	}

	/**
	 * Test method for {@link FileSystemUtils#deleteRecursively(File)}
	 */
	public void testDeleteRecursively() throws IOException {
		File root = createDirectoryStructure();

		FileSystemUtils.deleteRecursively(root);

		assertFalse(root.exists());
	}

	/**
	 * Test method for {@link FileSystemUtils#ensureDirectoryExists(File)}.
	 */
	public void testEnsureDirectoryExists() throws IOException {
		// we do not use methods in CCSMTestCaseBase as they already include
		// class to the tested method

		File file = new File("tmp");

		// delete in case it should be there
		file.delete();

		FileSystemUtils.ensureDirectoryExists(file);

		assertTrue(file.exists());
		assertTrue(file.isDirectory());

		file.delete();
	}

	/**
	 * Test method for {@link FileSystemUtils#ensureParentDirectoryExists(File)}
	 */
	public void testEnsureParentDirectoryExists() throws IOException {
		// we do not use methods in CCSMTestCaseBase as they already include
		// the tested method

		File file = new File("tmp" + File.separator + "test.txt");

		// delete in case it should be there
		file.getParentFile().delete();

		FileSystemUtils.ensureDirectoryExists(file);

		assertTrue(file.getParentFile().exists());
		assertTrue(file.getParentFile().isDirectory());

		file.delete();
	}

	/**
	 * Test method for {@link FileSystemUtils#ensureParentDirectoryExists(File)}
	 * . This tests CR #2462.
	 */
	public void testEnsureParentDirectoryExistsFileWithoutParent()
			throws IOException {
		FileSystemUtils.ensureParentDirectoryExists(new File("test"));
		// no need to delete the directory as this is the current working
		// directory.
	}

	/**
	 * Test method for
	 * {@link FileSystemUtils#listFilesRecursively(File, java.io.FileFilter)}
	 * and {@link FileExtensionFilter}.
	 */
	public void testFileExtensionFilter() throws IOException {
		File root = createDirectoryStructure();

		List<File> files = FileSystemUtils.listFilesRecursively(root,
				new FileExtensionFilter("txt", "bak"));

		for (File file : files) {
			assertTrue(file.getAbsolutePath().endsWith(".txt")
					|| file.getAbsolutePath().endsWith(".bak"));
		}

		for (String filename : filenames) {
			if (filename.endsWith(".txt") || filename.endsWith("bak")) {
				File file = new File(getTmpDirectory(), filename);
				assertTrue(file + "", files.contains(file));
			}
		}
	}

	/**
	 * Test method for
	 * {@link FileSystemUtils#listFilesRecursively(File, java.io.FileFilter)}
	 * and {@link FileOnlyFilter}.
	 */
	public void testFileOnlyFilter() throws IOException {
		File root = createDirectoryStructure();

		List<File> files = FileSystemUtils.listFilesRecursively(root,
				new FileOnlyFilter());

		assertEquals(files.size(), filenames.length);

		for (File file : files) {
			assertTrue(file.isFile());
		}

		for (String filename : filenames) {

			File file = new File(getTmpDirectory(), filename);
			assertTrue(file + "", files.contains(file));

		}
	}

	/**
	 * Test method for {@link FileSystemUtils#listFilesRecursively(File)}.
	 */
	public void testListRecursively() throws IOException {
		File root = createDirectoryStructure();

		List<File> files = FileSystemUtils.listFilesRecursively(root);

		for (String filename : filenames) {
			File file = new File(getTmpDirectory(), filename);
			assertTrue(files.contains(file));

			File directory = file.getParentFile();

			if (!directory.equals(getTmpDirectory())) {
				assertTrue(directory + " ist not in list", files
						.contains(directory));
			}

		}

	}

	/**
	 * Test for {@link FileSystemUtils#newFile(File, String[])}
	 */
	public void testNewFile() {
		File parent = new File("parent");
		File newFile = FileSystemUtils.newFile(parent);
		assertEquals(parent, newFile);

		newFile = FileSystemUtils.newFile(parent, "child1");
		assertEquals("parent" + File.separator + "child1", newFile.getPath());

		newFile = FileSystemUtils.newFile(parent, "child1", "child2");
		assertEquals("parent" + File.separator + "child1" + File.separator
				+ "child2", newFile.getPath());

		newFile = FileSystemUtils.newFile(parent, "child1", "child2", "child3");
		assertEquals("parent" + File.separator + "child1" + File.separator
				+ "child2" + File.separator + "child3", newFile.getPath());
	}

	/**
	 * Test for {@link FileSystemUtils#readFile(File)} and
	 * {@link FileSystemUtils#writeFile(File, String)}.
	 */
	public void testReadWriteFile() throws IOException {
		String content = "Test\nString\n";
		File file = new File(getTmpDirectory() + "test.txt");
		FileSystemUtils.writeFile(file, content);
		String readContent = FileSystemUtils.readFile(file);
		assertEquals(content, readContent);
	}

	
	/** Tests reading and writing in the presence of UTF-8 BOMs. */
	public void testReadFileUTF() throws IOException {
		File file = new File(getTmpDirectory() + "test.txt");
		final String content = "Test\nString\n\u00C4\u00D6\u00DC\u00E4\u00F6\u00FC";

		for (EByteOrderMark bom : EByteOrderMark.values()) {
			FileSystemUtils.writeFileWithBOM(file, content, bom);

			// we use the default charset, but readFile should be smart enough
			// to switch to the correct encoding
			String read = FileSystemUtils.readFile(file, Charset
					.defaultCharset().name());
			assertEquals("Expected correct content for encoding "
					+ bom.getEncoding(), content, read);
		}
	}

	/**
	 * Create a test directory structure in {@link #getTmpDirectory()} using the
	 * filenames defined by {@link #filenames}.
	 */
	private File createDirectoryStructure() throws IOException {
		deleteTmpDirectory();

		for (String filename : filenames) {
			createTmpFile(filename, FILE_CONTENT);
		}

		return getTmpDirectory();
	}

	/**
	 * Test {@link FileSystemUtils#getFileExtension(File)} with files with and
	 * without extension.
	 */
	public void testGetFileExtension() {
		assertEquals("a", extension("name.a"));
		assertEquals("ab", extension("name.ab"));
		assertEquals("abc", extension("othername.abc"));
		assertEquals("java", extension("name with spaces.java"));

		assertEquals("c", extension("a.b.c"));

		assertEquals("cs", extension("c:/windows/system21/somefile.cs"));

		assertEquals(null, extension("name_without_extension"));
		assertEquals(null, extension(StringUtils.EMPTY_STRING));
		assertEquals(StringUtils.EMPTY_STRING, extension("somename."));
	}

	/**
	 * Helper method that wraps creation of file and call to
	 * {@link FileSystemUtils#getFileExtension(File)}
	 */
	private String extension(String filename) {
		return FileSystemUtils.getFileExtension(new File(filename));
	}

	/**
	 * Test if errors are handled correctly in
	 * {@link FileSystemUtils#mergeTemplate(File, File, Object...)}.
	 */
	public void testMergeTemplateError() {
		File templateFile = useTestFile("template01.txt");
		File outFile = new File(getTmpDirectory(), "out1.txt");
		try {
			FileSystemUtils.mergeTemplate(templateFile, outFile);
			fail();
		} catch (IOException e) {
			// expected
		}
	}

	/**
	 * Simple test for relative indexing in
	 * {@link FileSystemUtils#mergeTemplate(File, File, Object...)}
	 */
	public void testMergeTemplate() throws IOException {
		File templateFile = useTestFile("template01.txt");
		File outFile = new File(getTmpDirectory(), "out1.txt");
		FileSystemUtils.mergeTemplate(templateFile, outFile, "arg1");
		assertEquals("test template: arg1", FileSystemUtils.readFile(outFile));
	}

	/** Test {@link FileSystemUtils#readStream(java.io.InputStream)} */
	public void testReadStream() throws IOException {
		FileInputStream stream = new FileInputStream(useTestFile("input01.txt"));
		assertEquals("1234567890" + StringUtils.CR + "1234567890",
				FileSystemUtils.readStream(stream));
		stream.close();
	}

	/**
	 * Test
	 * {@link FileSystemUtils#mergeTemplate(java.io.InputStream, Object...)}
	 */
	public void testMergeTemplateStreams() throws IOException {
		InputStream inStream = new FileInputStream(
				useTestFile("template01.txt"));
		InputStream stream = FileSystemUtils.mergeTemplate(inStream, "arg1");
		assertEquals("test template: arg1", FileSystemUtils.readStream(stream));
		inStream.close();
		stream.close();
	}

	/**
	 * Assert correct functionality for different and inconsistent file
	 * delimiter styles for {@link FileSystemUtils#commonRoot(Iterable)}
	 */
	public void testCommonRootDifferentPathStyles() {
		String root = determineRoot();
		assertCommonRoot(new String[] { root.toLowerCase() + "root/a.txt",
				root + "root" + File.separator + "b.txt",
				root.toUpperCase() + "root/c.txt" }, new File(root + "root/"));
	}

	/** Determines the root to be used. */
	private static String determineRoot() {
		return File.listRoots()[0].getPath();
	}

	/**
	 * Test case for typical case behavior of
	 * {@link FileSystemUtils#commonRoot(Iterable)}
	 */
	public void testCommonRootNeighbouringFiles() {
		String root = determineRoot() + "root/";
		assertCommonRoot(new String[] { root + "a.txt", root + "b.txt",
				root + "c.txt" }, new File(root));
	}

	/**
	 * Assert correct functionality for files with the same prefix for
	 * {@link FileSystemUtils#commonRoot(Iterable)}
	 */
	public void testCommonRootFilesWithEqualPrefix() {
		String root = determineRoot() + "root/";
		assertCommonRoot(new String[] { root + "PREa.txt", root + "PREb.txt" },
				new File(root));
	}

	/**
	 * Assert that an exception is raised if no common root directory can be
	 * determined for {@link FileSystemUtils#commonRoot(Iterable)}
	 */
	public void testCommonRootNoRoot() {
		// this test makes sense for windows only, as unix systems have only one
		// root
		if (System.getProperty("os.name").startsWith("Windows")) {
			assertCommonRoot(
					new String[] { "a:/root1/a.txt", "b:/root2/b.txt" }, null);
		}
	}

	/** Call {@link FileSystemUtils#commonRoot(Iterable)} and assert result */
	private void assertCommonRoot(String[] paths, File expectedRoot) {
		List<File> files = new ArrayList<File>();
		for (String path : paths) {
			files.add(new File(path));
		}

		File actualRoot = null;
		try {
			actualRoot = FileSystemUtils.commonRoot(files);
		} catch (IOException e) {
			fail("IOException thrown: " + e.getMessage());
		}
		assertEquals(expectedRoot, actualRoot);
	}

	/**
	 * Tests the {@link FileSystemUtils#autoDecompressStream(InputStream)}
	 * method.
	 */
	public void testAutoDecompressStream() throws IOException {
		byte[] data = "Some redundant text to actually allow compression, redundant, compression!"
				.getBytes();
		byte[] result = new byte[2 * data.length];

		// uncompressed case
		ByteArrayInputStream bis = new ByteArrayInputStream(data);
		int read = FileSystemUtils.autoDecompressStream(bis).read(result);
		assertEquals(data.length, read);
		for (int i = 0; i < data.length; ++i) {
			assertEquals(data[i], result[i]);
		}

		// compressed case
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		GZIPOutputStream out = new GZIPOutputStream(bos);
		out.write(data);
		out.close();
		bis = new ByteArrayInputStream(bos.toByteArray());
		read = FileSystemUtils.autoDecompressStream(bis).read(result);
		assertEquals(data.length, read);
		for (int i = 0; i < data.length; ++i) {
			assertEquals(data[i], result[i]);
		}
	}

	/**
	 * Tests the {@link FileSystemUtils#extractJarFileFromJarURL(java.net.URL)}
	 * method.
	 */
	public void testExtractJarFileFromJarURL() {
		// we use one of the JRE JAR files for this test
		URL jreURL = Integer.class.getResource(Integer.class.getSimpleName()
				+ ".class");
		assertEquals("jar", jreURL.getProtocol());

		File jarFile = FileSystemUtils.extractJarFileFromJarURL(jreURL);
		assertTrue("JDK JARs should exist!", jarFile.isFile());
		assertTrue(jarFile.getPath().toLowerCase().endsWith(".jar"));
	}

	/** Test case for {@link FileSystemUtils#isAbsolutePath(String)} */
	public void testIsAbsolutePath() {
		String[] absolutePaths = new String[] { "c:\\a.txt", "C:\\b.txt",
				"f:/a/b.txt", "Y:\\a", "C:\\", "z:/", "\\\\smbbroy\\juergens",
				"~/usr/html.txt", "/usr/trala/a.txt", "/usr/trala/a.txt" };
		for (String absolutePath : absolutePaths) {
			assertTrue("Path '" + absolutePath
					+ "' is considered not absolute, but should be!",
					FileSystemUtils.isAbsolutePath(absolutePath));
		}

		String[] relativePaths = new String[] { "a", "a.txt", "a/b/c",
				"a\\b\\c", "a\\b.txt" };
		for (String relativePath : relativePaths) {
			assertFalse("Path '" + relativePath
					+ "' is considered not relative, but should be!",
					FileSystemUtils.isAbsolutePath(relativePath));
		}
	}

	/** Test for {@link FileSystemUtils#safeRead(InputStream, byte[])} */
	public void testSafeReadHappyCase() throws Exception {
		byte[] sampleData = new byte[] { 1, 2, 3 };
		ByteArrayInputStream bis = new ByteArrayInputStream(sampleData);
		byte[] buf = new byte[3];
		FileSystemUtils.safeRead(bis, buf);
		assertTrue(Arrays.equals(sampleData, buf));
	}

	/** Test for {@link FileSystemUtils#safeRead(InputStream, byte[])} */
	public void testSafeReadExceptionCase() throws Exception {
		byte[] sampleData = new byte[] { 1, 2, 3 };
		ByteArrayInputStream bis = new ByteArrayInputStream(sampleData);
		try {
			FileSystemUtils.safeRead(bis, new byte[4]);
			fail("EOFException expected");
		} catch (EOFException expected) {
			// expected exception
		}
	}

	/** Test for {@link FileSystemUtils#safeRead(InputStream, byte[])} */
	public void testSafeReadMultipleReads() throws Exception {
		ByteArrayInputStream bis = new ByteArrayInputStream(new byte[0]) {
			private boolean firstCall = true;

			@Override
			public synchronized int read(byte[] b, int off, int len) {
				if (firstCall) {
					b[0] = 1;
					firstCall = false;
					return 1;
				}
				b[1] = 2;
				b[2] = 3;
				return 2;
			}
		};
		byte[] buf = new byte[3];
		FileSystemUtils.safeRead(bis, buf);
		assertTrue(Arrays.equals(new byte[] { 1, 2, 3 }, buf));
	}
}
