/*--------------------------------------------------------------------------+
$Id: LineSplitterTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.string;

import junit.framework.TestCase;

/**
 * Test class for the <code>LineSplitter</code>.
 * 
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 4490A53F9CD17ED7F50B2CAD5E6A8AD2
 */
public class LineSplitterTest extends TestCase {

    /** Splitter under test. */
    private final LineSplitter splitter = new LineSplitter();

    /** Test if splitter returns null if content is not set. */
    public void testUninitializedContent() {
        assertNull(splitter.getNextLine());
    }
    
    /** Test if explicitly set null content returns null. */
    public void testNullContent() {
        splitter.setContent(null);
        assertNull(splitter.getNextLine());
    }

    /** Test if empty strings yields null. */
    public void testEmptyContent() {
        splitter.setContent(StringUtils.EMPTY_STRING);
        assertNull(splitter.getNextLine());
    }

    /** Test if string without line terminator works correctly. */
    public void testOneLineContent() {
        splitter.setContent("test");
        assertEquals("test", splitter.getNextLine());
        assertNull(splitter.getNextLine());
    }

    /** Test if splitter works correctly for line terminator '\n' */
    public void testMultiLineContentN() {
        test("\n");
    }

    /** Test if splitter works correctly for line terminator '\r\n' */
    public void testMultiLineContentRN() {
        test("\r\n");
    }

    /** Test if splitter works correctly for line terminator '\r' */
    public void testMultiLineContentR() {
        test("\r");
    }

    /**
     * Test if splitter works correctly for line terminator obtained from
     * <code>System.getProperty("line.separator")</code> (this should be one
     * of '\n', '\r', '\r\n').
     */
    public void testMultiLineContentNative() {
        test(System.getProperty("line.separator"));
    }

    /**
     * This method creates a random string array, concatenates it with the
     * specified line terminator to a single string and tests if the splitter
     * works correctly.
     * 
     * @param lineTerminator
     */
    private void test(String lineTerminator) {
        String[] array = StringUtils.randomStringArray(100, 80);
        String test = StringUtils.concat(array, lineTerminator);
        splitter.setContent(test);
        for (String expected : array) {
            assertEquals(expected, splitter.getNextLine());
        }
        assertNull(splitter.getNextLine());
    }
}