/*--------------------------------------------------------------------------+
$Id: SimulinkModelBuilderTest.java 26285 2010-02-18 11:22:54Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.simulink.builder;

import java.io.FileNotFoundException;

import edu.tum.cs.commons.collections.IdentityHashSet;
import edu.tum.cs.commons.test.DeepCloneTestUtils;
import edu.tum.cs.simulink.model.SimulinkAnnotation;
import edu.tum.cs.simulink.model.SimulinkBlock;
import edu.tum.cs.simulink.model.SimulinkInPort;
import edu.tum.cs.simulink.model.SimulinkLine;
import edu.tum.cs.simulink.model.SimulinkModel;
import edu.tum.cs.simulink.model.SimulinkModelTest;
import edu.tum.cs.simulink.model.SimulinkOutPort;
import edu.tum.cs.simulink.testutils.SimulinkTestBase;
import edu.tum.cs.simulink.util.SimulinkUtils;

/**
 * Tests for Simulink builder. This tests if the model is <em>built</em>
 * properly. Correct implementation of the model itself, e.g. deep cloning,
 * element removal, is tested in {@link SimulinkModelTest}.
 * <p>
 * The test specified here are far from complete but meant as basis to
 * conveniently add more tests.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26285 $
 * @levd.rating GREEN Hash: 08560ACF89C9D05191BD188B36F805AD
 */
public class SimulinkModelBuilderTest extends SimulinkTestBase {

	/** Test if annotations are built correctly. */
	public void testAnnotations() throws FileNotFoundException,
			SimulinkModelBuildingException {

		SimulinkModel model = loadModel("annotations.mdl");
		assertEquals(2, model.getAnnotations().size());
		assertEquals(1, model.getBlock("annotations/Strecke").getAnnotations()
				.size());
		assertEquals(0, model.getBlock("annotations/Sum").getAnnotations()
				.size());

		SimulinkAnnotation annotation = model.getBlock("annotations/Strecke")
				.getAnnotations().iterator().next();
		assertEquals("DeepAnnotation", annotation.getName());

		// directly defined parameter
		assertTrue(annotation.getParameterNames().contains("Position"));
		assertEquals("[293, 181]", annotation.getParameter("Position"));

		// default parameter
		assertTrue(annotation.getParameterNames().contains("DropShadow"));
		assertEquals("off", annotation.getParameter("DropShadow"));

	}

	/** Test if blocks are built correctly. */
	public void testBlocks() throws FileNotFoundException,
			SimulinkModelBuildingException {

		SimulinkModel model = loadModel("model01.mdl");

		assertEquals(25, SimulinkUtils.createIdToNodeMap(model).size());

		checkBlock(model, "model01", 0, 0);
		SimulinkBlock block = checkBlock(model, "model01/Controller/PID", 1, 1);
		checkBlock(model, "model01/Controller/PID/Sum1", 2, 1);

		// directly defined parameters

		// parameter is part of the "Block" section in the MDL file
		assertTrue(block.getParameterNames().contains("Position"));
		assertEquals("[230, 25, 270, 85]", block.getParameter("Position"));

		// parameter is part of the "System" section in the MDL file (the block
		// describes a subsystem)
		assertTrue(block.getParameterNames().contains("System.ZoomFactor"));
		assertEquals("100", block.getParameter("System.ZoomFactor"));

		// default parameter
		assertTrue(block.getParameterNames().contains("DropShadow"));
		assertEquals("off", block.getParameter("DropShadow"));
	}

	/** Check if branches in lines are resolved correctly. */
	public void testBranching() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModel model = loadModel("branching.mdl");

		checkLine(model, "branching/In1", "1", "branching/Out1", "1");
		checkLine(model, "branching/In1", "1", "branching/Out2", "1");
		checkLine(model, "branching/In1", "1", "branching/Out3", "1");
		checkLine(model, "branching/In1", "1", "branching/Out4", "1");

	}

	/** Test if lines are built correctly. */
	public void testLines() throws FileNotFoundException,
			SimulinkModelBuildingException {

		SimulinkModel model = loadModel("model01.mdl");

		checkLine(model, "model01/Controller/PID/Sum1", "1",
				"model01/Controller/PID/Sum", "1");
		SimulinkLine line = checkLine(model, "model01/Controller/PID/Sum1",
				"1", "model01/Controller/PID/I-Delay", "1");

		System.out.println(line);

		// parameter
		assertTrue(line.getParameterNames().contains("Name"));
		assertEquals("TestLine", line.getParameter("Name"));

		// default parameter
		assertTrue(line.getParameterNames().contains("FontName"));
		assertEquals("arial", line.getParameter("FontName"));
	}

	/**
	 * Test if ports are built correctly. This test loads a special MDL file
	 * that includes all block types and guarantees that all ports of the blocks
	 * are connected. This test iterates over all ports and checks if they are
	 * connected. If we find an unconnected port, this must have been made up by
	 * the builder. If the builder would built to few ports, an exception is
	 * thrown as lines cannot be connected.
	 */
	public void testPorts() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModel model = loadModel("ports.mdl");

		IdentityHashSet<SimulinkBlock> blocks = DeepCloneTestUtils
				.getAllReferencedObjects(model, SimulinkBlock.class,
						SimulinkBlock.class.getPackage().getName());

		for (SimulinkBlock block : blocks) {
			if (block.getType().equals("SubSystem")) {
				continue;
			}
			for (SimulinkInPort inPort : block.getInPorts()) {
				assertNotNull("Type " + block.getType(), inPort.getLine());
			}
			for (SimulinkOutPort outPort : block.getOutPorts()) {
				assertFalse("Type " + block.getType(), outPort.getLines()
						.isEmpty());
			}
		}

	}

	/** Check if unknown port types are reported correctly. */
	public void testUnknownPorts() throws FileNotFoundException {
		try {
			loadModel("internal_dc_motor_lib.mdl");
			fail("Error for unknwon port expected.");
		} catch (SimulinkModelBuildingException e) {
			assertTrue(e.getMessage().contains("RCONN"));
		}
	}

	/**
	 * Obtains a block specified by an id from the model and check if has the
	 * correct number of in and outports.
	 * 
	 * @return the block
	 */
	private SimulinkBlock checkBlock(SimulinkModel model, String id,
			int inPortCount, int outPortCount) {
		SimulinkBlock block = model.getBlock(id);
		assertNotNull(block);

		assertEquals(inPortCount, block.getInPorts().size());
		assertEquals(outPortCount, block.getOutPorts().size());

		return block;
	}

	/**
	 * Obtains a source and destination port from the model and ensures that
	 * they are connected by a line.
	 * 
	 * @return the connecting line
	 */
	private SimulinkLine checkLine(SimulinkModel model, String srcBlockId,
			String srcPortIndex, String dstBlockId, String dstPortIndex) {

		SimulinkBlock srcBlock = model.getBlock(srcBlockId);
		SimulinkBlock dstBlock = model.getBlock(dstBlockId);

		SimulinkOutPort srcPort = srcBlock.getOutPort(srcPortIndex);
		SimulinkInPort dstPort = dstBlock.getInPort(dstPortIndex);

		for (SimulinkLine line : srcPort.getLines()) {
			if (line.getDstPort() == dstPort) {
				return line;
			}
		}

		fail();

		return null;
	}
}