package org.jboss.cache.factories;

import org.jboss.cache.CacheException;
import org.jboss.cache.CacheSPI;
import org.jboss.cache.DataContainer;
import org.jboss.cache.Fqn;
import org.jboss.cache.RPCManager;
import org.jboss.cache.buddyreplication.BuddyFqnTransformer;
import org.jboss.cache.buddyreplication.BuddyGroup;
import org.jboss.cache.buddyreplication.BuddyManager;
import org.jboss.cache.commands.DataCommand;
import org.jboss.cache.commands.ReplicableCommand;
import org.jboss.cache.commands.ReversibleCommand;
import org.jboss.cache.commands.read.ExistsCommand;
import org.jboss.cache.commands.read.GetChildrenNamesCommand;
import org.jboss.cache.commands.read.GetDataMapCommand;
import org.jboss.cache.commands.read.GetKeyValueCommand;
import org.jboss.cache.commands.read.GetKeysCommand;
import org.jboss.cache.commands.read.GetNodeCommand;
import org.jboss.cache.commands.read.GravitateDataCommand;
import org.jboss.cache.commands.remote.AnnounceBuddyPoolNameCommand;
import org.jboss.cache.commands.remote.AssignToBuddyGroupCommand;
import org.jboss.cache.commands.remote.ClusteredGetCommand;
import org.jboss.cache.commands.remote.DataGravitationCleanupCommand;
import org.jboss.cache.commands.remote.RemoveFromBuddyGroupCommand;
import org.jboss.cache.commands.remote.ReplicateCommand;
import org.jboss.cache.commands.tx.CommitCommand;
import org.jboss.cache.commands.tx.OptimisticPrepareCommand;
import org.jboss.cache.commands.tx.PrepareCommand;
import org.jboss.cache.commands.tx.RollbackCommand;
import org.jboss.cache.commands.write.*;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.factories.annotations.Inject;
import org.jboss.cache.factories.annotations.NonVolatile;
import org.jboss.cache.interceptors.InterceptorChain;
import org.jboss.cache.notifications.Notifier;
import org.jboss.cache.transaction.GlobalTransaction;
import org.jboss.cache.transaction.TransactionTable;
import org.jgroups.Address;

import javax.transaction.TransactionManager;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Factory for all types of cache commands.
 * Here are some of the purposes of this class:
 * <pre>
 *   - not creating <code>CacheCommands</code> directly (i.e. through new usage) as this would reduce unit testability
 *   - reduce the coupling between commands and other components. e.g. considering a commands that needs to knwo whether
 *     locking type is optimistic, we will pass in a 'optimistic' boolean flag rather than entire Configuration object
 * </pre>
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
@NonVolatile
public class CommandsFactory
{
   private RPCManager rpcManager;
   private DataContainer dataContainer;
   private Notifier notifier;
   private InterceptorChain invoker;
   private BuddyManager buddyManager;
   private TransactionTable transactionTable;
   private CacheSPI cacheSpi;
   private Configuration configuration;
   private TransactionManager txManager;
   private BuddyFqnTransformer buddyFqnTransformer;

   public CommandsFactory()
   {
   }

   @Inject
   public void initialize(RPCManager rpc, DataContainer dataContainer, Notifier notifier, BuddyManager buddyManager,
                          InterceptorChain invoker, TransactionTable transactionTable, CacheSPI cacheSpi,
                          Configuration configuration, TransactionManager txManager, BuddyFqnTransformer buddyFqnTransformer)
   {
      this.rpcManager = rpc;
      this.dataContainer = dataContainer;
      this.notifier = notifier;
      this.buddyManager = buddyManager;
      this.invoker = invoker;
      this.transactionTable = transactionTable;
      this.cacheSpi = cacheSpi;
      this.configuration = configuration;
      this.txManager = txManager;
      this.buddyFqnTransformer = buddyFqnTransformer;
   }

   public PutDataMapCommand buildPutDataMapCommand(GlobalTransaction gtx, Fqn fqn, Map data)
   {
      PutDataMapCommand cmd = new PutDataMapCommand(gtx, fqn, data);
      cmd.initialize(notifier, dataContainer);
      return cmd;
   }

   public PutKeyValueCommand buildPutKeyValueCommand(GlobalTransaction gtx, Fqn fqn, Object key, Object value)
   {
      PutKeyValueCommand command = new PutKeyValueCommand(gtx, fqn, key, value);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public PutForExternalReadCommand buildPutForExternalReadCommand(GlobalTransaction gtx, Fqn fqn, Object key, Object value)
   {
      PutForExternalReadCommand command = new PutForExternalReadCommand(gtx, fqn, key, value);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public ReplicateCommand buildReplicateCommand(ReplicableCommand command)
   {
      ReplicateCommand cmd = new ReplicateCommand(command);
      cmd.initialize(invoker);
      return cmd;
   }

   public ReplicateCommand buildReplicateCommand(List<ReplicableCommand> modifications)
   {
      ReplicateCommand cmd = new ReplicateCommand(modifications);
      cmd.initialize(invoker);
      return cmd;
   }

   public PrepareCommand buildPrepareCommand(GlobalTransaction gtx, ReversibleCommand command, boolean onePhaseCommit)
   {
      return buildPrepareCommand(gtx, Collections.singletonList(command), rpcManager.getLocalAddress(), onePhaseCommit);
   }

   public PrepareCommand buildPrepareCommand(GlobalTransaction gtx, List<ReversibleCommand> modifications, Address address, boolean onePhaseCommit)
   {
      return new PrepareCommand(gtx, modifications, address, onePhaseCommit);
   }

   public CommitCommand buildCommitCommand(GlobalTransaction gtx)
   {
      return new CommitCommand(gtx);
   }

   public DataGravitationCleanupCommand buildDataGravitationCleanupCommand(Fqn primaryFqn, Fqn backupFqn)
   {
      DataGravitationCleanupCommand command = new DataGravitationCleanupCommand(primaryFqn, backupFqn);
      command.initialize(buddyManager, invoker, transactionTable, this, dataContainer, buddyFqnTransformer);
      return command;
   }

   public GravitateDataCommand buildGravitateDataCommand(Fqn fqn, Boolean searchSubtrees)
   {
      GravitateDataCommand command = new GravitateDataCommand(fqn, searchSubtrees, rpcManager.getLocalAddress());
      command.initialize(dataContainer, cacheSpi, buddyFqnTransformer);
      return command;
   }

   public RemoveNodeCommand buildRemoveNodeCommand(GlobalTransaction gtx, Fqn fqn)
   {
      RemoveNodeCommand command = new RemoveNodeCommand(gtx, fqn);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public ClearDataCommand buildClearDataCommand(GlobalTransaction gtx, Fqn fqn)
   {
      ClearDataCommand command = new ClearDataCommand(gtx, fqn);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public EvictCommand buildEvictFqnCommand(Fqn fqn)
   {
      EvictCommand command = new EvictCommand(fqn);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public InvalidateCommand buildInvalidateCommand(Fqn fqn)
   {
      if (configuration.isNodeLockingOptimistic())
      {
         OptimisticInvalidateCommand command = new OptimisticInvalidateCommand(fqn);
         command.initialize(txManager);
         command.initialize(cacheSpi, dataContainer, notifier);
         return command;
      }
      else
      {
         InvalidateCommand command = new InvalidateCommand(fqn);
         command.initialize(cacheSpi, dataContainer, notifier);
         return command;
      }
   }

   public RemoveKeyCommand buildRemoveKeyCommand(GlobalTransaction tx, Fqn<?> fqn, Object key)
   {
      RemoveKeyCommand command = new RemoveKeyCommand(tx, fqn, key);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public GetDataMapCommand buildGetDataMapCommand(Fqn fqn)
   {
      GetDataMapCommand command = new GetDataMapCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   public ExistsCommand buildExistsNodeCommand(Fqn fqn)
   {
      ExistsCommand command = new ExistsCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   public GetKeyValueCommand buildGetKeyValueCommand(Fqn<?> fqn, Object key, boolean sendNodeEvent)
   {
      GetKeyValueCommand command = new GetKeyValueCommand(fqn, key, sendNodeEvent);
      command.initialize(dataContainer, notifier);
      return command;
   }

   public GetNodeCommand buildGetNodeCommand(Fqn fqn)
   {
      GetNodeCommand command = new GetNodeCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   public GetKeysCommand buildGetKeysCommand(Fqn fqn)
   {
      GetKeysCommand command = new GetKeysCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   public GetChildrenNamesCommand buildGetChildrenNamesCommand(Fqn fqn)
   {
      GetChildrenNamesCommand command = new GetChildrenNamesCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   public MoveCommand buildMoveCommand(Fqn from, Fqn to)
   {
      MoveCommand command = new MoveCommand(from, to);
      command.initialize(notifier, dataContainer);
      return command;
   }

   public RollbackCommand buildRollbackCommand(GlobalTransaction gtx)
   {
      return new RollbackCommand(gtx);
   }

   public OptimisticPrepareCommand buildOptimisticPrepareCommand(GlobalTransaction gtx, List<ReversibleCommand> modifications, Map data, Address address, boolean onePhaseCommit)
   {
      return new OptimisticPrepareCommand(gtx, modifications, data, address, onePhaseCommit);
   }

   public AnnounceBuddyPoolNameCommand buildAnnounceBuddyPoolNameCommand(Address address, String buddyPoolName)
   {
      AnnounceBuddyPoolNameCommand command = new AnnounceBuddyPoolNameCommand(address, buddyPoolName);
      command.initialize(buddyManager);
      return command;
   }

   public RemoveFromBuddyGroupCommand buildRemoveFromBuddyGroupCommand(String groupName)
   {
      RemoveFromBuddyGroupCommand command = new RemoveFromBuddyGroupCommand(groupName);
      command.initialize(buddyManager);
      return command;
   }

   public AssignToBuddyGroupCommand buildAssignToBuddyGroupCommand(BuddyGroup group, Map<Fqn, byte[]> state)
   {
      AssignToBuddyGroupCommand command = new AssignToBuddyGroupCommand(group, state);
      command.initialize(buddyManager);
      return command;
   }

   public ClusteredGetCommand buildClusteredGetCommand(Boolean searchBackupSubtrees, DataCommand dataCommand)
   {
      ClusteredGetCommand command = new ClusteredGetCommand(searchBackupSubtrees, dataCommand);
      command.initialize(dataContainer, invoker);
      return command;
   }

   public CreateNodeCommand buildCreateNodeCommand(Fqn fqn)
   {
      CreateNodeCommand command = new CreateNodeCommand(fqn);
      command.initialize(dataContainer);
      return command;
   }

   /**
    * Builds a cache command based on the ID passed in and an object array of parameters
    *
    * @param id         id of the command to build
    * @param parameters parameters attached to the command
    * @return a newly constructed cache command
    */
   public ReplicableCommand fromStream(int id, Object[] parameters)
   {
      ReplicableCommand command;
      switch (id)
      {
         case ExistsCommand.METHOD_ID:
         {
            ExistsCommand result = new ExistsCommand();
            result.initialize(dataContainer);
            command = result;
            break;
         }
         case GetChildrenNamesCommand.METHOD_ID:
         {
            GetChildrenNamesCommand returnValue = new GetChildrenNamesCommand();
            returnValue.initialize(dataContainer);
            command = returnValue;
            break;
         }
         case GetDataMapCommand.METHOD_ID:
         {
            GetDataMapCommand returnValue = new GetDataMapCommand();
            returnValue.initialize(dataContainer);
            command = returnValue;
            break;
         }
         case GetKeysCommand.METHOD_ID:
         {
            GetKeysCommand returnValue = new GetKeysCommand();
            returnValue.initialize(dataContainer);
            command = returnValue;
            break;
         }
         case GetKeyValueCommand.METHOD_ID:
         {
            GetKeyValueCommand returnValue = new GetKeyValueCommand();
            returnValue.initialize(dataContainer, notifier);
            command = returnValue;
            break;
         }
         case GetNodeCommand.METHOD_ID:
         {
            GetNodeCommand returnValue = new GetNodeCommand();
            returnValue.initialize(dataContainer);
            command = returnValue;
            break;
         }
         case MoveCommand.METHOD_ID:
         {
            MoveCommand returnValue = new MoveCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case PutDataMapCommand.METHOD_ID:
         case PutDataMapCommand.ERASE_METHOD_ID:
         case PutDataMapCommand.ERASE_VERSIONED_METHOD_ID:
         case PutDataMapCommand.VERSIONED_METHOD_ID:
         {
            PutDataMapCommand returnValue = new PutDataMapCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case PutKeyValueCommand.METHOD_ID:
         case PutKeyValueCommand.VERSIONED_METHOD_ID:
         {
            PutKeyValueCommand returnValue = new PutKeyValueCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case PutForExternalReadCommand.METHOD_ID:
         case PutForExternalReadCommand.VERSIONED_METHOD_ID:
         {
            PutForExternalReadCommand returnValue = new PutForExternalReadCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case ClearDataCommand.METHOD_ID:
         case ClearDataCommand.VERSIONED_METHOD_ID:
         {
            ClearDataCommand returnValue = new ClearDataCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case RemoveKeyCommand.METHOD_ID:
         case RemoveKeyCommand.VERSIONED_METHOD_ID:
         {
            RemoveKeyCommand returnValue = new RemoveKeyCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }

         case RemoveNodeCommand.METHOD_ID:
         case RemoveNodeCommand.VERSIONED_METHOD_ID:
         {
            RemoveNodeCommand returnValue = new RemoveNodeCommand();
            returnValue.initialize(notifier, dataContainer);
            command = returnValue;
            break;
         }
         case CreateNodeCommand.METHOD_ID:
         {
            CreateNodeCommand returnValue = new CreateNodeCommand();
            returnValue.initialize(dataContainer);
            command = returnValue;
            break;
         }
         // --- transactional method calls

         case PrepareCommand.METHOD_ID:
         {
            command = new PrepareCommand();
            break;
         }

         case OptimisticPrepareCommand.METHOD_ID:
         {
            command = new OptimisticPrepareCommand();
            break;
         }

         case CommitCommand.METHOD_ID:
         {
            command = new CommitCommand();
            break;
         }

         case RollbackCommand.METHOD_ID:
         {
            command = new RollbackCommand();
            break;
         }

         // --- replicate methods
         case ReplicateCommand.MULTIPLE_METHOD_ID:
         case ReplicateCommand.SINGLE_METHOD_ID:
         {
            ReplicateCommand returnValue = new ReplicateCommand();
            returnValue.initialize(invoker);
            command = returnValue;
            break;
         }

         case InvalidateCommand.METHOD_ID:
         {
            if (configuration.isNodeLockingOptimistic())
            {
               OptimisticInvalidateCommand returnValue = new OptimisticInvalidateCommand();
               returnValue.initialize(txManager);
               returnValue.initialize(cacheSpi, dataContainer, notifier);
               command = returnValue;
            }
            else
            {
               InvalidateCommand returnValue = new InvalidateCommand();
               returnValue.initialize(cacheSpi, dataContainer, notifier);
               command = returnValue;
            }
            break;
         }

         case ClusteredGetCommand.METHOD_ID:
         {
            ClusteredGetCommand returnValue = new ClusteredGetCommand();
            returnValue.initialize(dataContainer, invoker);
            command = returnValue;
            break;
         }
         // ---- Buddy replication - group organisation commands
         case AnnounceBuddyPoolNameCommand.METHOD_ID:
         {
            AnnounceBuddyPoolNameCommand returnValue = new AnnounceBuddyPoolNameCommand();
            returnValue.initialize(buddyManager);
            command = returnValue;
            break;
         }
         case AssignToBuddyGroupCommand.METHOD_ID:
         {
            AssignToBuddyGroupCommand returnValue = new AssignToBuddyGroupCommand();
            returnValue.initialize(buddyManager);
            command = returnValue;
            break;
         }
         case RemoveFromBuddyGroupCommand.METHOD_ID:
         {
            RemoveFromBuddyGroupCommand returnValue = new RemoveFromBuddyGroupCommand();
            returnValue.initialize(buddyManager);
            command = returnValue;
            break;
         }
         case DataGravitationCleanupCommand.METHOD_ID:
         {
            DataGravitationCleanupCommand returnValue = new DataGravitationCleanupCommand();
            returnValue.initialize(buddyManager, invoker, transactionTable, this, dataContainer, buddyFqnTransformer);
            command = returnValue;
            break;
         }
         case GravitateDataCommand.METHOD_ID:
         {
            GravitateDataCommand returnValue = new GravitateDataCommand(rpcManager.getLocalAddress());
            returnValue.initialize(dataContainer, cacheSpi, buddyFqnTransformer);
            command = returnValue;
            break;
         }
         default:
            throw new CacheException("Unknown command id " + id + "!");
      }

      command.setParameters(id, parameters);
      return command;
   }
}
