package JIRA::REST::Class::Sprint;
use base qw( JIRA::REST::Class::Abstract );
use strict;
use warnings;
use v5.10;

our $VERSION = '0.03';

# ABSTRACT: A helper class for C<JIRA::REST::Class> that represents the sprint of a JIRA issue as an object.

__PACKAGE__->mk_ro_accessors(qw/ id rapidViewId state name startDate endDate
                                 completeDate sequence /);

sub init {
    my $self = shift;
    $self->SUPER::init(@_);

    my $data = $self->data;
    $data =~ s{com\.atlassian\.greenhopper\.service\.sprint\.Sprint[^\[]+\[}{};
    $data =~ s{\]$}{};
    my @fields = split /,/, $data;
    foreach my $field (@fields) {
        my ($k, $v) = split /=/, $field;
        if ($v && $v eq '<null>') {
            undef $v;
        }
        $self->{$k} = $v;
    }
}

1;

#pod =accessor B<id>
#pod
#pod =accessor B<rapidViewId>
#pod
#pod =accessor B<state>
#pod
#pod =accessor B<name>
#pod
#pod =accessor B<startDate>
#pod
#pod =accessor B<endDate>
#pod
#pod =accessor B<completeDate>
#pod
#pod =accessor B<sequence>
#pod
#pod =for stopwords Atlassian ScriptRunner TODO aggregateprogress aggregatetimeestimate aggregatetimeoriginalestimate assigneeType avatar avatarUrls completeDate displayName duedate emailAddress endDate fieldtype fixVersions fromString genericized iconUrl isAssigneeTypeValid issueTypes issuelinks issuetype jira jql lastViewed maxResults originalEstimate originalEstimateSeconds parentkey projectId rapidViewId remainingEstimate remainingEstimateSeconds resolutiondate sprintlist startDate subtaskIssueTypes timeSpent timeSpentSeconds timeestimate timeoriginalestimate timespent timetracking toString updateAuthor worklog workratio
#pod
#pod
#pod =cut

=pod

=encoding UTF-8

=for :stopwords Packy Anderson Alexey Melezhik Atlassian ScriptRunner TODO
aggregateprogress aggregatetimeestimate aggregatetimeoriginalestimate
assigneeType avatar avatarUrls completeDate displayName duedate
emailAddress endDate fieldtype fixVersions fromString genericized iconUrl
isAssigneeTypeValid issueTypes issuelinks issuetype jira jql lastViewed
maxResults originalEstimate originalEstimateSeconds parentkey projectId
rapidViewId remainingEstimate remainingEstimateSeconds resolutiondate
sprintlist startDate subtaskIssueTypes timeSpent timeSpentSeconds
timeestimate timeoriginalestimate timespent timetracking toString
updateAuthor worklog workratio

=head1 NAME

JIRA::REST::Class::Sprint - A helper class for C<JIRA::REST::Class> that represents the sprint of a JIRA issue as an object.

=head1 VERSION

version 0.03

=head1 READ-ONLY ACCESSORS

=head2 B<id>

=head2 B<rapidViewId>

=head2 B<state>

=head2 B<name>

=head2 B<startDate>

=head2 B<endDate>

=head2 B<completeDate>

=head2 B<sequence>

=head1 AUTHOR

Packy Anderson <packy@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Packy Anderson.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut

__END__

# These methods don't work, probably because JIRA doesn't have a well-defined
# interface for adding/removing issues from a sprint.

sub greenhopper_api_url {
    my $self = shift;
    my $url  = $self->jira->rest_api_url_base;
    $url =~ s{/rest/api/.+}{/rest/greenhopper/latest};
    return $url;
}

sub add_issues {
    my $self = shift;
    my $url = join '/', q{},
      'sprint', $self->id, 'issues', 'add';

    my $args = { issueKeys => \@_ };
    my $host = $self->jira->{rest}->getHost;

    $self->jira->{rest}->setHost($self->greenhopper_api_url);
    $self->jira->{rest}->PUT($url, undef, $args);
    $self->jira->_content;
    $self->jira->{rest}->setHost($host);
}

sub remove_issues {
    my $self = shift;
    my $url = join '/', q{},
      'sprint', $self->id, 'issues', 'remove';
    my $args = { issueKeys => \@_ };
    my $host = $self->jira->{rest}->getHost;

    $self->jira->{rest}->setHost($self->greenhopper_api_url);
    $self->jira->{rest}->PUT($url, undef, $args);
    $self->jira->_content;
    $self->jira->{rest}->setHost($host);
}
