package Ancient;
use strict;
use warnings;
our $VERSION = '0.06';

1;

__END__

=head1 NAME

Ancient - Ancient Perl utilities

=head1 DESCRIPTION

We've had a Modern perl so with that logic I guess it's time for an Ancient version.

This distribution currently provides seven independent modules:

=over 4

=item * L<slot> - Global reactive state slots with optional watchers

=item * L<util> - Functional programming utilities with XS acceleration

=item * L<noop> - No-operation functions for benchmarking and testing

=item * L<const> - Fast read-only constants with compile-time optimization

=item * L<doubly> - Doubly linked list implementation

=item * L<lru> - LRU cache with O(1) operations

=item * L<object> - Objects with prototype chains

=back

=head1 MODULES

=head2 slot

    use slot qw(app_name debug);
    
    app_name("MyApp");
    print app_name();

Global reactive state slots shared across packages with optional
watchers for reactive programming. All slot:: functions are optimized
to custom ops when called with constant names.

See L<slot> for full documentation.

=head2 util

    use util qw(is_array is_hash memo pipeline trim ...);

Fast functional programming utilities including type predicates,
memoization, pipelines, lazy evaluation, and more. Many functions
use custom ops for compile-time optimization.

See L<util> for full documentation.

=head2 noop

    use noop;

    noop::pp();   # Pure Perl no-op
    noop::xs();   # XS no-op

Minimal no-operation functions for benchmarking overhead and
baseline performance testing.

See L<noop> for full documentation.

=head2 const

    use const;

    my $pi = const::c(3.14159);
    my $name = const::c("immutable");
    const::const(my @list => qw/a b c/);

Fast read-only constants with compile-time optimization. When called
with literal values, C<const::c()> is optimized away at compile time
for zero runtime overhead.

See L<const> for full documentation.

=head2 doubly

    use doubly;

    my $list = doubly->new(1);
    $list->add(2)->add(3);
    $list = $list->start;
    $list = $list->next;

Fast doubly linked list implementation with full navigation, insertion,
and removal operations. Approximately 3x faster than pure Perl implementations.

See L<doubly> for full documentation.

=head2 lru

    use lru;

    my $cache = lru::new(1000);
    $cache->set('key', $value);
    my $val = $cache->get('key');

LRU (Least Recently Used) cache with O(1) operations for get, set,
exists, peek, and delete. Automatic eviction when capacity is reached.

See L<lru> for full documentation.

=head2 object

    use object;

    object::define('Cat', qw(name age));
    my $cat = new Cat 'Whiskers', 3;
    print $cat->name;

Objects with prototype chains stored as arrays for speed. Property
accessors are compiled to custom ops. Getters are 2.4x faster and
setters 2x faster than traditional blessed hash references.

See L<object> for full documentation.

=head2 heap

    use heap;

    my $pq = heap::new('min');
    $pq->push(5)->push(1)->push(3);
    print $pq->pop;  # 1

Binary heap (priority queue) with configurable min/max behavior.
Supports custom comparison callbacks for complex objects.
O(log n) push and pop, O(1) peek.

See L<heap> for full documentation.

=head1 AUTHOR

LNATION <email@lnation.org>

=head1 LICENSE

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
