package OpenInteract::PageDirectory;

# $Id: PageDirectory.pm,v 1.7 2003/05/18 20:08:57 lachoy Exp $

use strict;
use File::Basename ();

$OpenInteract::PageDirectory::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

sub list_directory_actions {
    my ( $class ) = @_;
    my $CONFIG = OpenInteract::Request->instance->CONFIG;
    return [ sort grep { $CONFIG->{action}{ $_ }{is_directory} eq 'yes' }
                  keys %{ $CONFIG->{action} } ];
}


# Ensures that we search only for directories with a '/' at the end,
# and also that we find parent directories that have 'subdirs_inherit'
# flagged on

sub fetch_by_directory {
    my ( $class, $dir ) = @_;
    die "Must pass directory to fetch\n" unless ( $dir );
    my $R = OpenInteract::Request->instance;

    my $initial_dir_name = $class->_ensure_trailing_slash( $dir );
    $R->DEBUG && $R->scrib( 1, "Initial fetch attempt [$initial_dir_name]" );
    my $dir = $class->fetch( $initial_dir_name );
    return $dir if ( $dir );

    # We only want parents that have inheritance enabled

    my $where = ' subdirs_inherit = ? ';
    my $value = [ 'yes' ];

    my $all_parents = $class->explode_dir( $initial_dir_name );

    # First item is the directory itself -- don't need it

    if ( scalar @{ $all_parents } > 1 ) { shift @{ $all_parents } }

    $R->DEBUG && $R->scrib( 1, "Found parents of [$dir]: [",
                            join( '] [', @{ $all_parents } ), "]" );


    # Form a WHERE clause from the parents

    my $parent_where = join( ' OR ', map { "directory = ?" } @{ $all_parents } );
    $where .= " AND ( $parent_where ) ";
    push @{ $value }, @{ $all_parents };

    my $parent_dirs = $class->fetch_group({ where => $where,
                                            value => $value });

    # No parents found (most of the time)

    return undef unless ( ref $parent_dirs eq 'ARRAY' );
    my $num_parents = scalar @{ $parent_dirs };
    return undef unless ( $num_parents > 0 );

    # Only one parent found (what happens most of the rest of the time)

    return $parent_dirs->[0] if ( $num_parents == 1 );

    # With multiple parents, we prefer to return the one with a longer
    # name since they're closer to the original directory.

    my ( $immediate_parent );
    my $max_dir_length = 0;
    foreach my $parent_dir ( @{ $parent_dirs } ) {
        my $this_dir_length = length $parent_dir->{directory};
        if ( $this_dir_length > $max_dir_length ) {
            $immediate_parent = $parent_dir;
            $max_dir_length = $this_dir_length;
        }
    }
    return $immediate_parent;
}



sub explode_dir {
    my ( $class, $dir ) = @_;
    $dir =~ s|\\|/|g;
    my @exploded = ( $dir );
    while ( $dir ne '/' ) {
        my $next_parent = File::Basename::dirname( $dir );
        $next_parent = "$next_parent/" unless ( $next_parent eq '/' );
        push @exploded, $next_parent;
        $dir = $next_parent;
    }
    return \@exploded;
}


sub _ensure_trailing_slash {
    my ( $class, $dir ) = @_;
    return unless ( $dir );
    $dir = "$dir/" unless ( $dir =~ m|/$| );
    return $dir;
}


########################################
# RULES
########################################

# Here we add a rule so we ensure that every directory has a '/' at
# the end

sub ruleset_factory {
    my ( $class, $rs_table ) = @_;
    push @{ $rs_table->{pre_save_action} }, \&check_directory_syntax;
    return __PACKAGE__;
}

sub check_directory_syntax {
    my ( $self ) = @_;
    die "No directory defined" unless ( $self->{directory} );
    return $self->{directory} = $self->_ensure_trailing_slash( $self->{directory} );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::PageDirectory - Methods supporting the directory handler objects

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

B<fetch_by_directory( $dir )>

Try to retrieve a directory object by C<$dir>, or retrieve its closest
parent directory that has 'subdirs_inherit' set. If you are using the
directory to execute an action, you should B<always> use this.

B<explode_dir( $dir )>

Utility method that splits up C<$dir> into parent directories. So
given:

 my $dir_pieces = OpenInteract::PageDirectory->explode_dir( '/path/to/my/home/' );

You would have:

 [ '/path/to/my/home/',
   '/path/to/my/',
   '/path/to/',
   '/path/',
   '/' ]

B<list_directory_actions()>

Scans the action table and finds all available actions known to be
directory handlers.

Returns an arrayref strings representing actions.

=head1 RULES

B<pre_save_action>

Ensure that every directory saved to the database has a '/' at the
end.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut

