package Color::Library::Dictionary::NBS_ISCC::F;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::F;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::F - (NBS/ISCC F) Colors; (for) Ready-Mixed Paints

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (F) Colors; (for) Ready-Mixed Paints

    Federal Specification TT-C-595,
    Colors; (for) Ready-Mixed Paints 

This dictionary is a hybrid of combinatorial and idiosyncratic terms. 134 names map to 174 colors, using only 73 out of 267 centroids.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#F>

=head1 COLORS

	aircraft cream                                    aircraftcream                            #c2b280

	aircraft gray                                     aircraftgray                             #b9b8b5

	anti-corrosive brown                              anticorrosivebrown                       #422518

	anti-corrosive green                              anticorrosivegreen                       #403d21

	azure blue                                        azureblue                                #8791bf

	battleship gray                                   battleshipgray                           #848482

	black                                             black                                    #222222

	black boottopping                                 blackboottopping                         #222222

	black deck                                        blackdeck                                #222222

	blue                                              blue                                     #436b95

	blue drab                                         bluedrab                                 #4e5755

	blue gray                                         bluegray                                 #848482

	blue green                                        bluegreen                                #93c592

	blue striping                                     bluestriping                             #0067a5

	blue tint                                         bluetint                                 #c7e6d7

	bright blue                                       brightblue                               #0067a5

	bright green                                      brightgreen                              #3b7861

	brilliant yellow                                  brilliantyellow                          #f3c300

	brown striping                                    brownstriping                            #56070c

	brown striping                                    brownstriping                            #593319

	buff                                              buff                                     #e3a857

	buff                                              buff                                     #c19a6b

	buff                                              buff                                     #c2b280

	buff                                              buff                                     #d99058

	buff                                              buff                                     #be8a3d

	cameo rose                                        cameorose                                #fad6a5

	cameo rose                                        cameorose                                #f4c2c2

	canary yellow                                     canaryyellow                             #f8de7e

	canyon brown                                      canyonbrown                              #635147

	chrome green                                      chromegreen                              #173620

	chrome yellow                                     chromeyellow                             #f6a600

	chrome yellow                                     chromeyellow                             #eaa221

	clear blue                                        clearblue                                #367588

	cream                                             cream                                    #c2b280

	cream striping                                    creamstriping                            #f8de7e

	dark earth                                        darkearth                                #7e6d5a

	dark earth                                        darkearth                                #6c541e

	dark gray                                         darkgray                                 #555555

	dark green                                        darkgreen                                #355e3b

	dark green                                        darkgreen                                #3a4b47

	dark green deck                                   darkgreendeck                            #3a4b47

	dark gull gray                                    darkgullgray                             #848482

	dark putty                                        darkputty                                #57554c

	dark red boottopping                              darkredboottopping                       #3e322c

	dark zinc yellow primer                           darkzincyellowprimer                     #515744

	deep navy gray                                    deepnavygray                             #555555

	deep reddish brown                                deepreddishbrown                         #3e1d1e

	desert sand                                       desertsand                               #958070

	desert sand                                       desertsand                               #a67b5b

	dull red                                          dullred                                  #79443b

	earth brown                                       earthbrown                               #635147

	earth red                                         earthred                                 #6f4e37

	earth yellow                                      earthyellow                              #c19a6b

	eggshell                                          eggshell                                 #f3e5ab

	eggshell                                          eggshell                                 #f0ead6

	engine gray                                       enginegray                               #555555

	engine gray                                       enginegray                               #848482

	exterior green #14 or exterior green no. fourteen exteriorgreen14orexteriorgreennofourteen #2b3d26

	exterior green #15 or exterior green no. fifteen  exteriorgreen15orexteriorgreennofifteen  #8a9a5b

	field drab                                        fielddrab                                #7e6d5a

	field drab                                        fielddrab                                #6c541e

	fire red                                          firered                                  #be0032

	flat cream                                        flatcream                                #f3e5ab

	forest green                                      forestgreen                              #2b3d26

	forest green                                      forestgreen                              #355e3b

	gloss black                                       glossblack                               #222222

	government wall green                             governmentwallgreen                      #8f9779

	gray                                              gray                                     #848482

	gray                                              gray                                     #c2b280

	gray                                              gray                                     #bfb8a5

	gray #12                                          gray12                                   #bfb8a5

	gray #7                                           gray7                                    #8c8767

	gray #7                                           gray7                                    #8a8776

	gray deck                                         graydeck                                 #555555

	gray deck                                         graydeck                                 #848482

	gray tint                                         graytint                                 #c2b280

	gray wood                                         graywood                                 #848482

	green                                             green                                    #355e3b

	green                                             green                                    #355e3b

	green                                             green                                    #1c352d

	green                                             green                                    #2b3d26

	green                                             green                                    #b9b57d

	green                                             green                                    #403d21

	green                                             green                                    #3b7861

	green boottopping                                 greenboottopping                         #355e3b

	green striping                                    greenstriping                            #355e3b

	green tint                                        greentint                                #dadfb7

	green zinc yellow primer                          greenzincyellowprimer                    #8a9a5b

	haze-gray                                         hazegray                                 #848482

	highlight buff                                    highlightbuff                            #c2b280

	hospital ship green                               hospitalshipgreen                        #679267

	hull black                                        hullblack                                #222222

	hull gray                                         hullgray                                 #91a3b0

	insignia blue                                     insigniablue                             #252440

	insignia blue                                     insigniablue                             #36454f

	insignia red                                      insigniared                              #be0032

	insignia red                                      insigniared                              #ab4e52

	insignia white                                    insigniawhite                            #f2f3f4

	instrument black                                  instrumentblack                          #222222

	interior green                                    interiorgreen                            #4a5d23

	intermediate blue                                 intermediateblue                         #536878

	international orange                              internationalorange                      #e25822

	ivory                                             ivory                                    #f8de7e

	ivory cream                                       ivorycream                               #f3e5ab

	jet                                               jet                                      #222222

	jet black                                         jetblack                                 #222222

	lemon yellow                                      lemonyellow                              #f3c300

	light blue                                        lightblue                                #436b95

	light blue                                        lightblue                                #317873

	light blue                                        lightblue                                #367588

	light blue                                        lightblue                                #8ed1b2

	light blue                                        lightblue                                #317873

	light blue                                        lightblue                                #367588

	light blue enamel                                 lightblueenamel                          #00304e

	light blue,, light blue 5                         lightbluelightblue5                      #c7e6d7

	light gray                                        lightgray                                #b9b8b5

	light gray                                        lightgray                                #f3e5ab

	light gray                                        lightgray                                #c2b280

	light gray                                        lightgray                                #848482

	light gray                                        lightgray                                #b9b8b5

	light gray boottopping                            lightgrayboottopping                     #848482

	light green                                       lightgreen                               #44944a

	light green                                       lightgreen                               #c9dc89

	light green                                       lightgreen                               #679267

	light green                                       lightgreen                               #b9b57d

	light green                                       lightgreen                               #8a9a5b

	light green                                       lightgreen                               #b2beb5

	light gull gray                                   lightgullgray                            #b9b8b5

	light ivory                                       lightivory                               #f3e5ab

	light navy gray                                   lightnavygray                            #848482

	light olive                                       lightolive                               #8c8767

	light red boottopping                             lightredboottopping                      #56070c

	light yellow                                      lightyellow                              #f3c300

	light yellow                                      lightyellow                              #f8de7e

	loam                                              loam                                     #57554c

	machinery green                                   machinerygreen                           #5e716a

	marine green                                      marinegreen                              #3b3121

	maroon                                            maroon                                   #56070c

	maroon                                            maroon                                   #674c47

	medium blue                                       mediumblue                               #536878

	medium blue                                       mediumblue                               #436b95

	medium gray                                       mediumgray                               #8c8767

	medium gray                                       mediumgray                               #848482

	medium gray                                       mediumgray                               #8a8776

	medium green                                      mediumgreen                              #3a4b47

	medium green                                      mediumgreen                              #00622d

	medium green                                      mediumgreen                              #5e716a

	medium navy gray                                  mediumnavygray                           #848482

	medium tan                                        mediumtan                                #826644

	metallic brown                                    metallicbrown                            #56070c

	metallic brown                                    metallicbrown                            #56070c

	metallic brown                                    metallicbrown                            #593319

	metallic red                                      metallicred                              #56070c

	middle stone                                      middlestone                              #967117

	middlestone                                       middlestone                              #967117

	mosstone                                          mosstone                                 #8a9a5b

	national school bus chrome                        nationalschoolbuschrome                  #f6a600

	navy gray                                         navygray                                 #555555

	ocean gray                                        oceangray                                #536878

	ocean gray                                        oceangray                                #848482

	olive drab                                        olivedrab                                #3b3121

	olive drab                                        olivedrab                                #5b5842

	olive drab                                        olivedrab                                #403d21

	olive green                                       olivegreen                               #403d21

	orange                                            orange                                   #d9603b

	orange                                            orange                                   #e25822

	orange                                            orange                                   #ed872d

	orange yellow                                     orangeyellow                             #f6a600

	orange yellow                                     orangeyellow                             #eaa221

	oyster gray                                       oystergray                               #b9b8b5

	peach                                             peach                                    #fad6a5

	pea green                                         peagreen                                 #679267

	pearl gray                                        pearlgray                                #c2b280

	pearl gray                                        pearlgray                                #b9b8b5

	pullman green                                     pullmangreen                             #2b3d26

	pullman green                                     pullmangreen                             #403d21

	pullman green                                     pullmangreen                             #5b5842

	purple                                            purple                                   #86608e

	red                                               red                                      #be0032

	red                                               red                                      #be0032

	red                                               red                                      #ab4e52

	red boottopping                                   redboottopping                           #56070c

	red deck                                          reddeck                                  #56070c

	red lead                                          redlead                                  #d9603b

	red lead and oil                                  redleadandoil                            #e25822

	red lead modified                                 redleadmodified                          #aa381e

	red oxide                                         redoxide                                 #56070c

	red striping                                      redstriping                              #be0032

	road and tire gray                                roadandtiregray                          #848482

	royal blue                                        royalblue                                #00304e

	sand                                              sand                                     #958070

	sand                                              sand                                     #c2b280

	sand                                              sand                                     #a18f60

	sea blue                                          seablue                                  #202830

	sea blue                                          seablue                                  #202830

	sea blue                                          seablue                                  #51585e

	sea blue                                          seablue                                  #51585e

	sea gray                                          seagray                                  #555555

	seal brown                                        sealbrown                                #6f4e37

	slate                                             slate                                    #555555

	soft yellow                                       softyellow                               #f8de7e

	spring green                                      springgreen                              #8f9779

	spruce green                                      sprucegreen                              #515744

	suntan                                            suntan                                   #f3e5ab

	terra cotta                                       terracotta                               #a87c6d

	vermilion                                         vermilion                                #be0032

	vineyard green                                    vineyardgreen                            #1a2421

	vivid orange                                      vividorange                              #e25822

	warm gray                                         warmgray                                 #c2b280

	willow green                                      willowgreen                              #44944a

	yellow                                            yellow                                   #f6a600

	yellow                                            yellow                                   #eaa221

	yellow                                            yellow                                   #f6a600

	yellow striping                                   yellowstriping                           #f6a600

	yellow striping                                   yellowstriping                           #eaa221

	yellow tint                                       yellowtint                               #fad6a5

	zinc chromate                                     zincchromate                             #8a9a5b

	zinc yellow primer                                zincyellowprimer                         #c9ae5d


=cut

sub _load_color_list() {
    return [
['nbs-iscc-f:aircraftcream.90','aircraftcream','aircraft cream',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:aircraftgray.264','aircraftgray','aircraft gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:anticorrosivebrown.59','anticorrosivebrown','anti-corrosive brown',[66,37,24],'422518',4334872],
['nbs-iscc-f:anticorrosivegreen.108','anticorrosivegreen','anti-corrosive green',[64,61,33],'403d21',4209953],
['nbs-iscc-f:azureblue.199','azureblue','azure blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-f:battleshipgray.265','battleshipgray','battleship gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-f:blackboottopping.267','blackboottopping','black boottopping',[34,34,34],'222222',2236962],
['nbs-iscc-f:blackdeck.267','blackdeck','black deck',[34,34,34],'222222',2236962],
['nbs-iscc-f:blue.182','blue','blue',[67,107,149],'436b95',4418453],
['nbs-iscc-f:bluedrab.156','bluedrab','blue drab',[78,87,85],'4e5755',5134165],
['nbs-iscc-f:bluegray.265','bluegray','blue gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:bluegreen.135','bluegreen','blue green',[147,197,146],'93c592',9684370],
['nbs-iscc-f:bluestriping.178','bluestriping','blue striping',[0,103,165],'0067a5',26533],
['nbs-iscc-f:bluetint.148','bluetint','blue tint',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-f:brightblue.178','brightblue','bright blue',[0,103,165],'0067a5',26533],
['nbs-iscc-f:brightgreen.145','brightgreen','bright green',[59,120,97],'3b7861',3897441],
['nbs-iscc-f:brilliantyellow.82','brilliantyellow','brilliant yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-f:brownstriping.41','brownstriping','brown striping',[86,7,12],'56070c',5637900],
['nbs-iscc-f:brownstriping.56','brownstriping','brown striping',[89,51,25],'593319',5845785],
['nbs-iscc-f:buff.71','buff','buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-f:buff.76','buff','buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-f:buff.90','buff','buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:buff.53','buff','buff',[217,144,88],'d99058',14258264],
['nbs-iscc-f:buff.72','buff','buff',[190,138,61],'be8a3d',12487229],
['nbs-iscc-f:cameorose.73','cameorose','cameo rose',[250,214,165],'fad6a5',16438949],
['nbs-iscc-f:cameorose.28','cameorose','cameo rose',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-f:canaryyellow.86','canaryyellow','canary yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-f:canyonbrown.61','canyonbrown','canyon brown',[99,81,71],'635147',6508871],
['nbs-iscc-f:chromegreen.138','chromegreen','chrome green',[23,54,32],'173620',1521184],
['nbs-iscc-f:chromeyellow.66','chromeyellow','chrome yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:chromeyellow.68','chromeyellow','chrome yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-f:clearblue.173','clearblue','clear blue',[54,117,136],'367588',3569032],
['nbs-iscc-f:cream.90','cream','cream',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:creamstriping.86','creamstriping','cream striping',[248,222,126],'f8de7e',16309886],
['nbs-iscc-f:darkearth.80','darkearth','dark earth',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-f:darkearth.95','darkearth','dark earth',[108,84,30],'6c541e',7099422],
['nbs-iscc-f:darkgray.266','darkgray','dark gray',[85,85,85],'555555',5592405],
['nbs-iscc-f:darkgreen.137','darkgreen','dark green',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:darkgreen.151','darkgreen','dark green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-f:darkgreendeck.151','darkgreendeck','dark green deck',[58,75,71],'3a4b47',3820359],
['nbs-iscc-f:darkgullgray.265','darkgullgray','dark gull gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:darkputty.113','darkputty','dark putty',[87,85,76],'57554c',5723468],
['nbs-iscc-f:darkredboottopping.62','darkredboottopping','dark red boottopping ',[62,50,44],'3e322c',4076076],
['nbs-iscc-f:darkzincyellowprimer.127','darkzincyellowprimer','dark zinc yellow primer',[81,87,68],'515744',5330756],
['nbs-iscc-f:deepnavygray.266','deepnavygray','deep navy gray',[85,85,85],'555555',5592405],
['nbs-iscc-f:deepreddishbrown.44','deepreddishbrown','deep reddish brown',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-f:desertsand.60','desertsand','desert sand',[149,128,112],'958070',9797744],
['nbs-iscc-f:desertsand.57','desertsand','desert sand',[166,123,91],'a67b5b',10910555],
['nbs-iscc-f:dullred.43','dullred','dull red',[121,68,59],'79443b',7947323],
['nbs-iscc-f:earthbrown.61','earthbrown','earth brown',[99,81,71],'635147',6508871],
['nbs-iscc-f:earthred.58','earthred','earth red ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-f:earthyellow.76','earthyellow','earth yellow',[193,154,107],'c19a6b',12687979],
['nbs-iscc-f:eggshell.89','eggshell','eggshell ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:eggshell.92','eggshell','eggshell ',[240,234,214],'f0ead6',15788758],
['nbs-iscc-f:enginegray.266','enginegray','engine gray',[85,85,85],'555555',5592405],
['nbs-iscc-f:enginegray.265','enginegray','engine gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:exteriorgreen14orexteriorgreennofourteen.126','exteriorgreen14orexteriorgreennofourteen','exterior green #14 or exterior green no. fourteen',[43,61,38],'2b3d26',2833702],
['nbs-iscc-f:exteriorgreen15orexteriorgreennofifteen.120','exteriorgreen15orexteriorgreennofifteen','exterior green #15 or exterior green no. fifteen',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-f:fielddrab.80','fielddrab','field drab',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-f:fielddrab.95','fielddrab','field drab',[108,84,30],'6c541e',7099422],
['nbs-iscc-f:firered.11','firered','fire red',[190,0,50],'be0032',12451890],
['nbs-iscc-f:flatcream.89','flatcream','flat cream',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:forestgreen.126','forestgreen','forest green',[43,61,38],'2b3d26',2833702],
['nbs-iscc-f:forestgreen.137','forestgreen','forest green',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:glossblack.267','glossblack','gloss black',[34,34,34],'222222',2236962],
['nbs-iscc-f:governmentwallgreen.122','governmentwallgreen','government wall green',[143,151,121],'8f9779',9410425],
['nbs-iscc-f:gray.265','gray','gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:gray.90','gray','gray',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:gray.93','gray','gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-f:gray12.93','gray12','gray #12 ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-f:gray7.109','gray7','gray #7 ',[140,135,103],'8c8767',9209703],
['nbs-iscc-f:gray7.112','gray7','gray #7 ',[138,135,118],'8a8776',9078646],
['nbs-iscc-f:graydeck.266','graydeck','gray deck',[85,85,85],'555555',5592405],
['nbs-iscc-f:graydeck.265','graydeck','gray deck',[132,132,130],'848482',8684674],
['nbs-iscc-f:graytint.90','graytint','gray tint',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:graywood.265','graywood','gray wood',[132,132,130],'848482',8684674],
['nbs-iscc-f:green.137','green','green',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:green.137','green','green',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:green.147','green','green',[28,53,45],'1c352d',1848621],
['nbs-iscc-f:green.126','green','green',[43,61,38],'2b3d26',2833702],
['nbs-iscc-f:green.105','green','green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-f:green.108','green','green',[64,61,33],'403d21',4209953],
['nbs-iscc-f:green.145','green','green',[59,120,97],'3b7861',3897441],
['nbs-iscc-f:greenboottopping.137','greenboottopping','green boottopping',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:greenstriping.137','greenstriping','green striping',[53,94,59],'355e3b',3497531],
['nbs-iscc-f:greentint.121','greentint','green tint',[218,223,183],'dadfb7',14344119],
['nbs-iscc-f:greenzincyellowprimer.120','greenzincyellowprimer','green zinc yellow primer',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-f:hazegray.265','hazegray','haze-gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:highlightbuff.90','highlightbuff','highlight buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:hospitalshipgreen.136','hospitalshipgreen','hospital ship green',[103,146,103],'679267',6787687],
['nbs-iscc-f:hullblack.267','hullblack','hull black',[34,34,34],'222222',2236962],
['nbs-iscc-f:hullgray.185','hullgray','hull gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-f:insigniablue.201','insigniablue','insignia blue',[37,36,64],'252440',2434112],
['nbs-iscc-f:insigniablue.187','insigniablue','insignia blue',[54,69,79],'36454f',3556687],
['nbs-iscc-f:insigniared.11','insigniared','insignia red',[190,0,50],'be0032',12451890],
['nbs-iscc-f:insigniared.15','insigniared','insignia red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-f:insigniawhite.263','insigniawhite','insignia white',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-f:instrumentblack.267','instrumentblack','instrument black',[34,34,34],'222222',2236962],
['nbs-iscc-f:interiorgreen.125','interiorgreen','interior green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-f:intermediateblue.186','intermediateblue','intermediate blue',[83,104,120],'536878',5466232],
['nbs-iscc-f:internationalorange.34','internationalorange','international orange',[226,88,34],'e25822',14833698],
['nbs-iscc-f:ivory.86','ivory','ivory',[248,222,126],'f8de7e',16309886],
['nbs-iscc-f:ivorycream.89','ivorycream','ivory cream',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:jet.267','jet','jet',[34,34,34],'222222',2236962],
['nbs-iscc-f:jetblack.267','jetblack','jet black',[34,34,34],'222222',2236962],
['nbs-iscc-f:lemonyellow.82','lemonyellow','lemon yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-f:lightblue.182','lightblue','light blue',[67,107,149],'436b95',4418453],
['nbs-iscc-f:lightblue.164','lightblue','light blue',[49,120,115],'317873',3242099],
['nbs-iscc-f:lightblue.173','lightblue','light blue',[54,117,136],'367588',3569032],
['nbs-iscc-f:lightblue.143','lightblue','light blue',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-f:lightblue.164','lightblue','light blue',[49,120,115],'317873',3242099],
['nbs-iscc-f:lightblue.173','lightblue','light blue',[54,117,136],'367588',3569032],
['nbs-iscc-f:lightblueenamel.183','lightblueenamel','light blue enamel',[0,48,78],'00304e',12366],
['nbs-iscc-f:lightbluelightblue5.148','lightbluelightblue5','light blue,, light blue 5',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-f:lightgray.264','lightgray','light gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:lightgray.89','lightgray','light gray',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:lightgray.90','lightgray','light gray',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:lightgray.265','lightgray','light gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:lightgray.264','lightgray','light gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:lightgrayboottopping.265','lightgrayboottopping','light gray boottopping',[132,132,130],'848482',8684674],
['nbs-iscc-f:lightgreen.131','lightgreen','light green',[68,148,74],'44944a',4494410],
['nbs-iscc-f:lightgreen.119','lightgreen','light green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-f:lightgreen.136','lightgreen','light green',[103,146,103],'679267',6787687],
['nbs-iscc-f:lightgreen.105','lightgreen','light green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-f:lightgreen.120','lightgreen','light green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-f:lightgreen.154','lightgreen','light green',[178,190,181],'b2beb5',11714229],
['nbs-iscc-f:lightgullgray.264','lightgullgray','light gull gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:lightivory.89','lightivory','light ivory',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:lightnavygray.265','lightnavygray','light navy gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:lightolive.109','lightolive','light olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-f:lightredboottopping.41','lightredboottopping','light red boottopping ',[86,7,12],'56070c',5637900],
['nbs-iscc-f:lightyellow.82','lightyellow','light yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-f:lightyellow.86','lightyellow','light yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-f:loam.113','loam','loam',[87,85,76],'57554c',5723468],
['nbs-iscc-f:machinerygreen.150','machinerygreen','machinery green',[94,113,106],'5e716a',6189418],
['nbs-iscc-f:marinegreen.96','marinegreen','marine green',[59,49,33],'3b3121',3879201],
['nbs-iscc-f:maroon.41','maroon','maroon',[86,7,12],'56070c',5637900],
['nbs-iscc-f:maroon.46','maroon','maroon',[103,76,71],'674c47',6769735],
['nbs-iscc-f:mediumblue.186','mediumblue','medium blue',[83,104,120],'536878',5466232],
['nbs-iscc-f:mediumblue.182','mediumblue','medium blue',[67,107,149],'436b95',4418453],
['nbs-iscc-f:mediumgray.109','mediumgray','medium gray',[140,135,103],'8c8767',9209703],
['nbs-iscc-f:mediumgray.265','mediumgray','medium gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:mediumgray.112','mediumgray','medium gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-f:mediumgreen.151','mediumgreen','medium green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-f:mediumgreen.132','mediumgreen','medium green',[0,98,45],'00622d',25133],
['nbs-iscc-f:mediumgreen.150','mediumgreen','medium green',[94,113,106],'5e716a',6189418],
['nbs-iscc-f:mediumnavygray.265','mediumnavygray','medium navy gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:mediumtan.77','mediumtan','medium tan',[130,102,68],'826644',8545860],
['nbs-iscc-f:metallicbrown.41','metallicbrown','metallic brown ',[86,7,12],'56070c',5637900],
['nbs-iscc-f:metallicbrown.41','metallicbrown','metallic brown ',[86,7,12],'56070c',5637900],
['nbs-iscc-f:metallicbrown.56','metallicbrown','metallic brown ',[89,51,25],'593319',5845785],
['nbs-iscc-f:metallicred.41','metallicred','metallic red ',[86,7,12],'56070c',5637900],
['nbs-iscc-f:middlestone.94','middlestone','middle stone ',[150,113,23],'967117',9859351],
['nbs-iscc-f:middlestone.94','middlestone','middlestone ',[150,113,23],'967117',9859351],
['nbs-iscc-f:mosstone.120','mosstone','mosstone ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-f:nationalschoolbuschrome.66','nationalschoolbuschrome','national school bus chrome ',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:navygray.266','navygray','navy gray',[85,85,85],'555555',5592405],
['nbs-iscc-f:oceangray.186','oceangray','ocean gray',[83,104,120],'536878',5466232],
['nbs-iscc-f:oceangray.265','oceangray','ocean gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:olivedrab.96','olivedrab','olive drab',[59,49,33],'3b3121',3879201],
['nbs-iscc-f:olivedrab.110','olivedrab','olive drab',[91,88,66],'5b5842',5986370],
['nbs-iscc-f:olivedrab.108','olivedrab','olive drab',[64,61,33],'403d21',4209953],
['nbs-iscc-f:olivegreen.108','olivegreen','olive green',[64,61,33],'403d21',4209953],
['nbs-iscc-f:orange.35','orange','orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-f:orange.34','orange','orange',[226,88,34],'e25822',14833698],
['nbs-iscc-f:orange.50','orange','orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-f:orangeyellow.66','orangeyellow','orange yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:orangeyellow.68','orangeyellow','orange yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-f:oystergray.264','oystergray','oyster gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:peach.73','peach','peach',[250,214,165],'fad6a5',16438949],
['nbs-iscc-f:peagreen.136','peagreen','pea green',[103,146,103],'679267',6787687],
['nbs-iscc-f:pearlgray.90','pearlgray','pearl gray',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:pearlgray.264','pearlgray','pearl gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-f:pullmangreen.126','pullmangreen','pullman green',[43,61,38],'2b3d26',2833702],
['nbs-iscc-f:pullmangreen.108','pullmangreen','pullman green',[64,61,33],'403d21',4209953],
['nbs-iscc-f:pullmangreen.110','pullmangreen','pullman green',[91,88,66],'5b5842',5986370],
['nbs-iscc-f:purple.223','purple','purple',[134,96,142],'86608e',8806542],
['nbs-iscc-f:red.11','red','red',[190,0,50],'be0032',12451890],
['nbs-iscc-f:red.11','red','red',[190,0,50],'be0032',12451890],
['nbs-iscc-f:red.15','red','red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-f:redboottopping.41','redboottopping','red boottopping',[86,7,12],'56070c',5637900],
['nbs-iscc-f:reddeck.41','reddeck','red deck',[86,7,12],'56070c',5637900],
['nbs-iscc-f:redlead.35','redlead','red lead',[217,96,59],'d9603b',14245947],
['nbs-iscc-f:redleadandoil.34','redleadandoil','red lead and oil',[226,88,34],'e25822',14833698],
['nbs-iscc-f:redleadmodified.36','redleadmodified','red lead modified',[170,56,30],'aa381e',11155486],
['nbs-iscc-f:redoxide.41','redoxide','red oxide',[86,7,12],'56070c',5637900],
['nbs-iscc-f:redstriping.11','redstriping','red striping',[190,0,50],'be0032',12451890],
['nbs-iscc-f:roadandtiregray.265','roadandtiregray','road and tire gray',[132,132,130],'848482',8684674],
['nbs-iscc-f:royalblue.183','royalblue','royal blue',[0,48,78],'00304e',12366],
['nbs-iscc-f:sand.60','sand','sand',[149,128,112],'958070',9797744],
['nbs-iscc-f:sand.90','sand','sand',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:sand.91','sand','sand',[161,143,96],'a18f60',10588000],
['nbs-iscc-f:seablue.188','seablue','sea blue',[32,40,48],'202830',2107440],
['nbs-iscc-f:seablue.188','seablue','sea blue',[32,40,48],'202830',2107440],
['nbs-iscc-f:seablue.192','seablue','sea blue',[81,88,94],'51585e',5331038],
['nbs-iscc-f:seablue.192','seablue','sea blue',[81,88,94],'51585e',5331038],
['nbs-iscc-f:seagray.266','seagray','sea gray',[85,85,85],'555555',5592405],
['nbs-iscc-f:sealbrown.58','sealbrown','seal brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-f:slate.266','slate','slate',[85,85,85],'555555',5592405],
['nbs-iscc-f:softyellow.86','softyellow','soft yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-f:springgreen.122','springgreen','spring green',[143,151,121],'8f9779',9410425],
['nbs-iscc-f:sprucegreen.127','sprucegreen','spruce green',[81,87,68],'515744',5330756],
['nbs-iscc-f:suntan.89','suntan','suntan',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-f:terracotta.42','terracotta','terra cotta',[168,124,109],'a87c6d',11041901],
['nbs-iscc-f:vermilion.11','vermilion','vermilion',[190,0,50],'be0032',12451890],
['nbs-iscc-f:vineyardgreen.152','vineyardgreen','vineyard green ',[26,36,33],'1a2421',1713185],
['nbs-iscc-f:vividorange.34','vividorange','vivid orange',[226,88,34],'e25822',14833698],
['nbs-iscc-f:warmgray.90','warmgray','warm gray ',[194,178,128],'c2b280',12759680],
['nbs-iscc-f:willowgreen.131','willowgreen','willow green',[68,148,74],'44944a',4494410],
['nbs-iscc-f:yellow.66','yellow','yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:yellow.68','yellow','yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-f:yellow.66','yellow','yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:yellowstriping.66','yellowstriping','yellow striping',[246,166,0],'f6a600',16164352],
['nbs-iscc-f:yellowstriping.68','yellowstriping','yellow striping',[234,162,33],'eaa221',15376929],
['nbs-iscc-f:yellowtint.73','yellowtint','yellow tint',[250,214,165],'fad6a5',16438949],
['nbs-iscc-f:zincchromate.120','zincchromate','zinc chromate',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-f:zincyellowprimer.87','zincyellowprimer','zinc yellow primer ',[201,174,93],'c9ae5d',13217373]
    ];
}

sub _description {
    return {
          'subtitle' => 'Colors; (for) Ready-Mixed Paints',
          'title' => 'NBS/ISCC F',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (F) Colors; (for) Ready-Mixed Paints

    Federal Specification TT-C-595,
    Colors; (for) Ready-Mixed Paints 

This dictionary is a hybrid of combinatorial and idiosyncratic terms. 134 names map to 174 colors, using only 73 out of 267 centroids.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#F]
'
        }

}

1;
