#!/usr/bin/perl

use v5.12;
use warnings;
use Test::More tests => 72;
BEGIN { unshift @INC, 'lib', '../lib'}
use Graphics::Toolkit::Color::Space::Util ':all';

my $module = 'Graphics::Toolkit::Color';
eval "use $module qw/color/";
is( not( $@), 1, 'could load the module');
is( ref Graphics::Toolkit::Color->new(),        '', 'constructor need arguments');
is( ref Graphics::Toolkit::Color->new('red'), $module, 'constructor accepts color name');
is( ref Graphics::Toolkit::Color->new( 'red', 'green'), '', 'constructor needs only one color name');
#~ is( ref Graphics::Toolkit::Color->new('SVG::red'), $module, 'constructor accepts color name from a scheme');
#~ is( ref Graphics::Toolkit::Color->new('SVG::red'), $module, 'constructor accepts color name from a scheme');
is( ref Graphics::Toolkit::Color->new('#000'),     $module, 'short hex string with min value');
is( ref Graphics::Toolkit::Color->new('#FFFFFF'),  $module, 'long hex string with max value');
is( ref Graphics::Toolkit::Color->new('#1ab2cc'),  $module, 'long hex string mixed lc vlaues');
is( ref Graphics::Toolkit::Color->new('#abj'),          '', 'short hex string has typo');
is( ref Graphics::Toolkit::Color->new('#AABBGG'),       '', 'long hex string has typo');
is( ref Graphics::Toolkit::Color->new('#AA'),           '', 'short hex string is too short');
is( ref Graphics::Toolkit::Color->new('#AABF'),         '', 'short hex string is too long');
is( ref Graphics::Toolkit::Color->new('#AABBF'),        '', 'long hex string is too short');
is( ref Graphics::Toolkit::Color->new('#AABBFFF'),      '', 'long hex string is too long');
is( ref Graphics::Toolkit::Color->new('rgb(0, 0, 0)'),          $module, 'CSS string format');
is( ref Graphics::Toolkit::Color->new('lab( 12.3, 5.4, 1.2)'),  $module, 'CSS string in LAB space');
is( ref Graphics::Toolkit::Color->new('lab( 12.3, 5.4, 1.2%)'),      '', 'CSS string with bad suffix');
is( ref Graphics::Toolkit::Color->new('YIQ:5.22,   0, -10  '), $module, 'named string in YIQ space and additional spacing');
is( ref Graphics::Toolkit::Color->new('NCol: B10,  100, 0'),   $module, 'named string in Ncol space with min and max values');
is( ref Graphics::Toolkit::Color->new( 4),      '', 'constructor needs more than one number');
is( ref Graphics::Toolkit::Color->new( 4,5),    '', 'constructor needs more than two numbers');
is( ref Graphics::Toolkit::Color->new( 4,5,6,7), '', 'constructor needs less than four numbers');
is( ref Graphics::Toolkit::Color->new( 1,2,3), $module, 'constructor got three RGB numbers');
is( ref Graphics::Toolkit::Color->new( 1,2,'e4'),   '', 'third RGB value has to be number');
is( ref Graphics::Toolkit::Color->new( 1, '2a', 4), '', 'second RGB value has to be number');
is( ref Graphics::Toolkit::Color->new( '%', 2, 4), '', 'first RGB value has to be number');
is( ref Graphics::Toolkit::Color->new( [4,5]),    '', 'constructor needs more than two numbers in an ARRAY');
is( ref Graphics::Toolkit::Color->new( [4,5,6,7]), '', 'constructor needs less than four numbers in an ARRAY');
is( ref Graphics::Toolkit::Color->new( [1,2,3]), $module, 'constructor got three RGB numbers in an ARRAY');
is( ref Graphics::Toolkit::Color->new( ['YUV',1,2,3]), $module, 'named ARRAY in YUV space');
is( ref Graphics::Toolkit::Color->new( ['YUV',1,2]),        '', 'named ARRAY in YUV space got not enough values');
is( ref Graphics::Toolkit::Color->new( ['YUV',1,2,3,4]),    '', 'named ARRAY in YUV space got too many values');
is( ref Graphics::Toolkit::Color->new( ['cmyk',1,0,0,0]), $module, 'named ARRAY in CMYK space');
is( ref Graphics::Toolkit::Color->new( ['cmyk',1,0,0]),        '', 'CMYK ARRAY got not enough values');
is( ref Graphics::Toolkit::Color->new( ['cmyk',1,0,0,0,0]),    '', 'CMYK ARRAY got too much values');
is( ref Graphics::Toolkit::Color->new( ['cmk', 0,0,0]),        '', 'only known color space names are accepted ');
is( ref Graphics::Toolkit::Color->new( ['CIELCHab', 0, 0, 0]),          $module, 'long mixed case alias names work too');
is( ref Graphics::Toolkit::Color->new( ['hsb', 100.23, 0.173, .214]),   $module, 'different number shapes');
is( ref Graphics::Toolkit::Color->new( ['NCol','B10','100%','0%']),     $module, 'named ARRAY with values that need preprocessing');
is( ref Graphics::Toolkit::Color->new( ['ncol','B0','100','0']),        $module, 'try single digit string value');
is( ref Graphics::Toolkit::Color->new( OKLAB => [0,0,0] ),              $module, 'named ARRAY ref in uc oklab space');
is( ref Graphics::Toolkit::Color->new( 'hunterlab', [1,2,3] ),          $module, 'named ARRAY ref in lc hunterlab space');
is( ref Graphics::Toolkit::Color->new( { }),                                 '', 'HASH needs keys');
is( ref Graphics::Toolkit::Color->new( {r=> 1 }),                            '', 'HASH one key is not enough');
is( ref Graphics::Toolkit::Color->new( r=> 1 ),                              '', 'even without a HASH ref');
is( ref Graphics::Toolkit::Color->new( {r=> 1, g=>2 }),                      '', 'HASH two keys are not enough');
is( ref Graphics::Toolkit::Color->new( r=> 1, g=>2 ),                        '', 'also without a HASH ref');
is( ref Graphics::Toolkit::Color->new( { }),                                 '', 'HASH needs keys');
is( ref Graphics::Toolkit::Color->new( {r => 0, g => 0, b => 0 }),         $module, 'RGB short HASH');
is( ref Graphics::Toolkit::Color->new( r => 0, g => 0, b => 0 ),           $module, 'RGB short HASH without ref');
is( ref Graphics::Toolkit::Color->new( red => 0, G => 0, b => 0 ),         $module, 'can mix long and short, lc and uc axis names');
is( ref Graphics::Toolkit::Color->new( {r => 0, g => 0, b => 0, y=> 1 }),       '', 'too many keys');
is( ref Graphics::Toolkit::Color->new( r => 0, g => 0, b => 0, y=> 1),          '', 'also without ref');
is( ref Graphics::Toolkit::Color->new( c => 0, m => 0, k => 0, y=> 1),     $module, 'CMYK hash');
is( ref Graphics::Toolkit::Color->new( c => 0, m => 0, kkey => 0, y=> 1),       '', 'one key has typo');
is( ref Graphics::Toolkit::Color->new( c => 0, m => 0, k=> 1),                  '', 'one key is missing');
is( ref Graphics::Toolkit::Color->new( luma => 0, Pb => 0, Pr=> .5),       $module, 'YPbPr hash');
is( ref Graphics::Toolkit::Color->new( 'L*' => 0, 'a*' => '100','b*'=> .5),$module, 'LAB hash');
is( ref Graphics::Toolkit::Color->new( 'L*' => 0, 'a*' => '10e0', 'b*' => .5),  '', 'dont accept scientific notation');
is( ref Graphics::Toolkit::Color->new( l => 0,  chroma => '0',  hue => 0), $module, 'CIELCHuv hash');
is( ref Graphics::Toolkit::Color->new( l => "0%",  c => '0',  hue => 0),        '', 'CIELCHuv has no value suffix');
is( ref Graphics::Toolkit::Color->new( h => "0",  w => '0',  b => 0),      $module, 'HWB hash');
is( ref Graphics::Toolkit::Color->new( h=> "R100",w => '0%', b=> '100%'),  $module, 'NCol hash');
is( ref Graphics::Toolkit::Color->new( Y => 0, U => 0, V => 100),          $module, 'YPbPr short hash');
is( ref Graphics::Toolkit::Color->new( X => 0, Y => 0, Z => 0),            $module, 'XYZ short hash (has no long names)');

is( ref color( Y => 0, U => 0, V => 100),          $module, 'short named constructor method');
is( ref color( ),                                       '', 'needs also args');
is( ref color( 1,1),                                    '', 'two are not enough');
is( ref color( 1,1,1,1),                                '', 'four are too many');
is( ref color( '#0000ff' ),                        $module, 'can also recieve lc hex sting');
is( ref color( [1,2,3] ),                          $module, 'ARRAY ref');
is( ref color( {l => 50, c => 12.4, h => .6} ),    $module, 'LCH short axis name HASH');
is( ref color( {hue => 0, whiteness => '0%', blackness => '100%'} ), $module, 'HWB long axis name HASH');

exit 0;
