// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief Measurement of $R$ for energies between 2.6 and 3.65 GeV
  class BESII_2009_I814778 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESII_2009_I814778);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      // Book histograms
      book(_mult, 1, 1, 1);
      const YODA::BinnedEstimate<string>& ref = refData<YODA::BinnedEstimate<string>>(1,1,1);
      book(_c_hadrons, "/TMP/sigma_hadrons", ref);
      book(_c_muons,   "/TMP/sigma_muons"  , ref);
      for (const string& en : _c_hadrons.binning().edges<0>()) {
        double eval = stod(en);
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // mu+mu- + photons
      if (nCount[-13]==1 && nCount[13]==1 &&  ntotal==2+nCount[22]) {
        _c_muons->fill(_sqs);
      }
      else { // everything else
        _c_hadrons->fill(_sqs);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      divide(_c_hadrons, _c_muons, _mult);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _c_hadrons, _c_muons;
    BinnedEstimatePtr<string> _mult;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESII_2009_I814778);


}
