
#include "gtk-local.h"

#include "magnify-tool.h"

#include "gtk-toolbar.h"
#include "gtk-graph.h"
#include <gdk/gdkkeysyms.h>

#ifdef GAG_USE_X11
#include <gdk/gdkx.h>
#endif

/*****************************************************************************/

static int _magnify_programmatically = 0;
static GtkWidget *_magnify_window = NULL;
static GtkWidget *_magnifying_drawing_area = NULL;
static G_env *_magnify_genv = NULL;
static gint _magnify_x = 0;
static gint _magnify_y = 0;
static gint _magnify_x_offset = 0;
static gint _magnify_y_offset = 0;
static gint _magnify_width = 400;
static gint _magnify_height = 400;
static float _magnify_window_size_factor = 1.2;
static float _magnify_zoom_factor = 5.0;
static int _magnify_pin_window = 0;
const int _magnify_grab_pointer = 1; /* enables to release the mouse button */
static gboolean _x11_backend = FALSE;

static void _magnify_reset()
{
    _magnify_x = 0;
    _magnify_y = 0;
    _magnify_x_offset = 0;
    _magnify_y_offset = 0;
    _magnify_width = 400;
    _magnify_height = 400;
    _magnify_window_size_factor = 1.2;
    _magnify_zoom_factor = 5.0;
    _magnify_pin_window = 0;
}

static void _magnify_reset_zoom()
{
    _magnify_zoom_factor = 1.0;
    _magnify_x_offset = 0;
    _magnify_y_offset = 0;
}

static void _magnify_zoom()
{
    _magnify_zoom_factor *= 1.5;
    if (_magnify_zoom_factor > 1e10)
        _magnify_zoom_factor = 1e10;
}

static void _magnify_unzoom()
{
    _magnify_zoom_factor /= 1.5;
    if (_magnify_zoom_factor < 1.0)
        _magnify_zoom_factor = 1.0;
}

static void _move_origin( ggtk_env_t *genv, gint _x, gint _y, gint
 _x_root, gint _y_root)
{
    if (_magnify_grab_pointer) {
        gint x;
        gint y;
        GdkWindow *win = gtk_widget_get_window(genv->main_widget);
        gdk_window_get_origin(win, &x, &y);
        _magnify_x = _x_root - x;
        _magnify_y = _y_root - y;
    } else {
        _magnify_x = _x;
        _magnify_y = _y;
    }
}

static void _move_magnify_offset( ggtk_env_t *genv, gint _x, gint _y, gint
 _x_root, gint _y_root)
{
    int ox = _magnify_x;
    int oy = _magnify_y;
    _move_origin( genv, _x, _y, _x_root, _y_root);
    _magnify_x_offset -= _magnify_x - ox;
    _magnify_y_offset -= _magnify_y - oy;
}

static void _magnify_flip_pin_window()
{
    _magnify_pin_window = 1 - _magnify_pin_window;
}

static void _move_magnify_window( ggtk_env_t *genv, gint _x, gint _y, gint
 _x_root, gint _y_root)
{
    _move_origin( genv, _x, _y, _x_root, _y_root);
    if (!_magnify_pin_window) {
        GdkWindow *win = gtk_widget_get_window(_magnify_window);
        gint new_x = _x_root - _magnify_width / 2 + _magnify_x_offset;
        gint new_y = _y_root - _magnify_height / 2 + _magnify_y_offset;
        gdk_window_move(win, new_x, new_y);
    }
}

static gboolean _magnify_pointer_is_grabbed()
{
    if (!_magnify_window) return FALSE;
    GdkDisplay *display = gtk_widget_get_display(_magnify_window);
    GdkSeat   *seat    = gdk_display_get_default_seat(display);
    GdkDevice *pointer = gdk_seat_get_pointer(seat);
    return gdk_display_device_is_grabbed(display, pointer);
}

static void _grab(GdkWindow *window)
{ 
    if (!_magnify_window || !window) return;
    GdkDisplay *display = gtk_widget_get_display(_magnify_window);
    GdkSeat *seat = gdk_display_get_default_seat(display);
    GdkGrabStatus status = gdk_seat_grab(
        seat,window,GDK_SEAT_CAPABILITY_POINTER,
        TRUE,NULL,NULL,NULL,NULL
    );
    if (status != GDK_GRAB_SUCCESS)
        fprintf( stderr, "gdk_pointer_grab error\n");    
    GdkGrabStatus statusKeyboard = gdk_seat_grab(
        seat,window,GDK_SEAT_CAPABILITY_KEYBOARD,
        TRUE,NULL,NULL,NULL,NULL
    );
    if (statusKeyboard != GDK_GRAB_SUCCESS)
        fprintf( stderr, "gdk_keyboard_grab error\n");
}

static void _ungrab()
{
    if (!_magnify_window) return;
    GdkDisplay *display = gtk_widget_get_display(_magnify_window);
    GdkSeat *seat = gdk_display_get_default_seat(display);
    gdk_seat_ungrab(seat);
}

static void _destroy_magnify_window()
{
    if (_magnify_pointer_is_grabbed()) _ungrab();
    if (_magnify_window != NULL) {
        gtk_widget_destroy( _magnify_window);
    }
    if (_magnify_programmatically) {
        sic_post_widget_created();
    }
}

static void _refresh_magnify_data(G_env *ref_env)
{
    ggtk_env_t * genv = (ggtk_env_t *)_magnify_genv;
    polystore_free(&genv->polylines_store);
    imagestore_free(&genv->images_store);
    gint _x_frame_offset = 0;
    gint _y_frame_offset = 0;
    // CAUTION: With GTK3 some additional margins are added around the window
    // The margins change depending on the backend...
    if (_x11_backend) { 
        _x_frame_offset = 0;
        _y_frame_offset = 45;
    } else {
        _x_frame_offset = 22;
        _y_frame_offset = 102;
    }
    if (gtv_lens_limits(ref_env, _magnify_genv, 
        _magnify_x - _x_frame_offset, 
        _magnify_y - _y_frame_offset,
     _magnify_zoom_factor)) // error with too big zoom factor
        _magnify_unzoom();
    gtv_refresh( _magnify_genv, GTV_REFRESH_MODE_UPDATE);
}

static gboolean _draw_magnify( GtkWidget *widget, cairo_t *cr, gpointer data)
{
    draw_scene_from_genv(_magnify_window, cr, (ggtk_env_t *)_magnify_genv, ((G_env *)data)->win_backg, TRUE);
    return FALSE;
}

static void _magnify_destroy( GtkWidget *widget, gpointer data)
{
    if (_magnify_pointer_is_grabbed()) _ungrab();
    _magnify_window = NULL;
    /* deactivate magnify */
    ggtk_set_pointer_event_handler( NULL, NULL);
    _magnify_reset();
}

static void _magnify_update_window_size( ggtk_env_t *genv, gdouble x, gdouble y,
 gdouble x_root, gdouble y_root)
{
    gtk_window_resize( GTK_WINDOW(_magnify_window), _magnify_width,
     _magnify_height);
    gtv_on_resize( _magnify_genv, _magnify_width, _magnify_height);
    _move_magnify_window( genv, x, y, x_root, y_root);
    gtk_widget_queue_draw( _magnify_window);
}

static void _magnify_increase_window_size()
{
    _magnify_width = (gint)(_magnify_width *
     _magnify_window_size_factor);
    _magnify_height = (gint)(_magnify_height *
     _magnify_window_size_factor);
}

static void _magnify_decrease_window_size()
{
    if (_magnify_width > 10 && _magnify_height > 10) {
        _magnify_width = (gint)(_magnify_width /
         _magnify_window_size_factor);
        _magnify_height = (gint)(_magnify_height /
         _magnify_window_size_factor);
    }
}

static void _on_scroll(GdkEvent *event, gpointer data) 
{
    GdkEventScroll *e = (GdkEventScroll *)event;
    if(!e) return;
    if (e->state & GDK_CONTROL_MASK) {
        if (e->direction == GDK_SCROLL_DOWN 
            || e->direction == GDK_SCROLL_RIGHT) {
            _magnify_decrease_window_size();
            _magnify_update_window_size( (ggtk_env_t *)data, e->x, e->y,
             e->x_root, e->y_root);
        } else if (e->direction == GDK_SCROLL_UP 
            || e->direction == GDK_SCROLL_LEFT) {
            _magnify_increase_window_size();
            _magnify_update_window_size( (ggtk_env_t *)data, e->x, e->y,
             e->x_root, e->y_root);
        }
    } else {
        if (e->direction == GDK_SCROLL_DOWN) {
            _magnify_unzoom();
        } else if (e->direction == GDK_SCROLL_UP) {
            _magnify_zoom();
        }
        _refresh_magnify_data((G_env *)data);
    }
}

static gboolean _on_scroll_discrete(GtkEventControllerScroll *c,
                                   double dx, double dy, gpointer data)
{
    GdkEvent *event = gtk_get_current_event(); // Copy current event
    if (event) {
        _on_scroll(event, data);
        gdk_event_free(event);
    }
    return TRUE;
}

static void _create_magnify_window( G_env *ref_env)
{
    _magnify_pin_window = 0;
    _magnify_window = gtk_window_new(GTK_WINDOW_POPUP);
    GdkDisplay *display = gtk_widget_get_display(_magnify_window);
    #ifdef GAG_USE_X11
    _x11_backend = GDK_IS_X11_DISPLAY(display) > 0;
    #endif
    ggtk_env_t *genv = (ggtk_env_t *)ref_env;
    gtk_window_set_transient_for(GTK_WINDOW(_magnify_window), GTK_WINDOW(genv->main_widget));
    gtk_window_set_destroy_with_parent(GTK_WINDOW(_magnify_window), TRUE);

    gtk_window_set_decorated( GTK_WINDOW(_magnify_window), FALSE);
    gtk_window_set_position( GTK_WINDOW(_magnify_window),
     GTK_WIN_POS_MOUSE);
    gtk_window_set_default_size( GTK_WINDOW(_magnify_window),
     _magnify_width, _magnify_height);
    g_signal_connect( _magnify_window, "draw",
     G_CALLBACK(_draw_magnify), ref_env);
    g_signal_connect( _magnify_window, "destroy",
     G_CALLBACK (_magnify_destroy), ref_env);

    gtk_widget_show_all( _magnify_window);
    gtk_widget_set_app_paintable(_magnify_window, TRUE);
    _magnify_genv = ggtk_new_genv_from( ref_env, _magnify_window, 
        _magnify_width, _magnify_height, FALSE);
    ggtk_env_t * mag_genv = (ggtk_env_t *)_magnify_genv;
    mag_genv->antialiasing = CAIRO_ANTIALIAS_BEST;
    gtv_lens_register(ref_env, _magnify_genv);

    // Scroll events are not managed the same way between Wayland and X11.
    // For X11 we do not have the choice but to listen the scroll discrete event
    if (_x11_backend) {
        GtkEventController *ctrl = gtk_event_controller_scroll_new(
            GTK_WIDGET(_magnify_window),
            GTK_EVENT_CONTROLLER_SCROLL_VERTICAL | GTK_EVENT_CONTROLLER_SCROLL_DISCRETE);
        g_signal_connect(ctrl, "scroll", 
            G_CALLBACK(_on_scroll_discrete), ref_env);
    }
}

static int _magnify_handler( GdkEvent *event, gpointer data, gpointer user_data)
{
    if (event == NULL || event->type == GDK_BUTTON_PRESS) {
        GdkEventButton *e = (GdkEventButton *)event;
        if (_magnify_window != NULL) {
            if (e == NULL ) {
                return 1;
            } else if (e->button == 1) {
                _magnify_decrease_window_size();
                _magnify_update_window_size( (ggtk_env_t *)data, e->x, e->y,
                 e->x_root, e->y_root);
            } else if (e->button == 3) {
                _magnify_increase_window_size();
                _magnify_update_window_size( (ggtk_env_t *)data, e->x, e->y,
                 e->x_root, e->y_root);
            } else if (_magnify_grab_pointer) {
                _destroy_magnify_window();
            }
        } else if (!e || e->button == 2) {
            GdkWindow *window;
            _create_magnify_window( (G_env *)data);
            if (e != NULL) {
                window = e->window;
                _move_magnify_window( (ggtk_env_t *)data, e->x, e->y, e->x_root, e->y_root);
            } else {
                window = gtk_widget_get_window(((ggtk_env_t *)data)->drawing_area); 
            }
            if (_magnify_grab_pointer) _grab(window);
            gtv_on_resize( _magnify_genv, _magnify_width, _magnify_height);
            _refresh_magnify_data((G_env *)data);
        }
        return 1;
    } else if (_magnify_window == NULL) {
        return 0;
    } else if (event->type == GDK_BUTTON_RELEASE) {
        if (!_magnify_grab_pointer) {
            _destroy_magnify_window();
            return 1;
        }
    } else if (event->type == GDK_MOTION_NOTIFY) {
        GdkEventMotion *e = (GdkEventMotion *)event;
        if (e->state & GDK_CONTROL_MASK) {
            _move_magnify_offset( (ggtk_env_t *)data, e->x, e->y, e->x_root, e->y_root);
        } else {
            _move_magnify_window( (ggtk_env_t *)data, e->x, e->y, e->x_root, e->y_root);
        }
        _refresh_magnify_data((G_env *)data);
        return 1;
    } else if (event->type == GDK_SCROLL && !_x11_backend) {
        _on_scroll(event, data);
        return 1;
    } else if (event->type == GDK_KEY_PRESS) {
        GdkEventKey *e = (GdkEventKey *)event;
        if (e->keyval == GDK_KEY_E || e->keyval == GDK_KEY_e) {
            _destroy_magnify_window();
        } else if (e->keyval == GDK_KEY_Shift_L || e->keyval == GDK_KEY_Shift_R) {
            _magnify_flip_pin_window();
        } else if (e->keyval == GDK_KEY_0) {
            _magnify_reset_zoom();
            _refresh_magnify_data((G_env *)data);
        } else if (e->keyval == GDK_KEY_plus || e->keyval == GDK_KEY_KP_Add) {
            _magnify_zoom();
            _refresh_magnify_data((G_env *)data);
        } else if (e->keyval == GDK_KEY_minus || e->keyval == GDK_KEY_KP_Subtract) {
            _magnify_unzoom();
            _refresh_magnify_data((G_env *)data);
        }
        return 1;
    }
    return 0;
}

void ggtk_activate_magnify( GdkEventButton *event, gpointer data)
{
    _magnify_programmatically = (event == NULL);
    ggtk_set_pointer_event_handler( _magnify_handler, NULL);
    ggtk_call_pointer_event_handler( (GdkEvent *)event, data);
}

