/*
 * MIT License
 *
 * Copyright (c) 2019 Alexei Sintotski
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

import 'package:pubspec_lock/pubspec_lock.dart';
import 'package:test/test.dart';

void main() {
  void runTests(
    String rawPubspec,
    HostedPackageDependency expectedHostedDepedency,
  ) {
    group('when $PubspecLock.loadFromYamlString is called', () {
      group('given pubspec.lock with single hosted dependency', () {
        final pubspecLock = rawPubspec.loadPubspecLockFromYaml();
        test('it produces exactly one dependency object', () {
          expect(pubspecLock.packages.length, 1);
        });
        test('it produces PackageDependency with correct package name', () {
          expect(pubspecLock.packages.first.package(), package);
        });
        test('it produces PackageDependency with correct version', () {
          expect(pubspecLock.packages.first.version(), version);
        });
        test('it produces PackageDependency of correct type', () {
          expect(pubspecLock.packages.first.type(), DependencyType.transitive);
        });
        test('it produces HostedPackageDependency object', () {
          expect(isHostedDependency(pubspecLock.packages.first), isTrue);
        });
        test('it produces HostedPackageDependency object with correct data',
            () {
          expect(
            hostedPackageDependency(pubspecLock.packages.first),
            expectedHostedDepedency,
          );
        });
      });
    });

    group('when $PubspecLock.toYaml is called', () {
      group('given pubspec.lock with single hosted dependency', () {
        final pubspecLock = rawPubspec.loadPubspecLockFromYaml();
        final yamlOutput = pubspecLock.toYamlString();
        test('it produces equivalent YAML ouptut', () {
          expect(yamlOutput, rawPubspec);
        });
      });
    });
  }

  group('For dart <2.19', () {
    const pubspecWithHostedDependency = '''
# Generated by pub
# See https://dart.dev/tools/pub/glossary#lockfile
packages:
  $package:
    dependency: transitive
    description:
      name: $name
      url: "$url"
    source: hosted
    version: "$version"
''';

    const expectedPackageDependency = HostedPackageDependency(
      package: package,
      version: version,
      name: name,
      url: url,
      type: DependencyType.transitive,
    );

    runTests(pubspecWithHostedDependency, expectedPackageDependency);
  });

  group('For dart >=2.19', () {
    const pubspecWithHostedDependency = '''
# Generated by pub
# See https://dart.dev/tools/pub/glossary#lockfile
packages:
  $package:
    dependency: transitive
    description:
      name: $name
      sha256: $sha256
      url: "$url"
    source: hosted
    version: "$version"
''';

    const expectedPackageDependency = HostedPackageDependency(
      package: package,
      version: version,
      name: name,
      sha256: sha256,
      url: url,
      type: DependencyType.transitive,
    );

    runTests(pubspecWithHostedDependency, expectedPackageDependency);
  });
}

const package = 'analyzer';
const version = '0.36.4';
const name = 'analyzer';
const url = 'https://pub.dartlang.org';
const sha256 =
    'cbb075ec8fa11d3241cff5e4c20db8aa0732cc0d0bbb3f59b2172d23782dfc0b';

bool isHostedDependency(PackageDependency dependency) => dependency.iswitcho(
      hosted: (d) => true,
      otherwise: () => false,
    );

HostedPackageDependency hostedPackageDependency(PackageDependency dependency) =>
    dependency.iswitcho(
      hosted: (d) => d,
      otherwise: () =>
          throw AssertionError('Expected HostedPackageDependency: $dependency'),
    );
