// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/android/jank_metric_uma_recorder.h"

#include <cstdint>

#include "base/android/jni_android.h"
#include "base/android/jni_array.h"
#include "base/base_jni/JankMetricUMARecorder_jni.h"
#include "base/metrics/histogram_functions.h"
#include "base/time/time.h"
#include "base/trace_event/base_tracing.h"

namespace base::android {

namespace {

void RecordJankMetricReportingIntervalTraceEvent(
    int64_t reporting_interval_start_time,
    int64_t reporting_interval_duration,
    uint64_t janky_frame_count,
    uint64_t non_janky_frame_count) {
  if (reporting_interval_start_time < 0) {
    return;
  }

  // The following code does nothing if base tracing is disabled.
  [[maybe_unused]] auto t =
      perfetto::Track(static_cast<uint64_t>(reporting_interval_start_time));
  TRACE_EVENT_BEGIN(
      "android_webview.timeline", "JankMetricsReportingInterval", t,
      base::TimeTicks::FromUptimeMillis(reporting_interval_start_time),
      "janky_frames", janky_frame_count, "non_janky_frames",
      non_janky_frame_count);
  TRACE_EVENT_END(
      "android_webview.timeline", t,
      base::TimeTicks::FromUptimeMillis(
          (reporting_interval_start_time + reporting_interval_duration)));
}

}  // namespace

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class FrameJankStatus {
  kJanky = 0,
  kNonJanky = 1,
  kMaxValue = kNonJanky,
};

// This function is called from Java with JNI, it's declared in
// base/base_jni/JankMetricUMARecorder_jni.h which is an autogenerated
// header. The actual implementation is in RecordJankMetrics for simpler
// testing.
void JNI_JankMetricUMARecorder_RecordJankMetrics(
    JNIEnv* env,
    const base::android::JavaParamRef<jlongArray>& java_durations_ns,
    const base::android::JavaParamRef<jbooleanArray>& java_jank_status,
    jlong java_reporting_interval_start_time,
    jlong java_reporting_interval_duration) {
  RecordJankMetrics(env, java_durations_ns, java_jank_status,
                    java_reporting_interval_start_time,
                    java_reporting_interval_duration);
}

void RecordJankMetrics(
    JNIEnv* env,
    const base::android::JavaParamRef<jlongArray>& java_durations_ns,
    const base::android::JavaParamRef<jbooleanArray>& java_jank_status,
    jlong java_reporting_interval_start_time,
    jlong java_reporting_interval_duration) {
  std::vector<int64_t> durations_ns;
  JavaLongArrayToInt64Vector(env, java_durations_ns, &durations_ns);

  std::vector<bool> jank_status;
  JavaBooleanArrayToBoolVector(env, java_jank_status, &jank_status);

  std::string frame_duration_histogram_name = "Android.Jank.FrameDuration";
  std::string janky_frames_histogram_name = "Android.Jank.FrameJankStatus";

  for (const int64_t frame_duration_ns : durations_ns) {
    base::UmaHistogramTimes(frame_duration_histogram_name,
                            base::Nanoseconds(frame_duration_ns));
  }

  uint64_t janky_frame_count = 0;

  for (bool is_janky : jank_status) {
    base::UmaHistogramEnumeration(
        janky_frames_histogram_name,
        is_janky ? FrameJankStatus::kJanky : FrameJankStatus::kNonJanky);
    if (is_janky) {
      ++janky_frame_count;
    }
  }

  RecordJankMetricReportingIntervalTraceEvent(
      java_reporting_interval_start_time, java_reporting_interval_duration,
      janky_frame_count, jank_status.size() - janky_frame_count);
}

}  // namespace base::android
