use super::*;
use crate::syscalls::*;

/// ### `path_rename()`
/// Rename a file or directory
/// Inputs:
/// - `Fd old_fd`
///     The base directory for `old_path`
/// - `const char* old_path`
///     Pointer to UTF8 bytes, the file to be renamed
/// - `u32 old_path_len`
///     The number of bytes to read from `old_path`
/// - `Fd new_fd`
///     The base directory for `new_path`
/// - `const char* new_path`
///     Pointer to UTF8 bytes, the new file name
/// - `u32 new_path_len`
///     The number of bytes to read from `new_path`
#[instrument(level = "debug", skip_all, fields(%old_fd, %new_fd, old_path = field::Empty, new_path = field::Empty), ret)]
pub fn path_rename<M: MemorySize>(
    ctx: FunctionEnvMut<'_, WasiEnv>,
    old_fd: WasiFd,
    old_path: WasmPtr<u8, M>,
    old_path_len: M::Offset,
    new_fd: WasiFd,
    new_path: WasmPtr<u8, M>,
    new_path_len: M::Offset,
) -> Result<Errno, WasiError> {
    let env = ctx.data();
    let (memory, mut state, inodes) = unsafe { env.get_memory_and_wasi_state_and_inodes(&ctx, 0) };
    let mut source_str = unsafe { get_input_str_ok!(&memory, old_path, old_path_len) };
    Span::current().record("old_path", source_str.as_str());
    source_str = ctx.data().state.fs.relative_path_to_absolute(source_str);
    let source_path = std::path::Path::new(&source_str);
    let mut target_str = unsafe { get_input_str_ok!(&memory, new_path, new_path_len) };
    Span::current().record("new_path", target_str.as_str());
    target_str = ctx.data().state.fs.relative_path_to_absolute(target_str);
    let target_path = std::path::Path::new(&target_str);

    {
        let source_fd = wasi_try_ok!(state.fs.get_fd(old_fd));
        if !source_fd.rights.contains(Rights::PATH_RENAME_SOURCE) {
            return Ok(Errno::Access);
        }
        let target_fd = wasi_try_ok!(state.fs.get_fd(new_fd));
        if !target_fd.rights.contains(Rights::PATH_RENAME_TARGET) {
            return Ok(Errno::Access);
        }
    }

    // this is to be sure the source file is fetch from filesystem if needed
    wasi_try_ok!(state.fs.get_inode_at_path(
        inodes,
        old_fd,
        source_path.to_str().as_ref().unwrap(),
        true
    ));
    // Create the destination inode if the file exists.
    let _ =
        state
            .fs
            .get_inode_at_path(inodes, new_fd, target_path.to_str().as_ref().unwrap(), true);
    let (source_parent_inode, source_entry_name) =
        wasi_try_ok!(state
            .fs
            .get_parent_inode_at_path(inodes, old_fd, source_path, true));
    let (target_parent_inode, target_entry_name) =
        wasi_try_ok!(state
            .fs
            .get_parent_inode_at_path(inodes, new_fd, target_path, true));
    let mut need_create = true;
    let host_adjusted_target_path = {
        let guard = target_parent_inode.read();
        match guard.deref() {
            Kind::Dir { entries, path, .. } => {
                if entries.contains_key(&target_entry_name) {
                    need_create = false;
                }
                let mut out_path = path.clone();
                out_path.push(std::path::Path::new(&target_entry_name));
                out_path
            }
            Kind::Root { .. } => return Ok(Errno::Notcapable),
            Kind::Socket { .. }
            | Kind::Pipe { .. }
            | Kind::EventNotifications { .. }
            | Kind::Epoll { .. } => return Ok(Errno::Inval),
            Kind::Symlink { .. } | Kind::File { .. } | Kind::Buffer { .. } => {
                debug!("fatal internal logic error: parent of inode is not a directory");
                return Ok(Errno::Inval);
            }
        }
    };

    let source_entry = {
        let mut guard = source_parent_inode.write();
        match guard.deref_mut() {
            Kind::Dir { entries, .. } => {
                wasi_try_ok!(entries.remove(&source_entry_name).ok_or(Errno::Noent))
            }
            Kind::Root { .. } => return Ok(Errno::Notcapable),
            Kind::Socket { .. }
            | Kind::Pipe { .. }
            | Kind::EventNotifications { .. }
            | Kind::Epoll { .. } => {
                return Ok(Errno::Inval);
            }
            Kind::Symlink { .. } | Kind::File { .. } | Kind::Buffer { .. } => {
                debug!("fatal internal logic error: parent of inode is not a directory");
                return Ok(Errno::Inval);
            }
        }
    };

    {
        let mut guard = source_entry.write();
        match guard.deref_mut() {
            Kind::File { ref path, .. } => {
                let result = {
                    let path_clone = path.clone();
                    drop(guard);
                    let state = state;
                    let host_adjusted_target_path = host_adjusted_target_path.clone();
                    __asyncify_light(env, None, async move {
                        state
                            .fs_rename(path_clone, &host_adjusted_target_path)
                            .await
                    })?
                };
                // if the above operation failed we have to revert the previous change and then fail
                if let Err(e) = result {
                    let mut guard = source_parent_inode.write();
                    if let Kind::Dir { entries, .. } = guard.deref_mut() {
                        entries.insert(source_entry_name, source_entry);
                        return Ok(e);
                    }
                } else {
                    {
                        let mut guard = source_entry.write();
                        if let Kind::File { ref mut path, .. } = guard.deref_mut() {
                            *path = host_adjusted_target_path;
                        } else {
                            unreachable!()
                        }
                    }
                }
            }
            Kind::Dir { ref path, .. } => {
                let cloned_path = path.clone();
                let res = {
                    let state = state;
                    let host_adjusted_target_path = host_adjusted_target_path.clone();
                    __asyncify_light(env, None, async move {
                        state
                            .fs_rename(cloned_path, &host_adjusted_target_path)
                            .await
                    })?
                };
                if let Err(e) = res {
                    return Ok(e);
                }
                {
                    drop(guard);
                    let mut guard = source_entry.write();
                    if let Kind::Dir { path, .. } = guard.deref_mut() {
                        *path = host_adjusted_target_path;
                    }
                }
            }
            Kind::Buffer { .. } => {}
            Kind::Symlink { .. } => {}
            Kind::Socket { .. } => {}
            Kind::Pipe { .. } => {}
            Kind::Epoll { .. } => {}
            Kind::EventNotifications { .. } => {}
            Kind::Root { .. } => unreachable!("The root can not be moved"),
        }
    }

    if need_create {
        let mut guard = target_parent_inode.write();
        if let Kind::Dir { entries, .. } = guard.deref_mut() {
            let result = entries.insert(target_entry_name, source_entry);
            assert!(
                result.is_none(),
                "fatal error: race condition on filesystem detected or internal logic error"
            );
        }
    }

    Ok(Errno::Success)
}
