/**
 * Copyright (c) 2023 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.vertiflex.options;

import java.util.EnumSet;
import org.eclipse.elk.alg.vertiflex.EdgeRoutingStrategy;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

/**
 * Declarations for the ELK VertiFlex tree layout algorithm.
 */
@SuppressWarnings("all")
public class VertiFlexMetaDataProvider implements ILayoutMetaDataProvider {
  /**
   * The Y position that the node should be fixed at.
   */
  public static final IProperty<Double> VERTICAL_CONSTRAINT = new Property<Double>(
            "org.eclipse.elk.vertiflex.verticalConstraint");

  /**
   * Default value for {@link #LAYOUT_STRATEGY}.
   */
  private static final EdgeRoutingStrategy LAYOUT_STRATEGY_DEFAULT = EdgeRoutingStrategy.STRAIGHT;

  /**
   * Strategy for the layout of the children. 'straight' for straight line drawings, 'bend' for a possible bend.
   * When straight edges are prioritized the nodes will be reordered in order to guarantee that straight edges are
   * possible. If bend points are enabled on the other hand, the given model order of the nodes is maintained and
   * bend points are introduced to prevent edge node overlaps.
   */
  public static final IProperty<EdgeRoutingStrategy> LAYOUT_STRATEGY = new Property<EdgeRoutingStrategy>(
            "org.eclipse.elk.vertiflex.layoutStrategy",
            LAYOUT_STRATEGY_DEFAULT,
            null,
            null);

  /**
   * Default value for {@link #LAYER_DISTANCE}.
   */
  private static final double LAYER_DISTANCE_DEFAULT = 50.0;

  /**
   * The distance to use between nodes of different layers if no vertical constraints are set.
   */
  public static final IProperty<Double> LAYER_DISTANCE = new Property<Double>(
            "org.eclipse.elk.vertiflex.layerDistance",
            LAYER_DISTANCE_DEFAULT,
            null,
            null);

  /**
   * Default value for {@link #CONSIDER_NODE_MODEL_ORDER}.
   */
  private static final boolean CONSIDER_NODE_MODEL_ORDER_DEFAULT = true;

  /**
   * Consider node model as a secondary criterion when using straight line routing.
   */
  public static final IProperty<Boolean> CONSIDER_NODE_MODEL_ORDER = new Property<Boolean>(
            "org.eclipse.elk.vertiflex.considerNodeModelOrder",
            CONSIDER_NODE_MODEL_ORDER_DEFAULT,
            null,
            null);

  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.vertiflex.verticalConstraint")
        .group("")
        .name("Fixed vertical position")
        .description("The Y position that the node should be fixed at.")
        .type(LayoutOptionData.Type.DOUBLE)
        .optionClass(Double.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.VISIBLE)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.vertiflex.layoutStrategy")
        .group("")
        .name("Edge layout strategy")
        .description("Strategy for the layout of the children. \'straight\' for straight line drawings, \'bend\' for a possible bend. When straight edges are prioritized the nodes will be reordered in order to guarantee that straight edges are possible. If bend points are enabled on the other hand, the given model order of the nodes is maintained and bend points are introduced to prevent edge node overlaps.")
        .defaultValue(LAYOUT_STRATEGY_DEFAULT)
        .type(LayoutOptionData.Type.ENUM)
        .optionClass(EdgeRoutingStrategy.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.VISIBLE)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.vertiflex.layerDistance")
        .group("")
        .name("Layer distance")
        .description("The distance to use between nodes of different layers if no vertical constraints are set.")
        .defaultValue(LAYER_DISTANCE_DEFAULT)
        .type(LayoutOptionData.Type.DOUBLE)
        .optionClass(Double.class)
        .targets(EnumSet.of(LayoutOptionData.Target.PARENTS))
        .visibility(LayoutOptionData.Visibility.VISIBLE)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.vertiflex.considerNodeModelOrder")
        .group("")
        .name("Consider node model order")
        .description("Consider node model as a secondary criterion when using straight line routing.")
        .defaultValue(CONSIDER_NODE_MODEL_ORDER_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.PARENTS))
        .visibility(LayoutOptionData.Visibility.VISIBLE)
        .create()
    );
    new org.eclipse.elk.alg.vertiflex.options.VertiFlexOptions().apply(registry);
  }
}
