/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.builder.ImportBuilder;
import fr.inria.diverse.melange.builder.ModelTypingSpaceBuilder;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.MatchingHelper;
import fr.inria.diverse.melange.lib.ModelUtils;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ExternalLanguage;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.ImportDsl;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.LanguageOperator;
import fr.inria.diverse.melange.metamodel.melange.Mapping;
import fr.inria.diverse.melange.metamodel.melange.MelangePackage;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import fr.inria.diverse.melange.metamodel.melange.NamedElement;
import fr.inria.diverse.melange.metamodel.melange.ResourceType;
import fr.inria.diverse.melange.metamodel.melange.Slice;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.validation.AbstractMelangeValidator;
import fr.inria.diverse.melange.validation.MelangeValidationConstants;
import java.util.Collections;
import java.util.Set;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class MelangeValidator extends AbstractMelangeValidator {
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  private ModelUtils modelUtils;
  
  @Inject
  private MatchingHelper matchingHelper;
  
  @Inject
  private ModelTypingSpaceBuilder builder;
  
  @Check
  public void checkElementsAreNamed(final NamedElement e) {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(e.getName());
    if (_isNullOrEmpty) {
      this.error(
        "All elements must be named", 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.ELEMENT_NAME_EMPTY);
    }
  }
  
  @Check
  public void checkNamesAreUnique(final NamedElement e) {
    EObject _xifexpression = null;
    EObject _eContainer = e.eContainer();
    if ((_eContainer instanceof ModelTypingSpace)) {
      _xifexpression = e.eContainer();
    } else {
      EObject _xifexpression_1 = null;
      EObject _eContainer_1 = e.eContainer().eContainer();
      if ((_eContainer_1 instanceof ModelTypingSpace)) {
        _xifexpression_1 = e.eContainer().eContainer();
      }
      _xifexpression = _xifexpression_1;
    }
    final EObject root = _xifexpression;
    final Function1<NamedElement, Boolean> _function = new Function1<NamedElement, Boolean>() {
      @Override
      public Boolean apply(final NamedElement e_) {
        return Boolean.valueOf(((!Objects.equal(e_, e)) && Objects.equal(e_.getName(), e.getName())));
      }
    };
    boolean _exists = IterableExtensions.<NamedElement>exists(Iterables.<NamedElement>filter(((ModelTypingSpace) root).getElements(), NamedElement.class), _function);
    if (_exists) {
      this.error(
        "Names must be unique", 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.ELEMENT_NAME_DUPLICATED);
    }
  }
  
  @Check
  public void checkNameIsValid(final Language l) {
    if ((!(l instanceof ImportDsl))) {
      if ((StringExtensions.isNullOrEmpty(l.getName()) || (!Character.isUpperCase(l.getName().charAt(0))))) {
        this.error(
          "Language name should start with an uppercase", 
          MelangePackage.Literals.NAMED_ELEMENT__NAME, 
          MelangeValidationConstants.LANGUAGE_NAME_INVALID);
      }
    }
  }
  
  @Check
  public void checkLanguageNameIsntAmbiguous(final Language l) {
    if (((!StringExtensions.isNullOrEmpty(l.getName())) && l.getName().endsWith("MT"))) {
      this.error(
        ("Language shouldn\'t end with \'MT\' to avoid ambiguities " + "with model type names"), 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.LANGUAGE_NAME_AMBIGUOUS);
    }
  }
  
  @Check
  public void checkNameIsValid(final ModelType mt) {
    Language _extracted = mt.getExtracted();
    boolean _not = (!(_extracted instanceof ImportDsl));
    if (_not) {
      if ((StringExtensions.isNullOrEmpty(mt.getName()) || (!Character.isUpperCase(mt.getName().charAt(0))))) {
        this.error(
          "Model type name should start with an uppercase", 
          MelangePackage.Literals.NAMED_ELEMENT__NAME, 
          MelangeValidationConstants.MODELTYPE_NAME_INVALID);
      }
    }
  }
  
  @Check
  public void checkEcoreIsSet(final ModelType mt) {
    Language _extracted = mt.getExtracted();
    boolean _tripleEquals = (_extracted == null);
    if (_tripleEquals) {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(mt.getEcoreUri());
      if (_isNullOrEmpty) {
        this.error(
          "Please specify the Ecore file defining the model type\'s syntax", 
          MelangePackage.Literals.NAMED_ELEMENT__NAME, 
          MelangeValidationConstants.MODELTYPE_ECORE_EMPTY);
      } else {
        if (((mt.getEcoreUri() != null) && (this.modelUtils.loadPkg(mt.getEcoreUri()) == null))) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("The Ecore file \"");
          String _ecoreUri = mt.getEcoreUri();
          _builder.append(_ecoreUri);
          _builder.append("\" couldn\'t be loaded");
          this.error(_builder.toString(), 
            MelangePackage.Literals.MODELING_ELEMENT__ECORE_URI, 
            MelangeValidationConstants.MODELTYPE_ECORE_UNLOADABLE);
        }
      }
    }
  }
  
  @Check
  public void checkEcoreIsSet(final Language l) {
    if (((!this._languageExtensions.hasSyntax(l)) && (!this._languageExtensions.isXmof(l)))) {
      this.error(
        "Please specify the Ecore file defining the language\'s syntax", 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.METAMODEL_ECORE_EMPTY);
    }
  }
  
  @Check
  public void checkExactTypeIsSet(final Language l) {
    ModelType _exactType = l.getExactType();
    boolean _tripleEquals = (_exactType == null);
    if (_tripleEquals) {
      this.error(
        "Please specify the language\'s exact type name", 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.LANGUAGE_EXACTTYPE_EMPTY);
    }
  }
  
  @Check
  public void checkLanguageImplementsExactType(final Language l) {
    boolean _match = this.matchingHelper.match(
      IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(l.getSyntax())), IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(l.getExactType())), null);
    boolean _not = (!_match);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      String _name = l.getName();
      _builder.append(_name);
      _builder.append(" doesn\'t match its exact type ");
      String _name_1 = l.getExactType().getName();
      _builder.append(_name_1);
      this.error(_builder.toString(), 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.METAMODEL_IMPLEMENTS_ERROR);
    }
  }
  
  @Check
  public void checkLanguageOperatorIsAllowed(final LanguageOperator o) {
    EObject _eContainer = o.eContainer();
    final Language lang = ((Language) _eContainer);
    if ((lang instanceof ExternalLanguage)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Language operator ");
      String _name = o.eClass().getName();
      _builder.append(_name);
      _builder.append(" is no allowed in external languages (ie.legacy)");
      this.error(_builder.toString(), 
        MelangePackage.Literals.LANGUAGE_OPERATOR__TARGET_LANGUAGE, 
        MelangeValidationConstants.LANGUAGEOPERATOR_INVALID);
    }
  }
  
  @Check
  public void checkImportIsValid(final Import i) {
    try {
      EObject _eContainer = i.eContainer();
      final Language lang = ((Language) _eContainer);
      final EPackage ecore = this.modelUtils.loadPkg(i.getEcoreUri());
      if ((ecore == null)) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("The Ecore file \"");
        String _ecoreUri = i.getEcoreUri();
        _builder.append(_ecoreUri);
        _builder.append("\" couldn\'t be loaded");
        this.error(_builder.toString(), 
          MelangePackage.Literals.IMPORT__ECORE_URI, 
          MelangeValidationConstants.IMPORT_INVALID_URI);
      }
      boolean _isXcore = this._modelingElementExtensions.isXcore(lang.getSyntax());
      boolean _not = (!_isXcore);
      if (_not) {
        boolean _isEmpty = i.getGenmodelUris().isEmpty();
        if (_isEmpty) {
          String _substring = i.getEcoreUri().substring(0, 
            i.getEcoreUri().lastIndexOf("."));
          final String speculativeGenmodelPath = (_substring + ".genmodel");
          GenModel _loadGenmodel = this.modelUtils.loadGenmodel(speculativeGenmodelPath);
          boolean _tripleEquals = (_loadGenmodel == null);
          if (_tripleEquals) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("No associated genmodel found at \"");
            _builder_1.append(speculativeGenmodelPath);
            _builder_1.append("\"");
            this.error(_builder_1.toString(), 
              MelangePackage.Literals.IMPORT__ECORE_URI, 
              MelangeValidationConstants.IMPORT_INVALID_GENMODEL);
          }
        } else {
          final Procedure2<String, Integer> _function = new Procedure2<String, Integer>() {
            @Override
            public void apply(final String gmUri, final Integer n) {
              GenModel _loadGenmodel = MelangeValidator.this.modelUtils.loadGenmodel(gmUri);
              boolean _tripleEquals = (_loadGenmodel == null);
              if (_tripleEquals) {
                StringConcatenation _builder = new StringConcatenation();
                _builder.append("Couldn\'t load the specified Genmodel \"");
                _builder.append(gmUri);
                _builder.append("\"");
                MelangeValidator.this.error(_builder.toString(), 
                  MelangePackage.Literals.IMPORT__GENMODEL_URIS, (n).intValue(), 
                  MelangeValidationConstants.IMPORT_INVALID_GENMODEL);
              }
            }
          };
          IterableExtensions.<String>forEach(i.getGenmodelUris(), _function);
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        this.error(
          "Unexpected error while loading Ecore/Genmodel", 
          MelangePackage.Literals.IMPORT__ECORE_URI, 
          MelangeValidationConstants.IMPORT_LOADING_EXCEPTION);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Check
  public void checkHasAnnotationProcessorDependency(final Aspect asp) {
    boolean _and = false;
    boolean _and_1 = false;
    JvmTypeReference _aspectTypeRef = asp.getAspectTypeRef();
    JvmType _type = null;
    if (_aspectTypeRef!=null) {
      _type=_aspectTypeRef.getType();
    }
    boolean _tripleNotEquals = (_type != null);
    if (!_tripleNotEquals) {
      _and_1 = false;
    } else {
      JvmType _type_1 = asp.getAspectTypeRef().getType();
      _and_1 = (_type_1 instanceof JvmDeclaredType);
    }
    if (!_and_1) {
      _and = false;
    } else {
      JvmType _type_2 = asp.getAspectTypeRef().getType();
      final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationReference it) {
          return Boolean.valueOf(it.getAnnotation().eIsProxy());
        }
      };
      boolean _exists = IterableExtensions.<JvmAnnotationReference>exists(((JvmDeclaredType) _type_2).getAnnotations(), _function);
      _and = _exists;
    }
    if (_and) {
      this.error(
        ("Cannot find dependency to annotation processor. " + "Please add k3.al.annotationprocessor"), 
        MelangePackage.Literals.ASPECT__ASPECT_TYPE_REF, 
        MelangeValidationConstants.INVALID_ASPECT_IMPORT);
    }
  }
  
  @Check
  public void checkImplements(final Language l) {
    final Procedure2<ModelType, Integer> _function = new Procedure2<ModelType, Integer>() {
      @Override
      public void apply(final ModelType mt, final Integer i) {
        final Function1<Mapping, Boolean> _function = new Function1<Mapping, Boolean>() {
          @Override
          public Boolean apply(final Mapping it) {
            ModelType _to = it.getTo();
            return Boolean.valueOf(Objects.equal(_to, mt));
          }
        };
        boolean _match = MelangeValidator.this.matchingHelper.match(
          IterableExtensions.<EPackage>toList(MelangeValidator.this._modelingElementExtensions.getPkgs(l.getSyntax())), IterableExtensions.<EPackage>toList(MelangeValidator.this._modelingElementExtensions.getPkgs(mt)), 
          IterableExtensions.<Mapping>findFirst(l.getMappings(), _function));
        boolean _not = (!_match);
        if (_not) {
          StringConcatenation _builder = new StringConcatenation();
          String _name = l.getName();
          _builder.append(_name);
          _builder.append(" doesn\'t match the interface ");
          String _name_1 = mt.getName();
          _builder.append(_name_1);
          MelangeValidator.this.error(_builder.toString(), 
            MelangePackage.Literals.NAMED_ELEMENT__NAME, 
            MelangeValidationConstants.METAMODEL_IMPLEMENTS_ERROR);
        }
      }
    };
    IterableExtensions.<ModelType>forEach(l.getImplements(), _function);
  }
  
  @Check
  public void checkNoCyclicInheritance(final Language l) {
    final Function1<Language, Boolean> _function = new Function1<Language, Boolean>() {
      @Override
      public Boolean apply(final Language ll) {
        return Boolean.valueOf(MelangeValidator.this._languageExtensions.getAllSuperLanguages(ll).contains(l));
      }
    };
    boolean _exists = IterableExtensions.<Language>exists(this._languageExtensions.getAllSuperLanguages(l), _function);
    if (_exists) {
      this.error(
        "Cannot inherit from self", 
        MelangePackage.Literals.LANGUAGE_OPERATOR__TARGET_LANGUAGE, 
        MelangeValidationConstants.METAMODEL_SELF_INHERITANCE);
    }
  }
  
  @Check
  public void checkXtextResourceProperlyConfigured(final Language l) {
    if ((Objects.equal(l.getResourceType(), ResourceType.XTEXT) && (l.getXtextSetupRef() == null))) {
      this.error(
        "Xtext resources needs a \'setup\' class", 
        MelangePackage.Literals.LANGUAGE__RESOURCE_TYPE, 
        MelangeValidationConstants.METAMODEL_XTEXT_SETUP);
    }
  }
  
  @Check
  public void checkRuntimeHasBeenGenerated(final Language l) {
    if ((this._languageExtensions.isGeneratedByMelange(l) && (!this._languageExtensions.getRuntimeHasBeenGenerated(l)))) {
      String _name = l.getName();
      String _plus = ("Cannot find EMF runtime for" + _name);
      this.warning(_plus, 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.LANGUAGE_NO_EMF_RUNTIME);
    }
  }
  
  @Check
  public void checkAspectAnnotationIsValid(final Aspect asp) {
    final String clsName = this._aspectExtensions.getAspectAnnotationValue(asp.getAspectTypeRef());
    EObject _eContainer = asp.eContainer();
    final Language lang = ((Language) _eContainer);
    final Function1<Weave, Boolean> _function = new Function1<Weave, Boolean>() {
      @Override
      public Boolean apply(final Weave it) {
        String _simpleName = it.getAspectTypeRef().getSimpleName();
        String _simpleName_1 = asp.getAspectTypeRef().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
      }
    };
    final Weave correspondingWeave = IterableExtensions.<Weave>findFirst(Iterables.<Weave>filter(lang.getOperators(), Weave.class), _function);
    if ((((clsName != null) && (correspondingWeave != null)) && (asp.getAspectedClass() == null))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Cannot find target class ");
      _builder.append(clsName);
      this.error(_builder.toString(), correspondingWeave, 
        MelangePackage.Literals.WEAVE__ASPECT_TYPE_REF, 
        MelangeValidationConstants.WEAVE_INVALID_TARGET);
    }
  }
  
  @Check
  public void checkFindAspectedClass(final Aspect asp) {
    EObject _eContainer = asp.eContainer();
    final Language lang = ((Language) _eContainer);
    final Function1<Weave, Boolean> _function = new Function1<Weave, Boolean>() {
      @Override
      public Boolean apply(final Weave it) {
        String _simpleName = it.getAspectTypeRef().getSimpleName();
        String _simpleName_1 = asp.getAspectTypeRef().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
      }
    };
    final Weave correspondingWeave = IterableExtensions.<Weave>findFirst(Iterables.<Weave>filter(lang.getOperators(), Weave.class), _function);
    JvmTypeReference _aspectTypeRef = null;
    if (correspondingWeave!=null) {
      _aspectTypeRef=correspondingWeave.getAspectTypeRef();
    }
    final String clsName = this._aspectExtensions.getAspectAnnotationValue(_aspectTypeRef);
    if (((this._aspectExtensions.hasAspectAnnotation(asp) && (clsName == null)) && (correspondingWeave != null))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Cannot find in the classpath the class targeted by ");
      String _qualifiedName = asp.getAspectTypeRef().getQualifiedName();
      _builder.append(_qualifiedName);
      this.error(_builder.toString(), correspondingWeave, 
        MelangePackage.Literals.WEAVE__ASPECT_TYPE_REF, 
        MelangeValidationConstants.WEAVE_INVALID_TARGET);
    }
  }
  
  @Check
  public void checkWildcardAspectImport(final Weave w) {
    if (((w.getAspectWildcardImport() != null) && (!w.getAspectWildcardImport().endsWith(".*")))) {
      this.error(
        "Only wildcard imports are supported, e.g. my.pkg.*", 
        MelangePackage.Literals.WEAVE__ASPECT_WILDCARD_IMPORT, 
        MelangeValidationConstants.ASPECT_INVALID_WILDCARD);
    }
  }
  
  @Check
  public void checkSliceCriteria(final Slice s) {
    final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
      @Override
      public Boolean apply(final String clsName) {
        EClassifier _findClassifier = MelangeValidator.this._modelingElementExtensions.findClassifier(s.getTargetLanguage().getSyntax(), clsName);
        return Boolean.valueOf((_findClassifier == null));
      }
    };
    final Iterable<String> invalidRoots = IterableExtensions.<String>filter(s.getRoots(), _function);
    final int invalidRootsSize = IterableExtensions.size(invalidRoots);
    if ((invalidRootsSize > 0)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Invalid slicing criterion: cannot find class");
      {
        if ((invalidRootsSize > 1)) {
          _builder.append("es");
        }
      }
      _builder.append(": ");
      String _join = IterableExtensions.join(invalidRoots, ", ");
      _builder.append(_join);
      this.error(_builder.toString(), 
        MelangePackage.Literals.SLICE__ROOTS, 
        MelangeValidationConstants.SLICE_INVALID_ROOT);
    }
  }
  
  @Check
  public void checkEveryLanguagePackageHasACorrectGenPackage(final Language l) {
    boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(l);
    boolean _not = (!_isGeneratedByMelange);
    if (_not) {
      final Iterable<EPackage> invalidPkgs = this.getEPackagesWithoutGenPackage(l.getSyntax());
      int _size = IterableExtensions.size(invalidPkgs);
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Unexpected error: cannot find a GenPackage for: ");
        final Function1<EPackage, String> _function = new Function1<EPackage, String>() {
          @Override
          public String apply(final EPackage it) {
            return it.getName();
          }
        };
        String _join = IterableExtensions.join(IterableExtensions.<EPackage, String>map(invalidPkgs, _function), ", ");
        _builder.append(_join);
        _builder.append(".");
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(" ");
        _builder_1.append("Please check whether the associated Genmodel is up to date.");
        String _plus = (_builder.toString() + _builder_1);
        this.error(_plus, 
          MelangePackage.Literals.NAMED_ELEMENT__NAME, 
          MelangeValidationConstants.METAMODEL_NO_GENPACKAGE);
      } else {
        final Iterable<EClassifier> invalidCls = this.getEClassifiersWithoutGenClassifier(l.getSyntax());
        int _size_1 = IterableExtensions.size(invalidCls);
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          StringConcatenation _builder_2 = new StringConcatenation();
          _builder_2.append("Unexpected error: cannot find a GenClassifier for: ");
          final Function1<EClassifier, String> _function_1 = new Function1<EClassifier, String>() {
            @Override
            public String apply(final EClassifier it) {
              return it.getName();
            }
          };
          String _join_1 = IterableExtensions.join(IterableExtensions.<EClassifier, String>map(invalidCls, _function_1), ", ");
          _builder_2.append(_join_1);
          _builder_2.append(".");
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append(" ");
          _builder_3.append("Please check whether the associated Genmodel is up to date.");
          String _plus_1 = (_builder_2.toString() + _builder_3);
          this.error(_plus_1, 
            MelangePackage.Literals.NAMED_ELEMENT__NAME, 
            MelangeValidationConstants.METAMODEL_NO_GENPACKAGE);
        }
      }
    }
  }
  
  @Check
  public void checkEveryModelTypePackageHasACorrectGenPackage(final ModelType mt) {
    final Iterable<EPackage> invalidPkgs = this.getEPackagesWithoutGenPackage(mt);
    int _size = IterableExtensions.size(invalidPkgs);
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Cannot find a GenPackage for: ");
      final Function1<EPackage, String> _function = new Function1<EPackage, String>() {
        @Override
        public String apply(final EPackage it) {
          return it.getName();
        }
      };
      String _join = IterableExtensions.join(IterableExtensions.<EPackage, String>map(invalidPkgs, _function), ", ");
      _builder.append(_join);
      _builder.append(".");
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(" ");
      _builder_1.append("Please check whether the associated Genmodel is up to date.");
      String _plus = (_builder.toString() + _builder_1);
      this.error(_plus, 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.METAMODEL_NO_GENPACKAGE);
    } else {
      final Iterable<EClassifier> invalidCls = this.getEClassifiersWithoutGenClassifier(mt);
      int _size_1 = IterableExtensions.size(invalidCls);
      boolean _greaterThan_1 = (_size_1 > 0);
      if (_greaterThan_1) {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("Cannot find a GenClassifier for: ");
        final Function1<EClassifier, String> _function_1 = new Function1<EClassifier, String>() {
          @Override
          public String apply(final EClassifier it) {
            return it.getName();
          }
        };
        String _join_1 = IterableExtensions.join(IterableExtensions.<EClassifier, String>map(invalidCls, _function_1), ", ");
        _builder_2.append(_join_1);
        _builder_2.append(".");
        StringConcatenation _builder_3 = new StringConcatenation();
        _builder_3.append(" ");
        _builder_3.append("Please check whether the associated Genmodel is up to date.");
        String _plus_1 = (_builder_2.toString() + _builder_3);
        this.error(_plus_1, 
          MelangePackage.Literals.NAMED_ELEMENT__NAME, 
          MelangeValidationConstants.METAMODEL_NO_GENPACKAGE);
      }
    }
  }
  
  private Iterable<EPackage> getEPackagesWithoutGenPackage(final ModelingElement m) {
    final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
      @Override
      public Boolean apply(final EPackage it) {
        GenPackage _genPkgFor = MelangeValidator.this._modelingElementExtensions.getGenPkgFor(m, it);
        return Boolean.valueOf((_genPkgFor == null));
      }
    };
    return IterableExtensions.<EPackage>filter(this._modelingElementExtensions.getPkgs(m), _function);
  }
  
  private Iterable<EClassifier> getEClassifiersWithoutGenClassifier(final ModelingElement m) {
    final Function1<EClassifier, Boolean> _function = new Function1<EClassifier, Boolean>() {
      @Override
      public Boolean apply(final EClassifier it) {
        return Boolean.valueOf((((!Objects.equal(it.getName(), "EObject")) && (!MelangeValidator.this._ecoreExtensions.isAspectSpecific(it))) && (MelangeValidator.this._modelingElementExtensions.getGenClassifierFor(m, it) == null)));
      }
    };
    return IterableExtensions.<EClassifier>filter(this._modelingElementExtensions.getAllClassifiers(m), _function);
  }
  
  @Check
  public void checkLanguageNameDoesntConflictWithClassifier(final Language l) {
    EClassifier _findClassifier = this._modelingElementExtensions.findClassifier(l.getSyntax(), l.getName());
    boolean _tripleNotEquals = (_findClassifier != null);
    if (_tripleNotEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Language name conflicts with one of its meta-classes: ");
      String _name = l.getName();
      _builder.append(_name);
      this.error(_builder.toString(), 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.LANGUAGE_NAME_CONFLICTS_METACLASS);
    }
  }
  
  @Check
  public void checkOperators(final LanguageOperator op) {
    final Language targetLang = op.getTargetLanguage();
    boolean _isInError = this._languageExtensions.isInError(targetLang);
    if (_isInError) {
      String _name = targetLang.getName();
      String _plus = ("Language \'" + _name);
      String _plus_1 = (_plus + "\' has errors in its definition");
      this.error(_plus_1, 
        MelangePackage.Literals.LANGUAGE_OPERATOR__TARGET_LANGUAGE, 
        MelangeValidationConstants.METAMODEL_IN_ERROR);
    }
  }
  
  @Check
  public void checkEntryPoints(final Language lang) {
    final Set<JvmOperation> entries = this._languageExtensions.getEntryPoints(lang);
    if (((entries.isEmpty() && lang.getEcl().isEmpty()) && (!lang.getSemantics().isEmpty()))) {
      String _name = lang.getName();
      String _plus = ("Language " + _name);
      String _plus_1 = (_plus + " doesn\'t define an entry point. An Aspect\'s ");
      String _plus_2 = (_plus_1 + "method should be tagged with @Main to be identified as an ");
      String _plus_3 = (_plus_2 + "entry point for the execution.");
      this.warning(_plus_3, 
        MelangePackage.Literals.NAMED_ELEMENT__NAME, 
        MelangeValidationConstants.LANGUAGE_NO_MAIN);
    }
  }
  
  @Check
  public void checkExternalAspect(final Weave weave) {
    final Language lang = weave.getOwningLanguage();
    if ((lang instanceof ExternalLanguage)) {
      final Set<EPackage> importFragment = IterableExtensions.<ImportBuilder>head(Iterables.<ImportBuilder>filter(this.builder.getBuilder(lang).getSubBuilders(), ImportBuilder.class)).getModel();
      final Function1<Aspect, Boolean> _function = new Function1<Aspect, Boolean>() {
        @Override
        public Boolean apply(final Aspect it) {
          String _simpleName = it.getAspectTypeRef().getSimpleName();
          String _simpleName_1 = weave.getAspectTypeRef().getSimpleName();
          return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
        }
      };
      final Aspect correspondingAspect = IterableExtensions.<Aspect>findFirst(((ExternalLanguage)lang).getSemantics(), _function);
      EPackage _ecoreFragment = correspondingAspect.getEcoreFragment();
      boolean _match = this.matchingHelper.match(IterableExtensions.<EPackage>toList(importFragment), Collections.<EPackage>unmodifiableList(CollectionLiterals.<EPackage>newArrayList(_ecoreFragment)), null);
      boolean _not = (!_match);
      if (_not) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Aspect in External Language must match aspected class. Check it doesn\'t introduce new features");
        this.error(_builder.toString(), weave, 
          MelangePackage.Literals.WEAVE__ASPECT_TYPE_REF, 
          MelangeValidationConstants.WEAVE_INVALID_TARGET);
      }
    }
  }
}
