/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.processors;

import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import fr.inria.diverse.melange.ast.ASTHelper;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelTypeExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.ast.NamingHelper;
import fr.inria.diverse.melange.eclipse.EclipseProjectHelper;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ExternalLanguage;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.Subtyping;
import fr.inria.diverse.melange.utils.PluginXmlChanger;
import java.util.Arrays;
import java.util.function.Consumer;
import javax.inject.Inject;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.pde.core.plugin.IPluginBase;
import org.eclipse.pde.core.plugin.IPluginExtensionPoint;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.core.plugin.WorkspacePluginModel;
import org.eclipse.pde.internal.core.project.PDEProject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.documentation.IEObjectDocumentationProvider;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.w3c.dom.Element;

/**
 * For each {@link ModelType}, creates the corresponding
 * 'fr.inria.diverse.melange.modeltype' contribution in plugin.xml.
 * For each {@link Language}, creates the corresponding
 * 'fr.inria.diverse.melange.language' contribution in plugin.xml,
 * with the list of adapters towards implemented model types.
 */
@SuppressWarnings("all")
public class ExtensionPointProcessor extends DispatchMelangeProcessor {
  @Inject
  @Extension
  private NamingHelper _namingHelper;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  @Extension
  private ModelTypeExtensions _modelTypeExtensions;

  @Inject
  @Extension
  private ASTHelper _aSTHelper;

  @Inject
  @Extension
  private EclipseProjectHelper _eclipseProjectHelper;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  private IEObjectDocumentationProvider documentationProvider;

  private static final Logger log = Logger.getLogger(ExtensionPointProcessor.class);

  private static final String MELANGE_RESOURCE_PLUGIN = "fr.inria.diverse.melange.resource";

  private static final String LANGUAGE_EXTENSION_POINT = "fr.inria.diverse.melange.language";

  private static final String MODELTYPE_EXTENSION_POINT = "fr.inria.diverse.melange.modeltype";

  protected void _preProcess(final ModelTypingSpace root, final boolean preLinkingPhase) {
    final IProject project = this._eclipseProjectHelper.getProject(root.eResource());
    if (((project != null) && (!preLinkingPhase))) {
      final IFile pluginXml = PDEProject.getPluginXml(project);
      final PluginXmlChanger changer = new PluginXmlChanger();
      changer.load(pluginXml.getLocation().toString());
      final IPluginModelBase pluginModel = PDECore.getDefault().getModelManager().findModel(project);
      final WorkspacePluginModel fModel = new WorkspacePluginModel(pluginXml, false);
      final IPluginBase pluginBase = fModel.getPluginBase();
      final Function1<IPluginModelBase, Boolean> _function = (IPluginModelBase it) -> {
        String _name = it.getBundleDescription().getName();
        return Boolean.valueOf(Objects.equal(_name, ExtensionPointProcessor.MELANGE_RESOURCE_PLUGIN));
      };
      final IPluginModelBase melangeResourcePlugin = IterableExtensions.<IPluginModelBase>findFirst(((Iterable<IPluginModelBase>)Conversions.doWrapArray(PDECore.getDefault().getModelManager().getAllModels())), _function);
      final Function1<IPluginExtensionPoint, Boolean> _function_1 = (IPluginExtensionPoint it) -> {
        String _fullId = it.getFullId();
        return Boolean.valueOf(Objects.equal(_fullId, ExtensionPointProcessor.LANGUAGE_EXTENSION_POINT));
      };
      final IPluginExtensionPoint languageExtensionPoint = IterableExtensions.<IPluginExtensionPoint>findFirst(((Iterable<IPluginExtensionPoint>)Conversions.doWrapArray(melangeResourcePlugin.getExtensions().getExtensionPoints())), _function_1);
      final Function1<IPluginExtensionPoint, Boolean> _function_2 = (IPluginExtensionPoint it) -> {
        String _fullId = it.getFullId();
        return Boolean.valueOf(Objects.equal(_fullId, ExtensionPointProcessor.MODELTYPE_EXTENSION_POINT));
      };
      final IPluginExtensionPoint modeltypeExtensionPoint = IterableExtensions.<IPluginExtensionPoint>findFirst(((Iterable<IPluginExtensionPoint>)Conversions.doWrapArray(melangeResourcePlugin.getExtensions().getExtensionPoints())), _function_2);
      fModel.setEditable(true);
      fModel.load();
      if (((pluginModel != null) && (pluginBase != null))) {
        final Function1<ModelType, Boolean> _function_3 = (ModelType it) -> {
          return Boolean.valueOf(this._modelTypeExtensions.isValid(it));
        };
        final Iterable<ModelType> modelTypes = IterableExtensions.<ModelType>filter(this._aSTHelper.getModelTypes(root), _function_3);
        final Function1<Language, Boolean> _function_4 = (Language it) -> {
          return Boolean.valueOf(this._languageExtensions.isValid(it));
        };
        final Iterable<Language> languages = IterableExtensions.<Language>filter(this._aSTHelper.getLanguages(root), _function_4);
        if (((IterableExtensions.size(modelTypes) > 0) && (modeltypeExtensionPoint != null))) {
          changer.deleteExtensions(ExtensionPointProcessor.MODELTYPE_EXTENSION_POINT);
          final Element extensionTag = changer.addExtension(
            ExtensionPointProcessor.MODELTYPE_EXTENSION_POINT);
          final Consumer<ModelType> _function_5 = (ModelType mt) -> {
            try {
              final String fqn = this._iQualifiedNameProvider.getFullyQualifiedName(mt).toString();
              final String doc = this.documentationProvider.getDocumentation(mt);
              final Element modelTypeTag = changer.addChild(extensionTag, 
                "modeltype");
              changer.addAttribute(modelTypeTag, "id", fqn);
              changer.addAttribute(modelTypeTag, "uri", this._modelTypeExtensions.getUri(mt));
              if (((doc != null) && (!doc.isEmpty()))) {
                changer.addAttribute(modelTypeTag, 
                  "description", doc);
              }
              final Consumer<Subtyping> _function_6 = (Subtyping superMt) -> {
                final Element subtypingTag = changer.addChild(modelTypeTag, 
                  "subtyping");
                changer.addAttribute(subtypingTag, 
                  "modeltypeId", 
                  this._iQualifiedNameProvider.getFullyQualifiedName(superMt.getSuperType()).toString());
              };
              mt.getSubtypingRelations().forEach(_function_6);
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Registered new modeltype contribution ");
              _builder.append("<");
              _builder.append(fqn);
              _builder.append(", ");
              String _uri = this._modelTypeExtensions.getUri(mt);
              _builder.append(_uri);
              _builder.append("> in plugin.xml");
              ExtensionPointProcessor.log.debug(_builder);
            } catch (final Throwable _t) {
              if (_t instanceof CoreException) {
                final CoreException e = (CoreException)_t;
                StringConcatenation _builder_1 = new StringConcatenation();
                _builder_1.append("Failed to register new model type contribution");
                ExtensionPointProcessor.log.error(_builder_1, e);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          };
          modelTypes.forEach(_function_5);
        }
        if (((IterableExtensions.size(languages) > 0) && (languageExtensionPoint != null))) {
          changer.deleteExtensions(ExtensionPointProcessor.LANGUAGE_EXTENSION_POINT);
          final Element extensionTag_1 = changer.addExtension(ExtensionPointProcessor.LANGUAGE_EXTENSION_POINT);
          final Consumer<Language> _function_6 = (Language l) -> {
            try {
              final String fqn = this._iQualifiedNameProvider.getFullyQualifiedName(l).toString();
              final String doc = this.documentationProvider.getDocumentation(l);
              final Element languageTag = changer.addChild(extensionTag_1, 
                "language");
              changer.addAttribute(languageTag, 
                "id", fqn);
              changer.addAttribute(languageTag, 
                "exactType", 
                this._iQualifiedNameProvider.getFullyQualifiedName(l.getExactType()).toString());
              changer.addAttribute(languageTag, 
                "uri", 
                IterableExtensions.<EPackage>head(this._modelingElementExtensions.getPkgs(l.getSyntax())).getNsURI());
              if (((doc != null) && (!doc.isEmpty()))) {
                changer.addAttribute(languageTag, 
                  "description", doc);
              }
              final Function1<JvmOperation, String> _function_7 = (JvmOperation it) -> {
                String _xblockexpression = null;
                {
                  final Function1<JvmFormalParameter, String> _function_8 = (JvmFormalParameter it_1) -> {
                    return it_1.getParameterType().getType().getQualifiedName();
                  };
                  final String args = IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(it.getParameters(), _function_8), ",");
                  StringConcatenation _builder = new StringConcatenation();
                  String _qualifiedName = it.getDeclaringType().getQualifiedName();
                  _builder.append(_qualifiedName);
                  _builder.append(".");
                  String _simpleName = it.getSimpleName();
                  _builder.append(_simpleName);
                  _builder.append("(");
                  _builder.append(args);
                  _builder.append(")");
                  _xblockexpression = _builder.toString();
                }
                return _xblockexpression;
              };
              changer.addAttribute(languageTag, 
                "entryPoints", 
                IterableExtensions.join(IterableExtensions.<JvmOperation, String>map(this._languageExtensions.getEntryPoints(l), _function_7), ";"));
              changer.addAttribute(languageTag, 
                "aspects", 
                this.getSerializedAspects(l));
              final Function1<ModelType, Boolean> _function_8 = (ModelType mt) -> {
                return Boolean.valueOf((!((l instanceof ExternalLanguage) && Objects.equal(l.getExactType(), mt))));
              };
              final Consumer<ModelType> _function_9 = (ModelType mt) -> {
                final Element adapterTag = changer.addChild(languageTag, 
                  "adapter");
                changer.addAttribute(adapterTag, 
                  "modeltypeId", 
                  this._iQualifiedNameProvider.getFullyQualifiedName(mt).toString());
                changer.addAttribute(adapterTag, 
                  "class", 
                  this._namingHelper.adapterNameFor(l.getSyntax(), mt));
              };
              IterableExtensions.<ModelType>filter(l.getImplements(), _function_8).forEach(_function_9);
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Registered new language contribution ");
              _builder.append("<");
              _builder.append(fqn);
              _builder.append("> in plugin.xml");
              ExtensionPointProcessor.log.debug(_builder);
            } catch (final Throwable _t) {
              if (_t instanceof CoreException) {
                final CoreException e = (CoreException)_t;
                StringConcatenation _builder_1 = new StringConcatenation();
                _builder_1.append("Failed to register language contribution");
                ExtensionPointProcessor.log.debug(_builder_1, e);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          };
          languages.forEach(_function_6);
        }
      }
      changer.save(2);
    }
  }

  /**
   * Returns all the fully qualified name of the aspects woven on the
   * {@link Language} {@code lang}, following the template:
   * <br>
   * aspectizedClass:aspect,...,aspect;...;aspectizedClass:aspect,...,aspect
   */
  private String getSerializedAspects(final Language lang) {
    final ArrayListMultimap<String, Object> mapping = ArrayListMultimap.<String, Object>create();
    final Consumer<Aspect> _function = (Aspect asp) -> {
      JvmTypeReference _aspectTypeRef = asp.getAspectTypeRef();
      String _qualifiedName = null;
      if (_aspectTypeRef!=null) {
        _qualifiedName=_aspectTypeRef.getQualifiedName();
      }
      final String aspName = _qualifiedName;
      EClass _aspectedClass = asp.getAspectedClass();
      String _name = null;
      if (_aspectedClass!=null) {
        _name=_aspectedClass.getName();
      }
      final String targetName = _name;
      if (((aspName != null) && (targetName != null))) {
        mapping.put(targetName, aspName);
      }
    };
    lang.getSemantics().forEach(_function);
    final Function1<String, String> _function_1 = (String target) -> {
      String _xblockexpression = null;
      {
        final String aspects = IterableExtensions.join(mapping.get(target), ",");
        StringConcatenation _builder = new StringConcatenation();
        _builder.append(target);
        _builder.append(":");
        _builder.append(aspects);
        _xblockexpression = _builder.toString();
      }
      return _xblockexpression;
    };
    return IterableExtensions.join(IterableExtensions.<String, String>map(mapping.keySet(), _function_1), ";");
  }

  public void preProcess(final EObject root, final boolean preLinkingPhase) {
    if (root instanceof ModelTypingSpace) {
      _preProcess((ModelTypingSpace)root, preLinkingPhase);
      return;
    } else if (root != null) {
      _preProcess(root, preLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root, preLinkingPhase).toString());
    }
  }
}
