/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.jvmmodel;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import fr.inria.diverse.melange.ast.ModelTypeExtensions;
import fr.inria.diverse.melange.ast.NamingHelper;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EGenericType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.ETypeParameter;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.common.types.JvmTypeParameter;
import org.eclipse.xtext.common.types.JvmTypeParameterDeclarator;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.TypesFactory;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class holds the generation of {@link JvmTypeReference} towards elements
 * of a {@link Metamodel} or a {@link ModelType}. The various inferrers use it
 * to harmonize the code generation.
 */
@Singleton
@SuppressWarnings("all")
public class MelangeTypesBuilder {
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private ModelTypeExtensions _modelTypeExtensions;

  @Inject
  @Extension
  private NamingHelper _namingHelper;

  @Inject
  private TypeReferences typeReferences;

  @Inject
  private JvmTypeReferenceBuilder.Factory builderFactory;

  @Inject
  @Extension
  private JvmTypeReferenceBuilder builder;

  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;

  /**
   * Should be invoked first to inject the required dependencies as the
   * framework doesn't know anything about this class and won't inject
   * it automatically.
   */
  public void setContext(final ResourceSet rs) {
    this.builder = this.builderFactory.create(rs);
  }

  /**
   * @param ctx  The {@link ModelType} defining the group of types of interest
   * @param f    The feature for which a {@link JvmTypeReference} must be created
   * @param decl A list of type parameter declarators in case {@code f}
   *              depends on generic types
   * @return The {@link JvmTypeReference} to be used in the JVM model
   */
  protected JvmTypeReference _typeRef(final ModelType ctx, final ETypedElement f, final Iterable<JvmTypeParameterDeclarator> decl) {
    JvmTypeReference _switchResult = null;
    boolean _matched = false;
    EGenericType _eGenericType = f.getEGenericType();
    boolean _tripleNotEquals = (_eGenericType != null);
    if (_tripleNotEquals) {
      _matched=true;
      _switchResult = this.typeRef(ctx, f.getEGenericType(), decl);
    }
    if (!_matched) {
      EClassifier _eType = f.getEType();
      boolean _tripleNotEquals_1 = (_eType != null);
      if (_tripleNotEquals_1) {
        _matched=true;
        _switchResult = this.typeRef(ctx, f.getEType(), decl);
      }
    }
    if (!_matched) {
      if (f instanceof EOperation) {
        _matched=true;
        _switchResult = this.builder.typeRef(Void.TYPE);
      }
    }
    if (!_matched) {
      _switchResult = this.builder.typeRef(Object.class);
    }
    final JvmTypeReference baseType = _switchResult;
    JvmTypeReference _xifexpression = null;
    boolean _isMany = f.isMany();
    if (_isMany) {
      _xifexpression = this.builder.typeRef(EList.class, this._jvmTypesBuilder.cloneWithProxies(baseType));
    } else {
      _xifexpression = baseType;
    }
    return _xifexpression;
  }

  /**
   * @param ctx  The {@link ModelType} defining the group of types of interest
   * @param t    The {@link EGenericType} for which a type reference is requested
   * @param decl A list of type parameter declarators in case {@code t}
   *              depends on generic types
   * @return The {@link JvmTypeReference} to be used in the JVM model
   */
  protected JvmTypeReference _typeRef(final ModelType ctx, final EGenericType t, final Iterable<JvmTypeParameterDeclarator> decl) {
    JvmTypeReference _xifexpression = null;
    ETypeParameter _eTypeParameter = t.getETypeParameter();
    boolean _tripleNotEquals = (_eTypeParameter != null);
    if (_tripleNotEquals) {
      _xifexpression = this.createTypeParameterReference(((JvmTypeParameterDeclarator[])Conversions.unwrapArray(decl, JvmTypeParameterDeclarator.class)), t.getETypeParameter().getName());
    } else {
      JvmTypeReference _xifexpression_1 = null;
      boolean _isEmpty = t.getETypeArguments().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final Function1<EGenericType, JvmTypeReference> _function = (EGenericType ta) -> {
          JvmTypeReference _xifexpression_2 = null;
          EClassifier _eClassifier = ta.getEClassifier();
          boolean _tripleNotEquals_1 = (_eClassifier != null);
          if (_tripleNotEquals_1) {
            JvmTypeReference _xifexpression_3 = null;
            boolean _isExtracted = this._modelTypeExtensions.isExtracted(ctx);
            if (_isExtracted) {
              _xifexpression_3 = this.builder.typeRef(this._namingHelper.getFqnFor(ctx.getExtracted().getSyntax(), ta.getEClassifier()));
            } else {
              _xifexpression_3 = this.typeRef(ctx, ta.getEClassifier(), decl);
            }
            _xifexpression_2 = _xifexpression_3;
          } else {
            JvmTypeReference _xifexpression_4 = null;
            ETypeParameter _eTypeParameter_1 = ta.getETypeParameter();
            boolean _tripleNotEquals_2 = (_eTypeParameter_1 != null);
            if (_tripleNotEquals_2) {
              _xifexpression_4 = this.createTypeParameterReference(((JvmTypeParameterDeclarator[])Conversions.unwrapArray(decl, JvmTypeParameterDeclarator.class)), ta.getETypeParameter().getName());
            } else {
              _xifexpression_4 = TypesFactory.eINSTANCE.createJvmWildcardTypeReference();
            }
            _xifexpression_2 = _xifexpression_4;
          }
          return _xifexpression_2;
        };
        _xifexpression_1 = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, t.getEClassifier()), 
          ((JvmTypeReference[])Conversions.unwrapArray(ListExtensions.<EGenericType, JvmTypeReference>map(t.getETypeArguments(), _function), JvmTypeReference.class)));
      } else {
        JvmTypeReference _xifexpression_2 = null;
        EClassifier _eClassifier = t.getEClassifier();
        boolean _tripleNotEquals_1 = (_eClassifier != null);
        if (_tripleNotEquals_1) {
          _xifexpression_2 = this.typeRef(ctx, t.getEClassifier(), decl);
        } else {
          _xifexpression_2 = this.builder.typeRef(Object.class);
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  /**
   * @param ctx  The {@link ModelType} defining the group of types of interest
   * @param cls  The {@link EClassifier} for which a {@link JvmTypeReference}
   *              must be created
   * @param decl A list of type parameter declarators in case {@code cls}
   *              depends on generic types
   * @return The {@link JvmTypeReference} to be used in the JVM model
   */
  protected JvmTypeReference _typeRef(final ModelType ctx, final EClassifier cls, final Iterable<JvmTypeParameterDeclarator> decl) {
    JvmTypeReference _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(cls, null)) {
      _matched=true;
      _switchResult = this.builder.typeRef(Object.class);
    }
    if (!_matched) {
      if (cls instanceof EClass) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        boolean _isEmpty = ((EClass)cls).getETypeParameters().isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final Function1<ETypeParameter, JvmTypeReference> _function = (ETypeParameter p) -> {
            return this.createTypeParameterReference(((JvmTypeParameterDeclarator[])Conversions.unwrapArray(decl, JvmTypeParameterDeclarator.class)), p.getName());
          };
          _xifexpression = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls), 
            ((JvmTypeReference[])Conversions.unwrapArray(ListExtensions.<ETypeParameter, JvmTypeReference>map(((EClass)cls).getETypeParameters(), _function), JvmTypeReference.class)));
        } else {
          JvmTypeReference _xifexpression_1 = null;
          boolean _isAbstractable = this._ecoreExtensions.isAbstractable(((EClass)cls));
          if (_isAbstractable) {
            _xifexpression_1 = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls));
          } else {
            JvmTypeReference _xifexpression_2 = null;
            Class<?> _instanceClass = ((EClass)cls).getInstanceClass();
            boolean _tripleNotEquals = (_instanceClass != null);
            if (_tripleNotEquals) {
              _xifexpression_2 = this.builder.typeRef(((EClass)cls).getInstanceClass().getName());
            } else {
              JvmTypeReference _xifexpression_3 = null;
              String _instanceTypeName = ((EClass)cls).getInstanceTypeName();
              boolean _tripleNotEquals_1 = (_instanceTypeName != null);
              if (_tripleNotEquals_1) {
                _xifexpression_3 = this.builder.typeRef(((EClass)cls).getInstanceTypeName());
              } else {
                _xifexpression_3 = this.builder.typeRef(EObject.class);
              }
              _xifexpression_2 = _xifexpression_3;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (cls instanceof EEnum) {
        _matched=true;
        _switchResult = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls));
      }
    }
    if (!_matched) {
      if (cls instanceof EDataType) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        Class<?> _instanceClass = ((EDataType)cls).getInstanceClass();
        boolean _tripleNotEquals = (_instanceClass != null);
        if (_tripleNotEquals) {
          _xifexpression = this.builder.typeRef(((EDataType)cls).getInstanceClass().getName());
        } else {
          JvmTypeReference _xifexpression_1 = null;
          String _instanceTypeName = ((EDataType)cls).getInstanceTypeName();
          boolean _tripleNotEquals_1 = (_instanceTypeName != null);
          if (_tripleNotEquals_1) {
            _xifexpression_1 = this.builder.typeRef(((EDataType)cls).getInstanceTypeName());
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      _switchResult = this.builder.typeRef(Object.class);
    }
    return _switchResult;
  }

  /**
   * @param ctx  The {@link Metamodel} defining the group of types of interest
   * @param cls  The {@link EClassifier} for which a {@link JvmTypeReference}
   *              must be created
   * @param decl A list of type parameter declarators in case {@code cls}
   *              depends on generic types
   * @return The {@link JvmTypeReference} to be used in the JVM model
   */
  protected JvmTypeReference _typeRef(final Metamodel ctx, final EClassifier cls, final Iterable<JvmTypeParameterDeclarator> decl) {
    JvmTypeReference _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(cls, null)) {
      _matched=true;
      _switchResult = this.builder.typeRef(Object.class);
    }
    if (!_matched) {
      if (cls instanceof EClass) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        boolean _isEmpty = ((EClass)cls).getETypeParameters().isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final Function1<ETypeParameter, JvmTypeReference> _function = (ETypeParameter p) -> {
            return this.createTypeParameterReference(((JvmTypeParameterDeclarator[])Conversions.unwrapArray(decl, JvmTypeParameterDeclarator.class)), p.getName());
          };
          _xifexpression = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls), 
            ((JvmTypeReference[])Conversions.unwrapArray(ListExtensions.<ETypeParameter, JvmTypeReference>map(((EClass)cls).getETypeParameters(), _function), JvmTypeReference.class)));
        } else {
          JvmTypeReference _xifexpression_1 = null;
          boolean _isAbstractable = this._ecoreExtensions.isAbstractable(((EClass)cls));
          if (_isAbstractable) {
            _xifexpression_1 = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls));
          } else {
            JvmTypeReference _xifexpression_2 = null;
            Class<?> _instanceClass = ((EClass)cls).getInstanceClass();
            boolean _tripleNotEquals = (_instanceClass != null);
            if (_tripleNotEquals) {
              _xifexpression_2 = this.builder.typeRef(((EClass)cls).getInstanceClass().getName());
            } else {
              JvmTypeReference _xifexpression_3 = null;
              String _instanceTypeName = ((EClass)cls).getInstanceTypeName();
              boolean _tripleNotEquals_1 = (_instanceTypeName != null);
              if (_tripleNotEquals_1) {
                _xifexpression_3 = this.builder.typeRef(((EClass)cls).getInstanceTypeName());
              } else {
                _xifexpression_3 = this.builder.typeRef(EObject.class);
              }
              _xifexpression_2 = _xifexpression_3;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (cls instanceof EEnum) {
        _matched=true;
        _switchResult = this.builder.typeRef(this._namingHelper.getFqnFor(ctx, cls));
      }
    }
    if (!_matched) {
      if (cls instanceof EDataType) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        Class<?> _instanceClass = ((EDataType)cls).getInstanceClass();
        boolean _tripleNotEquals = (_instanceClass != null);
        if (_tripleNotEquals) {
          _xifexpression = this.builder.typeRef(((EDataType)cls).getInstanceClass().getName());
        } else {
          JvmTypeReference _xifexpression_1 = null;
          String _instanceTypeName = ((EDataType)cls).getInstanceTypeName();
          boolean _tripleNotEquals_1 = (_instanceTypeName != null);
          if (_tripleNotEquals_1) {
            _xifexpression_1 = this.builder.typeRef(((EDataType)cls).getInstanceTypeName());
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      _switchResult = this.builder.typeRef(Object.class);
    }
    return _switchResult;
  }

  protected JvmTypeReference _typeRef(final Metamodel ctx, final Metamodel sup, final EClassifier cls) {
    JvmTypeReference _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(cls, null)) {
      _matched=true;
      _switchResult = this.builder.typeRef(Object.class);
    }
    if (!_matched) {
      if (cls instanceof EClass) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        boolean _isAbstractable = this._ecoreExtensions.isAbstractable(((EClass)cls));
        if (_isAbstractable) {
          _xifexpression = this.builder.typeRef(this._namingHelper.adapterNameFor(ctx, sup, ((EClass)cls)));
        } else {
          JvmTypeReference _xifexpression_1 = null;
          Class<?> _instanceClass = ((EClass)cls).getInstanceClass();
          boolean _tripleNotEquals = (_instanceClass != null);
          if (_tripleNotEquals) {
            _xifexpression_1 = this.builder.typeRef(((EClass)cls).getInstanceClass().getName());
          } else {
            JvmTypeReference _xifexpression_2 = null;
            String _instanceTypeName = ((EClass)cls).getInstanceTypeName();
            boolean _tripleNotEquals_1 = (_instanceTypeName != null);
            if (_tripleNotEquals_1) {
              _xifexpression_2 = this.builder.typeRef(((EClass)cls).getInstanceTypeName());
            } else {
              _xifexpression_2 = this.builder.typeRef(EObject.class);
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (cls instanceof EEnum) {
        _matched=true;
        _switchResult = this.builder.typeRef(this._namingHelper.getFqnFor(sup, cls));
      }
    }
    if (!_matched) {
      if (cls instanceof EDataType) {
        _matched=true;
        JvmTypeReference _xifexpression = null;
        Class<?> _instanceClass = ((EDataType)cls).getInstanceClass();
        boolean _tripleNotEquals = (_instanceClass != null);
        if (_tripleNotEquals) {
          _xifexpression = this.builder.typeRef(((EDataType)cls).getInstanceClass().getName());
        } else {
          JvmTypeReference _xifexpression_1 = null;
          String _instanceTypeName = ((EDataType)cls).getInstanceTypeName();
          boolean _tripleNotEquals_1 = (_instanceTypeName != null);
          if (_tripleNotEquals_1) {
            _xifexpression_1 = this.builder.typeRef(((EDataType)cls).getInstanceTypeName());
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      _switchResult = this.builder.typeRef(Object.class);
    }
    return _switchResult;
  }

  /**
   * Returns a new {@link JvmParameterizedTypeReference} to the type parameter
   * named {@code find} in {@code lst}
   */
  public JvmTypeReference createTypeParameterReference(final JvmTypeParameterDeclarator[] lst, final String find) {
    final Function1<JvmTypeParameterDeclarator, EList<JvmTypeParameter>> _function = (JvmTypeParameterDeclarator it) -> {
      return it.getTypeParameters();
    };
    final Function1<JvmTypeParameter, Boolean> _function_1 = (JvmTypeParameter it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, find));
    };
    final JvmTypeParameter findRef = IterableExtensions.<JvmTypeParameter>findFirst(Iterables.<JvmTypeParameter>concat(ListExtensions.<JvmTypeParameterDeclarator, EList<JvmTypeParameter>>map(((List<JvmTypeParameterDeclarator>)Conversions.doWrapArray(lst)), _function)), _function_1);
    JvmTypeReference _xifexpression = null;
    if ((findRef != null)) {
      _xifexpression = this.typeReferences.createTypeRef(findRef);
    } else {
      _xifexpression = this.builder.typeRef(Object.class);
    }
    return _xifexpression;
  }

  public JvmTypeReference typeRef(final ModelingElement ctx, final EObject sup, final Object cls) {
    if (ctx instanceof Metamodel
         && sup instanceof Metamodel
         && cls instanceof EClassifier) {
      return _typeRef((Metamodel)ctx, (Metamodel)sup, (EClassifier)cls);
    } else if (ctx instanceof Metamodel
         && sup instanceof EClassifier
         && cls instanceof Iterable) {
      return _typeRef((Metamodel)ctx, (EClassifier)sup, (Iterable<JvmTypeParameterDeclarator>)cls);
    } else if (ctx instanceof ModelType
         && sup instanceof EClassifier
         && cls instanceof Iterable) {
      return _typeRef((ModelType)ctx, (EClassifier)sup, (Iterable<JvmTypeParameterDeclarator>)cls);
    } else if (ctx instanceof ModelType
         && sup instanceof ETypedElement
         && cls instanceof Iterable) {
      return _typeRef((ModelType)ctx, (ETypedElement)sup, (Iterable<JvmTypeParameterDeclarator>)cls);
    } else if (ctx instanceof ModelType
         && sup instanceof EGenericType
         && cls instanceof Iterable) {
      return _typeRef((ModelType)ctx, (EGenericType)sup, (Iterable<JvmTypeParameterDeclarator>)cls);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ctx, sup, cls).toString());
    }
  }
}
