/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.Language;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jface.text.Document;
import org.eclipse.text.edits.TextEdit;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class AspectRenamer {
  @Inject
  @Extension
  private IQualifiedNameConverter _iQualifiedNameConverter;

  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  private static final Logger log = Logger.getLogger(AspectRenamer.class);

  /**
   * Apply renaming rules on the Aspects' files generated by Kermeta3
   */
  public void processRenaming(final List<Aspect> aspects, final Language l, final Set<String> sourceAspectNs, final List<RenamingRuleManager> rulesManagers) {
    final List<EClass> allClasses = IterableExtensions.<EClass>toList(this._modelingElementExtensions.getAllClasses(l.getSyntax()));
    final IWorkspace workspace = ResourcesPlugin.getWorkspace();
    final IWorkspaceRoot root = workspace.getRoot();
    final IProject[] projects = root.getProjects();
    final Function1<IProject, Boolean> _function = (IProject it) -> {
      String _name = it.getName();
      String _externalRuntimeName = this._languageExtensions.getExternalRuntimeName(l);
      return Boolean.valueOf(Objects.equal(_name, _externalRuntimeName));
    };
    final IProject targetProject = IterableExtensions.<IProject>findFirst(((Iterable<IProject>)Conversions.doWrapArray(projects)), _function);
    final IJavaProject javaProject = JavaCore.create(targetProject);
    final ArrayList<IPackageFragmentRoot> roots = CollectionLiterals.<IPackageFragmentRoot>newArrayList();
    try {
      IPackageFragmentRoot[] _allPackageFragmentRoots = javaProject.getAllPackageFragmentRoots();
      Iterables.<IPackageFragmentRoot>addAll(roots, ((Iterable<? extends IPackageFragmentRoot>)Conversions.doWrapArray(_allPackageFragmentRoots)));
    } catch (final Throwable _t) {
      if (_t instanceof JavaModelException) {
        final JavaModelException e = (JavaModelException)_t;
        AspectRenamer.log.error(e);
        return;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final Function1<IPackageFragmentRoot, Boolean> _function_1 = (IPackageFragmentRoot it) -> {
      String _elementName = it.getElementName();
      return Boolean.valueOf(Objects.equal(_elementName, "src-gen"));
    };
    final IPackageFragmentRoot src_genFolder = IterableExtensions.<IPackageFragmentRoot>findFirst(roots, _function_1);
    final List<Pair<String, String>> k3Patterns = this.convertToPattern(aspects, rulesManagers);
    final Function1<Aspect, String> _function_2 = (Aspect it) -> {
      JvmType _type = it.getSource().getAspectTypeRef().getType();
      return ((JvmDeclaredType) _type).getPackageName();
    };
    final Set<String> allWeaveNamespaces = IterableExtensions.<String>toSet(ListExtensions.<Aspect, String>map(aspects, _function_2));
    allWeaveNamespaces.addAll(sourceAspectNs);
    final String targetAspectNamespace = this._languageExtensions.getAspectsNamespace(l);
    final Consumer<Aspect> _function_3 = (Aspect asp) -> {
      final IPackageFragment aspectNamespace = src_genFolder.getPackageFragment(targetAspectNamespace.toString());
      this.processRenaming(asp, aspectNamespace, rulesManagers, k3Patterns, allClasses, allWeaveNamespaces, targetAspectNamespace);
      try {
        targetProject.refreshLocal(IResource.DEPTH_INFINITE, null);
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException e = (CoreException)_t;
          AspectRenamer.log.error("Couldn\'t refresh resource", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    };
    aspects.forEach(_function_3);
  }

  /**
   * Apply renaming rules on the Aspect's files generated by Kermeta3
   */
  private void processRenaming(final Aspect asp, final IPackageFragment aspectNamespace, final List<RenamingRuleManager> rulesManagers, final List<Pair<String, String>> k3Pattern, final List<EClass> allClasses, final Set<String> allAspectNamespaces, final String targetAspectNamespace) {
    final String aspName = asp.getSource().getAspectTypeRef().getSimpleName();
    final String fileName1 = (aspName + ".java");
    final ICompilationUnit cu1 = aspectNamespace.getCompilationUnit(fileName1);
    final Consumer<RenamingRuleManager> _function = (RenamingRuleManager rulesManager) -> {
      RenamerVisitor _renamerVisitor = new RenamerVisitor(rulesManager, allClasses);
      this.applyRenaming(cu1, _renamerVisitor, k3Pattern, allAspectNamespaces, targetAspectNamespace, rulesManager);
    };
    rulesManagers.forEach(_function);
    boolean _hasAspectAnnotation = this._aspectExtensions.hasAspectAnnotation(asp);
    if (_hasAspectAnnotation) {
      final String targetClass = asp.getAspectedClass().getName();
      JvmType _type = asp.getSource().getAspectTypeRef().getType();
      final String targetFqName = this._aspectExtensions.extractAspectAnnotationValue(((JvmDeclaredType) _type)).toString();
      final Function1<RenamingRuleManager, Pair<String, String>> _function_1 = (RenamingRuleManager it) -> {
        return it.getClassRule(targetFqName);
      };
      final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>head(IterableExtensions.<Pair<String, String>>filterNull(ListExtensions.<RenamingRuleManager, Pair<String, String>>map(rulesManagers, _function_1)));
      String _xifexpression = null;
      if ((rule != null)) {
        _xifexpression = this._iQualifiedNameConverter.toQualifiedName(rule.getValue()).getLastSegment();
      } else {
        _xifexpression = this._iQualifiedNameConverter.toQualifiedName(targetClass).getLastSegment();
      }
      final String newClass = _xifexpression;
      final String fileName2 = ((aspName + targetClass) + "AspectContext.java");
      final String fileName3 = ((aspName + targetClass) + "AspectProperties.java");
      final ICompilationUnit cu2 = aspectNamespace.getCompilationUnit(fileName2);
      final ICompilationUnit cu3 = aspectNamespace.getCompilationUnit(fileName3);
      final Consumer<RenamingRuleManager> _function_2 = (RenamingRuleManager rulesManager) -> {
        RenamerVisitor _renamerVisitor = new RenamerVisitor(rulesManager, allClasses);
        this.applyRenaming(cu2, _renamerVisitor, k3Pattern, allAspectNamespaces, targetAspectNamespace, rulesManager);
        RenamerVisitor _renamerVisitor_1 = new RenamerVisitor(rulesManager, allClasses);
        this.applyRenaming(cu3, _renamerVisitor_1, k3Pattern, allAspectNamespaces, targetAspectNamespace, rulesManager);
      };
      rulesManagers.forEach(_function_2);
      try {
        cu2.rename(((aspName + newClass) + "AspectContext.java"), true, null);
        cu3.rename(((aspName + newClass) + "AspectProperties.java"), true, null);
      } catch (final Throwable _t) {
        if (_t instanceof JavaModelException) {
          final JavaModelException e = (JavaModelException)_t;
          AspectRenamer.log.error("Couldn\'t rename aspect classes", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }

  /**
   * Visit {@link sourceUnit} with {@link renamer} and apply changes in
   * the corresponding textual file.
   */
  private void applyRenaming(final ICompilationUnit sourceUnit, final ASTVisitor renamer, final List<Pair<String, String>> k3pattern, final Set<String> allAspectNamespaces, final String targetAspectNamespace, final RenamingRuleManager rulesManager) {
    try {
      String newSource = sourceUnit.getSource();
      newSource = this.replacePatterns(newSource, k3pattern);
      newSource = this.replaceNamespace(newSource, allAspectNamespaces, targetAspectNamespace);
      newSource = this.replaceFactories(newSource, rulesManager);
      sourceUnit.getBuffer().setContents(newSource);
      sourceUnit.getBuffer().save(null, true);
      final String source = sourceUnit.getSource();
      final Document document = new Document(source);
      final ASTParser parser = ASTParser.newParser(AST.JLS8);
      parser.setSource(sourceUnit);
      ASTNode _createAST = parser.createAST(null);
      final CompilationUnit astRoot = ((CompilationUnit) _createAST);
      astRoot.recordModifications();
      astRoot.accept(renamer);
      final TextEdit edits = astRoot.rewrite(document, sourceUnit.getJavaProject().getOptions(true));
      edits.apply(document);
      sourceUnit.getBuffer().setContents(document.get());
      sourceUnit.getBuffer().save(null, true);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        AspectRenamer.log.error("Couldn\'t apply renaming rules", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  /**
   * Rename AspectContext & AspectProperties in fileContent when exist a renaming rule for their base Class
   */
  private String replacePatterns(final String fileContent, final List<Pair<String, String>> patternRules) {
    final StringBuilder newContent = new StringBuilder(fileContent);
    for (final Pair<String, String> rule : patternRules) {
      {
        final String oldPattern = rule.getKey();
        final String newPattern = rule.getValue();
        this.replaceAll(newContent, oldPattern, newPattern);
      }
    }
    return newContent.toString();
  }

  /**
   * Change the package name of the class
   */
  private String replaceNamespace(final String fileContent, final Set<String> allAspectNamespaces, final String targetAspectNamespace) {
    final StringBuilder newContent = new StringBuilder(fileContent);
    final Consumer<String> _function = (String sourceAspectNamespace) -> {
      this.replaceAll(newContent, sourceAspectNamespace, targetAspectNamespace);
    };
    allAspectNamespaces.forEach(_function);
    return newContent.toString();
  }

  private String replaceFactories(final String fileContent, final RenamingRuleManager ruleManager) {
    final StringBuilder newContent = new StringBuilder(fileContent);
    final Consumer<Pair<String, String>> _function = (Pair<String, String> rule) -> {
      int _lastIndexOf = rule.getKey().lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      final String oldSimpleName = rule.getKey().substring(_plus);
      String _firstUpper = StringExtensions.toFirstUpper(oldSimpleName);
      final String oldFactoryName = (_firstUpper + "Factory");
      int _lastIndexOf_1 = rule.getValue().lastIndexOf(".");
      int _plus_1 = (_lastIndexOf_1 + 1);
      final String newSimpleName = rule.getValue().substring(_plus_1);
      String _firstUpper_1 = StringExtensions.toFirstUpper(newSimpleName);
      final String newFactoryName = (_firstUpper_1 + "Factory");
      this.replaceAll(newContent, oldFactoryName, newFactoryName);
    };
    ruleManager.getAllPackageRules().forEach(_function);
    return newContent.toString();
  }

  /**
   * Deduce k3's Aspect patterns from {@link rulesManager} and associated replacement patterns
   */
  private List<Pair<String, String>> convertToPattern(final List<Aspect> aspects, final List<RenamingRuleManager> rulesManagers) {
    final ArrayList<Pair<String, String>> res = CollectionLiterals.<Pair<String, String>>newArrayList();
    final Function1<Aspect, Boolean> _function = (Aspect it) -> {
      return Boolean.valueOf(this._aspectExtensions.hasAspectAnnotation(it));
    };
    final Consumer<Aspect> _function_1 = (Aspect asp) -> {
      final String targetFqName = this._aspectExtensions.getTargetedClassFqn(asp);
      final String aspName = asp.getAspectTypeRef().getSimpleName();
      final Function1<RenamingRuleManager, Pair<String, String>> _function_2 = (RenamingRuleManager it) -> {
        return it.getClassRule(targetFqName);
      };
      final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>head(IterableExtensions.<Pair<String, String>>filterNull(ListExtensions.<RenamingRuleManager, Pair<String, String>>map(rulesManagers, _function_2)));
      if ((rule != null)) {
        int _lastIndexOf = rule.getKey().lastIndexOf(".");
        int _plus = (_lastIndexOf + 1);
        final String oldClassName = rule.getKey().substring(_plus);
        int _lastIndexOf_1 = rule.getValue().lastIndexOf(".");
        int _plus_1 = (_lastIndexOf_1 + 1);
        final String newClassName = rule.getValue().substring(_plus_1);
        Pair<String, String> _mappedTo = Pair.<String, String>of(((aspName + oldClassName) + "AspectContext"), ((aspName + newClassName) + "AspectContext"));
        res.add(_mappedTo);
        Pair<String, String> _mappedTo_1 = Pair.<String, String>of(((aspName + oldClassName) + "AspectProperties"), ((aspName + newClassName) + "AspectProperties"));
        res.add(_mappedTo_1);
      }
    };
    IterableExtensions.<Aspect>filter(aspects, _function).forEach(_function_1);
    return res;
  }

  private void replaceAll(final StringBuilder string, final String oldPattern, final String newPattern) {
    final int oldPatternSize = oldPattern.length();
    final int newPatternSize = newPattern.length();
    int startIndex = 0;
    int index = string.indexOf(oldPattern);
    while ((index != (-1))) {
      {
        final char previousChar = string.charAt((index - 1));
        final char followingChar = string.charAt((index + oldPatternSize));
        if (((!Character.isJavaIdentifierPart(previousChar)) && (!Character.isJavaIdentifierPart(followingChar)))) {
          string.replace(index, (index + oldPatternSize), newPattern);
          startIndex = (index + newPatternSize);
        } else {
          startIndex = (index + oldPatternSize);
        }
        index = string.indexOf(oldPattern, startIndex);
      }
    }
  }
}
