/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFXMLParserImpl.java,v 1.2 2007/03/18 08:39:03 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.xml;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.eodm.rdf.resource.parser.ErrorHandler;
import org.eclipse.eodm.rdf.resource.parser.NamespaceHandler;
import org.eclipse.eodm.rdf.resource.parser.RDFXMLParser;
import org.eclipse.eodm.rdf.resource.parser.StatementHandler;
import org.eclipse.eodm.rdf.resource.parser.element.RDFBlankNode;
import org.eclipse.eodm.rdf.resource.parser.element.RDFLiteralElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFResourceElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFSURIConstant;
import org.eclipse.eodm.rdf.resource.parser.element.RDFTriple;
import org.eclipse.eodm.rdf.resource.parser.element.RDFValue;
import org.eclipse.eodm.rdf.resource.parser.element.URIReference;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserIOException;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * The core implementation of the RDF Parser.
 * <p>
 * The parsing process is devided into 2 steps: First, call XML Parser to parse
 * the rdf document in XML syntax and create possible events. Second, the rdf
 * syntax parser will generate all the node elements and property elements
 * according to RDF semantics. A state machine is maintained for the
 * transformation from XML parser events to RDF elements. All the states in the
 * state chain is stored in a stack named _stateStack in RDFXMLContentHandler
 * class.
 * 
 * <p>
 * <li>Old terms, such as rdf:aboutEach, rdf:aboutEachPrefix and rdf:bagID, are
 * not supported in the RDF Parser.
 * <li>
 * <p>
 * 
 * @since Java <font size="-2"> <sup>TM </sup> </font> 1.4
 */
public class RDFXMLParserImpl implements RDFXMLParser {

    private StatementHandler statementHandler;

    private ErrorHandler errorHandler;

    private NamespaceHandler namespaceHandler;

    // Note: two identical rdf:ID's are allowed, as long as they refer to
    // different resources
    // it is a must to map its uri and rdfId by the pair <uri, rdfID>
    private Set rdfIDs = new HashSet();

    public Stack xmlBaseURI = new Stack();

    private Stack elementStack = new Stack();

    private Stack langStack = new Stack();

    private HashMap namespaces = new HashMap();

    // the XML Parser content handler
    private RDFXMLContentHandler xmlParser;

    private Locator locator;

    public final String XML = "xml";

    public final String XML_LANG = "xml:lang";

    public RDFXMLParserImpl() {
        xmlParser = new RDFXMLContentHandler(this);
    }

    /**
     * Parse a RDF file
     */
    public void parse(InputSource inputSource, String baseURI)
            throws ParserIOException, ParserException {

        if (inputSource == null || baseURI == null)
            throw new IllegalArgumentException(
                    "source or baseURI must NOT be null");

        // call the xml parser to parsing the document in XML syntax
        try {
            SAXParserFactory spf = SAXParserFactory.newInstance();
            XMLReader reader = null;
            try {
                reader = spf.newSAXParser().getXMLReader();
            } catch (ParserConfigurationException e1) {
                throw new ParserException(
                        "System property javax.xml.parsers.SAXParserFactory is not valid");
            }

            // XMLReader reader = XMLReaderFactory.createXMLReader();
            inputSource.setSystemId(baseURI);
            pushXmlBase(baseURI);
            reader.setContentHandler(xmlParser);

            // Set attibutes of SAX parser
            reader.setFeature("http://xml.org/sax/features/validation", false);
            reader.setFeature("http://xml.org/sax/features/namespaces", true);

            // Start to parse the file
            reader.parse(inputSource);
        } catch (SAXException e) {
            if (locator != null) {
                e.printStackTrace();
                e.getCause();
                e.getLocalizedMessage();
                error("meet error on line "
                      + locator.getLineNumber() + locator.getColumnNumber());

            }

            throw new ParserException(e);
        } catch (IOException e) {
            throw new ParserIOException(e);
        }
    }

    /**
     * Set statement handler
     */
    public void setStatementHandler(StatementHandler statementHandler) {
        this.statementHandler = statementHandler;
    }

    /**
     * Set error handler
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * Set namespace handler
     */
    public void setNamespaceHandler(NamespaceHandler namespaceHandler) {
        this.namespaceHandler = namespaceHandler;
    }

    /**
     * Deal with namespace
     * 
     * @param abb
     *            the abbreviation of a URI
     * @param uri
     *            the URI of a resource
     */
    protected void addNamespace(String abb, String uri) {
        namespaces.put(abb, uri);
        if (namespaceHandler != null) {
            namespaceHandler.exportNamespace(abb, uri);
        }
    }

    /**
     * Remove a namespace from namespaces hashset
     * 
     * @param abb
     *            the abbreviation of a URI
     */
    protected void removeNamespace(String abb) {
        namespaces.remove(abb);
    }

    /**
     * Add a node element.
     * 
     * @param namespaceURI
     *            the namespace uri
     * @param localName
     *            the local name
     * @param attributes
     *            all the attribtues
     * @param isEmptyElement
     *            whether the node element is an empty node.
     */
    protected void addNodeElement(String namespaceURI, String localName,
            Attributes attributes) {

        // Get a node element from attributes and create a node element
        RDFResourceElement nodeResource = getNodeResource(attributes);
        NodeElement nodeElement = new NodeElement(nodeResource);

        // Set the language of the nodeelement
        nodeElement.setXmlLang(attributes.getValue(XML_LANG));

        // Check whether the element stack is empty or not.
        // If the stack is not empty, the node element should be an object
        if (!elementStack.isEmpty()) {
            // Node is an object
            NodeElement subject = (NodeElement) elementStack.get(elementStack
                    .size() - 2);
            PropertyElement predicate = (PropertyElement) elementStack
                    .get(elementStack.size() - 1);

            if (predicate.isCollectionContext()) { // parseType is Collection
                RDFResourceElement lastLIResource = predicate
                        .getLastLIResource();
                RDFBlankNode currListResource = new RDFBlankNode();
                if (lastLIResource != null) {
                    exportStatement(lastLIResource, RDFSURIConstant.P_REST_URI,
                            currListResource);
                } else { // the first collection element
                    exportStatement(subject.getResource(), predicate,
                            currListResource);
                }
                exportStatement(currListResource, RDFSURIConstant.P_FIRST_URI,
                        nodeResource);
                predicate.setLastLIResource(currListResource);
            } else {
                exportStatement(subject.getResource(), predicate, nodeResource);
            }
        }
        // The node element is a subject. If not rdf:Description, export a
        // <rdf:type> statement
        if (!(namespaceURI.equals(RDF.NAMESPACE) && localName
                .equals(RDF.S_DESCRIPTION))) {
            URIReference rdfClassName;
            rdfClassName = new URIReference(namespaceURI, localName);
            exportStatement(nodeResource, RDFSURIConstant.P_TYPE_URI,
                    rdfClassName);
        }
        String rdfType = attributes.getValue(RDF.NAMESPACE, RDF.P_TYPE);
        if (rdfType != null) {
            URIReference reference = new URIReference(rdfType);
            exportStatement(nodeResource, RDFSURIConstant.P_TYPE_URI, reference);
        }
        // Generate other attribute triples
        processSubjectsOfNodeAtts(nodeElement, attributes);
        pushElement(nodeElement);
    }

    /**
     * Add a property element. Such attributes as rdf:ID, rdf:parseType and so
     * on should be treated specially.
     * 
     * @param namespaceURI
     *            the namespace uri
     * @param localName
     *            the local name
     * @param attributes
     *            all the attributes
     * @param isEmptyElement
     *            whether the element is an empty element
     */
    protected void addPropertyElement(String namespaceURI, String localName,
            Attributes attributes) {
        URIReference referenceURI;
        boolean propertyAndNodeElement = false;

        if (localName.equals(RDF.S_LI) && namespaceURI.equals(RDF.NAMESPACE)) {
            // This is a list property
            NodeElement subject = (NodeElement) elementStack.get(elementStack
                    .size() - 1);
            referenceURI = new URIReference(RDF.NAMESPACE, "_"
                                                           + subject.nextLI());
        } else {
            referenceURI = new URIReference(namespaceURI, localName);
        }

        // Construct a property element based on the input namespace and
        // localname
        PropertyElement predicate = new PropertyElement(referenceURI);
        predicate.setXmlLang(attributes.getValue(XML_LANG));
        pushElement(predicate);

        // If the rdfID is not null, there is a refied statement.
        String rdfID = attributes.getValue(RDF.NAMESPACE, RDF.S_ID);
        if (rdfID != null) {
            URIReference reificationURI = createResourceFromID(rdfID);
            predicate.setReificationURI(reificationURI);
        }

        // Check the parsetype of a property.
        String rdfParseType = null;
        rdfParseType = attributes.getValue(RDF.NAMESPACE, RDF.S_PARSETYPE);

        // Deal with parsetype
        if (rdfParseType != null) {
            predicate.setParseType(rdfParseType);
            // XMLLiteral
            if (rdfParseType.equals(RDFS.C_LITERAL)) {
                if (attributes.getValue(RDF.NAMESPACE, RDF.S_RESOURCE) != null)
                    error("rdf:resource is forbidden since there is an rdf:parseType of \"Literal\"");
                pushElement(null);
            }
            // Blank node
            else if (rdfParseType.equals(RDFS.C_RESOURCE)) {
                propertyAndNodeElement = true;
                RDFBlankNode blankNode = new RDFBlankNode();
                NodeElement subject = (NodeElement) elementStack
                        .get(elementStack.size() - 2);
                exportStatement(subject.getResource(), predicate, blankNode);

                NodeElement nodeElement = new NodeElement(blankNode);
                // Set the current blanknode is implicit
                nodeElement.setExplicit(false);
                pushElement(nodeElement);
            }
            // Collection
            else if (rdfParseType.equals("Collection"))
                predicate.setCollectionContext(true);
            else if (rdfParseType.equals("daml:collection"))
                predicate.setCollectionContext(true);
            else {
                // no other type is allowed
                error("Unknown parseType: '" + rdfParseType + "'");
            }
        }

        // Deal with non-blank node
        if (!propertyAndNodeElement) {
            RDFResourceElement propertyResource = getPropertyResource(attributes);
            if (propertyResource != null) {
                NodeElement propertyResourceNodeElement = new NodeElement(
                        propertyResource);
                NodeElement subject = (NodeElement) elementStack
                        .get(elementStack.size() - 2);
                exportStatement(subject.getResource(), predicate,
                        propertyResource);

                String rdfType = null;
                rdfType = attributes.getValue(RDF.NAMESPACE, RDF.P_TYPE);
                if (rdfType != null) {
                    URIReference className = new URIReference(rdfType);
                    exportStatement(propertyResource,
                            RDFSURIConstant.P_TYPE_URI, className);
                }
                processSubjectsOfPropertyAtts(propertyResourceNodeElement,
                        attributes);
            }
        }
        // Set the datatype of the property
        String dataType = attributes.getValue(RDF.NAMESPACE, RDF.S_DATATYPE);
        if (dataType != null) {
            predicate.setDataType(dataType);
        }
    }

    /**
     * Remove the top element in elementStack while necessary.
     * 
     */
    public void removeElement() {
        Object topElement = elementStack.peek();
        if (topElement instanceof NodeElement) {
        } else if (topElement instanceof PropertyElement) {
            PropertyElement propertyElement = (PropertyElement) topElement;
            if (propertyElement.isCollectionContext()) {
                RDFResourceElement resource = propertyElement
                        .getLastLIResource();
                exportStatement(resource, RDFSURIConstant.P_REST_URI,
                        RDFSURIConstant.NIL_URI);
            } else if (propertyElement.isEmpty()) {
                NodeElement subject = (NodeElement) elementStack
                        .get(elementStack.size() - 2);
                if (propertyElement.getParseType() != null
                    && propertyElement.getParseType().equals("Literal"))
                    exportStatement(subject.getResource(), propertyElement,
                            new RDFLiteralElement("", null,
                                    RDFSURIConstant.C_XMLLITERAL_URI));
                else
                    exportStatement(subject.getResource(), propertyElement,
                            new RDFLiteralElement(""));
            }
        }
        popElement();
    }

    /**
     * Handle text content.
     * 
     * @param text
     *            the text content
     */
    public void addText(String text) {
        Object topElement = elementStack.peek();
        int offset = 0;
        if (topElement == null) {
            // Here is in a literal context. The subject and predicate can be
            // arbitrarily far down the stack.
            // Calculate the offset to the first non-null stack element, which
            // should be the predicate.
            do {
                offset++;
                topElement = elementStack.get(elementStack.size() - offset - 1);
            } while ((offset < elementStack.size()) && topElement == null);
        }
        if (topElement instanceof PropertyElement) {
            PropertyElement propertyElement = (PropertyElement) topElement;
            String type = propertyElement.getDataType();
            URIReference dataType;
            if (type != null) {
                dataType = new URIReference(type);
            } else {
                dataType = null;
            }
            RDFLiteralElement literal;
            if (RDFS.C_LITERAL.equals(propertyElement.getParseType())) {
                // The language set in the attributes of the predicate is not
                // used for literals.
                literal = new RDFLiteralElement(text, null,
                        RDFSURIConstant.C_XMLLITERAL_URI);
            } else {
                literal = new RDFLiteralElement(text, peekXmlLang(), dataType);
            }
            NodeElement nodeElement = (NodeElement) elementStack
                    .get(elementStack.size() - 2 - offset);
            exportStatement(nodeElement.getResource(), propertyElement, literal);
        } else {
            error("Unexpected element type while adding text");
        }
    }

    /**
     * Deal with the rest attributes of a node element.
     * 
     * @param nodeElement
     *            the node element to be processed
     * @param attributes
     *            all the attributes related with the node element
     */
    private void processSubjectsOfNodeAtts(NodeElement nodeElement,
            Attributes attributes) {
        for (int i = 0; i < attributes.getLength(); i++) {
            if (attributes.getQName(i).startsWith("xmlns"))
                continue;
            String namespace = attributes.getURI(i);
            String localName = attributes.getLocalName(i);
            String uri = namespace + localName;
            if (!(RDF.S_ABOUT_STR.equals(uri)
                  || RDF.S_ID_STR.equals(uri)
                  || RDF.S_NODEID_STR.equals(uri)
                  || RDF.P_TYPE_STR.equals(uri)
                  || org.eclipse.eodm.vocabulary.XML.NAMESPACE
                          .equals(namespace)
                  || org.eclipse.eodm.vocabulary.XML.S_LANG_STR
                          .equals(uri) || org.eclipse.eodm.vocabulary.XML.S_BASE_STR
                    .equals(uri))) {
                String value = attributes.getValue(i);
                RDFLiteralElement literal = new RDFLiteralElement(value,
                        nodeElement.getXmlLang());
                exportStatement(nodeElement.getResource(),
                        new URIReference(uri), literal);
            }
        }
    }

    /**
     * Deal with the rest attributes of a node element.
     * 
     * @param nodeElement
     *            the node element to be processed
     * @param attributes
     *            all the attributes related with the node element
     */
    private void processSubjectsOfPropertyAtts(NodeElement nodeElement,
            Attributes attributes) {
        for (int i = 0; i < attributes.getLength(); i++) {
            // filter out xmlns namespace attribute
            if (attributes.getQName(i).startsWith("xmlns"))
                continue;
            //

            String namespace = attributes.getURI(i);
            String localName = attributes.getLocalName(i);
            String uri = namespace + localName;
            if (!(RDF.S_ID_STR.equals(uri)
                  || RDF.S_PARSETYPE_STR.equals(uri)
                  || RDF.S_RESOURCE_STR.equals(uri)
                  || RDF.S_NODEID_STR.equals(uri)
                  || RDF.P_TYPE_STR.equals(uri)
                  || org.eclipse.eodm.vocabulary.XML.NAMESPACE
                          .equals(namespace)
                  || org.eclipse.eodm.vocabulary.XML.S_LANG_STR
                          .equals(uri) || org.eclipse.eodm.vocabulary.XML.S_BASE_STR
                    .equals(uri))) {
                String value = attributes.getValue(i);
                if (!uri.startsWith(XML)) {
                    exportStatement(nodeElement.getResource(),
                            new URIReference(uri), new RDFLiteralElement(value));
                }
            }
        }
    }

    /**
     * Generate a Node Element
     * 
     * @param attributes
     *            the attributes related with the node element
     * @return a node element
     */
    private RDFResourceElement getNodeResource(Attributes attributes) {

        String rdfAbout = null;
        String rdfID = null;
        String rdfNodeID = null;
        rdfAbout = attributes.getValue(RDF.NAMESPACE, RDF.S_ABOUT);
        rdfID = attributes.getValue(RDF.NAMESPACE, RDF.S_ID);
        rdfNodeID = attributes.getValue(RDF.NAMESPACE, RDF.S_NODEID);

        int resourceCount = 0;
        if (rdfAbout != null) {
            resourceCount++;
        }
        if (rdfID != null) {
            resourceCount++;
        }
        if (rdfNodeID != null) {
            resourceCount++;
        }
        if (resourceCount > 1) {
            // only ONE resource declaration is allowed
            error("only ONE resource declaration is allowed here");
        }
        if (rdfAbout != null) {
            if (rdfAbout.length() == 0) {
                return (RDFResourceElement) xmlBaseURI.peek();
            } else {
                return ((URIReference) xmlBaseURI.peek()).relative(rdfAbout);
            }
        } else if (rdfID != null) {
            return createResourceFromID(rdfID);
        } else if (rdfNodeID != null) {
            checkIDValid(rdfNodeID);
            return new RDFBlankNode(rdfNodeID);
        }
        return new RDFBlankNode();
    }

    /**
     * Handle all the attributes of a property resource. eg. ref:resource or
     * rdf:nodeID
     * 
     * @param attributes
     *            all the attributes
     * @return a property element
     */
    private RDFResourceElement getPropertyResource(Attributes attributes) {
        String rdfResource = null;
        rdfResource = attributes.getValue(RDF.NAMESPACE, RDF.S_RESOURCE);
        String rdfNodeID = null;
        rdfNodeID = attributes.getValue(RDF.NAMESPACE, RDF.S_NODEID);
        if (rdfResource != null && rdfNodeID != null) {
            error("only ONE resource declaration is allowed here");
        }
        if (rdfResource != null) {
            return createResource(rdfResource);
        }
        if (rdfNodeID != null) {
            checkIDValid(rdfNodeID);
            return new RDFBlankNode(rdfNodeID);
        }
        int count = attributes.getLength();
        if (attributes.getValue(RDF.NAMESPACE, RDF.S_DATATYPE) != null) {
            count--;
        }
        if (attributes.getValue(RDF.NAMESPACE, RDF.S_ID) != null) {
            count--;
        }
        if (attributes.getValue(XML_LANG) != null) {
            count--;
        }
        // RTG Change
        String parseType = attributes.getValue(RDF.NAMESPACE, RDF.S_PARSETYPE);
        if (parseType != null) {
            count--;
        }
        // Need to test for attributes in the XML namespace and attributes that
        // are xml reserved.
        for (int i = 0; i < attributes.getLength(); i++) {
            // filter out xmlns attributes
            if (attributes.getQName(i).startsWith("xmlns")) {
                count--;
                continue;
            }
            //
            String namespace = attributes.getURI(i);
            String localName = attributes.getLocalName(i);
            if ((org.eclipse.eodm.vocabulary.XML.NAMESPACE
                    .equals(namespace))
                || (localName
                        .startsWith(org.eclipse.eodm.vocabulary.XML.S_XML))) {
                count--;
            }
        }
        if (count > 0) {
            if (parseType != null && parseType.equals(RDFS.C_LITERAL))
                error("rdf:parseType=\"Literal\" is forbidden since an additional resource node will be created.");
            return new RDFBlankNode();
        }
        return null;
    }

    /**
     * Create a URIReference from a given rdfID
     * 
     * @param rdfID
     *            the input rdfID
     * @return the URIReference generated from the given rdfID
     */
    private URIReference createResourceFromID(String rdfID) {
        checkIDValid(rdfID);
        URIReference id = ((URIReference) xmlBaseURI.peek()).relative("#"
                                                                      + rdfID);
        /*
         * <li> Two identical rdf:ID's are allowed, as long as they refer to
         * different resources. </li>
         */
        boolean noConflict = rdfIDs.add(id);
        if (!noConflict) {
            error("'" + rdfID + "' is already used.");
        }
        return id;
    }

    private void checkIDValid(String id) {
        boolean valid = XMLNCName.validateNCName(id);
        if (!valid)
            error("ID " + id + " is not valid.");
    }

    /**
     * Create a URIReference from a given string
     * 
     * @param resource
     *            the input string
     * @return the URIReference generated from the given string
     */
    private URIReference createResource(String resource) {
        try {
            return ((URIReference) xmlBaseURI.peek()).relative(resource);
        } catch (IllegalArgumentException ie) {
            try {
                int anchor = resource.indexOf('#');
                if (anchor > 0) {
                    return ((URIReference) xmlBaseURI.peek())
                            .relative(resource.substring(0, anchor)
                                      + URLEncoder.encode(resource.substring(
                                              anchor, resource.length()),
                                              "UTF8"));
                } else if (anchor == 0) {
                    return ((URIReference) xmlBaseURI.peek())
                            .relative(URLEncoder.encode(resource, "UTF8"));
                } else {
                    throw ie;
                }
            } catch (UnsupportedEncodingException ue) {
                throw ie;
            }
        }
    }

    // Push element into _elementstack
    public void pushElement(Element element) {
        elementStack.push(element);
        if ((element != null) && (element.getXmlLang() != null)) {
            langStack.push(element.getXmlLang());
        }
    }

    // Pop up an element
    protected Element popElement() {
        if (elementStack.isEmpty()) {
            return null;
        }
        Element element = (Element) elementStack.pop();
        if ((element != null) && (element.getXmlLang() != null)) {
            langStack.pop();
        }
        return element;
    }

    // Peek the top element in _elementstack
    protected Element peekElement() {
        if (elementStack.isEmpty()) {
            return null;
        }
        return (Element) elementStack.peek();
    }

    // Peek the top language in langStack
    private String peekXmlLang() {
        if (!langStack.isEmpty())
            return (String) langStack.peek();
        return null;
    }

    /**
     * Export a statement
     * 
     * @param subject
     *            the subject of a statement
     * @param predicate
     *            the predicate of a statement
     * @param object
     *            the object of a statement
     */
    private void exportStatement(RDFResourceElement subject,
            PropertyElement predicate, RDFValue object) {
        predicate.setEmpty(false);
        exportStatement(subject, predicate.getUriReference(), object);
        if (predicate.getReificationURI() != null) {
            exportReificationStatement(predicate.getReificationURI(), subject,
                    predicate.getUriReference(), object);
        }
    }

    /**
     * Export a statement
     * 
     * @param subject
     *            the subject of a statement
     * @param predicate
     *            the predicate of a statement
     * @param object
     *            the object of a statement
     */
    private void exportStatement(RDFResourceElement subject,
            URIReference predicate, RDFValue object) {
        /* check whether the statement to be exported is valid */
        if (!checkStatementValid(subject, predicate, object)) {
            throw new ParserException("The statement "
                                      + new RDFTriple(subject, predicate,
                                              object) + " is not valid.");
        }
        statementHandler.exportStatement(subject, predicate, object);
    }

    /**
     * Export the reification.
     * 
     * @param reifyNode
     *            the reification node
     * @param subject
     *            the subject of the triple
     * @param predicate
     *            the predicate of the triple
     * @param object
     *            the object of the triple
     */
    private void exportReificationStatement(RDFResourceElement reifyNode,
            RDFResourceElement subject, URIReference predicate, RDFValue object) {
        statementHandler.exportReification(reifyNode, subject, predicate,
                object);
        if (!statementHandler.isSeparateExport()) {
            exportStatement(reifyNode, RDFSURIConstant.P_TYPE_URI,
                    RDFSURIConstant.C_STATEMENT_URI);
            exportStatement(reifyNode, RDFSURIConstant.P_SUBJECT_URI, subject);
            exportStatement(reifyNode, RDFSURIConstant.P_PREDICATE_URI,
                    predicate);
            exportStatement(reifyNode, RDFSURIConstant.P_OBJECT_URI, object);
        }
    }

    /**
     * Handle xml:base attribute. xml:base are devided into two parts: namespace
     * and local name.
     * 
     * @param xmlBase
     *            the xml base
     */
    public void pushXmlBase(String xmlBase) {

        try {
            URI baseURL = new URI(xmlBase);
            String fragment = baseURL.getFragment();
            if (fragment == null) {
                xmlBaseURI.push(new URIReference(baseURL));
            } else {
                xmlBaseURI.push(new URIReference(xmlBase.substring(0,
                        xmlBase.length() - (fragment.length() + 1))));
            }
        } catch (URISyntaxException murle) {
            throw new ParserException(murle);
        }
    }

    /**
     * Pop up XML base
     * 
     */
    public void popXmlBase() {
        xmlBaseURI.pop();
    }

    /**
     * Handle the error message.
     * 
     * @param msg
     *            the message
     */
    protected void error(String msg) {
        if (errorHandler != null) {
            errorHandler.error(msg);
        } else {
            System.err.println(msg);
            throw new ParserException(msg);
        }
    }

    public void setLocator(Locator locator) {
        this.locator = locator;
    }

    /* NOTE: rdf:bagID, rdf:aboutEach, rdf:aboutEachPrefix are omitted */
    private final static String[] FORBIDDEN_RDF_TYPE_KEYWORD = new String[] {
            RDF.S_RDF, RDF.S_ID, RDF.S_ABOUT, RDF.S_PARSETYPE, RDF.S_RESOURCE,
            RDF.S_NODEID, RDF.S_LI, RDF.P_ABOUTEACH, RDF.p_ABOUTEACHPREFIX,
            RDF.P_BAGID };

    /* NOTE: rdf:bagID, rdf:aboutEach, rdf:aboutEachPrefix are omitted */
    private static final String[] FORBIDDEN_RDF_PROPERTY_KEYWORD = new String[] {
            RDF.S_DESCRIPTION, RDF.S_RDF, RDF.S_ID, RDF.S_ABOUT,
            RDF.S_PARSETYPE, RDF.S_RESOURCE, RDF.S_NODEID, RDF.S_LI,
            RDF.P_ABOUTEACH, RDF.p_ABOUTEACHPREFIX, RDF.P_BAGID };

    /**
     * Check the validity of a statement according to W3C specification
     * 
     * @param subject
     * @param predicate
     * @param object
     * @return true if the statement is valid
     */
    private boolean checkStatementValid(RDFResourceElement subject,
            URIReference predicate, RDFValue object) {
        // forbidden as a node element name
        // rdf:RDF rdf:ID rdf:about rdf:bagID rdf:parseType
        // rdf:resource rdf:nodeID rdf:li rdf:aboutEach rdf:aboutEachPrefix
        if (predicate.getFullURI().equals(RDF.NAMESPACE + RDF.P_TYPE)
            && object instanceof URIReference) {
            URIReference refObject = (URIReference) object;
            for (int i = 0; i < FORBIDDEN_RDF_TYPE_KEYWORD.length; i++) {
                if (refObject.getFullURI().equals(
                        RDF.NAMESPACE + FORBIDDEN_RDF_TYPE_KEYWORD[i]))
                    return false;
            }
        }

        // forbidden as a property element name
        // rdf:Description rdf:RDF rdf:ID rdf:about rdf:bagID
        // rdf:parseType rdf:resource rdf:nodeID rdf:aboudEach
        // rdf:aboutEachPrefix
        // rdf:li (not allowed as as an attribute)
        for (int i = 0; i < FORBIDDEN_RDF_PROPERTY_KEYWORD.length; i++)
            if (predicate.getFullURI().equals(
                    RDF.NAMESPACE + FORBIDDEN_RDF_PROPERTY_KEYWORD[i])) {
                return false;
            }

        return true;
    }
}