/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.tooling.ui.queryresult.properties;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.ui.celleditor.ExtendedComboBoxCellEditor;
import org.eclipse.emf.common.util.Enumerator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.AdapterFactoryItemDelegator;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.emf.edit.ui.provider.ExtendedImageRegistry;
import org.eclipse.emf.edit.ui.provider.PropertyDescriptor;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.viatra.query.runtime.api.scope.IBaseIndex;
import org.eclipse.viatra.query.runtime.base.api.IndexingLevel;
import org.eclipse.viatra.query.runtime.base.api.NavigationHelper;
import org.eclipse.viatra.query.runtime.emf.EMFBaseIndexWrapper;
import org.eclipse.viatra.query.runtime.emf.types.EClassTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EDataTypeInSlotsKey;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PParameter;
import org.eclipse.viatra.query.tooling.ui.queryresult.QueryResultTreeMatcher;
import org.eclipse.viatra.query.tooling.ui.queryresult.util.QueryResultViewUtil;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@Accessors
@SuppressWarnings("all")
public class MatchParameterPropertyDescriptor implements IPropertyDescriptor {
  private String category = "Filters";

  private final PParameter parameter;

  private final QueryResultTreeMatcher<?> matcher;

  private final AdapterFactory adapterFactory = QueryResultViewUtil.getGenericAdapterFactory();

  private final AdapterFactoryItemDelegator adapterFactoryItemDelegator = new AdapterFactoryItemDelegator(this.adapterFactory);

  private final AdapterFactoryLabelProvider adapterFactoryLabelProvider = new AdapterFactoryLabelProvider(this.adapterFactory);

  /**
   * delegator handles EList as well
   */
  private final LabelProvider labelProvider = new LabelProvider() {
    @Override
    public String getText(final Object element) {
      return MatchParameterPropertyDescriptor.this.adapterFactoryItemDelegator.getText(element);
    }

    @Override
    public Image getImage(final Object element) {
      return ExtendedImageRegistry.INSTANCE.getImage(MatchParameterPropertyDescriptor.this.adapterFactoryItemDelegator.getImage(element));
    }
  };

  @Override
  public CellEditor createPropertyEditor(final Composite parent) {
    final IInputKey typeKey = this.parameter.getDeclaredUnaryType();
    if ((typeKey != null)) {
      final CellEditor result = this.prepareEditorForDeclaredType(parent, typeKey);
      return result;
    } else {
      return this.prepareEditorForUnknownType(parent);
    }
  }

  protected CellEditor prepareEditorForDeclaredType(final Composite parent, final IInputKey typeKey) {
    CellEditor _switchResult = null;
    boolean _matched = false;
    if (typeKey instanceof JavaTransitiveInstancesKey) {
      _matched=true;
      return this.prepareEditorForUnknownType(parent);
    }
    if (!_matched) {
      if (typeKey instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        ExtendedComboBoxCellEditor _xblockexpression = null;
        {
          final EClass eClass = ((EClassTransitiveInstancesKey)typeKey).getEmfKey();
          final ArrayList<EObject> choiceOfValues = this.getChoiceOfValues(eClass);
          ArrayList<Object> _arrayList = new ArrayList<Object>(choiceOfValues);
          _xblockexpression = new ExtendedComboBoxCellEditor(parent, _arrayList, 
            this.labelProvider, 
            true);
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (typeKey instanceof EDataTypeInSlotsKey) {
        _matched=true;
        CellEditor _xblockexpression = null;
        {
          final EDataType dataType = ((EDataTypeInSlotsKey)typeKey).getEmfKey();
          CellEditor _xifexpression = null;
          if ((dataType instanceof EEnum)) {
            ExtendedComboBoxCellEditor _xblockexpression_1 = null;
            {
              final Function1<EEnumLiteral, Enumerator> _function = (EEnumLiteral it) -> {
                return it.getInstance();
              };
              final List<Enumerator> choiceOfValues = ListExtensions.<EEnumLiteral, Enumerator>map(((EEnum)dataType).getELiterals(), _function);
              ArrayList<Object> _arrayList = new ArrayList<Object>(choiceOfValues);
              _xblockexpression_1 = new ExtendedComboBoxCellEditor(parent, _arrayList, 
                this.labelProvider, 
                false);
            }
            _xifexpression = _xblockexpression_1;
          } else {
            CellEditor _xifexpression_1 = null;
            if ((Objects.equals(dataType.getInstanceClass(), Boolean.class) || Objects.equals(dataType.getInstanceClass(), Boolean.TYPE))) {
              List<Boolean> _asList = Arrays.<Boolean>asList(new Boolean[] { Boolean.FALSE, Boolean.TRUE });
              _xifexpression_1 = new ExtendedComboBoxCellEditor(parent, _asList, 
                this.labelProvider, 
                true);
            } else {
              _xifexpression_1 = new PropertyDescriptor.EDataTypeCellEditor(dataType, parent);
            }
            _xifexpression = _xifexpression_1;
          }
          _xblockexpression = _xifexpression;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      ExtendedComboBoxCellEditor _xifexpression = null;
      String _typeName = this.parameter.getTypeName();
      String _name = Boolean.class.getName();
      boolean _equals = Objects.equals(_typeName, _name);
      if (_equals) {
        List<Boolean> _asList = Arrays.<Boolean>asList(new Boolean[] { Boolean.FALSE, Boolean.TRUE });
        _xifexpression = new ExtendedComboBoxCellEditor(parent, _asList, 
          this.labelProvider, 
          true);
      } else {
        _xifexpression = null;
      }
      _switchResult = _xifexpression;
    }
    final CellEditor result = _switchResult;
    return result;
  }

  /**
   * we don't know the type, so we just provide a list of possible values
   */
  protected ExtendedComboBoxCellEditor prepareEditorForUnknownType(final Composite parent) {
    final Set<Object> choiceOfValues = this.matcher.getMatcher().getAllValues(this.parameter.getName());
    ArrayList<Object> _arrayList = new ArrayList<Object>(choiceOfValues);
    final ExtendedComboBoxCellEditor editor = new ExtendedComboBoxCellEditor(parent, _arrayList, 
      this.labelProvider, 
      false);
    return editor;
  }

  protected ArrayList<EObject> getChoiceOfValues(final EClass eClass) {
    final ArrayList<EObject> choiceOfValues = CollectionLiterals.<EObject>newArrayList();
    IBaseIndex _baseIndex = this.matcher.getMatcher().getEngine().getBaseIndex();
    final EMFBaseIndexWrapper emfBaseIndex = ((EMFBaseIndexWrapper) _baseIndex);
    final NavigationHelper navigationHelper = emfBaseIndex.getNavigationHelper();
    if ((navigationHelper.isInWildcardMode() || Objects.equals(navigationHelper.getIndexingLevel(eClass), IndexingLevel.FULL))) {
      final Set<EObject> allInstances = navigationHelper.getAllInstances(eClass);
      Iterables.<EObject>addAll(choiceOfValues, allInstances);
    } else {
      Iterable<EObject> _filter = Iterables.<EObject>filter(this.matcher.getMatcher().getAllValues(this.parameter.getName()), EObject.class);
      Iterables.<EObject>addAll(choiceOfValues, _filter);
    }
    return choiceOfValues;
  }

  protected ArrayList<Object> getChoiceOfValues(final EDataType eDataType) {
    final ArrayList<Object> choiceOfValues = CollectionLiterals.<Object>newArrayList();
    IBaseIndex _baseIndex = this.matcher.getMatcher().getEngine().getBaseIndex();
    final EMFBaseIndexWrapper emfBaseIndex = ((EMFBaseIndexWrapper) _baseIndex);
    final NavigationHelper navigationHelper = emfBaseIndex.getNavigationHelper();
    if ((navigationHelper.isInWildcardMode() || Objects.equals(navigationHelper.getIndexingLevel(eDataType), IndexingLevel.FULL))) {
      final Set<Object> allInstances = navigationHelper.getDataTypeInstances(eDataType);
      Iterables.<Object>addAll(choiceOfValues, allInstances);
    } else {
      Set<Object> _allValues = this.matcher.getMatcher().getAllValues(this.parameter.getName());
      Iterables.<Object>addAll(choiceOfValues, _allValues);
    }
    return choiceOfValues;
  }

  @Override
  public String getCategory() {
    return this.category;
  }

  @Override
  public String getDescription() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Filter for parameter ");
    String _name = this.parameter.getName();
    _builder.append(_name);
    _builder.append(" with type ");
    IInputKey _declaredUnaryType = this.parameter.getDeclaredUnaryType();
    _builder.append(_declaredUnaryType);
    return _builder.toString();
  }

  @Override
  public String getDisplayName() {
    return this.parameter.getName();
  }

  @Override
  public String[] getFilterFlags() {
    return null;
  }

  @Override
  public Object getHelpContextIds() {
    return null;
  }

  @Override
  public Object getId() {
    return this.parameter;
  }

  @Override
  public ILabelProvider getLabelProvider() {
    return this.labelProvider;
  }

  @Override
  public boolean isCompatibleWith(final IPropertyDescriptor anotherProperty) {
    return false;
  }

  public MatchParameterPropertyDescriptor(final PParameter parameter, final QueryResultTreeMatcher<?> matcher) {
    super();
    this.parameter = parameter;
    this.matcher = matcher;
  }

  public void setCategory(final String category) {
    this.category = category;
  }

  @Pure
  public PParameter getParameter() {
    return this.parameter;
  }

  @Pure
  public QueryResultTreeMatcher<?> getMatcher() {
    return this.matcher;
  }

  @Pure
  public AdapterFactory getAdapterFactory() {
    return this.adapterFactory;
  }

  @Pure
  public AdapterFactoryItemDelegator getAdapterFactoryItemDelegator() {
    return this.adapterFactoryItemDelegator;
  }

  @Pure
  public AdapterFactoryLabelProvider getAdapterFactoryLabelProvider() {
    return this.adapterFactoryLabelProvider;
  }
}
