//
// This file is released under the terms of the NASA Open Source Agreement (NOSA)
// version 1.3 as detailed in the LICENSE file which accompanies this software.
//
//////////////////////////////////////////////////////////////////////

#include "glviewer.H"
#include <FL/fl_ask.H>
#include <cstring>

/*##############################################################################
#                                                                              #
#                             GL_VIEWER Constructor                            #
#                                                                              #
##############################################################################*/

GL_VIEWER::GL_VIEWER(int x,int y,int w,int h,const char *l) : Fl_Gl_Window(x,y,w,h,l)
{
#ifdef __APPLE__
#if FL_API_VERSION >= 10304
    Fl::use_high_res_GL( true );
#endif
#endif

    int i;
    char tempname[80];

    // Rotation and translation data

    vAng   = 0.;
    hAng   = 0.;
    zAng   = 0.;
    xshift = 0.;
    yshift = 0.;

    size   = 1.;

    OldY = NewY = 0;
    OldX = NewX = 0;

    OldScale = NewScale = 1.;

    NewTransX = NewTransY = 0.;
    OldTransX = OldTransY = 0.;

    FAST_DRAW_ON = 0;

    ViewSize = 1.;

    ZoomScale = 1.;

    Brightness = 0.5;

    ForceView_ = 0;

    BoxSize_ = 10.;

    // Aerothermal database data

    NumberOfMachs   = 0;
    NumberOfBetas    = 0;
    NumberOfAlphas  = 0;

    Nx = NULL;
    Ny = NULL;
    Nz = NULL;

    Nodal_Nx = NULL;
    Nodal_Ny = NULL;
    Nodal_Nz = NULL;

    Cp                       = NULL;
    MachList                 = NULL;
    BetaList                 = NULL;
    AlphaList                = NULL;

    LastFunction = NULL;

    // Data for tris and nodes

    NumberOfTris  = 0;
    NumberOfNodes = 0;
    NumberOfEdges = 0;

    TriList  = NULL;
    NodeList = NULL;
    EdgeList = NULL;

    // Initialize quaternion

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);
    trackball(OldQuat, 0.0, 0.0, 0.0, 0.0);

    // What to draw

    DrawSmoothShadeIsOn                  = 0;
    DrawLegendIsOn                       = 0;
    DrawLabelIsOn                        = 0;
    DrawWithWhiteBackgroundIsOn          = 1;

    DrawWireFrameIsOn                    = 0;
    DrawShadedIsOn                       = 1;
    DrawCpIsOn                           = 0;
    DrawCpSteadyIsOn                     = 0;
    DrawCpUnsteadyIsOn                   = 0;

    DrawComGeomTagsIsOn           = 0;
    DrawComGeomTagsShadedIsOn     = 1;
    DrawComGeomTagsBordersIsOn    = 0;

    DrawCGIsOn                           = 0;
    DrawCGLabelIsOn                      = 0;
    
    DrawAxesIsOn                         = 0;

    DrawFlatShadedIsOn = 0;
    
    DrawWakesToInfinityIsOn = 0;

     // Mesh Min/Max

    XMin = 1.;
    XMax = 1.;

    YMin = 1.;
    YMax = 1.;

    ZMin = 1.;
    ZMax = 1.;

    // Font stuff

    WriteFontFile();

    glfLoadFont( (char *) "cbviewer_font.glf");\

    sprintf(tempname,"cbviewer_font.glf");

    ::remove(tempname);

    UserTouchedEdgeMach   = 0;
    UserTouchedBeta  = 0;
    UserTouchedAlpha  = 0;

    UserChoiceMach = 1;
    UserChoiceBeta = 1;
    UserChoiceAlpha = 1;

    CurrentChoiceMach = 1;
    CurrentChoiceBeta = 1;
    CurrentChoiceAlpha = 1;

    ByteSwapForADB = 0;

    UseEnglishUnits = 0;

    for ( i = 0 ; i <= 10000 ; i++ ) {

       SurfaceID_R[i] = 1.00;
       SurfaceID_G[i] = 1.00;
       SurfaceID_B[i] = 1.00;

    }

    // Body

    SurfaceID_R[BODY] = 0.00;
    SurfaceID_G[BODY] = 0.55;
    SurfaceID_B[BODY] = 0.00;

    // Wing

    SurfaceID_R[WING] = 0.55;
    SurfaceID_G[WING] = 0.55;
    SurfaceID_B[WING] = 0.00;
    
    // Propulsion stuff
    
    NumberOfPropulsionElements = 0;
    PropulsionElement = NULL;

    // Coarse grid data
    
    NumberOfMeshLevels = 0;
    
    CoarseMeshLevelSelected = 1;
    
    // Control surfaces
    
    DrawControlSurfacesIsOn = 0;
    
    DrawControlSurfacesDeflectedIsOn = 0;
    
    // Load in the first solution to begin with
    
    UserSelectedSolutionCase_ = 1;
    
    // User sets the plot limits
    
    UserSetPlotLimits = 0;
    
    TimeAccurate_ = 0;
    
    NumberOfTrailingVortexEdges_ = 0;
    
    DrawBEAM3DFEMIsOn = 0;
    
    XWake_ = NULL;
    YWake_ = NULL;
    ZWake_ = NULL;
    SWake_ = NULL;
    WingWake_ = NULL;
    
    DrawWakeLinesIsOn = 1;
    
    DrawWakePointsIsOn = 0;    
    
    DrawWakesColored_ = 0;
    
    NumberOfLineContourLevels = 20;

    DrawPerTriFunctionIsOn = 1;

    DrawSmoothFunctionsIsOn = 0;
        
    DrawLineContoursIsOn = 0;
    
}

/*##############################################################################
#                                                                              #
#                               GL_VIEWER destructor                           #
#                                                                              #
##############################################################################*/

GL_VIEWER::~GL_VIEWER(void)
{

    int i;

    // Rotation and translation data

    vAng   = 0.;
    hAng   = 0.;
    zAng   = 0.;
    xshift = 0.;
    yshift = 0.;

    size   = 1.;

    // Aerothermal database data

    NumberOfMachs   = 0;
    NumberOfBetas   = 0;
    NumberOfAlphas  = 0;

    if ( Cp != NULL ) delete [] Cp;
 
    // Data for tris and nodes

    NumberOfTris  = 0;
    NumberOfNodes = 0;

    if ( TriList  != NULL ) delete [] TriList;
    if ( NodeList != NULL ) delete [] NodeList;

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER LoadInitialData                       #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadInitialData(char *name)
{

    char CommandLine[2000], file_name_w_ext[2000];
    FILE *adb_file;
    
    // Save the file name

    sprintf(file_name,"%s",name);

    char* pathsep = NULL;
    pathsep = strrchr( file_name, '/' );
    if ( !pathsep )
    {
        pathsep = strrchr( file_name, '\\' );
    }

    if ( pathsep )
    {
        sprintf( file_name_no_path, "%s", pathsep + 1 );
        sprintf( path, "%.*s", pathsep - file_name, file_name );
    }
    else
    {
        sprintf( file_name_no_path, "%s", file_name );
        sprintf( path, "." );
    }

    // Determine if an adb file exists. Add the .adb extension if not already present.
    
    if (strstr(file_name,".adb") )    
        sprintf(file_name_w_ext,"%s",file_name);
    else
        sprintf(file_name_w_ext,"%s.adb",file_name);

    if ( (adb_file = fopen(file_name_w_ext,"rb")) != NULL ) {
     
       fclose(adb_file);

       // Load in the Mesh

       LoadMeshData();
       
       CreateTriEdges();
 
       CalculateSurfaceNormals(1);
           
       // Load ADB Case list
       
       LoadSolutionCaseList();
       
       // Load in the solution data
       
       LoadSolutionData();
       
       // Load in BEAM3D FEM data
       
       LoadBeam3DFEMData();
       
       // Load in ADJOINT and gradient data
       
      // LoadAdjointandGradients();
       
    }
    
    // Load in the degen geometry data if it exists
    
    else {
     
       printf("No %s.adb file found! \n",file_name);
       printf("Please run vspaero first. \n");
       printf("You can run vspaero with the -geom option to only write out a viewable geometry. \n");
       fflush(NULL);
       exit(1);  
 
    }

    // Load in the ComGeom II tags
    
    PanelComGeomTagsBrowser_Update();

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER CalculateSurfaceNormals               #
#                                                                              #
##############################################################################*/

void GL_VIEWER::CalculateSurfaceNormals(int Case)
{

    int i, nod1, nod2, nod3;
    float vec1[3], vec2[3], normal[3], dot, *TotalArea;

    if ( Case == 1 ) {

       Nx = new float[NumberOfTris + 1];
       Ny = new float[NumberOfTris + 1];
       Nz = new float[NumberOfTris + 1];

       Nodal_Nx = new float[NumberOfNodes + 1];
       Nodal_Ny = new float[NumberOfNodes + 1];
       Nodal_Nz = new float[NumberOfNodes + 1];

    }

    TotalArea = new float[NumberOfNodes + 1];

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       Nodal_Nx[i] = 0.;
       Nodal_Ny[i] = 0.;
       Nodal_Nz[i] = 0.;

       TotalArea[i] = 0.;

	}

    for ( i = 1 ; i <= NumberOfTris ; i++ ) {

       nod1 = TriList[i].node1;
       nod2 = TriList[i].node2;
       nod3 = TriList[i].node3;

       vec1[0] = NodeList[nod2].x - NodeList[nod1].x;
       vec1[1] = NodeList[nod2].y - NodeList[nod1].y;
       vec1[2] = NodeList[nod2].z - NodeList[nod1].z;

       vec2[0] = NodeList[nod3].x - NodeList[nod1].x;
       vec2[1] = NodeList[nod3].y - NodeList[nod1].y;
       vec2[2] = NodeList[nod3].z - NodeList[nod1].z;

       vector_cross(vec1,vec2,normal);

       dot = sqrt(vector_dot(normal,normal));

       Nx[i] = normal[0]/dot;
       Ny[i] = normal[1]/dot;
       Nz[i] = normal[2]/dot;

       Nodal_Nx[nod1] += normal[0];
       Nodal_Ny[nod1] += normal[1];
       Nodal_Nz[nod1] += normal[2];

       Nodal_Nx[nod2] += normal[0];
       Nodal_Ny[nod2] += normal[1];
       Nodal_Nz[nod2] += normal[2];

       Nodal_Nx[nod3] += normal[0];
       Nodal_Ny[nod3] += normal[1];
       Nodal_Nz[nod3] += normal[2];

       TotalArea[nod1] += dot;
       TotalArea[nod2] += dot;
       TotalArea[nod3] += dot;

    }

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       Nodal_Nx[i] /= TotalArea[i];
       Nodal_Ny[i] /= TotalArea[i];
       Nodal_Nz[i] /= TotalArea[i];

	}

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       dot = sqrt(Nodal_Nx[i]*Nodal_Nx[i] + Nodal_Ny[i]*Nodal_Ny[i] + Nodal_Nz[i]*Nodal_Nz[i]);

       Nodal_Nx[i] /= dot;
       Nodal_Ny[i] /= dot;
       Nodal_Nz[i] /= dot;

	}

    delete [] TotalArea;

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER LoadMeshData                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadMeshData(void)
{

    char file_name_w_ext[2000], DumChar[1000], GridName[1000];
    int i, j, k, p, DumInt, Level, Edge, NumberOfControlSurfaceNodes;
    int TotNum, i_size, f_size, c_size;
    float DumFloat;
    FILE *adb_file, *madb_file;
    BINARYIO BIO;

    // Sizeof ints and floats

    i_size = sizeof(int);
    f_size = sizeof(float);
    c_size = sizeof(char);

    // Check on endian issues

    if ( ByteSwapForADB ) BIO.TurnByteSwapForReadsOn();

    // Open the aerothermal data base file. Add the .adb extension if not already present.

    if (strstr(file_name,".adb") )    
        sprintf(file_name_w_ext,"%s",file_name);
    else
        sprintf(file_name_w_ext,"%s.adb",file_name);

    if ( (adb_file = fopen(file_name_w_ext,"rb")) == NULL ) {

       printf("Could not open either an adb or madb file... ! \n");fflush(NULL);
                     
       exit(1);
       
    }

    // Read in the default in to check on endianess

    BIO.fread(&DumInt, i_size, 1, adb_file);

    if ( DumInt != -123789456  ) {

       BIO.TurnByteSwapForReadsOn();

       rewind(adb_file);

       BIO.fread(&DumInt, i_size, 1, adb_file);

    }
    
    // Read in model type... VLM or PANEL
    
    BIO.fread(&ModelType, i_size, 1, adb_file);
    
    // Read in symmetry flag
    
    BIO.fread(&SymmetryFlag, i_size, 1, adb_file);    
    
    // Read in unsteady analysis flag
    
    BIO.fread(&TimeAccurate_, i_size, 1, adb_file);        

    // Read in header

    BIO.fread(&NumberOfVortexLoops,        i_size, 1, adb_file);
    BIO.fread(&NumberOfNodes,              i_size, 1, adb_file);
    BIO.fread(&NumberOfTris,               i_size, 1, adb_file);
    BIO.fread(&NumberOfSurfaceVortexEdges, i_size, 1, adb_file);
    
    BIO.fread(&Sref,                f_size, 1, adb_file);
    BIO.fread(&Cref,                f_size, 1, adb_file);
    BIO.fread(&Bref,                f_size, 1, adb_file);
    BIO.fread(&Xcg,                 f_size, 1, adb_file);
    BIO.fread(&Ycg,                 f_size, 1, adb_file);
    BIO.fread(&Zcg,                 f_size, 1, adb_file);
    
    NumberOfMachs = NumberOfAlphas = NumberOfBetas = 1;

    printf("NumberOfNodes:  %d \n",NumberOfNodes);
    printf("NumberOfTris:   %d \n",NumberOfTris);
    printf("Sref:           %f \n",Sref);
    printf("Cref:           %f \n",Cref);
    printf("Bref:           %f \n",Bref);
    printf("Xcg:            %f \n",Xcg);
    printf("Ycg:            %f \n",Ycg);
    printf("Zcg:            %f \n",Zcg);fflush(NULL);

    TotNum = NumberOfMachs * NumberOfBetas * NumberOfAlphas;

    // Create space to store grid and solution

    NodeList = new NODE[NumberOfNodes + 1];

    TriList = new TRI[NumberOfTris + 1];

    Cp         = new float[NumberOfTris + 1];
    CpSteady   = new float[NumberOfTris + 1];
    CpUnsteady = new float[NumberOfTris + 1];
    Gamma      = new float[NumberOfTris + 1];
    
    GammaN       = new double[NumberOfVortexLoops + 1];
    dCp_Unsteady = new double[NumberOfVortexLoops + 1];

    Fx  = new double[NumberOfSurfaceVortexEdges + 1];
    Fy  = new double[NumberOfSurfaceVortexEdges + 1];
    Fz  = new double[NumberOfSurfaceVortexEdges + 1];

    U = new double[NumberOfVortexLoops + 1];
    V = new double[NumberOfVortexLoops + 1];
    W = new double[NumberOfVortexLoops + 1];
    
    TempNodalArray = new float[NumberOfNodes + 1];
    TempTotalArea  = new float[NumberOfNodes + 1];

    MachList = new float[NumberOfMachs + 1];
    BetaList = new float[NumberOfBetas + 1];
    AlphaList = new float[NumberOfAlphas + 1];

    // Read in wing ID flags, names...
 
    fread(&NumberOfWings_, i_size, 1, adb_file);
    
    WingListName_ = new char*[NumberOfWings_ + 1];
    
    WingGroupID_ = new int[NumberOfWings_ + 1];
  
    for ( i = 1 ; i <= NumberOfWings_ ; i++ ) { 
     
       fread(&DumInt, i_size, 1, adb_file);
       
       WingListName_[i] = new char[200];
 
       fread(WingListName_[i], c_size, 100, adb_file);
       
       fread(&(WingGroupID_[i]), i_size, 1, adb_file);
 
    }
    
    // Read in body ID flags, names...

    fread(&NumberOfBodies_, i_size, 1, adb_file);
    
    BodyListName_ = new char*[NumberOfBodies_ + 1];
    
    BodyGroupID_ = new int[NumberOfBodies_ + 1];
     
    for ( i = 1 ; i <= NumberOfBodies_ ; i++ ) { 
     
       fread(&DumInt, i_size, 1, adb_file);
       
       BodyListName_[i] = new char[200];

       fread(BodyListName_[i], c_size, 100, adb_file);
       
       fread(&(BodyGroupID_[i]), i_size, 1, adb_file);

    } 
    
    NumberOfSurfaces_ = NumberOfWings_ + NumberOfBodies_;
    
    // Read in Cart3d ID flags, names...

    fread(&NumberOfCart3dSurfaces_, i_size, 1, adb_file);
 
    Cart3dListName_ = new char*[NumberOfCart3dSurfaces_ + 1];
    
    for ( i = 1 ; i <= NumberOfCart3dSurfaces_ ; i++ ) { 
     
       fread(&DumInt, i_size, 1, adb_file);
       
       Cart3dListName_[i] = new char[200];

       fread(Cart3dListName_[i], c_size, 100, adb_file);

    }     
    
    // End of the header information... now geometry

    // Store the current location in the file

    fgetpos(adb_file, &StartOfWallTemperatureData);

    // Load in the geometry and surface information

    for ( i = 1 ; i <= NumberOfTris ; i++ ) {

       // Geometry

       BIO.fread(&(TriList[i].node1),        i_size, 1, adb_file);
       BIO.fread(&(TriList[i].node2),        i_size, 1, adb_file);
       BIO.fread(&(TriList[i].node3),        i_size, 1, adb_file);
      
       BIO.fread(&(TriList[i].surface_type), i_size, 1, adb_file);
       BIO.fread(&(TriList[i].surface_id),   i_size, 1, adb_file);
       BIO.fread(&(TriList[i].area),         f_size, 1, adb_file);

    }

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       BIO.fread(&(NodeList[i].x), f_size, 1, adb_file);
       BIO.fread(&(NodeList[i].y), f_size, 1, adb_file);
       BIO.fread(&(NodeList[i].z), f_size, 1, adb_file);

    }
    
    // Find Min/Max of geometry

    FindMeshMinMax();
    
    // Read in any propulsion data
    
    BIO.fread(&(NumberOfPropulsionElements), i_size, 1, adb_file); 
 
    PropulsionElement = new PROPULSION_ELEMENT[NumberOfPropulsionElements + 1];
    
    for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {
     
       PropulsionElement[i].Rotor.Read_Binary_STP_Data(adb_file);
    
    }
    
    DrawPropulsionElementsIsOn = 0;    
    
    // Read in any coarse mesh edge data
    
    BIO.fread(&(NumberOfMeshLevels), i_size, 1, adb_file); 
 
    CoarseNodeList = new NODE*[NumberOfMeshLevels + 1];
    CoarseEdgeList = new EDGE*[NumberOfMeshLevels + 1];
    
    NumberOfCourseNodesForLevel = new int[NumberOfMeshLevels + 1];
    NumberOfCourseEdgesForLevel = new int[NumberOfMeshLevels + 1];
    
    for ( Level = 1 ; Level <= NumberOfMeshLevels ; Level++ ) {
     
       BIO.fread(&(NumberOfCourseNodesForLevel[Level]), i_size, 1, adb_file);    
       BIO.fread(&(NumberOfCourseEdgesForLevel[Level]), i_size, 1, adb_file);          

       CoarseNodeList[Level] = new NODE[NumberOfCourseNodesForLevel[Level] + 1];
       CoarseEdgeList[Level] = new EDGE[NumberOfCourseEdgesForLevel[Level] + 1];

       for ( i = 1 ; i <= NumberOfCourseNodesForLevel[Level] ; i++ ) {
 
          BIO.fread(&(CoarseNodeList[Level][i].x), f_size, 1, adb_file);       
          BIO.fread(&(CoarseNodeList[Level][i].y), f_size, 1, adb_file);  
          BIO.fread(&(CoarseNodeList[Level][i].z), f_size, 1, adb_file);       
          
          CoarseNodeList[Level][i].x -= GeometryXShift;
          CoarseNodeList[Level][i].y -= GeometryYShift;
          CoarseNodeList[Level][i].z -= GeometryZShift;
          
       }
         
       for ( i = 1 ; i <= NumberOfCourseEdgesForLevel[Level] ; i++ ) {
 
          BIO.fread(&(CoarseEdgeList[Level][i].SurfaceID), i_size, 1, adb_file);   
          
          CoarseEdgeList[Level][i].IsBoundaryEdge = 0;
          
          if ( CoarseEdgeList[Level][i].SurfaceID < 0 ) {
             
             CoarseEdgeList[Level][i].SurfaceID = -CoarseEdgeList[Level][i].SurfaceID;
             
             CoarseEdgeList[Level][i].IsBoundaryEdge = 1;    
             
          }
        
          BIO.fread(&(CoarseEdgeList[Level][i].node1), i_size, 1, adb_file);       
          BIO.fread(&(CoarseEdgeList[Level][i].node2), i_size, 1, adb_file);       
          
          CoarseEdgeList[Level][i].IsKuttaEdge = 0;
          
       }
       
       for ( i = 1 ; i <= NumberOfCourseEdgesForLevel[Level] ; i++ ) {

          CoarseNodeList[Level][CoarseEdgeList[Level][i].node1].SurfID = CoarseEdgeList[Level][i].SurfaceID;
          CoarseNodeList[Level][CoarseEdgeList[Level][i].node2].SurfID = CoarseEdgeList[Level][i].SurfaceID;    

       }       
    
    }    
    
    // Read in the kutta edge data
    
    Level = 1;
    
    BIO.fread(&(NumberOfKuttaEdges), i_size, 1, adb_file);       

    for ( i = 1 ; i <= NumberOfKuttaEdges; i++ ) {
       
       BIO.fread(&(Edge), i_size, 1, adb_file);      
       
       CoarseEdgeList[Level][Edge].IsKuttaEdge = 1;
        
    }
    
    // Read in the kutta node data
    
    Level = 1;
    
    BIO.fread(&(NumberOfKuttaNodes), i_size, 1, adb_file);       

    for ( i = 1 ; i <= NumberOfKuttaNodes; i++ ) {
       
       BIO.fread(&(DumInt), i_size, 1, adb_file);      

    }    
    
    // Read in any control surfaces
    
    BIO.fread(&(NumberOfControlSurfaces), i_size, 1, adb_file);       

    ControlSurface = new CONTROL_SURFACE[NumberOfControlSurfaces + 1];
    
    for ( i = 1 ; i <= NumberOfControlSurfaces ; i++ ) {
       
       BIO.fread(&(NumberOfControlSurfaceNodes), i_size, 1, adb_file);       
       
       ControlSurface[i].NumberOfNodes = NumberOfControlSurfaceNodes;
       
       ControlSurface[i].NodeList = new float*[NumberOfControlSurfaceNodes + 1];

       for ( j = 1 ; j <= NumberOfControlSurfaceNodes ; j++ ) {
          
          ControlSurface[i].NodeList[j] = new float[3];
          
       }
       
       for ( j = 1 ; j <= NumberOfControlSurfaceNodes ; j++ ) {

          BIO.fread(&(ControlSurface[i].NodeList[j][0]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][0] -= GeometryXShift;      
          BIO.fread(&(ControlSurface[i].NodeList[j][1]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][1] -= GeometryYShift;       
          BIO.fread(&(ControlSurface[i].NodeList[j][2]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][2] -= GeometryZShift;  
          
       }          
       
       // Hinge nodes and vector
       
       BIO.fread(&(ControlSurface[i].HingeNode1[0]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[0] -= GeometryXShift;      
       BIO.fread(&(ControlSurface[i].HingeNode1[1]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[1] -= GeometryYShift;       
       BIO.fread(&(ControlSurface[i].HingeNode1[2]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[2] -= GeometryZShift;            
                       
       BIO.fread(&(ControlSurface[i].HingeNode2[0]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[0] -= GeometryXShift;      
       BIO.fread(&(ControlSurface[i].HingeNode2[1]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[1] -= GeometryYShift;       
       BIO.fread(&(ControlSurface[i].HingeNode2[2]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[2] -= GeometryZShift;  
       
       BIO.fread(&(ControlSurface[i].HingeVec[0]), f_size, 1, adb_file);   
       BIO.fread(&(ControlSurface[i].HingeVec[1]), f_size, 1, adb_file);    
       BIO.fread(&(ControlSurface[i].HingeVec[2]), f_size, 1, adb_file);
              
       // Affected loops
       
       BIO.fread(&(ControlSurface[i].NumberOfLoops), i_size, 1, adb_file);
       
       ControlSurface[i].LoopList = new int[ControlSurface[i].NumberOfLoops + 1];
       
       for ( p = 1 ; p <= ControlSurface[i].NumberOfLoops ; p++ ) {
          
          BIO.fread(&(ControlSurface[i].LoopList[p]), i_size, 1, adb_file);
          
       }          
       
       // Zero out control surface deflection
       
       ControlSurface[i].DeflectionAngle = 0.;
                            
    }
    
    // Mark all the loops on a control surface
    
    ControlSurfaceLoop = new int[NumberOfTris + 1];
    
    for ( j = 1 ; j <= NumberOfTris ; j++ ) {
        
       ControlSurfaceLoop[j] = 0;
        
    }
    
    printf("NumberOfControlSurfaces: %d \n",NumberOfControlSurfaces);
    
    for ( i = 1 ; i <= NumberOfControlSurfaces ; i++ ) {
       
       for ( j = 1 ; j <= ControlSurface[i].NumberOfLoops ; j++ ) {
          
          ControlSurfaceLoop[ControlSurface[i].LoopList[j]] = i;

       }       
       
    } 
    
    // End of the geometry section         

    // Close the adb file

    fclose(adb_file);

    // Force the view box to a fixed value

    if ( ForceView_ ) FixViewingBox(-BoxSize_, BoxSize_,
                                    -BoxSize_, BoxSize_,
                                    -BoxSize_, BoxSize_);

    // Zero out arrays

    zero_float_array(TempNodalArray,          NumberOfNodes);
    zero_float_array(TempTotalArea,           NumberOfNodes);

    // Size additional working arrays and structures

    ComGeom2PanelTag = new int[NumberOfTris + 1];

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER UpdateMeshData                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::UpdateMeshData(FILE *adb_file)
{

    char file_name_w_ext[2000], DumChar[1000], GridName[1000];
    int i, j, k, p, DumInt, Level, Edge, NumberOfControlSurfaceNodes;
    int TotNum, i_size, f_size, c_size;
    float DumFloat;
    BINARYIO BIO;

    // Sizeof ints and floats

    i_size = sizeof(int);
    f_size = sizeof(float);
    c_size = sizeof(char);

    // Load in the geometry and surface information

    for ( i = 1 ; i <= NumberOfTris ; i++ ) {

       // Geometry

       BIO.fread(&(TriList[i].node1),        i_size, 1, adb_file);
       BIO.fread(&(TriList[i].node2),        i_size, 1, adb_file);
       BIO.fread(&(TriList[i].node3),        i_size, 1, adb_file);
       
       BIO.fread(&(TriList[i].surface_type), i_size, 1, adb_file);
       BIO.fread(&(TriList[i].surface_id),   i_size, 1, adb_file);
       BIO.fread(&(TriList[i].area),         f_size, 1, adb_file);

    }

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       BIO.fread(&(NodeList[i].x), f_size, 1, adb_file);
       BIO.fread(&(NodeList[i].y), f_size, 1, adb_file);
       BIO.fread(&(NodeList[i].z), f_size, 1, adb_file);

    }

    // Center geometry

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       NodeList[i].x -= GeometryXShift;
       NodeList[i].y -= GeometryYShift;
       NodeList[i].z -= GeometryZShift;

    }
    
    // Read in any propulsion data
    
    BIO.fread(&(NumberOfPropulsionElements), i_size, 1, adb_file); 

    for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {
     
       PropulsionElement[i].Rotor.Read_Binary_STP_Data(adb_file);
    
    }

    // Read in any coarse mesh edge data
    
    BIO.fread(&(NumberOfMeshLevels), i_size, 1, adb_file); 

    for ( Level = 1 ; Level <= NumberOfMeshLevels ; Level++ ) {
     
       BIO.fread(&(NumberOfCourseNodesForLevel[Level]), i_size, 1, adb_file);    
       BIO.fread(&(NumberOfCourseEdgesForLevel[Level]), i_size, 1, adb_file);          

       for ( i = 1 ; i <= NumberOfCourseNodesForLevel[Level] ; i++ ) {
 
          BIO.fread(&(CoarseNodeList[Level][i].x), f_size, 1, adb_file);       
          BIO.fread(&(CoarseNodeList[Level][i].y), f_size, 1, adb_file);  
          BIO.fread(&(CoarseNodeList[Level][i].z), f_size, 1, adb_file);       
          
          CoarseNodeList[Level][i].x -= GeometryXShift;
          CoarseNodeList[Level][i].y -= GeometryYShift;
          CoarseNodeList[Level][i].z -= GeometryZShift;
          
       }
         
       for ( i = 1 ; i <= NumberOfCourseEdgesForLevel[Level] ; i++ ) {
 
          BIO.fread(&(CoarseEdgeList[Level][i].SurfaceID), i_size, 1, adb_file);   
          
          CoarseEdgeList[Level][i].IsBoundaryEdge = 0;
          
          if ( CoarseEdgeList[Level][i].SurfaceID < 0 ) {
             
             CoarseEdgeList[Level][i].SurfaceID = -CoarseEdgeList[Level][i].SurfaceID;
             
             CoarseEdgeList[Level][i].IsBoundaryEdge = 1;    
             
          }
        
          BIO.fread(&(CoarseEdgeList[Level][i].node1), i_size, 1, adb_file);       
          BIO.fread(&(CoarseEdgeList[Level][i].node2), i_size, 1, adb_file);       
          
          CoarseEdgeList[Level][i].IsKuttaEdge = 0;
          
       }
       
       for ( i = 1 ; i <= NumberOfCourseEdgesForLevel[Level] ; i++ ) {

          CoarseNodeList[Level][CoarseEdgeList[Level][i].node1].SurfID = CoarseEdgeList[Level][i].SurfaceID;
          CoarseNodeList[Level][CoarseEdgeList[Level][i].node2].SurfID = CoarseEdgeList[Level][i].SurfaceID;    

       }       
    
    }    
    
    // Read in the kutta edge data
    
    Level = 1;
    
    BIO.fread(&(NumberOfKuttaEdges), i_size, 1, adb_file);       

    for ( i = 1 ; i <= NumberOfKuttaEdges; i++ ) {
       
       BIO.fread(&(Edge), i_size, 1, adb_file);      
       
       CoarseEdgeList[Level][Edge].IsKuttaEdge = 1;
        
    }
    
    // Read in the kutta node data
    
    Level = 1;
    
    BIO.fread(&(NumberOfKuttaNodes), i_size, 1, adb_file);       

    for ( i = 1 ; i <= NumberOfKuttaNodes; i++ ) {
       
       BIO.fread(&(DumInt), i_size, 1, adb_file);      

    }    
    
    // Read in any control surfaces
    
    BIO.fread(&(NumberOfControlSurfaces), i_size, 1, adb_file);       

    for ( i = 1 ; i <= NumberOfControlSurfaces ; i++ ) {
       
       BIO.fread(&(NumberOfControlSurfaceNodes), i_size, 1, adb_file);       
       
       ControlSurface[i].NumberOfNodes = NumberOfControlSurfaceNodes;

       for ( j = 1 ; j <= NumberOfControlSurfaceNodes ; j++ ) {
          
          ControlSurface[i].NodeList[j] = new float[3];
          
       }
       
       for ( j = 1 ; j <= NumberOfControlSurfaceNodes ; j++ ) {

          BIO.fread(&(ControlSurface[i].NodeList[j][0]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][0] -= GeometryXShift;      
          BIO.fread(&(ControlSurface[i].NodeList[j][1]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][1] -= GeometryYShift;       
          BIO.fread(&(ControlSurface[i].NodeList[j][2]), f_size, 1, adb_file); ControlSurface[i].NodeList[j][2] -= GeometryZShift;  
          
       }          
       
       // Hinge nodes and vector
       
       BIO.fread(&(ControlSurface[i].HingeNode1[0]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[0] -= GeometryXShift;      
       BIO.fread(&(ControlSurface[i].HingeNode1[1]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[1] -= GeometryYShift;       
       BIO.fread(&(ControlSurface[i].HingeNode1[2]), f_size, 1, adb_file); ControlSurface[i].HingeNode1[2] -= GeometryZShift;            
                        
       BIO.fread(&(ControlSurface[i].HingeNode2[0]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[0] -= GeometryXShift;      
       BIO.fread(&(ControlSurface[i].HingeNode2[1]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[1] -= GeometryYShift;       
       BIO.fread(&(ControlSurface[i].HingeNode2[2]), f_size, 1, adb_file); ControlSurface[i].HingeNode2[2] -= GeometryZShift;  
       
       BIO.fread(&(ControlSurface[i].HingeVec[0]), f_size, 1, adb_file);   
       BIO.fread(&(ControlSurface[i].HingeVec[1]), f_size, 1, adb_file);    
       BIO.fread(&(ControlSurface[i].HingeVec[2]), f_size, 1, adb_file);
              
       // Affected loops
       
       BIO.fread(&(ControlSurface[i].NumberOfLoops), i_size, 1, adb_file);
       
       ControlSurface[i].LoopList = new int[ControlSurface[i].NumberOfLoops + 1];
       
       for ( p = 1 ; p <= ControlSurface[i].NumberOfLoops ; p++ ) {
          
          BIO.fread(&(ControlSurface[i].LoopList[p]), i_size, 1, adb_file);
          
       }          
       
       // Zero out control surface deflection
       
       ControlSurface[i].DeflectionAngle = 0.;
                            
    }     
    
    // Mark all the loops on a control surface
    
    ControlSurfaceLoop = new int[NumberOfTris + 1];
    
    for ( j = 1 ; j <= NumberOfTris ; j++ ) {
        
       ControlSurfaceLoop[j] = 0;
        
    }
    
    for ( i = 1 ; i <= NumberOfControlSurfaces ; i++ ) {
       
       for ( j = 1 ; j <= ControlSurface[i].NumberOfLoops ; j++ ) {

          ControlSurfaceLoop[ControlSurface[i].LoopList[j]] = i;

       }       
       
    }          

    // Calculate the surface normals

    CalculateSurfaceNormals(0);
    
}

/*##############################################################################
#                                                                              #
#                        GL_VIEWER LoadSolutionCaseList                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadSolutionCaseList(void)
{
 
    int i;
    char file_name_w_ext[2000], DumChar[2000];
    FILE *adb_file;
    
    // Open the solution case list. Add the .adb extension if not already present.

    if (strstr(file_name,".adb") )    
        sprintf(file_name_w_ext,"%s.cases",file_name);
    else
        sprintf(file_name_w_ext,"%s.adb.cases",file_name);

    if ( (adb_file = fopen(file_name_w_ext,"r")) == NULL ) {

       printf("Could not open the adb case list file... ! \n");fflush(NULL);
                     
       exit(1);

    }       
    
    // Read in the cases until eof
    
    NumberOfADBCases_ = 0;
    
    while ( fgets(DumChar,2000,adb_file) != NULL ) {
       
       NumberOfADBCases_++;
       
    }
    
    rewind(adb_file);

    printf("There are %d ADB cases \n",NumberOfADBCases_);
        
    ADBCaseList_ = new SOLUTION_CASE[NumberOfADBCases_ + 1];

    for ( i = 1 ; i <= NumberOfADBCases_ ; i++ ) {

       fscanf(adb_file,"%f %f %f",
       &(ADBCaseList_[i].Mach),
       &(ADBCaseList_[i].Alpha),
       &(ADBCaseList_[i].Beta));
        
       fgets(ADBCaseList_[i].CommentLine,200,adb_file);

    }

    fclose(adb_file); 
   
}


/*##############################################################################
#                                                                              #
#                              GL_VIEWER MakeMovie                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::MakeMovie(char *FileName)
{

    int i;
    char DumChar[2000], Command[2000], Path[2000];

    sprintf(Path,"%s/MoviePNGFiles/", path);

    // Check for ffmpeg  & delete any old png files
#ifdef WIN32
    system( " ffmpeg -h > temp.txt" );

    // Get size of temp file
    FILE * fp = fopen( "temp.txt", "r" );
    fseek( fp, 0L, SEEK_END );
    size_t sz = ftell( fp );

    fclose( fp );
    system( "del temp.txt" );

    if ( sz == 0 )
    {
        fl_alert( "ERROR: ffmpeg not found in the VSPViewer directory or system PATH" );
        return;
    }

    // rm may not be available on Windows 
    sprintf( Command, "rd /s /q %s", Path );
#else
    // Check for ffmpeg
    if ( system( "which ffmpeg > /dev/null 2>&1" ) )
    {
        // Command not available
        fl_alert( "ERROR: ffmpeg not found in the VSPViewer directory or system PATH" );
        return;
    }

    sprintf( Command, "rm -rf %s", Path );
#endif

    system(Command);

    // Create a sub directory to store all the movie files in
    sprintf( Command, "mkdir %s", Path );

    system(Command);
    

    for ( i = 1 ; i <= NumberOfADBCases_ ; i++ ) {
       
       printf("Saving video frame %d of %d \r",i,NumberOfADBCases_);

       LoadExistingSolutionData(i);

       if ( !UserSetPlotLimits ) FindSolutionMinMax();
       
       sprintf(DumChar,"%s.%d",FileName,i);
       
       Draw();

    // vui->CommentOutput->value("Test");
      
    // vui->CommentOutput->redraw();
      
    // vui->LoadSolutionCase->value(i);
    
    // vui->LoadSolutionCase->redraw();
   
    // vui->MainView->redraw();
       
    // vui->OutputPanel->redraw();
       
       WritePNGFile(Path,DumChar);
       
    }
    
    // Now run ffpmeg
    // 10 fps
    // list of input file names
    // video codec is libx264
    // output file name
    // -y means it will over write any existing (previous) file with that name
    
    printf("Running ffmpeg to create video \n");
    
    sprintf(Command,"ffmpeg -r 10 -i %s%s.%s.%%d.png -vcodec libx264 -y %s.%s.mp4",Path,file_name_no_path,FileName,file_name,FileName);

    system(Command);
      
}


/*##############################################################################
#                                                                              #
#                              GL_VIEWER LoadSolutionData                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadSolutionData(void)
{

    int i;

    // Get the user selected case

    LoadExistingSolutionData(UserSelectedSolutionCase_);

    if ( !UserSetPlotLimits ) FindSolutionMinMax();

}

/*##############################################################################
#                                                                              #
#                      GL_VIEWER LoadExistingSolutionData                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadExistingSolutionData(int Case)
{

    char file_name_w_ext[2000], DumChar[100], GridName[100];
    int i, j, k, m, p, Level;
    int i_size, f_size, c_size, d_size;
    int DumInt, nod1, nod2, nod3, CFDCaseFlag, Edge;
    float FreeStreamPressure, DynamicPressure, Xc, Yc, Zc, Cf;
    float BoundaryLayerThicknessCode, LaminarDelta, TurbulentDelta, DumFloat;
    FILE *adb_file, *madb_file;
    BINARYIO BIO;
    long OffSet;

    // Sizeof ints and floats

    i_size = sizeof(int);
    f_size = sizeof(float);
    d_size = sizeof(double);
    c_size = sizeof(char);

    // Check on endian issues

    if ( ByteSwapForADB ) BIO.TurnByteSwapForReadsOn();

    // Open the aerothermal data base file. Add the .adb extension if not already present.

    if (strstr(file_name,".adb") )    
        sprintf(file_name_w_ext,"%s",file_name);
    else
        sprintf(file_name_w_ext,"%s.adb",file_name);

    if ( (adb_file = fopen(file_name_w_ext,"rb")) == NULL ) {

       printf("Could not open either an adb or madb file... ! \n");fflush(NULL);
                     
       exit(1);

    } 

    // Read in the default text string to check on endianess

    BIO.fread(&DumInt, i_size, 1, adb_file);

    if ( DumInt != -123789456  ) {

       BIO.TurnByteSwapForReadsOn();

       rewind(adb_file);

       BIO.fread(&DumInt, i_size, 1, adb_file);

    }

    // Set the file position to the top of the temperature data

    fsetpos(adb_file, &StartOfWallTemperatureData);

    for ( p = 1 ; p <= Case ; p++ ) {  

       // Reload in the mesh data if this is an unsteady path case

       if ( p == 1 || TimeAccurate_ ) UpdateMeshData(adb_file);

       // Read in the EdgeMach, Q, and Alpha lists
   
       for ( k = 1 ; k <= NumberOfMachs  ; k++ ) BIO.fread(&MachList[k],    f_size, 1, adb_file);
       for ( k = 1 ; k <= NumberOfAlphas ; k++ ) { BIO.fread(&AlphaList[k], f_size, 1, adb_file); AlphaList[k] /= TORAD; };
       for ( k = 1 ; k <= NumberOfBetas  ; k++ ) { BIO.fread(&BetaList[k],  f_size, 1, adb_file); BetaList[k]  /= TORAD; };

       // Read in data set 
   
       BIO.fread(&(CpMinSoln), f_size, 1, adb_file); // Min Cp from solver
       BIO.fread(&(CpMaxSoln), f_size, 1, adb_file); // Max Cp from solver

       // Solution on computational mesh
       
       for ( m = 1 ; m <= NumberOfVortexLoops ; m++ ) {
 
          BIO.fread(&(GammaN[m]),       d_size, 1, adb_file); // Gamma
          BIO.fread(&(dCp_Unsteady[m]), d_size, 1, adb_file); // Unsteady dCP

       }

       // Vortex edge forces on computational mesh
       
       for ( m = 1 ; m <= NumberOfSurfaceVortexEdges ; m++ ) {
 
          BIO.fread(&(Fx[m]), d_size, 1, adb_file); 
          BIO.fread(&(Fy[m]), d_size, 1, adb_file); 
          BIO.fread(&(Fz[m]), d_size, 1, adb_file);

       }
      
       // Solution on computational mesh
      
       for ( m = 1 ; m <= NumberOfVortexLoops ; m++ ) {
   
          BIO.fread(&(U[m]), d_size, 1, adb_file); // U
          BIO.fread(&(V[m]), d_size, 1, adb_file); // V
          BIO.fread(&(W[m]), d_size, 1, adb_file); // W

       }
       
       // Solution on input mesh
                
       for ( m = 1 ; m <= NumberOfTris ; m++ ) {
   
          BIO.fread(&(Cp[m]),         f_size, 1, adb_file); // Total Cp
          BIO.fread(&(CpUnsteady[m]), f_size, 1, adb_file); // Unsteady Cp
          BIO.fread(&(Gamma[m]),      f_size, 1, adb_file); // Vorticity
          
          CpSteady[m] = Cp[m] - CpUnsteady[m]; // Steady state component of Cp

       }
      
       // Delete any old wake data
       
       if ( NumberOfTrailingVortexEdges_ != 0 ) {
          
          for ( i = 1 ; i <= NumberOfTrailingVortexEdges_ ; i++ ) {
             
             if ( XWake_[i] != NULL ) delete XWake_[i];
             if ( YWake_[i] != NULL ) delete YWake_[i];
             if ( ZWake_[i] != NULL ) delete ZWake_[i];
             
          }
          
          if ( XWake_ != NULL ) delete XWake_;
          if ( YWake_ != NULL ) delete YWake_;
          if ( ZWake_ != NULL ) delete ZWake_;
          if ( SWake_ != NULL ) delete SWake_;
          
          if ( WingWake_ != NULL ) delete [] WingWake_;

       }
       
       // Read in the wake location data
       
       BIO.fread(&(NumberOfTrailingVortexEdges_), i_size, 1, adb_file); // Number of trailing wake vortices

       XWake_ = new float*[NumberOfTrailingVortexEdges_ + 1];
       YWake_ = new float*[NumberOfTrailingVortexEdges_ + 1];
       ZWake_ = new float*[NumberOfTrailingVortexEdges_ + 1];   
           
       SWake_ = new float[NumberOfTrailingVortexEdges_ + 1];     
       
       WingWake_ = new int[NumberOfTrailingVortexEdges_ + 1];
       
       MaxWings_ = 0;
       
       for ( i = 1 ; i <= NumberOfTrailingVortexEdges_ ; i++ ) {
          
          BIO.fread(&(WingWake_[i]), i_size, 1, adb_file); // Wing ID
          BIO.fread(&(SWake_[i]), f_size, 1, adb_file); // Span location of this trailing vortex
    
          MaxWings_ = MAX(MaxWings_, WingWake_[i]);
   
          BIO.fread(&(NumberOfSubVortexNodes_), i_size, 1, adb_file); // Number of sub vortices

          XWake_[i] = new float[NumberOfSubVortexNodes_ + 1];
          YWake_[i] = new float[NumberOfSubVortexNodes_ + 1];
          ZWake_[i] = new float[NumberOfSubVortexNodes_ + 1];

          for ( j = 1 ; j <= NumberOfSubVortexNodes_ ; j++ ) {

             BIO.fread(&(XWake_[i][j]), f_size, 1, adb_file); // X
             BIO.fread(&(YWake_[i][j]), f_size, 1, adb_file); // Y
             BIO.fread(&(ZWake_[i][j]), f_size, 1, adb_file); // Z

             XWake_[i][j] -= GeometryXShift;
             YWake_[i][j] -= GeometryYShift;
             ZWake_[i][j] -= GeometryZShift;
             
           }
          
       }
       
       CurrentEdgeMach  =  MachList[1];
       CurrentAlpha     = AlphaList[1];
       CurrentBeta      = BetaList[1];
   
       CurrentChoiceMach  = 1;
       CurrentChoiceAlpha = 1;
       CurrentChoiceBeta  = 1;
       
       // Read in any control surface deflection data
   
       for ( i = 1 ; i <= NumberOfControlSurfaces ; i++ ) {
   
          BIO.fread(&(ControlSurface[i].DeflectionAngle), f_size, 1, adb_file); 

       }       
    
    }
    
    // Close the adb file

    fclose(adb_file);

}

/*##############################################################################
#                                                                              #
#                   GL_VIEWER RotateControlSurfaceNode                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::RotateControlSurfaceNode( float xyz[3], int ConSurf )
{

    QUAT Quat, InvQuat, Vec;
   
    // Rotate point about control surface hinge line

    Quat.FormRotationQuatf(ControlSurface[ConSurf].HingeVec,ControlSurface[ConSurf].DeflectionAngle);

    InvQuat = Quat;

    InvQuat.FormInverse(); 

    Vec(0) = xyz[0] - ControlSurface[ConSurf].HingeNode1[0];
    Vec(1) = xyz[1] - ControlSurface[ConSurf].HingeNode1[1];
    Vec(2) = xyz[2] - ControlSurface[ConSurf].HingeNode1[2];
    
    Vec = Quat * Vec * InvQuat;
    
    xyz[0] = Vec(0) + ControlSurface[ConSurf].HingeNode1[0];
    xyz[1] = Vec(1) + ControlSurface[ConSurf].HingeNode1[1];
    xyz[2] = Vec(2) + ControlSurface[ConSurf].HingeNode1[2];
 
}

/*##############################################################################
#                                                                              #
#                               GL_VIEWER LoadCaseFile                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadCaseFile(char *FileName)
{

    int i, j, Done;
    FILE *case_file;
    char file_name_w_ext[2000], DumChar[200], file_type[100], PlanetType[80];
    fpos_t StartOfRotorData;

    // Open the case file

    sprintf(file_name_w_ext,"%s.vspaero",FileName);

    if ( (case_file = fopen(file_name_w_ext,"r")) == NULL ) {

       printf("Could not open the file: %s for input! \n",file_name_w_ext);fflush(NULL);

       exit(1);

    }

    // Search for rotor data
    
    NumberOfPropulsionElements = 0;
    
    Done = 0;
   
    fgetpos(case_file, &StartOfRotorData);    
    
    while ( fgets(DumChar,200,case_file) != NULL && !Done ) {

       if ( strncmp(DumChar,"NumberOfRotors",14) == 0 ) {

          fsetpos(case_file, &StartOfRotorData);
       
          fscanf(case_file,"NumberOfRotors = %d \n",&NumberOfPropulsionElements);
          
          printf("NumberOfRotors: %d \n",NumberOfPropulsionElements);fflush(NULL);

          PropulsionElement = new PROPULSION_ELEMENT[NumberOfPropulsionElements + 1];

          for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {
           
             fgets(DumChar,200,case_file);
             fgets(DumChar,200,case_file);
             
             PropulsionElement[i].Rotor.Load_STP_Data(case_file);
 
          }
              
          Done = 1;
          
       }
       
       fgetpos(case_file, &StartOfRotorData);
       
    }
    
    fclose(case_file);
        
}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER LoadBeam3DFEMData                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadBeam3DFEMData(void)
{
 
    int i;
    char FEM_File_Name[2000];
    FILE *File;
    
    printf("NumberOfSurfaces_: %d \n",NumberOfSurfaces_);
    
    FemData_ = new FEM_NODE[NumberOfSurfaces_ + 1];
    
    for ( i = 1 ; i <= NumberOfSurfaces_ ; i++ ) {

       sprintf(FEM_File_Name,"%s.Surface.%d.dfm",file_name,i);
    
       // If a deformation solution exists... load it in
       
       if ( (File = fopen(FEM_File_Name,"r")) != NULL ) {
        
          fclose(File);
          
          printf("Reading in a FEM deformation file for surface: %d \n",i);
          
          LoadFEMDeformationData(i,FEM_File_Name);

       }             

    }    

}

/*##############################################################################
#                                                                              #
#                   GL_VIEWER LoadAdjointandGradients                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadAdjointandGradients(void)
{

    int i, DumInt;
    char OptFileName[2000], DumChar[2000];
    FILE *OptFile;
    
    sprintf(OptFileName,"%s.gradient",file_name);
    
    // If an optimization solution exists... load it in
    
    if ( (OptFile = fopen(OptFileName,"r")) != NULL ) {
     
       printf("Reading in a adjoint optimization filen \n");
       
       fscanf(OptFile,"%d",&DumInt);
       
       OptimizationData_.SizeList(DumInt);
       
       fgets(DumChar,2000,OptFile);
       
       for ( i = 1 ; i <= OptimizationData_.NumberOfOptNodes() ; i++ ) {
          
          fscanf(OptFile,"%lf %lf %lf %lf %lf %lf \n",
           &(OptimizationData_.x(i)),
           &(OptimizationData_.y(i)),
           &(OptimizationData_.z(i)),
           &(OptimizationData_.dFdx(i)),
           &(OptimizationData_.dFdy(i)),
           &(OptimizationData_.dFdz(i)));
           
       }
       
       fclose(OptFile);
       
    }      
    
}

/*##############################################################################
#                                                                              #
#                      GL_VIEWER LoadFEMDeformationData                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LoadFEMDeformationData(int i, char *FileName)
{
 
    int j, NumberOfFEMNodes;
    FILE *FEMFile;
    
    // Open the FEM deformation file

    printf("Reading in dfm file: %s \n",FileName);
    
    if ( (FEMFile = fopen(FileName, "r")) == NULL ) {

       printf("Could not open the FEM deformation file for input! \n");

       exit(1);

    }

    fscanf(FEMFile,"%d",&NumberOfFEMNodes);

    FemData(i).SizeList(NumberOfFEMNodes);

    printf("NumberOfFEMNodes: %d \n",NumberOfFEMNodes);

    for ( j = 1 ; j <= FemData(i).NumberOfFEMNodes() ; j++ ) { 
    
       fscanf(FEMFile,"%lf %lf %lf %lf %lf %lf %lf %lf %lf \n",
             &(FemData(i).x(j)),
             &(FemData(i).y(j)),   
             &(FemData(i).z(j)),        
             &(FemData(i).delta_xv(j)),
             &(FemData(i).delta_yv(j)),
             &(FemData(i).delta_zv(j)), 
             &(FemData(i).delta_phix(j)), 
             &(FemData(i).delta_phiy(j)), 
             &(FemData(i).delta_phiz(j)));


    }

    fclose(FEMFile);

}

/*##############################################################################
#                                                                              #
#                              EDGES CreateTriEdges                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::CreateTriEdges(void)
{

    int i, j, k, nod1, nod2, noda, nodb, start_edge, surf_edge;
    int level, edge_to_node[4][3], nod_list[4], tri1, tri2;
    int max_edge, new_edge, *jump_pnt, Error;
    GL_EDGE_ENTRY *list, *tlist;

    printf("Finding tri edges... \n");fflush(NULL);

    Error = 0;

    // Make space for a linked list of edges

    jump_pnt = new int[NumberOfNodes + 1];

    max_edge = 4*NumberOfNodes;

    list = new GL_EDGE_ENTRY[max_edge + 1];

    // Zero out the lists

    for ( i = 0 ; i <= NumberOfNodes ; i++ ) {

       jump_pnt[i] = 0;

    }

    for ( i = 0 ; i <= max_edge ; i++ ) {

       list[i].node  = 0;
       list[i].next  = 0;
       list[i].tri_1 = 0;
       list[i].tri_2 = 0;

    }

    // Initialize number of edges

    new_edge = 0;

    // Keep track of first edge

    start_edge = new_edge + 1;

    // List of nodes for each triangle edge

    edge_to_node[1][1] = 1;
    edge_to_node[1][2] = 2;

    edge_to_node[2][1] = 2;
    edge_to_node[2][2] = 3;

    edge_to_node[3][1] = 1;
    edge_to_node[3][2] = 3;

    // Loop over triangles and create the edge list

    for ( j = 1 ; j <= NumberOfTris ; j++ ) {

       // Local copy of node pointers

       nod_list[1] = TriList[j].node1;
       nod_list[2] = TriList[j].node2;
       nod_list[3] = TriList[j].node3;

       // Loop over all edge cases

       for ( i = 1 ; i <= 3 ; i++ ) {

          noda = nod_list[edge_to_node[i][1]];
          nodb = nod_list[edge_to_node[i][2]];

          nod1 = MIN(noda,nodb);
          nod2 = MAX(noda,nodb);

          level = jump_pnt[nod1];

          // New edge, since first edge for this node

          if ( level == 0 ) {

             new_edge++;

             // If this is a 3d case, the surface should not introduce new edges

             if ( start_edge != 1 ) {

                 printf("Error: tris contain edges not in tet grid! \n");fflush(NULL);

                 exit(1);

             }

             // Make sure there is enough room, if not reallocate space

             if ( new_edge >= max_edge ) {

                max_edge = (int) (1.2*max_edge);

                tlist = new GL_EDGE_ENTRY[max_edge + 1];

                for ( k = 1 ; k <= new_edge ; k++ ) {

                   tlist[k] = list[k];

                }

                delete [] list;

                list = tlist;

             }

             list[new_edge].node = nod2;
             list[new_edge].next = 0;

             jump_pnt[nod1] = new_edge;

             // Pack tri to edge pointer

             if ( noda == nod1 ) {

                if ( i == 1 ) TriList[j].edge1 = new_edge;
                if ( i == 2 ) TriList[j].edge2 = new_edge;
                if ( i == 3 ) TriList[j].edge3 = new_edge;

             }

             else {

                if ( i == 1 ) TriList[j].edge1 = -new_edge;
                if ( i == 2 ) TriList[j].edge2 = -new_edge;
                if ( i == 3 ) TriList[j].edge3 = -new_edge;

             }

             // Pack surface edge to tri pointer

             if ( list[new_edge].tri_1 == 0 ) {

                list[new_edge].tri_1 = j;

             }

             else {

                list[new_edge].tri_2 = j;

             }

          }

          // Must check if this edge exists

          else {

             while ( list[level].node != nod2 && list[level].next != 0 ) {

                level = list[level].next;

             }

             // Old edge

             if ( list[level].node == nod2 ) {

                // Pack tri to edge pointer

                if ( noda == nod1 ) {

                   if ( i == 1 ) TriList[j].edge1 = level;
                   if ( i == 2 ) TriList[j].edge2 = level;
                   if ( i == 3 ) TriList[j].edge3 = level;

                }

                else {

                   if ( i == 1 ) TriList[j].edge1 = -level;
                   if ( i == 2 ) TriList[j].edge2 = -level;
                   if ( i == 3 ) TriList[j].edge3 = -level;

                }

                // Pack surface edge to tri pointer

                if ( list[level].tri_1 == 0 ) {

                   list[level].tri_1 = j;

                }

                else {

                   list[level].tri_2 = j;

                }

             }

             // New edge

             else {

                new_edge++;

                // If this is a 3d case, the surface should not introduce new edges

                if ( start_edge != 1 ) {

                    printf("Error: tris contain edges not in tet grid! \n");fflush(NULL);

                    exit(1);

                }

                // Make sure there is enough room, if not reallocate space

                if ( new_edge >= max_edge ) {

                   max_edge = (int) (1.2*max_edge);

                   tlist = new GL_EDGE_ENTRY[max_edge + 1];

                   for ( k = 1 ; k <= new_edge ; k++ ) {

                      tlist[k] = list[k];

                   }

                   delete [] list;

                   list = tlist;

                }

                list[level].next = new_edge;

                list[new_edge].node  = nod2;
                list[new_edge].next  = 0;

                // Pack tri to edge pointer

                if ( noda == nod1 ) {

                   if ( i == 1 ) TriList[j].edge1 = new_edge;
                   if ( i == 2 ) TriList[j].edge2 = new_edge;
                   if ( i == 3 ) TriList[j].edge3 = new_edge;

                }

                else {

                   if ( i == 1 ) TriList[j].edge1 = -new_edge;
                   if ( i == 2 ) TriList[j].edge2 = -new_edge;
                   if ( i == 3 ) TriList[j].edge3 = -new_edge;

                }

                // Pack surface edge to tri pointer

                if ( list[new_edge].tri_1 == 0 ) {

                   list[new_edge].tri_1 = j;

                }

                else {

                   list[new_edge].tri_2 = j;

                }

             }

          }

       }

    }

    // Store edge to node pointers

    if ( start_edge == 1 ) {

       NumberOfEdges = new_edge;

       EdgeList = new EDGE[NumberOfEdges + 1];

       // Fill the edge list

       for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

          level = jump_pnt[i];

          while ( level != 0 ) {

			 EdgeList[level].node1 = i;
			 EdgeList[level].node2 = list[level].node;

             if ( list[level].tri_2 == 0 ) {

                Error = 1;

             }

             level = list[level].next;

          }

       }

    }

    // Store edge to tri pointers

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       level = jump_pnt[i];

       while ( level != 0 ) {

          if ( list[level].tri_1 != 0 && list[level].tri_1 != 0) {

             if ( start_edge == 1 ) {

                surf_edge = level;

             }

             else {

               surf_edge++;

             }

             // Store pointers to tris

             EdgeList[level].tri1 = list[level].tri_1;
             EdgeList[level].tri2 = list[level].tri_2;

          }

          else {

            if ( start_edge == 1 ) {

               printf("Surface triangulation as input is messed up! \n");
               printf("Error found while storing surface edges...\n");fflush(NULL);

               exit(1);

            }

          }

          level = list[level].next;

       }

    }

    printf("Number of nodes: %d \n",NumberOfNodes);
    printf("Number of tris: %d \n",NumberOfTris);
    printf("Number of edges is: %d \n",NumberOfEdges);fflush(NULL);

    // Free up the scratch space

    delete [] jump_pnt;
    delete [] list;

    // Error noting

    if ( Error ) printf("Note... mesh may not be closed! \n");fflush(NULL);

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER FindMeshMinMax                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FindMeshMinMax(void)
{

    int i;

    XMin = 1.e6;
    XMax = -XMin;

    YMin = 1.e6;
    YMax = -YMin;

    ZMin = 1.e6;
    ZMax = -ZMin;

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

	   XMin = MIN(XMin, NodeList[i].x);
	   YMin = MIN(YMin, NodeList[i].y);
	   ZMin = MIN(ZMin, NodeList[i].z);

	   XMax = MAX(XMax, NodeList[i].x);
	   YMax = MAX(YMax, NodeList[i].y);
	   ZMax = MAX(ZMax, NodeList[i].z);

	}

	printf("Xmin, Xmax: %f, %f \n",XMin, XMax);
	printf("Ymin, Ymax: %f, %f \n",YMin, YMax);
	printf("Zmin, Zmax: %f, %f \n",ZMin, ZMax);fflush(NULL);

	GeometryXShift = XMin + 0.5*(XMax - XMin);
	GeometryYShift = YMin + 0.5*(YMax - YMin);
	GeometryZShift = ZMin + 0.5*(ZMax - ZMin);

    // Center geometry

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       NodeList[i].x -= GeometryXShift;
       NodeList[i].y -= GeometryYShift;
       NodeList[i].z -= GeometryZShift;

    }

    XMin = 1.e6;
    XMax = -XMin;

    YMin = 1.e6;
    YMax = -YMin;

    ZMin = 1.e6;
    ZMax = -ZMin;

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

	   XMin = MIN(XMin, NodeList[i].x);
	   YMin = MIN(YMin, NodeList[i].y);
	   ZMin = MIN(ZMin, NodeList[i].z);

	   XMax = MAX(XMax, NodeList[i].x);
	   YMax = MAX(YMax, NodeList[i].y);
	   ZMax = MAX(ZMax, NodeList[i].z);

	}

	ViewSize = MAX3(XMax-XMin, YMax-YMin, ZMax-ZMin);

	ZoomScale = sqrt( pow(XMax-XMin,2) + pow(YMax-YMin,2) + pow(ZMax-ZMin,2) );
printf("ViewSize: %f ... ZoomScale: %f \n",ViewSize,ZoomScale);

// To force fixed perspective... we need to
// 1) Set ViewSize to a fixed value for all frames
// 2) Set Min, Max of x,y,z to some fixed values for all frames
// 3) Recalculate geometry xyz shifts based on 2

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER FixViewingBox                           #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FixViewingBox(float x1, float x2, float y1, float y2, float z1, float z2 )
{

    int i;

    // Un-Center geometry

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       NodeList[i].x += GeometryXShift;
       NodeList[i].y += GeometryYShift;
       NodeList[i].z += GeometryZShift;

    }

    GeometryXShift = 0.;
    GeometryYShift = 0.;
    GeometryZShift = 0.;

    XMin = x1;
    XMax = x2;

    YMin = y1;
    YMax = y2;

    ZMin = z1;
    ZMax = z2;

	ViewSize = MAX3(XMax-XMin, YMax-YMin, ZMax-ZMin);

	ZoomScale = sqrt( pow(XMax-XMin,2) + pow(YMax-YMin,2) + pow(ZMax-ZMin,2) );

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER FindSolutionMinMax                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FindSolutionMinMax(void)
{

FindSolutionMinMax(Cp,                 CpMinActual,         CpMaxActual,         CpMin,         CpMax);
FindSolutionMinMax(CpSteady,     CpSteadyMinActual,   CpSteadyMaxActual,   CpSteadyMin,   CpSteadyMax);
FindSolutionMinMax(CpUnsteady, CpUnsteadyMinActual, CpUnsteadyMaxActual, CpUnsteadyMin, CpUnsteadyMax);
FindSolutionMinMax(Gamma,           GammaMinActual,      GammaMaxActual,      GammaMin,      GammaMax);

if ( ModelType == PANEL_MODEL ) CpMax = CpMaxSoln;

/*
    int i, j, m, Hits;
    float Big = 1.e9, Avg, StdDev;

    CpMinActual = Big;
    CpMaxActual = -Big;
    
    Avg   = 0.;   Hits = 0;
 
    for ( m = 1 ; m <= NumberOfTris ; m++ ) {

       CpMinActual = MIN(CpMinActual, Cp[m]);
       CpMaxActual = MAX(CpMaxActual, Cp[m]);
       
       Avg += Cp[m]; Hits++;

    }
    
    Avg /= Hits;
  
    // Now calculate some statistics
  
    StdDev   = 0.;   Hits = 0;

    for ( m = 1 ; m <= NumberOfTris ; m++ ) {

       StdDev += pow(Avg- Cp[m],2.0f); Hits++;

    }    
    
    StdDev = sqrt(StdDev/Hits);

    // Set Cp min, and max
    
    if ( ModelType == VLM_MODEL ) {
    
       CpMin = MAX(Avg - 1. * StdDev, CpMinActual);
       CpMax = MIN(Avg + 1. * StdDev, CpMaxActual);
       
    }
    
    else {

       CpMin = MAX(Avg - 2. * StdDev, CpMinActual);
       CpMax = CpMaxSoln;
       
    }
*/
}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER FindSolutionMinMax                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FindSolutionMinMax(float *Function, float &FMinActual, float &FmaxActual, float &FMin, float &FMax)
{

    int i, j, m, Hits;
    float Big = 1.e9, Avg, StdDev;

    FMinActual = Big;
    FmaxActual = -Big;
    
    Avg   = 0.;   Hits = 0;
 
    for ( m = 1 ; m <= NumberOfTris ; m++ ) {

       FMinActual = MIN(FMinActual, Function[m]);
       FmaxActual = MAX(FmaxActual, Function[m]);
       
       Avg += Function[m]; Hits++;

    }
    
    Avg /= Hits;
  
    // Now calculate some statistics
  
    StdDev   = 0.;   Hits = 0;

    for ( m = 1 ; m <= NumberOfTris ; m++ ) {

       StdDev += pow(Avg- Function[m],2.0f); Hits++;

    }    
    
    StdDev = sqrt(StdDev/Hits);

    // Set Cp min, and max
    
    if ( ModelType == VLM_MODEL ) {
    
       FMin = MAX(Avg - 1. * StdDev, FMinActual);
       FMax = MIN(Avg + 1. * StdDev, FmaxActual);
       
    }
    
    else {

       FMin = MAX(Avg - 2. * StdDev, FMinActual);
       FMax = MIN(Avg + 2. * StdDev, FmaxActual);
       
    }

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER SetSolutionMin                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetSolutionMin(float MinVal)
{

    int i;

    if ( DrawCpIsOn         ) CpMin         = MinVal;
    if ( DrawCpSteadyIsOn   ) CpSteadyMin   = MinVal;
    if ( DrawCpUnsteadyIsOn ) CpUnsteadyMin = MinVal;
    if ( DrawGammaIsOn      ) GammaMin      = MinVal;

    UserSetPlotLimits = 1;

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER SetSolutionMinMax                       #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetSolutionMax(float MaxVal)
{

    int i;

    if ( DrawCpIsOn         ) CpMax         = MaxVal;
    if ( DrawCpSteadyIsOn   ) CpSteadyMax   = MaxVal;
    if ( DrawCpUnsteadyIsOn ) CpUnsteadyMax = MaxVal;
    if ( DrawGammaIsOn      ) GammaMax      = MaxVal;

    UserSetPlotLimits = 1;

}

/*##############################################################################
#                                                                              #
#                               GL_VIEWER ResetView                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::ResetView(void)
{

    xshift = yshift = 0.;
    hAng = vAng = zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 1.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER FitToWindow                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FitToWindow(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = vAng = zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER TopView                                #
#                                                                              #
##############################################################################*/

void GL_VIEWER::TopView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = vAng = zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER BottomView                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::BottomView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = 0.;
    vAng = 180.;
    zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER LeftView                               #
#                                                                              #
##############################################################################*/

void GL_VIEWER::LeftView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = 0.;
    vAng = -90.;
    zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER RightView                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::RightView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = 180.;
    vAng = -90.;
    zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER FrontView                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::FrontView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = 90.;
    vAng = -90.;
    zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER RearView                               #
#                                                                              #
##############################################################################*/

void GL_VIEWER::RearView(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng = -90.;
    vAng = -90.;
    zAng = 0.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

    trackball(NewQuat, 0.0, 0.0, 0.0, 0.0);

}

/*##############################################################################
#                                                                              #
#                         GL_VIEWER IsoViewDown                                #
#                                                                              #
##############################################################################*/

void GL_VIEWER::IsoViewDown(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng =   0.;
    vAng = -55.;
    zAng =  45.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

}

/*##############################################################################
#                                                                              #
#                           GL_VIEWER IsoViewUp                                #
#                                                                              #
##############################################################################*/

void GL_VIEWER::IsoViewUp(void)
{

    ResetView();

    xshift = yshift = 0.;
    hAng =   0.;
    vAng =  -134.;
    zAng =   45.;

    NewTransX = NewTransY = OldTransX = OldTransY = 0.;
    NewScale = OldScale = 2.;

}

/*##############################################################################
#                                                                              #
#                             GL_VIEWER ZeroViews                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::ZeroAllViews(void)
{

    int i;

    DrawLegendIsOn                    = 0;
    DrawLabelIsOn                     = 0;
    DrawWithWhiteBackgroundIsOn       = 0;
 
    DrawWireFrameIsOn                 = 0;
    DrawShadedIsOn                    = 0;
    DrawCpIsOn                        = 0;
    DrawCpUnsteadyIsOn                = 0;
    DrawCpSteadyIsOn                  = 0;
    DrawGammaIsOn                     = 0;
 
    DrawCGIsOn                        = 0;
    DrawCGLabelIsOn                   = 0;
    
    DrawWakesIsOn                     = 0;
       
}

/*##############################################################################
#                                                                              #
#                           GL_VIEWER ClearADBFile                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::ClearADBFile(void)
{

// this is test code...

    int i;
    const Fl_Menu_Item *MenuList;
    int MenuListSize, UserSelectedMenuItem;

    MenuList = vui->ADBMenu->menu();

    MenuListSize = vui->ADBMenu->size();

	printf("vui->ADBMenu->value(): %d --> name: %s \n",vui->ADBMenu->value(),MenuList[vui->ADBMenu->value()].label());fflush(NULL);

    UserSelectedMenuItem = vui->ADBMenu->value();

    for ( i = 0 ; i < vui->ADBMenu->size() ; i++ ) {

       MenuList[5].do_callback(NULL);
    //   if ( i != UserSelectedMenuItem ) vui->ADBMenu->mode(i,FL_MENU_TOGGLE);

    }

}

/*##############################################################################
#                                                                              #
#                        GL_VIEWER ToggleFlag                                  #
#                                                                              #
##############################################################################*/

void GL_VIEWER::ToggleFlag(int &Flag, const char *Text)
{

    Flag = 1 - Flag;

}

/*##############################################################################
#                                                                              #
#                               GL_VIEWER Draw                                 #
#                                                                              #
##############################################################################*/

int GL_VIEWER::handle(int event)
{

    float q;
    char key;

    switch(event) {

       // Focus on widget

	   case FL_FOCUS: return 1;

	   case FL_KEYBOARD:
     
          // Left arrow

          if ( Fl::event_key() == FL_Left ) {
           
             if ( Fl::event_state(FL_CTRL) ) {

                trackball(OldQuat,
                          0.00,0.00, 
                         -0.25,0.00);

                add_quats(OldQuat, NewQuat, NewQuat);                
                
             }
             
             else {

                NewTransX -= 0.01;
                
             }

             redraw();
             
          }
          
          // Right arrow

          if ( Fl::event_key() == FL_Right ) {

             if (  Fl::event_state(FL_CTRL) ) {
              
                trackball(OldQuat,
                          0.00,0.00, 
                          0.25,0.00);

                add_quats(OldQuat, NewQuat, NewQuat);                
                
             }
             
             else {

                NewTransX += 0.01;
                
             }
             
             redraw();

          }          
          
          // Up arrow

          if ( Fl::event_key() == FL_Up ) {

            if (  Fl::event_state(FL_CTRL)) {
              
                trackball(OldQuat,
                          0.00,0.00, 
                          0.00,0.25);

                add_quats(OldQuat, NewQuat, NewQuat);                
                
             }
             
             else if ( Fl::event_state(FL_ALT) ) {
              
                NewScale = OldScale * 1.1;
                
                OldScale = NewScale;
              
             }
             
             else {

                NewTransY += 0.01;
                
             }

             redraw();           

          }

          // Down arrow

          if ( Fl::event_key() == FL_Down ) {
    
             if ( Fl::event_state(FL_CTRL) ) {
              
                trackball(OldQuat,
                          0.00, 0.00, 
                          0.00,-0.25);

                add_quats(OldQuat, NewQuat, NewQuat);                
                
             }
             
             else if ( Fl::event_state(FL_ALT) ) {
              
                NewScale = OldScale / 1.1;
                
                OldScale = NewScale;
              
             }
                          
             
             else {

                NewTransY -= 0.01;
                
             }

             redraw();     

          }

          // Control key...
      
          if ( Fl::event_state(FL_CTRL) ) {

             if ( Fl::event_key() == 'q' )  exit(1);
             if ( Fl::event_key() == 'r' )  { ResetView(); Draw(); };
             
             if ( Fl::event_key() == 65470 ) { TopView();     Draw(); } // F1
             if ( Fl::event_key() == 65471 ) { BottomView();  Draw(); } // F2
             if ( Fl::event_key() == 65472 ) { LeftView();    Draw(); } // F3
             if ( Fl::event_key() == 65473 ) { RightView();   Draw(); } // F4
             if ( Fl::event_key() == 65474 ) { FrontView();   Draw(); } // F5
             if ( Fl::event_key() == 65475 ) { RearView();    Draw(); } // F6
             if ( Fl::event_key() == 65476 ) { IsoViewUp();   Draw(); } // F7
             if ( Fl::event_key() == 65477 ) { IsoViewDown(); Draw(); } // F8        
             
          }
          
          // Some ASCII key was pushed...

          if ( Fl::event_text()[0]) {

              key = Fl::event_text()[0];

              // Insert keys...

              if ( key == 'f' ) FAST_DRAW_ON = 1 - FAST_DRAW_ON;
              
              if ( Fl::event_key() == '=' )  { CoarseMeshLevelSelected-- ; CoarseMeshLevelSelected = MIN(NumberOfMeshLevels,MAX(1,CoarseMeshLevelSelected)); Draw(); };
              if ( Fl::event_key() == '-' )  { CoarseMeshLevelSelected++ ; CoarseMeshLevelSelected = MIN(NumberOfMeshLevels,MAX(1,CoarseMeshLevelSelected)); Draw(); };
               
              redraw();
              return 1;

          }          
          
          redraw();
          return 1;

       // One of the buttons was pushed

	    case FL_PUSH:      
         
         // Button 1 Down - Rotation Control

	      if ( Fl::event_button1() && !Fl::event_state(FL_CTRL) && !Fl::event_state(FL_META) ) {

		     OldX = Fl::event_x();
		     OldY = Fl::event_y();

		   }

         // Button 2 Down - Zoom Control

	      if ( Fl::event_button2() || ( Fl::event_state(FL_CTRL) && Fl::event_button1() ) ) {

		     OldX = Fl::event_x();
		     OldY = Fl::event_y();

		     OldScale = NewScale;

		   }

          // Button 3 Down - Translation Control

	      if ( Fl::event_button3() || ( Fl::event_state(FL_META) && Fl::event_button1() )  ) {

		     OldX = Fl::event_x();
		     OldY = Fl::event_y();

		     OldTransX = NewTransX;
		     OldTransY = NewTransY;

		   }

	      redraw();

	      Fl::focus(this);

	      return 1;

       // One of the buttons was pushed and the mouse dragged

	    case FL_DRAG:

          // Button 1 is dragged - rotation

	      if ( Fl::event_button1() && !Fl::event_state(FL_CTRL) && !Fl::event_state(FL_META) ) {

             NewX = Fl::event_x();
             NewY = Fl::event_y();

             trackball(OldQuat,
                       2.*(2.*OldX - w()) / w(), 2.*(h() - 2.*OldY) / h(),
                       2.*(2.*NewX - w()) / w(), 2.*(h() - 2.*NewY) / h() );

             add_quats(OldQuat, NewQuat, NewQuat);

		       OldX = Fl::event_x();
		       OldY = Fl::event_y();

		       redraw();

		  }

        // Button 2 is dragged - zooming

		  if ( Fl::event_button2() || ( Fl::event_state(FL_CTRL) && Fl::event_button1() ) ) {

             NewX = Fl::event_x();
             NewY = Fl::event_y();

             q = (float) -( ( OldY - NewY )/400. );

             NewScale = OldScale * exp(q);

             redraw();

	      }

         // Button 3 is dragged - translation

         if ( Fl::event_button3() || ( Fl::event_state(FL_META) && Fl::event_button1() ) ) {

             NewX = Fl::event_x();
             NewY = Fl::event_y();

             NewTransX = ( OldTransX + ( ( NewX - OldX )/1000. ) );
             NewTransY = ( OldTransY - ( ( NewY - OldY )/1000. ) );

             redraw();

	      }

	      return 1;

	   default:

	      return Fl_Gl_Window::handle(event);

	}

}

/*##############################################################################
#                                                                              #
#                               GL_VIEWER Draw                                 #
#                                                                              #
##############################################################################*/

void GL_VIEWER::Draw(void)
{

    int i;
    GLfloat m[4][4];
    float AR;

    float position1[] = { -0.5f*ViewSize,  0.3f*ViewSize, 0.5f*ViewSize, 0.f };
    float position2[] = {  0.5f*ViewSize,  0.3f*ViewSize, 0.5f*ViewSize, 0.f };

    GLfloat ambient1[] = { 1.0f*Brightness, 1.0f*Brightness, 1.0f*Brightness, 1.0f };
    GLfloat ambient2[] = { 1.0f*Brightness, 1.0f*Brightness, 1.0f*Brightness, 1.0f };
   
 //   GLfloat ambient1[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };
 //   GLfloat ambient2[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };

    GLfloat ambient3[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };
    GLfloat ambient4[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };
       
    // Enable Lighting

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glLightfv(GL_LIGHT0,GL_DIFFUSE,ambient1);
    glLightfv(GL_LIGHT1,GL_DIFFUSE,ambient2);
    glEnable(GL_NORMALIZE);

    AR = h()/(float)w();

    glMatrixMode(GL_PROJECTION);

    // Position the lights

    glLightfv(GL_LIGHT0, GL_POSITION, position1);
    glLightfv(GL_LIGHT1, GL_POSITION, position2);

    glEnable(GL_DEPTH_TEST);

    glLoadIdentity();
    glViewport(0, 0, pixel_w(), pixel_h());
    glOrtho(-1.5*ViewSize,1.5*ViewSize,
            -1.5*AR*ViewSize,1.5*AR*ViewSize,
            -15.*ViewSize*NewScale,15.*ViewSize*NewScale);

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    if ( DrawWithWhiteBackgroundIsOn ) {

       BackgroundRGBA[0] = 1.0;
       BackgroundRGBA[1] = 1.0;
       BackgroundRGBA[2] = 1.0;
       BackgroundRGBA[3] = 0.0;

       TextRGBA[0] = 0.0;
       TextRGBA[1] = 0.0;
       TextRGBA[2] = 0.0;
       TextRGBA[3] = 0.0;

       glClearColor(BackgroundRGBA[0],
                    BackgroundRGBA[1],
                    BackgroundRGBA[2],
                    BackgroundRGBA[3]);

    }

    else {

       BackgroundRGBA[0] = 0.0;
       BackgroundRGBA[1] = 0.0;
       BackgroundRGBA[2] = 0.9;
       BackgroundRGBA[3] = 0.0;

       TextRGBA[0] = 1.0;
       TextRGBA[1] = 1.0;
       TextRGBA[2] = 1.0;
       TextRGBA[3] = 0.0;

       glClearColor(BackgroundRGBA[0],
                    BackgroundRGBA[1],
                    BackgroundRGBA[2],
                    BackgroundRGBA[3]);

    }
    
    if ( DrawFlatShadedIsOn ) glDisable(GL_LIGHTING);

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glPushMatrix();

       glTranslatef(-xshift*ABS(XMax-XMin), -yshift*ABS(YMax-YMin), 0);

       glRotatef(hAng,0,1,0); glRotatef(vAng,1,0,0); glRotatef(zAng,0,0,1);

       glScalef(float(size),float(size),float(size));

       build_rotmatrix(m, NewQuat, ZoomScale*NewTransX,
                                   ZoomScale*NewTransY, 0., NewScale);
       glMultMatrixf(&m[0][0]);

       // Draw all the 3D stuff

       if ( FAST_DRAW_ON ) {

          DrawBBox();

       }

       else {
      
          if ( DrawWireFrameIsOn          ) DrawWireFrame();
              
          if ( DrawShadedIsOn             ) DrawShadedSurface();
                  
          if ( DrawCpIsOn                 ) DrawCp();          
          
          if ( DrawCpSteadyIsOn           ) DrawCpSteady();
              
          if ( DrawCpUnsteadyIsOn         ) DrawCpUnsteady();

          if ( DrawGammaIsOn              ) DrawGamma();

          if ( DrawWakesIsOn              ) DrawWakes();

              
          if ( DrawPropulsionElementsIsOn ) DrawRotorSurfacesShaded();
                  
          if ( DrawCGIsOn                 ) DrawCGMarker();         
               
     
          if ( DrawCoarseMeshesIsOn       ) { DrawCoarseMeshEdgesForLevel(CoarseMeshLevelSelected); DrawCoarseMeshNodesForLevel(CoarseMeshLevelSelected); };

          if ( DrawAxesIsOn               ) DrawAxes();

          if ( DrawControlSurfacesIsOn    ) DrawControlSurfaces();
          
          if ( DrawBEAM3DFEMIsOn          ) DrawBEAM3DFEM();
          

          
       }

    glPopMatrix();

    // Now Draw all the 2D stuf

    glDisable(GL_DEPTH_TEST);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho( 0., w(), 0., h() , -1., 1.);

    glDisable(GL_LIGHTING);

    glPushMatrix();

       if ( DrawLegendIsOn           ) DrawLegend();
       if ( DrawLabelIsOn            ) DrawLabel();
       if ( DrawCGLabelIsOn          ) DrawCGLabel();

    glPopMatrix();

    glEnable(GL_LIGHTING);

    glLoadIdentity();
    glViewport(0, 0, pixel_w(), pixel_h());
    glOrtho(-1.5*ViewSize,1.5*ViewSize,
            -1.5*AR*ViewSize,1.5*AR*ViewSize,
            -15.*ViewSize*NewScale,15.*ViewSize*NewScale);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawLegend                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawLegend(void)
{

    // Draw RGB legend

    int i, no_box;
    float cvec[3];
    float x1, x2, y1, y2, y_old, y_delt, lim_delt, LegendVal;
    float prcent;
    char string[80];
        
    no_box = 50;

    lim_delt = 0.05*h();

    x1 = 0.01*w();
    x2 = 0.05*w();
    y1 = 0.15*h();
    y2 = 0.85*h();

    y_delt = (y2 - y1)/(float)(no_box-1);

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

    // Draw bottom clip region
    
    prcent = 0.;
    
    percent_to_rgb(prcent, cvec, 0);

    glColor3fv( cvec);
    glBegin( GL_POLYGON );

       glVertex2f( x1, y1 - 3.* y_delt );
       glVertex2f( x2, y1 - 3.* y_delt );
       glVertex2f( x2, y1 - 2.* y_delt );
       glVertex2f( x1, y1 - 2.* y_delt );

    glEnd();
       
    // Draw majority of box

    y_old = y1;

    for (i = 0 ; i < no_box ; i++) {

       prcent = (float)i/(float)(no_box-1);

       if (prcent >= 1.0) prcent = 0.9999;

       percent_to_rgb(prcent, cvec, 0);

       glColor3fv( cvec);
       glBegin( GL_POLYGON );

          glVertex2f( x1, y_old );
          glVertex2f( x2, y_old );
          glVertex2f( x2, y_old + y_delt );
          glVertex2f( x1, y_old + y_delt );

       glEnd();

       y_old += y_delt;

    }
    
    // Draw top clip region
    
    prcent = 1.;
    
    percent_to_rgb(prcent, cvec, 0);

    glColor3fv( cvec);
    glBegin( GL_POLYGON );

       glVertex2f( x1, y2 + 3.* y_delt );
       glVertex2f( x2, y2 + 3.* y_delt );
       glVertex2f( x2, y2 + 4.* y_delt );
       glVertex2f( x1, y2 + 4.* y_delt );

    glEnd();    

    // Draw labels on legend

    glDisable(GL_DEPTH_TEST);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glColor3fv(TextRGBA);
    glfStringCentering(GL_FALSE);
    glfSetRotateAngle(0.0f);
    glfStringDirection(GLF_LEFT);

    // First the legend title

    glPushMatrix();

       sprintf(string, "%s", LegendTitle);
       glTranslatef(0.05*w(), 0.95*h(), 0);
       glScalef(15., 15., 1);
       glfDrawSolidString(string);

    glPopMatrix();

    // Now, the legend values

    no_box = 5;

    y1 = 0.15*h();
    y2 = 0.85*h();
    y_old = y1;
    y_delt = (y2 - y1)/(float)(no_box-1);

    for (i = 1 ; i <= no_box ; i++) {

       prcent = (i-1)/(float)(no_box-1);

       LegendVal = LegendMin + prcent*(LegendMax - LegendMin);

       glPushMatrix();

          sprintf(string, "%6.5f", LegendVal);
          glTranslatef(0.075*w(), y_old, 0);
          glScalef(15., 15., 1);
          glfDrawSolidString(string);

       glPopMatrix();

       y_old += y_delt;

    }
    
    // Low clipped value

    no_box = 50;

    x1 = 0.01*w();
    x2 = 0.05*w();
    y1 = 0.15*h();
    y2 = 0.85*h();

    y_delt = (y2 - y1)/(float)(no_box-1);

    glPushMatrix();

       sprintf(string, "%6.5f", LegendMinClip);
       glTranslatef(0.075*w(), y1 - 4.*y_delt, 0);
       glScalef(15., 15., 1);
       glfDrawSolidString(string);

    glPopMatrix();    

    glPushMatrix();

       sprintf(string, "%6.5f", LegendMaxClip);
       glTranslatef(0.075*w(), y2 + 4.*y_delt, 0);
       glScalef(15., 15., 1);
       glfDrawSolidString(string);

    glPopMatrix();   
    
}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawLabel                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawLabel(void)
{

    char string[80];

    // Draw the title

    glDisable(GL_DEPTH_TEST);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glColor3fv(TextRGBA);
	glfStringCentering(GL_FALSE);
	glfSetRotateAngle(0.0f);
	glfStringDirection(GLF_LEFT);

    glPushMatrix();

       sprintf(string, "Mach: %2.3f, Beta: %2.8f, Alpha: %2.3f",
       CurrentEdgeMach, CurrentBeta, CurrentAlpha);
       glTranslatef(0.25*w(), 0.075*h(), 0);
       glScalef(15., 15., 1);
       glfDrawSolidString(string);

    glPopMatrix();

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawBBox                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawBBox(void)
{

    int j, node1, node2, node3;
    float vec[3], rgb1[3], rgb2[3], rgb3[3];

    // Draw triangles as wire frames

    rgb1[0] = 1.;
    rgb1[1] = 0.;
    rgb1[2] = 0.;

    rgb2[0] = 0.;
    rgb2[1] = 1.;
    rgb2[2] = 0.;

    rgb3[0] = 1.;
    rgb3[1] = 1.;
    rgb3[2] = 1.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glLineWidth(5.);
    glDisable(GL_LIGHTING);

    // Draw the bounding box

       glBegin(GL_QUADS);

          // Quad 1

             glColor3fv(rgb1);

             vec[0] = XMin;
             vec[1] = YMin;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMin;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMin;
             vec[2] = ZMax;

             glVertex3fv(vec);


             vec[0] = XMin;
             vec[1] = YMin;
             vec[2] = ZMax;

             glVertex3fv(vec);

          // Quad 2

             glColor3fv(rgb2);

             vec[0] = XMin;
             vec[1] = YMax;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMax;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMax;
             vec[2] = ZMax;

             glVertex3fv(vec);


             vec[0] = XMin;
             vec[1] = YMax;
             vec[2] = ZMax;

             glVertex3fv(vec);

          // Quad 3

             glColor3fv(rgb2);

             vec[0] = XMax;
             vec[1] = YMin;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMax;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMax;
             vec[1] = YMax;
             vec[2] = ZMax;

             glVertex3fv(vec);


             vec[0] = XMax;
             vec[1] = YMin;
             vec[2] = ZMax;

             glVertex3fv(vec);

          // Quad 4

             glColor3fv(rgb3);

             vec[0] = XMin;
             vec[1] = YMin;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMin;
             vec[1] = YMax;
             vec[2] = ZMin;

             glVertex3fv(vec);

             vec[0] = XMin;
             vec[1] = YMax;
             vec[2] = ZMax;

             glVertex3fv(vec);


             vec[0] = XMin;
             vec[1] = YMin;
             vec[2] = ZMax;

             glVertex3fv(vec);

       glEnd();

    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawWireFrame                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawWireFrame(void)
{

    int j, node1, node2, node3, SurfaceID, SurfID;
    float vec1[3], vec2[3], vec3[3], rgb[3];

    // Draw triangles as wire frames

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor3fv(rgb);
    glLineWidth(1.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    for ( j = 1 ; j <= NumberOfTris; j++ ) {

       SurfaceID = TriList[j].surface_id;
       
       SurfID = SurfaceID;
       
       if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

          node1 = TriList[j].node1;
          node2 = TriList[j].node2;
          node3 = TriList[j].node3;
          
          vec1[0] = NodeList[node1].x;
          vec1[1] = NodeList[node1].y;
          vec1[2] = NodeList[node1].z;

          vec2[0] = NodeList[node2].x;
          vec2[1] = NodeList[node2].y;
          vec2[2] = NodeList[node2].z;

          vec3[0] = NodeList[node3].x;
          vec3[1] = NodeList[node3].y;
          vec3[2] = NodeList[node3].z;             
          
          if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {

             RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
             RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
             RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
        
          }     

          glBegin(GL_TRIANGLES);
   
             glVertex3fv(vec1);

             glVertex3fv(vec2);

             glVertex3fv(vec3);
   
          glEnd();
          
          if ( DrawReflectedGeometryIsOn ) {

             vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
             vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
             vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;

             glBegin(GL_TRIANGLES);

                glVertex3fv(vec1);

                glVertex3fv(vec2);

                glVertex3fv(vec3);

             glEnd();
   
		     }
        
       }

    }

    glDisable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(1.,1.);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawWireFrame                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCoarseMeshEdgesForLevel(int Level)
{

    int j, node1, node2, node3, SurfaceID, SurfID;
    float vec[3], rgb[3];

    // Draw triangles as wire frames

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 1.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor3fv(rgb);
    glLineWidth(1.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    for ( j = 1 ; j <= NumberOfCourseEdgesForLevel[Level]; j++ ) {

      SurfaceID = CoarseEdgeList[Level][j].SurfaceID;

      if ( CoarseEdgeList[Level][j].IsBoundaryEdge && !CoarseEdgeList[Level][j].IsKuttaEdge) {

          rgb[0] = 1.;
          rgb[1] = 0.;
          rgb[2] = 0.;
      
          glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
          glColor3fv(rgb);         
         
         
      }
      
      else if ( CoarseEdgeList[Level][j].IsKuttaEdge ) {
         
          rgb[0] = 0.;
          rgb[1] = 1.;
          rgb[2] = 0.;
      
          glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
          glColor3fv(rgb);               
         
      }
      
      else {
         
          rgb[0] = 0.;
          rgb[1] = 0.;
          rgb[2] = 1.;
      
          glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
          glColor3fv(rgb);         
         
      }         
                  
      if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfaceID]) == 2 ) {

       glBegin(GL_LINES);

          node1 = CoarseEdgeList[Level][j].node1;
          node2 = CoarseEdgeList[Level][j].node2;

          vec[0] = CoarseNodeList[Level][node1].x;
          vec[1] = CoarseNodeList[Level][node1].y;
          vec[2] = CoarseNodeList[Level][node1].z;

          glVertex3fv(vec);

          vec[0] = CoarseNodeList[Level][node2].x;
          vec[1] = CoarseNodeList[Level][node2].y;
          vec[2] = CoarseNodeList[Level][node2].z;

          glVertex3fv(vec);

       glEnd();
       
       if ( DrawReflectedGeometryIsOn ) {

          glBegin(GL_LINES);

             node1 = CoarseEdgeList[Level][j].node1;
             node2 = CoarseEdgeList[Level][j].node2;

             vec[0] = CoarseNodeList[Level][node1].x;
             vec[1] = CoarseNodeList[Level][node1].y;
             vec[2] = CoarseNodeList[Level][node1].z;
             
             vec[1] = -(vec[1] + GeometryYShift) - GeometryYShift;
             
             glVertex3fv(vec);

             vec[0] = CoarseNodeList[Level][node2].x;
             vec[1] = CoarseNodeList[Level][node2].y;
             vec[2] = CoarseNodeList[Level][node2].z;
             
             vec[1] = -(vec[1] + GeometryYShift) - GeometryYShift;
             
             glVertex3fv(vec);                
          
          glEnd();

        }          
          
      }

    }

    glDisable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(1.,1.);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                GL_VIEWER DrawCoarseMeshNodesForLevel                         #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCoarseMeshNodesForLevel(int Level)
{

    int j, node1, node2, node3, SurfaceID, SurfID;
    float vec[3], rgb[3];

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glPointSize(4.);
    glDisable(GL_LIGHTING);

    rgb[0] = 1.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    glColor3fv(rgb);
    
    for ( j = 1 ; j <= NumberOfCourseNodesForLevel[Level]; j++ ) {
       
       SurfID = CoarseNodeList[Level][j].SurfID;
       
       if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

          vec[0] = CoarseNodeList[Level][j].x;
          vec[1] = CoarseNodeList[Level][j].y;
          vec[2] = CoarseNodeList[Level][j].z;
             
          glBegin(GL_POINTS);
   
             glVertex3fv(vec);
   
          glEnd();
          
          if ( DrawReflectedGeometryIsOn ) {
   
             glBegin(GL_POINTS);
   
                vec[1] = -(vec[1] + GeometryYShift) - GeometryYShift;
    
                glVertex3fv(vec);                
             
             glEnd();

           }   
           
       }
   
    }
    
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawWakes                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawWakes(void)
{

    int i, j, node1, node2, node3, NumberNodes, SurfID;
    float vec[3], rgb[4], Alpha;

    // Draw triangles as wire frames

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;
    rgb[3] = 0.25;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor4fv(rgb);
    glLineWidth(1.);
    glDisable(GL_LIGHTING);
    glPointSize(2.);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    NumberNodes = NumberOfSubVortexNodes_ - 1;
    
    if ( TimeAccurate_ ) NumberNodes = MAX(0, NumberOfSubVortexNodes_ - 1);
    
   //     if ( TimeAccurate_ ) NumberNodes = NumberOfSubVortexNodes_;

    if ( DrawWakesToInfinityIsOn ) NumberNodes = NumberOfSubVortexNodes_;
     
    for ( i = 1 ; i <= NumberOfTrailingVortexEdges_; i++ ) {

       SurfID = WingWake_[i];

       if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {
 
          if ( DrawWakeLinesIsOn ) {
                 
             glBegin(GL_LINE_STRIP);
                 
                for ( j = 1 ; j <= NumberNodes; j++ ) {
                   
                   Alpha = 1.;
                   
                   Alpha = (float) j / (float) (NumberOfSubVortexNodes_-1);
                   
                   Alpha = 1. - pow(Alpha,0.5);
                   
                   Alpha = MAX(Alpha,0.1);
                   
                   rgb[0] = 0.;
                   rgb[1] = 0.;
                   rgb[2] = 0.;
                   rgb[3] = Alpha;
                       
                   glColor4fv(rgb);
                       
                   if ( DrawWakesColored_ == 1 ) SetTagSurfaceColor(WingWake_[i], MaxWings_, Alpha);
                   
                   if ( DrawWakesColored_ == 2 ) { percent_to_rgb(SWake_[i],rgb,0); glColor4fv(rgb); };
   
                   vec[0] = XWake_[i][j];
                   vec[1] = YWake_[i][j];
                   vec[2] = ZWake_[i][j];
                      
                   glVertex3fv(vec);
      
                }
                
             glEnd();
             
          }
          
          if ( DrawWakePointsIsOn ) {
             
             rgb[0] = 0.;
             rgb[1] = 0.;
             rgb[2] = 0.;
             rgb[3] = 0.5;
         
             glColor4fv(rgb);
             glPointSize(2.);
                 
             glBegin(GL_POINTS);
   
                for ( j = 1 ; j <= NumberNodes; j++ ) {
   
                   vec[0] = XWake_[i][j];
                   vec[1] = YWake_[i][j];
                   vec[2] = ZWake_[i][j];
                      
                   glVertex3fv(vec);
      
                }
                
                glVertex3fv(vec);                
             
             glEnd();
             
          }
                 
          if ( DrawReflectedGeometryIsOn ) {
   
             if ( DrawWakeLinesIsOn ) { 
                
                glBegin(GL_LINE_STRIP);
                 
                for ( j = 1 ; j <= NumberNodes; j++ ) {
                      
                      Alpha = 1.;
                      
                      Alpha = (float) j / (float) (NumberOfSubVortexNodes_-1);
                      
                      Alpha = 1. - pow(Alpha,0.5);
                      
                      Alpha = MAX(Alpha,0.1);
                      
                      rgb[0] = 0.;
                      rgb[1] = 0.;
                      rgb[2] = 0.;
                      rgb[3] = Alpha;
                          
                      glColor4fv(rgb);
                          
                      if ( DrawWakesColored_ == 1 ) SetTagSurfaceColor(WingWake_[i], MaxWings_, Alpha);
                      
                      if ( DrawWakesColored_ == 2 ) { percent_to_rgb(SWake_[i],rgb,0); glColor4fv(rgb); };
         
                      vec[0] = XWake_[i][j];
                      vec[1] = YWake_[i][j];
                      vec[2] = ZWake_[i][j];
                   
                      vec[1] = -(vec[1] + GeometryYShift) - GeometryYShift;
                   
                      glVertex3fv(vec);
      
                   }
                
                glEnd();
                
             }
             
             if ( DrawWakePointsIsOn ) {
                
                rgb[0] = 0.;
                rgb[1] = 0.;
                rgb[2] = 0.;
                rgb[3] = 0.5;
             
                glColor4fv(rgb);
                glPointSize(2.);
                                
                glBegin(GL_POINTS);
      
                   for ( j = 1 ; j <= NumberNodes; j++ ) {
      
                      vec[0] = XWake_[i][j];
                      vec[1] = YWake_[i][j];
                      vec[2] = ZWake_[i][j];
                      
                      vec[1] = -(vec[1] + GeometryYShift) - GeometryYShift;
                         
                      glVertex3fv(vec);
         
                   }
                   
                   glVertex3fv(vec);                
                
                glEnd();
                
             }                
   
          } 
           
       }  
           
    }

    glDisable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(1.,1.);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                       GL_VIEWER DrawControlSurfaces                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawControlSurfaces(void)
{

    int j, k, node1, node2, node3, SurfaceID, SurfID;
    float vec[3], rgb[3];

    // Draw triangles as wire frames

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor3fv(rgb);
    glLineWidth(3.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    for ( j = 1 ; j <= NumberOfControlSurfaces; j++ ) {

       glBegin(GL_LINE_STRIP);
       
          for ( k = 1 ; k <= ControlSurface[j].NumberOfNodes ; k++ ) {
             
             glVertex3fv(ControlSurface[j].NodeList[k]);
   
          }       

       glEnd();
          
    }

    glDisable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(1.,1.);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                           GL_VIEWER DrawBEAM3DFEM                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawBEAM3DFEM(void)
{

    int i, j, k, node1, node2, node3, SurfaceID, SurfID;
    float vec[3], rgb[3];

    // Draw triangles as wire frames

    rgb[0] = 1.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor3fv(rgb);
    glLineWidth(3.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    for ( i = 1 ; i <= NumberOfSurfaces_; i++ ) {

       if ( FemData(i).NumberOfFEMNodes() > 0 ) {
          
          glBegin(GL_LINE_STRIP);
          
             for ( j = 1 ; j <= FemData(i).NumberOfFEMNodes() ; j++ ) {
                
                if ( DrawBEAM3DFEMIsOn == 1 ) {
                   
                   vec[0] = FemData(i).x(j) - GeometryXShift;
                   vec[1] = FemData(i).y(j) - GeometryYShift;
                   vec[2] = FemData(i).z(j) - GeometryZShift;
                   
                }
                
                else {

                   vec[0] = FemData(i).x(j) + FemData(i).delta_xv(j) - GeometryXShift;
                   vec[1] = FemData(i).y(j) + FemData(i).delta_yv(j) - GeometryYShift;
                   vec[2] = FemData(i).z(j) + FemData(i).delta_zv(j) - GeometryZShift;
                   
                }

                glVertex3fv(vec);
      
             }       
   
          glEnd();
          
       }
          
    }

    glDisable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(1.,1.);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#               GL_VIEWER DrawCurrentSelectedComGeomRegions                    #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCurrentSelectedComGeomRegions(void)
{

    int j, node1, node2, tri1, tri2;
    float vec[3];
    GLfloat rgb[3];

    // Set up picking stuff

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glLineWidth(5.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    rgb[0] = 0.;
    rgb[1] = 1.;
    rgb[2] = 0.;

    glColor3fv(rgb);

    for ( j = 1 ; j <= NumberOfEdges ; j++ ) {

       tri1 = EdgeList[j].tri1;
       tri2 = EdgeList[j].tri2;

       if ( PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[tri1]) ||
            PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[tri2]) ) {

          if ( ComGeom2PanelTag[tri1] != ComGeom2PanelTag[tri2] ) {

             glBegin(GL_LINES);

                node1 = EdgeList[j].node1;
                node2 = EdgeList[j].node2;

                vec[0] = NodeList[node1].x;
                vec[1] = NodeList[node1].y;
                vec[2] = NodeList[node1].z;

                glVertex3fv(vec);

                vec[0] = NodeList[node2].x;
                vec[1] = NodeList[node2].y;
                vec[2] = NodeList[node2].z;

                glVertex3fv(vec);

             glEnd();

		  }

	   }

    }

    glLineWidth(1.);

}

/*##############################################################################
#                                                                              #
#                   GL_VIEWER PanelComGeomTagsBrowser_UnSelectAll              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::PanelComGeomTagsBrowser_UnSelectAll(void)
{

    // Unselect all the panels

    PanelComGeomTagsBrowser->deselect();

}

/*##############################################################################
#                                                                              #
#                   GL_VIEWER PanelComGeomTagsBrowser_SelectAll                #
#                                                                              #
##############################################################################*/

void GL_VIEWER::PanelComGeomTagsBrowser_SelectAll(void)
{

    int i;

    // Unselect all the panels

    for ( i = 1 ; i <= PanelComGeomTagsBrowser->size(); i++ ) {

       PanelComGeomTagsBrowser->select(i);

    }

}

/*##############################################################################
#                                                                              #
#                   GL_VIEWER PanelComGeomTagsBrowser_Update                   #
#                                                                              #
##############################################################################*/

void GL_VIEWER::PanelComGeomTagsBrowser_Update(void)
{

    int i, Type, IndependentPanelType, NumItems;
    float Emissivity;
    char Line[200];

    PanelComGeomTagsBrowser->clear();

    NumItems = 0;
    
    for ( i = 1 ; i <= NumberOfWings_ ; i++ ) {

       sprintf(Line,"(%d) Wing: (%d): %-100s \n", WingGroupID_[i],i,WingListName_[i]);

       PanelComGeomTagsBrowser->add(Line);
       
       ComGeom2PanelTag[++NumItems] = i;

    }
    
    for ( i = 1 ; i <= NumberOfBodies_ ; i++ ) {

       sprintf(Line,"(%d) Body: (%d): %-100s \n", BodyGroupID_[i], i,BodyListName_[i]);

       PanelComGeomTagsBrowser->add(Line);
       
       ComGeom2PanelTag[++NumItems] = NumberOfWings_ + i;

    }    

    for ( i = 1 ; i <= NumberOfCart3dSurfaces_ ; i++ ) {

       sprintf(Line,"(%d): %-100s \n",i, Cart3dListName_[i]);

       PanelComGeomTagsBrowser->add(Line);
       
       ComGeom2PanelTag[++NumItems] = NumberOfBodies_ + NumberOfWings_ + i;

    } 

    printf("NumItems: %d \n",NumItems);fflush(NULL);

    // Add and select a blank line at the end of the list

    PanelComGeomTagsBrowser->add("");

    PanelComGeomTagsBrowser->select(NumberOfWings_+1);

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER DrawShadedSurface                       #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawShadedSurface(void)
{

    int j, node1, node2, node3, LastTri, LastCon, LastSurface;
    int LastMaterialType, SurfaceID, SurfID;
    float vec1[3], vec2[3], vec3[3], rgb[4], LastEmissivity;

    GLfloat ambient1[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };
    GLfloat ambient2[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };

    GLfloat ambient3[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };
    GLfloat ambient4[] = { 2.0f*Brightness, 2.0f*Brightness, 2.0f*Brightness, 1.0f };

    // Modify lighting for just simple shaded surface... brighten up things some

/*    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glLightfv(GL_LIGHT0,GL_DIFFUSE,ambient3);
    glLightfv(GL_LIGHT1,GL_DIFFUSE,ambient4);
*/
    glShadeModel(GL_SMOOTH);

    // Draw triangles as shaded surface

    rgb[0] = 0.9;
    rgb[1] = 0.9;
    rgb[2] = 0.9;
    rgb[3] = 1.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
    glColor3fv(rgb);

    glEnable(GL_POLYGON_OFFSET_FILL);
    glPolygonOffset(1.,0.);

    // Draw the surface triangles
    
    LastTri = 0;

    for ( j = 1 ; j <= NumberOfTris; j++ ) {

       // Draw it according to its surface ID flag

       SurfaceID = TriList[j].surface_id;
       
       SurfID = SurfaceID;

       if ( ComGeom2PanelTag[SurfID] != 0 &&
                 DrawComGeomTagsIsOn &&
                 DrawComGeomTagsShadedIsOn &&
                 PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) ) {
       
          if ( LastTri != GOMGEOM_TRI || LastSurface != SurfID ) {

             SetTagSurfaceColor(SurfID,2*(NumberOfBodies_ + NumberOfWings_));

             LastTri = GOMGEOM_TRI;

             LastSurface = SurfID;

          }

       }

       else {

          if ( LastTri != SRF_TRI ) {

             rgb[0] = 0.9;
             rgb[1] = 0.9;
             rgb[2] = 0.9;
             rgb[3] = 1.00;

             glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
             glColor3fv(rgb);

             LastTri = SRF_TRI;

          }

       }

       if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

          node1 = TriList[j].node1;
          node2 = TriList[j].node2;
          node3 = TriList[j].node3;
          
          vec1[0] = NodeList[node1].x;
          vec1[1] = NodeList[node1].y;
          vec1[2] = NodeList[node1].z;

          vec2[0] = NodeList[node2].x;
          vec2[1] = NodeList[node2].y;
          vec2[2] = NodeList[node2].z;

          vec3[0] = NodeList[node3].x;
          vec3[1] = NodeList[node3].y;
          vec3[2] = NodeList[node3].z;             
          
          if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {
 
             RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
             RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
             RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
          
          }   
          
          if ( !DrawSmoothShadeIsOn ) {
   
             glBegin(GL_TRIANGLES);
   
                glNormal3f( Nx[j], Ny[j], Nz[j] );
  
                glVertex3fv(vec1);

                glVertex3fv(vec2);

                glVertex3fv(vec3);
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {

                vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;

                glBegin(GL_TRIANGLES);
   
                   glNormal3f( Nx[j], -Ny[j], Nz[j] );

                   glVertex3fv(vec1);
   
                   glVertex3fv(vec2);
   
                   glVertex3fv(vec3);
   
                glEnd();
   
		     }
   
          }
   
          else {

             node1 = TriList[j].node1;
             node2 = TriList[j].node2;
             node3 = TriList[j].node3;
             
             vec1[0] = NodeList[node1].x;
             vec1[1] = NodeList[node1].y;
             vec1[2] = NodeList[node1].z;
   
             vec2[0] = NodeList[node2].x;
             vec2[1] = NodeList[node2].y;
             vec2[2] = NodeList[node2].z;
   
             vec3[0] = NodeList[node3].x;
             vec3[1] = NodeList[node3].y;
             vec3[2] = NodeList[node3].z;       
          
             glBegin(GL_TRIANGLES);
   
                glNormal3f( Nodal_Nx[node1], Nodal_Ny[node1], Nodal_Nz[node1] );
                glVertex3fv(vec1);
   
                glNormal3f( Nodal_Nx[node2], Nodal_Ny[node2], Nodal_Nz[node2] );
                glVertex3fv(vec2);
   
                glNormal3f( Nodal_Nx[node3], Nodal_Ny[node3], Nodal_Nz[node3] );
                glVertex3fv(vec3);
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {

                vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;
                
                glBegin(GL_TRIANGLES);

                   glNormal3f( Nodal_Nx[node1], -Nodal_Ny[node1], Nodal_Nz[node1] );
                   glVertex3fv(vec1);
   
                   glNormal3f( Nodal_Nx[node2], -Nodal_Ny[node2], Nodal_Nz[node2] );
                   glVertex3fv(vec2);
  
   
                   glNormal3f( Nodal_Nx[node3], -Nodal_Ny[node3], Nodal_Nz[node3] );
                   glVertex3fv(vec3);
   
                glEnd();
   
		     }
   
	      }

       }

    }

    if ( DrawXPlaneIsOn || DrawYPlaneIsOn || DrawZPlaneIsOn ) DrawSymmetryPlane();

    glDisable(GL_POLYGON_OFFSET_FILL);

    // Enable Lighting
/*
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glLightfv(GL_LIGHT0,GL_DIFFUSE,ambient1);
    glLightfv(GL_LIGHT1,GL_DIFFUSE,ambient2);
*/
}

/*##############################################################################
#                          DrawShadedSolution                                  #
#                             GL_VIEWER                                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawShadedSolution(float *Function, float FMin, float FMax)
{

    int i, node1, node2, node3;
    float Area, NewMin, NewMax;

    // Per tri contour shading
    
    if ( DrawPerTriFunctionIsOn ) {

       DrawShadedSolutionPerTri(Function, FMin, FMax);

    }

    // Line contours
    
    else if ( DrawLineContoursIsOn ) {
       
       DrawShadedLineContours(Function, FMin, FMax);
       
    }

    // Smooth shading of contours, node averaging
    
    else if ( DrawSmoothFunctionsIsOn ) {

       if ( LastFunction != Function ) {

          for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

	         TempNodalArray[i] = 0.;

	         TempTotalArea[i] = 0.;

	      }

          for ( i = 1 ; i <= NumberOfTris ; i++ ) {

             node1 = TriList[i].node1;
             node2 = TriList[i].node2;
             node3 = TriList[i].node3;

             Area = TriList[i].area;

	          TempNodalArray[node1] += Function[i] * Area;
	          TempNodalArray[node2] += Function[i] * Area;
	          TempNodalArray[node3] += Function[i] * Area;

	          TempTotalArea[node1] += Area;
	          TempTotalArea[node2] += Area;
	          TempTotalArea[node3] += Area;

          }

	       NodalMin = 1.e9;
	       NodalMax = -NodalMin;

          for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

	         if ( TempTotalArea[i] > 0. ) {
              
                TempNodalArray[i] /= TempTotalArea[i];
                
             }
             
             else {
              
                TempTotalArea[i] = 0.;
             
             }

	         NodalMin = MIN(NodalMin,TempNodalArray[i]);
	         NodalMax = MAX(NodalMax,TempNodalArray[i]);

	      }

	      LastFunction = Function;

	   }

      DrawShadedSolutionPerNode(TempNodalArray, FMin, FMax);

	}

}

/*##############################################################################
#                                                                              #
#                          DrawShadedLineContours                              #
#                             GL_VIEWER                                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawShadedLineContours(float *Function, float FMin, float FMax)
{

    int c, i, j, node1, node2, node3, NumContourLevels;
    float Area, NewMin, NewMax, Vec[3], Value, Per2RGB, Per[3], v1, v2;
    float xyz1[3], xyz2[3];
    float rgb[3];
    
    DrawShadedSurface();
    
    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glColor3fv(rgb);
    glLineWidth(1.);
    glDisable(GL_LIGHTING);

    // Draw the surface triangles

    glEnable(GL_POLYGON_OFFSET_LINE);

    glPolygonOffset(0.,-10.);

    // Create nodal values
    
    if ( LastFunction != Function ) {

       for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

         TempNodalArray[i] = 0.;

         TempTotalArea[i] = 0.;

      }

       for ( i = 1 ; i <= NumberOfTris ; i++ ) {

          node1 = TriList[i].node1;
          node2 = TriList[i].node2;
          node3 = TriList[i].node3;

          Area = TriList[i].area;

          TempNodalArray[node1] += Function[i] * Area;
          TempNodalArray[node2] += Function[i] * Area;
          TempNodalArray[node3] += Function[i] * Area;

          TempTotalArea[node1] += Area;
          TempTotalArea[node2] += Area;
          TempTotalArea[node3] += Area;

       }

       NodalMin = 1.e9;
       NodalMax = -NodalMin;

       for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

         if ( TempTotalArea[i] > 0. ) {
           
             TempNodalArray[i] /= TempTotalArea[i];
             
          }
          
          else {
           
             TempTotalArea[i] = 0.;
          
          }

         NodalMin = MIN(NodalMin,TempNodalArray[i]);
         NodalMax = MAX(NodalMax,TempNodalArray[i]);

      }

      LastFunction = Function;

   }

   // Loop over contour values and draw some lines
   
   for ( c = 1 ; c <= NumberOfLineContourLevels ; c++ ) {
      
       Value = FMin + (c-1)*( FMax - FMin )/(NumberOfLineContourLevels-1);

       if ( FMax != FMin ) {
   
          Per2RGB = (Value - FMin)/(FMax - FMin);
   
       }
   
       else {
   
          Per2RGB = 1.;
   
       }
   
       percent_to_rgb(Per2RGB, rgb, 0);
 
       for ( i = 1 ; i <= NumberOfTris ; i++ ) {
      
          node1 = TriList[i].node1;
          node2 = TriList[i].node2;
          node3 = TriList[i].node3;
      
          Vec[0] = TempNodalArray[node1];
          Vec[1] = TempNodalArray[node2];
          Vec[2] = TempNodalArray[node3];
          
          if ( Value >= MIN3(Vec[0],Vec[1],Vec[2]) && Value <= MAX3(Vec[0],Vec[1],Vec[2]) ) {
     
             for ( j = 0 ; j <= 2 ; j++ ) {
                
                if ( j == 0 ) { v1 = Vec[0]; v2 = Vec[1]; };
                if ( j == 1 ) { v1 = Vec[1]; v2 = Vec[2]; };
                if ( j == 2 ) { v1 = Vec[2]; v2 = Vec[0]; };
                
                Per[j] = -1.;
                
                if ( v1 - v2 != 0. ) {
                   
                   Per[j] = (Value - v1)/(v2 - v1);
                
                }
                
             }
             
             glColor3fv(rgb);

             if ( ( Per[0] >= 0. && Per[0] <= 1. ) && ( Per[1] >= 0. && Per[1] <= 1. ) ) {
                
                xyz1[0] = NodeList[node1].x + Per[0]*(NodeList[node2].x - NodeList[node1].x);
                xyz1[1] = NodeList[node1].y + Per[0]*(NodeList[node2].y - NodeList[node1].y);
                xyz1[2] = NodeList[node1].z + Per[0]*(NodeList[node2].z - NodeList[node1].z);
                                                             
                xyz2[0] = NodeList[node2].x + Per[1]*(NodeList[node3].x - NodeList[node2].x);
                xyz2[1] = NodeList[node2].y + Per[1]*(NodeList[node3].y - NodeList[node2].y);
                xyz2[2] = NodeList[node2].z + Per[1]*(NodeList[node3].z - NodeList[node2].z);
           
                glBegin(GL_LINES);
               
                   glVertex3fv(xyz1);
               
                   glVertex3fv(xyz2);
        
                glEnd();
 
                if ( DrawReflectedGeometryIsOn ) {
        
                   glBegin(GL_LINES);
                   
                      xyz1[1] = -(xyz1[1] + GeometryYShift) - GeometryYShift;
                      xyz2[1] = -(xyz2[1] + GeometryYShift) - GeometryYShift;
                  
                      glVertex3fv(xyz1);
                  
                      glVertex3fv(xyz2);
           
                   glEnd();

                }
                            
             }
             
             else if ( ( Per[0] >= 0. && Per[0] <= 1. ) && ( Per[2] >= 0. && Per[2] <= 1. ) ) {
   
                xyz1[0] = NodeList[node1].x + Per[0]*(NodeList[node2].x - NodeList[node1].x);
                xyz1[1] = NodeList[node1].y + Per[0]*(NodeList[node2].y - NodeList[node1].y);
                xyz1[2] = NodeList[node1].z + Per[0]*(NodeList[node2].z - NodeList[node1].z);
                                                              
                xyz2[0] = NodeList[node3].x + Per[2]*(NodeList[node1].x - NodeList[node3].x);
                xyz2[1] = NodeList[node3].y + Per[2]*(NodeList[node1].y - NodeList[node3].y);
                xyz2[2] = NodeList[node3].z + Per[2]*(NodeList[node1].z - NodeList[node3].z);
   
                glBegin(GL_LINES);
               
                   glVertex3fv(xyz1);
               
                   glVertex3fv(xyz2);
        
                glEnd();

                if ( DrawReflectedGeometryIsOn ) {
        
                   glBegin(GL_LINES);
                   
                      xyz1[1] = -(xyz1[1] + GeometryYShift) - GeometryYShift;
                      xyz2[1] = -(xyz2[1] + GeometryYShift) - GeometryYShift;
                  
                      glVertex3fv(xyz1);
                  
                      glVertex3fv(xyz2);
           
                   glEnd();

                }
                                             
             }
             
             else if ( ( Per[1] >= 0. && Per[1] <= 1. ) && ( Per[2] >= 0. && Per[2] <= 1. ) ) {
                
                xyz1[0] = NodeList[node2].x + Per[1]*(NodeList[node3].x - NodeList[node2].x);
                xyz1[1] = NodeList[node2].y + Per[1]*(NodeList[node3].y - NodeList[node2].y);
                xyz1[2] = NodeList[node2].z + Per[1]*(NodeList[node3].z - NodeList[node2].z);
                                                              
                xyz2[0] = NodeList[node3].x + Per[2]*(NodeList[node1].x - NodeList[node3].x);
                xyz2[1] = NodeList[node3].y + Per[2]*(NodeList[node1].y - NodeList[node3].y);
                xyz2[2] = NodeList[node3].z + Per[2]*(NodeList[node1].z - NodeList[node3].z);
   
                glBegin(GL_LINES);
               
                   glVertex3fv(xyz1);
               
                   glVertex3fv(xyz2);
        
                glEnd();

                if ( DrawReflectedGeometryIsOn ) {
        
                   glBegin(GL_LINES);
                   
                      xyz1[1] = -(xyz1[1] + GeometryYShift) - GeometryYShift;
                      xyz2[1] = -(xyz2[1] + GeometryYShift) - GeometryYShift;
                  
                      glVertex3fv(xyz1);
                  
                      glVertex3fv(xyz2);
           
                   glEnd();

                }
                                             
             }
          
          }          
          
       }
       
   }
   
}

/*##############################################################################
#                                                                              #
#                         GL_VIEWER DrawShadedSolutionPerTri                   #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawShadedSolutionPerTri(float *Function, float FMin, float FMax)
{

    int j, node1, node2, node3, SurfaceID, SurfID;
    float vec1[3], vec2[3], vec3[3], rgb[4], per;

    // Draw triangles as function shaded surface

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glEnable(GL_POLYGON_OFFSET_FILL);
    glPolygonOffset(1.,0.);

    // Draw the surface triangles

    rgb[3] = 1.;

    if ( !DrawSmoothShadeIsOn ) {

       for ( j = 1 ; j <= NumberOfTris; j++ ) {

          SurfaceID = TriList[j].surface_id;
       
          SurfID = SurfaceID;
               
          if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

             node1 = TriList[j].node1;
             node2 = TriList[j].node2;
             node3 = TriList[j].node3;
   
             if ( FMax != FMin ) {
   
                per = (Function[j] - FMin)/(FMax - FMin);
   
             }
   
             else {
   
                per = 1.;
   
             }
   
             percent_to_rgb(per, rgb, 0);
   
             glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
             glColor3fv(rgb);
             
             vec1[0] = NodeList[node1].x;
             vec1[1] = NodeList[node1].y;
             vec1[2] = NodeList[node1].z;

             vec2[0] = NodeList[node2].x;
             vec2[1] = NodeList[node2].y;
             vec2[2] = NodeList[node2].z;

             vec3[0] = NodeList[node3].x;
             vec3[1] = NodeList[node3].y;
             vec3[2] = NodeList[node3].z;             
             
             if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {
   
                RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
             
             }       
   
             glBegin(GL_TRIANGLES);
   
                if ( !DrawFlatShadedIsOn ) glNormal3f( Nx[j], Ny[j], Nz[j] );
   
                glVertex3fv(vec1);
   
                glVertex3fv(vec2);
   
                glVertex3fv(vec3);
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {
   
                glBegin(GL_TRIANGLES);
   
                   if ( !DrawFlatShadedIsOn ) glNormal3f( Nx[j], -Ny[j], Nz[j] );
   
                   node1 = TriList[j].node1;
                   node2 = TriList[j].node2;
                   node3 = TriList[j].node3;
   
                   vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                   vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                   vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;
   
                   glVertex3fv(vec1);
   
                   glVertex3fv(vec2);
   
                   glVertex3fv(vec3);
   
                glEnd();
   
             }

          }

       }

    }

    else {

       for ( j = 1 ; j <= NumberOfTris; j++ ) {

          SurfaceID = TriList[j].surface_id;
       
          SurfID = SurfaceID;
        
          if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

             node1 = TriList[j].node1;
             node2 = TriList[j].node2;
             node3 = TriList[j].node3;
   
             if ( FMax != FMin ) {
   
                per = (Function[j] - FMin)/(FMax - FMin);
   
             }
   
             else {
   
                per = 1.;
   
             }
   
             percent_to_rgb(per, rgb, 0);
   
             glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
             glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
             glColor3fv(rgb);
   
             vec1[0] = NodeList[node1].x;
             vec1[1] = NodeList[node1].y;
             vec1[2] = NodeList[node1].z;

             vec2[0] = NodeList[node2].x;
             vec2[1] = NodeList[node2].y;
             vec2[2] = NodeList[node2].z;

             vec3[0] = NodeList[node3].x;
             vec3[1] = NodeList[node3].y;
             vec3[2] = NodeList[node3].z;             
             
             if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {
   
                RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
             
             }       
                
             glBegin(GL_TRIANGLES);

                glNormal3f( Nodal_Nx[node1], Nodal_Ny[node1], Nodal_Nz[node1] );
                glVertex3fv(vec1);

                glNormal3f( Nodal_Nx[node2], Nodal_Ny[node2], Nodal_Nz[node2] );
                glVertex3fv(vec2);

                glNormal3f( Nodal_Nx[node3], Nodal_Ny[node3], Nodal_Nz[node3] );
                glVertex3fv(vec3);
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {
   
                glBegin(GL_TRIANGLES);
   
                   vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                   vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                   vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;
           
                   glNormal3f( Nodal_Nx[node1], -Nodal_Ny[node1], Nodal_Nz[node1] );
                   glVertex3fv(vec1);
   
                   glNormal3f( Nodal_Nx[node2], -Nodal_Ny[node2], Nodal_Nz[node2] );
                   glVertex3fv(vec2);
   
                   glNormal3f( Nodal_Nx[node3], -Nodal_Ny[node3], Nodal_Nz[node3] );
                   glVertex3fv(vec3);
   
                glEnd();
   
             }

           }

	   }

    }

    if ( DrawXPlaneIsOn || DrawYPlaneIsOn || DrawZPlaneIsOn ) DrawSymmetryPlane();

    glDisable(GL_POLYGON_OFFSET_FILL);

}

/*##############################################################################
#                                                                              #
#                    GL_VIEWER DrawShadedSolutionPerNode                       #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawShadedSolutionPerNode(float *Function, float FMin, float FMax)
{

    int j, k, node[3], SurfaceID, SurfID;
    float vec1[3], vec2[3], vec3[3], rgb[4], per;

    // Draw triangles as function shaded surface

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glEnable(GL_POLYGON_OFFSET_FILL);
    glPolygonOffset(1.,0.);

    // Draw the surface triangles

    rgb[3] = 1.;

    if ( !DrawSmoothShadeIsOn ) {

       for ( j = 1 ; j <= NumberOfTris; j++ ) {

          SurfaceID = TriList[j].surface_id;
       
          SurfID = SurfaceID;
 
          if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

             node[0] = TriList[j].node1;
             node[1] = TriList[j].node2;
             node[2] = TriList[j].node3;
   
             vec1[0] = NodeList[node[0]].x;
             vec1[1] = NodeList[node[0]].y;
             vec1[2] = NodeList[node[0]].z;

             vec2[0] = NodeList[node[1]].x;
             vec2[1] = NodeList[node[1]].y;
             vec2[2] = NodeList[node[1]].z;

             vec3[0] = NodeList[node[2]].x;
             vec3[1] = NodeList[node[2]].y;
             vec3[2] = NodeList[node[2]].z;             
             
             if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {
   
                RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
             
             }       
                
             glBegin(GL_TRIANGLES);
   
                glNormal3f( Nx[j], Ny[j], Nz[j] );
   
                for ( k = 0 ; k <= 2 ; k++ ) {
   
                   if ( FMax != FMin ) {
   
                      per = (Function[node[k]] - FMin)/(FMax - FMin);
   
                   }
   
                   else {
   
                      per = 1.;
   
                   }
   
                   percent_to_rgb(per, rgb, 0);
   
                   glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
                   glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
                   glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
                   glColor3fv(rgb);
   
                   if ( k == 0 ) glVertex3fv(vec1);
                   if ( k == 1 ) glVertex3fv(vec2);
                   if ( k == 2 ) glVertex3fv(vec3);
   
                }
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {
   
                glBegin(GL_TRIANGLES);
   
                   glNormal3f( Nx[j], -Ny[j], Nz[j] );
   
                   for ( k = 0 ; k <= 2 ; k++ ) {
   
                      if ( k == 0 ) vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                      if ( k == 1 ) vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                      if ( k == 2 ) vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;
   
                      if ( FMax != FMin ) {
   
                         per = (Function[node[k]] - FMin)/(FMax - FMin);
   
                      }
   
                      else {
   
                         per = 1.;
   
                      }
   
                      percent_to_rgb(per, rgb, 0);
   
                      glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
                      glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
                      glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
                      glColor3fv(rgb);
   
                      if ( k == 0 ) glVertex3fv(vec1);
                      if ( k == 1 ) glVertex3fv(vec2);
                      if ( k == 2 ) glVertex3fv(vec3);
   
                   }
   
                glEnd();
   
	         }

          }

       }

    }

    else {

       for ( j = 1 ; j <= NumberOfTris; j++ ) {

          SurfaceID = TriList[j].surface_id;
       
          SurfID = SurfaceID;
        
          if ( !DrawOnlySelectedIsOn || DrawOnlySelectedIsOn + PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) == 2 ) {

             node[0] = TriList[j].node1;
             node[1] = TriList[j].node2;
             node[2] = TriList[j].node3;
             
             vec1[0] = NodeList[node[0]].x;
             vec1[1] = NodeList[node[0]].y;
             vec1[2] = NodeList[node[0]].z;

             vec2[0] = NodeList[node[1]].x;
             vec2[1] = NodeList[node[1]].y;
             vec2[2] = NodeList[node[1]].z;

             vec3[0] = NodeList[node[2]].x;
             vec3[1] = NodeList[node[2]].y;
             vec3[2] = NodeList[node[2]].z;             
             
             if ( DrawControlSurfacesDeflectedIsOn && ControlSurfaceLoop[j] != 0 ) {
   
                RotateControlSurfaceNode( vec1, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec2, ControlSurfaceLoop[j] );
                RotateControlSurfaceNode( vec3, ControlSurfaceLoop[j] );
             
             }       
                     
             glBegin(GL_TRIANGLES);
   
                for ( k = 0 ; k <= 2 ; k++ ) {

                   if ( FMax != FMin ) {
   
                      per = (Function[node[k]] - FMin)/(FMax - FMin);
   
                   }
   
                   else {
   
                      per = 1.;
   
                   }
   
                   percent_to_rgb(per, rgb, 0);
   
                   glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
                   glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
                   glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
                   glColor3fv(rgb);
   
                   glNormal3f( Nodal_Nx[node[k]], Nodal_Ny[node[k]], Nodal_Nz[node[k]] );
   
                   if ( k == 0 ) glVertex3fv(vec1);
                   if ( k == 1 ) glVertex3fv(vec2);
                   if ( k == 2 ) glVertex3fv(vec3);   
                   
                }
   
             glEnd();
   
             if ( DrawReflectedGeometryIsOn ) {
   
                glBegin(GL_TRIANGLES);
   
                   for ( k = 0 ; k <= 2 ; k++ ) {
   
                      if ( k == 0 ) vec1[1] = -(vec1[1] + GeometryYShift) - GeometryYShift;
                      if ( k == 1 ) vec2[1] = -(vec2[1] + GeometryYShift) - GeometryYShift;
                      if ( k == 2 ) vec3[1] = -(vec3[1] + GeometryYShift) - GeometryYShift;
   
                      if ( FMax != FMin ) {
   
                         per = (Function[node[k]] - FMin)/(FMax - FMin);
   
                      }
   
                      else {
   
                         per = 1.;
   
                      }
   
                      percent_to_rgb(per, rgb, 0);
   
                      glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
                      glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
                      glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
                      glColor3fv(rgb);
   
                      glNormal3f( Nodal_Nx[node[k]], Nodal_Ny[node[k]], Nodal_Nz[node[k]] );
   
                      if ( k == 0 ) glVertex3fv(vec1);
                      if ( k == 1 ) glVertex3fv(vec2);
                      if ( k == 2 ) glVertex3fv(vec3);  
                      
                   }
   
                glEnd();
   
	         }

          }

       }

    }

    if ( DrawXPlaneIsOn || DrawYPlaneIsOn || DrawZPlaneIsOn ) DrawSymmetryPlane();

    glDisable(GL_POLYGON_OFFSET_FILL);

}

/*##############################################################################
#                                                                              #
#                                GL_VIEWER DrawCp                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCp(void)
{

    if ( ModelType == VLM_MODEL ) {
    
       sprintf(LegendTitle,"Delta-Cp");
       
    }
    
    else {
       
       sprintf(LegendTitle,"Cp");
       
    }

    LegendMin = CpMin;
    LegendMax = CpMax;

    LegendMinClip = CpMinActual;
    LegendMaxClip = CpMaxActual;
       
    // Draw shaded solution

    DrawShadedSolution(Cp, CpMin, CpMax);

}

/*##############################################################################
#                                                                              #
#                        GL_VIEWER DrawCpSteady                                #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCpSteady(void)
{

    if ( ModelType == VLM_MODEL ) {
    
       sprintf(LegendTitle,"Steady Delta-Cp");
       
    }
    
    else {
       
       sprintf(LegendTitle,"Steady Cp");
       
    }

    LegendMin = CpSteadyMin;
    LegendMax = CpSteadyMax;

    LegendMinClip = CpSteadyMinActual;
    LegendMaxClip = CpSteadyMaxActual;
       
    // Draw shaded solution

    DrawShadedSolution(CpSteady, CpSteadyMin, CpSteadyMax);

}

/*##############################################################################
#                                                                              #
#                        GL_VIEWER DrawCpUnsteady                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCpUnsteady(void)
{

    if ( ModelType == VLM_MODEL ) {
    
       sprintf(LegendTitle,"Unsteady Delta-Cp");
       
    }
    
    else {
       
       sprintf(LegendTitle,"Unsteady Cp");
       
    }

    LegendMin = CpUnsteadyMin;
    LegendMax = CpUnsteadyMax;

    LegendMinClip = CpUnsteadyMinActual;
    LegendMaxClip = CpUnsteadyMaxActual;
       
    // Draw shaded solution

    DrawShadedSolution(CpUnsteady, CpUnsteadyMin, CpUnsteadyMax);

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER DrawGamma                               #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawGamma(void)
{

    if ( ModelType == VLM_MODEL ) {
    
       sprintf(LegendTitle,"Vorticity");
       
    }
    
    else {
       
       sprintf(LegendTitle,"Vorticity");
       
    }

    LegendMin = GammaMin;
    LegendMax = GammaMax;

    LegendMinClip = GammaMinActual;
    LegendMaxClip = GammaMaxActual;
       
    // Draw shaded solution

    DrawShadedSolution(Gamma, GammaMin, GammaMax);

}

/*##############################################################################
#                                                                              #
#                       GL_VIEWER DrawRotorSurfacesShaded                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawRotorSurfacesShaded(void)
{

    int i, j, node1, node2, node3;
    float vec[3], cvec[3], rgb[4], prcent;

    // Draw triangles as shaded surface

    rgb[0] = 1.;
    rgb[1] = 0.5;
    rgb[2] = 0.5;

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
  //  glColor3fv(rgb);
  
    glLineWidth(1.);

    rgb[0] = 0.5;
    rgb[1] = 0.2;
    rgb[2] = 0.5;
    rgb[3] = 0.5;
    
    rgb[0] = 0.7;
    rgb[1] = 0.3;
    rgb[2] = 0.7;
    rgb[3] = 0.5;    
    
    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
   
    glEnable(GL_POLYGON_OFFSET_FILL);
    glPolygonOffset(1.,0.);
    
    // Loop over rotors
    
    for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {
     
       PropulsionElement[i].Rotor.CalculateRotorGeometry();
    
       // Loop over nodes defining the tips of each rotor and draw the rotor disk
     
       for ( j = 1 ; j < NUM_ROTOR_NODES ; j++ ) {

          glBegin(GL_TRIANGLES);

             glNormal3f( -PropulsionElement[i].Rotor.Normal(0),
                         -PropulsionElement[i].Rotor.Normal(1),
                         -PropulsionElement[i].Rotor.Normal(2) );

             vec[0] = PropulsionElement[i].Rotor.XYZ(0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.XYZ(1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.XYZ(2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,2) - GeometryZShift;
      
             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,2) - GeometryZShift;
      
             glVertex3fv(vec);

          glEnd();
        
          if ( DrawReflectedGeometryIsOn ) {

             glBegin(GL_TRIANGLES);
             
                glNormal3f( -PropulsionElement[i].Rotor.Normal(0),
                            -PropulsionElement[i].Rotor.Normal(1),
                            -PropulsionElement[i].Rotor.Normal(2) );
   
                vec[0] = PropulsionElement[i].Rotor.XYZ(0) - GeometryXShift;;
                vec[1] = -PropulsionElement[i].Rotor.XYZ(1) - GeometryYShift;;
                vec[2] = PropulsionElement[i].Rotor.XYZ(2) - GeometryZShift;;

                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,0) - GeometryXShift;;
                vec[1] = -PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,1) - GeometryYShift;;
                vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,2) - GeometryZShift;;
  
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,0) - GeometryXShift;;
                vec[1] = -PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,1) - GeometryYShift;;
                vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,2) - GeometryZShift;;
  
                glVertex3fv(vec);
                
             glEnd();             
             
          }  
          
       }
     
    }
    
    if ( DrawWithWhiteBackgroundIsOn ) {
     
       rgb[0] = 0.0;
       rgb[1] = 0.0;
       rgb[2] = 0.0;
       rgb[3] = 1.0;
       
    }
    
    else {
     
       rgb[0] = 1.0;
       rgb[1] = 1.0;
       rgb[2] = 1.0;
       rgb[3] = 1.0;
       
    }     
     
    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);

    for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {

       // Now outline the rotor, indicating the direction of rotation
       
       for ( j = 1 ; j < NUM_ROTOR_NODES ; j++ ) {

          glBegin(GL_LINES);

             vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,2) - GeometryZShift;
      
             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,2) - GeometryZShift;
      
             glVertex3fv(vec);
             
          glEnd();
                  
          if ( DrawReflectedGeometryIsOn ) {

             glBegin(GL_LINES);
   
                vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j  ,2) - GeometryZShift;
         
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorRadiusXYZ(j+1,2) - GeometryZShift;
         
                glVertex3fv(vec);
                
             glEnd();
              
          }
                       

       }       
 
    }
    
    // Now draw an arrow head
    
    for ( i = 1 ; i <= NumberOfPropulsionElements ; i++ ) {
    
       glBegin(GL_TRIANGLES);

             // Triangle 1
          
             glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,0),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,1),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,2) );
          
             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;
             
             glVertex3fv(vec);
             
       glEnd();
             
       glBegin(GL_TRIANGLES);             
             
             // Triangle 2
           
             glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,0),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,1),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,2) );
           
             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;
             
             glVertex3fv(vec);             

       glEnd();
             
       glBegin(GL_TRIANGLES);       
       
             // Triangle 3
     
             glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,0),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,1),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,2) );
     
             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;
             
             glVertex3fv(vec);             
      
       glEnd();
             
       glBegin(GL_TRIANGLES);       
         
             // Triangle 4
           
             glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,0),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,1),
                         PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,2) );
           
             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;

             glVertex3fv(vec);

             vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
             vec[1] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
             vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;
             
             glVertex3fv(vec);             
                          
       glEnd(); 
       
       if ( DrawReflectedGeometryIsOn ) {
       
          glBegin(GL_TRIANGLES);
   
                // Triangle 1
             
                glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,0),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,1),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(1,2) );
             
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;
                
                glVertex3fv(vec);
                
          glEnd();
                
          glBegin(GL_TRIANGLES);             
                
                // Triangle 2
              
                glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,0),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,1),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(2,2) );
              
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;
                
                glVertex3fv(vec);             
   
          glEnd();
                
          glBegin(GL_TRIANGLES);       
          
                // Triangle 3
        
                glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,0),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,1),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(3,2) );
        
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(2,2) - GeometryZShift;
                
                glVertex3fv(vec);             
         
          glEnd();
                
          glBegin(GL_TRIANGLES);       
            
                // Triangle 4
              
                glNormal3f( PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,0),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,1),
                            PropulsionElement[i].Rotor.RotorDirectionVectorNormal(4,2) );
              
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(1,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(3,2) - GeometryZShift;
   
                glVertex3fv(vec);
   
                vec[0] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,0) - GeometryXShift;
                vec[1] = -PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,1) - GeometryYShift;
                vec[2] = PropulsionElement[i].Rotor.RotorDirectionVectorXYZ(4,2) - GeometryZShift;
                
                glVertex3fv(vec);             
                             
          glEnd();   
          
       }     
                                  
    }
 
}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER DrawCGMarker                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCGMarker(void)
{

    static float zero[3]  = { 0.0,    0.0,   0.0};
    static float vxy_1[3] = { 1.0,    0.0,   0.0};
    static float vxy_2[3] = { 0.707,  0.707, 0.0};
    static float vxy_3[3] = { 0.0,    1.0,   0.0};
    static float vxy_4[3] = {-0.707,  0.707, 0.0};
    static float vxy_5[3] = {-1.0,    0.0,   0.0};
    static float vxy_6[3] = {-0.707, -0.707, 0.0};
    static float vxy_7[3] = {0.0,    -1.0,   0.0};
    static float vxy_8[3] = { 0.707, -0.707, 0.0};
    static float vxz_1[3] = { 1.0,    0.0,   0.0};
    static float vxz_2[3] = { 0.707,  0.0,   0.707};
    static float vxz_3[3] = { 0.0,    0.0,   1.0};
    static float vxz_4[3] = {-0.707,  0.0,   0.707};
    static float vxz_5[3] = {-1.0,    0.0,   0.0};
    static float vxz_6[3] = {-0.707,  0.0,  -0.707};
    static float vxz_7[3] = {0.0,     0.0,  -1.0};
    static float vxz_8[3] = { 0.707, 0.0,   -0.707};
    static float vyz_1[3] = {0.0,     1.0,    0.0};
    static float vyz_2[3] = {0.0,     0.707,  0.707};
    static float vyz_3[3] = {0.0,     0.0,    1.0};
    static float vyz_4[3] = {0.0,    -0.707,  0.707};
    static float vyz_5[3] = {0.0,    -1.0,    0.0};
    static float vyz_6[3] = {0.0,    -0.707, -0.707};
    static float vyz_7[3] = {0.0,    0.0,    -1.0};
    static float vyz_8[3] = {0.0,     0.707, -0.707};

    float Scale, InvScale;

       Scale = 0.025*Cref;
    InvScale = 1./Scale;

    glTranslatef( Xcg - GeometryXShift,
                  Ycg - GeometryYShift,
                  Zcg - GeometryZShift);

    glScalef(Scale, Scale, Scale);
    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glLineWidth(2);
    glColor3f(1,0,0);

    glBegin( GL_LINE_LOOP );
      glVertex3fv(vxy_1);  glVertex3fv(vxy_2);  glVertex3fv(vxy_3);  glVertex3fv(vxy_4);
      glVertex3fv(vxy_5);  glVertex3fv(vxy_6);  glVertex3fv(vxy_7);  glVertex3fv(vxy_8);
    glEnd();

    glBegin( GL_LINE_LOOP );
      glVertex3fv(vxz_1);  glVertex3fv(vxz_2);  glVertex3fv(vxz_3);  glVertex3fv(vxz_4);
      glVertex3fv(vxz_5);  glVertex3fv(vxz_6);  glVertex3fv(vxz_7);  glVertex3fv(vxz_8);
    glEnd();

    glBegin( GL_LINE_LOOP );
      glVertex3fv(vyz_1);  glVertex3fv(vyz_2);  glVertex3fv(vyz_3);  glVertex3fv(vyz_4);
      glVertex3fv(vyz_5);  glVertex3fv(vyz_6);  glVertex3fv(vyz_7);  glVertex3fv(vyz_8);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vxy_1);   glVertex3fv(vxy_2);  glVertex3fv(vxy_3);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vxy_5);   glVertex3fv(vxy_6);  glVertex3fv(vxy_7);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vxz_1);   glVertex3fv(vxz_2);  glVertex3fv(vxz_3);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vxz_5);   glVertex3fv(vxz_6);  glVertex3fv(vxz_7);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vyz_1);   glVertex3fv(vyz_2);  glVertex3fv(vyz_3);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(zero);   glVertex3fv(vyz_5);   glVertex3fv(vyz_6);  glVertex3fv(vyz_7);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxy_2);  glVertex3fv(vxz_2);  glVertex3fv(vyz_2);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxz_2);  glVertex3fv(vyz_2);  glVertex3fv(vyz_3);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxz_2);  glVertex3fv(vxy_2);  glVertex3fv(vxy_1);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxy_2);  glVertex3fv(vyz_2);  glVertex3fv(vyz_1);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxy_6);  glVertex3fv(vxz_6);  glVertex3fv(vyz_6);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxz_6);  glVertex3fv(vxy_6);  glVertex3fv(vxy_5);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxz_6);  glVertex3fv(vyz_6);  glVertex3fv(vyz_7);
    glEnd();

    glBegin( GL_POLYGON );
      glVertex3fv(vxy_6);  glVertex3fv(vyz_6);  glVertex3fv(vxy_7);
    glEnd();

    glScalef(InvScale, InvScale, InvScale);
    glTranslatef( -(Xcg - GeometryXShift),
                  -(Ycg - GeometryYShift),
                  -(Zcg - GeometryZShift));

    glLineWidth(1);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                         GL_VIEWER DrawSymmetryPlane                          #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawSymmetryPlane(void)
{

    float Scale, InvScale, dalpha, alpha;

    int i, j, node1, node2, node3, LastTri, LastCon, LastSurface;
    int LastMaterialType, SurfaceID, SurfID;
    float vec1[3], vec2[3], vec3[3], vec4[3], Normal[3], rgb[4], LastEmissivity;
    float xyz1[3], xyz2[3];

    glTranslatef( 0. - GeometryXShift,
                  0. - GeometryYShift,
                  0. - GeometryZShift);

    // Draw triangles as shaded surface

    rgb[0] = 0.7;
    rgb[1] = 0.7;
    rgb[2] = 0.7;
    rgb[3] = 1.;

    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
    glColor3fv(rgb);

    if ( DrawXPlaneIsOn ) {
       
       vec1[0] =         0;
       vec1[1] = -ViewSize + GeometryYShift;
       vec1[2] =  ViewSize + GeometryZShift;
       
       vec2[0] =         0;
       vec2[1] = -ViewSize + GeometryYShift;
       vec2[2] = -ViewSize + GeometryZShift;
       
       vec3[0] =         0;
       vec3[1] =  ViewSize + GeometryYShift;
       vec3[2] = -ViewSize + GeometryZShift;
       
       vec4[0] =         0;
       vec4[1] =  ViewSize + GeometryYShift;
       vec4[2] =  ViewSize + GeometryZShift;
       
       Normal[0] = 1.;
       Normal[1] = 0.;
       Normal[2] = 0.;
       
    }

    else if ( DrawYPlaneIsOn ) {
       
       vec1[0] = -ViewSize + GeometryXShift;
       vec1[1] =        0.;
       vec1[2] =  ViewSize + GeometryZShift;
       
       vec2[0] =  ViewSize + GeometryXShift;
       vec2[1] =        0.;
       vec2[2] =  ViewSize + GeometryZShift;
       
       vec3[0] =  ViewSize + GeometryXShift;
       vec3[1] =        0.;
       vec3[2] = -ViewSize + GeometryZShift;
       
       vec4[0] = -ViewSize + GeometryXShift;
       vec4[1] =        0.;
       vec4[2] = -ViewSize + GeometryZShift;
       
       Normal[0] = 0.;
       Normal[1] = 1.;
       Normal[2] = 0.;
       
    }
        
    else if ( DrawZPlaneIsOn ) {
       
       vec1[0] = -ViewSize + GeometryXShift;
       vec1[1] = -ViewSize + GeometryYShift;
       vec1[2] =         0.;
       
       vec2[0] = -ViewSize + GeometryXShift;
       vec2[1] =  ViewSize + GeometryYShift;
       vec2[2] =        0.;
       
       vec3[0] =  ViewSize + GeometryXShift;
       vec3[1] =  ViewSize + GeometryYShift;
       vec3[2] =        0.;
       
       vec4[0] =  ViewSize + GeometryXShift;
       vec4[1] = -ViewSize + GeometryYShift;
       vec4[2] =        0.;
       
       Normal[0] = 0.;
       Normal[1] = 0.;
       Normal[2] = 1.;
       
    }
                
    // Draw the plane
              
    glBegin(GL_TRIANGLES);

       glNormal3f( Normal[0], Normal[1], Normal[2] );

       glVertex3fv(vec1);

       glVertex3fv(vec2);

       glVertex3fv(vec3);
       
    glEnd();       
    
    glBegin(GL_TRIANGLES);
       
       glNormal3f( Normal[0], Normal[1], Normal[2] );

       glVertex3fv(vec3);

       glVertex3fv(vec4);

       glVertex3fv(vec1);       

    glEnd();
    
    // Draw some grid lines
   
    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glLineWidth(5.);
    glDisable(GL_LIGHTING);    

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;
    rgb[3] = 1.;
    
    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
    glColor3fv(rgb);    
        
    int NumLines = 5;
    
    dalpha = 1./(NumLines-1);
    
    for ( i = 1 ; i <= NumLines ;  i++ ) {
    
       alpha = (i-1)*dalpha;
       
       xyz1[0] = vec1[0] + alpha*(vec2[0] - vec1[0]);
       xyz1[1] = vec1[1] + alpha*(vec2[1] - vec1[1]);
       xyz1[2] = vec1[2] + alpha*(vec2[2] - vec1[2]);
       
       xyz2[0] = vec4[0] + alpha*(vec3[0] - vec4[0]);
       xyz2[1] = vec4[1] + alpha*(vec3[1] - vec4[1]);
       xyz2[2] = vec4[2] + alpha*(vec3[2] - vec4[2]);

       glBegin(GL_LINES);
       
          glVertex3fv(xyz1);
          glVertex3fv(xyz2);
          
       glEnd();
       
    }
    
    for ( i = 1 ; i <= NumLines ;  i++ ) {
    
       alpha = (i-1)*dalpha;
       
       xyz1[0] = vec1[0] + alpha*(vec4[0] - vec1[0]);
       xyz1[1] = vec1[1] + alpha*(vec4[1] - vec1[1]);
       xyz1[2] = vec1[2] + alpha*(vec4[2] - vec1[2]);
       
       xyz2[0] = vec2[0] + alpha*(vec3[0] - vec2[0]);
       xyz2[1] = vec2[1] + alpha*(vec3[1] - vec2[1]);
       xyz2[2] = vec2[2] + alpha*(vec3[2] - vec2[2]);

       glBegin(GL_LINES);
       
          glVertex3fv(xyz1);
          glVertex3fv(xyz2);
          
       glEnd();
       
    }    
    
    glEnable(GL_LIGHTING);

    glTranslatef( 0. + GeometryXShift,
                  0. + GeometryYShift,
                  0. + GeometryZShift);     
  
}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER DrawAxes                                  #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawAxes(void)
{

    static float zero[3]  = { 0.0,   0.0,   0.0};
    static float vxy_1[3] = { 1.0,   0.0,   0.0};
    static float vxy_2[3] = { 0.0,   1.0,   0.0};
    static float vxy_3[3] = { 0.0,   0.0,   1.0};

    float Scale, InvScale;

       Scale = 0.1*ABS(XMax - XMin);
    InvScale = 1./Scale;

    glTranslatef( 0. - GeometryXShift,
                  0. - GeometryYShift,
                  0. - GeometryZShift);

    glScalef(Scale, Scale, Scale);
    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glLineWidth(2);
    
    glColor3f(1,0,0);
    glBegin( GL_LINE_LOOP );
      glVertex3fv(zero);  glVertex3fv(vxy_1);
    glEnd();

    glColor3f(0,1,0);
    glBegin( GL_LINE_LOOP );
      glVertex3fv(zero);  glVertex3fv(vxy_2);
    glEnd();
    
    glColor3f(0,0,1);
    glBegin( GL_LINE_LOOP );
      glVertex3fv(zero);  glVertex3fv(vxy_3);
    glEnd();
 
    glScalef(InvScale, InvScale, InvScale);
    glTranslatef( -(0. - GeometryXShift),
                  -(0. - GeometryYShift),
                  -(0. - GeometryZShift));

    glLineWidth(1);
    glEnable(GL_LIGHTING);

}

/*##############################################################################
#                                                                              #
#                              GL_VIEWER DrawTPSCGLabel                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::DrawCGLabel(void)
{

    char string[80];
    float rgb[3];

    rgb[0] = 0.;
    rgb[1] = 0.;
    rgb[2] = 0.;

    // Draw the title

    glDisable(GL_DEPTH_TEST);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glColor3fv(rgb);
    glfStringCentering(GL_FALSE);
    glfSetRotateAngle(0.0f);
    glfStringDirection(GLF_LEFT);

    glPushMatrix();

       sprintf(string, "Vehicle CG: %f, %f, %f \n", Xcg, Ycg, Zcg);

       glTranslatef(0.25*w(), 0.025*h(), 0);
       glScalef(15., 15., 1);
       glfDrawSolidString(string);

    glPopMatrix();

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER SetTagSurfaceColor                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetTagSurfaceColor(int SurfaceID, int MaxVals)
{

  int rdd[] = {155,   0,   0, 175, 155,   0, 155,   2, 100, 198, 175, 106, 113, 142,  30,   0, 175, 124, 113, 142, 175,  56, 105,  0, 105, 63,  0, 175};
  int grn[] = {  0, 155,   0, 105, 155, 155,   0,  25, 149, 113,  92,  90, 175, 142, 144, 175, 149, 175, 113,  56, 130, 142,  50, 72, 139, 36, 36, 175};
  int blu[] = {  0,   0, 155, 105,   0, 155, 155, 112, 175, 113, 175,   5, 113,  56, 175, 127,  12,   0, 175, 172,  71, 142, 175,  0,  34,  0, 63,   0};

  int index;
  int NumberOfColors = 28;
  float rgb[4], Scale;

  // Adjust SurfaceID to fit into 15 set colors, we simply rotate back
  // through the color list if there is more than 15 control surfaces...

  if ( SurfaceID <= 0 ) {

     printf("Incorrect SurfaceID... ID is <= 0! \n");fflush(NULL);
     exit(1);

  }

  index = SurfaceID - 1;

  if ( SurfaceID > NumberOfColors ) {

     index = SurfaceID
           - (SurfaceID/NumberOfColors)*NumberOfColors;

  }

  // Set the color

  rgb[0] = rdd[index]/255.;
  rgb[1] = grn[index]/255.;
  rgb[2] = blu[index]/255.;
  rgb[3] = 1.;

  if ( SurfaceID > NumberOfColors ) {

     Scale = 3.*(SurfaceID)/((float)NumberOfColors)/MaxVals;

     rgb[0] /= (1. + Scale);
     rgb[1] /= (1. + Scale);
     rgb[2] /= (1. + Scale);

  }
  
  rgb[0] *= 1.5;
  rgb[1] *= 1.5;
  rgb[2] *= 1.5;
  
  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);

  glColor3fv(rgb); // For those things that are not doing lighting...

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER SetTagSurfaceColor                        #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetTagSurfaceColor(int SurfaceID, int MaxVals, double Alpha)
{

  int rdd[] = {155,   0,   0, 175, 155,   0, 155,   2, 100, 198, 175, 106, 113, 142,  30,   0, 175, 124, 113, 142, 175,  56, 105,  0, 105, 63,  0, 175};
  int grn[] = {  0, 155,   0, 105, 155, 155,   0,  25, 149, 113,  92,  90, 175, 142, 144, 175, 149, 175, 113,  56, 130, 142,  50, 72, 139, 36, 36, 175};
  int blu[] = {  0,   0, 155, 105,   0, 155, 155, 112, 175, 113, 175,   5, 113,  56, 175, 127,  12,   0, 175, 172,  71, 142, 175,  0,  34,  0, 63,   0};

  int index;
  int NumberOfColors = 28;
  float rgb[4], Scale;

  // Adjust SurfaceID to fit into 15 set colors, we simply rotate back
  // through the color list if there is more than 15 control surfaces...

  if ( SurfaceID <= 0 ) {

     printf("Incorrect SurfaceID... ID is <= 0! \n");fflush(NULL);
     exit(1);

  }

  index = SurfaceID - 1;

  if ( SurfaceID > NumberOfColors ) {

     index = SurfaceID
           - (SurfaceID/NumberOfColors)*NumberOfColors;

  }

  // Set the color

  rgb[0] = rdd[index]/255.;
  rgb[1] = grn[index]/255.;
  rgb[2] = blu[index]/255.;
  rgb[3] = Alpha;

  if ( SurfaceID > NumberOfColors ) {

     Scale = 3.*(SurfaceID)/((float)NumberOfColors)/MaxVals;

     rgb[0] /= (1. + Scale);
     rgb[1] /= (1. + Scale);
     rgb[2] /= (1. + Scale);

  }
  
  rgb[0] *= 1.5;
  rgb[1] *= 1.5;
  rgb[2] *= 1.5;
  
  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);

  glColor4fv(rgb); // For those things that are not doing lighting...

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER percent_to_rgb                            #
#                                                                              #
##############################################################################*/

void GL_VIEWER::percent_to_rgb(float per, float col[3], int colmap)
{

    int   R = 0, G = 1, B = 2;
    int   j, num_regions;
    float color[8][3], percent[8], t;

    // Set the color map

    if ( colmap == 0 ) {

       num_regions = 5;

       // blue

       percent[0] = 0./5.;
       color[0][R] = 0.0;
       color[0][G] = 0.0;
       color[0][B] = 1.0;

       // cyan

       percent[1] = 1./5.;
       color[1][R] = 0.0;
       color[1][G] = 1.0;
       color[1][B] = 1.0;

       // green

       percent[2] = 2./5.;
       color[2][R] = 0.0;
       color[2][G] = 1.0;
       color[2][B] = 0.0;

       // yellow

       percent[3] = 3./5.;
       color[3][R] = 1.0;
       color[3][G] = 1.0;
       color[3][B] = 0.0;

       // red

       percent[4] = 4./5.;
       color[4][R] = 1.0;
       color[4][G] = 0.0;
       color[4][B] = 0.0;

       // magenta

       percent[5] =  1.;
       color[5][R] = 1.0;
       color[5][G] = 0.0;
       color[5][B] = 1.0;

    }

    else if ( colmap == 999 ) {

       num_regions = 6;

       // black

       percent[0] = 0.;
       color[0][R] = 0.0;
       color[0][G] = 0.0;
       color[0][B] = 0.0;

       // blue

       percent[1] = 1./6.;
       color[1][R] = 0.0;
       color[1][G] = 0.0;
       color[1][B] = 1.0;

       // cyan

       percent[2] = 2./6.;
       color[2][R] = 0.0;
       color[2][G] = 1.0;
       color[2][B] = 1.0;

       // green

       percent[3] = 3./6.;
       color[3][R] = 0.0;
       color[3][G] = 1.0;
       color[3][B] = 0.0;

       // yellow

       percent[4] = 4./6.;
       color[4][R] = 1.0;
       color[4][G] = 1.0;
       color[4][B] = 0.0;

       // red

       percent[5] = 5./6.;
       color[5][R] = 1.0;
       color[5][G] = 0.0;
       color[5][B] = 0.0;

       // magenta

       percent[6] =  1.;
       color[6][R] = 1.0;
       color[6][G] = 0.0;
       color[6][B] = 1.0;

       // white

       percent[7] = 1.;
       color[7][R] = 1.0;
       color[7][G] = 1.0;
       color[7][B] = 1.0;

    }

    // Modify color map for drag contours

    else {

       num_regions = 6;

       // blue

       percent[0] = 0./6.;
       color[0][R] = 0.0;
       color[0][G] = 0.0;
       color[0][B] = 1.0;

       // cyan

       percent[1] = 1./6.;
       color[1][R] = 0.0;
       color[1][G] = 1.0;
       color[1][B] = 1.0;

       // green

       percent[2] = 2./6.;
       color[2][R] = 0.0;
       color[2][G] = 1.0;
       color[2][B] = 0.0;

       // white

       percent[3] = 3./6.;
       color[3][R] = 1.0;
       color[3][G] = 1.0;
       color[3][B] = 1.0;

       // yellow

       percent[4] = 4./6.;
       color[4][R] = 1.0;
       color[4][G] = 1.0;
       color[4][B] = 0.0;

       // red

       percent[5] = 5./6.;
       color[5][R] = 1.0;
       color[5][G] = 0.0;
       color[5][B] = 0.0;

       // magenta

       percent[6] = 6./6.;
       color[6][R] = 1.0;
       color[6][G] = 0.0;
       color[6][B] = 1.0;

    }

    // Set up color map

    if ( per > 1. )  {

       col[R] = 1.;
       col[G] = 0.;
       col[B] = 1.;

    }

    else {

       per = MIN(MAX(per,0.0),1.0);

       j = (int) ( MIN(num_regions*per + 1, num_regions) );

       t = (per-percent[j-1])/(percent[j]-percent[j-1]);

       col[R] = color[j-1][R] + t*(color[j][R] - color[j-1][R]);
       col[G] = color[j-1][G] + t*(color[j][G] - color[j-1][G]);
       col[B] = color[j-1][B] + t*(color[j][B] - color[j-1][B]);

    }

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER SetSurfaceRegionColor                     #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetSurfaceRegionColor(int SurfaceRegionID)
{

  int rdd[] = {155,   0,   0, 255,   0, 255, 198, 113, 142, 113, 142,  56,  0, 63,  0};
  int grn[] = {  0, 255,   0, 255, 255,   0, 113, 198, 142, 113,  56, 142, 72, 36, 36};
  int blu[] = {  0,   0, 255,   0, 255, 255, 113, 113,  56, 198, 142, 142,  0,  0, 63};
  int index;
  int NumberOfColors = 15;
  float rgb[4];

  // Adjust SurfaceRegionID to fit into 15 set colors, we simply rotate back
  // through the color list if there is more than 15 surface regions...

  if ( SurfaceRegionID <= 0 ) {

     printf("Incorrect ControlSurfaceID... ID is <= 0! \n");fflush(NULL);
     exit(1);

  }

  index = SurfaceRegionID - 1;

  if ( SurfaceRegionID > NumberOfColors ) {

     index = SurfaceRegionID
           - (SurfaceRegionID/NumberOfColors)*NumberOfColors - 1;

  }

  // Set the color

  rgb[0] = rdd[index]/255.;
  rgb[1] = grn[index]/255.;
  rgb[2] = blu[index]/255.;
  rgb[3] = 1.;

  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);
  glColor3fv(rgb);


}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER SetMaterialTypeColor                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetMaterialTypeColor(int MaterialID)
{

  int rdd[] = {155,   0,   0, 155,   0, 155, 198, 113, 142, 113, 142,  56,  0, 63,  0};
  int grn[] = {  0, 155,   0, 155, 155,   0, 113, 198, 142, 113,  56, 142, 72, 36, 36};
  int blu[] = {  0,   0, 155,   0, 155, 155, 113, 113,  56, 198, 142, 142,  0,  0, 63};
  int index;
  int NumberOfColors = 15;
  float rgb[4];

  // Adjust ControlSurfaceID to fit into 15 set colors, we simply rotate back
  // through the color list if there is more than 15 control surfaces...

  if ( MaterialID < 0 ) {

     printf("Incorrect MaterialID... ID is < 0! \n");fflush(NULL);
     exit(1);

  }

  index = MaterialID;

  if ( MaterialID > NumberOfColors ) {

     index = MaterialID
           - (MaterialID/NumberOfColors)*NumberOfColors;

  }

  // Set the color

  rgb[0] = rdd[index]/255.;
  rgb[1] = grn[index]/255.;
  rgb[2] = blu[index]/255.;
  rgb[3] = 1.;

  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);

  glColor3fv(rgb); // For those things that are not doing lighting...

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER SetControlSurfaceColor                    #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SetControlSurfaceColor(int ControlSurfaceID)
{

  int rdd[] = {155,   0,   0, 155,   0, 155, 198, 113, 142, 113, 142,  56,  0, 63,  0};
  int grn[] = {  0, 155,   0, 155, 155,   0, 113, 198, 142, 113,  56, 142, 72, 36, 36};
  int blu[] = {  0,   0, 155,   0, 155, 155, 113, 113,  56, 198, 142, 142,  0,  0, 63};
  int index;
  int NumberOfColors = 15;
  float rgb[4];

  // Adjust ControlSurfaceID to fit into 15 set colors, we simply rotate back
  // through the color list if there is more than 15 control surfaces...

  if ( ControlSurfaceID <= 0 ) {

     printf("Incorrect ControlSurfaceID... ID is <= 0! \n");fflush(NULL);
     exit(1);

  }

  index = ControlSurfaceID - 1;

  if ( ControlSurfaceID > NumberOfColors ) {

     index = ControlSurfaceID
           - (ControlSurfaceID/NumberOfColors)*NumberOfColors;

  }

  // Set the color

  rgb[0] = rdd[index]/255.;
  rgb[1] = grn[index]/255.;
  rgb[2] = blu[index]/255.;
  rgb[3] = 1.;

  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,rgb);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,rgb);

  rgb[0] *= 2.25;
  rgb[1] *= 2.25;
  rgb[2] *= 2.25;

  glColor3fv(rgb); // For those things that are not doing lighting...

}

/*##############################################################################
#                                                                              #
#                            GL_VIEWER SwapSurfaceNormals                      #
#                                                                              #
##############################################################################*/

void GL_VIEWER::SwapSurfaceNormals(void)
{

    int j, SurfID;
    int i, nod1, nod2, nod3;
    float vec1[3], vec2[3], normal[3], dot, *TotalArea;

    // Loop over surface triangles and swap normals...

    for ( j = 1 ; j <= NumberOfTris; j++ ) {

       // Draw it according to its surface ID flag

       SurfID = TriList[j].surface_id;
  
       if ( PanelComGeomTagsBrowser->selected(ComGeom2PanelTag[SurfID]) ) {
       
          Nx[j] *= -1.;
          Ny[j] *= -1.;
          Nz[j] *= -1.;
          
          Cp[j] *= -1.;

       }
       
    }
    

    TotalArea = new float[NumberOfNodes + 1];

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       Nodal_Nx[i] = 0.;
       Nodal_Ny[i] = 0.;
       Nodal_Nz[i] = 0.;

       TotalArea[i] = 0.;

	}

    for ( i = 1 ; i <= NumberOfTris ; i++ ) {

       nod1 = TriList[i].node1;
       nod2 = TriList[i].node2;
       nod3 = TriList[i].node3;

       vec1[0] = NodeList[nod2].x - NodeList[nod1].x;
       vec1[1] = NodeList[nod2].y - NodeList[nod1].y;
       vec1[2] = NodeList[nod2].z - NodeList[nod1].z;

       vec2[0] = NodeList[nod3].x - NodeList[nod1].x;
       vec2[1] = NodeList[nod3].y - NodeList[nod1].y;
       vec2[2] = NodeList[nod3].z - NodeList[nod1].z;

       vector_cross(vec1,vec2,normal);

       dot = sqrt(vector_dot(normal,normal));

       Nodal_Nx[nod1] += Nx[i];
       Nodal_Ny[nod1] += Ny[i];
       Nodal_Nz[nod1] += Nz[i];

       Nodal_Nx[nod2] += Nx[i];
       Nodal_Ny[nod2] += Ny[i];
       Nodal_Nz[nod2] += Nz[i];

       Nodal_Nx[nod3] += Nx[i];
       Nodal_Ny[nod3] += Ny[i];
       Nodal_Nz[nod3] += Nz[i];

       TotalArea[nod1] += dot;
       TotalArea[nod2] += dot;
       TotalArea[nod3] += dot;

    }

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       Nodal_Nx[i] /= TotalArea[i];
       Nodal_Ny[i] /= TotalArea[i];
       Nodal_Nz[i] /= TotalArea[i];

	}

    for ( i = 1 ; i <= NumberOfNodes ; i++ ) {

       dot = sqrt(Nodal_Nx[i]*Nodal_Nx[i] + Nodal_Ny[i]*Nodal_Ny[i] + Nodal_Nz[i]*Nodal_Nz[i]);

       Nodal_Nx[i] /= dot;
       Nodal_Ny[i] /= dot;
       Nodal_Nz[i] /= dot;

	}

    delete [] TotalArea;
    
}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER WritePNGFile                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::WritePNGFile(char *FileName)
{

    int width, height, DumInt;
    char rgbstr[256];

    if ( FileName != NULL ) {

       // Make sure everything is drawn to the screen

       flush(); glFlush ();

       // Now write to png file

       sprintf(rgbstr,"%s.%s.png",file_name,FileName);
       width = w();
       height = h();
       
       WritePNG( rgbstr, ( char * ) "VSPVIEWER", 0, 0, width, height, 0 );

       fflush(NULL);

    }

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER WritePNGFile                              #
#                                                                              #
##############################################################################*/

void GL_VIEWER::WritePNGFile(char *Path, char *FileName)
{

    int width, height, DumInt;
    char rgbstr[256];

    if ( FileName != NULL ) {

       // Make sure everything is drawn to the screen

       flush(); glFlush ();

       // Now write to png file

       sprintf(rgbstr,"%s%s.%s.png",Path,file_name_no_path,FileName);
       width = w();
       height = h();
       
       WritePNG( rgbstr, ( char * )"VSPVIEWER", 0, 0, width, height, 0 );

       fflush(NULL);

    }

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER WritePNG                                  #
#                                                                              #
##############################################################################*/

int GL_VIEWER::WritePNG(char *filename, char *description,
                        int x, int y, int width, int height, int compression)
{

    GLubyte *image, *flipped_image, *SrcLine, *DestLine;
    int i, DumInt, ScanLength;
    
    image = (GLubyte *) malloc(width * height * sizeof(GLubyte) * 4);
    flipped_image = (GLubyte *) malloc(width * height * sizeof(GLubyte) * 4);

    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    
    glReadPixels(x, y, width, height, GL_RGBA, GL_UNSIGNED_BYTE, image);
    
    ScanLength = 4 * width;
    
    for ( i = 0 ; i < height ; i++ ) {
       
        SrcLine = &image[ i * ScanLength ];
        
       DestLine = &flipped_image[(height - i - 1) * ScanLength ];
       
       memcpy(  DestLine, SrcLine, ScanLength );
       
    }
    
    DumInt = stbi_write_png(filename, width, height, 4, &flipped_image[0], width*4);
    
    if ( image != NULL ) free(image);

    if ( flipped_image != NULL ) free(flipped_image);

    return 0;

}

/*##############################################################################
#                                                                              #
#                          GL_VIEWER WriteFontFile                             #
#                                                                              #
##############################################################################*/

void GL_VIEWER::WriteFontFile(void)
{

    FILE *font_file;
    int c_size;

    // Sizeof ints and floats

    c_size = sizeof(char);

    // Open the font file

    if ( (font_file = fopen("cbviewer_font.glf","wb")) == NULL ) {

       printf("Could not open the font file for writing! \n");fflush(NULL);

       exit(1);

    }

    // The following hex data contains the glf Times New font data that was
    // contained in the times_new1.glf binary font file. I used the bin2hex.pl
    // perl script found on the web to convert this binary file to hex and store
    // it as a c character array - so the user would not have to carry around
    // the font file to get this viewer to work. This routine simply write the
    // data out to a file that the glf routine then reads in...

/* begin binary data: */
unsigned char bin_data[] = /* 45361 */
{0x47,0x4C,0x46,0x54,0x69,0x6D,0x65,0x73,0x20,0x4E,0x65,0x77,0x20,0x52,0x6F
,0x6D,0x61,0x6E,0x20,0x31,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5F,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x21,0x1F,0x1B,0x02,0x49,0x81,0xC5
,0x3D,0xCB,0x4D,0xDC,0x3E,0xA9,0xF9,0xCA,0x3D,0x2F,0x87,0xED,0x3E,0xCD,0xCC
,0xCC,0x3D,0xEA,0x7A,0xFA,0x3E,0xA5,0x12,0xBE,0x3D,0xF6,0x09,0x0C,0x3F,0x22
,0xE3,0x91,0x3D,0x0C,0xAD,0x16,0x3F,0x31,0xD1,0x20,0x3D,0xC6,0x17,0x1D,0x3F
,0x42,0x60,0x65,0x3A,0x64,0x3B,0x1F,0x3F,0xC6,0x8A,0x1A,0xBD,0xC6,0x17,0x1D
,0x3F,0x61,0x18,0x90,0xBD,0x0C,0xAD,0x16,0x3F,0xD3,0x9F,0xBD,0xBD,0x0B,0xB7
,0x0C,0x3F,0xCD,0xCC,0xCC,0xBD,0x4C,0xE3,0xFF,0x3E,0x9B,0x20,0xCA,0xBD,0xD9
,0x22,0xF1,0x3E,0x8A,0x1C,0xC2,0xBD,0xCB,0x4D,0xDC,0x3E,0xBC,0x74,0x93,0xBC
,0x4E,0x62,0x90,0xBE,0x6F,0x12,0x83,0x3C,0x4E,0x62,0x90,0xBE,0x73,0x2D,0x1A
,0xBD,0x74,0xD3,0xDE,0xBE,0x73,0xF3,0x8D,0xBD,0x5C,0xE5,0xE9,0xBE,0x08,0x04
,0xBA,0xBD,0xF9,0x12,0xFA,0xBE,0x39,0xB4,0xC8,0xBD,0x4C,0x8B,0x06,0xBF,0x08
,0x04,0xBA,0xBD,0x19,0x38,0x10,0xBF,0x73,0xF3,0x8D,0xBD,0x31,0x5D,0x18,0xBF
,0x73,0x2D,0x1A,0xBD,0x25,0xE6,0x1D,0xBF,0x00,0x00,0x00,0x00,0x77,0xBE,0x1F
,0xBF,0x73,0x2D,0x1A,0x3D,0x25,0xE6,0x1D,0xBF,0xF9,0xF3,0x8D,0x3D,0x31,0x5D
,0x18,0xBF,0x08,0x04,0xBA,0x3D,0x19,0x38,0x10,0xBF,0x39,0xB4,0xC8,0x3D,0x4C
,0x8B,0x06,0xBF,0x08,0x04,0xBA,0x3D,0xF9,0x12,0xFA,0xBE,0xF9,0xF3,0x8D,0x3D
,0x5C,0xE5,0xE9,0xBE,0x73,0x2D,0x1A,0x3D,0x74,0xD3,0xDE,0xBE,0x00,0x00,0x00
,0x00,0xD1,0x22,0xDB,0xBE,0x0C,0x0D,0x0E,0x0C,0x0E,0x00,0x0C,0x00,0x01,0x0B
,0x0C,0x01,0x0B,0x01,0x02,0x0A,0x0B,0x02,0x0A,0x02,0x03,0x09,0x0A,0x03,0x08
,0x09,0x03,0x08,0x03,0x04,0x07,0x08,0x04,0x07,0x04,0x05,0x06,0x07,0x05,0x1E
,0x0F,0x10,0x1D,0x1E,0x10,0x1C,0x1D,0x10,0x1B,0x1C,0x10,0x1A,0x1B,0x10,0x19
,0x1A,0x10,0x18,0x19,0x10,0x17,0x18,0x10,0x16,0x17,0x10,0x15,0x16,0x10,0x14
,0x15,0x10,0x13,0x14,0x10,0x12,0x13,0x10,0x12,0x10,0x11,0x0E,0x1E,0x22,0x1F
,0x1B,0x02,0xA4,0x70,0x3D,0x3E,0x1B,0x2F,0xDD,0x3D,0xC5,0x20,0x70,0x3E,0xF8
,0x53,0xBB,0x3E,0xCD,0xCC,0x7C,0x3E,0x2D,0x5C,0xE6,0x3E,0x12,0x83,0x80,0x3E
,0xB0,0x1E,0x07,0x3F,0xAB,0x98,0x7A,0x3E,0xB3,0x08,0x11,0x3F,0xF8,0x4F,0x67
,0x3E,0x31,0xB2,0x18,0x3F,0xF0,0x52,0x4A,0x3E,0x13,0x99,0x1D,0x3F,0xB4,0xC8
,0x26,0x3E,0x64,0x3B,0x1F,0x3F,0x26,0x57,0x01,0x3E,0x13,0x99,0x1D,0x3F,0x42
,0x08,0xC8,0x3D,0x31,0xB2,0x18,0x3F,0xD5,0x22,0xA2,0x3D,0xFB,0x3E,0x10,0x3F
,0x06,0x81,0x95,0x3D,0xCF,0xF7,0x03,0x3F,0x55,0x6A,0x96,0x3D,0x08,0xAC,0xFC
,0x3E,0xC8,0x26,0x99,0x3D,0x73,0x68,0xF1,0x3E,0x78,0x99,0xC1,0x3D,0x48,0xE1
,0xBA,0x3E,0x98,0x6E,0x12,0x3E,0x1B,0x2F,0xDD,0x3D,0xE1,0x7A,0x14,0xBE,0x1B
,0x2F,0xDD,0x3D,0x82,0x70,0xC5,0xBD,0x58,0x39,0xBC,0x3E,0x28,0x7D,0xA1,0xBD
,0xAE,0x9D,0xF0,0x3E,0x06,0x81,0x95,0xBD,0x58,0xE5,0x06,0x3F,0x41,0xB8,0xA2
,0xBD,0x80,0xD6,0x10,0x3F,0xF2,0x5D,0xCA,0xBD,0x7D,0x95,0x18,0x3F,0x35,0x08
,0x03,0xBE,0xEF,0x91,0x1D,0x3F,0x35,0xB6,0x27,0xBE,0x64,0x3B,0x1F,0x3F,0x30
,0x84,0x4C,0xBE,0x13,0x99,0x1D,0x3F,0x50,0xE1,0x68,0xBE,0x31,0xB2,0x18,0x3F
,0xCE,0xFC,0x7A,0xBE,0xA1,0x30,0x10,0x3F,0x12,0x83,0x80,0xBE,0x77,0xBE,0x03
,0x3F,0x63,0x0C,0x7C,0xBE,0x3B,0xE0,0xEA,0x3E,0xDA,0x1E,0x6D,0xBE,0x18,0xB0
,0xBC,0x3E,0x5A,0x64,0x3B,0xBE,0x1B,0x2F,0xDD,0x3D,0x0E,0x0F,0x00,0x0E,0x00
,0x01,0x0E,0x01,0x02,0x0D,0x0E,0x02,0x0D,0x02,0x03,0x0C,0x0D,0x03,0x0B,0x0C
,0x03,0x0A,0x0B,0x03,0x0A,0x03,0x04,0x0A,0x04,0x05,0x0A,0x05,0x06,0x0A,0x06
,0x07,0x0A,0x07,0x08,0x09,0x0A,0x08,0x1D,0x1E,0x10,0x1D,0x10,0x11,0x1D,0x11
,0x12,0x1C,0x1D,0x12,0x1B,0x1C,0x12,0x1B,0x12,0x13,0x1A,0x1B,0x13,0x1A,0x13
,0x14,0x1A,0x14,0x15,0x1A,0x15,0x16,0x1A,0x16,0x17,0x1A,0x17,0x18,0x19,0x1A
,0x18,0x0F,0x1E,0x23,0x20,0x20,0x02,0xBD,0x1C,0x8E,0xBE,0x77,0xBE,0x1F,0xBF
,0xFE,0xD4,0x48,0xBE,0x42,0x60,0x65,0xBE,0xD0,0xD2,0xD5,0x3D,0x42,0x60,0x65
,0xBE,0x7F,0x6A,0xBC,0x3C,0x77,0xBE,0x1F,0xBF,0x82,0x70,0xC5,0x3D,0x77,0xBE
,0x1F,0xBF,0xBD,0x1C,0x36,0x3E,0x42,0x60,0x65,0xBE,0x06,0x81,0xD5,0x3E,0x42
,0x60,0x65,0xBE,0x06,0x81,0xD5,0x3E,0xE3,0xA5,0x1B,0xBE,0xB8,0x1E,0x45,0x3E
,0xE3,0xA5,0x1B,0xBE,0x3A,0x08,0x82,0x3E,0x9A,0x99,0x19,0x3E,0x06,0x81,0xD5
,0x3E,0x9A,0x99,0x19,0x3E,0x06,0x81,0xD5,0x3E,0xF8,0x53,0x63,0x3E,0x37,0x89
,0x89,0x3E,0xF8,0x53,0x63,0x3E,0xAC,0x70,0xB3,0x3E,0x77,0xBE,0x1F,0x3F,0xF2
,0xD2,0x8D,0x3E,0x77,0xBE,0x1F,0x3F,0x1A,0xDF,0x47,0x3E,0xF8,0x53,0x63,0x3E
,0x98,0xBE,0xD7,0xBD,0xF8,0x53,0x63,0x3E,0x12,0x83,0xC0,0xBC,0x77,0xBE,0x1F
,0x3F,0xF0,0xA7,0xC6,0xBD,0x77,0xBE,0x1F,0x3F,0xC1,0x1A,0x37,0xBE,0xF8,0x53
,0x63,0x3E,0x06,0x81,0xD5,0xBE,0xF8,0x53,0x63,0x3E,0x06,0x81,0xD5,0xBE,0x9A
,0x99,0x19,0x3E,0xBD,0x1C,0x46,0xBE,0x9A,0x99,0x19,0x3E,0x3C,0x87,0x82,0xBE
,0xE3,0xA5,0x1B,0xBE,0x06,0x81,0xD5,0xBE,0xE3,0xA5,0x1B,0xBE,0x06,0x81,0xD5
,0xBE,0x42,0x60,0x65,0xBE,0x3A,0x08,0x8A,0xBE,0x42,0x60,0x65,0xBE,0x46,0xB6
,0xB3,0xBE,0x77,0xBE,0x1F,0xBF,0x8F,0xC2,0xF5,0xBD,0x9A,0x99,0x19,0x3E,0x1E
,0xDD,0x38,0x3E,0x9A,0x99,0x19,0x3E,0x4D,0xD7,0xF3,0x3D,0xE3,0xA5,0x1B,0xBE
,0x02,0xD3,0x39,0xBE,0xE3,0xA5,0x1B,0xBE,0x1E,0x01,0x02,0x01,0x1E,0x1F,0x10
,0x11,0x12,0x10,0x12,0x13,0x0C,0x0D,0x0E,0x0C,0x0E,0x0F,0x02,0x03,0x04,0x02
,0x04,0x05,0x1E,0x02,0x05,0x1A,0x1B,0x00,0x1A,0x00,0x01,0x18,0x19,0x1A,0x17
,0x18,0x1A,0x17,0x1A,0x01,0x17,0x01,0x1F,0x16,0x17,0x1F,0x16,0x1F,0x1C,0x14
,0x15,0x16,0x13,0x14,0x16,0x13,0x16,0x1C,0x10,0x13,0x1C,0x10,0x1C,0x1D,0x0F
,0x10,0x1D,0x0A,0x0B,0x0C,0x09,0x0A,0x0C,0x09,0x0C,0x0F,0x09,0x0F,0x1D,0x08
,0x09,0x1D,0x08,0x1D,0x1E,0x08,0x1E,0x05,0x08,0x05,0x06,0x07,0x08,0x06,0x1B
,0x1F,0x24,0x40,0x42,0x03,0xEC,0x51,0xB8,0xBE,0x7D,0x3F,0x0D,0xBF,0x92,0x03
,0x8E,0xBE,0x44,0x4C,0x15,0xBF,0x6E,0x16,0x4F,0xBE,0x4A,0xB6,0x1A,0xBF,0x30
,0x0D,0x03,0xBE,0x9F,0x04,0x1E,0xBF,0xA6,0x9B,0x44,0xBD,0x77,0xBE,0x1F,0xBF
,0xA6,0x9B,0x44,0xBD,0x7F,0x6A,0x3C,0xBF,0x6F,0x12,0x03,0x3C,0x7F,0x6A,0x3C
,0xBF,0x6F,0x12,0x03,0x3C,0x77,0xBE,0x1F,0xBF,0x09,0x54,0x1F,0x3E,0xEB,0x1C
,0x17,0xBF,0xD7,0xA3,0x88,0x3E,0x9C,0x6F,0x04,0xBF,0x6F,0x66,0xAC,0x3E,0xF0
,0xBF,0xD5,0xBE,0xEC,0x51,0xB8,0x3E,0x23,0x2F,0x9B,0xBE,0xF1,0x29,0xB0,0x3E
,0x6C,0x92,0x4F,0xBE,0x25,0xB2,0x97,0x3E,0xBD,0x1C,0xF6,0xBD,0xEC,0xF7,0x44
,0x3E,0x0D,0x6F,0xD6,0xBC,0x6F,0x12,0x03,0x3C,0x88,0x66,0xDE,0x3D,0x6F,0x12
,0x03,0x3C,0x5C,0x8F,0x0E,0x3F,0x44,0x33,0xEF,0x3D,0xAF,0x06,0x08,0x3F,0x02
,0x2B,0x47,0x3E,0x16,0x85,0xF5,0x3E,0xD1,0xCE,0x79,0x3E,0x79,0x94,0xCA,0x3E
,0x4C,0x37,0x89,0x3E,0xBA,0x49,0x8C,0x3E,0xBE,0x9F,0x9A,0x3E,0xBA,0x49,0x8C
,0x3E,0xBE,0x9F,0x9A,0x3E,0xE1,0xCE,0x09,0x3F,0xA0,0x1A,0x3F,0x3E,0xC8,0xB6
,0x14,0x3F,0x9C,0x14,0x06,0x3E,0x6B,0x66,0x19,0x3F,0x10,0x25,0x9A,0x3D,0xD0
,0x0E,0x1C,0x3F,0x6F,0x12,0x03,0x3C,0x2D,0xB2,0x1D,0x3F,0x6F,0x12,0x03,0x3C
,0x21,0xB0,0x32,0x3F,0xA6,0x9B,0x44,0xBD,0x21,0xB0,0x32,0x3F,0xA6,0x9B,0x44
,0xBD,0x2D,0xB2,0x1D,0x3F,0x6C,0x3F,0x39,0xBE,0xB6,0x49,0x15,0x3F,0x79,0xE9
,0x8E,0xBE,0xD7,0xA3,0x04,0x3F,0xC6,0xF7,0xAD,0xBE,0x75,0x3E,0xDC,0x3E,0xEC
,0x51,0xB8,0xBE,0x37,0x35,0xA8,0x3E,0xF0,0x52,0xB2,0xBE,0xB4,0x74,0x75,0x3E
,0xFC,0x55,0xA0,0xBE,0x31,0x60,0x29,0x3E,0x40,0xDB,0x5A,0xBE,0xA0,0x16,0x83
,0x3D,0xA6,0x9B,0x44,0xBD,0xAA,0xF1,0x52,0xBD,0xA6,0x9B,0x44,0xBD,0x4E,0x62
,0x10,0xBF,0x9A,0x42,0x27,0xBE,0x28,0xF1,0x09,0xBF,0x3B,0xE3,0x7B,0xBE,0xB0
,0xC6,0xF9,0xBE,0x6A,0xA5,0x98,0xBE,0x58,0xE2,0xD1,0xBE,0x2F,0xDD,0xA4,0xBE
,0xE3,0xA5,0x9B,0xBE,0xEC,0x51,0xB8,0xBE,0xE3,0xA5,0x9B,0xBE,0xBA,0xF9,0x06
,0xBE,0x4C,0x38,0x54,0x3E,0x52,0xB8,0x3E,0xBE,0xA0,0xC4,0x87,0x3E,0x5B,0x3F
,0x5D,0xBE,0xAD,0x89,0xA5,0x3E,0x8B,0x6C,0x67,0xBE,0xB4,0x74,0xC5,0x3E,0x3B
,0x34,0x5C,0xBE,0xB7,0xB5,0xE5,0x3E,0x4C,0x8B,0x3A,0xBE,0xD7,0xF9,0xFF,0x3E
,0x94,0xC1,0x01,0xBE,0xA6,0x64,0x09,0x3F,0xA6,0x9B,0x44,0xBD,0x04,0x56,0x0E
,0x3F,0xA6,0x9B,0x44,0xBD,0x78,0x0E,0x15,0x3E,0x6F,0x12,0x03,0x3C,0xFD,0xF9
,0xB6,0xBD,0x4D,0x2D,0xFB,0x3D,0x23,0x82,0x31,0xBE,0xC3,0xF1,0x3C,0x3E,0x06
,0x85,0x71,0xBE,0x33,0x32,0x58,0x3E,0xC8,0x0C,0x94,0xBE,0xAE,0x47,0x61,0x3E
,0x70,0xE9,0xB0,0xBE,0xD1,0xCE,0x59,0x3E,0x70,0xD2,0xCC,0xBE,0x7C,0x64,0x43
,0x3E,0x56,0x0C,0xE7,0xBE,0xA4,0x18,0x20,0x3E,0x92,0x03,0xFE,0xBE,0x12,0xF8
,0xE3,0x3D,0x6A,0x12,0x08,0xBF,0xD4,0x45,0x8A,0x3D,0x99,0xD8,0x0C,0xBF,0x6F
,0x12,0x03,0x3C,0x4E,0x62,0x10,0xBF,0x33,0x06,0x07,0x06,0x33,0x34,0x07,0x3D
,0x3E,0x3D,0x07,0x08,0x07,0x3E,0x3F,0x33,0x07,0x3F,0x33,0x3F,0x35,0x2A,0x2B
,0x00,0x29,0x2A,0x00,0x28,0x29,0x00,0x28,0x00,0x01,0x27,0x28,0x01,0x27,0x01
,0x02,0x27,0x02,0x03,0x26,0x27,0x03,0x26,0x03,0x04,0x1B,0x1C,0x1D,0x1A,0x1B
,0x1D,0x14,0x15,0x16,0x13,0x14,0x16,0x12,0x13,0x16,0x12,0x16,0x17,0x11,0x12
,0x17,0x11,0x17,0x18,0x10,0x11,0x18,0x10,0x18,0x19,0x10,0x19,0x1A,0x10,0x1A
,0x1D,0x04,0x05,0x06,0x3C,0x3D,0x08,0x3C,0x08,0x09,0x3B,0x3C,0x09,0x3B,0x09
,0x0A,0x3A,0x3B,0x0A,0x39,0x3A,0x0A,0x39,0x0A,0x0B,0x38,0x39,0x0B,0x38,0x0B
,0x0C,0x37,0x38,0x0C,0x37,0x0C,0x0D,0x37,0x0D,0x0E,0x36,0x37,0x0E,0x35,0x36
,0x0E,0x35,0x0E,0x0F,0x33,0x35,0x0F,0x33,0x0F,0x10,0x33,0x10,0x1D,0x33,0x1D
,0x1E,0x32,0x33,0x1E,0x31,0x32,0x1E,0x31,0x1E,0x1F,0x30,0x31,0x1F,0x30,0x1F
,0x20,0x2F,0x30,0x20,0x2F,0x20,0x21,0x2E,0x2F,0x21,0x2E,0x21,0x22,0x2D,0x2E
,0x22,0x2D,0x22,0x23,0x2D,0x23,0x24,0x2C,0x2D,0x24,0x34,0x2C,0x24,0x34,0x24
,0x25,0x06,0x34,0x25,0x06,0x25,0x26,0x04,0x06,0x26,0x2B,0x34,0x3F,0x25,0x50
,0x4E,0x05,0xB8,0x76,0xCA,0x3E,0x64,0x3B,0x1F,0x3F,0xD7,0x4B,0xF3,0xBE,0x66
,0x66,0x26,0xBF,0xC1,0x1E,0xCB,0xBE,0x66,0x66,0x26,0xBF,0xBE,0x9F,0xF2,0x3E
,0x64,0x3B,0x1F,0x3F,0x57,0x7A,0xFD,0xBE,0x65,0xC5,0x1C,0x3F,0xCB,0xA0,0x0E
,0xBF,0x56,0x63,0x15,0x3F,0xB5,0xA7,0x1C,0xBF,0x86,0xFF,0x08,0x3F,0x21,0x5A
,0x27,0xBF,0x25,0x08,0xEF,0x3E,0x47,0x20,0x2E,0xBF,0x2E,0x1B,0xC5,0x3E,0x4E
,0x62,0x30,0xBF,0x9B,0x70,0x97,0x3E,0x6B,0x27,0x2E,0xBF,0xCD,0x22,0x54,0x3E
,0xC5,0x76,0x27,0xBF,0x09,0xFC,0x01,0x3E,0x91,0x0C,0x1D,0xBF,0x2E,0xFE,0x76
,0x3D,0xD7,0xA4,0x0F,0xBF,0xAB,0x07,0x4C,0x3C,0xD1,0x02,0x00,0xBF,0x08,0x3E
,0x86,0xBC,0xF2,0xD2,0xDD,0xBE,0xF4,0xFD,0xD4,0xBC,0xB3,0x5E,0xAC,0xBE,0x98
,0xF8,0xA3,0xBB,0x95,0x99,0x82,0xBE,0xA1,0x80,0x6D,0x3D,0xF9,0x4E,0x4C,0xBE
,0x92,0x40,0x23,0x3E,0x10,0x58,0x39,0xBE,0x9B,0x70,0x97,0x3E,0x0B,0x2A,0x4A
,0xBE,0x8F,0x56,0xD5,0x3E,0xB6,0x9F,0x7C,0xBE,0x48,0xE2,0x05,0x3F,0x2E,0xC8
,0xA6,0xBE,0x1D,0xE5,0x18,0x3F,0x81,0xE9,0xDC,0xBE,0x64,0x3B,0x1F,0x3F,0xF8
,0xA9,0xCA,0xBE,0x54,0x8C,0x0F,0x3F,0x98,0x6E,0xBA,0xBE,0x58,0xE5,0x06,0x3F
,0x33,0xDF,0xA9,0xBE,0x0A,0x2E,0xDE,0x3E,0x1D,0x5A,0xA4,0xBE,0x04,0x02,0x95
,0x3E,0xE7,0xFB,0xA9,0xBE,0xC1,0x70,0x1E,0x3E,0x48,0xE1,0xBA,0xBE,0x67,0xF1
,0x82,0x3D,0xC1,0xC8,0xCB,0xBE,0x6E,0x88,0xF1,0x3C,0xF9,0xD6,0xDF,0xBE,0xBC
,0x74,0x93,0x3C,0x25,0xB1,0xEC,0xBE,0x1C,0x09,0xB4,0x3C,0xB6,0x47,0xF7,0xBE
,0x05,0xE1,0x0A,0x3D,0x9F,0xAE,0x02,0xBF,0x4F,0x06,0x87,0x3D,0x81,0x95,0x07
,0xBF,0x64,0x3B,0xDF,0x3D,0xCF,0x4D,0x0B,0xBF,0x85,0x97,0x40,0x3E,0x44,0x8B
,0x0C,0xBF,0x56,0xB6,0x97,0x3E,0xB1,0x51,0x0A,0xBF,0x6E,0xA5,0xDF,0x3E,0xD7
,0xA4,0x03,0xBF,0x23,0xDC,0x04,0x3F,0xB1,0x30,0xF4,0xBE,0xFA,0x09,0x0F,0x3F
,0x0B,0x7F,0xDE,0xBE,0x98,0x6E,0x12,0x3F,0x54,0x8D,0xBE,0x3E,0x26,0x1B,0x8F
,0xBC,0x6B,0x64,0x9F,0x3E,0x6D,0xE7,0x3B,0xBD,0x8A,0xAE,0x83,0x3E,0x53,0x5A
,0xBF,0xBD,0x88,0x12,0x5D,0x3E,0x26,0x35,0x24,0xBE,0xBF,0x46,0x42,0x3E,0x67
,0xB7,0x76,0xBE,0x10,0x58,0x39,0x3E,0x19,0xAC,0xA8,0xBE,0xBF,0x46,0x42,0x3E
,0x0F,0x44,0xD6,0xBE,0x88,0x12,0x5D,0x3E,0x64,0x91,0xFE,0xBE,0x8A,0xAE,0x83
,0x3E,0xED,0xF0,0x0F,0xBF,0x6B,0x64,0x9F,0x3E,0x89,0x41,0x1C,0xBF,0x50,0xE3
,0xBE,0x3E,0x33,0xDD,0x23,0xBF,0x6B,0x64,0xDF,0x3E,0x66,0x66,0x26,0xBF,0x99
,0x2E,0x08,0x3F,0x47,0x21,0x21,0xBF,0xF2,0x28,0x1D,0x3F,0xE7,0x51,0x11,0xBF
,0xFF,0x93,0x2B,0x3F,0x73,0x11,0xEF,0xBE,0x4E,0x62,0x30,0x3F,0x19,0xAC,0xA8
,0xBE,0xFC,0x34,0x2E,0x3F,0xBF,0x48,0x78,0xBE,0x08,0xAD,0x27,0x3F,0x47,0xE5
,0x26,0xBE,0xAE,0x28,0x1D,0x3F,0xE0,0x2C,0xC5,0xBD,0x2D,0x06,0x0F,0x3F,0x6F
,0x12,0x43,0xBD,0xA4,0x1A,0xFE,0x3E,0x9B,0xAF,0x92,0xBC,0x5B,0x0C,0xDE,0x3E
,0x6F,0x12,0x03,0xBC,0xCB,0xA0,0xF2,0x3E,0xBC,0x95,0x85,0xBD,0x25,0xB0,0x01
,0x3F,0x3B,0x8F,0xCA,0xBD,0x84,0xD4,0x09,0x3F,0x77,0x66,0x42,0xBE,0x44,0x8B
,0x0C,0x3F,0x5C,0x3B,0xA9,0xBE,0x84,0xD4,0x09,0x3F,0xE7,0xA6,0xED,0xBE,0x25
,0xB0,0x01,0x3F,0x3C,0x33,0x0D,0xBF,0xB2,0xF4,0xF1,0x3E,0x06,0x80,0x16,0xBF
,0xA2,0x45,0xDE,0x3E,0x9A,0x99,0x19,0xBF,0xC9,0x21,0xCA,0x3E,0xD4,0xB9,0x16
,0xBF,0xFA,0x2A,0xB9,0x3E,0xA4,0x1A,0x0E,0xBF,0x54,0x8E,0xA9,0x3E,0x1D,0x05
,0xF0,0xBE,0x1D,0x5A,0xA4,0x3E,0xFE,0x7C,0xAB,0xBE,0x9F,0x71,0xA9,0x3E,0xD7
,0x4B,0x43,0xBE,0x4A,0xB8,0xB8,0x3E,0x43,0x00,0xD0,0xBD,0x14,0x05,0xCA,0x3E
,0xDC,0xF1,0x86,0xBD,0xA2,0x45,0xDE,0x3E,0x1B,0x2F,0x5D,0xBD,0x00,0x01,0x02
,0x00,0x02,0x03,0x05,0x25,0x26,0x05,0x26,0x27,0x25,0x05,0x06,0x25,0x06,0x07
,0x25,0x07,0x08,0x25,0x08,0x09,0x05,0x27,0x28,0x04,0x05,0x28,0x17,0x04,0x28
,0x17,0x28,0x29,0x16,0x17,0x29,0x16,0x29,0x18,0x16,0x18,0x19,0x15,0x16,0x19
,0x15,0x19,0x1A,0x14,0x15,0x1A,0x14,0x1A,0x1B,0x13,0x14,0x1B,0x12,0x13,0x1B
,0x12,0x1B,0x1C,0x11,0x12,0x1C,0x11,0x1C,0x1D,0x10,0x11,0x1D,0x10,0x1D,0x1E
,0x0F,0x10,0x1E,0x0F,0x1E,0x1F,0x0E,0x0F,0x1F,0x0E,0x1F,0x20,0x0E,0x20,0x21
,0x0D,0x0E,0x21,0x0D,0x21,0x22,0x25,0x09,0x0A,0x24,0x25,0x0A,0x24,0x0A,0x0B
,0x23,0x24,0x0B,0x23,0x0B,0x0C,0x23,0x0C,0x0D,0x23,0x0D,0x22,0x2B,0x4B,0x4C
,0x2B,0x4C,0x4D,0x4B,0x2B,0x2C,0x4B,0x2C,0x2D,0x4B,0x2D,0x2E,0x4B,0x2E,0x2F
,0x2A,0x2B,0x4D,0x2A,0x4D,0x4E,0x3F,0x2A,0x4E,0x3F,0x4E,0x4F,0x3E,0x3F,0x4F
,0x3E,0x4F,0x40,0x4B,0x2F,0x30,0x4A,0x4B,0x30,0x4A,0x30,0x31,0x4A,0x31,0x32
,0x49,0x4A,0x32,0x49,0x32,0x33,0x49,0x33,0x34,0x48,0x49,0x34,0x48,0x34,0x35
,0x47,0x48,0x35,0x47,0x35,0x36,0x46,0x47,0x36,0x45,0x46,0x36,0x45,0x36,0x37
,0x44,0x45,0x37,0x44,0x37,0x38,0x43,0x44,0x38,0x43,0x38,0x39,0x43,0x39,0x3A
,0x42,0x43,0x3A,0x42,0x3A,0x3B,0x42,0x3B,0x3C,0x41,0x42,0x3C,0x41,0x3C,0x3D
,0x41,0x3D,0x3E,0x41,0x3E,0x40,0x03,0x17,0x29,0x3F,0x4F,0x26,0x5E,0x60,0x03
,0x1B,0x2F,0x5D,0x3E,0x4C,0x37,0x09,0x3E,0x76,0xE3,0x7D,0x3E,0x13,0xB6,0xFF
,0x3D,0x0C,0x02,0x8B,0x3E,0x10,0x58,0xD9,0x3D,0xF2,0x28,0x95,0x3E,0x95,0x9F
,0x94,0x3D,0x54,0x8B,0x98,0x3E,0x73,0xF3,0x0D,0x3D,0xD1,0x76,0x84,0x3E,0x94
,0x14,0xD8,0xBD,0xD2,0x72,0x10,0x3E,0x0F,0xD5,0x8C,0xBE,0x7A,0xFE,0xB4,0x3C
,0xF5,0x49,0xCE,0xBD,0x12,0x83,0xC0,0xBD,0xA3,0x20,0xB8,0x3D,0xBD,0xA7,0x32
,0x3D,0x7E,0x3A,0x2E,0x3E,0xA6,0x43,0x07,0x3E,0xEC,0xA5,0x79,0x3E,0xF3,0x57
,0x38,0x3E,0x2E,0xC7,0xA3,0x3E,0x39,0xB4,0x48,0x3E,0x85,0x97,0xD0,0x3E,0xCD
,0xCD,0x37,0x3E,0xF4,0x50,0xFB,0x3E,0x87,0x1A,0x05,0x3E,0x29,0x5C,0x0F,0x3F
,0xF3,0x59,0x5E,0x3D,0x96,0x43,0x1B,0x3F,0xFA,0x09,0x27,0xBD,0x64,0x3B,0x1F
,0x3F,0x1D,0x5A,0x14,0xBE,0xA6,0x47,0x1B,0x3F,0xC5,0xC8,0x72,0xBE,0x8B,0x6C
,0x0F,0x3F,0xD3,0xFA,0xA3,0xBE,0x3D,0x0C,0xED,0x3E,0x0E,0x2D,0xB2,0xBE,0xC3
,0xF5,0xA8,0x3E,0xFF,0x3D,0xB0,0xBE,0x43,0xC9,0x8C,0x3E,0xB0,0x70,0xAA,0xBE
,0x84,0xBC,0x5E,0x3E,0x34,0x4A,0x9F,0xBE,0x12,0x30,0x1A,0x3E,0xDF,0x4F,0x8D
,0xBE,0x94,0x68,0x89,0x3D,0xCB,0xA2,0xE8,0xBE,0xB7,0x09,0x37,0xBD,0x00,0xAA
,0x10,0xBF,0x5F,0x62,0x1C,0xBE,0x0D,0x6E,0x1F,0xBF,0x8D,0x99,0x84,0xBE,0x1D
,0x5A,0x24,0xBF,0xC1,0x1E,0xBB,0xBE,0xFC,0xDE,0x1E,0xBF,0xDB,0x4F,0xEE,0xBE
,0x8B,0x6D,0x0E,0xBF,0x89,0xEB,0x0C,0xBF,0x7A,0x53,0xE9,0xBE,0x53,0xCE,0x1B
,0xBF,0x87,0x6E,0xA6,0xBE,0x9C,0xC4,0x20,0xBF,0x90,0xBE,0x69,0xBE,0x54,0x57
,0x1E,0xBF,0x49,0xBC,0x0C,0xBE,0x5A,0x0F,0x17,0xBF,0x6D,0xFD,0x34,0xBD,0xC7
,0x7F,0x09,0xBF,0x7D,0x3F,0x75,0x3D,0xC0,0x76,0xE8,0xBE,0xE3,0xFE,0x13,0x3E
,0xD5,0xED,0x08,0xBF,0x14,0xB2,0x63,0x3E,0xA2,0x9A,0x16,0xBF,0x97,0xAB,0x97
,0x3E,0x19,0x3A,0x1E,0xBF,0x91,0xED,0xBC,0x3E,0x9C,0xC4,0x20,0xBF,0xBC,0x08
,0xEB,0x3E,0x0B,0xED,0x1C,0xBF,0x23,0x86,0x09,0x3F,0x6B,0x66,0x11,0xBF,0x18
,0x79,0x19,0x3F,0xC5,0xCB,0xFB,0xBE,0x1D,0x5A,0x24,0x3F,0x9A,0x41,0xC4,0xBE
,0x00,0x00,0x1C,0x3F,0x62,0x10,0xB8,0xBE,0xFF,0xEA,0x15,0x3F,0x1D,0x04,0xD5
,0xBE,0x10,0xAE,0x0C,0x3F,0xC7,0xF3,0xE9,0xBE,0x97,0xAE,0x00,0x3F,0xD7,0xA5
,0xF6,0xBE,0xE8,0xA3,0xE4,0x3E,0x48,0xE1,0xFA,0xBE,0x01,0xC1,0xC4,0x3E,0xC6
,0x35,0xF6,0xBE,0x8F,0x18,0xA5,0x3E,0x40,0x33,0xE8,0xBE,0xFD,0x69,0x83,0x3E
,0x78,0x7D,0xCE,0xBE,0x68,0xE9,0x3A,0x3E,0x52,0xB8,0xA6,0xBE,0x85,0xEC,0x8C
,0x3E,0x74,0x41,0x4D,0xBE,0x49,0x64,0xB7,0x3E,0x4B,0x5C,0x87,0xBD,0xC4,0x0B
,0xDA,0x3E,0x56,0x45,0x38,0x3D,0x73,0x12,0xF2,0x3E,0x05,0x89,0xCD,0x3D,0x7F
,0xF5,0x04,0x3F,0x70,0xEF,0xFA,0x3D,0x19,0x04,0x16,0x3F,0x4C,0x37,0x09,0x3E
,0x19,0x04,0x16,0x3F,0x31,0x08,0x2C,0x3E,0x1B,0x2F,0x5D,0x3E,0x31,0x08,0x2C
,0x3E,0xE9,0x80,0x24,0xBE,0x07,0x27,0x72,0x3E,0x3A,0x5C,0x3B,0xBE,0xEC,0xF9
,0x9A,0x3E,0x60,0xE7,0x46,0xBE,0x8C,0x15,0xB5,0x3E,0x83,0xC0,0x4A,0xBE,0x96
,0x97,0xCC,0x3E,0x18,0xB4,0x40,0xBE,0x6C,0x26,0xE7,0x3E,0x5C,0x8F,0x22,0xBE
,0xD5,0x23,0x01,0x3F,0x37,0x89,0xE1,0xBD,0xC2,0x6B,0x0B,0x3F,0xA0,0x1A,0x2F
,0xBD,0x17,0xD9,0x0E,0x3F,0xC5,0x20,0x30,0x3C,0xA5,0x2F,0x0C,0x3F,0x89,0x41
,0x60,0x3D,0x2F,0x33,0x04,0x3F,0xC2,0x6A,0xAC,0x3D,0x99,0x45,0xF0,0x3E,0x12
,0x83,0xC0,0x3D,0xEA,0x7A,0xD2,0x3E,0x59,0xC0,0xA4,0x3D,0x4E,0xB9,0xAA,0x3E
,0x67,0xF1,0x22,0x3D,0xF6,0xD0,0x86,0x3E,0xC1,0x1A,0xE7,0xBC,0xE7,0x55,0x4D
,0x3E,0xA4,0x70,0xFD,0xBD,0x3D,0x62,0x14,0x3E,0x04,0x00,0x87,0xBD,0x6A,0xA4
,0xED,0xBE,0x9C,0x18,0x02,0xBE,0x5E,0xBC,0xFF,0xBE,0x76,0x17,0x38,0xBE,0xE4
,0x30,0x04,0xBF,0xC5,0x20,0x70,0xBE,0xCB,0xA1,0x05,0xBF,0x13,0x9B,0xA7,0xBE
,0xDC,0x0E,0x01,0xBF,0xA1,0x47,0xCC,0xBE,0x00,0xAC,0xE6,0xBE,0xBF,0xB5,0xE3
,0xBE,0xCD,0x23,0xBF,0xBE,0x1F,0x85,0xEB,0xBE,0x5E,0xBA,0x91,0xBE,0xEC,0xA6
,0xE4,0xBE,0x0D,0xFE,0x4E,0xBE,0x52,0x0C,0xD0,0xBE,0x20,0xD5,0x00,0xBE,0xE5
,0xD1,0xAD,0xBE,0x8C,0xF7,0x63,0xBD,0xF6,0x28,0x7C,0xBE,0x40,0x33,0x08,0x3C
,0xF8,0x4F,0x47,0xBE,0xEF,0x01,0xBA,0xBD,0xBA,0xF9,0x16,0xBE,0x65,0x8F,0x30
,0xBE,0x3B,0x37,0xAD,0xBD,0x4F,0x75,0x88,0xBE,0x42,0x60,0x65,0x3C,0x77,0x6A
,0xCE,0xBE,0x08,0x4B,0x4C,0x59,0x06,0x07,0x4B,0x08,0x09,0x06,0x59,0x5A,0x08
,0x4C,0x4D,0x06,0x5A,0x5B,0x59,0x07,0x08,0x06,0x5B,0x5C,0x06,0x5C,0x5D,0x4B
,0x09,0x0A,0x4A,0x4B,0x0A,0x4A,0x0A,0x0B,0x2C,0x2D,0x2E,0x2B,0x2C,0x2E,0x2B
,0x2E,0x2F,0x2A,0x2B,0x2F,0x2A,0x2F,0x30,0x49,0x4A,0x0B,0x49,0x0B,0x0C,0x49
,0x0C,0x0D,0x48,0x49,0x0D,0x48,0x0D,0x0E,0x47,0x48,0x0E,0x47,0x0E,0x0F,0x46
,0x47,0x0F,0x46,0x0F,0x10,0x45,0x46,0x10,0x45,0x10,0x11,0x44,0x45,0x11,0x59
,0x08,0x4D,0x3C,0x3D,0x00,0x3C,0x00,0x01,0x3C,0x01,0x02,0x3C,0x02,0x03,0x44
,0x11,0x12,0x43,0x44,0x12,0x43,0x12,0x13,0x42,0x43,0x13,0x41,0x42,0x13,0x41
,0x13,0x14,0x40,0x41,0x14,0x3F,0x40,0x14,0x3F,0x14,0x15,0x3F,0x15,0x16,0x3E
,0x3F,0x16,0x3E,0x16,0x17,0x4D,0x3E,0x17,0x59,0x4D,0x17,0x59,0x17,0x18,0x59
,0x18,0x19,0x58,0x59,0x19,0x57,0x58,0x19,0x57,0x19,0x1A,0x56,0x57,0x1A,0x56
,0x1A,0x1B,0x55,0x56,0x1B,0x55,0x1B,0x1C,0x54,0x55,0x1C,0x54,0x1C,0x1D,0x54
,0x1D,0x1E,0x53,0x54,0x1E,0x53,0x1E,0x1F,0x52,0x53,0x1F,0x52,0x1F,0x20,0x51
,0x52,0x20,0x51,0x20,0x21,0x51,0x21,0x22,0x50,0x51,0x22,0x4F,0x50,0x22,0x4F
,0x22,0x23,0x4E,0x4F,0x23,0x4E,0x23,0x24,0x5D,0x4E,0x24,0x06,0x5D,0x24,0x3C
,0x03,0x04,0x29,0x2A,0x30,0x29,0x30,0x31,0x28,0x29,0x31,0x28,0x31,0x32,0x27
,0x28,0x32,0x27,0x32,0x33,0x26,0x27,0x33,0x26,0x33,0x34,0x25,0x26,0x34,0x24
,0x25,0x34,0x24,0x34,0x35,0x06,0x24,0x35,0x06,0x35,0x36,0x05,0x06,0x36,0x05
,0x36,0x37,0x04,0x05,0x37,0x04,0x37,0x38,0x3A,0x3B,0x3C,0x3A,0x3C,0x04,0x39
,0x3A,0x04,0x39,0x04,0x38,0x3D,0x4D,0x5D,0x27,0x0F,0x0D,0x01,0xE3,0xA5,0x9B
,0x3C,0x1B,0x2F,0xDD,0x3D,0x14,0xAE,0x87,0x3D,0x58,0x39,0xBC,0x3E,0x02,0xA0
,0xAA,0x3D,0x5A,0x66,0xF1,0x3E,0xA2,0x45,0xB6,0x3D,0xB0,0x1E,0x07,0x3F,0x82
,0xE5,0xA8,0x3D,0xDA,0xE4,0x10,0x3F,0x9C,0xC4,0x80,0x3D,0x7D,0x95,0x18,0x3F
,0x8D,0x7A,0x08,0x3D,0xEF,0x91,0x1D,0x3F,0xB5,0xC4,0x4A,0xBB,0x64,0x3B,0x1F
,0x3F,0x8B,0xA7,0x1E,0xBD,0x13,0x99,0x1D,0x3F,0xC1,0x1A,0x87,0xBD,0x31,0xB2
,0x18,0x3F,0xC8,0x7A,0xAA,0xBD,0xA1,0x30,0x10,0x3F,0xA2,0x45,0xB6,0xBD,0x77
,0xBE,0x03,0x3F,0x2E,0x91,0xAB,0xBD,0xA4,0x19,0xEB,0x3E,0xDE,0x74,0x8B,0xBD
,0x18,0xB0,0xBC,0x3E,0xE3,0xA5,0x9B,0xBC,0x1B,0x2F,0xDD,0x3D,0x0E,0x00,0x01
,0x0D,0x0E,0x01,0x0D,0x01,0x02,0x0C,0x0D,0x02,0x0B,0x0C,0x02,0x0B,0x02,0x03
,0x0A,0x0B,0x03,0x0A,0x03,0x04,0x0A,0x04,0x05,0x0A,0x05,0x06,0x0A,0x06,0x07
,0x0A,0x07,0x08,0x09,0x0A,0x08,0x0E,0x28,0x1A,0x18,0x01,0xEA,0x79,0x27,0x3E
,0x7E,0x8A,0x63,0xBF,0xF9,0x4B,0xCB,0x3D,0x79,0x94,0x52,0xBF,0xFB,0xB1,0x49
,0x3D,0xAD,0xDB,0x40,0xBF,0x59,0x4F,0x2D,0x3C,0x73,0x67,0x2E,0xBF,0xDD,0x7C
,0xA3,0xBC,0xDC,0x83,0x18,0xBF,0xC6,0xC0,0x3A,0xBD,0xD3,0xF9,0xF8,0xBE,0x09
,0x8B,0x8A,0xBD,0x14,0xB0,0xA5,0xBE,0x9A,0x99,0x99,0xBD,0x54,0x3B,0x13,0xBE
,0x03,0xCD,0x87,0xBD,0xE0,0x14,0xD6,0x3C,0x68,0xCC,0x24,0xBD,0x37,0x89,0x41
,0x3E,0x82,0xC5,0x61,0x3B,0x18,0x41,0xAB,0x3E,0x14,0x06,0x85,0x3D,0x37,0x8B
,0xE7,0x3E,0x04,0xA9,0x14,0x3E,0x0D,0x8D,0x0B,0x3F,0x48,0xE1,0x7A,0x3E,0x2D
,0xB2,0x1D,0x3F,0x48,0xE1,0x7A,0x3E,0x8B,0x6C,0x27,0x3F,0xFE,0x99,0x41,0x3D
,0xD0,0x61,0x06,0x3F,0x6F,0x9D,0xDF,0xBD,0x13,0xD7,0xA9,0x3E,0xB4,0x1C,0x58
,0xBE,0x84,0x64,0xC1,0x3D,0x48,0xE1,0x7A,0xBE,0x48,0xE1,0x2A,0xBE,0xD4,0x27
,0x69,0xBE,0x43,0x75,0xB3,0xBE,0x01,0xFC,0x33,0xBE,0x4A,0x0B,0x07,0xBF,0x0F
,0x2B,0xBC,0xBD,0xB8,0x93,0x30,0xBF,0x9A,0x98,0xAE,0x3C,0x79,0x3D,0x54,0xBF
,0x2B,0xFC,0xF9,0x3D,0xCD,0xCB,0x69,0xBF,0x48,0xE1,0x7A,0x3E,0x7F,0x6A,0x7C
,0xBF,0x48,0xE1,0x7A,0x3E,0x46,0xB6,0x73,0xBF,0x0D,0x0E,0x0F,0x0C,0x0D,0x0F
,0x0B,0x0C,0x0F,0x0B,0x0F,0x10,0x0A,0x0B,0x10,0x17,0x18,0x19,0x17,0x19,0x00
,0x09,0x0A,0x10,0x09,0x10,0x11,0x08,0x09,0x11,0x08,0x11,0x12,0x07,0x08,0x12
,0x17,0x00,0x01,0x16,0x17,0x01,0x16,0x01,0x02,0x15,0x16,0x02,0x15,0x02,0x03
,0x15,0x03,0x04,0x14,0x15,0x04,0x14,0x04,0x05,0x06,0x07,0x12,0x06,0x12,0x13
,0x06,0x13,0x14,0x05,0x06,0x14,0x19,0x29,0x1A,0x18,0x01,0x85,0xEA,0x26,0xBE
,0x66,0x86,0x0D,0x3F,0x56,0x66,0xCA,0xBD,0xC0,0x20,0xF9,0x3E,0x32,0xC9,0x48
,0xBD,0xF9,0xBD,0xD5,0x3E,0x54,0xE3,0x25,0xBC,0x15,0x02,0xB1,0x3E,0xDA,0x1F
,0xA8,0x3C,0x6B,0x66,0x85,0x3E,0xC6,0xC0,0x3A,0x3D,0xA3,0xC8,0x1A,0x3E,0x09
,0x8B,0x8A,0x3D,0x84,0xD6,0x43,0xBC,0x9A,0x99,0x99,0x3D,0xAE,0xEF,0x43,0xBE
,0x42,0x08,0x88,0x3D,0x18,0x05,0xB9,0xBE,0x76,0xA8,0x26,0x3D,0xC3,0x49,0x06
,0xBF,0xC8,0x45,0x35,0xBB,0x4A,0x96,0x2B,0xBF,0x93,0x18,0x84,0xBD,0x10,0xAD
,0x49,0xBF,0x22,0x8B,0x14,0xBE,0x4C,0x6D,0x61,0xBF,0x48,0xE1,0x7A,0xBE,0x46
,0xB6,0x73,0xBF,0x48,0xE1,0x7A,0xBE,0x7F,0x6A,0x7C,0xBF,0x6C,0x5C,0x3F,0xBD
,0xC4,0x5F,0x5B,0xBF,0x12,0x83,0xE0,0x3D,0x7D,0xE9,0x29,0xBF,0x47,0x39,0x58
,0x3E,0x08,0x55,0xDA,0xBE,0x48,0xE1,0x7A,0x3E,0x87,0x16,0x29,0xBE,0xD4,0x27
,0x69,0x3E,0xD8,0x10,0x9C,0x3C,0x01,0xFC,0x33,0x3E,0xE7,0xA7,0x48,0x3E,0x8C
,0xA1,0xBC,0x3D,0x86,0x56,0xB7,0x3E,0x7C,0x2A,0xA7,0xBC,0xBB,0xF1,0xFE,0x3E
,0xA6,0x97,0xF8,0xBD,0xD7,0xF8,0x14,0x3F,0x48,0xE1,0x7A,0xBE,0x8B,0x6C,0x27
,0x3F,0x48,0xE1,0x7A,0xBE,0x2D,0xB2,0x1D,0x3F,0x0D,0x0E,0x0F,0x0C,0x0D,0x0F
,0x0B,0x0C,0x0F,0x0B,0x0F,0x10,0x0A,0x0B,0x10,0x17,0x18,0x19,0x17,0x19,0x00
,0x09,0x0A,0x10,0x09,0x10,0x11,0x08,0x09,0x11,0x08,0x11,0x12,0x07,0x08,0x12
,0x17,0x00,0x01,0x16,0x17,0x01,0x16,0x01,0x02,0x15,0x16,0x02,0x15,0x02,0x03
,0x15,0x03,0x04,0x14,0x15,0x04,0x14,0x04,0x05,0x06,0x07,0x12,0x06,0x12,0x13
,0x06,0x13,0x14,0x05,0x06,0x14,0x19,0x2A,0x64,0x62,0x01,0xE0,0x2C,0x85,0xBD
,0x79,0xE9,0xB6,0x3E,0x02,0xD3,0xE9,0xBD,0x3E,0x5E,0xD0,0x3E,0xE8,0xA5,0x22
,0xBE,0xFF,0x3D,0xE8,0x3E,0xB4,0x1C,0x48,0xBE,0x12,0x85,0xF6,0x3E,0x48,0x8A
,0x68,0xBE,0x45,0xA1,0xFD,0x3E,0x81,0x95,0x83,0xBE,0x00,0x00,0x00,0x3F,0x0F
,0x42,0x90,0xBE,0xC0,0x75,0xFD,0x3E,0x6C,0x3D,0x9B,0xBE,0x02,0xD7,0xF5,0x3E
,0x95,0xD7,0xA2,0xBE,0xFE,0x2B,0xEB,0x3E,0x42,0x60,0xA5,0xBE,0xEE,0x7C,0xDF
,0x3E,0xE8,0x4E,0xA0,0xBE,0x0A,0xD7,0xCB,0x3E,0x97,0x1A,0x91,0xBE,0x3A,0x5C
,0xBB,0x3E,0x06,0xD9,0x72,0xBE,0xF0,0x4F,0xB1,0x3E,0xE3,0xFD,0x18,0xBE,0xDD
,0x24,0xA6,0x3E,0x6B,0xEF,0xB3,0xBD,0x00,0x02,0x9E,0x3E,0x04,0xCB,0x11,0xBD
,0xF1,0x2A,0x93,0x3E,0x1E,0xE1,0xB4,0xBD,0x75,0x3D,0x89,0x3E,0xC1,0xCA,0x11
,0xBE,0x29,0x04,0x82,0x3E,0xBB,0xEF,0x58,0xBE,0x98,0xC2,0x73,0x3E,0xA0,0x6E
,0x80,0xBE,0xC1,0x1A,0x67,0x3E,0x9E,0x99,0x90,0xBE,0x4A,0xB4,0x54,0x3E,0xE6
,0x24,0x9C,0xBE,0xF7,0x03,0x3E,0x3E,0x62,0x11,0xA3,0xBE,0x76,0xC0,0x25,0x3E
,0x42,0x60,0xA5,0xBE,0xAE,0x9F,0x0E,0x3E,0x97,0xAC,0xA2,0xBE,0x71,0xE5,0xEC
,0x3D,0x75,0x91,0x9A,0xBE,0x66,0x16,0xC1,0x3D,0x40,0xA1,0x8E,0xBE,0x74,0xEF
,0xA1,0x3D,0xA0,0x6E,0x80,0xBE,0x50,0x8D,0x97,0x3D,0x83,0x18,0x68,0xBE,0x93
,0xC8,0x9E,0x3D,0x69,0x39,0x50,0xBE,0xE1,0x7A,0xB4,0x3D,0x21,0xAC,0x26,0xBE
,0xF8,0x4F,0xF7,0x3D,0x51,0x68,0xD9,0xBD,0x07,0x29,0x38,0x3E,0xF4,0xDC,0x82
,0xBD,0xDE,0x76,0x61,0x3E,0x8F,0xC2,0x75,0xBC,0x3A,0x5C,0x83,0x3E,0x69,0x57
,0xA1,0xBC,0x93,0x19,0x5F,0x3E,0xA5,0x85,0xCB,0xBC,0x2A,0x8B,0x42,0x3E,0x8E
,0x58,0x0B,0xBD,0x4A,0x0C,0x22,0x3E,0x84,0x10,0x50,0xBD,0x7E,0x37,0xDD,0x3D
,0x98,0xA3,0x87,0xBD,0x7C,0x7E,0x58,0x3D,0x0E,0x2D,0x92,0xBD,0x54,0xE1,0xCF
,0x3B,0x6B,0xB8,0x88,0xBD,0x69,0x1C,0xEA,0xBC,0xF6,0xB3,0x58,0xBD,0xBF,0x2A
,0x57,0xBD,0xCA,0x8B,0xCC,0xBC,0x53,0xB4,0x92,0xBD,0xB2,0x9F,0xC5,0x3B,0x77
,0xBE,0x9F,0xBD,0x7F,0xC1,0xEE,0x3C,0x4A,0x60,0x93,0xBD,0x96,0x93,0x50,0x3D
,0x87,0x8B,0x5C,0xBD,0x52,0xB6,0x88,0x3D,0xFC,0xAA,0xDC,0xBC,0xFD,0x84,0x93
,0x3D,0x4C,0x35,0xB3,0x3B,0x99,0xF5,0x82,0x3D,0xB3,0xED,0x74,0x3D,0xD6,0x8E
,0x22,0x3D,0x94,0xC0,0x16,0x3E,0x61,0x70,0xCD,0x3C,0xEC,0xFB,0x50,0x3E,0xE3
,0xA5,0x9B,0x3C,0x3A,0x5C,0x83,0x3E,0x84,0x9B,0x8C,0x3D,0x4D,0x87,0x5E,0x3E
,0x81,0x20,0xE0,0x3D,0x05,0xA6,0x33,0x3E,0xE5,0x7F,0x22,0x3E,0x57,0xED,0xFA
,0x3D,0x7E,0x3B,0x49,0x3E,0xE5,0x80,0xBD,0x3D,0xDB,0xA6,0x68,0x3E,0x57,0x0A
,0xA1,0x3D,0x6C,0x93,0x82,0x3E,0x50,0x8D,0x97,0x3D,0xF7,0x3C,0x8F,0x3E,0x0F
,0x7B,0xA1,0x3D,0x48,0x37,0x9A,0x3E,0xC8,0x43,0xBF,0x3D,0x70,0xD1,0xA1,0x3E
,0x85,0x95,0xEA,0x3D,0x1D,0x5A,0xA4,0x3E,0x6D,0x8F,0x0E,0x3E,0x4B,0x21,0xA0
,0x3E,0x55,0x68,0x30,0x3E,0xD5,0x76,0x93,0x3E,0x71,0x8D,0x4F,0x3E,0x74,0x40
,0x72,0x3E,0x16,0xDD,0x6A,0x3E,0xB9,0xC6,0x17,0x3E,0xB7,0x9B,0x80,0x3E,0xF3
,0x55,0xB2,0x3D,0x67,0x64,0x88,0x3E,0x4D,0xD7,0x13,0x3D,0xB1,0x1A,0x93,0x3E
,0x13,0xD7,0xB1,0x3D,0xC8,0x78,0x9C,0x3E,0xF9,0xA3,0x18,0x3E,0x4A,0xB4,0xA4
,0x3E,0xBC,0x76,0x79,0x3E,0x47,0xCB,0xB1,0x3E,0x00,0xAC,0x96,0x3E,0x50,0xE3
,0xBE,0x3E,0x29,0xB3,0xA1,0x3E,0xC6,0xE1,0xCC,0x3E,0x42,0x60,0xA5,0x3E,0x08
,0xAC,0xDC,0x3E,0x12,0xDA,0xA2,0x3E,0xC8,0xB5,0xE9,0x3E,0xA6,0x47,0x9B,0x3E
,0xF6,0xD2,0xF4,0x3E,0x19,0x02,0x90,0x3E,0x22,0x70,0xFC,0x3E,0x45,0x62,0x82
,0x3E,0xDB,0xF9,0xFE,0x3E,0xFE,0x7D,0x66,0x3E,0x89,0xD4,0xFC,0x3E,0xE7,0xA7
,0x48,0x3E,0x4E,0x64,0xF6,0x3E,0xF7,0x02,0x23,0x3E,0xB7,0x09,0xE7,0x3E,0xBE
,0x9F,0xDA,0x3D,0xEE,0x24,0xCA,0x3E,0x4C,0xC2,0x85,0x3D,0x81,0x42,0xB5,0x3E
,0xE3,0xA5,0x9B,0x3C,0x98,0xC2,0xA3,0x3E,0x2F,0xA2,0xED,0x3C,0x50,0xE2,0xC3
,0x3E,0x49,0x81,0x45,0x3D,0x17,0xD9,0xE6,0x3E,0xE7,0x53,0x87,0x3D,0xBF,0x9E
,0x03,0x3F,0xFD,0x84,0x93,0x3D,0xFA,0xD2,0x0F,0x3F,0x24,0x29,0x89,0x3D,0x2B
,0xDD,0x19,0x3F,0x17,0x29,0x54,0x3D,0xCF,0x4B,0x21,0x3F,0x3A,0x21,0xF4,0x3C
,0x60,0xE4,0x25,0x3F,0xA1,0x47,0x8C,0x3B,0x8B,0x6C,0x27,0x3F,0x6C,0x96,0xCB
,0xBC,0x19,0xE5,0x25,0x3F,0x36,0x73,0x48,0xBD,0xD3,0x4E,0x21,0x3F,0xDE,0x57
,0x85,0xBD,0x6D,0xC6,0x19,0x3F,0x4E,0x62,0x90,0xBD,0x7B,0x68,0x0F,0x3F,0xFF
,0xCC,0x80,0xBD,0xFB,0x94,0x03,0x3F,0x1A,0x19,0x24,0xBD,0x6E,0xBE,0xE1,0x3E
,0x9E,0xEE,0xBC,0xBC,0x9A,0xED,0xC2,0x3E,0x8F,0xC2,0x75,0xBC,0x5C,0xE3,0xA3
,0x3E,0x09,0x0A,0x0B,0x08,0x09,0x0B,0x07,0x08,0x0B,0x06,0x07,0x0B,0x05,0x06
,0x0B,0x04,0x05,0x0B,0x04,0x0B,0x0C,0x03,0x04,0x0C,0x02,0x03,0x0C,0x02,0x0C
,0x0D,0x01,0x02,0x0D,0x00,0x01,0x0D,0x00,0x0D,0x0E,0x63,0x00,0x0E,0x63,0x0E
,0x0F,0x2A,0x2B,0x2C,0x2A,0x2C,0x2D,0x29,0x2A,0x2D,0x29,0x2D,0x2E,0x28,0x29
,0x2E,0x27,0x28,0x2E,0x27,0x2E,0x2F,0x27,0x2F,0x30,0x26,0x27,0x30,0x25,0x26
,0x30,0x25,0x30,0x31,0x24,0x25,0x31,0x24,0x31,0x32,0x23,0x24,0x32,0x22,0x23
,0x32,0x22,0x32,0x33,0x21,0x22,0x33,0x59,0x5A,0x5B,0x59,0x5B,0x5C,0x59,0x5C
,0x5D,0x58,0x59,0x5D,0x58,0x5D,0x5E,0x57,0x58,0x5E,0x57,0x5E,0x5F,0x57,0x5F
,0x60,0x56,0x57,0x60,0x56,0x60,0x61,0x55,0x56,0x61,0x54,0x55,0x61,0x54,0x61
,0x62,0x53,0x54,0x62,0x53,0x62,0x63,0x53,0x63,0x0F,0x3C,0x3D,0x3E,0x3B,0x3C
,0x3E,0x3A,0x3B,0x3E,0x39,0x3A,0x3E,0x39,0x3E,0x3F,0x38,0x39,0x3F,0x38,0x3F
,0x40,0x37,0x38,0x40,0x36,0x37,0x40,0x36,0x40,0x41,0x35,0x36,0x41,0x34,0x35
,0x41,0x34,0x41,0x42,0x33,0x34,0x42,0x33,0x42,0x43,0x21,0x33,0x43,0x47,0x48
,0x49,0x47,0x49,0x4A,0x47,0x4A,0x4B,0x47,0x4B,0x4C,0x47,0x4C,0x4D,0x46,0x47
,0x4D,0x46,0x4D,0x4E,0x46,0x4E,0x4F,0x46,0x4F,0x50,0x45,0x46,0x50,0x45,0x50
,0x51,0x44,0x45,0x51,0x44,0x51,0x52,0x44,0x52,0x53,0x43,0x44,0x53,0x43,0x53
,0x0F,0x21,0x43,0x0F,0x20,0x21,0x0F,0x20,0x0F,0x10,0x20,0x10,0x11,0x1F,0x20
,0x11,0x1F,0x11,0x12,0x1E,0x1F,0x12,0x1E,0x12,0x13,0x1D,0x1E,0x13,0x1D,0x13
,0x14,0x1D,0x14,0x15,0x1C,0x1D,0x15,0x1B,0x1C,0x15,0x1A,0x1B,0x15,0x19,0x1A
,0x15,0x19,0x15,0x16,0x18,0x19,0x16,0x17,0x18,0x16,0x63,0x2B,0x0C,0x0A,0x01
,0xBC,0x74,0x13,0x3D,0x46,0xB6,0xF3,0xBE,0xBC,0x74,0x13,0x3D,0xBC,0x74,0x13
,0xBD,0x46,0xB6,0xF3,0x3E,0xBC,0x74,0x13,0xBD,0x46,0xB6,0xF3,0x3E,0xBC,0x74
,0x13,0x3D,0xBC,0x74,0x13,0x3D,0xBC,0x74,0x13,0x3D,0xBC,0x74,0x13,0x3D,0xFC
,0xA9,0xF1,0x3E,0xBC,0x74,0x13,0xBD,0xFC,0xA9,0xF1,0x3E,0xBC,0x74,0x13,0xBD
,0xBC,0x74,0x13,0x3D,0x46,0xB6,0xF3,0xBE,0xBC,0x74,0x13,0x3D,0x46,0xB6,0xF3
,0xBE,0xBC,0x74,0x13,0xBD,0xBC,0x74,0x13,0xBD,0xBC,0x74,0x13,0xBD,0xBC,0x74
,0x13,0xBD,0x46,0xB6,0xF3,0xBE,0x0A,0x0B,0x00,0x0A,0x00,0x01,0x08,0x09,0x0A
,0x07,0x08,0x0A,0x07,0x0A,0x01,0x05,0x06,0x07,0x04,0x05,0x07,0x04,0x07,0x01
,0x03,0x04,0x01,0x03,0x01,0x02,0x0B,0x2C,0x1D,0x1B,0x01,0x55,0x34,0x56,0xBC
,0x73,0xBE,0x58,0xBF,0x82,0x8D,0x8B,0x3D,0x17,0xD9,0x46,0xBF,0x8B,0x1A,0xEC
,0x3D,0x97,0x02,0x32,0xBF,0xDD,0x24,0x06,0x3E,0x27,0x87,0x1B,0xBF,0xCA,0x1A
,0xF5,0x3D,0x39,0x29,0x08,0xBF,0x71,0x8D,0xAF,0x3D,0x73,0x12,0xF2,0xBE,0x75
,0x93,0x18,0x3D,0xDE,0x90,0xDE,0xBE,0x4B,0x21,0x90,0xBC,0x62,0x10,0xD8,0xBE
,0x5D,0xA5,0x7B,0xBD,0xB7,0x08,0xDC,0xBE,0x7A,0xC4,0xC8,0xBD,0xB7,0xF1,0xE7
,0xBE,0x8C,0x68,0xFB,0xBD,0xF9,0x68,0xF9,0xBE,0xDD,0x24,0x06,0xBE,0x2D,0x06
,0x07,0xBF,0x4B,0x39,0xFF,0xBD,0x10,0x58,0x11,0xBF,0xFF,0x07,0xD8,0xBD,0xFA
,0x2A,0x19,0xBF,0x8B,0xC6,0x9A,0xBD,0x98,0x19,0x1E,0xBF,0x3D,0x0A,0x17,0xBD
,0x77,0xBE,0x1F,0xBF,0xBE,0x89,0xA1,0xBC,0xEB,0x3B,0x1F,0xBF,0x4B,0x3F,0x61
,0xBB,0x35,0xB4,0x1D,0xBF,0xB5,0x8D,0xBF,0x3C,0x4D,0x4D,0x1A,0xBF,0xD3,0xD8
,0x1E,0x3D,0xFA,0x2A,0x19,0xBF,0xE9,0x10,0x38,0x3D,0x10,0xAD,0x19,0xBF,0xF1
,0x47,0x51,0x3D,0x33,0x33,0x1B,0xBF,0xF3,0x90,0x69,0x3D,0x57,0x79,0x1E,0xBF
,0xFC,0xA9,0x71,0x3D,0x8F,0x6C,0x22,0xBF,0x61,0xC4,0x3E,0x3D,0x9B,0x03,0x34
,0xBF,0x72,0x52,0x18,0x3C,0xD7,0xA3,0x44,0xBF,0x5B,0x60,0x4F,0xBD,0x04,0xAB
,0x52,0xBF,0xDD,0x24,0x06,0xBE,0xD1,0x76,0x5C,0xBF,0xDD,0x24,0x06,0xBE,0x66
,0x66,0x66,0xBF,0x1B,0x1C,0x00,0x1A,0x1B,0x00,0x19,0x1A,0x00,0x19,0x00,0x01
,0x18,0x19,0x01,0x18,0x01,0x02,0x17,0x18,0x02,0x17,0x02,0x03,0x16,0x17,0x03
,0x16,0x03,0x04,0x15,0x16,0x04,0x14,0x15,0x04,0x14,0x04,0x05,0x13,0x14,0x05
,0x13,0x05,0x06,0x13,0x06,0x07,0x12,0x13,0x07,0x12,0x07,0x08,0x12,0x08,0x09
,0x12,0x09,0x0A,0x12,0x0A,0x0B,0x11,0x12,0x0B,0x11,0x0B,0x0C,0x11,0x0C,0x0D
,0x11,0x0D,0x0E,0x11,0x0E,0x0F,0x10,0x11,0x0F,0x1C,0x2D,0x04,0x02,0x01,0xD5
,0x78,0x69,0xBE,0xDD,0x24,0x86,0xBE,0xD5,0x78,0x69,0x3E,0xDD,0x24,0x86,0xBE
,0xD5,0x78,0x69,0x3E,0x6F,0x12,0x03,0xBE,0xD5,0x78,0x69,0xBE,0x6F,0x12,0x03
,0xBE,0x02,0x03,0x00,0x02,0x00,0x01,0x03,0x2E,0x10,0x0E,0x01,0x73,0x2D,0x1A
,0xBD,0x94,0xC0,0xDE,0xBE,0x73,0xF3,0x8D,0xBD,0x9A,0x99,0xE9,0xBE,0x08,0x04
,0xBA,0xBD,0xA0,0xC2,0xF9,0xBE,0x39,0xB4,0xC8,0xBD,0xF0,0xA7,0x06,0xBF,0x08
,0x04,0xBA,0xBD,0xC7,0x4A,0x10,0xBF,0x73,0xF3,0x8D,0xBD,0x6F,0x66,0x18,0xBF
,0x73,0x2D,0x1A,0xBD,0x71,0xE8,0x1D,0xBF,0x00,0x00,0x00,0x00,0x77,0xBE,0x1F
,0xBF,0x73,0x2D,0x1A,0x3D,0x71,0xE8,0x1D,0xBF,0xF9,0xF3,0x8D,0x3D,0x6F,0x66
,0x18,0xBF,0x08,0x04,0xBA,0x3D,0xC7,0x4A,0x10,0xBF,0x39,0xB4,0xC8,0x3D,0xF0
,0xA7,0x06,0xBF,0xB4,0x3D,0xBA,0x3D,0x31,0x0A,0xFA,0xBE,0x17,0xD9,0x8E,0x3D
,0x02,0xD3,0xE9,0xBE,0x05,0x6B,0x1C,0x3D,0xDD,0xCE,0xDE,0xBE,0x00,0x00,0x00
,0x00,0xD1,0x22,0xDB,0xBE,0x01,0x02,0x03,0x00,0x01,0x03,0x0F,0x00,0x03,0x0F
,0x03,0x04,0x0F,0x04,0x05,0x0F,0x05,0x06,0x0F,0x06,0x07,0x0F,0x07,0x08,0x0F
,0x08,0x09,0x0F,0x09,0x0A,0x0F,0x0A,0x0B,0x0E,0x0F,0x0B,0x0D,0x0E,0x0B,0x0D
,0x0B,0x0C,0x0F,0x2F,0x04,0x02,0x01,0x7F,0x16,0x3B,0x3E,0x8B,0x6C,0x27,0x3F
,0x25,0xB0,0x81,0xBE,0x77,0xBE,0x1F,0xBF,0x7F,0x16,0x3B,0xBE,0x77,0xBE,0x1F
,0xBF,0x25,0xB0,0x81,0x3E,0x8B,0x6C,0x27,0x3F,0x02,0x03,0x00,0x02,0x00,0x01
,0x03,0x30,0x2A,0x2A,0x02,0xFA,0xD4,0xB9,0xBE,0xF1,0x7E,0x6C,0xBE,0x4F,0xE5
,0x94,0xBE,0x1F,0x2D,0xD6,0xBE,0xF9,0x82,0x26,0xBE,0xDD,0xCF,0x11,0xBF,0x7F
,0x6A,0xBC,0xBB,0x52,0xB8,0x1E,0xBF,0x61,0xC2,0xA8,0x3D,0x66,0x85,0x1A,0xBF
,0x18,0x08,0x32,0x3E,0x92,0xEC,0x0D,0xBF,0xAC,0x73,0x84,0x3E,0xE5,0x0D,0xF0
,0xBE,0x72,0x6A,0xA7,0x3E,0x2B,0xDB,0xAF,0xBE,0x42,0x76,0xBE,0x3E,0x4F,0xE7
,0x3A,0xBE,0xDD,0x24,0xC6,0x3E,0x39,0xB6,0x1E,0x3C,0xDB,0xA2,0xB4,0x3E,0xBC
,0x21,0x8D,0x3E,0xB4,0x1C,0x80,0x3E,0x7F,0xBE,0xF5,0x3E,0xA2,0xED,0x08,0x3E
,0x09,0xC2,0x15,0x3F,0xB5,0xC4,0x4A,0x3B,0x52,0xB8,0x1E,0x3F,0xF3,0x76,0xA4
,0xBD,0x4E,0x0E,0x1B,0x3F,0xB3,0xED,0x24,0xBE,0x62,0x10,0x10,0x3F,0x19,0xAF
,0x81,0xBE,0x94,0x83,0xF1,0x3E,0x1B,0x83,0xA6,0xBE,0xCB,0x9F,0xAF,0x3E,0x75
,0x3C,0xBE,0xBE,0xF8,0xA6,0x39,0x3E,0xDD,0x24,0xC6,0xBE,0x35,0x60,0x10,0xBC
,0x7C,0x42,0x56,0xBE,0x3D,0xB8,0xFB,0x3D,0x54,0x8F,0x44,0xBE,0xA6,0x99,0x8E
,0x3E,0x16,0x84,0x22,0xBE,0x3A,0x21,0xD4,0x3E,0xBF,0x2A,0xD7,0xBD,0xCD,0x20
,0x02,0x3F,0x26,0x39,0x60,0xBD,0x9C,0x6F,0x0C,0x3F,0xA4,0x6C,0x11,0x3B,0x3B
,0xDF,0x0F,0x3F,0x57,0x41,0x4C,0x3D,0x0C,0x03,0x0E,0x3F,0xEC,0x51,0xB8,0x3D
,0xA0,0x6E,0x08,0x3F,0xD1,0x22,0x0B,0x3E,0x04,0xAE,0xF3,0x3E,0x96,0x93,0x30
,0x3E,0x49,0x64,0xC7,0x3E,0x7C,0x43,0x51,0x3E,0x4E,0xB6,0x71,0x3E,0xF6,0x28
,0x5C,0x3E,0x14,0xAE,0x47,0x3D,0x67,0xB8,0x51,0x3E,0xD1,0x7A,0x48,0xBE,0x77
,0x66,0x32,0x3E,0xFC,0xA9,0xC1,0xBE,0xE1,0xCF,0x10,0x3E,0x86,0xA9,0xED,0xBE
,0x14,0x06,0xC5,0x3D,0xB8,0x73,0x05,0xBF,0x5E,0x0E,0x3B,0x3D,0xA4,0xA6,0x0D
,0xBF,0xB5,0xC4,0x4A,0xBB,0x4E,0x62,0x10,0xBF,0x7A,0x18,0xBA,0xBD,0xEB,0xFC
,0x07,0xBF,0xE5,0x80,0x1D,0xBE,0xAA,0x99,0xDD,0xBE,0xF1,0x7E,0x4C,0xBE,0x48
,0x37,0x82,0xBE,0xF6,0x28,0x5C,0xBE,0x23,0xDB,0xF9,0xBC,0x14,0x15,0x01,0x29
,0x14,0x01,0x28,0x29,0x01,0x00,0x01,0x15,0x13,0x00,0x15,0x12,0x13,0x15,0x27
,0x28,0x01,0x27,0x01,0x02,0x26,0x27,0x02,0x11,0x12,0x15,0x26,0x02,0x03,0x25
,0x26,0x03,0x25,0x03,0x04,0x24,0x25,0x04,0x11,0x15,0x16,0x10,0x11,0x16,0x0F
,0x10,0x16,0x0F,0x16,0x17,0x0E,0x0F,0x17,0x0E,0x17,0x18,0x0D,0x0E,0x18,0x0D
,0x18,0x19,0x0C,0x0D,0x19,0x0C,0x19,0x1A,0x0C,0x1A,0x1B,0x23,0x24,0x04,0x23
,0x04,0x05,0x22,0x23,0x05,0x22,0x05,0x06,0x21,0x22,0x06,0x21,0x06,0x07,0x20
,0x21,0x07,0x20,0x07,0x08,0x1F,0x20,0x08,0x1F,0x08,0x09,0x1F,0x09,0x0A,0x1E
,0x1F,0x0A,0x1E,0x0A,0x0B,0x1D,0x1E,0x0B,0x1C,0x1D,0x0B,0x1C,0x0B,0x0C,0x1B
,0x1C,0x0C,0x13,0x29,0x31,0x1C,0x1A,0x01,0xF2,0x22,0x63,0xBE,0x66,0x66,0xE6
,0x3E,0xC3,0x45,0x2E,0xBE,0x6D,0xE6,0xF0,0x3E,0xEA,0x76,0x06,0xBE,0x6F,0x66
,0xF4,0x3E,0xED,0x82,0xE1,0xBD,0xB7,0xB4,0xF2,0x3E,0xED,0xD4,0xBC,0xBD,0xB2
,0x9F,0xED,0x3E,0x42,0xB2,0xA0,0xBD,0x28,0xF2,0xE4,0x3E,0x57,0xE9,0x8E,0xBD
,0xC0,0x76,0xD8,0x3E,0x61,0x6C,0x81,0xBD,0x00,0xAD,0xB9,0x3E,0x23,0xDB,0x79
,0xBD,0x6F,0x12,0x83,0x3E,0x23,0xDB,0x79,0xBD,0x14,0xAE,0xC7,0xBE,0xA8,0xC8
,0x81,0xBD,0x49,0xB9,0xF3,0xBE,0xEA,0x59,0x90,0xBD,0x58,0xE3,0x04,0xBF,0xC7
,0x47,0xAB,0xBD,0x16,0xDA,0x09,0xBF,0x19,0x37,0xD5,0xBD,0x25,0xB0,0x0D,0xBF
,0xE7,0x54,0x12,0xBE,0x03,0x41,0x10,0xBF,0x62,0x10,0x58,0xBE,0x73,0x68,0x11
,0xBF,0x62,0x10,0x58,0xBE,0x9A,0x99,0x19,0xBF,0xD7,0xA3,0x70,0x3E,0x9A,0x99
,0x19,0xBF,0xD7,0xA3,0x70,0x3E,0x73,0x68,0x11,0xBF,0x80,0xEE,0x2B,0x3E,0xDE
,0x39,0x10,0xBF,0x3D,0x0E,0x03,0x3E,0x71,0x93,0x0D,0xBF,0x51,0x68,0xD9,0x3D
,0x7D,0x75,0x09,0xBF,0xC0,0x22,0xBF,0x3D,0x48,0xE0,0x03,0xBF,0x0C,0x58,0xB2
,0x3D,0x12,0x6B,0xF1,0xBE,0x7B,0x14,0xAE,0x3D,0x14,0xAE,0xC7,0xBE,0x7B,0x14
,0xAE,0x3D,0x52,0xB8,0x1E,0x3F,0x20,0x9B,0x64,0x3D,0x52,0xB8,0x1E,0x3F,0xD7
,0xA3,0x70,0xBE,0x6A,0xBC,0xF4,0x3E,0x18,0x19,0x1A,0x10,0x11,0x12,0x10,0x12
,0x13,0x10,0x13,0x14,0x10,0x14,0x15,0x10,0x15,0x16,0x1A,0x1B,0x00,0x1A,0x00
,0x01,0x1A,0x01,0x02,0x1A,0x02,0x03,0x1A,0x03,0x04,0x0E,0x0F,0x10,0x0E,0x10
,0x16,0x0D,0x0E,0x16,0x0C,0x0D,0x16,0x0B,0x0C,0x16,0x0A,0x0B,0x16,0x0A,0x16
,0x17,0x09,0x0A,0x17,0x09,0x17,0x18,0x09,0x18,0x1A,0x08,0x09,0x1A,0x07,0x08
,0x1A,0x06,0x07,0x1A,0x05,0x06,0x1A,0x05,0x1A,0x04,0x1B,0x32,0x25,0x23,0x01
,0xEC,0x51,0xB8,0x3E,0x91,0xED,0xBC,0xBE,0xD3,0xA1,0xAB,0x3E,0x95,0xEE,0xCE
,0xBE,0x91,0xED,0x9C,0x3E,0xCB,0x4D,0xDC,0xBE,0x85,0x42,0x8C,0x3E,0xFA,0x7F
,0xE5,0xBE,0x18,0x5C,0x73,0x3E,0xEC,0xF9,0xEA,0xBE,0xD9,0xD0,0x3D,0x3E,0x14
,0xB0,0xED,0xBE,0x33,0xA8,0xD6,0x3D,0x8D,0x97,0xEE,0xBE,0x7D,0x97,0x52,0xBE
,0x8D,0x97,0xEE,0xBE,0x9D,0x64,0x8B,0x3D,0x34,0xDB,0x35,0xBE,0xD5,0xD0,0x46
,0x3E,0x47,0x91,0xB5,0xBC,0x22,0x33,0x8F,0x3E,0x77,0xBE,0xFF,0x3D,0xB4,0x72
,0x9F,0x3E,0x0E,0xDD,0x5C,0x3E,0x2F,0xDD,0xA4,0x3E,0x4F,0xE5,0x9C,0x3E,0xD4
,0x62,0x98,0x3E,0xCE,0x8D,0xD9,0x3E,0x86,0xE7,0x65,0x3E,0xF2,0xD1,0x06,0x3F
,0x38,0x87,0xEB,0x3D,0xB6,0xBE,0x18,0x3F,0x56,0x0F,0x98,0xBC,0x52,0xB8,0x1E
,0x3F,0x6A,0x68,0x13,0xBE,0x78,0x29,0x19,0x3F,0x44,0x8B,0x7C,0xBE,0xE9,0x7C
,0x08,0x3F,0xF6,0x28,0xA4,0xBE,0x12,0xDA,0xDA,0x3E,0xEC,0x51,0xB8,0xBE,0x73
,0x68,0x91,0x3E,0x70,0x95,0xA7,0xBE,0x73,0x68,0x91,0x3E,0x31,0xB1,0x91,0xBE
,0x8B,0x18,0xBE,0x3E,0x0F,0xD5,0x64,0xBE,0x2B,0x87,0xDE,0x3E,0x68,0x41,0x18
,0xBE,0x81,0x41,0xF2,0x3E,0xEE,0xCC,0x84,0xBD,0xFE,0xD4,0xF8,0x3E,0x9C,0x15
,0xD1,0x3C,0xB4,0x72,0xEF,0x3E,0x6B,0x47,0xD1,0x3D,0xD7,0x4B,0xD3,0x3E,0x92
,0x41,0x1E,0x3E,0x58,0x8E,0xA8,0x3E,0xC5,0x20,0x30,0x3E,0xD5,0xD0,0x66,0x3E
,0x79,0x93,0x0F,0x3E,0xE8,0xA3,0x8C,0x3D,0x58,0xAE,0x37,0x3D,0x8E,0xE7,0xD3
,0xBD,0x16,0x87,0x03,0xBE,0x24,0x5E,0x9E,0xBE,0x5E,0xBA,0xC9,0xBE,0x23,0x31
,0x11,0xBF,0x5E,0xBA,0xC9,0xBE,0x9A,0x99,0x19,0xBF,0x69,0xE5,0x9E,0x3E,0x9A
,0x99,0x19,0xBF,0x5E,0xBA,0xC9,0x3E,0x91,0xED,0xBC,0xBE,0x23,0x24,0x00,0x23
,0x00,0x01,0x23,0x01,0x02,0x23,0x02,0x03,0x13,0x14,0x15,0x13,0x15,0x16,0x13
,0x16,0x17,0x12,0x13,0x17,0x12,0x17,0x18,0x11,0x12,0x18,0x11,0x18,0x19,0x10
,0x11,0x19,0x10,0x19,0x1A,0x0F,0x10,0x1A,0x23,0x03,0x04,0x22,0x23,0x04,0x22
,0x04,0x05,0x22,0x05,0x06,0x22,0x06,0x07,0x21,0x22,0x07,0x20,0x21,0x07,0x20
,0x07,0x08,0x1F,0x20,0x08,0x1F,0x08,0x09,0x1E,0x1F,0x09,0x1E,0x09,0x0A,0x1D
,0x1E,0x0A,0x1D,0x0A,0x0B,0x0F,0x1A,0x1B,0x0E,0x0F,0x1B,0x0D,0x0E,0x1B,0x0D
,0x1B,0x1C,0x0C,0x0D,0x1C,0x0C,0x1C,0x1D,0x0C,0x1D,0x0B,0x24,0x33,0x3D,0x3B
,0x01,0x29,0x5C,0x97,0xBE,0x2B,0x33,0xB5,0x3E,0x1D,0x5A,0x44,0xBE,0xDB,0x4C
,0xED,0x3E,0x49,0x81,0x85,0xBD,0x00,0x00,0x00,0x3F,0xBA,0x83,0x58,0x3C,0x9D
,0xD7,0xF8,0x3E,0x78,0xF1,0x9E,0x3D,0x31,0x5E,0xE3,0x3E,0x65,0x18,0xF7,0x3D
,0x30,0x48,0xC2,0x3E,0x71,0x3D,0x0A,0x3E,0xAA,0x49,0x98,0x3E,0xBC,0x20,0x02
,0x3E,0x81,0x3F,0x6C,0x3E,0xC4,0x95,0xD3,0x3D,0xD1,0xCA,0x2D,0x3E,0x94,0x31
,0x7E,0x3D,0x7E,0x1C,0xED,0x3D,0x03,0x27,0xDB,0x3A,0x97,0xC6,0x8F,0x3D,0xDC
,0x9F,0x8B,0xBD,0x39,0x60,0x17,0x3D,0x97,0x1E,0x0D,0xBE,0x0A,0xD7,0xA3,0x3C
,0x97,0x1E,0x0D,0xBE,0xA6,0x9B,0xC4,0xBB,0x91,0x62,0xE0,0xBD,0xA6,0x9B,0xC4
,0xBB,0x85,0x5C,0x29,0xBD,0xD1,0xAD,0x57,0xBC,0xF1,0x47,0xD1,0x3C,0x73,0xF3
,0x0D,0xBD,0x0F,0x7F,0xAD,0x3D,0x78,0x0E,0x85,0xBD,0x2F,0xDD,0x04,0x3E,0xA7
,0xEB,0xC9,0xBD,0x7E,0xC4,0x1F,0x3E,0x96,0x41,0x05,0xBE,0x90,0x6A,0x38,0x3E
,0x3D,0xBA,0x31,0xBE,0xCC,0x26,0x50,0x3E,0x5F,0x60,0x76,0xBE,0x62,0x10,0x58
,0x3E,0x77,0x6A,0x9E,0xBE,0x8D,0xF0,0x46,0x3E,0xFF,0xE8,0xCB,0xBE,0x4F,0x91
,0x13,0x3E,0x1F,0xDB,0xF2,0xBE,0xE6,0x1E,0x92,0x3D,0x14,0xCE,0x06,0xBF,0x43
,0x8C,0x57,0xBC,0x96,0x43,0x0B,0xBF,0x25,0xCB,0x49,0xBD,0xDB,0xA4,0x0A,0xBF
,0xB1,0x4D,0xAA,0xBD,0xBD,0xC8,0x08,0xBF,0xB3,0xD2,0xE4,0xBD,0xB1,0xFC,0x05
,0xBF,0xC0,0x76,0x20,0xBE,0xDF,0x50,0x00,0xBF,0x07,0x28,0x4D,0xBE,0xFD,0x67
,0xF5,0xBE,0x26,0x56,0x66,0xBE,0x07,0x29,0xF0,0xBE,0x2D,0xB2,0x7D,0xBE,0xBD
,0x72,0xED,0xBE,0xD3,0x4D,0x8A,0xBE,0x44,0x8B,0xEC,0xBE,0xA1,0x30,0x98,0xBE
,0x14,0x04,0xEF,0xBE,0x96,0xED,0xA3,0xBE,0xA8,0x6E,0xF6,0xBE,0x0F,0xED,0xAB
,0xBE,0x4C,0x8D,0x00,0xBF,0x8D,0x97,0xAE,0xBE,0x00,0xAC,0x06,0xBF,0xF8,0xFC
,0xA8,0xBE,0x74,0x28,0x0F,0xBF,0x17,0x2D,0x98,0xBE,0x77,0x14,0x17,0xBF,0x3E
,0xB4,0x6F,0xBE,0x5B,0xCF,0x1C,0xBF,0x8B,0x6C,0x07,0xBE,0x52,0xB8,0x1E,0xBF
,0x26,0xA6,0xAB,0x3D,0x03,0x40,0x15,0xBF,0x55,0xBE,0x77,0x3E,0x0C,0xAE,0xF1
,0xBE,0x6C,0xE9,0xA1,0x3E,0x54,0x38,0xB2,0xBE,0x8D,0x97,0xAE,0x3E,0x0D,0x52
,0x50,0xBE,0x0C,0x57,0xA7,0x3E,0x75,0x3D,0xD1,0xBD,0x89,0x95,0x91,0x3E,0x73
,0x7F,0x75,0xBC,0x00,0x52,0x5B,0x3E,0xE2,0x01,0x65,0x3D,0xB2,0x9D,0xEF,0x3D
,0xAF,0xD2,0xDD,0x3D,0x8F,0xC2,0x75,0x3E,0x03,0x27,0x7B,0x3E,0x3B,0xDF,0x8F
,0x3E,0xB5,0x70,0xC1,0x3E,0x35,0x60,0x88,0x3E,0x9F,0xC7,0xE8,0x3E,0xC9,0xC6
,0x63,0x3E,0xED,0xD2,0x06,0x3F,0xEE,0x78,0x03,0x3E,0xF9,0xBE,0x18,0x3F,0x0A
,0xD7,0x23,0x3A,0x52,0xB8,0x1E,0x3F,0x20,0xD3,0xDA,0xBD,0xE0,0x84,0x1A,0x3F
,0x46,0xB2,0x47,0xBE,0x79,0xEA,0x0D,0x3F,0x14,0x96,0x88,0xBE,0x84,0x2C,0xF3
,0x3E,0x66,0x66,0xA6,0xBE,0x7F,0x6A,0xBC,0x3E,0x3B,0x3C,0x00,0x3B,0x00,0x01
,0x3A,0x3B,0x01,0x3A,0x01,0x02,0x39,0x3A,0x02,0x38,0x39,0x02,0x38,0x02,0x03
,0x37,0x38,0x03,0x37,0x03,0x04,0x36,0x37,0x04,0x36,0x04,0x05,0x35,0x36,0x05
,0x34,0x35,0x05,0x33,0x34,0x05,0x33,0x05,0x06,0x33,0x06,0x07,0x32,0x33,0x07
,0x32,0x07,0x08,0x32,0x08,0x09,0x0C,0x0D,0x0E,0x0B,0x0C,0x0E,0x0B,0x0E,0x0F
,0x0A,0x0B,0x0F,0x0A,0x0F,0x10,0x09,0x0A,0x10,0x32,0x09,0x10,0x32,0x10,0x11
,0x31,0x32,0x11,0x31,0x11,0x12,0x30,0x31,0x12,0x30,0x12,0x13,0x2F,0x30,0x13
,0x2F,0x13,0x14,0x2E,0x2F,0x14,0x2E,0x14,0x15,0x2D,0x2E,0x15,0x2D,0x15,0x16
,0x2C,0x2D,0x16,0x2C,0x16,0x17,0x2C,0x17,0x18,0x2B,0x2C,0x18,0x2B,0x18,0x19
,0x2A,0x2B,0x19,0x2A,0x19,0x1A,0x2A,0x1A,0x1B,0x2A,0x1B,0x1C,0x2A,0x1C,0x1D
,0x29,0x2A,0x1D,0x29,0x1D,0x1E,0x29,0x1E,0x1F,0x28,0x29,0x1F,0x28,0x1F,0x20
,0x28,0x20,0x21,0x27,0x28,0x21,0x27,0x21,0x22,0x27,0x22,0x23,0x26,0x27,0x23
,0x26,0x23,0x24,0x26,0x24,0x25,0x3C,0x34,0x0E,0x0E,0x02,0x91,0xED,0x7C,0x3E
,0xC1,0xCA,0x21,0xBE,0x91,0xED,0x7C,0x3E,0x52,0xB8,0x1E,0x3F,0x6C,0x3F,0x19
,0x3E,0x52,0xB8,0x1E,0x3F,0x4E,0x62,0xD0,0xBE,0x4D,0x87,0x2E,0xBE,0x4E,0x62
,0xD0,0xBE,0x73,0x68,0x91,0xBE,0x17,0xD9,0xCE,0x3D,0x73,0x68,0x91,0xBE,0x17
,0xD9,0xCE,0x3D,0x9A,0x99,0x19,0xBF,0x91,0xED,0x7C,0x3E,0x9A,0x99,0x19,0xBF
,0x91,0xED,0x7C,0x3E,0x73,0x68,0x91,0xBE,0x4E,0x62,0xD0,0x3E,0x73,0x68,0x91
,0xBE,0x4E,0x62,0xD0,0x3E,0xC1,0xCA,0x21,0xBE,0xDF,0xFB,0xA3,0xBE,0xC1,0xCA
,0x21,0xBE,0x17,0xD9,0xCE,0x3D,0x74,0xEB,0xDD,0x3E,0x17,0xD9,0xCE,0x3D,0xC1
,0xCA,0x21,0xBE,0x0C,0x00,0x01,0x00,0x0C,0x0D,0x0C,0x01,0x02,0x0C,0x02,0x03
,0x0B,0x0C,0x03,0x09,0x0A,0x00,0x08,0x09,0x00,0x08,0x00,0x0D,0x06,0x07,0x08
,0x05,0x06,0x08,0x05,0x08,0x0D,0x04,0x05,0x0D,0x04,0x0D,0x0B,0x03,0x04,0x0B
,0x0A,0x0D,0x35,0x29,0x27,0x01,0xC2,0xA5,0xA3,0xBD,0x75,0x93,0x18,0x3F,0x00
,0xAC,0x9E,0xBE,0x02,0x2B,0x07,0x3E,0xD1,0x20,0x25,0xBE,0x24,0x7F,0xF0,0x3D
,0x5C,0x8F,0x42,0xBD,0x0D,0x52,0xB0,0x3D,0x94,0x31,0x7E,0x3D,0x5F,0x45,0xC6
,0x3C,0x74,0x97,0x14,0x3E,0x8D,0x0C,0x72,0xBD,0x2B,0x32,0x4A,0x3E,0xE1,0x7E
,0x20,0xBE,0x52,0x10,0x5C,0x3E,0xC6,0xF7,0x85,0xBE,0xEB,0x00,0x48,0x3E,0x96
,0x05,0xBB,0xBE,0xFA,0xD2,0x0B,0x3E,0x42,0x5E,0xE7,0xBE,0x13,0x7F,0x54,0x3D
,0x5C,0xAE,0x02,0xBF,0xA4,0xFB,0x39,0xBD,0x14,0xAE,0x07,0xBF,0x23,0xBA,0xE7
,0xBD,0xD4,0xB7,0x04,0xBF,0xEC,0x51,0x38,0xBE,0x8E,0xEB,0xF7,0xBE,0x30,0x84
,0x5C,0xBE,0x03,0xEC,0xEB,0xBE,0xCB,0x4D,0x74,0xBE,0x2F,0x87,0xE5,0xBE,0x9F
,0xAD,0x83,0xBE,0x7C,0xD5,0xE2,0xBE,0x00,0xAC,0x8E,0xBE,0x95,0xEF,0xE1,0xBE
,0x15,0x19,0x9D,0xBE,0x8A,0x58,0xE4,0xBE,0xE5,0xD2,0xA8,0xBE,0x68,0x93,0xEB
,0xBE,0x30,0x9B,0xB0,0xBE,0x24,0xF2,0xF5,0xBE,0x33,0x33,0xB3,0xBE,0x58,0xE3
,0x00,0xBF,0xF2,0xEA,0xAC,0xBE,0x84,0xA0,0x0B,0xBF,0x73,0x12,0x9A,0xBE,0x7F
,0x69,0x15,0xBF,0x24,0xD6,0x72,0xBE,0x99,0x64,0x1C,0xBF,0x9C,0x6C,0x13,0xBE
,0x52,0xB8,0x1E,0xBF,0x51,0xDD,0x1C,0xBD,0x38,0x9E,0x1B,0xBF,0x65,0x8B,0x84
,0x3D,0xDB,0x4F,0x12,0xBF,0x3B,0x38,0x08,0x3E,0x15,0x1A,0x08,0xBF,0x39,0x60
,0x47,0x3E,0xDD,0x24,0xF6,0xBE,0x4E,0x0B,0x7E,0x3E,0x58,0x37,0xD6,0xBE,0x8D
,0x43,0x95,0x3E,0x27,0x31,0xB0,0xBE,0x3E,0x09,0xA4,0x3E,0xF8,0x51,0x85,0xBE
,0xC3,0xF5,0xA8,0x3E,0x2D,0xB2,0x2D,0xBE,0x1F,0xD8,0x99,0x3E,0x4A,0x96,0x93
,0xBC,0xE3,0xFD,0x58,0x3E,0x5F,0x45,0xE6,0x3D,0x94,0xDD,0x4C,0x3D,0x58,0xE1
,0x66,0x3E,0x78,0x41,0x24,0xBE,0xD9,0x76,0x92,0x3E,0x30,0xB8,0xA6,0xBD,0x1D
,0x5A,0xE4,0x3E,0x77,0xBE,0x8F,0x3E,0x1D,0x5A,0xE4,0x3E,0x33,0x33,0xB3,0x3E
,0x75,0x93,0x18,0x3F,0x27,0x28,0x00,0x26,0x27,0x00,0x26,0x00,0x01,0x25,0x26
,0x01,0x25,0x01,0x02,0x24,0x25,0x02,0x24,0x02,0x03,0x24,0x03,0x04,0x23,0x24
,0x04,0x23,0x04,0x05,0x22,0x23,0x05,0x22,0x05,0x06,0x21,0x22,0x06,0x21,0x06
,0x07,0x20,0x21,0x07,0x1F,0x20,0x07,0x1F,0x07,0x08,0x1E,0x1F,0x08,0x1E,0x08
,0x09,0x1D,0x1E,0x09,0x1C,0x1D,0x09,0x1C,0x09,0x0A,0x1B,0x1C,0x0A,0x1B,0x0A
,0x0B,0x1A,0x1B,0x0B,0x19,0x1A,0x0B,0x19,0x0B,0x0C,0x19,0x0C,0x0D,0x18,0x19
,0x0D,0x17,0x18,0x0D,0x17,0x0D,0x0E,0x16,0x17,0x0E,0x16,0x0E,0x0F,0x16,0x0F
,0x10,0x15,0x16,0x10,0x15,0x10,0x11,0x15,0x11,0x12,0x15,0x12,0x13,0x15,0x13
,0x14,0x28,0x36,0x34,0x34,0x02,0xAC,0x1C,0x9A,0x3E,0x52,0xB8,0x1E,0x3F,0x32
,0x04,0x50,0x3E,0x79,0x3D,0x1C,0x3F,0xA8,0x51,0xC8,0x3D,0xCD,0xCC,0x14,0x3F
,0xED,0xB8,0x61,0xBC,0xBE,0x33,0x06,0x3F,0x58,0x8D,0x05,0xBE,0xF1,0x7E,0xDC
,0x3E,0x30,0x83,0x71,0xBE,0x5A,0x0F,0x9F,0x3E,0x00,0xAA,0xA0,0xBE,0x5C,0x8F
,0x32,0x3E,0x61,0x51,0xB9,0xBE,0xB8,0xCD,0xD4,0x3C,0x37,0x89,0xC1,0xBE,0xE6
,0x78,0x05,0xBE,0x90,0x15,0xAC,0xBE,0xC7,0x4B,0xB7,0xBE,0xAC,0x74,0x57,0xBE
,0xDF,0xF9,0x09,0xBF,0x4B,0xAC,0xEC,0xBD,0xB1,0x88,0x19,0xBF,0xA4,0x6C,0x11
,0xBB,0x52,0xB8,0x1E,0xBF,0xC7,0xF1,0x23,0x3E,0x63,0x26,0x15,0xBF,0x0C,0x02
,0x93,0x3E,0x50,0xE1,0xF0,0xBE,0x64,0xE7,0xB5,0x3E,0xBC,0x75,0xAE,0xBE,0x37
,0x89,0xC1,0x3E,0x3B,0x8F,0x4A,0xBE,0xA3,0x07,0xB6,0x3E,0x98,0xDF,0x69,0xBD
,0x06,0x83,0x93,0x3E,0x31,0x43,0x63,0x3D,0xA9,0xA0,0x42,0x3E,0xD5,0x79,0x04
,0x3E,0x90,0x12,0x9B,0x3D,0x2D,0xB2,0x1D,0x3E,0x7E,0x72,0x54,0xBD,0x1E,0x89
,0x07,0x3E,0xC7,0x4B,0x37,0xBE,0xEF,0x1C,0x8A,0x3D,0x06,0xD9,0x12,0xBE,0x60
,0xEA,0x37,0x3E,0xDF,0x4F,0xCD,0xBD,0xBA,0xF3,0x8C,0x3E,0x6D,0x37,0x41,0xBD
,0x6E,0xA5,0xB7,0x3E,0xE7,0xFC,0x94,0x3C,0x1F,0x2D,0xDE,0x3E,0x75,0x93,0xB8
,0x3D,0xC3,0xF4,0xFD,0x3E,0x38,0xD9,0x26,0x3E,0x37,0x33,0x0A,0x3F,0x7B,0xC0
,0x7C,0x3E,0x13,0x99,0x11,0x3F,0x58,0x39,0xB4,0x3E,0x19,0x04,0x16,0x3F,0x58
,0x39,0xB4,0x3E,0x52,0xB8,0x1E,0x3F,0x1F,0x2C,0x03,0xBE,0x45,0xF1,0x2A,0x3D
,0x93,0xE5,0xA4,0xBD,0x67,0xF1,0x82,0x3D,0xC7,0x64,0x31,0xBD,0x11,0x8B,0x98
,0x3D,0x9C,0xC5,0x0B,0xBC,0x77,0xBE,0x9F,0x3D,0x27,0xDD,0xB6,0x3D,0x35,0xF1
,0x4E,0x3D,0xD0,0xD2,0x25,0x3E,0xDE,0xAF,0x02,0xBD,0x9D,0xF3,0x53,0x3E,0x71
,0x91,0x1B,0xBE,0xF8,0x53,0x63,0x3E,0x79,0x95,0x95,0xBE,0xFE,0xD5,0x53,0x3E
,0xD5,0x22,0xD2,0xBE,0x10,0x5C,0x25,0x3E,0x89,0xED,0xFE,0xBE,0xB1,0x6C,0xC6
,0x3D,0x5D,0x4E,0x0D,0xBF,0xCB,0xA1,0xC5,0x3C,0x85,0xEB,0x11,0xBF,0x3A,0x79
,0x11,0xBD,0xAA,0x46,0x0F,0xBF,0x1E,0xFA,0xAE,0xBD,0x19,0x58,0x07,0xBF,0x08
,0x03,0x0F,0xBE,0x01,0xBE,0xEB,0xBE,0xAB,0xED,0x36,0xBE,0xEB,0x53,0xBE,0xBE
,0x08,0x03,0x4F,0xBE,0xAD,0x89,0x8D,0xBE,0x3D,0x0A,0x57,0xBE,0x81,0xED,0x40
,0xBE,0xAA,0xF1,0x52,0xBE,0x25,0x02,0xD5,0xBD,0xF0,0xA7,0x46,0xBE,0xF4,0xFD
,0x54,0x3B,0x33,0x04,0x05,0x33,0x05,0x06,0x33,0x06,0x07,0x33,0x07,0x08,0x32
,0x33,0x08,0x31,0x32,0x08,0x31,0x08,0x09,0x30,0x31,0x09,0x2F,0x30,0x09,0x2F
,0x09,0x0A,0x2E,0x2F,0x0A,0x2D,0x2E,0x0A,0x2D,0x0A,0x0B,0x2D,0x0B,0x0C,0x2C
,0x2D,0x0C,0x2B,0x2C,0x0C,0x2B,0x0C,0x0D,0x2A,0x2B,0x0D,0x29,0x2A,0x0D,0x29
,0x0D,0x0E,0x28,0x29,0x0E,0x28,0x0E,0x0F,0x27,0x28,0x0F,0x27,0x0F,0x10,0x26
,0x27,0x10,0x26,0x10,0x11,0x25,0x26,0x11,0x25,0x11,0x12,0x25,0x12,0x13,0x24
,0x25,0x13,0x24,0x13,0x14,0x23,0x24,0x14,0x23,0x14,0x15,0x22,0x23,0x15,0x22
,0x15,0x16,0x21,0x22,0x16,0x20,0x21,0x16,0x33,0x20,0x16,0x04,0x33,0x16,0x04
,0x16,0x17,0x04,0x17,0x18,0x04,0x18,0x19,0x03,0x04,0x19,0x03,0x19,0x1A,0x03
,0x1A,0x1B,0x02,0x03,0x1B,0x02,0x1B,0x1C,0x01,0x02,0x1C,0x01,0x1C,0x1D,0x00
,0x01,0x1D,0x00,0x1D,0x1E,0x1F,0x00,0x1E,0x1F,0x33,0x37,0x0C,0x0A,0x01,0x0E
,0x83,0xC1,0xBE,0x72,0x14,0xA0,0x3E,0x66,0x68,0xB4,0xBE,0xC1,0x1E,0x9B,0x3E
,0xD3,0xA3,0x99,0xBE,0xFA,0xD1,0xC0,0x3E,0xE5,0xD4,0x6E,0xBE,0x54,0x3B,0xDB
,0x3E,0x72,0xC2,0x34,0xBE,0x46,0x60,0xE4,0x3E,0x22,0xA8,0xBA,0xBD,0x8B,0x6C
,0xE7,0x3E,0x98,0xC2,0x73,0x3E,0x8B,0x6C,0xE7,0x3E,0x76,0xC6,0xF7,0xBD,0x77
,0xBE,0x1F,0xBF,0xC7,0xD6,0xB3,0xBC,0x77,0xBE,0x1F,0xBF,0x0E,0x83,0xC1,0x3E
,0x0A,0xD7,0x0F,0x3F,0x0E,0x83,0xC1,0x3E,0x75,0x93,0x18,0x3F,0xD9,0x24,0x87
,0xBE,0x75,0x93,0x18,0x3F,0x06,0x07,0x08,0x06,0x08,0x09,0x0B,0x00,0x01,0x0B
,0x01,0x02,0x0B,0x02,0x03,0x0B,0x03,0x04,0x0A,0x0B,0x04,0x0A,0x04,0x05,0x0A
,0x05,0x06,0x0A,0x06,0x09,0x0B,0x38,0x41,0x43,0x03,0x0E,0xD9,0x60,0xBE,0x75
,0x1E,0xB5,0xBD,0x7F,0xBE,0x95,0xBE,0x45,0x66,0x2E,0xBE,0xC7,0x4C,0xAA,0xBE
,0x77,0x6A,0x7E,0xBE,0xE9,0x26,0xB1,0xBE,0x50,0x8D,0xA7,0xBE,0x88,0x69,0xA7
,0xBE,0xC7,0x49,0xD9,0xBE,0x1F,0x31,0x8A,0xBE,0xEE,0x7C,0x03,0xBF,0x96,0x3F
,0x1F,0xBE,0x75,0xE9,0x17,0xBF,0x5A,0x68,0x67,0xBB,0x52,0xB8,0x1E,0xBF,0x3E
,0x5D,0x0D,0x3E,0x47,0x02,0x19,0xBF,0x40,0x89,0x7F,0x3E,0x48,0xE0,0x07,0xBF
,0x5F,0xCE,0xA4,0x3E,0xD8,0x64,0xDD,0xBE,0xE9,0x26,0xB1,0x3E,0x1C,0xB2,0xA1
,0xBE,0x54,0x8B,0xA8,0x3E,0xAC,0x1E,0x60,0xBE,0x52,0xB8,0x8E,0x3E,0xE1,0xD2
,0x01,0xBE,0xE5,0xD4,0x4E,0x3E,0x89,0x24,0x3A,0xBD,0xE9,0x9B,0x94,0x3D,0x5A
,0x64,0x7B,0x3D,0x20,0x09,0x4B,0x3E,0x65,0x39,0x19,0x3E,0xDF,0x51,0x8B,0x3E
,0x99,0xF1,0x66,0x3E,0xB7,0xB5,0x9D,0x3E,0xFA,0xD3,0x96,0x3E,0x0A,0xD7,0xA3
,0x3E,0xF5,0x80,0xB9,0x3E,0xC5,0x8C,0x98,0x3E,0xAD,0x2F,0xEA,0x3E,0x31,0x5C
,0x6D,0x3E,0xAA,0x9C,0x0A,0x3F,0x81,0x98,0x04,0x3E,0x64,0xB1,0x19,0x3F,0x9F
,0x22,0x87,0xBA,0x52,0xB8,0x1E,0x3F,0x47,0x90,0x0A,0xBE,0x00,0x54,0x19,0x3F
,0xE1,0xCE,0x75,0xBE,0xE9,0x26,0x09,0x3F,0x36,0x72,0x9D,0xBE,0x39,0x0B,0xE3
,0x3E,0xC3,0xF5,0xA8,0xBE,0x2F,0x89,0xAB,0x3E,0x12,0xC2,0xA3,0xBE,0x29,0x5E
,0x85,0x3E,0xDD,0x26,0x94,0xBE,0xC9,0x1E,0x41,0x3E,0x41,0x0F,0x65,0xBE,0x82
,0x39,0xDA,0x3D,0xDA,0x1E,0xDD,0xBD,0x06,0xBC,0xCC,0x3A,0x30,0xB8,0xE6,0xBD
,0x3B,0x37,0x5D,0x3E,0x9D,0x46,0x1A,0xBE,0x64,0x78,0x84,0x3E,0x90,0x16,0x37
,0xBE,0xD5,0x76,0x9B,0x3E,0x64,0xE8,0x48,0xBE,0xA3,0x06,0xB3,0x3E,0x17,0xD9
,0x4E,0xBE,0x9E,0x97,0xCA,0x3E,0xA0,0x6E,0x40,0xBE,0x36,0x1E,0xEC,0x3E,0xF9
,0x2E,0x15,0xBE,0x58,0x8E,0x04,0x3F,0x86,0xE7,0xA5,0xBD,0x36,0x94,0x0E,0x3F
,0x66,0xDC,0xD4,0x39,0x85,0xEB,0x11,0x3F,0x84,0x45,0xA5,0x3D,0xCE,0x8C,0x0E
,0x3F,0xE5,0x7C,0x11,0x3E,0xA8,0x70,0x04,0x3F,0x4F,0x5D,0x39,0x3E,0x9C,0xC4
,0xE8,0x3E,0xF0,0xA7,0x46,0x3E,0x14,0x5A,0xBE,0x3E,0x56,0x65,0x3F,0x3E,0x83
,0x6D,0x9C,0x3E,0xCB,0x9D,0x29,0x3E,0x03,0x7B,0x7C,0x3E,0x75,0x3D,0xF1,0x3D
,0x59,0x89,0x39,0x3E,0xB3,0xB2,0xFD,0x3C,0x12,0xF8,0xC3,0x3D,0x20,0xB4,0xBE
,0x3D,0xB6,0x4B,0x2B,0xBE,0xBC,0xCC,0x40,0x3E,0x7F,0x6A,0x8C,0xBE,0x18,0xB2
,0x5A,0x3E,0xC2,0xA1,0xA7,0xBE,0xF8,0x53,0x63,0x3E,0x3F,0x35,0xC6,0xBE,0x33
,0x87,0x54,0x3E,0xB3,0x45,0xEA,0xBE,0xE6,0x20,0x28,0x3E,0x89,0x41,0x04,0xBF
,0x90,0x49,0xC6,0x3D,0x02,0x81,0x0E,0xBF,0xE1,0x79,0x29,0x3C,0x85,0xEB,0x11
,0xBF,0xD0,0xD0,0x9F,0xBD,0x7D,0x94,0x0D,0xBF,0x50,0xE1,0x18,0xBE,0x54,0x8F
,0x00,0xBF,0x29,0x09,0x49,0xBE,0xBE,0xA1,0xD8,0xBE,0x87,0x16,0x59,0xBE,0xA6
,0x9B,0xA4,0xBE,0x89,0xF0,0x4F,0xBE,0x50,0x39,0x76,0xBE,0x13,0x7F,0x34,0xBE
,0x8D,0x47,0x29,0xBE,0x77,0x11,0x06,0xBE,0xB9,0x51,0xC4,0xBD,0x9E,0xEF,0x87
,0xBD,0x5C,0x8F,0x02,0xBD,0x00,0x01,0x30,0x1F,0x00,0x30,0x03,0x3A,0x3B,0x3A
,0x03,0x04,0x3A,0x04,0x05,0x3A,0x05,0x06,0x03,0x3B,0x3C,0x39,0x3A,0x06,0x39
,0x06,0x07,0x38,0x39,0x07,0x38,0x07,0x08,0x37,0x38,0x08,0x1F,0x30,0x20,0x1E
,0x1F,0x20,0x36,0x37,0x08,0x36,0x08,0x09,0x35,0x36,0x09,0x35,0x09,0x0A,0x34
,0x35,0x0A,0x34,0x0A,0x0B,0x33,0x34,0x0B,0x02,0x03,0x3C,0x02,0x3C,0x3D,0x01
,0x02,0x3D,0x01,0x3D,0x3E,0x1D,0x1E,0x20,0x1D,0x20,0x21,0x1D,0x21,0x22,0x1C
,0x1D,0x22,0x1B,0x1C,0x22,0x1B,0x22,0x23,0x1A,0x1B,0x23,0x1A,0x23,0x24,0x1A
,0x24,0x25,0x19,0x1A,0x25,0x19,0x25,0x26,0x18,0x19,0x26,0x18,0x26,0x27,0x17
,0x18,0x27,0x17,0x27,0x28,0x16,0x17,0x28,0x16,0x28,0x29,0x16,0x29,0x2A,0x15
,0x16,0x2A,0x15,0x2A,0x2B,0x14,0x15,0x2B,0x14,0x2B,0x2C,0x13,0x14,0x2C,0x12
,0x13,0x2C,0x12,0x2C,0x2D,0x11,0x12,0x2D,0x11,0x2D,0x2E,0x10,0x11,0x2E,0x10
,0x2E,0x2F,0x0F,0x10,0x2F,0x0F,0x2F,0x30,0x33,0x0B,0x0C,0x32,0x33,0x0C,0x32
,0x0C,0x0D,0x31,0x32,0x0D,0x31,0x0D,0x0E,0x31,0x0E,0x0F,0x40,0x31,0x0F,0x40
,0x0F,0x30,0x40,0x30,0x01,0x3F,0x40,0x01,0x3F,0x01,0x3E,0x1F,0x30,0x40,0x39
,0x30,0x30,0x02,0x4A,0x0C,0x9A,0xBE,0x77,0xBE,0x1F,0xBF,0x79,0x93,0x0F,0xBE
,0x97,0xC4,0x19,0xBF,0xAA,0xEF,0xFC,0x3B,0x0A,0xD7,0x07,0xBF,0xAF,0x97,0x26
,0x3E,0xE3,0xFC,0xCD,0xBE,0x4E,0x0E,0x8F,0x3E,0x9E,0x97,0x7A,0xBE,0x40,0x88
,0xB4,0x3E,0x21,0xAC,0x86,0xBD,0x25,0x06,0xC1,0x3E,0xFC,0xA9,0x01,0x3E,0x54
,0x8B,0xB0,0x3E,0x31,0x07,0xA9,0x3E,0x3F,0x35,0x7E,0x3E,0x12,0x2F,0xFF,0x3E
,0x96,0xE9,0x07,0x3E,0x28,0xF0,0x16,0x3F,0x63,0x0E,0x82,0xBB,0x52,0xB8,0x1E
,0x3F,0xBF,0xF1,0x25,0xBE,0x23,0x31,0x15,0x3F,0xEC,0xF9,0x92,0xBE,0x4C,0x37
,0xF1,0x3E,0x1F,0x83,0xB5,0xBE,0xCD,0x20,0xAE,0x3E,0x25,0x06,0xC1,0xBE,0xE1
,0x7A,0x44,0x3E,0x9D,0x85,0xB5,0xBE,0xF9,0x11,0x3F,0x3D,0x25,0x04,0x93,0xBE
,0xF1,0x82,0x88,0xBD,0xA8,0xC5,0x40,0xBE,0x14,0x04,0x0F,0xBE,0x8D,0x0C,0x92
,0xBD,0x9E,0xEF,0x27,0xBE,0x6D,0x37,0x41,0x3D,0x3C,0x33,0x11,0xBE,0x02,0x2B
,0x37,0x3E,0x2B,0xFC,0x99,0xBD,0x98,0x13,0x04,0x3E,0xC3,0x47,0x64,0xBE,0xBE
,0xDA,0x51,0x3D,0x19,0x58,0xB7,0xBE,0x5F,0x62,0x2C,0xBD,0x22,0x71,0xEF,0xBE
,0x29,0x08,0x0E,0xBE,0x48,0xE2,0x09,0xBF,0x1C,0xB1,0x76,0xBE,0x32,0xE9,0x13
,0xBF,0x46,0xB6,0xB3,0xBE,0x50,0x8D,0x17,0xBF,0x46,0xB6,0xB3,0xBE,0x77,0xBE
,0x1F,0xBF,0x01,0xDA,0x16,0x3E,0xF7,0x3A,0x29,0xBD,0xC0,0x97,0xC2,0x3D,0x80
,0x62,0x84,0xBD,0x8A,0xC8,0x30,0x3D,0xA3,0x1E,0xA2,0xBD,0xAA,0xB6,0x9B,0xBA
,0x31,0x08,0xAC,0xBD,0x1A,0xA4,0xA0,0xBD,0x9D,0x47,0x85,0xBD,0x44,0xDB,0x11
,0xBE,0x40,0x33,0x08,0xBC,0x0E,0x31,0x4E,0xBE,0xBB,0xF1,0xEE,0x3D,0xD3,0x4D
,0x62,0xBE,0xB0,0x72,0x90,0x3E,0xFA,0x28,0x53,0xBE,0x1A,0xDC,0xCE,0x3E,0x6F
,0xBA,0x25,0xBE,0x35,0xB2,0xFB,0x3E,0x0C,0x5A,0xC8,0xBD,0xBC,0x5D,0x0B,0x3F
,0xC8,0x60,0xC5,0xBC,0x3B,0xDF,0x0F,0x3F,0x9A,0xD0,0x24,0x3D,0xF6,0x7D,0x0C
,0x3F,0xA6,0x43,0xC7,0x3D,0x25,0x5A,0x02,0x3F,0x9B,0x1E,0x14,0x3E,0x7F,0xC0
,0xE3,0x3E,0xF5,0x2C,0x38,0x3E,0xDB,0xF9,0xB6,0x3E,0x61,0x8E,0x4E,0x3E,0x29
,0x08,0x86,0x3E,0x19,0x04,0x56,0x3E,0xE9,0x26,0x31,0x3E,0x85,0xEB,0x51,0x3E
,0x80,0x45,0xBE,0x3D,0xCB,0xA1,0x45,0x3E,0x48,0xDF,0x24,0xBC,0x04,0x2D,0x2E
,0x04,0x2E,0x2F,0x2D,0x04,0x05,0x2D,0x05,0x06,0x2D,0x06,0x07,0x2C,0x2D,0x07
,0x2B,0x2C,0x07,0x2B,0x07,0x08,0x2A,0x2B,0x08,0x2A,0x08,0x09,0x29,0x2A,0x09
,0x29,0x09,0x0A,0x28,0x29,0x0A,0x27,0x28,0x0A,0x27,0x0A,0x0B,0x26,0x27,0x0B
,0x25,0x26,0x0B,0x25,0x0B,0x0C,0x24,0x25,0x0C,0x24,0x0C,0x0D,0x23,0x24,0x0D
,0x23,0x0D,0x0E,0x22,0x23,0x0E,0x22,0x0E,0x0F,0x22,0x0F,0x10,0x21,0x22,0x10
,0x21,0x10,0x11,0x20,0x21,0x11,0x20,0x11,0x12,0x1F,0x20,0x12,0x1F,0x12,0x13
,0x1E,0x1F,0x13,0x1E,0x13,0x14,0x1D,0x1E,0x14,0x1C,0x1D,0x14,0x2F,0x1C,0x14
,0x04,0x2F,0x14,0x04,0x14,0x15,0x03,0x04,0x15,0x03,0x15,0x16,0x02,0x03,0x16
,0x02,0x16,0x17,0x01,0x02,0x17,0x01,0x17,0x18,0x01,0x18,0x19,0x00,0x01,0x19
,0x00,0x19,0x1A,0x1B,0x00,0x1A,0x1B,0x2F,0x3A,0x20,0x1C,0x02,0xE0,0x14,0x16
,0xBD,0x50,0x39,0x66,0x3E,0x29,0xE7,0x8B,0xBD,0x48,0x31,0x50,0x3E,0xBE,0xF7
,0xB7,0xBD,0xA9,0xC2,0x2F,0x3E,0xF0,0xA7,0xC6,0xBD,0x4C,0x37,0x09,0x3E,0xBE
,0xF7,0xB7,0xBD,0xDE,0x57,0xC5,0x3D,0x29,0xE7,0x8B,0xBD,0x9E,0x7A,0x84,0x3D
,0xE0,0x14,0x16,0xBD,0x20,0xD5,0x30,0x3D,0x6F,0x12,0x83,0x3A,0xBC,0x74,0x13
,0x3D,0x06,0x46,0x1E,0x3D,0x20,0xD5,0x30,0x3D,0x43,0x00,0x90,0x3D,0x9E,0x7A
,0x84,0x3D,0x52,0x10,0xBC,0x3D,0xDE,0x57,0xC5,0x3D,0x83,0xC0,0xCA,0x3D,0x4C
,0x37,0x09,0x3E,0x52,0x10,0xBC,0x3D,0xA9,0xC2,0x2F,0x3E,0x43,0x00,0x90,0x3D
,0x48,0x31,0x50,0x3E,0x06,0x46,0x1E,0x3D,0x50,0x39,0x66,0x3E,0x6F,0x12,0x83
,0x3A,0x68,0x91,0x6D,0x3E,0xE6,0x5B,0x1F,0xBD,0x17,0xD9,0xDE,0xBE,0x46,0x41
,0x90,0xBD,0xE7,0xFB,0xE9,0xBE,0x92,0x20,0xBC,0xBD,0xBA,0x4B,0xFA,0xBE,0x83
,0xC0,0xCA,0xBD,0xA4,0xC4,0x06,0xBF,0x92,0x20,0xBC,0xBD,0x25,0x5C,0x10,0xBF
,0x46,0x41,0x90,0xBD,0xA0,0x6E,0x18,0xBF,0xE6,0x5B,0x1F,0xBD,0x79,0xEA,0x1D
,0xBF,0xB8,0x59,0xBC,0xBA,0x77,0xBE,0x1F,0xBF,0x96,0x08,0x14,0x3D,0x79,0xEA
,0x1D,0xBF,0x96,0x43,0x8B,0x3D,0xA0,0x6E,0x18,0xBF,0xD9,0xCE,0xB7,0x3D,0x25
,0x5C,0x10,0xBF,0xF0,0xA7,0xC6,0x3D,0xA4,0xC4,0x06,0xBF,0xFF,0x07,0xB8,0x3D
,0xBA,0x4B,0xFA,0xBE,0x39,0x29,0x8C,0x3D,0xE7,0xFB,0xE9,0xBE,0x28,0x46,0x16
,0x3D,0x17,0xD9,0xDE,0xBE,0xB8,0x59,0xBC,0xBA,0xD1,0x22,0xDB,0xBE,0x0B,0x0C
,0x0D,0x0B,0x0D,0x0E,0x0B,0x0E,0x0F,0x0A,0x0B,0x0F,0x09,0x0A,0x0F,0x08,0x09
,0x0F,0x07,0x08,0x0F,0x06,0x07,0x0F,0x05,0x06,0x0F,0x04,0x05,0x0F,0x03,0x04
,0x0F,0x02,0x03,0x0F,0x01,0x02,0x0F,0x01,0x0F,0x00,0x1F,0x10,0x11,0x1E,0x1F
,0x11,0x1D,0x1E,0x11,0x1D,0x11,0x12,0x1D,0x12,0x13,0x1C,0x1D,0x13,0x1B,0x1C
,0x13,0x1A,0x1B,0x13,0x19,0x1A,0x13,0x18,0x19,0x13,0x17,0x18,0x13,0x16,0x17
,0x13,0x15,0x16,0x13,0x15,0x13,0x14,0x0F,0x1F,0x3B,0x2D,0x29,0x02,0x7B,0xD9
,0x36,0xBD,0xC2,0x4D,0x66,0x3E,0x77,0x49,0x9C,0xBD,0x12,0x83,0x50,0x3E,0x0C
,0x5A,0xC8,0xBD,0xBB,0x45,0x30,0x3E,0x3D,0x0A,0xD7,0xBD,0x1D,0xAA,0x09,0x3E
,0x0C,0x5A,0xC8,0xBD,0xD0,0xED,0xC5,0x3D,0x77,0x49,0x9C,0xBD,0x8B,0xC4,0x84
,0x3D,0x7B,0xD9,0x36,0xBD,0xD3,0xF9,0x30,0x3D,0x42,0x60,0xE5,0xBB,0xBC,0x74
,0x13,0x3D,0xD6,0x02,0xFB,0x3C,0xD3,0xF9,0x30,0x3D,0xEB,0x3B,0x7F,0x3D,0x8B
,0xC4,0x84,0x3D,0x04,0xAE,0xAB,0x3D,0xD0,0xED,0xC5,0x3D,0x35,0x5E,0xBA,0x3D
,0x1D,0xAA,0x09,0x3E,0x04,0xAE,0xAB,0x3D,0xBB,0x45,0x30,0x3E,0xEB,0x3B,0x7F
,0x3D,0x12,0x83,0x50,0x3E,0xD6,0x02,0xFB,0x3C,0xC2,0x4D,0x66,0x3E,0x42,0x60
,0xE5,0xBB,0x68,0x91,0x6D,0x3E,0x55,0x34,0x56,0xBC,0x73,0xBE,0x58,0xBF,0x82
,0x8D,0x8B,0x3D,0x17,0xD9,0x46,0xBF,0x8B,0x1A,0xEC,0x3D,0x97,0x02,0x32,0xBF
,0xDD,0x24,0x06,0x3E,0x27,0x87,0x1B,0xBF,0xCA,0x1A,0xF5,0x3D,0x39,0x29,0x08
,0xBF,0x71,0x8D,0xAF,0x3D,0x73,0x12,0xF2,0xBE,0x75,0x93,0x18,0x3D,0xDE,0x90
,0xDE,0xBE,0x4B,0x21,0x90,0xBC,0x62,0x10,0xD8,0xBE,0x5D,0xA5,0x7B,0xBD,0xB7
,0x08,0xDC,0xBE,0x7A,0xC4,0xC8,0xBD,0xB7,0xF1,0xE7,0xBE,0x8C,0x68,0xFB,0xBD
,0xF9,0x68,0xF9,0xBE,0xDD,0x24,0x06,0xBE,0x2D,0x06,0x07,0xBF,0x33,0x37,0xFF
,0xBD,0x10,0x58,0x11,0xBF,0x22,0x00,0xD8,0xBD,0xFA,0x2A,0x19,0xBF,0x31,0xB4
,0x9A,0xBD,0x98,0x19,0x1E,0xBF,0x2E,0xC8,0x16,0xBD,0x77,0xBE,0x1F,0xBF,0x9F
,0x05,0xA1,0xBC,0xA2,0x44,0x1F,0xBF,0x1B,0x2F,0x5D,0xBB,0x13,0xD7,0x1D,0xBF
,0x48,0x31,0xC0,0x3C,0xFC,0xA8,0x1A,0xBF,0xE3,0x1A,0x1F,0x3D,0x9A,0x99,0x19
,0xBF,0x6A,0x31,0x38,0x3D,0x2A,0x1B,0x1A,0xBF,0xF1,0x47,0x51,0x3D,0xBA,0x9F
,0x1B,0xBF,0xF3,0x90,0x69,0x3D,0x48,0xE1,0x1E,0xBF,0xFC,0xA9,0x71,0x3D,0xDD
,0xCE,0x22,0xBF,0x61,0xC4,0x3E,0x3D,0x0A,0x4B,0x34,0xBF,0x72,0x52,0x18,0x3C
,0xF2,0xD3,0x44,0xBF,0x5B,0x60,0x4F,0xBD,0xEB,0xC7,0x52,0xBF,0xDD,0x24,0x06
,0xBE,0x1B,0x85,0x5C,0xBF,0xDD,0x24,0x06,0xBE,0x66,0x66,0x66,0xBF,0x01,0x02
,0x03,0x00,0x01,0x03,0x0F,0x00,0x03,0x0F,0x03,0x04,0x0F,0x04,0x05,0x0F,0x05
,0x06,0x0F,0x06,0x07,0x0F,0x07,0x08,0x0F,0x08,0x09,0x0F,0x09,0x0A,0x0F,0x0A
,0x0B,0x0E,0x0F,0x0B,0x0D,0x0E,0x0B,0x0D,0x0B,0x0C,0x2B,0x2C,0x10,0x2A,0x2B
,0x10,0x29,0x2A,0x10,0x29,0x10,0x11,0x28,0x29,0x11,0x28,0x11,0x12,0x27,0x28
,0x12,0x27,0x12,0x13,0x26,0x27,0x13,0x26,0x13,0x14,0x25,0x26,0x14,0x24,0x25
,0x14,0x24,0x14,0x15,0x23,0x24,0x15,0x23,0x15,0x16,0x23,0x16,0x17,0x22,0x23
,0x17,0x22,0x17,0x18,0x22,0x18,0x19,0x22,0x19,0x1A,0x22,0x1A,0x1B,0x21,0x22
,0x1B,0x21,0x1B,0x1C,0x21,0x1C,0x1D,0x21,0x1D,0x1E,0x21,0x1E,0x1F,0x21,0x1F
,0x20,0x0F,0x2C,0x3C,0x07,0x05,0x01,0x21,0xB0,0xF2,0xBE,0x58,0x39,0xB4,0xBC
,0x21,0xB0,0xF2,0x3E,0x64,0x3B,0xDF,0xBE,0x21,0xB0,0xF2,0x3E,0x90,0x16,0xB7
,0xBE,0x3E,0x5E,0xB0,0xBE,0xDE,0x21,0x45,0xB9,0x21,0xB0,0xF2,0x3E,0xC9,0x22
,0xB5,0x3E,0x21,0xB0,0xF2,0x3E,0x1B,0x2F,0xDD,0x3E,0x21,0xB0,0xF2,0xBE,0x58
,0x39,0xB4,0x3C,0x00,0x01,0x02,0x00,0x02,0x03,0x04,0x05,0x06,0x03,0x04,0x06
,0x00,0x03,0x06,0x06,0x3D,0x08,0x04,0x02,0x0E,0x2D,0xF2,0xBE,0x87,0x16,0xD9
,0x3D,0x0E,0x2D,0xF2,0x3E,0x87,0x16,0xD9,0x3D,0x0E,0x2D,0xF2,0x3E,0xA2,0x45
,0x36,0x3E,0x0E,0x2D,0xF2,0xBE,0xA2,0x45,0x36,0x3E,0x0E,0x2D,0xF2,0xBE,0x35
,0x5E,0x3A,0xBE,0x0E,0x2D,0xF2,0x3E,0x35,0x5E,0x3A,0xBE,0x0E,0x2D,0xF2,0x3E
,0xAE,0x47,0xE1,0xBD,0x0E,0x2D,0xF2,0xBE,0xAE,0x47,0xE1,0xBD,0x02,0x03,0x00
,0x02,0x00,0x01,0x06,0x07,0x04,0x06,0x04,0x05,0x03,0x07,0x3E,0x07,0x05,0x01
,0x21,0xB0,0xF2,0x3E,0xBC,0x74,0x93,0x3C,0x21,0xB0,0xF2,0xBE,0x1B,0x2F,0xDD
,0x3E,0x21,0xB0,0xF2,0xBE,0x46,0x0A,0xB5,0x3E,0xD7,0x4F,0xAF,0x3E,0x97,0x72
,0x3E,0xBB,0x21,0xB0,0xF2,0xBE,0xE2,0x22,0xB7,0xBE,0x21,0xB0,0xF2,0xBE,0x64
,0x3B,0xDF,0xBE,0x21,0xB0,0xF2,0x3E,0xA6,0x9B,0xC4,0xBC,0x00,0x01,0x02,0x00
,0x02,0x03,0x04,0x05,0x06,0x03,0x04,0x06,0x00,0x03,0x06,0x06,0x3F,0x3D,0x39
,0x02,0x38,0x11,0xFD,0x3B,0xEA,0x22,0x65,0xBE,0xD6,0x02,0xFB,0x3C,0xE9,0xD6
,0x1B,0xBE,0x65,0xE1,0xAB,0x3D,0x6B,0x47,0x91,0xBD,0x64,0x3B,0x3F,0x3E,0x1C
,0x45,0x56,0x3D,0x14,0x05,0x82,0x3E,0x6D,0x3B,0x0D,0x3E,0x75,0x3F,0x97,0x3E
,0xFE,0xD4,0x58,0x3E,0x6D,0x3A,0xA2,0x3E,0x87,0xBF,0x8E,0x3E,0x54,0xE3,0xA5
,0x3E,0x17,0x2D,0xB0,0x3E,0x9C,0xC3,0x9D,0x3E,0x3A,0x5D,0xDE,0x3E,0x52,0x64
,0x85,0x3E,0xC5,0x20,0x04,0x3F,0xB7,0x99,0x1A,0x3E,0xB8,0x74,0x18,0x3F,0x87
,0x16,0xD9,0xBB,0x64,0x3B,0x1F,0x3F,0x7B,0x11,0x0D,0xBE,0xE7,0x50,0x1A,0x3F
,0x26,0x35,0x74,0xBE,0x60,0x91,0x0B,0x3F,0x24,0xF1,0x9A,0xBE,0x48,0xDF,0xEC
,0x3E,0x54,0xE3,0xA5,0xBE,0x6A,0xBC,0xBC,0x3E,0xE7,0xFB,0xA1,0xBE,0x38,0xF4
,0x9E,0x3E,0xA2,0x45,0x96,0xBE,0x2B,0x85,0x88,0x3E,0xD5,0xCD,0x85,0xBE,0x5C
,0xE6,0x74,0x3E,0xA6,0x43,0x67,0xBE,0x1F,0x85,0x6B,0x3E,0x15,0xAA,0x4B,0xBE
,0x2C,0x0C,0x71,0x3E,0x86,0x93,0x34,0xBE,0xEE,0xD0,0x80,0x3E,0xBA,0xF3,0x24
,0xBE,0x0A,0x2B,0x8D,0x3E,0x77,0xBE,0x1F,0xBE,0xA4,0x1C,0x9C,0x3E,0x44,0x89
,0x26,0xBE,0x96,0xAF,0xAB,0x3E,0x68,0xE9,0x3A,0xBE,0xC0,0x20,0xC1,0x3E,0xCF
,0x49,0x4F,0xBE,0x18,0x96,0xD7,0x3E,0x9C,0x14,0x56,0xBE,0x50,0x35,0xEA,0x3E
,0xF6,0xD4,0x4A,0xBE,0xF9,0x11,0xFF,0x3E,0x87,0x16,0x29,0xBE,0x0E,0x2E,0x09
,0x3F,0x0D,0x8D,0xE7,0xBD,0x79,0x1E,0x10,0x3F,0xC7,0x4B,0x37,0xBD,0x98,0x6E
,0x12,0x3F,0x3A,0x04,0x0E,0x3D,0xBE,0x33,0x0E,0x3F,0x83,0x35,0xCE,0x3D,0x0E
,0x83,0x01,0x3F,0xF8,0xA7,0x14,0x3E,0xDF,0x51,0xDB,0x3E,0x0A,0xD7,0x23,0x3E
,0x4C,0xE3,0xA7,0x3E,0xD5,0xCB,0x1F,0x3E,0x7B,0x16,0x8C,0x3E,0xF4,0xA9,0x13
,0x3E,0x8E,0x3F,0x61,0x3E,0xBE,0x16,0xF4,0x3D,0x2F,0xDB,0x1E,0x3E,0x90,0x12
,0x9B,0x3D,0x5E,0xBA,0x89,0x3D,0x12,0xA0,0xA6,0x3C,0x75,0x00,0x44,0xBD,0xB1
,0x89,0x4C,0xBC,0x02,0xD3,0x09,0xBE,0x4B,0xC9,0xF2,0xBC,0xAF,0xED,0x5D,0xBE
,0xE3,0xA5,0x1B,0xBD,0xE5,0xD0,0xA2,0xBE,0x6F,0x12,0x83,0xBB,0xE5,0xD0,0xA2
,0xBE,0x8A,0xCC,0x5C,0xBD,0xCD,0xCA,0xDE,0xBE,0x1E,0xFA,0xAE,0xBD,0xA0,0xC2
,0xE9,0xBE,0xE4,0xD8,0xDA,0xBD,0x29,0x04,0xFA,0xBE,0xD5,0x78,0xE9,0xBD,0xA4
,0xC4,0x06,0xBF,0xE4,0xD8,0xDA,0xBD,0x25,0x5C,0x10,0xBF,0x1E,0xFA,0xAE,0xBD
,0xA0,0x6E,0x18,0xBF,0x8A,0xCC,0x5C,0xBD,0x79,0xEA,0x1D,0xBF,0xFC,0xA8,0x86
,0xBC,0x77,0xBE,0x1F,0xBF,0xE4,0x2F,0xAD,0x3C,0x79,0xEA,0x1D,0xBF,0x87,0x16
,0x59,0x3D,0xA0,0x6E,0x18,0xBF,0x87,0x16,0x99,0x3D,0x25,0x5C,0x10,0xBF,0x9E
,0xEF,0xA7,0x3D,0xA4,0xC4,0x06,0xBF,0xAD,0x4F,0x99,0x3D,0x29,0x04,0xFA,0xBE
,0xCE,0xE1,0x5A,0x3D,0xA0,0xC2,0xE9,0xBE,0x08,0xAB,0xB1,0x3C,0xCD,0xCA,0xDE
,0xBE,0xFC,0xA8,0x86,0xBC,0xD1,0x22,0xDB,0xBE,0x2A,0x2B,0x2C,0x2A,0x2C,0x00
,0x29,0x2A,0x00,0x29,0x00,0x01,0x28,0x29,0x01,0x28,0x01,0x02,0x27,0x28,0x02
,0x27,0x02,0x03,0x26,0x27,0x03,0x26,0x03,0x04,0x25,0x26,0x04,0x25,0x04,0x05
,0x24,0x25,0x05,0x24,0x05,0x06,0x23,0x24,0x06,0x23,0x06,0x07,0x23,0x07,0x08
,0x22,0x23,0x08,0x22,0x08,0x09,0x22,0x09,0x0A,0x21,0x22,0x0A,0x21,0x0A,0x0B
,0x20,0x21,0x0B,0x1F,0x20,0x0B,0x1F,0x0B,0x0C,0x1E,0x1F,0x0C,0x1D,0x1E,0x0C
,0x1D,0x0C,0x0D,0x1C,0x1D,0x0D,0x1C,0x0D,0x0E,0x1B,0x1C,0x0E,0x1B,0x0E,0x0F
,0x1A,0x1B,0x0F,0x19,0x1A,0x0F,0x19,0x0F,0x10,0x18,0x19,0x10,0x18,0x10,0x11
,0x18,0x11,0x12,0x17,0x18,0x12,0x17,0x12,0x13,0x16,0x17,0x13,0x15,0x16,0x13
,0x14,0x15,0x13,0x36,0x37,0x38,0x36,0x38,0x39,0x36,0x39,0x3A,0x36,0x3A,0x3B
,0x36,0x3B,0x3C,0x35,0x36,0x3C,0x34,0x35,0x3C,0x33,0x34,0x3C,0x32,0x33,0x3C
,0x31,0x32,0x3C,0x30,0x31,0x3C,0x2F,0x30,0x3C,0x2E,0x2F,0x3C,0x2E,0x3C,0x2D
,0x2C,0x3C,0x40,0x71,0x71,0x02,0x37,0x35,0x80,0x3E,0x8B,0x6C,0x67,0x3E,0x54
,0x3B,0x63,0x3E,0x93,0x18,0x04,0x3E,0xF8,0x54,0x4E,0x3E,0x55,0x33,0x3B,0x3E
,0x75,0x8F,0x2C,0x3E,0x5C,0x8F,0x62,0x3E,0x54,0x8D,0xFE,0x3D,0x30,0x2D,0x7A
,0x3E,0xF3,0xAD,0x8F,0x3D,0x25,0x06,0x81,0x3E,0x8E,0xAC,0x7C,0xBC,0x6C,0x41
,0x6F,0x3E,0x65,0x8B,0xE4,0xBD,0x58,0xE1,0x36,0x3E,0x19,0x56,0x71,0xBE,0xA5
,0xC0,0x82,0x3D,0xD3,0x4D,0xAA,0xBE,0x70,0xED,0xA4,0xBD,0xE3,0x51,0xCA,0xBE
,0x61,0x8B,0x6D,0xBE,0xF4,0xFD,0xD4,0xBE,0x5F,0x62,0xBC,0xBE,0x3E,0xCC,0xCE
,0xBE,0x6C,0x92,0xE7,0xBE,0x3F,0x37,0xBC,0xBE,0x6D,0x91,0x04,0xBF,0x59,0xA4
,0xA1,0xBE,0x9B,0x59,0x0F,0xBF,0xCD,0x78,0x83,0xBE,0xAA,0xF1,0x12,0xBF,0xA2
,0xF1,0x44,0xBE,0xED,0xF0,0x0F,0xBF,0xEB,0x01,0xF3,0xBD,0x95,0xEE,0x06,0xBF
,0x50,0x52,0xE0,0xBC,0xEF,0x8F,0xE7,0xBE,0x62,0x10,0xB8,0x3D,0x4A,0xB4,0xA4
,0xBE,0x36,0xAE,0x9F,0x3D,0x02,0xD3,0xC9,0xBE,0x50,0x8D,0x97,0x3D,0x3D,0x0A
,0xE7,0xBE,0x91,0xF1,0xA8,0x3D,0x93,0x8C,0x00,0xBF,0xDA,0x1E,0xDD,0x3D,0xDF
,0x4E,0x0A,0xBF,0xEA,0xD0,0x19,0x3E,0xA1,0x66,0x10,0xBF,0x19,0x04,0x56,0x3E
,0x98,0x6E,0x12,0xBF,0x61,0xA4,0xA7,0x3E,0x9E,0xCE,0x0D,0xBF,0xD9,0xCC,0xE1
,0x3E,0x23,0xDD,0xFF,0xBE,0xD3,0xD8,0x0A,0x3F,0x9C,0xDC,0xCF,0xBE,0xA0,0xC4
,0x1F,0x3F,0x23,0xDB,0x89,0xBE,0xEA,0x92,0x2D,0x3F,0x24,0xD3,0xE1,0xBD,0x0E
,0x2D,0x32,0x3F,0xF2,0xD2,0x4D,0x3D,0xA6,0x45,0x2D,0x3F,0xD5,0xCD,0x55,0x3E
,0x5C,0x8F,0x1E,0x3F,0xB4,0x72,0xB7,0x3E,0x86,0x76,0x06,0x3F,0x9C,0xC4,0xF8
,0x3E,0xCD,0xCE,0xCA,0x3E,0xD1,0x21,0x14,0x3F,0x00,0x53,0x76,0x3E,0xD9,0x99
,0x22,0x3F,0x16,0x89,0x89,0x3D,0x8B,0x6C,0x27,0x3F,0x84,0x66,0x17,0xBE,0xCF
,0x2D,0x20,0x3F,0x81,0x95,0xB3,0xBE,0xAC,0x71,0x0A,0x3F,0x3A,0x93,0x06,0xBF
,0x65,0xA7,0xCF,0x3E,0xD9,0x23,0x28,0xBF,0x98,0xBE,0x67,0x3E,0x61,0x1B,0x3D
,0xBF,0x15,0xFE,0x8C,0x3C,0x93,0x18,0x44,0xBF,0x02,0x83,0x54,0xBE,0x95,0xB9
,0x3D,0xBF,0x8B,0x6E,0xD5,0xBE,0xAA,0x9C,0x2A,0xBF,0xBA,0x9F,0x1B,0xBF,0x37
,0xFF,0x0B,0xBF,0x54,0xE4,0x44,0xBF,0x81,0x3D,0xC6,0xBE,0x58,0x8F,0x63,0xBF
,0x9B,0xC8,0x4C,0xBE,0x0B,0x96,0x76,0xBF,0xC5,0x53,0x0F,0x3B,0x91,0xED,0x7C
,0xBF,0x9F,0x59,0x82,0x3E,0xB7,0x5E,0x73,0xBF,0x31,0xB2,0xF4,0x3E,0x29,0xB2
,0x56,0xBF,0xD4,0x82,0x27,0x3F,0xF3,0x93,0x2A,0xBF,0x93,0x18,0x44,0x3F,0x42
,0x60,0xE5,0xBE,0xA2,0xEF,0x36,0x3F,0x42,0x60,0xE5,0xBE,0xEB,0x52,0x17,0x3F
,0x58,0x58,0x28,0xBF,0xEC,0xA5,0xD9,0x3E,0x46,0x61,0x4F,0xBF,0xA3,0x20,0x68
,0x3E,0x52,0x2C,0x67,0xBF,0xB2,0x9F,0xC5,0x3B,0xA0,0x1A,0x6F,0xBF,0xCD,0xCE
,0x32,0xBE,0x33,0xA9,0x69,0xBF,0x92,0x95,0xAF,0xBE,0xFC,0x54,0x59,0xBF,0x51
,0x4D,0xF9,0xBE,0x66,0x66,0x3E,0xBF,0x3B,0x34,0x18,0xBF,0xDD,0x25,0x19,0xBF
,0x0B,0x61,0x29,0xBF,0xE9,0xBA,0xD8,0xBE,0xA0,0x1A,0x2F,0xBF,0x3A,0x5C,0x6B
,0xBE,0x75,0x93,0x28,0xBF,0x06,0x2F,0x7A,0xBC,0xF4,0xFD,0x14,0xBF,0x01,0xFC
,0x43,0x3E,0x15,0x6F,0xEC,0xBE,0xDB,0xA5,0xBD,0x3E,0x71,0x3B,0x9C,0xBE,0x27
,0x86,0x00,0x3F,0x54,0xE5,0xFB,0xBD,0x46,0x40,0x15,0x3F,0xFD,0xF9,0x96,0x3D
,0xF6,0x28,0x1C,0x3F,0xDB,0xF7,0x68,0x3E,0x79,0xC9,0x17,0x3F,0x93,0x18,0xBC
,0x3E,0x04,0xAB,0x0A,0x3F,0x4F,0x75,0xF8,0x3E,0xED,0xBB,0xEA,0x3E,0x5E,0xB9
,0x12,0x3F,0x60,0xE5,0xB0,0x3E,0x85,0xB6,0x20,0x3F,0x15,0x02,0x59,0x3E,0x42
,0x60,0x25,0x3F,0xC3,0xF5,0x88,0x3D,0x15,0x38,0x21,0x3F,0x18,0x23,0xB2,0xBD
,0x7F,0xBF,0x14,0x3F,0xDB,0x4D,0x70,0xBE,0xA0,0x70,0x02,0x3F,0x8E,0xEB,0xB7
,0xBE,0x50,0x8B,0xD9,0x3E,0xEA,0x24,0xE3,0xBE,0xB9,0x34,0xAE,0x3E,0xA1,0xDB
,0xFB,0xBE,0x89,0x95,0x89,0x3E,0x4A,0x0C,0x02,0xBF,0x77,0x68,0x78,0x3E,0x5B
,0xCF,0x00,0xBF,0x98,0x6E,0x62,0x3E,0x1F,0x31,0xFA,0xBE,0x66,0xBE,0x53,0x3E
,0xF6,0x29,0xEF,0xBE,0x17,0xD9,0x4E,0x3E,0x65,0xE3,0xE1,0xBE,0x24,0x09,0x52
,0x3E,0xE8,0xFA,0xCE,0xBE,0x91,0x99,0x5B,0x3E,0x00,0xAC,0xB6,0xBE,0x4A,0x0C
,0x92,0x3E,0xDA,0xA9,0xD9,0xBD,0x67,0xBA,0xC7,0x3E,0x6D,0xE7,0x7B,0x3E,0x8B
,0x70,0xF3,0x3D,0x71,0x39,0x4E,0x3E,0xF2,0xD2,0x1D,0x3E,0xC9,0xC6,0x33,0x3E
,0xC2,0x4D,0x36,0x3E,0x56,0x66,0x0A,0x3E,0xC9,0x76,0x3E,0x3E,0x56,0x99,0xA9
,0x3D,0x8A,0x1C,0x32,0x3E,0x53,0xCE,0x97,0xBC,0x56,0x0E,0x0D,0x3E,0xC5,0x70
,0x15,0xBE,0x67,0x64,0xD0,0x3D,0xE7,0xA7,0x68,0xBE,0xEE,0x41,0x48,0x3D,0xB6
,0xF3,0x9D,0xBE,0x9C,0xC3,0x35,0xBC,0x1B,0x2E,0xC2,0xBE,0x16,0x31,0x8C,0xBD
,0x91,0x99,0xDB,0xBE,0xE5,0x47,0xFC,0xBD,0xBB,0x9C,0xEA,0xBE,0xDD,0x7C,0x33
,0xBE,0xB2,0x9D,0xEF,0xBE,0x35,0x0A,0x59,0xBE,0xDD,0x23,0xEB,0xBE,0xA3,0xC8
,0x7A,0xBE,0x3D,0xB6,0xDD,0xBE,0x0D,0x55,0x89,0xBE,0xDD,0x25,0xC9,0xBE,0xDF
,0x4F,0x8D,0xBE,0xA6,0x43,0xAF,0xBE,0x45,0xB7,0x86,0xBE,0x41,0x0B,0x81,0xBE
,0x34,0xDB,0x65,0xBE,0x9A,0x41,0x0C,0xBE,0xAB,0xED,0x16,0xBE,0x3B,0x8E,0x9F
,0x3C,0xB1,0xFD,0x64,0xBD,0xAD,0x6C,0x0F,0x3E,0x14,0xAF,0x32,0x3C,0xEA,0x22
,0x45,0x3E,0x63,0x60,0x9D,0x3D,0x3D,0x0A,0x57,0x3E,0x01,0x5C,0x5D,0x5C,0x01
,0x02,0x5C,0x02,0x03,0x34,0x35,0x36,0x34,0x36,0x37,0x33,0x34,0x37,0x33,0x37
,0x38,0x32,0x33,0x38,0x32,0x38,0x39,0x59,0x5A,0x00,0x59,0x00,0x01,0x31,0x32
,0x39,0x31,0x39,0x3A,0x5B,0x5C,0x03,0x5B,0x03,0x04,0x01,0x5D,0x5E,0x59,0x01
,0x5E,0x59,0x5E,0x5F,0x59,0x5F,0x60,0x58,0x59,0x60,0x58,0x60,0x61,0x58,0x61
,0x62,0x70,0x5B,0x04,0x70,0x04,0x05,0x70,0x05,0x06,0x6F,0x70,0x06,0x30,0x31
,0x3A,0x30,0x3A,0x3B,0x30,0x3B,0x3C,0x2F,0x30,0x3C,0x2F,0x3C,0x3D,0x2E,0x2F
,0x3D,0x2E,0x3D,0x3E,0x2D,0x2E,0x3E,0x2D,0x3E,0x3F,0x2C,0x2D,0x3F,0x2C,0x3F
,0x40,0x2B,0x2C,0x40,0x2B,0x40,0x41,0x2A,0x2B,0x41,0x2A,0x41,0x42,0x29,0x2A
,0x42,0x28,0x29,0x42,0x28,0x42,0x43,0x27,0x28,0x43,0x27,0x43,0x44,0x26,0x27
,0x44,0x26,0x44,0x45,0x25,0x26,0x45,0x25,0x45,0x46,0x24,0x25,0x46,0x24,0x46
,0x47,0x23,0x24,0x47,0x23,0x47,0x48,0x23,0x48,0x49,0x22,0x23,0x49,0x22,0x49
,0x4A,0x21,0x22,0x4A,0x21,0x4A,0x4B,0x20,0x21,0x4B,0x20,0x4B,0x4C,0x1F,0x20
,0x4C,0x1F,0x4C,0x4D,0x1E,0x1F,0x4D,0x1E,0x4D,0x4E,0x1D,0x1E,0x4E,0x1D,0x4E
,0x4F,0x1C,0x1D,0x4F,0x1C,0x4F,0x50,0x1B,0x1C,0x50,0x1B,0x50,0x51,0x1A,0x1B
,0x51,0x1A,0x51,0x52,0x19,0x1A,0x52,0x19,0x52,0x53,0x6F,0x06,0x07,0x6E,0x6F
,0x07,0x6D,0x6E,0x07,0x6D,0x07,0x08,0x6C,0x6D,0x08,0x6C,0x08,0x09,0x6C,0x09
,0x0A,0x6B,0x6C,0x0A,0x6B,0x0A,0x0B,0x6A,0x6B,0x0B,0x69,0x6A,0x0B,0x69,0x0B
,0x0C,0x69,0x0C,0x0D,0x68,0x69,0x0D,0x68,0x0D,0x0E,0x68,0x0E,0x0F,0x67,0x68
,0x0F,0x67,0x0F,0x10,0x66,0x67,0x10,0x66,0x10,0x11,0x65,0x66,0x11,0x65,0x11
,0x12,0x64,0x65,0x12,0x63,0x64,0x12,0x63,0x12,0x13,0x62,0x63,0x13,0x58,0x62
,0x13,0x58,0x13,0x14,0x57,0x58,0x14,0x57,0x14,0x15,0x56,0x57,0x15,0x56,0x15
,0x16,0x55,0x56,0x16,0x55,0x16,0x17,0x55,0x17,0x18,0x54,0x55,0x18,0x54,0x18
,0x19,0x54,0x19,0x53,0x5A,0x70,0x41,0x23,0x23,0x02,0xAC,0xC8,0x80,0x3E,0xB2
,0x49,0xBE,0xBE,0xF8,0x54,0x8E,0x3E,0x2F,0x89,0xE3,0xBE,0x06,0xD9,0x92,0x3E
,0x6D,0x3B,0xFD,0xBE,0x3C,0xA1,0x8F,0x3E,0x9D,0x85,0x05,0xBF,0xDF,0xF9,0x85
,0x3E,0xC7,0x4B,0x0B,0xBF,0x3A,0x5D,0x66,0x3E,0x98,0x6D,0x0F,0xBF,0x87,0x16
,0x29,0x3E,0x73,0x68,0x11,0xBF,0x87,0x16,0x29,0x3E,0x9A,0x99,0x19,0xBF,0xA8
,0x1C,0x23,0x3F,0x9A,0x99,0x19,0xBF,0xA8,0x1C,0x23,0x3F,0x73,0x68,0x11,0xBF
,0x61,0x6F,0x12,0x3F,0x57,0x43,0x0E,0xBF,0x3F,0xE0,0x05,0x3F,0x93,0x19,0x07
,0xBF,0xEF,0x91,0xF5,0x3E,0xBB,0x0A,0xF1,0xBE,0x14,0x06,0xDD,0x3E,0x92,0x41
,0xBE,0xBE,0x07,0x97,0x8E,0x3C,0x64,0x3B,0x1F,0x3F,0x87,0x16,0x59,0xBC,0x64
,0x3B,0x1F,0x3F,0x37,0x89,0xE1,0xBE,0x8E,0xEB,0xB7,0xBE,0xC7,0x4A,0xFC,0xBE
,0x93,0xC3,0xEF,0xBE,0xAA,0x47,0x0A,0xBF,0x02,0x7F,0x08,0xBF,0xDB,0x4E,0x13
,0xBF,0xC9,0x77,0x0D,0xBF,0xA8,0x1C,0x23,0xBF,0x73,0x68,0x11,0xBF,0xA8,0x1C
,0x23,0xBF,0x9A,0x99,0x19,0xBF,0x46,0x0A,0x85,0xBE,0x9A,0x99,0x19,0xBF,0x46
,0x0A,0x85,0xBE,0x73,0x68,0x11,0xBF,0x60,0x8F,0xA9,0xBE,0x93,0xE3,0x0E,0xBF
,0x9D,0x9B,0xBE,0xBE,0x06,0x80,0x0A,0xBF,0x85,0x97,0xC8,0xBE,0x48,0xC0,0x04
,0xBF,0x7D,0xEB,0xCB,0xBE,0xCB,0x4D,0xFC,0xBE,0x7F,0x15,0xC8,0xBE,0x8B,0x18
,0xE6,0xBE,0x86,0x93,0xBC,0xBE,0x06,0xD5,0xC6,0xBE,0x9E,0x97,0x92,0xBE,0xCD
,0xCC,0x4C,0xBE,0x19,0xAC,0x38,0x3E,0xCD,0xCC,0x4C,0xBE,0x4C,0xDF,0x83,0xBE
,0x4C,0x37,0x09,0xBE,0x3A,0xCA,0x41,0xBD,0xD5,0xCC,0xB2,0x3E,0xE8,0xA3,0x1C
,0x3E,0x4C,0x37,0x09,0xBE,0x22,0x0B,0x0C,0x22,0x0C,0x0D,0x22,0x0D,0x0E,0x21
,0x22,0x0E,0x21,0x0E,0x0F,0x21,0x0F,0x10,0x20,0x21,0x10,0x15,0x16,0x17,0x15
,0x17,0x18,0x15,0x18,0x19,0x15,0x19,0x1A,0x14,0x15,0x1A,0x13,0x14,0x1A,0x12
,0x13,0x1A,0x11,0x12,0x1A,0x11,0x1A,0x1B,0x10,0x11,0x1B,0x10,0x1B,0x1C,0x10
,0x1C,0x1D,0x07,0x08,0x09,0x07,0x09,0x0A,0x07,0x0A,0x0B,0x06,0x07,0x0B,0x05
,0x06,0x0B,0x04,0x05,0x0B,0x03,0x04,0x0B,0x02,0x03,0x0B,0x10,0x1D,0x1E,0x20
,0x10,0x1E,0x20,0x1E,0x1F,0x22,0x20,0x1F,0x0B,0x22,0x1F,0x0B,0x1F,0x00,0x0B
,0x00,0x01,0x02,0x0B,0x01,0x1F,0x22,0x42,0x44,0x46,0x03,0x60,0x3B,0xB8,0x3E
,0x15,0xAA,0x5B,0x3D,0xF6,0xD4,0xDA,0x3E,0xA7,0x93,0xEC,0x3D,0xD5,0xCB,0xEF
,0x3E,0x69,0xE4,0x43,0x3E,0xB4,0xC8,0xF6,0x3E,0xAA,0x99,0x8D,0x3E,0xBE,0xDE
,0xED,0x3E,0x89,0x41,0xC0,0x3E,0xB8,0x20,0xD3,0x3E,0x4A,0xB4,0xEC,0x3E,0x32
,0x75,0xA7,0x3E,0x5C,0xE3,0x07,0x3F,0x2F,0x85,0x57,0x3E,0xF2,0x26,0x13,0x3F
,0xA9,0xF5,0xDE,0x3D,0x50,0x38,0x17,0x3F,0xA0,0x1A,0xAF,0xBC,0x75,0x93,0x18
,0x3F,0x5E,0xBA,0x09,0xBF,0x75,0x93,0x18,0x3F,0x5E,0xBA,0x09,0xBF,0x4E,0x62
,0x10,0x3F,0x44,0x8B,0xFC,0xBE,0x4E,0x62,0x10,0x3F,0x58,0x37,0xDE,0xBE,0x54
,0xE3,0x0D,0x3F,0x90,0x6A,0xC8,0xBE,0x66,0x66,0x06,0x3F,0xD7,0xA5,0xBE,0xBE
,0x7D,0x41,0xF3,0x3E,0x5A,0x64,0xBB,0xBE,0xA6,0x9B,0xC4,0x3E,0x5A,0x64,0xBB
,0xBE,0xF0,0xA7,0xC6,0xBE,0xDF,0xF9,0xBD,0xBE,0xAC,0x72,0xF1,0xBE,0x4D,0xBA
,0xC5,0xBE,0xE9,0x26,0x05,0xBF,0x88,0xBE,0xDB,0xBE,0x0C,0x58,0x0E,0xBF,0x44
,0x8B,0xFC,0xBE,0x73,0x68,0x11,0xBF,0x5E,0xBA,0x09,0xBF,0x73,0x68,0x11,0xBF
,0x5E,0xBA,0x09,0xBF,0x9A,0x99,0x19,0xBF,0x84,0xD5,0xD8,0x3C,0x9A,0x99,0x19
,0xBF,0x3D,0x63,0x4F,0x3E,0xB9,0x33,0x17,0xBF,0xE0,0x4D,0xA7,0x3E,0x08,0x02
,0x10,0xBF,0xBA,0x49,0xD4,0x3E,0x52,0x98,0x03,0xBF,0x54,0x8D,0xF6,0x3E,0x87
,0x14,0xE3,0xBE,0x75,0x1D,0x06,0x3F,0xC4,0x5E,0xB8,0xBE,0x5E,0xBA,0x09,0x3F
,0x88,0xBE,0x8B,0xBE,0x7D,0x40,0x04,0x3F,0x19,0xAC,0x28,0xBE,0xF4,0xA5,0xE7
,0x3E,0xF6,0xB3,0x98,0xBD,0x25,0x03,0xC0,0x3E,0xF6,0x29,0xC7,0xBC,0xFC,0x55
,0x88,0x3E,0x48,0xDF,0x24,0x3C,0x39,0xB4,0x48,0xBE,0x91,0x43,0x04,0x3F,0xE8
,0xDA,0xF7,0xBD,0x5D,0xFA,0x07,0x3F,0xF3,0x72,0x18,0xBD,0x4C,0x37,0x09,0x3F
,0xB2,0xA1,0xDB,0x3D,0x63,0x7C,0x04,0x3F,0x59,0x89,0x59,0x3E,0x96,0x97,0xEC
,0x3E,0xB2,0x9F,0x8D,0x3E,0xD9,0xCF,0xC2,0x3E,0x75,0x93,0x98,0x3E,0xE9,0x26
,0x91,0x3E,0x80,0x2C,0x94,0x3E,0xEB,0xA7,0x5F,0x3E,0xC2,0xF7,0x86,0x3E,0x77
,0x66,0x22,0x3E,0xAF,0xEC,0x62,0x3E,0x7A,0xC6,0xDE,0x3D,0xEC,0x51,0x28,0x3E
,0x99,0xF1,0x96,0x3D,0xB1,0x1A,0xAB,0x3D,0xEF,0xC8,0x58,0x3D,0xB2,0x9D,0xEF
,0xBC,0x7F,0x6A,0x3C,0x3D,0xC7,0xA1,0x9E,0xBD,0xE6,0x91,0x3F,0x3D,0x5A,0xBC
,0xF8,0xBD,0x03,0x06,0x49,0x3D,0x5C,0xE6,0x24,0xBE,0xEF,0xC8,0x58,0x3D,0x39
,0xB4,0x48,0xBE,0x9D,0xD9,0x6E,0x3D,0x39,0xB4,0x48,0xBE,0xA0,0x19,0xC4,0xBC
,0xF6,0x79,0x2C,0xBE,0x99,0x67,0xA5,0xBC,0x43,0x8F,0x08,0xBE,0x01,0x8A,0x91
,0xBC,0x76,0x33,0x23,0xBD,0x6F,0x12,0x83,0xBC,0xF8,0xFD,0x9B,0x3D,0xDD,0xE9
,0xCE,0xBC,0x32,0x04,0x30,0x3E,0x09,0x37,0x59,0xBD,0xDA,0xCA,0x7B,0x3E,0x75
,0x72,0xC6,0xBD,0x85,0xED,0x97,0x3E,0xFF,0x24,0x1E,0xBE,0x97,0x01,0xA7,0x3E
,0x8C,0x6A,0x61,0xBE,0x31,0x08,0xAC,0x3E,0x35,0xB2,0x93,0xBE,0xF2,0x3F,0xA1
,0x3E,0xA9,0x86,0xC5,0xBE,0x79,0xE7,0x80,0x3E,0x08,0xAA,0xEE,0xBE,0x35,0x07
,0x18,0x3E,0x04,0x21,0x05,0xBF,0x4E,0x64,0xE6,0x3B,0x5E,0xBA,0x09,0xBF,0xB3
,0x24,0xC0,0xBD,0x3B,0x34,0x08,0xBF,0x39,0xB4,0x48,0xBE,0xD3,0xA1,0x03,0xBF
,0x39,0x2B,0x2C,0x39,0x2C,0x2D,0x01,0x29,0x2A,0x29,0x01,0x02,0x38,0x39,0x2D
,0x38,0x2D,0x2E,0x29,0x02,0x03,0x01,0x2A,0x2B,0x00,0x01,0x2B,0x22,0x00,0x2B
,0x22,0x2B,0x39,0x22,0x39,0x3A,0x29,0x03,0x04,0x28,0x29,0x04,0x28,0x04,0x05
,0x28,0x05,0x06,0x27,0x28,0x06,0x27,0x06,0x07,0x26,0x27,0x07,0x26,0x07,0x08
,0x38,0x2E,0x2F,0x37,0x38,0x2F,0x37,0x2F,0x30,0x36,0x37,0x30,0x36,0x30,0x31
,0x21,0x22,0x3A,0x21,0x3A,0x3B,0x25,0x26,0x08,0x25,0x08,0x09,0x25,0x09,0x0A
,0x24,0x25,0x0A,0x23,0x24,0x0A,0x36,0x31,0x32,0x35,0x36,0x32,0x20,0x21,0x3B
,0x16,0x17,0x18,0x15,0x16,0x18,0x14,0x15,0x18,0x13,0x14,0x18,0x12,0x13,0x18
,0x0A,0x0B,0x0C,0x23,0x0A,0x0C,0x23,0x0C,0x0D,0x23,0x0D,0x0E,0x23,0x0E,0x0F
,0x33,0x23,0x0F,0x33,0x0F,0x10,0x33,0x10,0x11,0x35,0x32,0x33,0x34,0x35,0x33
,0x34,0x33,0x11,0x43,0x34,0x11,0x43,0x11,0x12,0x43,0x12,0x18,0x42,0x43,0x18
,0x41,0x42,0x18,0x40,0x41,0x18,0x40,0x18,0x19,0x3F,0x40,0x19,0x3F,0x19,0x1A
,0x20,0x3B,0x3C,0x1F,0x20,0x3C,0x1E,0x1F,0x3C,0x1E,0x3C,0x3D,0x1D,0x1E,0x3D
,0x1D,0x3D,0x3E,0x1C,0x1D,0x3E,0x1B,0x1C,0x3E,0x1B,0x3E,0x3F,0x1B,0x3F,0x1A
,0x22,0x33,0x43,0x43,0x2E,0x2C,0x01,0x66,0x66,0xE6,0x3E,0x64,0x3B,0x1F,0x3F
,0x93,0xC3,0xDF,0x3E,0x19,0xAF,0x15,0x3F,0x7F,0xBE,0xD5,0x3E,0xEA,0x7A,0x0E
,0x3F,0xB6,0xF5,0xCB,0x3E,0xD3,0x4C,0x0B,0x3F,0x85,0x97,0xC0,0x3E,0x71,0x3D
,0x0A,0x3F,0xBC,0xCB,0xB5,0x3E,0xD1,0x77,0x0B,0x3F,0xAA,0x99,0xA5,0x3E,0xF2
,0x26,0x0F,0x3F,0x79,0x91,0x49,0x3E,0x48,0x36,0x1B,0x3F,0xBA,0x49,0x8C,0x3D
,0x64,0x3B,0x1F,0x3F,0x3B,0xAA,0xBA,0xBD,0x0E,0xF7,0x19,0x3F,0x2F,0x89,0x73
,0xBE,0xFA,0x29,0x0A,0x3F,0x1F,0xD8,0xB9,0xBE,0x71,0x93,0xE1,0x3E,0x13,0x81
,0xEA,0xBE,0x92,0x97,0x9D,0x3E,0xE9,0x9B,0x04,0xBF,0x58,0x8C,0x1A,0x3E,0x5E
,0xBA,0x09,0xBF,0x9B,0xAE,0xA7,0xBC,0xAE,0xF1,0x01,0xBF,0x4C,0x37,0x69,0xBE
,0x1B,0x2F,0xD5,0xBE,0x2B,0xDB,0xCF,0xBE,0xB8,0x22,0x61,0xBE,0x88,0x2C,0x12
,0xBF,0x21,0xB0,0x32,0x3D,0x89,0x41,0x20,0xBF,0x81,0xE8,0x49,0x3E,0x17,0x83
,0x1B,0xBF,0x7B,0x12,0xA8,0x3E,0xAE,0x47,0x0D,0xBF,0x20,0x9A,0xE1,0x3E,0xEC
,0xA5,0xE9,0xBE,0x5E,0xBA,0x09,0x3F,0xE5,0xD0,0xA2,0xBE,0x5A,0xBA,0x02,0x3F
,0x9A,0x99,0x99,0xBE,0xA5,0x2E,0xD1,0x3E,0x10,0xEC,0xD8,0xBE,0x8C,0x6A,0xA1
,0x3E,0x56,0x0C,0xFF,0xBE,0x13,0xD4,0x60,0x3E,0xD2,0x18,0x09,0xBF,0x80,0xBA
,0xE1,0x3D,0xBA,0x49,0x0C,0xBF,0xFD,0xA2,0x84,0xBC,0xA8,0x3A,0x08,0xBF,0x78
,0xED,0x02,0xBE,0x9C,0x1A,0xF8,0xBE,0x0D,0x52,0x60,0xBE,0x2F,0x88,0xD0,0xBE
,0xAC,0x72,0x91,0xBE,0x9C,0xC2,0x9A,0xBE,0xE1,0x62,0xA5,0xBE,0x56,0xB9,0x30
,0xBE,0x31,0x08,0xAC,0xBE,0x81,0x96,0xAE,0xBC,0x93,0x00,0xA5,0xBE,0x30,0xD7
,0x22,0x3E,0x75,0xE9,0x8F,0xBE,0xCF,0x4B,0x9D,0x3E,0xB0,0x20,0x5D,0xBE,0x16
,0xC1,0xD7,0x3E,0x38,0x85,0x05,0xBE,0x02,0xD5,0xFF,0x3E,0x71,0xE5,0xEC,0xBC
,0xBF,0x7F,0x0B,0x3F,0xF0,0xF7,0xAB,0x3D,0x29,0x5C,0x0F,0x3F,0xFE,0x7F,0x5C
,0x3E,0xC6,0x35,0x0A,0x3F,0xE9,0x28,0xA7,0x3E,0x16,0x85,0xF5,0x3E,0x24,0xEF
,0xD4,0x3E,0x1F,0x30,0xBF,0x3E,0xB4,0xC8,0xF6,0x3E,0x1B,0x2F,0x5D,0x3E,0x5C
,0x8F,0x02,0x3F,0x1B,0x2F,0x5D,0x3E,0xB4,0xC8,0xF6,0x3E,0x64,0x3B,0x1F,0x3F
,0x2B,0x2C,0x2D,0x2A,0x2B,0x2D,0x2A,0x2D,0x00,0x2A,0x00,0x01,0x2A,0x01,0x02
,0x29,0x2A,0x02,0x29,0x02,0x03,0x29,0x03,0x04,0x29,0x04,0x05,0x28,0x29,0x05
,0x28,0x05,0x06,0x28,0x06,0x07,0x27,0x28,0x07,0x27,0x07,0x08,0x26,0x27,0x08
,0x26,0x08,0x09,0x25,0x26,0x09,0x25,0x09,0x0A,0x24,0x25,0x0A,0x24,0x0A,0x0B
,0x23,0x24,0x0B,0x23,0x0B,0x0C,0x22,0x23,0x0C,0x22,0x0C,0x0D,0x21,0x22,0x0D
,0x21,0x0D,0x0E,0x21,0x0E,0x0F,0x20,0x21,0x0F,0x20,0x0F,0x10,0x1F,0x20,0x10
,0x1F,0x10,0x11,0x1E,0x1F,0x11,0x1D,0x1E,0x11,0x1D,0x11,0x12,0x1C,0x1D,0x12
,0x1B,0x1C,0x12,0x1B,0x12,0x13,0x1A,0x1B,0x13,0x1A,0x13,0x14,0x19,0x1A,0x14
,0x19,0x14,0x15,0x18,0x19,0x15,0x18,0x15,0x16,0x17,0x18,0x16,0x2D,0x44,0x26
,0x26,0x02,0x55,0x33,0x8B,0xBD,0x9A,0x99,0x19,0xBF,0x13,0xD7,0x71,0x3E,0xCB
,0xA1,0x0D,0xBF,0xEB,0xFD,0xE6,0x3E,0xBC,0x74,0xD3,0xBE,0x65,0xA6,0x10,0x3F
,0xF4,0x51,0x66,0xBE,0x35,0x5E,0x1A,0x3F,0x7E,0x3B,0x89,0x3B,0x3A,0xEA,0x14
,0x3F,0x69,0x8F,0x37,0x3E,0x58,0x8E,0x04,0x3F,0xBC,0x20,0xAA,0x3E,0xE5,0x7B
,0xD6,0x3E,0x2A,0x72,0xE8,0x3E,0x1B,0xD9,0x95,0x3E,0x00,0xFF,0x08,0x3F,0x56
,0x0C,0xF7,0x3D,0x53,0xAE,0x14,0x3F,0x0E,0xDD,0xEC,0xBD,0x75,0x93,0x18,0x3F
,0x35,0x5E,0x1A,0xBF,0x75,0x93,0x18,0x3F,0x35,0x5E,0x1A,0xBF,0x4E,0x62,0x10
,0x3F,0x17,0xD9,0x0E,0xBF,0x4E,0x62,0x10,0x3F,0x54,0x37,0xFF,0xBE,0x54,0xE3
,0x0D,0x3F,0xF0,0x4F,0xE9,0xBE,0x66,0x66,0x06,0x3F,0x02,0xD5,0xDF,0xBE,0xFE
,0x7C,0xF3,0x3E,0x08,0xAC,0xDC,0xBE,0xA6,0x9B,0xC4,0x3E,0x08,0xAC,0xDC,0xBE
,0xF0,0xA7,0xC6,0xBE,0xFE,0x27,0xDF,0xBE,0x54,0x8E,0xF1,0xBE,0x9D,0x9B,0xE6
,0xBE,0x89,0xEB,0x04,0xBF,0x17,0x82,0xFC,0xBE,0x3D,0x49,0x0E,0xBF,0x17,0xD9
,0x0E,0xBF,0x73,0x68,0x11,0xBF,0x35,0x5E,0x1A,0xBF,0x73,0x68,0x11,0xBF,0x35
,0x5E,0x1A,0xBF,0x9A,0x99,0x19,0xBF,0xCB,0xA1,0x85,0xBE,0xB6,0x9F,0x00,0x3F
,0xA0,0x18,0x29,0xBE,0x2B,0x88,0x05,0x3F,0x5D,0x6A,0xA4,0xBD,0x02,0x2B,0x07
,0x3F,0x58,0x8D,0xE5,0x3D,0x5E,0x11,0xFC,0x3E,0x89,0x41,0x88,0x3E,0x8D,0x43
,0xC5,0x3E,0x41,0x0C,0xBC,0x3E,0x61,0x8B,0x5D,0x3E,0xDF,0x4F,0xCD,0x3E,0x0A
,0xD7,0xA3,0xBB,0x41,0x0C,0xBC,0x3E,0xCA,0xA4,0x66,0xBE,0x89,0x41,0x88,0x3E
,0x9E,0xED,0xC9,0xBE,0x0B,0x27,0xE9,0x3D,0x7F,0x89,0x00,0xBF,0x19,0x04,0x96
,0xBD,0x5E,0xBA,0x09,0xBF,0x65,0xE3,0x21,0xBE,0x67,0x26,0x08,0xBF,0xCB,0xA1
,0x85,0xBE,0x83,0x6A,0x03,0xBF,0x18,0x00,0x23,0x18,0x23,0x24,0x18,0x24,0x25
,0x23,0x00,0x01,0x22,0x23,0x01,0x21,0x22,0x01,0x21,0x01,0x02,0x20,0x21,0x02
,0x20,0x02,0x03,0x1F,0x20,0x03,0x1F,0x03,0x04,0x1F,0x04,0x05,0x1E,0x1F,0x05
,0x1E,0x05,0x06,0x1D,0x1E,0x06,0x1D,0x06,0x07,0x1D,0x07,0x08,0x1C,0x1D,0x08
,0x1C,0x08,0x09,0x1B,0x1C,0x09,0x1B,0x09,0x0A,0x1A,0x1B,0x0A,0x1A,0x0A,0x0B
,0x19,0x1A,0x0B,0x16,0x17,0x18,0x16,0x18,0x25,0x15,0x16,0x25,0x14,0x15,0x25
,0x13,0x14,0x25,0x13,0x25,0x19,0x12,0x13,0x19,0x11,0x12,0x19,0x10,0x11,0x19
,0x0B,0x0C,0x0D,0x0F,0x10,0x19,0x0E,0x0F,0x19,0x0E,0x19,0x0B,0x0D,0x0E,0x0B
,0x18,0x25,0x45,0x3A,0x38,0x01,0x20,0xD5,0x20,0x3E,0x14,0xAE,0x07,0x3F,0xAA
,0x43,0x6E,0x3E,0x02,0xD6,0x06,0x3F,0x72,0x14,0x90,0x3E,0xDB,0x4D,0x04,0x3F
,0xE3,0xE3,0xA3,0x3E,0x1D,0x04,0xFD,0x3E,0xF5,0x2A,0xB2,0x3E,0xFA,0x26,0xED
,0x3E,0x9D,0x2C,0xBD,0x3E,0xC3,0x9D,0xD3,0x3E,0x02,0x2B,0xC7,0x3E,0x0C,0x02
,0xAB,0x3E,0x75,0x93,0xD8,0x3E,0x0C,0x02,0xAB,0x3E,0x98,0x6E,0xD2,0x3E,0x75
,0x93,0x18,0x3F,0xF8,0x53,0x03,0xBF,0x75,0x93,0x18,0x3F,0xF8,0x53,0x03,0xBF
,0x4E,0x62,0x10,0x3F,0x40,0xDD,0xF0,0xBE,0x4E,0x62,0x10,0x3F,0x0E,0x2F,0xD0
,0xBE,0xB4,0x1E,0x0E,0x3F,0x77,0x66,0xBA,0xBE,0xF8,0x53,0x07,0x3F,0x50,0x8B
,0xB1,0xBE,0x2B,0xDD,0xF5,0x3E,0x8D,0x97,0xAE,0xBE,0xF0,0xA7,0xC6,0x3E,0x8D
,0x97,0xAE,0xBE,0xF0,0xA7,0xC6,0xBE,0xEB,0xE5,0xAF,0xBE,0x99,0x2E,0xEC,0xBE
,0xE1,0xD0,0xB3,0xBE,0x3F,0x36,0x01,0xBF,0x9E,0x5C,0xBB,0xBE,0x5F,0x99,0x07
,0xBF,0x50,0x8D,0xC7,0xBE,0xF6,0x28,0x0C,0xBF,0xCA,0xA3,0xDB,0xBE,0x93,0x18
,0x10,0xBF,0x40,0xDD,0xF0,0xBE,0x73,0x68,0x11,0xBF,0xF8,0x53,0x03,0xBF,0x73
,0x68,0x11,0xBF,0xF8,0x53,0x03,0xBF,0x9A,0x99,0x19,0xBF,0x23,0xDB,0xD1,0x3E
,0x9A,0x99,0x19,0xBF,0xF8,0x53,0x03,0x3F,0x9A,0x99,0x99,0xBE,0x33,0x33,0xF3
,0x3E,0x9A,0x99,0x99,0xBE,0x00,0x53,0xD6,0x3E,0xCD,0xCE,0xCA,0xBE,0xBC,0x20
,0xBA,0x3E,0xF2,0xD2,0xED,0xBE,0x1D,0x1C,0xA4,0x3E,0x78,0x80,0xFF,0xBE,0x33
,0x31,0x8D,0x3E,0x9C,0x19,0x05,0xBF,0x02,0x7E,0x5D,0x3E,0x9A,0xCD,0x07,0xBF
,0xDC,0x49,0x04,0x3E,0x39,0xB4,0x08,0xBF,0x72,0xDD,0x94,0xBD,0x39,0xB4,0x08
,0xBF,0xFF,0x5B,0xE9,0xBD,0x17,0x0E,0x08,0xBF,0x72,0xC0,0x0E,0xBE,0xA0,0x1B
,0x06,0xBF,0x35,0x09,0x1E,0xBE,0x9D,0x0F,0x03,0xBF,0x6C,0xEB,0x27,0xBE,0x50
,0x39,0xFE,0xBE,0xD3,0x4E,0x2D,0xBE,0xE6,0x78,0xED,0xBE,0xA0,0x1A,0x2F,0xBE
,0xDD,0xD0,0xCC,0xBE,0xA0,0x1A,0x2F,0xBE,0xA6,0x9B,0xC4,0xBB,0xC9,0x76,0xBE
,0x3D,0xA6,0x9B,0xC4,0xBB,0xA3,0x1E,0x22,0x3E,0xED,0x7D,0x2A,0xBC,0xC9,0x1E
,0x51,0x3E,0x37,0x89,0xC1,0xBC,0xD3,0xF9,0x70,0x3E,0x4E,0x7F,0x36,0xBD,0x43
,0x37,0x83,0x3E,0x05,0xA6,0x93,0xBD,0x3B,0x8C,0x89,0x3E,0xEC,0x30,0xE6,0xBD
,0x3B,0xDF,0x8F,0x3E,0x7F,0x6A,0x3C,0xBE,0x89,0x41,0xA0,0x3E,0x7F,0x6A,0x3C
,0xBE,0x89,0x41,0xA0,0x3E,0x68,0x91,0x6D,0x3E,0x3B,0xDF,0x8F,0x3E,0x68,0x91
,0x6D,0x3E,0xE0,0xF7,0x87,0x3E,0x06,0x81,0x15,0x3E,0x19,0x54,0x6B,0x3E,0x18
,0x79,0xB9,0x3D,0x21,0xAC,0x36,0x3E,0x5D,0x87,0x8A,0x3D,0xC9,0x76,0xBE,0x3D
,0x8F,0xC2,0x75,0x3D,0xA0,0x1A,0x2F,0xBE,0x8F,0xC2,0x75,0x3D,0xA0,0x1A,0x2F
,0xBE,0x14,0xAE,0x07,0x3F,0x19,0x1A,0x1B,0x19,0x1B,0x1C,0x19,0x1C,0x1D,0x19
,0x1D,0x1E,0x06,0x07,0x08,0x05,0x06,0x08,0x04,0x05,0x08,0x03,0x04,0x08,0x31
,0x32,0x33,0x31,0x33,0x34,0x31,0x34,0x35,0x17,0x18,0x19,0x16,0x17,0x19,0x15
,0x16,0x19,0x14,0x15,0x19,0x13,0x14,0x19,0x12,0x13,0x19,0x09,0x0A,0x0B,0x08
,0x09,0x0B,0x08,0x0B,0x0C,0x08,0x0C,0x0D,0x2F,0x30,0x31,0x2F,0x31,0x35,0x2E
,0x2F,0x35,0x2D,0x2E,0x35,0x2C,0x2D,0x35,0x2C,0x35,0x36,0x2B,0x2C,0x36,0x2B
,0x36,0x37,0x2A,0x2B,0x37,0x29,0x2A,0x37,0x29,0x37,0x38,0x02,0x03,0x08,0x19
,0x1E,0x1F,0x01,0x02,0x08,0x19,0x1F,0x20,0x00,0x01,0x08,0x19,0x20,0x21,0x39
,0x00,0x08,0x39,0x08,0x0D,0x39,0x0D,0x0E,0x38,0x39,0x0E,0x38,0x0E,0x0F,0x38
,0x0F,0x10,0x29,0x38,0x10,0x28,0x29,0x10,0x28,0x10,0x11,0x27,0x28,0x11,0x27
,0x11,0x12,0x26,0x27,0x12,0x25,0x26,0x12,0x24,0x25,0x12,0x19,0x21,0x22,0x23
,0x24,0x12,0x23,0x12,0x19,0x23,0x19,0x22,0x39,0x46,0x33,0x31,0x01,0xD7,0xFB
,0x1D,0x3E,0x14,0xAE,0x07,0x3F,0x3E,0x08,0x71,0x3E,0x5C,0x8F,0x06,0x3F,0x48
,0x89,0x95,0x3E,0x33,0x33,0x03,0x3F,0xB9,0x89,0xAA,0x3E,0x3A,0xB2,0xFA,0x3E
,0x08,0x02,0xBC,0x3E,0x17,0x81,0xE9,0x3E,0xF0,0xBE,0xCA,0x3E,0x75,0xE9,0xCF
,0x3E,0x50,0x8D,0xD7,0x3E,0x0C,0x02,0xAB,0x3E,0x79,0xE9,0xE6,0x3E,0x0C,0x02
,0xAB,0x3E,0x9C,0xC4,0xE0,0x3E,0x75,0x93,0x18,0x3F,0x79,0xE9,0xE6,0xBE,0x75
,0x93,0x18,0x3F,0x79,0xE9,0xE6,0xBE,0x4E,0x62,0x10,0x3F,0xC9,0x1E,0xD1,0xBE
,0x4E,0x62,0x10,0x3F,0x58,0x8F,0xBB,0xBE,0xA7,0x05,0x0F,0x3F,0x29,0x5C,0xA7
,0xBE,0xA2,0xEF,0x0A,0x3F,0x19,0xAD,0x9B,0xBE,0xB4,0x73,0x06,0x3F,0x7B,0xC0
,0x94,0xBE,0x13,0x2D,0x01,0x3F,0xF8,0x52,0x90,0xBE,0x87,0x6D,0xEB,0x3E,0x17
,0xD9,0x8E,0xBE,0xA6,0x9B,0xC4,0x3E,0x17,0xD9,0x8E,0xBE,0xF0,0xA7,0xC6,0xBE
,0x64,0x92,0x91,0xBE,0xF1,0xD4,0xF3,0xBE,0x6E,0xBE,0x99,0xBE,0xC3,0x49,0x06
,0xBF,0x44,0x36,0xB0,0xBE,0xCB,0xA0,0x0E,0xBF,0xC9,0x1E,0xD1,0xBE,0x73,0x68
,0x11,0xBF,0x79,0xE9,0xE6,0xBE,0x73,0x68,0x11,0xBF,0x79,0xE9,0xE6,0xBE,0x9A
,0x99,0x19,0xBF,0x25,0x06,0x81,0x3D,0x9A,0x99,0x19,0xBF,0x25,0x06,0x81,0x3D
,0x73,0x68,0x11,0xBF,0x0A,0xD7,0xA3,0x3C,0x73,0x68,0x11,0xBF,0x7B,0x14,0xAE
,0xBC,0xCC,0x0B,0x10,0xBF,0xFE,0xD4,0x78,0xBD,0xC7,0xF5,0x0B,0xBF,0xFF,0x03
,0xAC,0xBD,0xD9,0x79,0x07,0xBF,0x5D,0xDF,0xC7,0xBD,0x37,0x33,0x02,0xBF,0x1F
,0x64,0xD9,0xBD,0xD1,0x79,0xED,0xBE,0x64,0x3B,0xDF,0xBD,0xF0,0xA7,0xC6,0xBE
,0x64,0x3B,0xDF,0xBD,0xA6,0x9B,0x44,0xBC,0x64,0x3B,0xDF,0x3D,0xA6,0x9B,0x44
,0xBC,0xAD,0x6E,0x25,0x3E,0x90,0xD7,0x83,0xBC,0xD5,0x20,0x4C,0x3E,0xB0,0x72
,0xE8,0xBC,0x33,0xE1,0x67,0x3E,0x10,0x04,0x48,0xBD,0x51,0xDD,0x7C,0x3E,0xAE
,0x9F,0x9E,0xBD,0xE1,0x24,0x85,0x3E,0x7C,0x9D,0xF4,0xBD,0x14,0xAE,0x87,0x3E
,0xEC,0x51,0x38,0xBE,0x62,0x10,0x98,0x3E,0xEC,0x51,0x38,0xBE,0x62,0x10,0x98
,0x3E,0x8B,0x6C,0x67,0x3E,0x14,0xAE,0x87,0x3E,0x8B,0x6C,0x67,0x3E,0xC2,0xA3
,0x7D,0x3E,0x77,0x69,0x13,0x3E,0xE0,0xF7,0x5F,0x3E,0xAD,0x14,0xC2,0x3D,0x63
,0x0E,0x32,0x3E,0x4D,0xC0,0x8F,0x3D,0x64,0x3B,0xDF,0x3D,0xB6,0xF3,0x7D,0x3D
,0x64,0x3B,0xDF,0xBD,0xB6,0xF3,0x7D,0x3D,0x64,0x3B,0xDF,0xBD,0x14,0xAE,0x07
,0x3F,0x06,0x07,0x08,0x05,0x06,0x08,0x04,0x05,0x08,0x03,0x04,0x08,0x2A,0x2B
,0x2C,0x2A,0x2C,0x2D,0x2A,0x2D,0x2E,0x19,0x1A,0x1B,0x18,0x19,0x1B,0x18,0x1B
,0x1C,0x18,0x1C,0x1D,0x18,0x1D,0x1E,0x18,0x1E,0x1F,0x16,0x17,0x18,0x09,0x0A
,0x0B,0x08,0x09,0x0B,0x08,0x0B,0x0C,0x08,0x0C,0x0D,0x08,0x0D,0x0E,0x18,0x1F
,0x20,0x28,0x29,0x2A,0x28,0x2A,0x2E,0x27,0x28,0x2E,0x26,0x27,0x2E,0x25,0x26
,0x2E,0x25,0x2E,0x2F,0x24,0x25,0x2F,0x24,0x2F,0x30,0x23,0x24,0x30,0x22,0x23
,0x30,0x22,0x30,0x31,0x02,0x03,0x08,0x01,0x02,0x08,0x15,0x16,0x18,0x15,0x18
,0x20,0x14,0x15,0x20,0x13,0x14,0x20,0x12,0x13,0x20,0x12,0x20,0x21,0x11,0x12
,0x21,0x11,0x21,0x22,0x11,0x22,0x31,0x11,0x31,0x32,0x10,0x11,0x32,0x0F,0x10
,0x32,0x0E,0x0F,0x32,0x08,0x0E,0x32,0x08,0x32,0x00,0x01,0x08,0x00,0x32,0x47
,0x3D,0x3B,0x01,0x17,0xD9,0xCE,0x3E,0x64,0x3B,0x1F,0x3F,0x76,0x51,0xCC,0x3E
,0x0B,0x97,0x15,0x3F,0x72,0x6A,0xC7,0x3E,0x6F,0xBC,0x0F,0x3F,0xCC,0xB5,0xC0
,0x3E,0x15,0xC4,0x0C,0x3F,0x9C,0xC4,0xB8,0x3E,0xA8,0xC6,0x0B,0x3F,0x4E,0x0D
,0xAC,0x3E,0xCF,0xF6,0x0C,0x3F,0xEB,0xFD,0x96,0x3E,0x33,0x87,0x10,0x3F,0x46
,0x5B,0x55,0x3E,0x39,0x29,0x18,0x3F,0xEE,0x78,0x13,0x3E,0x1F,0x84,0x1C,0x3F
,0xE0,0xD8,0xB3,0x3D,0x97,0x8D,0x1E,0x3F,0x8B,0x6D,0xD2,0x3C,0x64,0x3B,0x1F
,0x3F,0x6E,0x6A,0x10,0xBE,0xF8,0x54,0x1A,0x3F,0x73,0x68,0x91,0xBE,0xD3,0xA1
,0x0B,0x3F,0x31,0x07,0xD9,0xBE,0xEE,0xCF,0xDD,0x3E,0xA0,0xC4,0x07,0xBF,0x72
,0x14,0x90,0x3E,0x45,0x4B,0x16,0xBF,0xF4,0xF9,0x08,0x3E,0xD1,0x22,0x1B,0xBF
,0x9B,0xAE,0xA7,0xBC,0xA5,0x86,0x12,0xBF,0xF9,0x82,0x66,0xBE,0x63,0x64,0xF1
,0xBE,0xCB,0xF5,0xCE,0xBE,0xB2,0x47,0x80,0xBE,0xE4,0x0F,0x12,0xBF,0x4F,0x02
,0x5B,0x3D,0x89,0x41,0x20,0xBF,0xC0,0x7A,0x2C,0x3E,0x02,0xA0,0x1E,0xBF,0xCD
,0x78,0x8B,0x3E,0x6B,0xBB,0x19,0xBF,0x5C,0x3A,0xBE,0x3E,0xE2,0x5A,0x11,0xBF
,0xFC,0xA9,0xF1,0x3E,0xA6,0x45,0x05,0xBF,0xFC,0xA9,0xF1,0x3E,0x77,0xBE,0x1F
,0xBE,0x7D,0xEB,0xF3,0x3E,0xF1,0xD6,0x99,0xBD,0x21,0xB0,0xFA,0x3E,0x3A,0xC9
,0xD6,0xBC,0x69,0xA7,0x06,0x3F,0xE6,0x5C,0x0A,0x3C,0xFE,0x7E,0x15,0x3F,0x0A
,0xD7,0xA3,0x3C,0xD1,0x22,0x1B,0x3F,0x0A,0xD7,0xA3,0x3C,0xD1,0x22,0x1B,0x3F
,0xF4,0xFD,0x54,0x3D,0x06,0x81,0x15,0x3E,0xF4,0xFD,0x54,0x3D,0x06,0x81,0x15
,0x3E,0x0A,0xD7,0xA3,0x3C,0xE3,0xFE,0x53,0x3E,0xB0,0xCA,0x85,0x3C,0x89,0x45
,0x7C,0x3E,0x81,0x96,0xAE,0x3B,0xD9,0xCF,0x8A,0x3E,0xFF,0xAE,0x4F,0xBC,0x0C
,0xAC,0x93,0x3E,0x06,0xF6,0x18,0xBD,0xC1,0xE2,0x98,0x3E,0x86,0xE7,0xA5,0xBD
,0xBE,0x9F,0x9A,0x3E,0x77,0xBE,0x1F,0xBE,0xBE,0x9F,0x9A,0x3E,0x54,0x37,0xFF
,0xBE,0xB0,0x75,0x79,0x3E,0x48,0xE1,0x06,0xBF,0x8C,0x68,0x3B,0x3E,0xCD,0x22
,0x0C,0xBF,0x48,0x33,0xF6,0x3D,0x76,0x52,0x0F,0xBF,0x51,0x2D,0x62,0x3D,0x4E
,0x62,0x10,0xBF,0x32,0x21,0x66,0xBD,0x76,0x32,0x0C,0xBF,0xE7,0xA7,0x28,0xBE
,0x9E,0x45,0xFF,0xBE,0x71,0xE8,0x85,0xBE,0x88,0x2A,0xD4,0xBE,0x0D,0x00,0xAD
,0xBE,0x02,0xD7,0x95,0xBE,0xDC,0x62,0xC6,0xBE,0xED,0x2A,0x14,0xBE,0x17,0xD9
,0xCE,0xBE,0x00,0xFF,0x94,0x3C,0xF2,0x27,0xC2,0xBE,0x31,0xB4,0x5A,0x3E,0x83
,0x14,0x9C,0xBE,0x5C,0x8F,0xC2,0x3E,0x15,0x00,0x23,0xBE,0x0A,0xD7,0x03,0x3F
,0x8B,0x6C,0x27,0x3D,0x29,0x5C,0x0F,0x3F,0x7F,0x14,0x45,0x3E,0x05,0xC0,0x08
,0x3F,0xE7,0xA7,0xA0,0x3E,0x13,0xD7,0xE9,0x3E,0x74,0xEA,0xC2,0x3E,0xD7,0x4E
,0xBC,0x3E,0x3F,0x35,0xDE,0x3E,0x23,0xDB,0x79,0x3E,0x8D,0x97,0xEE,0x3E,0x23
,0xDB,0x79,0x3E,0x64,0x3B,0xDF,0x3E,0x64,0x3B,0x1F,0x3F,0x3A,0x3B,0x3C,0x39
,0x3A,0x3C,0x38,0x39,0x3C,0x1F,0x20,0x21,0x1F,0x21,0x22,0x1F,0x22,0x23,0x1F
,0x23,0x24,0x1F,0x24,0x25,0x1D,0x1E,0x1F,0x1D,0x1F,0x25,0x1C,0x1D,0x25,0x1B
,0x1C,0x25,0x1A,0x1B,0x25,0x1A,0x25,0x26,0x19,0x1A,0x26,0x19,0x26,0x27,0x18
,0x19,0x27,0x18,0x27,0x28,0x17,0x18,0x28,0x16,0x17,0x28,0x16,0x28,0x29,0x15
,0x16,0x29,0x15,0x29,0x2A,0x15,0x2A,0x2B,0x14,0x15,0x2B,0x14,0x2B,0x2C,0x13
,0x14,0x2C,0x13,0x2C,0x2D,0x13,0x2D,0x2E,0x13,0x2E,0x2F,0x12,0x13,0x2F,0x12
,0x2F,0x30,0x11,0x12,0x30,0x11,0x30,0x31,0x10,0x11,0x31,0x10,0x31,0x32,0x0F
,0x10,0x32,0x0F,0x32,0x33,0x0E,0x0F,0x33,0x0D,0x0E,0x33,0x0D,0x33,0x34,0x0C
,0x0D,0x34,0x0C,0x34,0x35,0x0B,0x0C,0x35,0x0B,0x35,0x36,0x0A,0x0B,0x36,0x09
,0x0A,0x36,0x09,0x36,0x37,0x08,0x09,0x37,0x07,0x08,0x37,0x06,0x07,0x37,0x06
,0x37,0x38,0x05,0x06,0x38,0x04,0x05,0x38,0x3C,0x00,0x01,0x03,0x04,0x38,0x3C
,0x01,0x02,0x38,0x3C,0x02,0x03,0x38,0x02,0x3C,0x48,0x48,0x46,0x01,0xF2,0xD2
,0x8D,0xBE,0xA6,0x9B,0xC4,0x3E,0x42,0x78,0x8C,0xBE,0x6F,0x2A,0xEA,0x3E,0x77
,0x68,0x88,0xBE,0x6D,0x3C,0x00,0x3F,0x88,0xD8,0x80,0xBE,0x1E,0xA5,0x06,0x3F
,0xE7,0xFB,0x69,0xBE,0x2B,0x31,0x0B,0x3F,0x10,0x02,0x42,0xBE,0x09,0x16,0x0F
,0x3F,0xB9,0xC6,0x17,0xBE,0x4E,0x62,0x10,0x3F,0x3D,0x0A,0xD7,0xBD,0x4E,0x62
,0x10,0x3F,0x3D,0x0A,0xD7,0xBD,0x75,0x93,0x18,0x3F,0xC9,0x76,0x1E,0xBF,0x75
,0x93,0x18,0x3F,0xC9,0x76,0x1E,0xBF,0x4E,0x62,0x10,0x3F,0x60,0x91,0x13,0xBF
,0x4E,0x62,0x10,0x3F,0x6D,0xE6,0x08,0xBF,0x29,0x07,0x0F,0x3F,0x42,0x60,0xFD
,0xBE,0xCB,0xF5,0x0A,0x3F,0x31,0xB1,0xF1,0xBE,0x06,0x80,0x06,0x3F,0x93,0xC4
,0xEA,0xBE,0x7D,0x3F,0x01,0x3F,0x11,0x57,0xE6,0xBE,0x71,0x91,0xEB,0x3E,0x2F
,0xDD,0xE4,0xBE,0xA6,0x9B,0xC4,0x3E,0x2F,0xDD,0xE4,0xBE,0xF0,0xA7,0xC6,0xBE
,0x7D,0x96,0xE7,0xBE,0xF1,0xD4,0xF3,0xBE,0x87,0xC2,0xEF,0xBE,0xC3,0x49,0x06
,0xBF,0x2E,0x1D,0x03,0xBF,0xCB,0xA0,0x0E,0xBF,0x60,0x91,0x13,0xBF,0x73,0x68
,0x11,0xBF,0xC9,0x76,0x1E,0xBF,0x73,0x68,0x11,0xBF,0xC9,0x76,0x1E,0xBF,0x9A
,0x99,0x19,0xBF,0x3D,0x0A,0xD7,0xBD,0x9A,0x99,0x19,0xBF,0x3D,0x0A,0xD7,0xBD
,0x73,0x68,0x11,0xBF,0xB9,0xC6,0x17,0xBE,0x73,0x68,0x11,0xBF,0x10,0x02,0x42
,0xBE,0xCC,0x0B,0x10,0xBF,0xE7,0xFB,0x69,0xBE,0xC7,0xF5,0x0B,0xBF,0xB6,0xF4
,0x80,0xBE,0xD9,0x79,0x07,0xBF,0xAE,0xF3,0x87,0xBE,0x37,0x33,0x02,0xBF,0x29
,0x5B,0x8C,0xBE,0xD1,0x79,0xED,0xBE,0xF2,0xD2,0x8D,0xBE,0xF0,0xA7,0xC6,0xBE
,0xF2,0xD2,0x8D,0xBE,0xA6,0x9B,0xC4,0xBC,0xF2,0xD2,0x8D,0x3E,0xA6,0x9B,0xC4
,0xBC,0xF2,0xD2,0x8D,0x3E,0xF0,0xA7,0xC6,0xBE,0xF0,0xFC,0x8A,0x3E,0xF1,0xD4
,0xF3,0xBE,0xEA,0x7A,0x82,0x3E,0xC3,0x49,0x06,0xBF,0x92,0x3F,0x58,0x3E,0xCB
,0xA0,0x0E,0xBF,0xC1,0x1A,0x17,0x3E,0x73,0x68,0x11,0xBF,0x3D,0x0A,0xD7,0x3D
,0x73,0x68,0x11,0xBF,0x3D,0x0A,0xD7,0x3D,0x9A,0x99,0x19,0xBF,0xC9,0x76,0x1E
,0x3F,0x9A,0x99,0x19,0xBF,0xC9,0x76,0x1E,0x3F,0x73,0x68,0x11,0xBF,0x71,0x91
,0x13,0x3F,0x73,0x68,0x11,0xBF,0x6D,0xE6,0x08,0x3F,0xCC,0x0B,0x10,0xBF,0x42
,0x60,0xFD,0x3E,0xC7,0xF5,0x0B,0xBF,0x31,0xB1,0xF1,0x3E,0xD9,0x79,0x07,0xBF
,0x93,0xC4,0xEA,0x3E,0x37,0x33,0x02,0xBF,0x11,0x57,0xE6,0x3E,0xD1,0x79,0xED
,0xBE,0x2F,0xDD,0xE4,0x3E,0xF0,0xA7,0xC6,0xBE,0x2F,0xDD,0xE4,0x3E,0xA6,0x9B
,0xC4,0x3E,0x8C,0x2B,0xE6,0x3E,0x6F,0x2A,0xEA,0x3E,0x83,0x16,0xEA,0x3E,0x6D
,0x3C,0x00,0x3F,0x40,0xA2,0xF1,0x3E,0x1E,0xA5,0x06,0x3F,0xF2,0xD2,0xFD,0x3E
,0x2B,0x31,0x0B,0x3F,0xC7,0xF4,0x08,0x3F,0x09,0x16,0x0F,0x3F,0x71,0x91,0x13
,0x3F,0x4E,0x62,0x10,0x3F,0xC9,0x76,0x1E,0x3F,0x4E,0x62,0x10,0x3F,0xC9,0x76
,0x1E,0x3F,0x75,0x93,0x18,0x3F,0x3D,0x0A,0xD7,0x3D,0x75,0x93,0x18,0x3F,0x3D
,0x0A,0xD7,0x3D,0x4E,0x62,0x10,0x3F,0xC1,0x1A,0x17,0x3E,0x4E,0x62,0x10,0x3F
,0x69,0x8D,0x41,0x3E,0x29,0x07,0x0F,0x3F,0x5E,0xBA,0x69,0x3E,0xCB,0xF5,0x0A
,0x3F,0x5D,0xE2,0x80,0x3E,0x06,0x80,0x06,0x3F,0x8E,0xEB,0x87,0x3E,0x7D,0x3F
,0x01,0x3F,0x10,0x59,0x8C,0x3E,0x71,0x91,0xEB,0x3E,0xF2,0xD2,0x8D,0x3E,0xA6
,0x9B,0xC4,0x3E,0xF2,0xD2,0x8D,0x3E,0x31,0x08,0x2C,0x3D,0xF2,0xD2,0x8D,0xBE
,0x31,0x08,0x2C,0x3D,0x3D,0x3E,0x3F,0x3C,0x3D,0x3F,0x3C,0x3F,0x40,0x3C,0x40
,0x41,0x3C,0x41,0x42,0x3C,0x42,0x43,0x3A,0x3B,0x3C,0x2B,0x2C,0x2D,0x2A,0x2B
,0x2D,0x2A,0x2D,0x2E,0x2A,0x2E,0x2F,0x2A,0x2F,0x30,0x2A,0x30,0x31,0x28,0x29
,0x2A,0x19,0x1A,0x1B,0x18,0x19,0x1B,0x18,0x1B,0x1C,0x18,0x1C,0x1D,0x18,0x1D
,0x1E,0x18,0x1E,0x1F,0x16,0x17,0x18,0x09,0x0A,0x0B,0x08,0x09,0x0B,0x08,0x0B
,0x0C,0x08,0x0C,0x0D,0x08,0x0D,0x0E,0x08,0x0E,0x0F,0x06,0x07,0x08,0x08,0x0F
,0x10,0x3C,0x43,0x44,0x2A,0x31,0x32,0x18,0x1F,0x20,0x39,0x3A,0x3C,0x39,0x3C
,0x44,0x38,0x39,0x44,0x37,0x38,0x44,0x36,0x37,0x44,0x35,0x36,0x44,0x35,0x44
,0x45,0x34,0x35,0x45,0x33,0x34,0x45,0x33,0x45,0x46,0x05,0x06,0x08,0x05,0x08
,0x10,0x04,0x05,0x10,0x03,0x04,0x10,0x02,0x03,0x10,0x01,0x02,0x10,0x01,0x10
,0x11,0x00,0x01,0x11,0x47,0x00,0x11,0x47,0x11,0x12,0x15,0x16,0x18,0x15,0x18
,0x20,0x14,0x15,0x20,0x13,0x14,0x20,0x12,0x13,0x20,0x12,0x20,0x21,0x12,0x21
,0x22,0x47,0x12,0x22,0x46,0x47,0x22,0x46,0x22,0x23,0x33,0x46,0x23,0x33,0x23
,0x24,0x32,0x33,0x24,0x32,0x24,0x25,0x32,0x25,0x26,0x27,0x28,0x2A,0x27,0x2A
,0x32,0x27,0x32,0x26,0x47,0x49,0x20,0x1E,0x01,0x68,0xE9,0x5A,0x3E,0x73,0x68
,0x11,0xBF,0x10,0xAE,0x30,0x3E,0xCC,0x0B,0x10,0xBF,0x39,0xB4,0x08,0x3E,0xC7
,0xF5,0x0B,0xBF,0x69,0x8D,0xE1,0x3D,0xD9,0x79,0x07,0xBF,0x8A,0x91,0xC5,0x3D
,0x37,0x33,0x02,0xBF,0x9D,0xF3,0xB3,0x3D,0xD1,0x79,0xED,0xBE,0x7B,0x14,0xAE
,0x3D,0xF0,0xA7,0xC6,0xBE,0x7B,0x14,0xAE,0x3D,0xA6,0x9B,0xC4,0x3E,0x74,0x60
,0xB9,0x3D,0xA8,0xC8,0xF1,0x3E,0x52,0x43,0xDB,0x3D,0x9E,0x43,0x05,0x3F,0x93
,0x1A,0x1A,0x3E,0xA6,0x9A,0x0D,0x3F,0x68,0xE9,0x5A,0x3E,0x4E,0x62,0x10,0x3F
,0x81,0x95,0x83,0x3E,0x4E,0x62,0x10,0x3F,0x81,0x95,0x83,0x3E,0x75,0x93,0x18
,0x3F,0x81,0x95,0x83,0xBE,0x75,0x93,0x18,0x3F,0x81,0x95,0x83,0xBE,0x4E,0x62
,0x10,0x3F,0xA3,0x95,0x5B,0xBE,0x4E,0x62,0x10,0x3F,0x92,0xE9,0x30,0xBE,0xA7
,0x05,0x0F,0x3F,0x62,0x10,0x08,0xBE,0xA2,0xEF,0x0A,0x3F,0x84,0x64,0xE1,0xBD
,0xB4,0x73,0x06,0x3F,0x0C,0xB2,0xC5,0xBD,0x13,0x2D,0x01,0x3F,0x01,0xFC,0xB3
,0xBD,0x87,0x6D,0xEB,0x3E,0x7B,0x14,0xAE,0xBD,0xA6,0x9B,0xC4,0x3E,0x7B,0x14
,0xAE,0xBD,0xF0,0xA7,0xC6,0xBE,0xB1,0xF9,0xB8,0xBD,0xF1,0xD4,0xF3,0xBE,0xDA
,0xA9,0xD9,0xBD,0xC3,0x49,0x06,0xBF,0x97,0xC4,0x19,0xBE,0xCB,0xA0,0x0E,0xBF
,0xA3,0x95,0x5B,0xBE,0x73,0x68,0x11,0xBF,0x81,0x95,0x83,0xBE,0x73,0x68,0x11
,0xBF,0x81,0x95,0x83,0xBE,0x9A,0x99,0x19,0xBF,0x81,0x95,0x83,0x3E,0x9A,0x99
,0x19,0xBF,0x81,0x95,0x83,0x3E,0x73,0x68,0x11,0xBF,0x1E,0x1F,0x00,0x1D,0x1E
,0x00,0x1D,0x00,0x01,0x1D,0x01,0x02,0x1D,0x02,0x03,0x1D,0x03,0x04,0x1B,0x1C
,0x1D,0x0E,0x0F,0x10,0x0D,0x0E,0x10,0x0D,0x10,0x11,0x0D,0x11,0x12,0x0D,0x12
,0x13,0x0D,0x13,0x14,0x0B,0x0C,0x0D,0x0D,0x14,0x15,0x1D,0x04,0x05,0x1A,0x1B
,0x1D,0x1A,0x1D,0x05,0x19,0x1A,0x05,0x18,0x19,0x05,0x17,0x18,0x05,0x17,0x05
,0x06,0x16,0x17,0x06,0x16,0x06,0x07,0x15,0x16,0x07,0x15,0x07,0x08,0x15,0x08
,0x09,0x0A,0x0B,0x0D,0x0A,0x0D,0x15,0x0A,0x15,0x09,0x1F,0x4A,0x29,0x27,0x01
,0x54,0x3B,0x13,0xBE,0x4E,0x62,0x10,0x3F,0xF8,0xFF,0xD1,0xBD,0xA7,0x05,0x0F
,0x3F,0x4A,0x0C,0x82,0xBD,0xA2,0xEF,0x0A,0x3F,0x80,0x62,0x24,0xBD,0xB4,0x73
,0x06,0x3F,0x84,0xD5,0xD8,0xBC,0x13,0x2D,0x01,0x3F,0xD1,0x5D,0x92,0xBC,0x87
,0x6D,0xEB,0x3E,0x8F,0xC2,0x75,0xBC,0xA6,0x9B,0xC4,0x3E,0x8F,0xC2,0x75,0xBC
,0xA8,0x1A,0xDD,0xBE,0xF6,0xED,0xA4,0xBC,0xEA,0x3D,0xFD,0xBE,0xF4,0x88,0x11
,0xBD,0xA6,0xF0,0x08,0xBF,0x7A,0xC4,0x68,0xBD,0xEC,0x85,0x0E,0xBF,0xAF,0x22
,0xA3,0xBD,0x4E,0x62,0x10,0xBF,0x6E,0xC2,0xDD,0xBD,0xEC,0x50,0x0D,0xBF,0x46
,0x5E,0x06,0xBE,0xA4,0x1C,0x04,0xBF,0x70,0x41,0x26,0xBE,0x5B,0xD1,0xE6,0xBE
,0xA4,0x18,0x40,0xBE,0xF8,0x51,0xD5,0xBE,0x3E,0x5E,0x58,0xBE,0x74,0x29,0xCE
,0xBE,0x61,0x8D,0x73,0xBE,0xA8,0xC6,0xCB,0xBE,0x68,0x3D,0x8C,0xBE,0xC3,0x48
,0xCF,0xBE,0x68,0x3D,0x9C,0xBE,0xD1,0xCE,0xD9,0xBE,0xC3,0xA0,0xA4,0xBE,0x02
,0xD5,0xE7,0xBE,0x8B,0x6C,0xA7,0xBE,0xA6,0x47,0xFB,0xBE,0x14,0x5B,0xA1,0xBE
,0xB4,0x73,0x0A,0xBF,0xF2,0x26,0x8F,0xBE,0x66,0xBC,0x15,0xBF,0x8B,0xC4,0x64
,0xBE,0x49,0xA0,0x1D,0xBF,0x6C,0x3F,0x19,0xBE,0x89,0x41,0x20,0xBF,0xF1,0x65
,0x62,0xBD,0x6B,0x48,0x1C,0xBF,0xEA,0x25,0xC6,0x3C,0x31,0x5D,0x10,0xBF,0x12
,0x2F,0xAF,0x3D,0x90,0x2E,0xFE,0xBE,0xB7,0x9B,0x00,0x3E,0x9B,0x1C,0xD6,0xBE
,0x0A,0x31,0x17,0x3E,0x83,0xBE,0xA4,0xBE,0x52,0xB8,0x1E,0x3E,0xC7,0x9B,0x4C
,0xBE,0x52,0xB8,0x1E,0x3E,0xA6,0x9B,0xC4,0x3E,0xED,0x2A,0x24,0x3E,0xA8,0xC8
,0xF1,0x3E,0x02,0x83,0x34,0x3E,0x9E,0x43,0x05,0x3F,0x43,0x39,0x61,0x3E,0xA6
,0x9A,0x0D,0x3F,0x2B,0x2F,0x91,0x3E,0x4E,0x62,0x10,0x3F,0x8B,0x6C,0xA7,0x3E
,0x4E,0x62,0x10,0x3F,0x8B,0x6C,0xA7,0x3E,0x75,0x93,0x18,0x3F,0xEE,0x7C,0x3F
,0xBE,0x75,0x93,0x18,0x3F,0xEE,0x7C,0x3F,0xBE,0x4E,0x62,0x10,0x3F,0x27,0x28
,0x00,0x26,0x27,0x00,0x26,0x00,0x01,0x26,0x01,0x02,0x26,0x02,0x03,0x26,0x03
,0x04,0x24,0x25,0x26,0x26,0x04,0x05,0x23,0x24,0x26,0x23,0x26,0x05,0x22,0x23
,0x05,0x21,0x22,0x05,0x20,0x21,0x05,0x20,0x05,0x06,0x1F,0x20,0x06,0x1F,0x06
,0x07,0x1E,0x1F,0x07,0x1D,0x1E,0x07,0x1C,0x1D,0x07,0x1C,0x07,0x08,0x1B,0x1C
,0x08,0x1B,0x08,0x09,0x1A,0x1B,0x09,0x1A,0x09,0x0A,0x1A,0x0A,0x0B,0x19,0x1A
,0x0B,0x19,0x0B,0x0C,0x18,0x19,0x0C,0x18,0x0C,0x0D,0x17,0x18,0x0D,0x17,0x0D
,0x0E,0x16,0x17,0x0E,0x15,0x16,0x0E,0x15,0x0E,0x0F,0x14,0x15,0x0F,0x13,0x14
,0x0F,0x12,0x13,0x0F,0x11,0x12,0x0F,0x11,0x0F,0x10,0x28,0x4B,0x49,0x47,0x01
,0xA8,0x51,0x48,0x3D,0xCC,0xF1,0x7A,0x3E,0x9A,0x99,0x69,0x3E,0xFC,0xA9,0xD9
,0x3E,0xEC,0x68,0x94,0x3E,0x94,0x2E,0xF5,0x3E,0xA6,0x99,0xAE,0x3E,0xC2,0xF6
,0x03,0x3F,0xAA,0x45,0xC4,0x3E,0xDF,0xF9,0x09,0x3F,0xEB,0x53,0xD6,0x3E,0xE7
,0xA6,0x0D,0x3F,0x5B,0x7B,0xE7,0x3E,0xD1,0x96,0x0F,0x3F,0xA8,0x72,0xFA,0x3E
,0x4E,0x62,0x10,0x3F,0xA8,0x72,0xFA,0x3E,0x75,0x93,0x18,0x3F,0x27,0x31,0x48
,0x3D,0x75,0x93,0x18,0x3F,0x27,0x31,0x48,0x3D,0x4E,0x62,0x10,0x3F,0xB5,0x18
,0x9C,0x3D,0x4E,0x62,0x10,0x3F,0xD7,0x4D,0xE9,0x3D,0xAA,0x65,0x0F,0x3F,0xED
,0x28,0x0E,0x3E,0x9C,0x6F,0x0C,0x3F,0x52,0x61,0x1C,0x3E,0x5A,0x2D,0x08,0x3F
,0xC9,0x1E,0x21,0x3E,0xC7,0x4B,0x03,0x3F,0x3E,0x06,0x1B,0x3E,0x81,0xEA,0xF7
,0x3E,0x5A,0xBC,0x08,0x3E,0xEB,0xFD,0xE6,0x3E,0x2B,0x4C,0xDF,0x3C,0xF2,0x7D
,0xA9,0x3E,0xC4,0xD0,0x3A,0xBE,0xA2,0x45,0x06,0x3E,0xD1,0x22,0x9B,0xBE,0x0B
,0xED,0x9C,0x3C,0xD1,0x22,0x9B,0xBE,0xA6,0x9B,0xC4,0x3E,0x7F,0xC2,0x99,0xBE
,0xB6,0x48,0xEA,0x3E,0xCB,0xA1,0x95,0xBE,0x91,0x43,0x00,0x3F,0xAA,0xF2,0x8D
,0xBE,0xB2,0x9E,0x06,0x3F,0x75,0xE7,0x81,0xBE,0xFA,0x28,0x0B,0x3F,0xB5,0xC4
,0x5A,0xBE,0xF0,0x13,0x0F,0x3F,0x2D,0x5A,0x30,0xBE,0x4E,0x62,0x10,0x3F,0xDD
,0x24,0x06,0xBE,0x4E,0x62,0x10,0x3F,0xDD,0x24,0x06,0xBE,0x75,0x93,0x18,0x3F
,0xB8,0x1E,0x25,0xBF,0x75,0x93,0x18,0x3F,0xB8,0x1E,0x25,0xBF,0x4E,0x62,0x10
,0x3F,0x50,0x39,0x1A,0xBF,0x4E,0x62,0x10,0x3F,0x5C,0x8E,0x0F,0xBF,0xA7,0x05
,0x0F,0x3F,0x10,0x58,0x05,0xBF,0xA2,0xEF,0x0A,0x3F,0x11,0x01,0xFF,0xBE,0x0E
,0x82,0x06,0x3F,0x72,0x14,0xF8,0xBE,0x6B,0x66,0x01,0x3F,0xF0,0xA6,0xF3,0xBE
,0x3B,0x8A,0xEB,0x3E,0x0E,0x2D,0xF2,0xBE,0xA6,0x9B,0xC4,0x3E,0x0E,0x2D,0xF2
,0xBE,0xF0,0xA7,0xC6,0xBE,0x5C,0xE6,0xF4,0xBE,0xF1,0xD4,0xF3,0xBE,0x66,0x12
,0xFD,0xBE,0xC3,0x49,0x06,0xBF,0x1E,0xC5,0x09,0xBF,0xCB,0xA0,0x0E,0xBF,0x50
,0x39,0x1A,0xBF,0x73,0x68,0x11,0xBF,0xB8,0x1E,0x25,0xBF,0x73,0x68,0x11,0xBF
,0xB8,0x1E,0x25,0xBF,0x9A,0x99,0x19,0xBF,0xDD,0x24,0x06,0xBE,0x9A,0x99,0x19
,0xBF,0xDD,0x24,0x06,0xBE,0x73,0x68,0x11,0xBF,0x2D,0x5A,0x30,0xBE,0x73,0x68
,0x11,0xBF,0x75,0x3B,0x5B,0xBE,0xCC,0x0B,0x10,0xBF,0x75,0xE7,0x81,0xBE,0xC7
,0xF5,0x0B,0xBF,0x6B,0x10,0x8E,0xBE,0xD9,0x79,0x07,0xBF,0x0A,0x2B,0x95,0xBE
,0x37,0x33,0x02,0xBF,0xDF,0xA4,0x99,0xBE,0xD1,0x79,0xED,0xBE,0xD1,0x22,0x9B
,0xBE,0xF0,0xA7,0xC6,0xBE,0xD1,0x22,0x9B,0xBE,0x6F,0x12,0x03,0xBB,0xDB,0xF9
,0xDE,0x3D,0x11,0xAC,0xD2,0xBE,0x27,0x89,0x15,0x3E,0x6C,0x7C,0xE6,0xBE,0x9E
,0xEF,0x27,0x3E,0xFE,0xD2,0xF2,0xBE,0x24,0x5E,0x2E,0x3E,0x72,0xFD,0xFB,0xBE
,0x12,0x83,0x30,0x3E,0xD9,0x24,0x03,0xBF,0x90,0xC1,0x2A,0x3E,0x72,0x52,0x08
,0xBF,0x07,0x7D,0x19,0x3E,0xF4,0xFD,0x0C,0xBF,0x57,0x3F,0xF6,0x3D,0xDB,0x4D
,0x10,0xBF,0xF6,0x28,0x9C,0x3D,0x73,0x68,0x11,0xBF,0xF6,0x28,0x9C,0x3D,0x9A
,0x99,0x19,0xBF,0xB8,0x1E,0x25,0x3F,0x9A,0x99,0x19,0xBF,0xB8,0x1E,0x25,0x3F
,0x73,0x68,0x11,0xBF,0xE4,0x84,0x11,0x3F,0xBE,0xDE,0x0D,0xBF,0xF1,0xD4,0xFB
,0x3E,0xDF,0xF9,0x05,0xBF,0xE2,0x91,0xD0,0x3E,0xC1,0xC9,0xEE,0xBE,0x2B,0x33
,0x9D,0x3E,0x92,0x95,0xBF,0xBE,0x67,0x0E,0x09,0xBE,0xF8,0x53,0x83,0x3D,0x41
,0x42,0x43,0x41,0x43,0x44,0x41,0x44,0x45,0x41,0x45,0x46,0x2E,0x2F,0x30,0x2D
,0x2E,0x30,0x2D,0x30,0x31,0x2D,0x31,0x32,0x2D,0x32,0x33,0x2D,0x33,0x34,0x2B
,0x2C,0x2D,0x1E,0x1F,0x20,0x1D,0x1E,0x20,0x1D,0x20,0x21,0x1D,0x21,0x22,0x1D
,0x22,0x23,0x1D,0x23,0x24,0x1B,0x1C,0x1D,0x09,0x0A,0x0B,0x08,0x09,0x0B,0x08
,0x0B,0x0C,0x08,0x0C,0x0D,0x08,0x0D,0x0E,0x08,0x0E,0x0F,0x2D,0x34,0x35,0x1D
,0x24,0x25,0x06,0x07,0x08,0x06,0x08,0x0F,0x05,0x06,0x0F,0x04,0x05,0x0F,0x03
,0x04,0x0F,0x02,0x03,0x0F,0x01,0x02,0x0F,0x01,0x0F,0x10,0x01,0x10,0x11,0x00
,0x01,0x11,0x00,0x11,0x12,0x00,0x12,0x13,0x48,0x00,0x13,0x48,0x13,0x14,0x3F
,0x40,0x41,0x3F,0x41,0x46,0x3E,0x3F,0x46,0x3D,0x3E,0x46,0x3D,0x46,0x47,0x3C
,0x3D,0x47,0x3B,0x3C,0x47,0x3A,0x3B,0x47,0x3A,0x47,0x48,0x39,0x3A,0x48,0x38
,0x39,0x48,0x37,0x38,0x48,0x37,0x48,0x14,0x1A,0x1B,0x1D,0x1A,0x1D,0x25,0x19
,0x1A,0x25,0x18,0x19,0x25,0x17,0x18,0x25,0x16,0x17,0x25,0x16,0x25,0x26,0x15
,0x16,0x26,0x14,0x15,0x26,0x14,0x26,0x27,0x37,0x14,0x27,0x36,0x37,0x27,0x35
,0x36,0x27,0x35,0x27,0x28,0x35,0x28,0x29,0x2A,0x2B,0x2D,0x2A,0x2D,0x35,0x2A
,0x35,0x29,0x48,0x4C,0x26,0x24,0x01,0xD8,0x65,0xE0,0x3E,0x14,0x59,0xBB,0xBE
,0xEC,0xFB,0xC8,0x3E,0x8D,0x43,0xDD,0xBE,0xEF,0x54,0xB0,0x3E,0x9A,0x97,0xF3
,0xBE,0x44,0x35,0x95,0x3E,0xCB,0xF7,0x00,0xBF,0x87,0xC1,0x5C,0x3E,0x94,0xD9
,0x04,0xBF,0xB9,0x6E,0xCA,0x3D,0xDD,0x24,0x06,0xBF,0x18,0x05,0x41,0x3C,0xDD
,0x24,0x06,0xBF,0x17,0x0E,0xA4,0xBD,0xF7,0x93,0x05,0xBF,0x80,0x66,0x00,0xBE
,0x43,0xE1,0x03,0xBF,0x0D,0xFD,0x13,0xBE,0x7A,0xFE,0x00,0xBF,0x3D,0xBA,0x21
,0xBE,0x5E,0xBA,0xF9,0xBE,0x12,0x87,0x2C,0xBE,0xF2,0xD2,0xE5,0xBE,0xC5,0x20
,0x30,0xBE,0xDD,0x24,0xC6,0xBE,0xC5,0x20,0x30,0xBE,0x5A,0x64,0xBB,0x3E,0x69
,0x8E,0x2C,0xBE,0xA7,0xAF,0xE7,0x3E,0x13,0xD7,0x21,0xBE,0x15,0x03,0x00,0x3F
,0x83,0x68,0x0D,0xBE,0xF7,0x92,0x06,0x3F,0x74,0x60,0xD9,0xBD,0x60,0xE6,0x0B
,0x3F,0x5F,0xB6,0x5D,0xBD,0x07,0x60,0x0F,0x3F,0xF4,0xFD,0xD4,0x3C,0x4E,0x62
,0x10,0x3F,0xF4,0xFD,0xD4,0x3C,0x75,0x93,0x18,0x3F,0x81,0x95,0x03,0xBF,0x75
,0x93,0x18,0x3F,0x81,0x95,0x03,0xBF,0x4E,0x62,0x10,0x3F,0xC5,0x20,0xF0,0xBE
,0x4E,0x62,0x10,0x3F,0xEC,0xA5,0xD1,0xBE,0x54,0xE3,0x0D,0x3F,0x88,0xBE,0xBB
,0xBE,0x66,0x66,0x06,0x3F,0x9A,0x43,0xB2,0xBE,0xFE,0x7C,0xF3,0x3E,0xA0,0x1A
,0xAF,0xBE,0xA6,0x9B,0xC4,0x3E,0xA0,0x1A,0xAF,0xBE,0xF0,0xA7,0xC6,0xBE,0xE1
,0x79,0xB1,0xBE,0x64,0x92,0xF1,0xBE,0x85,0x97,0xB8,0xBE,0x94,0xC2,0x04,0xBF
,0xFA,0xD3,0xCE,0xBE,0xF3,0x3E,0x0E,0xBF,0xC5,0x20,0xF0,0xBE,0x73,0x68,0x11
,0xBF,0x81,0x95,0x03,0xBF,0x73,0x68,0x11,0xBF,0x81,0x95,0x03,0xBF,0x9A,0x99
,0x19,0xBF,0xAA,0xF1,0xD2,0x3E,0x9A,0x99,0x19,0xBF,0x81,0x95,0x03,0x3F,0x52
,0x64,0x8D,0xBE,0xD9,0xCE,0xF7,0x3E,0x71,0x3D,0x8A,0xBE,0x23,0x24,0x25,0x23
,0x25,0x00,0x23,0x00,0x01,0x23,0x01,0x02,0x21,0x22,0x23,0x20,0x21,0x23,0x1F
,0x20,0x23,0x1E,0x1F,0x23,0x15,0x16,0x17,0x14,0x15,0x17,0x14,0x17,0x18,0x14
,0x18,0x19,0x14,0x19,0x1A,0x23,0x02,0x03,0x12,0x13,0x14,0x23,0x03,0x04,0x23
,0x04,0x05,0x23,0x05,0x06,0x1E,0x23,0x06,0x1E,0x06,0x07,0x1E,0x07,0x08,0x1E
,0x08,0x09,0x1E,0x09,0x0A,0x1D,0x1E,0x0A,0x1D,0x0A,0x0B,0x1C,0x1D,0x0B,0x1C
,0x0B,0x0C,0x1B,0x1C,0x0C,0x1B,0x0C,0x0D,0x1B,0x0D,0x0E,0x1A,0x1B,0x0E,0x1A
,0x0E,0x0F,0x1A,0x0F,0x10,0x1A,0x10,0x11,0x14,0x1A,0x11,0x12,0x14,0x11,0x25
,0x4D,0x32,0x30,0x01,0xB6,0xF3,0xFD,0xBC,0x9A,0x99,0x19,0xBF,0xE3,0xA5,0xDB
,0x3E,0x62,0x10,0xD0,0x3E,0xE3,0xA5,0xDB,0x3E,0x39,0xB4,0xC8,0xBE,0xB4,0x1F
,0xD9,0x3E,0x89,0x42,0xF3,0xBE,0x47,0x8D,0xD1,0x3E,0x71,0x93,0x05,0xBF,0x3B
,0x8A,0xBB,0x3E,0x2E,0x73,0x0E,0xBF,0x9A,0x99,0x99,0x3E,0x73,0x68,0x11,0xBF
,0x81,0x95,0x83,0x3E,0x73,0x68,0x11,0xBF,0x81,0x95,0x83,0x3E,0x9A,0x99,0x19
,0xBF,0x42,0x60,0x45,0x3F,0x9A,0x99,0x19,0xBF,0x42,0x60,0x45,0x3F,0x73,0x68
,0x11,0xBF,0x95,0x99,0x3A,0x3F,0x73,0x68,0x11,0xBF,0x45,0xB7,0x2A,0x3F,0x9B
,0x04,0x0F,0xBF,0xA0,0xC4,0x1F,0x3F,0x12,0xD9,0x07,0xBF,0x2C,0xF3,0x1A,0x3F
,0x75,0xE6,0xF6,0xBE,0x10,0x58,0x19,0x3F,0x39,0xB4,0xC8,0xBE,0x10,0x58,0x19
,0x3F,0xF0,0xA7,0xC6,0x3E,0x28,0x9B,0x1A,0x3F,0x3F,0x36,0xF1,0x3E,0x5E,0x64
,0x1E,0x3F,0x4C,0x8D,0x04,0x3F,0x75,0x92,0x29,0x3F,0x09,0x6D,0x0D,0x3F,0x95
,0x99,0x3A,0x3F,0x4E,0x62,0x10,0x3F,0x42,0x60,0x45,0x3F,0x4E,0x62,0x10,0x3F
,0x42,0x60,0x45,0x3F,0x75,0x93,0x18,0x3F,0x48,0x35,0xDC,0x3E,0x75,0x93,0x18
,0x3F,0x31,0x08,0x2C,0x3B,0x1D,0xAE,0xAD,0xBE,0xCF,0xA3,0xDA,0xBE,0x75,0x93
,0x18,0x3F,0x42,0x60,0x45,0xBF,0x75,0x93,0x18,0x3F,0x42,0x60,0x45,0xBF,0x4E
,0x62,0x10,0x3F,0xBA,0xF3,0x34,0xBF,0x67,0x46,0x0F,0x3F,0x7F,0xBE,0x29,0xBF
,0xB2,0xF2,0x0B,0x3F,0x78,0x7D,0x22,0xBF,0x9C,0xF8,0x06,0x3F,0x8D,0xED,0x1D
,0xBF,0x81,0xE9,0x00,0x3F,0x78,0x7D,0x1A,0xBF,0x9E,0xED,0xE9,0x3E,0x10,0x58
,0x19,0xBF,0xF0,0xA7,0xC6,0x3E,0x10,0x58,0x19,0xBF,0x39,0xB4,0xC8,0xBE,0x28
,0x9B,0x1A,0xBF,0x89,0x42,0xF3,0xBE,0x5E,0x64,0x1E,0xBF,0x71,0x93,0x05,0xBF
,0x94,0x83,0x29,0xBF,0x2E,0x73,0x0E,0xBF,0x35,0x5E,0x3A,0xBF,0x73,0x68,0x11
,0xBF,0x42,0x60,0x45,0xBF,0x73,0x68,0x11,0xBF,0x42,0x60,0x45,0xBF,0x9A,0x99
,0x19,0xBF,0x21,0xB0,0xB2,0xBE,0x9A,0x99,0x19,0xBF,0x21,0xB0,0xB2,0xBE,0x73
,0x68,0x11,0xBF,0x39,0xB4,0xC8,0xBE,0x73,0x68,0x11,0xBF,0xC0,0x76,0xE8,0xBE
,0x9B,0x04,0x0F,0xBF,0x46,0x5E,0xFE,0xBE,0x12,0xD9,0x07,0xBF,0xF6,0xD3,0x03
,0xBF,0x75,0xE6,0xF6,0xBE,0x42,0x60,0x05,0xBF,0x39,0xB4,0xC8,0xBE,0x42,0x60
,0x05,0xBF,0x62,0x10,0xD0,0x3E,0x23,0xDB,0x79,0xBD,0x9A,0x99,0x19,0xBF,0x29
,0x2A,0x2B,0x28,0x29,0x2B,0x28,0x2B,0x2C,0x28,0x2C,0x2D,0x26,0x27,0x28,0x26
,0x28,0x2D,0x25,0x26,0x2D,0x24,0x25,0x2D,0x24,0x2D,0x2E,0x23,0x24,0x2E,0x23
,0x2E,0x2F,0x22,0x23,0x2F,0x21,0x22,0x2F,0x21,0x2F,0x30,0x20,0x21,0x30,0x19
,0x1A,0x1B,0x19,0x1B,0x1C,0x19,0x1C,0x1D,0x19,0x1D,0x1E,0x19,0x1E,0x1F,0x15
,0x16,0x17,0x14,0x15,0x17,0x13,0x14,0x17,0x12,0x13,0x17,0x09,0x0A,0x0B,0x08
,0x09,0x0B,0x08,0x0B,0x0C,0x08,0x0C,0x0D,0x08,0x0D,0x0E,0x06,0x07,0x08,0x11
,0x12,0x17,0x19,0x1F,0x20,0x19,0x20,0x30,0x18,0x19,0x30,0x18,0x30,0x31,0x18
,0x31,0x00,0x18,0x00,0x01,0x17,0x18,0x01,0x11,0x17,0x01,0x10,0x11,0x01,0x10
,0x01,0x02,0x0F,0x10,0x02,0x0F,0x02,0x03,0x0E,0x0F,0x03,0x0E,0x03,0x04,0x0E
,0x04,0x05,0x08,0x0E,0x05,0x08,0x05,0x06,0x31,0x4E,0x29,0x27,0x01,0x93,0x19
,0x27,0xBF,0x4E,0x62,0x10,0x3F,0x53,0x79,0x1B,0xBF,0xCE,0xE0,0x0F,0x3F,0x71
,0x3C,0x13,0xBF,0x2D,0x5C,0x0E,0x3F,0x2B,0x86,0x07,0xBF,0x7D,0xE9,0x09,0x3F
,0x52,0x0E,0xFE,0xBE,0xC0,0x20,0x05,0x3F,0x6F,0x10,0xED,0xBE,0x6B,0x0F,0xFB
,0x3E,0xE4,0xA3,0xD5,0xBE,0xAE,0x47,0xE1,0x3E,0xE4,0xA3,0xD5,0xBE,0x39,0xB4
,0xC8,0xBE,0xF1,0x29,0xD8,0xBE,0x89,0x42,0xF3,0xBE,0x5E,0xBC,0xDF,0xBE,0x71
,0x93,0x05,0xBF,0x8B,0x18,0xF6,0xBE,0x2E,0x73,0x0E,0xBF,0x77,0x13,0x0C,0xBF
,0x73,0x68,0x11,0xBF,0x23,0xDA,0x16,0xBF,0x73,0x68,0x11,0xBF,0x23,0xDA,0x16
,0xBF,0x9A,0x99,0x19,0xBF,0xC7,0x47,0x2B,0xBE,0x9A,0x99,0x19,0xBF,0xC7,0x47
,0x2B,0xBE,0x73,0x68,0x11,0xBF,0xF8,0x4F,0x57,0xBE,0x73,0x68,0x11,0xBF,0x1A
,0x31,0x8B,0xBE,0x9B,0x04,0x0F,0xBF,0xE7,0x51,0xA1,0xBE,0x12,0xD9,0x07,0xBF
,0xAE,0x9B,0xAA,0xBE,0x75,0xE6,0xF6,0xBE,0x46,0xB4,0xAD,0xBE,0x39,0xB4,0xC8
,0xBE,0x46,0xB4,0xAD,0xBE,0xA0,0x6E,0xB0,0x3E,0xC9,0xCC,0xE5,0x3E,0x64,0x91
,0x1E,0xBF,0xC4,0x22,0xF6,0x3E,0x64,0x91,0x1E,0xBF,0xC4,0x22,0xF6,0x3E,0xF0
,0xA7,0xC6,0x3E,0xF4,0xA8,0xF8,0x3E,0x3F,0x36,0xF1,0x3E,0xB0,0x1D,0x00,0x3F
,0x4C,0x8D,0x04,0x3F,0xE6,0x3C,0x0B,0x3F,0x09,0x6D,0x0D,0x3F,0x87,0x17,0x1C
,0x3F,0x4E,0x62,0x10,0x3F,0x93,0x19,0x27,0x3F,0x4E,0x62,0x10,0x3F,0x93,0x19
,0x27,0x3F,0x75,0x93,0x18,0x3F,0x89,0x45,0x6C,0x3E,0x75,0x93,0x18,0x3F,0x89
,0x45,0x6C,0x3E,0x4E,0x62,0x10,0x3F,0x1D,0xB0,0x8B,0x3E,0x4E,0x62,0x10,0x3F
,0xBC,0x74,0xAB,0x3E,0x76,0xFE,0x0D,0x3F,0x29,0x5A,0xC1,0x3E,0xED,0xD2,0x06
,0x3F,0xF0,0xFC,0xCA,0x3E,0x2B,0xDA,0xF4,0x3E,0x27,0x33,0xCE,0x3E,0xF0,0xA7
,0xC6,0x3E,0x27,0x33,0xCE,0x3E,0x0E,0x2D,0x9A,0xBE,0x5E,0x10,0xA9,0xBE,0x75
,0x93,0x18,0x3F,0x93,0x19,0x27,0xBF,0x75,0x93,0x18,0x3F,0x27,0x28,0x00,0x27
,0x00,0x01,0x27,0x01,0x02,0x27,0x02,0x03,0x27,0x03,0x04,0x27,0x04,0x05,0x27
,0x05,0x06,0x26,0x27,0x06,0x1F,0x20,0x21,0x1E,0x1F,0x21,0x1E,0x21,0x22,0x1E
,0x22,0x23,0x1C,0x1D,0x1E,0x1C,0x1E,0x23,0x1B,0x1C,0x23,0x1A,0x1B,0x23,0x1A
,0x23,0x24,0x19,0x1A,0x24,0x19,0x24,0x25,0x18,0x19,0x25,0x17,0x18,0x25,0x17
,0x25,0x26,0x16,0x17,0x26,0x15,0x16,0x26,0x15,0x26,0x06,0x15,0x06,0x07,0x14
,0x15,0x07,0x0E,0x0F,0x10,0x0D,0x0E,0x10,0x0D,0x10,0x11,0x0D,0x11,0x12,0x0B
,0x0C,0x0D,0x0B,0x0D,0x12,0x0A,0x0B,0x12,0x09,0x0A,0x12,0x09,0x12,0x13,0x08
,0x09,0x13,0x08,0x13,0x14,0x07,0x08,0x14,0x28,0x4F,0x20,0x20,0x02,0xCA,0xF9
,0x52,0xBE,0x18,0x43,0x15,0x3F,0x9E,0xEF,0xC7,0xBE,0x42,0xB4,0xEE,0x3E,0xB8
,0x1E,0x09,0xBF,0x42,0x60,0x85,0x3E,0x06,0x81,0x15,0xBF,0xFB,0xAD,0x1D,0x3B
,0xEB,0xC7,0x0A,0xBF,0xEC,0xF8,0x7F,0xBE,0x54,0x39,0xD5,0xBE,0xFE,0xD6,0xE6
,0xBE,0x6F,0xB9,0x6A,0xBE,0x0B,0x0C,0x15,0xBF,0x31,0x08,0x2C,0xBB,0x89,0x41
,0x20,0xBF,0xFC,0x51,0x64,0x3E,0xD2,0xC3,0x14,0xBF,0x33,0x33,0xD3,0x3E,0x7D
,0x95,0xE4,0xBE,0x2B,0x87,0x0A,0x3F,0x11,0xA9,0x79,0xBE,0x06,0x81,0x15,0x3F
,0x16,0xDB,0xA4,0x3B,0xAA,0x9C,0x0A,0x3F,0xA8,0x1A,0x7D,0x3E,0x2B,0xDF,0xD3
,0x3E,0xA4,0x1C,0xE4,0x3E,0x9A,0x99,0x69,0x3E,0x24,0xF0,0x13,0x3F,0x41,0x62
,0x3B,0x3C,0x64,0x3B,0x1F,0x3F,0xE5,0x2A,0x16,0x3E,0x14,0xAE,0x07,0x3F,0xF8
,0xFB,0x85,0x3E,0xAE,0x47,0xE1,0x3E,0x1F,0x2F,0xB4,0x3E,0x49,0x0E,0x80,0x3E
,0x81,0x95,0xC3,0x3E,0x98,0x6E,0x92,0xBC,0x0B,0x97,0xB5,0x3E,0xDC,0xBB,0x86
,0xBE,0xAA,0x9B,0x8B,0x3E,0xCD,0x76,0xDD,0xBE,0xC6,0x4E,0x18,0x3E,0x98,0xF8
,0x07,0xBF,0x35,0x60,0x90,0xBB,0x4E,0x62,0x10,0xBF,0x88,0xBC,0x15,0xBE,0x39
,0xB4,0x08,0xBF,0x88,0xBE,0x83,0xBE,0xF8,0x53,0xE3,0xBE,0xBA,0x9F,0xB3,0xBE
,0x3C,0x33,0x81,0xBE,0x81,0x95,0xC3,0xBE,0x96,0x3F,0xDF,0x3B,0x1F,0x2F,0xB4
,0xBE,0x5C,0x3C,0x84,0x3E,0xF8,0xFB,0x85,0xBE,0x6E,0xBE,0xE1,0x3E,0x37,0xDF
,0x18,0xBE,0xF5,0xBC,0x07,0x3F,0x3D,0x0C,0xAD,0xBB,0x29,0x5C,0x0F,0x3F,0x01
,0x1B,0x1C,0x1B,0x01,0x02,0x01,0x1C,0x1D,0x1B,0x02,0x03,0x00,0x01,0x1D,0x00
,0x1D,0x1E,0x0F,0x00,0x1E,0x0F,0x1E,0x1F,0x1B,0x03,0x04,0x1A,0x1B,0x04,0x1A
,0x04,0x05,0x19,0x1A,0x05,0x19,0x05,0x06,0x18,0x19,0x06,0x18,0x06,0x07,0x17
,0x18,0x07,0x17,0x07,0x08,0x16,0x17,0x08,0x15,0x16,0x08,0x15,0x08,0x09,0x14
,0x15,0x09,0x14,0x09,0x0A,0x13,0x14,0x0A,0x13,0x0A,0x0B,0x0F,0x1F,0x10,0x0E
,0x0F,0x10,0x0E,0x10,0x11,0x0D,0x0E,0x11,0x0D,0x11,0x12,0x0C,0x0D,0x12,0x0C
,0x12,0x13,0x0B,0x0C,0x13,0x0F,0x1F,0x50,0x34,0x34,0x02,0xF4,0xF9,0x68,0xBD
,0x48,0xC4,0x54,0xBD,0x69,0x8E,0xAC,0xBB,0x6A,0xBC,0x74,0xBD,0x62,0x81,0x2F
,0x3D,0x12,0xF8,0x83,0xBD,0xF2,0x22,0xB3,0x3D,0x02,0x2B,0x87,0xBD,0xA4,0x71
,0x78,0x3E,0xBD,0x35,0x30,0xBD,0x62,0x66,0xB7,0x3E,0x60,0x5A,0xD4,0x3C,0xA5
,0x2F,0xDC,0x3E,0x35,0xB3,0x06,0x3E,0xB0,0x72,0xE8,0x3E,0xAC,0x74,0x87,0x3E
,0x21,0x59,0xE0,0x3E,0x14,0xED,0xBA,0x3E,0x52,0x0C,0xC8,0x3E,0xA2,0xEF,0xE6
,0x3E,0x28,0xEF,0xA3,0x3E,0xAB,0xB1,0x04,0x3F,0xCD,0xC8,0x70,0x3E,0x87,0x17
,0x10,0x3F,0xBF,0xF2,0x00,0x3E,0x7E,0x74,0x16,0x3F,0xBA,0x49,0x8C,0xBC,0x75
,0x93,0x18,0x3F,0xB0,0x72,0xE8,0xBE,0x75,0x93,0x18,0x3F,0xB0,0x72,0xE8,0xBE
,0x4E,0x62,0x10,0x3F,0xBE,0x4B,0xD1,0xBE,0x4E,0x62,0x10,0x3F,0xF4,0xA6,0xB2
,0xBE,0x54,0xE3,0x0D,0x3F,0xE8,0xA3,0x9C,0xBE,0x66,0x66,0x06,0x3F,0xA4,0x72
,0x93,0xBE,0x46,0xB6,0xF3,0x3E,0x4E,0x62,0x90,0xBE,0xA6,0x9B,0xC4,0x3E,0x4E
,0x62,0x90,0xBE,0xF0,0xA7,0xC6,0xBE,0xF4,0xA6,0x92,0xBE,0x08,0xAB,0xF1,0xBE
,0xC5,0x74,0x99,0xBE,0x31,0xB2,0x04,0xBF,0xF2,0xD0,0xAF,0xBE,0xE2,0x3A,0x0E
,0xBF,0xBE,0x4B,0xD1,0xBE,0x73,0x68,0x11,0xBF,0xB0,0x72,0xE8,0xBE,0x73,0x68
,0x11,0xBF,0xB0,0x72,0xE8,0xBE,0x9A,0x99,0x19,0xBF,0x8F,0xC2,0x75,0x3D,0x9A
,0x99,0x19,0xBF,0x8F,0xC2,0x75,0x3D,0x73,0x68,0x11,0xBF,0x8B,0x6C,0x67,0x3C
,0x73,0x68,0x11,0xBF,0x01,0x16,0x39,0xBD,0x79,0xE9,0x0E,0xBF,0x28,0xD1,0xB2
,0xBD,0x8B,0x6C,0x07,0xBF,0x5A,0xBC,0xD8,0xBD,0x8F,0xC2,0xF5,0xBE,0x42,0x60
,0xE5,0xBD,0xF0,0xA7,0xC6,0xBE,0x42,0x60,0xE5,0xBD,0x1F,0x10,0x28,0xBD,0x42
,0x60,0xE5,0xBD,0x23,0xDB,0x01,0x3F,0x5D,0xDF,0x47,0xBD,0x77,0x12,0x05,0x3F
,0xA1,0x80,0xED,0x3A,0xDD,0x24,0x06,0x3F,0xD1,0x24,0x91,0x3D,0xF2,0x06,0x04
,0x3F,0x95,0x47,0x07,0x3E,0x21,0x5A,0xFB,0x3E,0xF4,0x50,0x3B,0x3E,0xC1,0x1F
,0xE6,0x3E,0x29,0x04,0x62,0x3E,0xD3,0x4F,0xC8,0x3E,0x52,0x0A,0x7A,0x3E,0x3A
,0xCB,0xA4,0x3E,0x25,0x06,0x81,0x3E,0x71,0xE5,0x7C,0x3E,0xA7,0x41,0x71,0x3E
,0xF4,0xFA,0x13,0x3E,0x37,0xE1,0x3E,0x3E,0x88,0x0E,0x81,0x3D,0x83,0xC0,0xEA
,0x3D,0x3E,0xCF,0x1F,0x3C,0x20,0x9A,0xF9,0x3C,0x6F,0x12,0x03,0xBC,0xB9,0x1C
,0x2F,0x3B,0x74,0xEF,0xE1,0xBB,0xFB,0x93,0xF8,0xBC,0xE1,0x7C,0x6A,0xBB,0x42
,0x60,0xE5,0xBD,0x33,0x34,0x1E,0x3C,0x01,0x30,0x31,0x30,0x01,0x02,0x01,0x31
,0x32,0x00,0x01,0x32,0x00,0x32,0x33,0x23,0x00,0x33,0x2F,0x30,0x02,0x2F,0x02
,0x03,0x2F,0x03,0x04,0x2E,0x2F,0x04,0x2E,0x04,0x05,0x2D,0x2E,0x05,0x2D,0x05
,0x06,0x2C,0x2D,0x06,0x2C,0x06,0x07,0x2C,0x07,0x08,0x2B,0x2C,0x08,0x2B,0x08
,0x09,0x2A,0x2B,0x09,0x2A,0x09,0x0A,0x29,0x2A,0x0A,0x29,0x0A,0x0B,0x28,0x29
,0x0B,0x28,0x0B,0x0C,0x27,0x28,0x0C,0x27,0x0C,0x0D,0x26,0x27,0x0D,0x25,0x26
,0x0D,0x25,0x0D,0x0E,0x24,0x25,0x0E,0x1C,0x1D,0x1E,0x1B,0x1C,0x1E,0x1B,0x1E
,0x1F,0x1B,0x1F,0x20,0x1B,0x20,0x21,0x19,0x1A,0x1B,0x0E,0x0F,0x10,0x0E,0x10
,0x11,0x24,0x0E,0x11,0x24,0x11,0x12,0x24,0x12,0x13,0x33,0x24,0x13,0x33,0x13
,0x14,0x33,0x14,0x15,0x23,0x33,0x15,0x22,0x23,0x15,0x22,0x15,0x16,0x21,0x22
,0x16,0x21,0x16,0x17,0x21,0x17,0x18,0x1B,0x21,0x18,0x1B,0x18,0x19,0x23,0x33
,0x51,0x2F,0x2F,0x02,0x46,0x0A,0xA5,0x3E,0x71,0x3C,0x07,0xBF,0xF4,0xFD,0xEC
,0x3E,0xA6,0x43,0xC7,0xBE,0xAE,0xF1,0x0D,0x3F,0x3D,0x0A,0x57,0xBE,0x8F,0xC2
,0x15,0x3F,0xF3,0x01,0x81,0xBB,0x08,0xCD,0x0A,0x3F,0xE4,0x4E,0x79,0x3E,0x23
,0xD9,0xD3,0x3E,0x06,0x83,0xE3,0x3E,0x85,0x3E,0x68,0x3E,0xF0,0xDC,0x13,0x3F
,0x4C,0x37,0x09,0x3C,0x64,0x3B,0x1F,0x3F,0xB8,0x22,0x61,0xBE,0x15,0xE4,0x13
,0x3F,0x63,0xB8,0xD2,0xBE,0x6F,0xBC,0xE3,0x3E,0xFC,0xA8,0x0A,0xBF,0xC1,0xC5
,0x7A,0x3E,0x8F,0xC2,0x15,0xBF,0x66,0xDC,0xD4,0xB9,0x51,0xF7,0x11,0xBF,0x35
,0x08,0x23,0xBE,0x85,0x95,0x06,0xBF,0x44,0x33,0x97,0xBE,0x1D,0xAF,0xE8,0xBE
,0x4A,0x0C,0xD2,0xBE,0x9E,0xEF,0xB7,0xBE,0x94,0x6C,0x01,0xBF,0x83,0x6A,0x8B
,0xBE,0x31,0xB3,0x0F,0xBF,0xE5,0xD0,0x22,0xBE,0x1B,0x2F,0x1D,0xBF,0x25,0x41
,0x78,0xBD,0xA5,0x4F,0x33,0xBF,0x89,0x41,0x60,0x3D,0xD9,0x23,0x48,0xBF,0xC3
,0x9A,0x3A,0x3E,0x3A,0x1F,0x5A,0xBF,0xBA,0x4B,0xA2,0x3E,0x39,0xB5,0x67,0xBF
,0x97,0x01,0xE7,0x3E,0xEC,0x87,0x70,0xBF,0x46,0xB6,0x13,0x3F,0x58,0x39,0x74
,0xBF,0x46,0xB6,0x13,0x3F,0x56,0x0E,0x6D,0xBF,0x95,0xEE,0xE6,0x3E,0x92,0x58
,0x66,0xBF,0xA0,0xC6,0xAD,0x3E,0xBA,0x9F,0x57,0xBF,0xB2,0x4B,0x74,0x3E,0x6A
,0xA6,0x3F,0xBF,0xE1,0x7A,0x14,0x3E,0x1B,0x2F,0x1D,0xBF,0x69,0x6E,0xC5,0x3D
,0x38,0x69,0x0A,0x3F,0xA8,0x1A,0x3D,0x3E,0x0A,0x2C,0x00,0x3F,0xC6,0xE1,0x84
,0x3E,0x1A,0x6D,0xDD,0x3E,0x93,0xC4,0xA2,0x3E,0xA6,0x47,0xAB,0x3E,0xA4,0xC3
,0xBB,0x3E,0x42,0x5E,0x2F,0x3E,0x93,0x18,0xC4,0x3E,0xD5,0x79,0xD4,0xBC,0x5C
,0xE6,0xB4,0x3E,0x0A,0x83,0x8A,0xBE,0xD7,0x4F,0x87,0x3E,0xF4,0xFD,0xE4,0xBE
,0xB6,0x9D,0x16,0x3E,0x1D,0xAE,0x09,0xBF,0xC5,0x53,0x0F,0xBB,0x73,0x68,0x11
,0xBF,0xAF,0x43,0x15,0xBE,0x1D,0xAE,0x09,0xBF,0xF6,0x28,0x84,0xBE,0xF4,0xFD
,0xE4,0xBE,0xA4,0x1C,0xB4,0xBE,0xD3,0xA1,0x83,0xBE,0x93,0x18,0xC4,0xBE,0xA6
,0x9B,0xC4,0xB9,0x0C,0x56,0xB4,0xBE,0x79,0xE8,0x83,0x3E,0x56,0x0E,0x85,0xBE
,0xD3,0xF9,0xE0,0x3E,0x0F,0x29,0x16,0xBE,0x78,0x7D,0x06,0x3F,0xC5,0x53,0x0F
,0xBB,0xF2,0xD2,0x0D,0x3F,0x01,0x22,0x23,0x22,0x01,0x02,0x22,0x02,0x03,0x01
,0x23,0x24,0x00,0x01,0x24,0x1C,0x00,0x24,0x1C,0x24,0x25,0x1C,0x25,0x26,0x16
,0x17,0x18,0x16,0x18,0x19,0x15,0x16,0x19,0x15,0x19,0x1A,0x22,0x03,0x04,0x21
,0x22,0x04,0x21,0x04,0x05,0x20,0x21,0x05,0x1F,0x20,0x05,0x1F,0x05,0x06,0x1E
,0x1F,0x06,0x1E,0x06,0x07,0x1D,0x1E,0x07,0x2E,0x1D,0x07,0x2E,0x07,0x08,0x2D
,0x2E,0x08,0x2C,0x2D,0x08,0x2C,0x08,0x09,0x2B,0x2C,0x09,0x2B,0x09,0x0A,0x2A
,0x2B,0x0A,0x2A,0x0A,0x0B,0x2A,0x0B,0x0C,0x29,0x2A,0x0C,0x29,0x0C,0x0D,0x29
,0x0D,0x0E,0x28,0x29,0x0E,0x28,0x0E,0x0F,0x28,0x0F,0x10,0x27,0x28,0x10,0x27
,0x10,0x11,0x26,0x27,0x11,0x26,0x11,0x12,0x14,0x15,0x1A,0x14,0x1A,0x1B,0x13
,0x14,0x1B,0x13,0x1B,0x1C,0x13,0x1C,0x26,0x13,0x26,0x12,0x1C,0x2E,0x52,0x38
,0x38,0x02,0x1F,0x84,0x18,0x3F,0x73,0x68,0x11,0xBF,0x84,0xD4,0x01,0x3F,0xA9
,0xDC,0x0C,0xBF,0xA0,0xC4,0xDF,0x3E,0x10,0xAE,0x04,0xBF,0x68,0x91,0xBD,0x3E
,0xC4,0x21,0xEB,0xBE,0xA6,0x45,0x95,0x3E,0xB0,0x72,0xB8,0xBE,0x47,0x1C,0x32
,0x3D,0x9C,0xC5,0x8B,0xBC,0x24,0xD6,0x32,0x3E,0x1C,0xCF,0xE7,0x3C,0x2D,0x5A
,0x88,0x3E,0x93,0x8D,0xC7,0x3D,0x55,0xF8,0xA3,0x3E,0x27,0x33,0x3E,0x3E,0x02
,0x2D,0xAD,0x3E,0x98,0xC2,0x93,0x3E,0x8F,0x18,0xA5,0x3E,0xA9,0xDC,0xC4,0x3E
,0x16,0xDB,0x8C,0x3E,0xC7,0xA1,0xEE,0x3E,0x88,0x11,0x52,0x3E,0x89,0x61,0x07
,0x3F,0xFE,0xD4,0xF8,0x3D,0xD5,0x78,0x11,0x3F,0x0C,0x76,0xC3,0x3B,0xD5,0xCC
,0x16,0x3F,0x0F,0x81,0x23,0xBE,0x75,0x93,0x18,0x3F,0x1F,0x84,0x18,0xBF,0x75
,0x93,0x18,0x3F,0x1F,0x84,0x18,0xBF,0x4E,0x62,0x10,0x3F,0x00,0xFF,0x0C,0xBF
,0x4E,0x62,0x10,0x3F,0x06,0x83,0xFB,0xBE,0x54,0xE3,0x0D,0x3F,0xA2,0x9B,0xE5
,0xBE,0x66,0x66,0x06,0x3F,0xB4,0x20,0xDC,0xBE,0x46,0xB6,0xF3,0x3E,0xDB,0xF7
,0xD8,0xBE,0xA6,0x9B,0xC4,0x3E,0xDB,0xF7,0xD8,0xBE,0xF0,0xA7,0xC6,0xBE,0xFC
,0x56,0xDB,0xBE,0x08,0xAB,0xF1,0xBE,0xC1,0x74,0xE2,0xBE,0x31,0xB2,0x04,0xBF
,0x36,0xB1,0xF8,0xBE,0xE2,0x3A,0x0E,0xBF,0x00,0xFF,0x0C,0xBF,0x73,0x68,0x11
,0xBF,0x1F,0x84,0x18,0xBF,0x73,0x68,0x11,0xBF,0x1F,0x84,0x18,0xBF,0x9A,0x99
,0x19,0xBF,0xEF,0x74,0xA7,0xBD,0x9A,0x99,0x19,0xBF,0xEF,0x74,0xA7,0xBD,0x73
,0x68,0x11,0xBF,0xB3,0x99,0x03,0xBE,0x73,0x68,0x11,0xBF,0x04,0xAA,0x3F,0xBE
,0x79,0xE9,0x0E,0xBF,0x6C,0x93,0x6A,0xBE,0x8B,0x6C,0x07,0xBF,0x48,0x89,0x7D
,0xBE,0x8F,0xC2,0xF5,0xBE,0x9E,0xED,0x81,0xBE,0xF0,0xA7,0xC6,0xBE,0x9E,0xED
,0x81,0xBE,0x0A,0xD7,0x23,0xBD,0x8B,0xC0,0x68,0xBE,0x53,0x58,0x29,0xBD,0x57
,0x3D,0x50,0xBE,0x8E,0x22,0x2B,0xBD,0xE3,0xA9,0x07,0xBE,0x0A,0xD7,0x23,0xBD
,0xCF,0xA1,0x8C,0x3E,0x9A,0x99,0x19,0xBF,0x1F,0x84,0x18,0x3F,0x9A,0x99,0x19
,0xBF,0x9E,0xED,0x81,0xBE,0x5C,0x8F,0x02,0x3F,0x6D,0x91,0x34,0xBE,0x19,0x04
,0x06,0x3F,0x44,0x33,0xEF,0xBD,0x02,0x2B,0x07,0x3F,0x6F,0xD6,0x60,0xBC,0x35
,0xB3,0x02,0x3F,0xC3,0xF5,0x88,0x3D,0x9E,0x97,0xEA,0x3E,0x39,0x7F,0xF3,0x3D
,0x02,0x82,0xC1,0x3E,0xED,0x80,0x0B,0x3E,0x7F,0xBE,0x8D,0x3E,0x0C,0x02,0xEB
,0x3D,0x86,0xE8,0x30,0x3E,0x3A,0x04,0x4E,0x3D,0x7C,0x0C,0xB6,0x3D,0x6D,0x54
,0x67,0xBD,0x81,0x7C,0x09,0x3D,0x13,0xD7,0x51,0xBE,0xD9,0xCE,0x77,0x3C,0x35
,0x62,0x66,0xBE,0xF4,0xFB,0x7E,0x3C,0x9E,0xED,0x81,0xBE,0x6F,0x12,0x83,0x3C
,0x05,0x33,0x34,0x33,0x05,0x06,0x32,0x33,0x06,0x32,0x06,0x07,0x29,0x2A,0x00
,0x29,0x00,0x01,0x29,0x01,0x02,0x29,0x02,0x03,0x28,0x29,0x03,0x28,0x03,0x04
,0x28,0x04,0x05,0x28,0x05,0x34,0x28,0x34,0x35,0x27,0x28,0x35,0x26,0x27,0x35
,0x26,0x35,0x36,0x26,0x36,0x37,0x25,0x26,0x37,0x1E,0x1F,0x20,0x1D,0x1E,0x20
,0x1D,0x20,0x21,0x1D,0x21,0x22,0x1D,0x22,0x23,0x1B,0x1C,0x1D,0x10,0x11,0x12
,0x0F,0x10,0x12,0x0F,0x12,0x13,0x0F,0x13,0x14,0x0F,0x14,0x15,0x32,0x07,0x08
,0x31,0x32,0x08,0x31,0x08,0x09,0x31,0x09,0x0A,0x30,0x31,0x0A,0x30,0x0A,0x0B
,0x30,0x0B,0x0C,0x2F,0x30,0x0C,0x2F,0x0C,0x0D,0x2F,0x0D,0x0E,0x2E,0x2F,0x0E
,0x2E,0x0E,0x0F,0x2D,0x2E,0x0F,0x2C,0x2D,0x0F,0x0F,0x15,0x16,0x2B,0x2C,0x0F
,0x2B,0x0F,0x16,0x37,0x2B,0x16,0x37,0x16,0x17,0x25,0x37,0x17,0x24,0x25,0x17
,0x24,0x17,0x18,0x23,0x24,0x18,0x23,0x18,0x19,0x23,0x19,0x1A,0x1D,0x23,0x1A
,0x1D,0x1A,0x1B,0x2A,0x37,0x53,0x4A,0x48,0x01,0xBC,0x74,0x93,0x3E,0x64,0x3B
,0x1F,0x3F,0xFE,0x2C,0x8E,0x3E,0x49,0x80,0x16,0x3F,0x8B,0x6C,0x87,0x3E,0xBB
,0x48,0x11,0x3F,0x35,0xB8,0x7D,0x3E,0x64,0xB0,0x0E,0x3F,0xCB,0x49,0x68,0x3E
,0xF2,0xD2,0x0D,0x3F,0x74,0xED,0x4B,0x3E,0xCA,0x37,0x0F,0x3F,0x53,0x93,0x20
,0x3E,0x62,0x66,0x13,0x3F,0xBC,0xE9,0x56,0x3D,0x20,0x46,0x1C,0x3F,0x91,0xED
,0x3C,0xBD,0x64,0x3B,0x1F,0x3F,0x6A,0x69,0x3E,0xBE,0xF9,0x49,0x19,0x3F,0xAA
,0xF3,0x98,0xBE,0xC9,0x75,0x07,0x3F,0xF9,0x11,0xBF,0xBE,0x5F,0xD0,0xDA,0x3E
,0xA8,0xC6,0xCB,0xBE,0x79,0x95,0x9D,0x3E,0xE1,0x7B,0xC7,0xBE,0xBB,0x9C,0x72
,0x3E,0xAE,0x9B,0xBA,0xBE,0x1C,0x5E,0x30,0x3E,0x26,0x6E,0xA5,0xBE,0x01,0xF8
,0xE7,0x3D,0x60,0x3B,0x88,0xBE,0x10,0x08,0x74,0x3D,0xE2,0x22,0x37,0xBE,0xA8
,0xA7,0x0F,0xB9,0x39,0xEF,0x3F,0xBD,0x38,0x14,0x9E,0xBD,0xAC,0xCA,0x9E,0x3D
,0xF4,0xA5,0x17,0xBE,0x73,0x68,0x11,0x3E,0x85,0xEB,0x41,0xBE,0x73,0x11,0x3F
,0x3E,0x50,0x38,0x6B,0xBE,0x69,0xE5,0x5E,0x3E,0x23,0xD9,0x8B,0xBE,0x7D,0x94
,0x71,0x3E,0x0D,0xC4,0xA2,0xBE,0xD9,0xCE,0x77,0x3E,0xC5,0x74,0xB9,0xBE,0x2D
,0x5F,0x67,0x3E,0xD8,0x62,0xDF,0xBE,0x6B,0x10,0x36,0x3E,0x64,0xE5,0xFF,0xBE
,0x38,0xD7,0xD0,0x3D,0x79,0x1F,0x0B,0xBF,0x97,0x72,0x3E,0x3B,0x17,0xD9,0x0E
,0xBF,0x00,0x02,0xB6,0xBD,0x14,0x22,0x0C,0xBF,0x57,0x0A,0x31,0xBE,0xEB,0xFC
,0x03,0xBF,0x4F,0x90,0x78,0xBE,0x67,0x7C,0xEF,0xBE,0x06,0x81,0x95,0xBE,0xC5
,0x74,0xD1,0xBE,0x72,0x6A,0xA7,0xBE,0x08,0x57,0xA8,0xBE,0x7D,0x3F,0xB5,0xBE
,0x1B,0x2F,0x5D,0xBE,0xCB,0xA1,0xC5,0xBE,0x1B,0x2F,0x5D,0xBE,0xCB,0xA1,0xC5
,0xBE,0x89,0x41,0x20,0xBF,0x7D,0x3F,0xB5,0xBE,0x89,0x41,0x20,0xBF,0x5C,0x3B
,0xB1,0xBE,0xAC,0x70,0x17,0xBF,0x1F,0x85,0xAB,0xBE,0x71,0x3D,0x12,0xBF,0x9C
,0xC5,0xA3,0xBE,0x25,0xB2,0x0F,0xBF,0xEC,0xA5,0x99,0xBE,0x17,0xD9,0x0E,0xBF
,0xC9,0xCC,0x85,0xBE,0xC3,0xA0,0x10,0xBF,0x61,0x8D,0x43,0xBE,0xD7,0xF7,0x15
,0xBF,0x6A,0xBC,0xF4,0xBD,0x89,0x97,0x1B,0xBF,0x09,0x87,0x9E,0xBD,0x50,0x39
,0x1E,0xBF,0x34,0x0E,0xF5,0xBC,0x83,0xBF,0x1F,0xBF,0xC9,0x77,0xA9,0x3C,0x89
,0x41,0x20,0xBF,0xE2,0xCC,0x2F,0x3E,0x22,0xFA,0x19,0xBF,0x3D,0xB6,0x95,0x3E
,0xDD,0x23,0x07,0xBF,0x9E,0x42,0xBE,0x3E,0x14,0x42,0xD7,0xBE,0xA8,0xC6,0xCB
,0x3E,0xF0,0xA7,0x96,0xBE,0xEF,0x8F,0xC7,0x3E,0x74,0xEB,0x65,0xBE,0x81,0xEB
,0xBA,0x3E,0xFC,0xA9,0x21,0xBE,0x4F,0x77,0xA6,0x3E,0x7B,0xBE,0xC6,0xBD,0xE5
,0xD0,0x8A,0x3E,0x0F,0x7D,0x37,0xBD,0xCA,0xA3,0x3B,0x3E,0xB9,0x6D,0x5F,0x3C
,0x7B,0x9F,0x2A,0x3D,0x09,0x87,0xBE,0x3D,0x63,0x0E,0x02,0xBE,0x9C,0xC0,0x44
,0x3E,0xE9,0x26,0x61,0xBE,0x13,0xD7,0x89,0x3E,0xA8,0x6E,0x7E,0xBE,0x83,0x16
,0xA2,0x3E,0x93,0x18,0x84,0xBE,0xDD,0xD0,0xBC,0x3E,0x5A,0xBC,0x78,0xBE,0x2D
,0x5F,0xDF,0x3E,0x35,0x5E,0x4A,0xBE,0xF8,0xFB,0xFD,0x3E,0x4E,0x5E,0x04,0xBE
,0x6B,0xBB,0x09,0x3F,0xE9,0x10,0x38,0xBD,0xDF,0x4F,0x0D,0x3F,0x77,0x12,0x11
,0x3D,0xE3,0xA6,0x0A,0x3F,0xB1,0xA5,0xE7,0x3D,0x00,0xAC,0x02,0x3F,0x29,0x09
,0x39,0x3E,0xF4,0x53,0xEC,0x3E,0xBA,0x49,0x6C,0x3E,0x0A,0xD7,0xCB,0x3E,0x39
,0x0A,0x88,0x3E,0x63,0x64,0xA1,0x3E,0xBC,0x74,0x93,0x3E,0xF4,0xFD,0x54,0x3E
,0x0A,0xD7,0xA3,0x3E,0xF4,0xFD,0x54,0x3E,0x0A,0xD7,0xA3,0x3E,0x64,0x3B,0x1F
,0x3F,0x48,0x49,0x00,0x48,0x00,0x01,0x48,0x01,0x02,0x48,0x02,0x03,0x47,0x48
,0x03,0x46,0x47,0x03,0x45,0x46,0x03,0x44,0x45,0x03,0x44,0x03,0x04,0x44,0x04
,0x05,0x43,0x44,0x05,0x43,0x05,0x06,0x43,0x06,0x07,0x42,0x43,0x07,0x42,0x07
,0x08,0x41,0x42,0x08,0x41,0x08,0x09,0x40,0x41,0x09,0x3F,0x40,0x09,0x3F,0x09
,0x0A,0x3E,0x3F,0x0A,0x3E,0x0A,0x0B,0x3D,0x3E,0x0B,0x3D,0x0B,0x0C,0x3C,0x3D
,0x0C,0x3C,0x0C,0x0D,0x23,0x24,0x25,0x23,0x25,0x26,0x23,0x26,0x27,0x23,0x27
,0x28,0x22,0x23,0x28,0x21,0x22,0x28,0x20,0x21,0x28,0x1F,0x20,0x28,0x1F,0x28
,0x29,0x1F,0x29,0x2A,0x1E,0x1F,0x2A,0x1E,0x2A,0x2B,0x1D,0x1E,0x2B,0x1D,0x2B
,0x2C,0x1D,0x2C,0x2D,0x1C,0x1D,0x2D,0x1C,0x2D,0x2E,0x1C,0x2E,0x2F,0x1B,0x1C
,0x2F,0x1B,0x2F,0x30,0x1A,0x1B,0x30,0x1A,0x30,0x31,0x19,0x1A,0x31,0x19,0x31
,0x32,0x18,0x19,0x32,0x18,0x32,0x33,0x17,0x18,0x33,0x17,0x33,0x34,0x16,0x17
,0x34,0x16,0x34,0x35,0x15,0x16,0x35,0x15,0x35,0x36,0x15,0x36,0x37,0x14,0x15
,0x37,0x14,0x37,0x38,0x13,0x14,0x38,0x13,0x38,0x39,0x12,0x13,0x39,0x11,0x12
,0x39,0x11,0x39,0x3A,0x10,0x11,0x3A,0x10,0x3A,0x3B,0x0F,0x10,0x3B,0x0E,0x0F
,0x3B,0x0D,0x0E,0x3B,0x3C,0x0D,0x3B,0x49,0x54,0x22,0x20,0x01,0x48,0xE1,0xFA
,0xBE,0x75,0x93,0x18,0x3F,0x12,0x83,0x00,0xBF,0x5A,0x10,0xA2,0x3E,0x3B,0xDF
,0xEF,0xBE,0x5A,0x10,0xA2,0x3E,0xDB,0x50,0xE9,0xBE,0xD7,0x4F,0xC7,0x3E,0xDF
,0xFB,0xDB,0xBE,0xDF,0xFB,0xE3,0x3E,0x79,0x91,0xC9,0xBE,0x90,0x6A,0xF8,0x3E
,0x98,0xC2,0xB3,0xBE,0xD1,0x78,0x02,0x3F,0xD5,0x76,0x9B,0xBE,0x8C,0xD7,0x04
,0x3F,0xC4,0x78,0x6D,0xBE,0xCB,0xA1,0x05,0x3F,0x58,0x39,0xB4,0xBD,0xCB,0xA1
,0x05,0x3F,0x58,0x39,0xB4,0xBD,0x39,0xB4,0xC8,0xBE,0x15,0x52,0xBE,0xBD,0x89
,0x42,0xF3,0xBE,0xC7,0x9B,0xDC,0xBD,0x71,0x93,0x05,0xBF,0xBC,0xCA,0x1A,0xBE
,0x2E,0x73,0x0E,0xBF,0x3F,0x35,0x5E,0xBE,0x73,0x68,0x11,0xBF,0xB8,0x1E,0x85
,0xBE,0x73,0x68,0x11,0xBF,0xB8,0x1E,0x85,0xBE,0x9A,0x99,0x19,0xBF,0x4A,0x0C
,0x82,0x3E,0x9A,0x99,0x19,0xBF,0x4A,0x0C,0x82,0x3E,0x73,0x68,0x11,0xBF,0xBD
,0xC4,0x58,0x3E,0x73,0x68,0x11,0xBF,0x42,0x5C,0x19,0x3E,0x9B,0x04,0x0F,0xBF
,0x1B,0xBA,0xD9,0x3D,0x12,0xD9,0x07,0xBF,0x3D,0x62,0xB4,0x3D,0x75,0xE6,0xF6
,0xBE,0x9E,0xEF,0xA7,0x3D,0x39,0xB4,0xC8,0xBE,0x9E,0xEF,0xA7,0x3D,0xCB,0xA1
,0x05,0x3F,0x72,0x14,0x80,0x3E,0xCB,0xA1,0x05,0x3F,0x3D,0x49,0xA2,0x3E,0x83
,0x14,0x04,0x3F,0xF2,0x28,0xBD,0x3E,0x17,0xD9,0xFE,0x3E,0xD4,0x0F,0xD2,0x3E
,0x00,0xFD,0xEE,0x3E,0x25,0x5A,0xE2,0x3E,0x79,0x3D,0xD8,0x3E,0x1D,0x03,0xEA
,0x3E,0x75,0x91,0xC2,0x3E,0x3B,0xDF,0xEF,0x3E,0x5A,0x10,0xA2,0x3E,0x12,0x83
,0x00,0x3F,0x5A,0x10,0xA2,0x3E,0x48,0xE1,0xFA,0x3E,0x75,0x93,0x18,0x3F,0x1F
,0x20,0x21,0x1E,0x1F,0x21,0x1D,0x1E,0x21,0x1C,0x1D,0x21,0x1B,0x1C,0x21,0x00
,0x01,0x02,0x00,0x02,0x03,0x00,0x03,0x04,0x00,0x04,0x05,0x11,0x12,0x13,0x10
,0x11,0x13,0x10,0x13,0x14,0x10,0x14,0x15,0x10,0x15,0x16,0x0E,0x0F,0x10,0x00
,0x05,0x06,0x1A,0x1B,0x21,0x1A,0x21,0x00,0x19,0x1A,0x00,0x18,0x19,0x00,0x00
,0x06,0x07,0x0D,0x0E,0x10,0x0D,0x10,0x16,0x0C,0x0D,0x16,0x0B,0x0C,0x16,0x0B
,0x16,0x17,0x0A,0x0B,0x17,0x09,0x0A,0x17,0x09,0x17,0x18,0x09,0x18,0x00,0x08
,0x09,0x00,0x08,0x00,0x07,0x21,0x55,0x38,0x36,0x01,0xCB,0xA1,0x85,0x3E,0x4E
,0x62,0x10,0x3F,0xA4,0x1C,0xA4,0x3E,0x54,0xE3,0x0D,0x3F,0x08,0x04,0xBA,0x3E
,0x66,0x66,0x06,0x3F,0xF6,0x7E,0xC3,0x3E,0xFE,0x7C,0xF3,0x3E,0xF0,0xA7,0xC6
,0x3E,0xA6,0x9B,0xC4,0x3E,0xF0,0xA7,0xC6,0x3E,0xB9,0x6E,0xCA,0xBD,0x4D,0x4D
,0xC2,0x3E,0x5A,0xBB,0x85,0xBE,0x64,0x3D,0xB5,0x3E,0xA6,0x9B,0xBC,0xBE,0x51
,0x4E,0x9C,0x3E,0xAF,0xB3,0xE1,0xBE,0x19,0xAC,0x68,0x3E,0xD5,0x7A,0xFF,0xBE
,0x18,0x5C,0x03,0x3E,0xF4,0x88,0x09,0xBF,0x93,0x02,0x8B,0x3C,0xCD,0xCC,0x0C
,0xBF,0x07,0xB6,0x8A,0xBD,0x89,0xB7,0x0A,0xBF,0x52,0x64,0x0D,0xBE,0xCD,0x77
,0x04,0xBF,0x17,0x80,0x46,0xBE,0xAB,0x05,0xF6,0xBE,0x8F,0x72,0x70,0xBE,0x9D
,0x9B,0xDE,0xBE,0x94,0xC0,0x86,0xBE,0xCE,0xE0,0xBF,0xBE,0xE0,0xF7,0x8F,0xBE
,0x14,0x04,0x97,0xBE,0x58,0xE3,0x94,0xBE,0x3E,0x5B,0x57,0xBE,0x2B,0x87,0x96
,0xBE,0x2D,0x0A,0x0B,0xBE,0x2B,0x87,0x96,0xBE,0xA6,0x9B,0xC4,0x3E,0x6F,0xBC
,0x93,0xBE,0x17,0x83,0xEF,0x3E,0x3A,0x5C,0x8B,0xBE,0x1D,0x5A,0x04,0x3F,0xAF
,0xCC,0x6B,0xBE,0x42,0x60,0x0D,0x3F,0x7A,0x6C,0x2B,0xBE,0x4E,0x62,0x10,0x3F
,0x23,0xDB,0xF9,0xBD,0x4E,0x62,0x10,0x3F,0x23,0xDB,0xF9,0xBD,0x75,0x93,0x18
,0x3F,0xF8,0x53,0x23,0xBF,0x75,0x93,0x18,0x3F,0xF8,0x53,0x23,0xBF,0x4E,0x62
,0x10,0x3F,0xD1,0xCC,0x17,0xBF,0x4E,0x62,0x10,0x3F,0xB8,0xC9,0x08,0xBF,0xBF
,0x2B,0x0E,0x3F,0x0A,0x2B,0xFD,0xBE,0x33,0x88,0x07,0x3F,0xC9,0x77,0xF1,0xBE
,0x0B,0x99,0xF3,0x3E,0x68,0x91,0xED,0xBE,0xA6,0x9B,0xC4,0x3E,0x68,0x91,0xED
,0xBE,0xF1,0x82,0xA8,0xBD,0xC7,0xF3,0xE9,0xBE,0x7D,0xEC,0x86,0xBE,0xA0,0x1A
,0xDF,0xBE,0x75,0x93,0xC0,0xBE,0x23,0xDB,0xC1,0xBE,0x0A,0x2B,0xF5,0xBE,0xE6
,0x78,0x95,0xBE,0x79,0xE9,0x0E,0xBF,0x2F,0xDF,0x2A,0xBE,0x8E,0xEB,0x1B,0xBF
,0xD1,0x24,0xB1,0x3B,0x89,0x41,0x20,0xBF,0x4C,0x8B,0x2A,0x3E,0x0B,0xB8,0x1B
,0xBF,0xD1,0x76,0x94,0x3E,0xA0,0x1B,0x0E,0xBF,0x9C,0x30,0xC1,0x3E,0xE3,0x4F
,0xF4,0xBE,0xD3,0x4B,0xDC,0x3E,0x33,0x31,0xC5,0xBE,0xB0,0x04,0xEA,0x3E,0x5B
,0x25,0x88,0xBE,0x8D,0x97,0xEE,0x3E,0x23,0xDB,0xD9,0xBD,0x8D,0x97,0xEE,0x3E
,0x37,0x89,0xC1,0x3E,0xCF,0xF6,0xF0,0x3E,0xBF,0x49,0xEB,0x3E,0x72,0x14,0xF8
,0x3E,0x9E,0x43,0x01,0x3F,0x88,0xD5,0x07,0x3F,0x99,0x9A,0x0C,0x3F,0xD9,0xCE
,0x17,0x3F,0x4E,0x62,0x10,0x3F,0xF8,0x53,0x23,0x3F,0x4E,0x62,0x10,0x3F,0xF8
,0x53,0x23,0x3F,0x75,0x93,0x18,0x3F,0x1B,0x2F,0x5D,0x3E,0x75,0x93,0x18,0x3F
,0x1B,0x2F,0x5D,0x3E,0x4E,0x62,0x10,0x3F,0x36,0x37,0x00,0x35,0x36,0x00,0x35
,0x00,0x01,0x35,0x01,0x02,0x33,0x34,0x35,0x33,0x35,0x02,0x32,0x33,0x02,0x31
,0x32,0x02,0x31,0x02,0x03,0x30,0x31,0x03,0x30,0x03,0x04,0x2F,0x30,0x04,0x2F
,0x04,0x05,0x2E,0x2F,0x05,0x2E,0x05,0x06,0x2D,0x2E,0x06,0x2C,0x2D,0x06,0x2C
,0x06,0x07,0x2B,0x2C,0x07,0x2B,0x07,0x08,0x2A,0x2B,0x08,0x2A,0x08,0x09,0x29
,0x2A,0x09,0x29,0x09,0x0A,0x28,0x29,0x0A,0x28,0x0A,0x0B,0x28,0x0B,0x0C,0x27
,0x28,0x0C,0x27,0x0C,0x0D,0x26,0x27,0x0D,0x26,0x0D,0x0E,0x26,0x0E,0x0F,0x25
,0x26,0x0F,0x25,0x0F,0x10,0x24,0x25,0x10,0x24,0x10,0x11,0x23,0x24,0x11,0x23
,0x11,0x12,0x22,0x23,0x12,0x22,0x12,0x13,0x22,0x13,0x14,0x21,0x22,0x14,0x1B
,0x1C,0x1D,0x1A,0x1B,0x1D,0x1A,0x1D,0x1E,0x1A,0x1E,0x1F,0x1A,0x1F,0x20,0x18
,0x19,0x1A,0x21,0x14,0x15,0x20,0x21,0x15,0x20,0x15,0x16,0x20,0x16,0x17,0x1A
,0x20,0x17,0x1A,0x17,0x18,0x37,0x56,0x22,0x20,0x01,0x70,0x3F,0x88,0x3E,0x75
,0x93,0x18,0x3F,0x70,0x3F,0x88,0x3E,0x4E,0x62,0x10,0x3F,0x85,0xED,0x8F,0x3E
,0x3E,0xB4,0x0F,0x3F,0x69,0x90,0xAA,0x3E,0xD1,0x57,0x0C,0x3F,0x0F,0x2B,0xBC
,0x3E,0x71,0x3C,0x07,0x3F,0xF8,0xFB,0xC5,0x3E,0x7A,0x1D,0x01,0x3F,0x85,0x41
,0xC9,0x3E,0x94,0x6C,0xF5,0x3E,0xBF,0xF3,0xC3,0x3E,0x87,0x16,0xD9,0x3E,0x4A
,0x0A,0xB4,0x3E,0x5A,0xB8,0xAC,0x3E,0xA7,0x93,0x8C,0x3D,0x2D,0xB2,0xB5,0xBE
,0x18,0x08,0x72,0xBE,0x8E,0x3F,0xB1,0x3E,0x85,0x97,0x88,0xBE,0xC5,0xC8,0xDA
,0x3E,0xB8,0xC8,0x8D,0xBE,0x14,0xAE,0xF7,0x3E,0xBE,0x9F,0x8A,0xBE,0xEB,0x3B
,0x03,0x3F,0xD1,0x24,0x81,0xBE,0xF2,0x28,0x09,0x3F,0x93,0x1A,0x5A,0xBE,0xE7
,0x8F,0x0D,0x3F,0xC5,0x1C,0x14,0xBE,0x4E,0x62,0x10,0x3F,0xC5,0x1C,0x14,0xBE
,0x75,0x93,0x18,0x3F,0xD3,0xF8,0x21,0xBF,0x75,0x93,0x18,0x3F,0xD3,0xF8,0x21
,0xBF,0x4E,0x62,0x10,0x3F,0x7F,0x69,0x15,0xBF,0xFC,0x89,0x0E,0x3F,0x62,0xBB
,0x0B,0xBF,0xAA,0x46,0x0B,0x3F,0x5F,0x24,0x04,0xBF,0x76,0x89,0x06,0x3F,0x2D
,0xB4,0xFB,0xBE,0x91,0x43,0x00,0x3F,0x65,0xE3,0xF1,0xBE,0xFA,0x7D,0xEF,0x3E
,0x2F,0xDF,0xE2,0xBE,0x4B,0x8F,0xCE,0x3E,0xAC,0x1F,0x9B,0x3B,0x89,0x41,0x20
,0xBF,0xAA,0xB6,0x1B,0x3D,0x89,0x41,0x20,0xBF,0xC2,0x4B,0xE8,0x3E,0x72,0x14
,0xC8,0x3E,0xF6,0x28,0xFC,0x3E,0x64,0xE5,0xEF,0x3E,0x2D,0x5D,0x09,0x3F,0xDF
,0xF9,0x05,0x3F,0x29,0xB1,0x13,0x3F,0x5D,0x4F,0x0C,0x3F,0xD3,0xF8,0x21,0x3F
,0x4E,0x62,0x10,0x3F,0xD3,0xF8,0x21,0x3F,0x75,0x93,0x18,0x3F,0x20,0x21,0x00
,0x1F,0x20,0x00,0x1E,0x1F,0x00,0x1D,0x1E,0x00,0x11,0x12,0x13,0x11,0x13,0x14
,0x11,0x14,0x15,0x11,0x15,0x16,0x11,0x16,0x17,0x11,0x17,0x18,0x0F,0x10,0x11
,0x0F,0x11,0x18,0x0E,0x0F,0x18,0x0D,0x0E,0x18,0x0C,0x0D,0x18,0x0C,0x18,0x19
,0x0B,0x0C,0x19,0x0B,0x19,0x1A,0x0A,0x0B,0x1A,0x09,0x0A,0x1A,0x09,0x1A,0x1B
,0x09,0x1B,0x1C,0x08,0x09,0x1C,0x07,0x08,0x1C,0x07,0x1C,0x1D,0x06,0x07,0x1D
,0x00,0x01,0x02,0x05,0x06,0x1D,0x00,0x02,0x03,0x1D,0x00,0x03,0x1D,0x03,0x04
,0x05,0x1D,0x04,0x21,0x57,0x3F,0x3D,0x01,0x68,0x93,0xF3,0x3E,0x75,0x93,0x18
,0x3F,0x68,0x93,0xF3,0x3E,0x4E,0x62,0x10,0x3F,0x07,0x42,0x06,0x3F,0x89,0x97
,0x0F,0x3F,0xD1,0x21,0x10,0x3F,0x4C,0x37,0x0D,0x3F,0x58,0xAD,0x14,0x3F,0x95
,0xB8,0x0A,0x3F,0x46,0x60,0x18,0x3F,0x33,0xDE,0x06,0x3F,0x39,0xD5,0x1A,0x3F
,0xC3,0x2A,0x02,0x3F,0xF0,0xA6,0x1B,0x3F,0x81,0x41,0xFA,0x3E,0x0A,0x2B,0x19
,0x3F,0x60,0xE6,0xDB,0x3E,0x4A,0xB7,0x11,0x3F,0xFC,0x51,0xAC,0x3E,0xB4,0x72
,0xAF,0x3E,0x56,0x0E,0xA5,0xBE,0xE7,0x53,0xE7,0x3D,0x25,0xB2,0xA7,0x3E,0x78
,0x5E,0xAA,0x3D,0x0D,0xAB,0xD8,0x3E,0x7C,0x0C,0x96,0x3D,0x90,0x6A,0xF8,0x3E
,0xC1,0xCA,0xA1,0x3D,0x48,0x8A,0x04,0x3F,0x14,0x06,0xC5,0x3D,0xAA,0xF1,0x0A
,0x3F,0x9C,0xC4,0x00,0x3E,0x2D,0x06,0x0F,0x3F,0x75,0x8F,0x2C,0x3E,0x4E,0x62
,0x10,0x3F,0x88,0x12,0x4D,0x3E,0x4E,0x62,0x10,0x3F,0x88,0x12,0x4D,0x3E,0x75
,0x93,0x18,0x3F,0xDB,0xA3,0x87,0xBE,0x75,0x93,0x18,0x3F,0xDB,0xA3,0x87,0xBE
,0x4E,0x62,0x10,0x3F,0xE0,0x2A,0x6F,0xBE,0x46,0x0B,0x10,0x3F,0x15,0x02,0x59
,0xBE,0x2D,0x06,0x0F,0x3F,0x90,0x68,0x42,0xBE,0xDF,0xA5,0x0C,0x3F,0xDA,0xCA
,0x2B,0xBE,0xD9,0x22,0x09,0x3F,0xAD,0xF7,0x1B,0xBE,0x16,0xDA,0x05,0x3F,0x48
,0xDD,0x0E,0xBE,0x54,0xE3,0x01,0x3F,0x6F,0xF1,0xF0,0xBD,0x3C,0x33,0xF1,0x3E
,0xD6,0xAB,0xC8,0xBD,0x6D,0xE7,0xDB,0x3E,0xB3,0xD0,0x8E,0xBD,0x11,0xAC,0xB2
,0x3E,0xA7,0xEB,0x09,0xBD,0xD1,0xCA,0x7D,0x3E,0x96,0x3F,0x6F,0xBE,0x5D,0x37
,0xA5,0xBE,0xB7,0xF1,0xEF,0xBE,0x9E,0x97,0xB2,0x3E,0x13,0xD6,0xFE,0xBE,0x5A
,0x66,0xE1,0x3E,0x68,0xE6,0x01,0xBF,0x81,0x95,0xFB,0x3E,0xEC,0x51,0x00,0xBF
,0x9A,0xEF,0x04,0x3F,0xE9,0x28,0xF7,0xBE,0xAA,0xF1,0x0A,0x3F,0x31,0xB3,0xE7
,0xBE,0x2D,0x06,0x0F,0x3F,0x95,0x99,0xD2,0xBE,0x4E,0x62,0x10,0x3F,0xA0,0xC4
,0xC7,0xBE,0x4E,0x62,0x10,0x3F,0xA0,0xC4,0xC7,0xBE,0x75,0x93,0x18,0x3F,0x6B
,0x11,0x55,0xBF,0x75,0x93,0x18,0x3F,0x6B,0x11,0x55,0xBF,0x4E,0x62,0x10,0x3F
,0x8A,0x21,0x49,0xBF,0x86,0x56,0x0F,0x3F,0xC7,0xF5,0x3F,0xBF,0x2F,0x33,0x0C
,0x3F,0xB1,0xFD,0x38,0xBF,0x79,0xE9,0x06,0x3F,0xF8,0xA8,0x33,0xBF,0x06,0xD5
,0xFE,0x3E,0x09,0x16,0x2F,0xBF,0x9C,0xC4,0xE8,0x3E,0xC1,0xC9,0x26,0xBF,0x6A
,0x68,0xBB,0x3E,0xC9,0x74,0x98,0xBE,0x89,0x41,0x20,0xBF,0xA0,0xC4,0x87,0xBE
,0x89,0x41,0x20,0xBF,0xA8,0xC7,0xB6,0x3B,0x97,0x1E,0x0D,0x3E,0x69,0x3B,0x8E
,0x3E,0x89,0x41,0x20,0xBF,0xAA,0xF3,0xA0,0x3E,0x89,0x41,0x20,0xBF,0x3F,0x36
,0x25,0x3F,0xBD,0xC8,0xAC,0x3E,0xB6,0x9E,0x2D,0x3F,0x83,0x68,0xDD,0x3E,0xA0
,0x6F,0x33,0x3F,0xD5,0x24,0xF8,0x3E,0xFA,0x5F,0x3A,0x3F,0x34,0xF2,0x05,0x3F
,0xF4,0xA7,0x41,0x3F,0x93,0x18,0x0C,0x3F,0x32,0x3C,0x4A,0x3F,0xE8,0x4F,0x0F
,0x3F,0x6B,0x11,0x55,0x3F,0x4E,0x62,0x10,0x3F,0x6B,0x11,0x55,0x3F,0x75,0x93
,0x18,0x3F,0x3D,0x3E,0x00,0x3C,0x3D,0x00,0x3B,0x3C,0x00,0x3A,0x3B,0x00,0x39
,0x3A,0x00,0x29,0x2A,0x2B,0x29,0x2B,0x2C,0x29,0x2C,0x2D,0x29,0x2D,0x2E,0x29
,0x2E,0x2F,0x29,0x2F,0x30,0x27,0x28,0x29,0x13,0x14,0x15,0x13,0x15,0x16,0x13
,0x16,0x17,0x13,0x17,0x18,0x13,0x18,0x19,0x13,0x19,0x1A,0x13,0x1A,0x1B,0x13
,0x1B,0x1C,0x11,0x12,0x13,0x00,0x01,0x02,0x35,0x36,0x37,0x32,0x33,0x34,0x26
,0x27,0x29,0x26,0x29,0x30,0x25,0x26,0x30,0x24,0x25,0x30,0x23,0x24,0x30,0x23
,0x30,0x31,0x22,0x23,0x31,0x21,0x22,0x31,0x21,0x31,0x32,0x20,0x21,0x32,0x20
,0x32,0x34,0x1F,0x20,0x34,0x10,0x11,0x13,0x10,0x13,0x1C,0x0F,0x10,0x1C,0x0E
,0x0F,0x1C,0x0D,0x0E,0x1C,0x0D,0x1C,0x1D,0x0C,0x0D,0x1D,0x0C,0x1D,0x1E,0x0B
,0x0C,0x1E,0x0B,0x1E,0x1F,0x0A,0x0B,0x1F,0x0A,0x1F,0x34,0x0A,0x34,0x35,0x0A
,0x35,0x37,0x09,0x0A,0x37,0x08,0x09,0x37,0x08,0x37,0x38,0x07,0x08,0x38,0x07
,0x38,0x39,0x06,0x07,0x39,0x05,0x06,0x39,0x04,0x05,0x39,0x00,0x02,0x03,0x39
,0x00,0x03,0x04,0x39,0x03,0x3E,0x58,0x4A,0x48,0x01,0x6B,0xBA,0xA6,0x3E,0x71
,0x91,0xBB,0x3E,0x79,0x93,0xC7,0x3E,0x61,0x8D,0xE3,0x3E,0x7B,0x12,0xE0,0x3E
,0xB6,0x9F,0xFC,0x3E,0x83,0x68,0xF5,0x3E,0xCE,0x37,0x06,0x3F,0x52,0x63,0x06
,0x3F,0xEC,0x51,0x0C,0x3F,0x35,0xB2,0x0F,0x3F,0x29,0x07,0x0F,0x3F,0x46,0x61
,0x1B,0x3F,0x4E,0x62,0x10,0x3F,0x46,0x61,0x1B,0x3F,0x75,0x93,0x18,0x3F,0x7F
,0x16,0x3B,0x3E,0x75,0x93,0x18,0x3F,0x7F,0x16,0x3B,0x3E,0x4E,0x62,0x10,0x3F
,0x25,0x06,0x71,0x3E,0x2C,0xBC,0x0F,0x3F,0x87,0x6C,0x88,0x3E,0x64,0x3C,0x0E
,0x3F,0x75,0xE7,0x91,0x3E,0xE1,0x9A,0x0B,0x3F,0x71,0x93,0x99,0x3E,0x58,0x8F
,0x07,0x3F,0xCF,0xA4,0x9D,0x3E,0x76,0xA7,0x03,0x3F,0x04,0x00,0x9F,0x3E,0x46
,0x5E,0xFE,0x3E,0x8D,0xED,0x9D,0x3E,0x6A,0xFA,0xF4,0x3E,0x4A,0xB6,0x9A,0x3E
,0xE1,0xD0,0xEB,0x3E,0x90,0xC1,0x92,0x3E,0x0B,0xEB,0xDE,0x3E,0xD5,0x76,0x83
,0x3E,0xE3,0x51,0xCA,0x3E,0x6B,0x81,0x3D,0x3D,0xC9,0x1E,0x01,0x3E,0x88,0x12
,0x0D,0xBE,0xA8,0x1A,0xCD,0x3E,0x68,0x91,0x2D,0xBE,0x46,0x5F,0xE9,0x3E,0x5E
,0x66,0x38,0xBE,0xB6,0xF3,0xFD,0x3E,0x6E,0xBE,0x31,0xBE,0x9F,0x05,0x05,0x3F
,0xA0,0xC6,0x1D,0xBE,0x02,0x81,0x0A,0x3F,0x31,0x7D,0xEF,0xBD,0xF7,0x92,0x0E
,0x3F,0xE6,0x95,0x6B,0xBD,0x4E,0x62,0x10,0x3F,0xE6,0x95,0x6B,0xBD,0x75,0x93
,0x18,0x3F,0xC3,0xA0,0x1C,0xBF,0x75,0x93,0x18,0x3F,0xC3,0xA0,0x1C,0xBF,0x4E
,0x62,0x10,0x3F,0x36,0x93,0x07,0xBF,0x27,0x11,0x0D,0x3F,0x71,0x3D,0xEA,0xBE
,0xD5,0x79,0x04,0x3F,0xB5,0x33,0xC4,0xBE,0xF5,0x13,0xE6,0x3E,0x39,0xB6,0x96
,0xBE,0x35,0x5E,0xAA,0x3E,0x4B,0x3F,0x81,0xBD,0xA8,0xC5,0x60,0xBC,0x89,0x43
,0xAE,0xBE,0x9F,0x72,0xBC,0xBE,0xCB,0xF7,0xCC,0xBE,0x8E,0x57,0xE0,0xBE,0x49
,0x0E,0xE8,0xBE,0x81,0x97,0xF9,0xBE,0x15,0x02,0x01,0xBF,0xB6,0xBD,0x05,0xBF
,0x04,0xAB,0x0E,0xBF,0xDF,0xA5,0x0C,0xBF,0x5F,0xCF,0x17,0xBF,0x6A,0xA6,0x0F
,0xBF,0x37,0x88,0x22,0xBF,0x73,0x68,0x11,0xBF,0x37,0x88,0x22,0xBF,0x9A,0x99
,0x19,0xBF,0x2B,0xDB,0x57,0xBE,0x9A,0x99,0x19,0xBF,0x2B,0xDB,0x57,0xBE,0x73
,0x68,0x11,0xBF,0x3D,0xB7,0x88,0xBE,0x7C,0xB5,0x0F,0xBF,0xB0,0x70,0x9A,0xBE
,0xDB,0xF8,0x0B,0xBF,0x81,0x95,0xA3,0xBE,0x57,0x98,0x06,0xBF,0xC7,0xA1,0xA6
,0xBE,0xAE,0xF3,0xFF,0xBE,0xBB,0x9D,0xA5,0xBE,0x17,0x9B,0xF6,0xBE,0x75,0x91
,0xA2,0xBE,0xA2,0x9B,0xED,0xBE,0xB0,0x70,0x9A,0xBE,0x01,0x6C,0xE0,0xBE,0x27
,0x2F,0x8A,0xBE,0x0A,0x83,0xCA,0xBE,0x78,0x5E,0xAA,0xBC,0xB6,0xF3,0x9D,0xBD
,0xF6,0xD4,0x3A,0x3E,0xFE,0x7C,0xC3,0xBE,0x91,0xED,0x5C,0x3E,0xD9,0x77,0xDD
,0xBE,0xF1,0x7E,0x6C,0x3E,0x85,0x3F,0xEB,0xBE,0x7D,0x97,0x72,0x3E,0xD7,0xA2
,0xF5,0xBE,0xD8,0x9F,0x74,0x3E,0x21,0x04,0x00,0xBF,0x71,0x93,0x71,0x3E,0xE9
,0x46,0x04,0xBF,0x7E,0x6E,0x68,0x3E,0xEE,0x7B,0x08,0xBF,0x65,0x8D,0x5A,0x3E
,0x81,0x20,0x0C,0xBF,0xBE,0x4B,0x49,0x3E,0x29,0xB2,0x0E,0xBF,0x65,0x39,0x29
,0x3E,0x34,0x9E,0x10,0xBF,0x63,0x43,0xF7,0x3D,0x73,0x68,0x11,0xBF,0x63,0x43
,0xF7,0x3D,0x9A,0x99,0x19,0xBF,0x37,0x88,0x22,0x3F,0x9A,0x99,0x19,0xBF,0x37
,0x88,0x22,0x3F,0x73,0x68,0x11,0xBF,0xB5,0xFD,0x0F,0x3F,0xB9,0xDF,0x0D,0xBF
,0x04,0x02,0x01,0x3F,0x2F,0x87,0x05,0xBF,0xC5,0x8C,0xE0,0x3E,0x6D,0x8F,0xE6
,0xBE,0xDB,0xF7,0xB0,0x3E,0x83,0x14,0xA4,0xBE,0xEB,0xA9,0xB5,0x3D,0x1A,0xA4
,0x80,0x3D,0x42,0x43,0x44,0x42,0x44,0x45,0x42,0x45,0x46,0x42,0x46,0x47,0x2B
,0x2C,0x2D,0x2B,0x2D,0x2E,0x2B,0x2E,0x2F,0x2B,0x2F,0x30,0x2A,0x2B,0x30,0x29
,0x2A,0x30,0x28,0x29,0x30,0x27,0x28,0x30,0x26,0x27,0x30,0x25,0x26,0x30,0x25
,0x30,0x31,0x25,0x31,0x32,0x24,0x25,0x32,0x24,0x32,0x33,0x24,0x33,0x34,0x24
,0x34,0x35,0x23,0x24,0x35,0x23,0x35,0x36,0x1C,0x1D,0x1E,0x1C,0x1E,0x1F,0x1C
,0x1F,0x20,0x1C,0x20,0x21,0x07,0x08,0x09,0x07,0x09,0x0A,0x07,0x0A,0x0B,0x07
,0x0B,0x0C,0x07,0x0C,0x0D,0x07,0x0D,0x0E,0x40,0x41,0x42,0x05,0x06,0x07,0x05
,0x07,0x0E,0x04,0x05,0x0E,0x03,0x04,0x0E,0x02,0x03,0x0E,0x01,0x02,0x0E,0x01
,0x0E,0x0F,0x01,0x0F,0x10,0x00,0x01,0x10,0x00,0x10,0x11,0x00,0x11,0x12,0x00
,0x12,0x13,0x49,0x00,0x13,0x49,0x13,0x14,0x1A,0x1B,0x1C,0x1A,0x1C,0x21,0x19
,0x1A,0x21,0x18,0x19,0x21,0x17,0x18,0x21,0x17,0x21,0x22,0x16,0x17,0x22,0x15
,0x16,0x22,0x15,0x22,0x23,0x14,0x15,0x23,0x49,0x14,0x23,0x49,0x23,0x36,0x48
,0x49,0x36,0x48,0x36,0x37,0x48,0x37,0x38,0x47,0x48,0x38,0x47,0x38,0x39,0x47
,0x39,0x3A,0x47,0x3A,0x3B,0x47,0x3B,0x3C,0x47,0x3C,0x3D,0x47,0x3D,0x3E,0x3F
,0x40,0x42,0x3F,0x42,0x47,0x3F,0x47,0x3E,0x49,0x59,0x35,0x33,0x01,0x5A,0x10
,0x5A,0x3E,0x4E,0x62,0x10,0x3F,0x35,0x5E,0x82,0x3E,0x08,0xAD,0x0F,0x3F,0xB2
,0x9B,0x91,0x3E,0x58,0x8D,0x0D,0x3F,0x49,0x63,0x9C,0x3E,0xD3,0x2E,0x0A,0x3F
,0x04,0x58,0xA4,0x3E,0x73,0xBD,0x05,0x3F,0x79,0x40,0xA9,0x3E,0x48,0x6B,0x00
,0x3F,0x61,0xE3,0xAA,0x3E,0x0F,0xD5,0xF4,0x3E,0xB0,0x1D,0xA4,0x3E,0x69,0x90
,0xDA,0x3E,0xC0,0xCC,0x8F,0x3E,0x94,0x6C,0xB5,0x3E,0xDA,0x1E,0x3D,0x3D,0x19
,0x03,0x6B,0xBC,0x1C,0x5E,0x50,0xBE,0x08,0x00,0xBE,0x3E,0x97,0xC4,0x79,0xBE
,0x14,0x05,0xE2,0x3E,0xC1,0xC8,0x83,0xBE,0x60,0xE5,0xF8,0x3E,0xEC,0xA4,0x7E
,0xBE,0x80,0x2B,0x05,0x3F,0x2B,0xDF,0x63,0xBE,0x62,0x66,0x0B,0x3F,0xEE,0x79
,0x3E,0xBE,0x57,0x23,0x0F,0x3F,0x1D,0xAE,0x15,0xBE,0x4E,0x62,0x10,0x3F,0xD9
,0x26,0xF5,0xBD,0x4E,0x62,0x10,0x3F,0xD9,0x26,0xF5,0xBD,0x75,0x93,0x18,0x3F
,0x4E,0x61,0x21,0xBF,0x75,0x93,0x18,0x3F,0x4E,0x61,0x21,0xBF,0x4E,0x62,0x10
,0x3F,0x70,0xD1,0x19,0xBF,0xEE,0xD1,0x0F,0x3F,0xAE,0x48,0x14,0xBF,0xCD,0x20
,0x0E,0x3F,0x11,0x6E,0x06,0xBF,0xC9,0xCB,0x06,0x3F,0x5C,0x8D,0xFC,0xBE,0x96
,0x97,0x00,0x3F,0xE1,0x24,0xED,0xBE,0xBD,0x72,0xED,0x3E,0xCD,0x22,0xD4,0xBE
,0xDE,0xCC,0xC8,0x3E,0xE1,0xD2,0xB1,0xBD,0x9C,0xC4,0xE0,0xBD,0xE1,0xD2,0xB1
,0xBD,0x39,0xB4,0xC8,0xBE,0x9E,0xEB,0xBB,0xBD,0x89,0x42,0xF3,0xBE,0x50,0x35
,0xDA,0xBD,0x71,0x93,0x05,0xBF,0x81,0x97,0x19,0xBE,0x2E,0x73,0x0E,0xBF,0x04
,0x02,0x5D,0xBE,0x73,0x68,0x11,0xBF,0x1B,0x85,0x84,0xBE,0x73,0x68,0x11,0xBF
,0x1B,0x85,0x84,0xBE,0x9A,0x99,0x19,0xBF,0xE8,0xA5,0x82,0x3E,0x9A,0x99,0x19
,0xBF,0xE8,0xA5,0x82,0x3E,0x73,0x68,0x11,0xBF,0xC3,0xF1,0x5C,0x3E,0x73,0x68
,0x11,0xBF,0xC8,0xCE,0x1B,0x3E,0x9B,0x04,0x0F,0xBF,0x91,0xBA,0xDD,0x3D,0x12
,0xD9,0x07,0xBF,0xF1,0x2E,0xB7,0x3D,0x75,0xE6,0xF6,0xBE,0x15,0x56,0xAA,0x3D
,0x39,0xB4,0xC8,0xBE,0x15,0x56,0xAA,0x3D,0xD9,0x43,0xBB,0xBD,0x31,0x06,0xBE
,0x3E,0xEE,0xD0,0xB8,0x3E,0xC2,0xF7,0xD6,0x3E,0x52,0xB8,0xDE,0x3E,0xBA,0x9F
,0xEB,0x3E,0x3E,0x5E,0xF8,0x3E,0xF2,0x26,0xFF,0x3E,0x18,0xCF,0x04,0x3F,0x21
,0x5B,0x0A,0x3F,0xF6,0x29,0x0B,0x3F,0xBA,0x49,0x14,0x3F,0x34,0x14,0x0F,0x3F
,0x0A,0x82,0x1B,0x3F,0x4E,0x62,0x10,0x3F,0x4E,0x61,0x21,0x3F,0x4E,0x62,0x10
,0x3F,0x4E,0x61,0x21,0x3F,0x75,0x93,0x18,0x3F,0x5A,0x10,0x5A,0x3E,0x75,0x93
,0x18,0x3F,0x33,0x34,0x00,0x33,0x00,0x01,0x33,0x01,0x02,0x33,0x02,0x03,0x33
,0x03,0x04,0x33,0x04,0x05,0x31,0x32,0x33,0x23,0x24,0x25,0x22,0x23,0x25,0x22
,0x25,0x26,0x22,0x26,0x27,0x22,0x27,0x28,0x20,0x21,0x22,0x12,0x13,0x14,0x12
,0x14,0x15,0x12,0x15,0x16,0x12,0x16,0x17,0x12,0x17,0x18,0x12,0x18,0x19,0x10
,0x11,0x12,0x33,0x05,0x06,0x33,0x06,0x07,0x30,0x31,0x33,0x0F,0x10,0x12,0x0F
,0x12,0x19,0x0E,0x0F,0x19,0x0D,0x0E,0x19,0x0C,0x0D,0x19,0x0C,0x19,0x1A,0x0B
,0x0C,0x1A,0x0B,0x1A,0x1B,0x0A,0x0B,0x1B,0x09,0x0A,0x1B,0x09,0x1B,0x1C,0x1F
,0x20,0x22,0x1F,0x22,0x28,0x1E,0x1F,0x28,0x1D,0x1E,0x28,0x1D,0x28,0x29,0x1C
,0x1D,0x29,0x09,0x1C,0x29,0x09,0x29,0x2A,0x09,0x2A,0x2B,0x08,0x09,0x2B,0x07
,0x08,0x2B,0x07,0x2B,0x2C,0x07,0x2C,0x2D,0x07,0x2D,0x2E,0x2F,0x30,0x33,0x2F
,0x33,0x07,0x2F,0x07,0x2E,0x34,0x5A,0x15,0x13,0x01,0xCB,0xA1,0xC5,0xBE,0x75
,0x93,0x18,0x3F,0x85,0xEB,0xD1,0xBE,0xC9,0xCA,0x97,0x3E,0x9C,0xC4,0xC0,0xBE
,0xC9,0xCA,0x97,0x3E,0xF9,0xC0,0xB6,0xBE,0xD9,0x21,0xC6,0x3E,0x5F,0xB8,0xAB
,0xBE,0xEC,0xF9,0xE2,0x3E,0x75,0x92,0x9D,0xBE,0x3E,0xB3,0xF4,0x3E,0x48,0x37
,0x8A,0xBE,0x06,0xD7,0x00,0x3F,0x39,0x0B,0x5B,0xBE,0x63,0xD1,0x04,0x3F,0x30
,0xB8,0x06,0xBE,0xDD,0x24,0x06,0x3F,0xD9,0x22,0x81,0x3E,0xDD,0x24,0x06,0x3F
,0x93,0x18,0x04,0xBF,0xAE,0x47,0x11,0xBF,0x93,0x18,0x04,0xBF,0x9A,0x99,0x19
,0xBF,0x23,0x2F,0xEB,0x3E,0x9A,0x99,0x19,0xBF,0x93,0x18,0x04,0x3F,0xAE,0x47
,0x91,0xBE,0xBE,0x4B,0xF9,0x3E,0x4B,0x8F,0x8E,0xBE,0xFE,0x80,0xDF,0x3E,0x2E
,0xC8,0xCE,0xBE,0x1F,0x31,0xC2,0x3E,0x3D,0x0C,0xF5,0xBE,0x73,0x69,0x9C,0x3E
,0xC5,0x01,0x04,0xBF,0x98,0x6E,0x52,0x3E,0x02,0x2B,0x07,0xBF,0xF3,0x55,0x92
,0xBE,0x02,0x2B,0x07,0xBF,0x6A,0xBC,0x00,0x3F,0x75,0x93,0x18,0x3F,0x00,0x01
,0x02,0x00,0x02,0x03,0x00,0x03,0x04,0x00,0x04,0x05,0x00,0x05,0x06,0x0A,0x0B
,0x0C,0x0C,0x0D,0x0E,0x0C,0x0E,0x0F,0x0C,0x0F,0x10,0x0C,0x10,0x11,0x00,0x06
,0x07,0x14,0x00,0x07,0x14,0x07,0x08,0x14,0x08,0x09,0x13,0x14,0x09,0x13,0x09
,0x0A,0x13,0x0A,0x0C,0x12,0x13,0x0C,0x12,0x0C,0x11,0x14,0x5B,0x08,0x06,0x01
,0xF0,0xA7,0x46,0x3E,0xF8,0x53,0x63,0xBF,0xB6,0xF3,0x7D,0xBD,0xF8,0x53,0x63
,0xBF,0xB6,0xF3,0x7D,0xBD,0xF2,0xD2,0x0D,0x3F,0xF0,0xA7,0x46,0x3E,0xF2,0xD2
,0x0D,0x3F,0xF0,0xA7,0x46,0x3E,0x77,0xBE,0x1F,0x3F,0xF0,0xA7,0x46,0xBE,0x77
,0xBE,0x1F,0x3F,0xF0,0xA7,0x46,0xBE,0x7D,0x3F,0x75,0xBF,0xF0,0xA7,0x46,0x3E
,0x7D,0x3F,0x75,0xBF,0x06,0x07,0x00,0x06,0x00,0x01,0x05,0x06,0x01,0x05,0x01
,0x02,0x04,0x05,0x02,0x04,0x02,0x03,0x07,0x5C,0x04,0x02,0x01,0xF0,0xA5,0x80
,0xBE,0x8B,0x6C,0x27,0x3F,0x55,0x12,0x39,0x3E,0x77,0xBE,0x1F,0xBF,0xF0,0xA5
,0x80,0x3E,0x77,0xBE,0x1F,0xBF,0xB9,0x1A,0x39,0xBE,0x8B,0x6C,0x27,0x3F,0x01
,0x02,0x03,0x00,0x01,0x03,0x03,0x5D,0x08,0x06,0x01,0xF0,0xA7,0x46,0xBE,0xF2
,0xD2,0x0D,0x3F,0xB6,0xF3,0x7D,0x3D,0xF2,0xD2,0x0D,0x3F,0xB6,0xF3,0x7D,0x3D
,0xF8,0x53,0x63,0xBF,0xF0,0xA7,0x46,0xBE,0xF8,0x53,0x63,0xBF,0xF0,0xA7,0x46
,0xBE,0x7D,0x3F,0x75,0xBF,0xF0,0xA7,0x46,0x3E,0x7D,0x3F,0x75,0xBF,0xF0,0xA7
,0x46,0x3E,0x77,0xBE,0x1F,0x3F,0xF0,0xA7,0x46,0xBE,0x77,0xBE,0x1F,0x3F,0x06
,0x07,0x00,0x06,0x00,0x01,0x05,0x06,0x01,0x05,0x01,0x02,0x04,0x05,0x02,0x04
,0x02,0x03,0x07,0x5E,0x07,0x05,0x01,0x58,0x39,0x34,0xBC,0x52,0xB8,0x1E,0x3F
,0x27,0x31,0xC8,0xBE,0xA6,0x9B,0x44,0xBC,0x31,0x5C,0x9D,0xBE,0xA6,0x9B,0x44
,0xBC,0x31,0x08,0x2C,0x3B,0x58,0xE3,0x00,0x3F,0x16,0x85,0x9D,0x3E,0xA6,0x9B
,0x44,0xBC,0x27,0x31,0xC8,0x3E,0xA6,0x9B,0x44,0xBC,0x96,0x43,0x8B,0x3C,0x52
,0xB8,0x1E,0x3F,0x00,0x01,0x02,0x00,0x02,0x03,0x04,0x05,0x06,0x03,0x04,0x06
,0x00,0x03,0x06,0x06,0x5F,0x04,0x02,0x01,0xB2,0x9D,0xEF,0x3E,0x0C,0x02,0x6B
,0xBF,0xB2,0x9D,0xEF,0xBE,0x0C,0x02,0x6B,0xBF,0xB2,0x9D,0xEF,0xBE,0xA4,0x70
,0x7D,0xBF,0xB2,0x9D,0xEF,0x3E,0xA4,0x70,0x7D,0xBF,0x02,0x03,0x00,0x02,0x00
,0x01,0x03,0x60,0x04,0x02,0x01,0x9D,0x64,0xEB,0x3D,0x2F,0xDD,0xA4,0x3E,0x06
,0x81,0x15,0x3E,0x2F,0xDD,0xA4,0x3E,0xCD,0xCC,0x4C,0x3D,0x89,0x41,0x20,0x3F
,0x06,0x81,0x15,0xBE,0x89,0x41,0x20,0x3F,0x02,0x03,0x00,0x01,0x02,0x00,0x03
,0x61,0x49,0x49,0x02,0xA9,0x6C,0xB8,0x3D,0xF5,0x12,0x0B,0xBF,0xE5,0x28,0xE0
,0x3D,0x23,0x31,0x15,0xBF,0x9F,0xC9,0x0E,0x3E,0x4B,0xCD,0x1A,0xBF,0x2B,0x87
,0x36,0x3E,0x08,0xAC,0x1C,0xBF,0xC1,0x74,0x8A,0x3E,0xF6,0x28,0x14,0xBF,0x6D
,0xE7,0xBB,0x3E,0x7D,0x3F,0xF5,0xBE,0x6D,0xE7,0xBB,0x3E,0xF6,0x28,0xDC,0xBE
,0xB6,0x49,0xA5,0x3E,0xB0,0xC6,0xF1,0xBE,0xCD,0x74,0x97,0x3E,0x2F,0xDD,0xFC
,0xBE,0x9A,0x99,0x91,0x3E,0x54,0x37,0xFF,0xBE,0x27,0xD9,0x8A,0x3E,0x00,0x00
,0x00,0xBF,0x80,0x7D,0x84,0x3E,0xD2,0xE0,0xFE,0xBE,0x23,0xD7,0x7D,0x3E,0x06
,0x83,0xFB,0xBE,0xBA,0xF3,0x74,0x3E,0xFA,0xD5,0xF4,0xBE,0x0D,0xFE,0x6E,0x3E
,0xA8,0xC8,0xE9,0xBE,0xCF,0x9E,0x6B,0x3E,0x70,0x27,0xD1,0xBE,0xFA,0x7E,0x6A
,0x3E,0x6E,0xBE,0xA1,0xBE,0xFA,0x7E,0x6A,0x3E,0x6B,0x47,0x31,0xBD,0x6E,0x6C
,0x66,0x3E,0x62,0x85,0x5B,0x3D,0x50,0x35,0x5A,0x3E,0x39,0x64,0xE3,0x3D,0x02
,0xD3,0x39,0x3E,0x7F,0x14,0x25,0x3E,0x7A,0x6C,0x0B,0x3E,0xBD,0xC4,0x48,0x3E
,0x0F,0x43,0x6B,0x3D,0x4E,0x5E,0x64,0x3E,0xF7,0xC8,0x26,0xBD,0x68,0x91,0x6D
,0x3E,0xD8,0x9E,0x29,0xBE,0xE7,0x55,0x5D,0x3E,0x31,0xB2,0x84,0xBE,0xE8,0xA3
,0x2C,0x3E,0xFF,0x94,0xA2,0xBE,0xDA,0x1A,0xD1,0x3D,0x44,0x8B,0xAC,0xBE,0xDD
,0x25,0xF1,0x3C,0xB9,0xDF,0xA9,0xBE,0xC2,0xDE,0xC4,0xBB,0x3C,0xDD,0xA1,0xBE
,0x93,0x8D,0x07,0xBD,0x61,0xA6,0x95,0xBE,0x10,0xEB,0x4D,0xBD,0x46,0x5E,0x86
,0xBE,0x42,0x60,0x65,0xBD,0x1B,0x80,0x6D,0xBE,0x17,0xD9,0x4E,0xBD,0xFD,0xD8
,0x54,0xBE,0x96,0x43,0x0B,0xBD,0x02,0xD4,0x44,0xBE,0x34,0x30,0xF2,0xBB,0xEE
,0x7C,0x3F,0xBE,0xB6,0xF4,0xE8,0x3C,0xCE,0x4F,0x41,0xBE,0xA5,0xC0,0xA2,0x3D
,0x41,0xBA,0x38,0xBE,0x05,0x89,0xED,0x3D,0xDB,0xF9,0x1E,0xBE,0xF4,0xA5,0x17
,0x3E,0xC5,0x53,0xEF,0xBD,0x95,0x9D,0x2E,0x3E,0x4C,0x37,0x89,0xBD,0xA2,0x45
,0x36,0x3E,0xF2,0xCE,0x21,0xBB,0xA3,0xC8,0x2A,0x3E,0xB4,0xC8,0x36,0x3D,0xEC
,0x51,0x08,0x3E,0xE3,0x55,0x96,0x3D,0xD7,0xFB,0x8D,0x3D,0xE7,0xFB,0xA9,0x3D
,0xA3,0x5A,0x04,0xBD,0xE7,0xFB,0xA9,0x3D,0x6F,0x12,0x83,0xBD,0x4D,0xBC,0xC3
,0xBD,0xDF,0x53,0x09,0xBE,0xBF,0xF3,0x5B,0xBE,0x42,0x60,0x45,0xBE,0x7A,0xFE
,0x94,0xBE,0x31,0x5C,0x7D,0xBE,0x14,0x5A,0xAE,0xBE,0xE1,0x7A,0x9C,0xBE,0x1F
,0x84,0xB8,0xBE,0xC3,0xA0,0xB4,0xBE,0x6D,0xE7,0xBB,0xBE,0x0E,0xD9,0xD0,0xBE
,0x4D,0xF5,0xB4,0xBE,0x77,0x13,0xFC,0xBE,0xAC,0x1E,0xA0,0xBE,0x2D,0x06,0x0F
,0xBF,0xD2,0x8D,0x80,0xBE,0x2D,0x07,0x1A,0xBF,0x06,0xD9,0x32,0xBE,0x2D,0xB2
,0x1D,0xBF,0x8D,0x9B,0xFA,0xBD,0xBA,0x49,0x1C,0xBF,0x97,0xE3,0x95,0xBD,0x62
,0x10,0x18,0xBF,0x06,0x81,0x95,0xBC,0x52,0x0D,0x0F,0xBF,0xE7,0xFB,0xA9,0x3D
,0x29,0x5C,0xF7,0xBE,0x5A,0x9E,0x87,0xBC,0x43,0xE1,0xFB,0xBE,0x8A,0x1C,0xC2
,0xBD,0x81,0x95,0x03,0xBF,0x77,0x12,0x11,0xBE,0x25,0xE7,0x00,0xBF,0xF1,0x2A
,0x3B,0xBE,0x45,0xB8,0xF1,0xBE,0x0A,0x85,0x58,0xBE,0x7C,0x81,0xD9,0xBE,0xD3
,0x4D,0x62,0xBE,0x98,0xC2,0xBB,0xBE,0x53,0x91,0x5A,0xBE,0xBC,0x08,0xA3,0xBE
,0x18,0x5C,0x43,0xBE,0x04,0x58,0x8C,0xBE,0xED,0x2C,0x1A,0xBE,0x6C,0x3E,0x6E
,0xBE,0x46,0x06,0xB9,0xBD,0xAA,0x99,0x45,0xBE,0x56,0xD4,0xE0,0xBC,0x97,0x1C
,0x27,0xBE,0xE7,0xFB,0xA9,0x3D,0xFC,0xA9,0xF1,0xBD,0xE7,0xFB,0xA9,0x3D,0x29
,0x04,0xDA,0xBE,0x01,0x47,0x48,0x00,0x01,0x48,0x3B,0x00,0x48,0x05,0x06,0x07
,0x04,0x05,0x07,0x04,0x07,0x08,0x04,0x08,0x09,0x3B,0x48,0x3C,0x3A,0x3B,0x3C
,0x3A,0x3C,0x3D,0x39,0x3A,0x3D,0x04,0x09,0x0A,0x38,0x39,0x3D,0x38,0x3D,0x3E
,0x37,0x38,0x3E,0x36,0x37,0x3E,0x36,0x3E,0x3F,0x35,0x36,0x3F,0x34,0x35,0x3F
,0x34,0x3F,0x40,0x33,0x34,0x40,0x33,0x40,0x41,0x32,0x33,0x41,0x31,0x32,0x41
,0x31,0x41,0x42,0x30,0x31,0x42,0x30,0x42,0x43,0x2F,0x30,0x43,0x2F,0x43,0x44
,0x2E,0x2F,0x44,0x2E,0x44,0x45,0x2E,0x45,0x46,0x2D,0x2E,0x46,0x2D,0x46,0x47
,0x04,0x0A,0x0B,0x03,0x04,0x0B,0x03,0x0B,0x0C,0x02,0x03,0x0C,0x01,0x02,0x0C
,0x01,0x0C,0x0D,0x47,0x01,0x0D,0x47,0x0D,0x0E,0x47,0x0E,0x0F,0x47,0x0F,0x10
,0x47,0x10,0x11,0x2D,0x47,0x11,0x2C,0x2D,0x11,0x2C,0x11,0x12,0x2B,0x2C,0x12
,0x2B,0x12,0x13,0x2B,0x13,0x14,0x2A,0x2B,0x14,0x2A,0x14,0x15,0x2A,0x15,0x16
,0x29,0x2A,0x16,0x29,0x16,0x17,0x28,0x29,0x17,0x28,0x17,0x18,0x27,0x28,0x18
,0x26,0x27,0x18,0x26,0x18,0x19,0x25,0x26,0x19,0x25,0x19,0x1A,0x24,0x25,0x1A
,0x24,0x1A,0x1B,0x23,0x24,0x1B,0x23,0x1B,0x1C,0x22,0x23,0x1C,0x22,0x1C,0x1D
,0x22,0x1D,0x1E,0x22,0x1E,0x1F,0x22,0x1F,0x20,0x21,0x22,0x20,0x3B,0x48,0x62
,0x2C,0x2C,0x02,0x29,0x5C,0x0F,0xBE,0x8B,0x6C,0x27,0x3F,0xBE,0xF7,0x37,0xBE
,0x8B,0x6C,0x27,0x3F,0x62,0x10,0xD8,0xBE,0x04,0x56,0x0E,0x3F,0xB0,0xC6,0xD1
,0xBE,0xDD,0x24,0x06,0x3F,0xF2,0x7A,0xC0,0xBE,0x94,0xD9,0x08,0x3F,0xE3,0x51
,0xB2,0xBE,0x87,0xC0,0x09,0x3F,0x2A,0x72,0xA8,0xBE,0xE6,0x04,0x09,0x3F,0x25
,0x5C,0xA0,0xBE,0xF2,0xD1,0x06,0x3F,0xB0,0x1F,0x9A,0xBE,0x81,0x96,0x02,0x3F
,0xC9,0xCC,0x95,0xBE,0x17,0x83,0xF7,0x3E,0x35,0x46,0x93,0xBE,0xB8,0xB1,0xD9
,0x3E,0x98,0x6E,0x92,0xBE,0x9C,0xC4,0xA0,0x3E,0x98,0x6E,0x92,0xBE,0xFC,0xA9
,0x05,0xBF,0x10,0x04,0x58,0xBE,0xEC,0x12,0x11,0xBF,0x49,0xBC,0x0C,0xBE,0x54
,0x39,0x19,0xBF,0x99,0x9D,0x85,0xBD,0x32,0x1D,0x1E,0xBF,0x06,0x7F,0xBF,0x3B
,0x77,0xBE,0x1F,0xBF,0x0B,0x27,0x19,0x3E,0xA1,0x85,0x18,0xBF,0x89,0xED,0x8E
,0x3E,0x1F,0xDB,0x02,0xBF,0xAC,0xC7,0xC5,0x3E,0x77,0xBE,0xB7,0xBE,0x62,0x10
,0xD8,0x3E,0x19,0xAC,0x28,0xBE,0xDE,0x39,0xCC,0x3E,0x0F,0x7E,0xE2,0x37,0x52
,0xB6,0xA8,0x3E,0xCD,0xC8,0x00,0x3E,0xAB,0x98,0x6A,0x3E,0x20,0x5F,0x52,0x3E
,0xFA,0x09,0xE7,0x3D,0x68,0x91,0x6D,0x3E,0x03,0x40,0x95,0xBC,0xA1,0x9D,0x43
,0x3E,0x29,0x5C,0x0F,0xBE,0x1F,0x85,0x8B,0x3D,0xF6,0xB3,0xB8,0xBD,0x02,0x83
,0x64,0x3D,0xDC,0x0E,0x4D,0xBD,0x6F,0x62,0xA8,0x3D,0xCB,0xA0,0x5A,0xBB,0xA3
,0xE9,0xCC,0x3D,0x7D,0xCA,0x31,0x3D,0x87,0x16,0xD9,0x3D,0x33,0xDF,0x01,0x3E
,0x86,0x76,0xAE,0x3D,0x89,0x99,0x4D,0x3E,0x35,0x5E,0xBA,0x3C,0x3D,0x61,0x81
,0x3E,0xF7,0xAB,0xA0,0xBD,0x71,0x3D,0x8A,0x3E,0x49,0xBC,0x5C,0xBE,0x3D,0x61
,0x81,0x3E,0x74,0x40,0xBA,0xBE,0x89,0x99,0x4D,0x3E,0xAA,0x9D,0xF1,0xBE,0x8E
,0xE7,0x03,0x3E,0xDC,0xB8,0x09,0xBF,0x16,0x4E,0x52,0x3D,0x29,0x5C,0x0F,0xBF
,0xBC,0x77,0xD4,0x3A,0x0B,0xEC,0x0D,0xBF,0x4B,0x72,0x40,0xBD,0xA2,0x9B,0x09
,0xBF,0x6F,0x0C,0xC1,0xBD,0xDD,0x79,0x02,0xBF,0x29,0x5C,0x0F,0xBE,0xFA,0x2A
,0xF1,0xBE,0x29,0x5C,0x0F,0xBE,0x12,0x83,0x40,0x3C,0x01,0x2A,0x2B,0x1A,0x00
,0x01,0x1A,0x01,0x2B,0x1A,0x2B,0x1B,0x19,0x1A,0x1B,0x19,0x1B,0x1C,0x19,0x1C
,0x1D,0x01,0x02,0x03,0x01,0x03,0x04,0x01,0x04,0x05,0x01,0x05,0x06,0x01,0x06
,0x07,0x19,0x1D,0x1E,0x18,0x19,0x1E,0x18,0x1E,0x1F,0x17,0x18,0x1F,0x17,0x1F
,0x20,0x16,0x17,0x20,0x15,0x16,0x20,0x15,0x20,0x21,0x14,0x15,0x21,0x14,0x21
,0x22,0x13,0x14,0x22,0x13,0x22,0x23,0x12,0x13,0x23,0x12,0x23,0x24,0x11,0x12
,0x24,0x11,0x24,0x25,0x10,0x11,0x25,0x10,0x25,0x26,0x10,0x26,0x27,0x0F,0x10
,0x27,0x0F,0x27,0x28,0x0E,0x0F,0x28,0x0E,0x28,0x29,0x0D,0x0E,0x29,0x0D,0x29
,0x2A,0x0C,0x0D,0x2A,0x0B,0x0C,0x2A,0x0B,0x2A,0x01,0x0A,0x0B,0x01,0x09,0x0A
,0x01,0x08,0x09,0x01,0x08,0x01,0x07,0x1A,0x2B,0x63,0x29,0x27,0x01,0xC1,0xCA
,0xA1,0x3E,0x29,0x5C,0x8F,0xBE,0xC4,0xCD,0x89,0x3E,0xF5,0xD5,0xBD,0xBE,0xB3
,0x45,0x62,0x3E,0xE1,0x26,0xDB,0xBE,0xB4,0xC8,0x16,0x3E,0x7F,0x69,0xF1,0xBE
,0x5D,0x6A,0x84,0x3D,0xFE,0xD4,0xF8,0xBE,0xBD,0x1C,0x36,0xBD,0x65,0xA8,0xEA
,0xBE,0x81,0x95,0x03,0xBE,0xDD,0x22,0xC0,0xBE,0xFA,0x7E,0x3A,0xBE,0x05,0x17
,0x83,0xBE,0xCD,0xCC,0x4C,0xBE,0xFE,0x5F,0xF5,0xBD,0x15,0x56,0x3A,0xBE,0xD4
,0xD6,0x08,0x3C,0xAA,0xF1,0x02,0xBE,0x11,0xA8,0xDE,0x3D,0x61,0xE1,0x84,0xBD
,0xDE,0x76,0x21,0x3E,0xB0,0x71,0x7D,0x3C,0x0E,0x2D,0x32,0x3E,0xC1,0x72,0x84
,0x3D,0x73,0x66,0x2B,0x3E,0x96,0x93,0xD0,0x3D,0x5E,0x12,0x17,0x3E,0xF7,0x78
,0x01,0x3E,0x07,0x7D,0xE9,0x3D,0x3B,0x37,0x0D,0x3E,0x1F,0x85,0x8B,0x3D,0x9F
,0x1C,0x15,0x3E,0xE1,0x40,0xC8,0x3C,0xA5,0xC0,0x22,0x3E,0xF2,0xCE,0x21,0xBB
,0x9D,0xF3,0x43,0x3E,0x0C,0x3D,0xE2,0xBC,0x69,0xE1,0x72,0x3E,0xBC,0x74,0x13
,0xBD,0xA6,0x47,0x8B,0x3E,0xC7,0x9B,0xFC,0xBC,0xF7,0xAB,0x98,0x3E,0x67,0x65
,0x7B,0xBC,0x4E,0x0C,0xA1,0x3E,0x08,0x22,0x0B,0x3C,0x0A,0xD7,0xA3,0x3E,0x4F
,0x77,0x1E,0x3D,0x38,0xF5,0x99,0x3E,0x11,0x52,0xD7,0x3D,0xC7,0x9F,0x78,0x3E
,0x6D,0x3B,0x2D,0x3E,0xDA,0xA8,0x1E,0x3E,0xEA,0x7B,0x5D,0x3E,0xDC,0x84,0x3B
,0x3D,0x68,0x91,0x6D,0x3E,0xE9,0xF3,0xD1,0xBD,0x93,0x19,0x4F,0x3E,0xB5,0x70
,0x69,0xBE,0x28,0x64,0xE7,0x3D,0x06,0x82,0xA0,0xBE,0xE5,0x9D,0xC3,0xBC,0xA0
,0x1A,0xAF,0xBE,0x0B,0x27,0x49,0xBE,0xBB,0xF2,0xA1,0xBE,0x9D,0xD9,0xBE,0xBE
,0xD3,0xF5,0x74,0xBE,0x1B,0x2E,0x02,0xBF,0xCF,0xA3,0x02,0xBE,0x60,0x5A,0x18
,0xBF,0x72,0x70,0xE9,0x3A,0x77,0xBE,0x1F,0xBF,0x79,0xE9,0xE6,0x3D,0x5F,0x45
,0x1A,0xBF,0xDF,0xA7,0x5A,0x3E,0x16,0xDA,0x09,0xBF,0x95,0x44,0x96,0x3E,0x5E
,0x4D,0xDE,0xBE,0xA0,0x1A,0xAF,0x3E,0x0D,0xAA,0x95,0xBE,0x28,0x00,0x01,0x27
,0x28,0x01,0x27,0x01,0x02,0x26,0x27,0x02,0x26,0x02,0x03,0x25,0x26,0x03,0x25
,0x03,0x04,0x24,0x25,0x04,0x24,0x04,0x05,0x23,0x24,0x05,0x22,0x23,0x05,0x22
,0x05,0x06,0x21,0x22,0x06,0x21,0x06,0x07,0x20,0x21,0x07,0x20,0x07,0x08,0x1F
,0x20,0x08,0x1F,0x08,0x09,0x1E,0x1F,0x09,0x1E,0x09,0x0A,0x1D,0x1E,0x0A,0x1D
,0x0A,0x0B,0x1C,0x1D,0x0B,0x1C,0x0B,0x0C,0x1C,0x0C,0x0D,0x1B,0x1C,0x0D,0x1B
,0x0D,0x0E,0x1A,0x1B,0x0E,0x1A,0x0E,0x0F,0x1A,0x0F,0x10,0x19,0x1A,0x10,0x19
,0x10,0x11,0x18,0x19,0x11,0x18,0x11,0x12,0x18,0x12,0x13,0x18,0x13,0x14,0x18
,0x14,0x15,0x17,0x18,0x15,0x17,0x15,0x16,0x28,0x64,0x38,0x38,0x02,0x98,0x6E
,0x12,0x3E,0x77,0xBE,0x1F,0xBF,0x6D,0xE7,0x3B,0x3E,0x77,0xBE,0x1F,0xBF,0x75
,0x93,0xD8,0x3E,0xF0,0xA7,0x06,0xBF,0xE3,0xA5,0xD3,0x3E,0x91,0xED,0xFC,0xBE
,0xE5,0x7C,0xC1,0x3E,0x8F,0x56,0x01,0xBF,0x3A,0x5C,0xB3,0x3E,0xCB,0x4B,0x02
,0xBF,0x19,0x1B,0xAA,0x3E,0xCF,0x81,0x01,0xBF,0x83,0x16,0xA2,0x3E,0xBB,0x47
,0xFE,0xBE,0xBC,0xB2,0x9B,0x3E,0x3F,0x8B,0xF5,0xBE,0x09,0x54,0x97,0x3E,0xC7
,0xF5,0xE7,0xBE,0xE6,0xCE,0x94,0x3E,0x4A,0xCE,0xC9,0xBE,0xCF,0xF7,0x93,0x3E
,0x29,0x5C,0x8F,0xBE,0xCF,0xF7,0x93,0x3E,0x8B,0x6C,0x27,0x3F,0x29,0x5C,0x7F
,0x3E,0x8B,0x6C,0x27,0x3F,0xBC,0x74,0x13,0x3C,0x04,0x56,0x0E,0x3F,0x79,0xE9
,0xA6,0x3C,0xDD,0x24,0x06,0x3F,0x4F,0x77,0x5E,0x3D,0x94,0xD9,0x08,0x3F,0x49
,0x81,0xA5,0x3D,0x87,0xC0,0x09,0x3F,0xFC,0xFB,0xCC,0x3D,0x18,0x05,0x09,0x3F
,0x7C,0x47,0xED,0x3D,0xED,0xD2,0x06,0x3F,0x94,0x15,0x03,0x3E,0x5F,0x9A,0x02
,0x3F,0x67,0xB6,0x0B,0x3E,0x89,0x97,0xF7,0x3E,0x6A,0xC0,0x10,0x3E,0x2E,0xC9
,0xD9,0x3E,0x98,0x6E,0x12,0x3E,0x9C,0xC4,0xA0,0x3E,0x98,0x6E,0x12,0x3E,0x46
,0xB6,0x23,0x3E,0x40,0x85,0x83,0x3D,0xB1,0x1A,0x5B,0x3E,0x05,0x6C,0x07,0xBD
,0x68,0x91,0x6D,0x3E,0x91,0x9B,0x41,0xBE,0x46,0xB5,0x48,0x3E,0x29,0xB0,0xA0
,0xBE,0xBC,0x41,0xB4,0x3D,0xA2,0x9A,0xCA,0xBE,0x99,0xBC,0x81,0xBD,0x75,0x93
,0xD8,0xBE,0xEC,0xF9,0x6A,0xBE,0xCB,0xF4,0xCB,0xBE,0x18,0x42,0xC6,0xBE,0x8B
,0x18,0xA6,0xBE,0xB0,0x73,0x03,0xBF,0x00,0xAC,0x5E,0xBE,0xC5,0xAB,0x18,0xBF
,0x41,0x10,0xC0,0xBD,0x77,0xBE,0x1F,0xBF,0x55,0xBD,0xFC,0xBC,0x97,0x02,0x1E
,0xBF,0xF2,0xD1,0xE2,0x3C,0xD5,0xCE,0x18,0xBF,0xE7,0x70,0xAD,0x3D,0xDE,0x8F
,0x0F,0xBF,0x98,0x6E,0x12,0x3E,0x2D,0xB2,0x01,0xBF,0x28,0x0C,0x8A,0x3D,0xFE
,0x7E,0x01,0xBF,0xD1,0x24,0x31,0xBC,0xF0,0xA7,0x06,0xBF,0xB7,0x7A,0xCE,0xBD
,0xCB,0xA2,0x00,0xBF,0xD9,0x76,0x3A,0xBE,0xFA,0x26,0xDD,0xBE,0x0E,0x87,0x75
,0xBE,0x6E,0x68,0xA2,0xBE,0xA6,0x9B,0x84,0xBE,0xF4,0xFD,0x24,0xBE,0x16,0xDB
,0x74,0xBE,0xD5,0x78,0xE9,0xBB,0xB9,0xC6,0x37,0xBE,0xB4,0xC8,0xD6,0x3D,0x5F
,0x95,0xEB,0xBD,0xF2,0x7A,0x20,0x3E,0xD8,0x2E,0x2D,0xBD,0x0E,0x2D,0x32,0x3E
,0x8B,0xDE,0x29,0xBA,0xDA,0xCA,0x2B,0x3E,0x34,0x49,0x2C,0x3D,0xF9,0xA3,0x18
,0x3E,0x2A,0x00,0xA6,0x3D,0x22,0xE3,0xF1,0x3D,0x5D,0x6A,0xE4,0x3D,0x23,0xDB
,0x99,0x3D,0x2D,0xB0,0x07,0x3E,0xC9,0x76,0xBE,0x3C,0x98,0x6E,0x12,0x3E,0xCD
,0xCC,0x0C,0xBD,0x98,0x6E,0x12,0x3E,0x31,0x08,0xE4,0xBE,0x01,0x36,0x37,0x02
,0x03,0x04,0x01,0x02,0x04,0x01,0x04,0x05,0x01,0x05,0x06,0x01,0x06,0x07,0x0B
,0x0C,0x0D,0x26,0x00,0x01,0x26,0x01,0x37,0x26,0x37,0x27,0x25,0x26,0x27,0x0E
,0x0F,0x10,0x0D,0x0E,0x10,0x0D,0x10,0x11,0x0D,0x11,0x12,0x0D,0x12,0x13,0x25
,0x27,0x28,0x24,0x25,0x28,0x23,0x24,0x28,0x23,0x28,0x29,0x22,0x23,0x29,0x21
,0x22,0x29,0x21,0x29,0x2A,0x20,0x21,0x2A,0x1F,0x20,0x2A,0x1F,0x2A,0x2B,0x1E
,0x1F,0x2B,0x1E,0x2B,0x2C,0x1D,0x1E,0x2C,0x1D,0x2C,0x2D,0x1C,0x1D,0x2D,0x1C
,0x2D,0x2E,0x1B,0x1C,0x2E,0x1B,0x2E,0x2F,0x1A,0x1B,0x2F,0x1A,0x2F,0x30,0x19
,0x1A,0x30,0x19,0x30,0x31,0x19,0x31,0x32,0x18,0x19,0x32,0x18,0x32,0x33,0x18
,0x33,0x34,0x0D,0x13,0x14,0x01,0x07,0x08,0x36,0x01,0x08,0x36,0x08,0x09,0x36
,0x09,0x0A,0x36,0x0A,0x0B,0x36,0x0B,0x0D,0x35,0x36,0x0D,0x0D,0x14,0x15,0x18
,0x34,0x35,0x18,0x35,0x0D,0x17,0x18,0x0D,0x16,0x17,0x0D,0x16,0x0D,0x15,0x26
,0x37,0x65,0x25,0x25,0x02,0xA0,0x1A,0xAF,0x3E,0xA6,0x9B,0xC4,0xBD,0x7D,0xEB
,0xA3,0x3E,0x2A,0x71,0x1D,0x3D,0x35,0x5E,0x82,0x3E,0x0E,0x2D,0x12,0x3E,0xFE
,0xD6,0x1E,0x3E,0x73,0xB8,0x56,0x3E,0x91,0x62,0x00,0x3D,0x68,0x91,0x6D,0x3E
,0xF3,0x05,0xED,0xBD,0x8E,0xEB,0x4F,0x3E,0xCE,0x4F,0x71,0xBE,0xF9,0xF3,0xED
,0x3D,0xF2,0x7D,0xA1,0xBE,0xCA,0x8C,0xB7,0xBC,0xA0,0x1A,0xAF,0xBE,0x40,0x85
,0x53,0xBE,0x74,0xD4,0xA1,0xBE,0x49,0xA0,0xC1,0xBE,0x21,0x04,0x74,0xBE,0xA0
,0xC5,0x02,0xBF,0xF5,0x2E,0xFE,0xBD,0x41,0x80,0x18,0xBF,0x5E,0xBB,0x34,0x3C
,0x77,0xBE,0x1F,0xBF,0xFF,0x25,0x09,0x3E,0xB3,0x5D,0x19,0xBF,0xAE,0xF3,0x6F
,0x3E,0x69,0x3B,0x06,0xBF,0xEA,0x93,0x9C,0x3E,0xB0,0x74,0xD6,0xBE,0xA0,0x1A
,0xAF,0x3E,0xEA,0x7A,0x9A,0xBE,0x9C,0xC4,0xA0,0x3E,0x73,0x68,0x91,0xBE,0xE9
,0x0E,0x8A,0x3E,0xF1,0x29,0xC0,0xBE,0x04,0x02,0x5D,0x3E,0x3B,0x8B,0xDE,0xBE
,0x83,0xC3,0x1B,0x3E,0x1F,0x30,0xEF,0xBE,0x7F,0xDF,0x9F,0x3D,0x6A,0xBC,0xF4
,0xBE,0xF6,0x95,0x07,0xBD,0x67,0xBA,0xE7,0xBE,0xAE,0xEF,0x03,0xBE,0x39,0xB4
,0xC0,0xBE,0x00,0x56,0x47,0xBE,0xBC,0xCA,0x82,0xBE,0x1B,0x2F,0x5D,0xBE,0xA6
,0x9B,0xC4,0xBD,0xB2,0xF3,0x46,0xBE,0xA3,0x95,0x3B,0x3D,0x83,0x18,0x18,0xBE
,0x2F,0x52,0xE8,0x3D,0xE9,0xF3,0xB1,0xBD,0x80,0x99,0x1F,0x3E,0x2C,0x9D,0x8F
,0xBC,0x7B,0x14,0x2E,0x3E,0x9C,0x6D,0xEE,0x3C,0x47,0xE5,0x26,0x3E,0xA1,0x12
,0x97,0x3D,0xEF,0x57,0x11,0x3E,0x69,0x6E,0xE5,0x3D,0x65,0x6E,0xDE,0x3D,0xF6
,0xD0,0x0E,0x3E,0xA6,0x9B,0x84,0x3D,0xBC,0xCA,0x1A,0x3E,0x11,0x6E,0xB2,0x3C
,0xC1,0xCA,0x21,0x3E,0x58,0x39,0x34,0xBD,0x1B,0x2F,0x5D,0xBE,0x58,0x39,0x34
,0xBD,0x00,0x21,0x22,0x21,0x00,0x01,0x21,0x01,0x02,0x00,0x22,0x23,0x19,0x00
,0x23,0x19,0x23,0x24,0x21,0x02,0x03,0x20,0x21,0x03,0x1F,0x20,0x03,0x1F,0x03
,0x04,0x1E,0x1F,0x04,0x1E,0x04,0x05,0x1D,0x1E,0x05,0x1C,0x1D,0x05,0x10,0x11
,0x12,0x0F,0x10,0x12,0x0F,0x12,0x13,0x0E,0x0F,0x13,0x0E,0x13,0x14,0x0D,0x0E
,0x14,0x0D,0x14,0x15,0x0C,0x0D,0x15,0x0C,0x15,0x16,0x0B,0x0C,0x16,0x1C,0x05
,0x06,0x1B,0x1C,0x06,0x1A,0x1B,0x06,0x1A,0x06,0x07,0x24,0x1A,0x07,0x0A,0x0B
,0x16,0x0A,0x16,0x17,0x09,0x0A,0x17,0x09,0x17,0x18,0x08,0x09,0x18,0x08,0x18
,0x19,0x07,0x08,0x19,0x24,0x07,0x19,0x19,0x24,0x66,0x35,0x33,0x01,0x08,0xAC
,0x1C,0x3E,0xBC,0x74,0x13,0x3E,0x08,0xAC,0x1C,0x3E,0xF4,0xFD,0x54,0x3E,0xAE
,0x47,0x61,0xBD,0xF4,0xFD,0x54,0x3E,0xAE,0x47,0x61,0xBD,0x4E,0x62,0x88,0x3E
,0xF8,0xA7,0x54,0xBD,0x40,0xDF,0xD6,0x3E,0xD6,0xC8,0x2E,0xBD,0xBA,0x49,0x00
,0x3F,0x29,0x21,0xD8,0xBC,0xBA,0x6A,0x0A,0x3F,0x1D,0x58,0x0E,0xBB,0xF2,0x7D
,0x11,0x3F,0x60,0x3D,0xEE,0x3C,0x2A,0xA7,0x15,0x3F,0x27,0x31,0x88,0x3D,0x3D
,0x0A,0x17,0x3F,0xE4,0x84,0xC9,0x3D,0xBE,0xF5,0x15,0x3F,0xB3,0x45,0x02,0x3E
,0x52,0xB8,0x12,0x3F,0x92,0x95,0x1F,0x3E,0xC0,0x5F,0x0C,0x3F,0x92,0x95,0x3F
,0x3E,0xF0,0xFD,0x01,0x3F,0x11,0x57,0x5E,0x3E,0x84,0x81,0xEF,0x3E,0x6C,0xEB
,0x77,0x3E,0x0C,0xAC,0xE3,0x3E,0x95,0x2C,0x87,0x3E,0xBB,0x09,0xDE,0x3E,0x02
,0xD3,0x91,0x3E,0xF6,0x28,0xDC,0x3E,0xE2,0xE4,0x9E,0x3E,0x1B,0x2D,0xDF,0x3E
,0x56,0x10,0xAB,0x3E,0x47,0x39,0xE8,0x3E,0x94,0xDA,0xB3,0x3E,0x0C,0xAF,0xF4
,0x3E,0xB4,0xC8,0xB6,0x3E,0xCB,0xF7,0x00,0x3F,0x79,0x3D,0xB0,0x3E,0x01,0x16
,0x0D,0x3F,0xA6,0x9B,0x9C,0x3E,0xAE,0xF3,0x17,0x3F,0xD8,0xF3,0x65,0x3E,0x5C
,0x8E,0x23,0x3F,0x0C,0x02,0x0B,0x3E,0x8B,0x6C,0x27,0x3F,0xCB,0x49,0x28,0x3D
,0x71,0x1C,0x24,0x3F,0xA2,0xD0,0x32,0xBD,0x02,0x2C,0x1A,0x3F,0x96,0xCE,0xE7
,0xBD,0x66,0xDA,0x0A,0x3F,0xBD,0x1C,0x26,0xBE,0xC4,0xCC,0xEE,0x3E,0x75,0xE5
,0x43,0xBE,0x04,0x00,0xBF,0x3E,0xF2,0xD2,0x4D,0xBE,0x1D,0xAE,0x85,0x3E,0xF2
,0xD2,0x4D,0xBE,0xF4,0xFD,0x54,0x3E,0xB4,0xC8,0xB6,0xBE,0xF4,0xFD,0x54,0x3E
,0xB4,0xC8,0xB6,0xBE,0xBC,0x74,0x13,0x3E,0xF2,0xD2,0x4D,0xBE,0xBC,0x74,0x13
,0x3E,0xF2,0xD2,0x4D,0xBE,0xA6,0x9B,0xC4,0xBE,0xFC,0xA6,0x50,0xBE,0xA0,0x1B
,0xEA,0xBE,0xD9,0x22,0x59,0xBE,0xE9,0x7C,0x00,0xBF,0x7B,0x15,0x69,0xBE,0x81
,0x76,0x07,0xBF,0xE9,0x26,0x81,0xBE,0x0A,0xD6,0x0C,0xBF,0x44,0xE0,0x90,0xBE
,0xD4,0x43,0x10,0xBF,0x1F,0x31,0xA2,0xBE,0x73,0x68,0x11,0xBF,0x6A,0xBC,0xB4
,0xBE,0x73,0x68,0x11,0xBF,0x6A,0xBC,0xB4,0xBE,0x9A,0x99,0x19,0xBF,0xDD,0x24
,0x06,0x3E,0x9A,0x99,0x19,0xBF,0xDD,0x24,0x06,0x3E,0x73,0x68,0x11,0xBF,0x1F
,0x85,0x6B,0x3D,0x73,0x68,0x11,0xBF,0x9C,0xDB,0x04,0x3C,0x5C,0xE4,0x0E,0xBF
,0xBA,0x4A,0xF7,0xBC,0x19,0x58,0x07,0xBF,0x5D,0xDF,0x47,0xBD,0x6F,0x66,0xF4
,0xBE,0xAE,0x47,0x61,0xBD,0xA6,0x9B,0xC4,0xBE,0xAE,0x47,0x61,0xBD,0xBC,0x74
,0x13,0x3E,0x2D,0x2E,0x2F,0x2C,0x2D,0x2F,0x2C,0x2F,0x30,0x2C,0x30,0x31,0x2C
,0x31,0x32,0x2A,0x2B,0x2C,0x21,0x22,0x23,0x20,0x21,0x23,0x00,0x01,0x02,0x34
,0x00,0x02,0x29,0x2A,0x2C,0x29,0x2C,0x32,0x28,0x29,0x32,0x27,0x28,0x32,0x26
,0x27,0x32,0x25,0x26,0x32,0x25,0x32,0x33,0x24,0x25,0x33,0x23,0x24,0x33,0x23
,0x33,0x34,0x20,0x23,0x34,0x20,0x34,0x02,0x1F,0x20,0x02,0x1F,0x02,0x03,0x1E
,0x1F,0x03,0x1E,0x03,0x04,0x1D,0x1E,0x04,0x1D,0x04,0x05,0x1C,0x1D,0x05,0x1B
,0x1C,0x05,0x1B,0x05,0x06,0x1B,0x06,0x07,0x1A,0x1B,0x07,0x1A,0x07,0x08,0x1A
,0x08,0x09,0x19,0x1A,0x09,0x19,0x09,0x0A,0x19,0x0A,0x0B,0x18,0x19,0x0B,0x18
,0x0B,0x0C,0x17,0x18,0x0C,0x17,0x0C,0x0D,0x17,0x0D,0x0E,0x16,0x17,0x0E,0x15
,0x16,0x0E,0x15,0x0E,0x0F,0x14,0x15,0x0F,0x13,0x14,0x0F,0x12,0x13,0x0F,0x12
,0x0F,0x10,0x11,0x12,0x10,0x34,0x67,0x67,0x69,0x03,0xED,0xD2,0x86,0xBE,0x3D
,0x9E,0xBE,0xBE,0x93,0x18,0x9C,0xBE,0xB2,0x47,0xD8,0xBE,0x7F,0xBE,0xA5,0xBE
,0x8C,0x2D,0xEC,0xBE,0xC3,0xF5,0xA8,0xBE,0x31,0x5C,0xFD,0xBE,0xA1,0x83,0xA6
,0xBE,0x2A,0x52,0x05,0xBF,0x1B,0x2D,0x9F,0xBE,0x58,0x3A,0x0B,0xBF,0xFF,0xEA
,0x91,0xBE,0x96,0x22,0x11,0xBF,0x7A,0x6C,0x7B,0xBE,0xA8,0xC6,0x17,0xBF,0x83
,0x14,0xAC,0xBE,0x17,0x2D,0x30,0xBF,0x72,0x14,0xC0,0xBE,0x32,0x73,0x3D,0xBF
,0x5C,0xE3,0xCB,0xBE,0xB4,0x72,0x47,0xBF,0xE8,0xA4,0xCF,0xBE,0xB5,0xDD,0x4C
,0xBF,0x60,0xE5,0xD0,0xBE,0xC3,0x49,0x52,0xBF,0xA0,0x6E,0xC8,0xBE,0x13,0x43
,0x5E,0xBF,0x3D,0x0A,0xAF,0xBE,0xB8,0xC8,0x69,0xBF,0x90,0x6A,0x58,0xBE,0xAD
,0x86,0x78,0xBF,0x52,0xB8,0x5E,0xBD,0xA4,0x70,0x7D,0xBF,0x0E,0x87,0x15,0x3E
,0x37,0x34,0x75,0xBF,0x7D,0xEB,0x9B,0x3E,0x02,0x7F,0x5C,0xBF,0xE6,0xCF,0xBF
,0x3E,0x39,0x5E,0x45,0xBF,0xA8,0xC6,0xCB,0x3E,0xF8,0xA7,0x2C,0xBF,0xA5,0x83
,0xC5,0x3E,0xFD,0x9F,0x1B,0xBF,0x5A,0xBA,0xB2,0x3E,0x23,0x86,0x0D,0xBF,0x43
,0x73,0x95,0x3E,0x1A,0x50,0x03,0xBF,0x6A,0x6C,0x5F,0x3E,0x6D,0xE7,0xFB,0xBE
,0xC2,0xA1,0x07,0x3E,0x0C,0x01,0xF8,0xBE,0xD9,0xCD,0x8C,0xBC,0x66,0x12,0xF5
,0xBE,0x9A,0x78,0xC7,0xBD,0xE9,0x7D,0xF3,0xBE,0xE0,0xF7,0x0F,0xBE,0xFC,0xA9
,0xF1,0xBE,0x2B,0x30,0x34,0xBE,0xA5,0x83,0xED,0xBE,0x1F,0x31,0x4A,0xBE,0x00
,0x56,0xE7,0xBE,0x1C,0x5D,0x55,0xBE,0xA7,0x08,0xE0,0xBE,0x87,0x16,0x59,0xBE
,0x12,0x83,0xD8,0xBE,0x85,0x96,0x55,0xBE,0x8A,0x94,0xCE,0xBE,0x7F,0x16,0x4B
,0xBE,0x73,0xBC,0xC2,0xBE,0xE3,0xA6,0x36,0xBE,0x48,0xDF,0xB4,0xBE,0x21,0x58
,0x15,0xBE,0x3F,0xE1,0xA4,0xBE,0xA4,0x1C,0xAC,0xBD,0xB4,0xC7,0xAB,0xBE,0x31
,0x07,0xC1,0xBC,0x7B,0x14,0xAE,0xBE,0xCB,0xD6,0xDA,0x3D,0xD1,0xCC,0xA3,0xBE
,0xBD,0xC8,0x54,0x3E,0xD3,0xF5,0x84,0xBE,0x6F,0x7E,0x8B,0x3E,0xF0,0x51,0x2F
,0xBE,0x2B,0x87,0x96,0x3E,0xDE,0xFF,0x87,0xBD,0x2B,0x34,0x90,0x3E,0xFE,0x99
,0xC1,0x3C,0xD9,0x76,0x7A,0x3E,0xA6,0x9B,0xC4,0x3D,0x2D,0x0A,0xB3,0x3E,0xA6
,0x9B,0xC4,0x3D,0xEA,0x7A,0xC2,0x3E,0xDF,0xF9,0xC5,0x3D,0x43,0xE3,0xC9,0x3E
,0x8C,0x14,0xCA,0x3D,0x97,0xC5,0xCC,0x3E,0xA7,0x74,0xD0,0x3D,0xDF,0xA3,0xCE
,0x3E,0xB5,0xA3,0xD8,0x3D,0xEF,0x54,0xD0,0x3E,0xCE,0x1C,0xF2,0x3D,0x60,0xE5
,0xD0,0x3E,0x76,0x16,0x0D,0x3E,0x5C,0x38,0xD0,0x3E,0xE5,0xD4,0x1E,0x3E,0x2F
,0x31,0xCE,0x3E,0x50,0x35,0x2A,0x3E,0x5F,0x62,0xCC,0x3E,0x37,0xE1,0x2E,0x3E
,0x0C,0xAE,0xC9,0x3E,0xA5,0xF3,0x31,0x3E,0xD1,0x78,0xC2,0x3E,0xDB,0xA7,0x33
,0x3E,0xE1,0x26,0xB3,0x3E,0x58,0x39,0x34,0x3E,0x8B,0x6C,0x37,0x3E,0x58,0x39
,0x34,0x3E,0x29,0x04,0xB2,0x3D,0x64,0x3B,0x5F,0x3E,0xE1,0x79,0xA9,0xBC,0x68
,0x91,0x6D,0x3E,0xBD,0x73,0x18,0xBE,0x93,0x72,0x57,0x3E,0x79,0xE7,0x80,0xBE
,0x98,0x16,0x15,0x3E,0xC5,0x8D,0xA3,0xBE,0x3E,0x20,0x50,0x3D,0xA0,0x1A,0xAF
,0xBE,0x7C,0xB4,0x78,0xBD,0xCC,0x09,0xAA,0xBE,0x52,0x0D,0x0B,0xBE,0x0E,0xD7
,0x9A,0xBE,0x92,0xE9,0x50,0xBE,0xA1,0x2E,0x82,0xBE,0x40,0xF7,0x85,0xBE,0xF7
,0x78,0x41,0xBE,0xE6,0x24,0x9C,0xBE,0xEC,0xA2,0xE8,0x3C,0x15,0xAC,0x31,0x3E
,0xE5,0xD0,0xA2,0x3D,0xD9,0x26,0x05,0x3E,0x77,0xBE,0xFF,0x3D,0xFE,0xB7,0x12
,0x3D,0x29,0x5C,0x0F,0x3E,0x88,0xA1,0xB5,0xBD,0x36,0x5C,0x04,0x3E,0xC7,0xF3
,0x39,0xBE,0x30,0xB8,0xC6,0x3D,0xE9,0xD6,0x7B,0xBE,0x7F,0x2F,0x45,0x3D,0xD1
,0x24,0x91,0xBE,0x11,0x57,0x4E,0xBC,0x50,0x8D,0x97,0xBE,0x60,0xE9,0x9C,0xBD
,0xD1,0x21,0x90,0xBE,0xBC,0x74,0x03,0xBE,0x66,0xBE,0x73,0xBE,0x8E,0x3F,0x31
,0xBE,0x1F,0x83,0x15,0xBE,0x12,0x83,0x40,0xBE,0x3B,0xE0,0xBA,0xBC,0xB6,0x49
,0x35,0xBE,0x98,0x68,0x90,0x3D,0xA1,0x9D,0x13,0xBE,0xAC,0x70,0x0B,0x3E,0x00
,0xA8,0xC2,0xBD,0x79,0x3E,0x33,0x3E,0xE2,0x05,0x11,0xBD,0x12,0x83,0x40,0x3E
,0xBD,0x18,0x8A,0xBD,0xA8,0xC5,0x1C,0xBF,0x08,0xAC,0xDC,0x3D,0x52,0xB8,0x1E
,0xBF,0xF5,0x2A,0x62,0x3E,0x5D,0xA4,0x20,0xBF,0x21,0x04,0x94,0x3E,0x9E,0x99
,0x24,0xBF,0x81,0x42,0xA5,0x3E,0x96,0x43,0x2B,0xBF,0x0C,0x02,0xAB,0x3E,0xE7
,0x51,0x35,0xBF,0xA3,0x59,0xA1,0x3E,0x32,0x1E,0x45,0xBF,0x46,0x60,0x84,0x3E
,0xBF,0xF3,0x53,0xBF,0xD1,0x23,0x26,0x3E,0x45,0xB7,0x5E,0xBF,0xF0,0xA7,0xC6
,0x3C,0xD3,0x4D,0x62,0xBF,0x88,0xBE,0xFB,0xBD,0xD1,0xCD,0x5E,0xBF,0x3E,0x5A
,0x6C,0xBE,0xDB,0x4D,0x54,0xBF,0xFA,0xD4,0x89,0xBE,0x7C,0x7F,0x4B,0xBF,0x4E
,0x62,0x90,0xBE,0x08,0x56,0x41,0xBF,0x45,0xA1,0x8D,0xBE,0x89,0x96,0x38,0xBF
,0x29,0x5E,0x85,0xBE,0x46,0x0C,0x2F,0xBF,0x76,0x6D,0x6F,0xBE,0xD4,0xB7,0x24
,0xBF,0x79,0x91,0x49,0xBE,0x9A,0x99,0x19,0xBF,0x55,0x56,0x05,0x66,0x55,0x05
,0x66,0x05,0x06,0x66,0x06,0x07,0x02,0x03,0x4E,0x65,0x66,0x07,0x65,0x07,0x08
,0x64,0x65,0x08,0x63,0x64,0x08,0x63,0x08,0x09,0x62,0x63,0x09,0x62,0x09,0x0A
,0x61,0x62,0x0A,0x61,0x0A,0x0B,0x61,0x0B,0x0C,0x61,0x0C,0x0D,0x61,0x0D,0x0E
,0x60,0x61,0x0E,0x60,0x0E,0x0F,0x5F,0x60,0x0F,0x5F,0x0F,0x10,0x5E,0x5F,0x10
,0x5E,0x10,0x11,0x5D,0x5E,0x11,0x5D,0x11,0x12,0x5C,0x5D,0x12,0x5B,0x5C,0x12
,0x5B,0x12,0x13,0x5A,0x5B,0x13,0x5A,0x13,0x14,0x59,0x5A,0x14,0x59,0x14,0x15
,0x58,0x59,0x15,0x58,0x15,0x16,0x58,0x16,0x17,0x57,0x58,0x17,0x57,0x17,0x18
,0x56,0x57,0x18,0x56,0x18,0x19,0x56,0x19,0x1A,0x05,0x56,0x1A,0x05,0x1A,0x1B
,0x05,0x1B,0x1C,0x05,0x1C,0x1D,0x05,0x1D,0x1E,0x04,0x05,0x1E,0x03,0x04,0x1E
,0x03,0x1E,0x1F,0x03,0x1F,0x20,0x03,0x20,0x21,0x03,0x21,0x22,0x4E,0x03,0x22
,0x4E,0x22,0x23,0x4E,0x23,0x24,0x4D,0x4E,0x24,0x4D,0x24,0x25,0x4C,0x4D,0x25
,0x4C,0x25,0x26,0x4C,0x26,0x27,0x4B,0x4C,0x27,0x4A,0x4B,0x27,0x4A,0x27,0x28
,0x49,0x4A,0x28,0x49,0x28,0x29,0x48,0x49,0x29,0x48,0x29,0x2A,0x47,0x48,0x2A
,0x47,0x2A,0x2B,0x47,0x2B,0x2C,0x46,0x47,0x2C,0x45,0x46,0x2C,0x30,0x31,0x32
,0x2F,0x30,0x32,0x2E,0x2F,0x32,0x2E,0x32,0x33,0x2D,0x2E,0x33,0x2D,0x33,0x34
,0x2D,0x34,0x35,0x2D,0x35,0x36,0x2D,0x36,0x37,0x2D,0x37,0x38,0x2D,0x38,0x39
,0x2D,0x39,0x3A,0x2C,0x2D,0x3A,0x45,0x2C,0x3A,0x45,0x3A,0x3B,0x45,0x3B,0x3C
,0x54,0x45,0x3C,0x54,0x3C,0x3D,0x53,0x54,0x3D,0x52,0x53,0x3D,0x52,0x3D,0x3E
,0x51,0x52,0x3E,0x51,0x3E,0x3F,0x50,0x51,0x3F,0x50,0x3F,0x40,0x4F,0x50,0x40
,0x4F,0x40,0x41,0x4F,0x41,0x42,0x4F,0x42,0x43,0x4E,0x4F,0x43,0x4E,0x43,0x44
,0x02,0x4E,0x44,0x02,0x44,0x00,0x02,0x00,0x01,0x44,0x54,0x66,0x68,0x43,0x41
,0x01,0xB6,0x4B,0x4B,0xBE,0x8B,0x6C,0x27,0x3F,0xAE,0x47,0xE1,0xBE,0x04,0x56
,0x0E,0x3F,0xAA,0xF1,0xDA,0xBE,0xDD,0x24,0x06,0x3F,0xCB,0xA2,0xC8,0xBE,0x94
,0xD9,0x08,0x3F,0x48,0x35,0xBC,0xBE,0x87,0xC0,0x09,0x3F,0xC6,0xA2,0xB1,0xBE
,0x18,0x05,0x09,0x3F,0xB5,0x70,0xA9,0xBE,0xED,0xD2,0x06,0x3F,0x11,0x56,0xA3
,0xBE,0x5F,0x9A,0x02,0x3F,0x3D,0x0A,0x9F,0xBE,0x89,0x97,0xF7,0x3E,0xF1,0x7E
,0x9C,0xBE,0x2E,0xC9,0xD9,0x3E,0xE3,0xA5,0x9B,0xBE,0x9C,0xC4,0xA0,0x3E,0xE3
,0xA5,0x9B,0xBE,0x19,0x04,0xD6,0xBE,0x04,0x02,0x9D,0xBE,0xCD,0x20,0xF6,0xBE
,0x87,0x16,0xA1,0xBE,0xA6,0x9B,0x04,0xBF,0x8F,0x6F,0xA7,0xBE,0xCE,0xE2,0x09
,0xBF,0xA2,0x99,0xAF,0xBE,0xE1,0x24,0x0D,0xBF,0x94,0x6B,0xC2,0xBE,0x8E,0x57
,0x10,0xBF,0xD1,0x22,0xDB,0xBE,0x73,0x68,0x11,0xBF,0xD1,0x22,0xDB,0xBE,0x9A
,0x99,0x19,0xBF,0x42,0x60,0xE5,0xBC,0x9A,0x99,0x19,0xBF,0x42,0x60,0xE5,0xBC
,0x73,0x68,0x11,0xBF,0xD8,0x9F,0xA4,0xBD,0xC2,0x89,0x10,0xBF,0x3B,0x8F,0xEA
,0xBD,0x8D,0xED,0x0D,0xBF,0x16,0xDD,0x0A,0xBE,0xEA,0x06,0x0A,0xBF,0x0C,0x5A
,0x18,0xBE,0xBB,0x48,0x05,0xBF,0x83,0x6E,0x1F,0xBE,0x28,0x44,0xF8,0xBE,0xC1
,0xCA,0x21,0xBE,0x19,0x04,0xD6,0xBE,0xC1,0xCA,0x21,0xBE,0xAA,0xF3,0x28,0x3C
,0x8D,0x41,0xC7,0xBD,0x7C,0x0E,0x8C,0x3D,0xA6,0xD6,0x3B,0xBD,0x4E,0x0A,0xD3
,0x3D,0xD5,0xE8,0x55,0x38,0xEC,0x4B,0xF6,0x3D,0x7A,0xFE,0x34,0x3D,0x25,0x06
,0x01,0x3E,0xF3,0xFF,0xAA,0x3D,0x9D,0x68,0xF7,0x3D,0xB3,0xD0,0xEE,0x3D,0x0F
,0x7D,0xD7,0x3D,0xCB,0xF6,0x11,0x3E,0x88,0x47,0xA2,0x3D,0x17,0x29,0x24,0x3E
,0xEB,0x8C,0x2F,0x3D,0x00,0xAC,0x2E,0x3E,0x61,0x18,0xB0,0xBC,0x0E,0x2D,0x32
,0x3E,0x4B,0x8F,0x06,0xBE,0x0E,0x2D,0x32,0x3E,0x19,0x04,0xD6,0xBE,0x17,0x81
,0x31,0x3E,0x77,0x15,0xF2,0xBE,0xEE,0x7C,0x2F,0x3E,0xBD,0x1E,0x00,0xBF,0xC1
,0x1E,0x23,0x3E,0x63,0x9C,0x07,0xBF,0x00,0x00,0x10,0x3E,0x31,0x07,0x0D,0xBF
,0xF7,0x78,0xE1,0x3D,0x27,0x50,0x10,0xBF,0xF1,0xBD,0x7F,0x3D,0x73,0x68,0x11
,0xBF,0x58,0x39,0x34,0x3D,0x73,0x68,0x11,0xBF,0x58,0x39,0x34,0x3D,0x9A,0x99
,0x19,0xBF,0xAE,0x47,0xE1,0x3E,0x9A,0x99,0x19,0xBF,0xAE,0x47,0xE1,0x3E,0x73
,0x68,0x11,0xBF,0xE3,0xE2,0xC8,0x3E,0x41,0x9F,0x10,0xBF,0x1E,0x87,0xB9,0x3E
,0x9A,0x43,0x0E,0xBF,0x28,0x44,0xB0,0x3E,0xD6,0x8E,0x0A,0xBF,0xFE,0x28,0xAA
,0x3E,0x5E,0xBA,0x05,0xBF,0x87,0x6B,0xA5,0x3E,0x79,0xE9,0xF6,0xBE,0x0A,0xD7
,0xA3,0x3E,0x19,0x04,0xD6,0xBE,0x0A,0xD7,0xA3,0x3E,0x1D,0x02,0x07,0xBE,0x1C
,0xB2,0xA1,0x3E,0x82,0xAC,0xA7,0xBB,0x96,0x43,0x9B,0x3E,0x8F,0x37,0x99,0x3D
,0x4B,0xCC,0x8B,0x3E,0x52,0xBA,0x14,0x3E,0x71,0x91,0x6B,0x3E,0x10,0xB0,0x46
,0x3E,0x46,0x07,0x34,0x3E,0x23,0xD9,0x63,0x3E,0xB8,0x1E,0xE5,0x3D,0x68,0x91
,0x6D,0x3E,0x49,0x64,0x5F,0x3D,0xE1,0xCE,0x65,0x3E,0x70,0x41,0x36,0xBB,0x4D
,0x87,0x4E,0x3E,0x0A,0xDB,0x8F,0xBD,0xC8,0xCE,0x1B,0x3E,0xC1,0xCA,0x21,0xBE
,0x00,0x75,0x83,0x3D,0xC1,0xCA,0x21,0xBE,0x8B,0x6C,0x27,0x3F,0x41,0x42,0x00
,0x2E,0x2F,0x30,0x2E,0x30,0x31,0x2E,0x31,0x32,0x2E,0x32,0x33,0x2E,0x33,0x34
,0x2C,0x2D,0x2E,0x12,0x13,0x14,0x12,0x14,0x15,0x12,0x15,0x16,0x12,0x16,0x17
,0x12,0x17,0x18,0x00,0x01,0x02,0x00,0x02,0x03,0x00,0x03,0x04,0x00,0x04,0x05
,0x00,0x05,0x06,0x2E,0x34,0x35,0x12,0x18,0x19,0x10,0x11,0x12,0x00,0x06,0x07
,0x41,0x00,0x07,0x41,0x07,0x08,0x41,0x08,0x09,0x41,0x09,0x0A,0x41,0x0A,0x0B
,0x2B,0x2C,0x2E,0x2B,0x2E,0x35,0x2A,0x2B,0x35,0x29,0x2A,0x35,0x28,0x29,0x35
,0x27,0x28,0x35,0x27,0x35,0x36,0x26,0x27,0x36,0x25,0x26,0x36,0x25,0x36,0x37
,0x25,0x37,0x38,0x24,0x25,0x38,0x24,0x38,0x39,0x23,0x24,0x39,0x23,0x39,0x3A
,0x22,0x23,0x3A,0x22,0x3A,0x3B,0x22,0x3B,0x3C,0x21,0x22,0x3C,0x21,0x3C,0x3D
,0x20,0x21,0x3D,0x20,0x3D,0x3E,0x1F,0x20,0x3E,0x1F,0x3E,0x3F,0x1E,0x1F,0x3F
,0x1E,0x3F,0x40,0x1D,0x1E,0x40,0x1D,0x40,0x41,0x1C,0x1D,0x41,0x1B,0x1C,0x41
,0x1B,0x41,0x0B,0x1A,0x1B,0x0B,0x19,0x1A,0x0B,0x19,0x0B,0x0C,0x19,0x0C,0x0D
,0x19,0x0D,0x0E,0x19,0x0E,0x0F,0x12,0x19,0x0F,0x10,0x12,0x0F,0x42,0x69,0x2C
,0x28,0x02,0xA0,0xE0,0xE2,0xBC,0x01,0xC0,0x25,0x3F,0x41,0xD5,0x68,0xBD,0x62
,0xBA,0x20,0x3F,0x96,0x97,0x9C,0xBD,0x6E,0x52,0x19,0x3F,0xE7,0xFB,0xA9,0xBD
,0x02,0x7F,0x10,0x3F,0x70,0x5E,0x9C,0xBD,0x61,0xA4,0x07,0x3F,0xFA,0x09,0x67
,0xBD,0xEE,0x26,0x00,0x3F,0x7C,0x65,0xDE,0xBC,0x7F,0x17,0xF6,0x3E,0x14,0xB2
,0xF3,0x3B,0x21,0xB0,0xF2,0x3E,0x04,0xAE,0x2B,0x3D,0xA0,0x17,0xF6,0x3E,0xA9
,0x16,0x91,0x3D,0xEE,0x26,0x00,0x3F,0x9E,0x43,0xB9,0x3D,0x61,0xA4,0x07,0x3F
,0xF0,0xA7,0xC6,0x3D,0x02,0x7F,0x10,0x3F,0x9E,0x43,0xB9,0x3D,0x6E,0x52,0x19
,0x3F,0xA9,0x16,0x91,0x3D,0x62,0xBA,0x20,0x3F,0x04,0xAE,0x2B,0x3D,0x01,0xC0
,0x25,0x3F,0x14,0xB2,0xF3,0x3B,0x8B,0x6C,0x27,0x3F,0xD5,0xED,0x2C,0x3D,0x68
,0x91,0x6D,0x3E,0xF2,0xD2,0x4D,0xBE,0x4C,0x37,0x09,0x3E,0xE9,0x26,0x41,0xBE
,0x60,0xE5,0xD0,0x3D,0x3E,0xB4,0x1F,0xBE,0x2B,0x33,0xE5,0x3D,0x25,0xAE,0x03
,0xBE,0xF0,0xF7,0xEB,0x3D,0xF1,0x4B,0xDD,0xBD,0xA9,0x4B,0xE6,0x3D,0x1B,0x2F
,0xBD,0xBD,0xE0,0x47,0xD5,0x3D,0x2E,0xAA,0xA5,0xBD,0xF5,0x9F,0xB5,0x3D,0x85
,0x60,0x95,0xBD,0x53,0x08,0x84,0x3D,0xE8,0x2E,0x89,0xBD,0xC3,0x2A,0xDE,0x3A
,0xB8,0x1E,0x85,0xBD,0x1B,0x2F,0xDD,0xBD,0xB8,0x1E,0x85,0xBD,0x19,0x04,0xD6
,0xBE,0x4B,0x3D,0x8B,0xBD,0x19,0x04,0xF6,0xBE,0x89,0x99,0x9D,0xBD,0xFE,0xD4
,0x04,0xBF,0x80,0x64,0xBA,0xBD,0x30,0x9D,0x0A,0xBF,0xB8,0xCE,0xDF,0xBD,0xF2
,0x7C,0x0E,0xBF,0x59,0xBE,0x0E,0xBE,0x8A,0xAD,0x10,0xBF,0x37,0x89,0x41,0xBE
,0x73,0x68,0x11,0xBF,0x37,0x89,0x41,0xBE,0x9A,0x99,0x19,0xBF,0xF2,0xD2,0x4D
,0x3E,0x9A,0x99,0x19,0xBF,0xF2,0xD2,0x4D,0x3E,0x73,0x68,0x11,0xBF,0xD3,0x4E
,0x1D,0x3E,0x41,0x9F,0x10,0xBF,0x40,0x68,0xFD,0x3D,0x9A,0x43,0x0E,0xBF,0x9D
,0x68,0xD7,0x3D,0x59,0x4E,0x0A,0xBF,0x11,0x00,0xBC,0x3D,0x5A,0xB8,0x04,0xBF
,0x62,0x6A,0xAB,0x3D,0x16,0x2F,0xF6,0xBE,0x54,0xE3,0xA5,0x3D,0x19,0x04,0xD6
,0xBE,0x54,0xE3,0xA5,0x3D,0x68,0x91,0x6D,0x3E,0x07,0x08,0x09,0x06,0x07,0x09
,0x05,0x06,0x09,0x05,0x09,0x0A,0x05,0x0A,0x0B,0x04,0x05,0x0B,0x03,0x04,0x0B
,0x02,0x03,0x0B,0x01,0x02,0x0B,0x00,0x01,0x0B,0x0F,0x00,0x0B,0x0F,0x0B,0x0C
,0x0F,0x0C,0x0D,0x0E,0x0F,0x0D,0x2A,0x2B,0x10,0x22,0x23,0x24,0x22,0x24,0x25
,0x22,0x25,0x26,0x22,0x26,0x27,0x22,0x27,0x28,0x10,0x11,0x12,0x10,0x12,0x13
,0x10,0x13,0x14,0x10,0x14,0x15,0x10,0x15,0x16,0x20,0x21,0x22,0x20,0x22,0x28
,0x1F,0x20,0x28,0x1E,0x1F,0x28,0x1D,0x1E,0x28,0x1C,0x1D,0x28,0x1C,0x28,0x29
,0x1B,0x1C,0x29,0x1B,0x29,0x2A,0x1B,0x2A,0x10,0x1A,0x1B,0x10,0x19,0x1A,0x10
,0x18,0x19,0x10,0x17,0x18,0x10,0x17,0x10,0x16,0x0F,0x2B,0x6A,0x35,0x31,0x02
,0xC2,0x51,0xF2,0x3D,0xED,0xBC,0x25,0x3F,0xFD,0xF9,0xB6,0x3D,0x10,0xAE,0x20
,0x3F,0x1B,0x83,0x8E,0x3D,0xBD,0x34,0x19,0x3F,0x25,0x06,0x81,0x3D,0xAA,0x45
,0x10,0x3F,0x1B,0x83,0x8E,0x3D,0x84,0x81,0x07,0x3F,0xFD,0xF9,0xB6,0x3D,0x8B
,0x16,0x00,0x3F,0xC2,0x51,0xF2,0x3D,0x5E,0x0F,0xF6,0x3E,0x7A,0x39,0x1C,0x3E
,0x21,0xB0,0xF2,0x3E,0xC7,0xF5,0x3F,0x3E,0x5E,0x0F,0xF6,0x3E,0x12,0xDB,0x5D
,0x3E,0x8B,0x16,0x00,0x3F,0x83,0x16,0x72,0x3E,0x84,0x81,0x07,0x3F,0xFE,0xD4
,0x78,0x3E,0xAA,0x45,0x10,0x3F,0x83,0x16,0x72,0x3E,0xBD,0x34,0x19,0x3F,0x12
,0xDB,0x5D,0x3E,0x10,0xAE,0x20,0x3F,0xC7,0xF5,0x3F,0x3E,0xED,0xBC,0x25,0x3F
,0x7A,0x39,0x1C,0x3E,0x8B,0x6C,0x27,0x3F,0x81,0x95,0x43,0x3E,0x68,0x91,0x6D
,0x3E,0x87,0x16,0x59,0xBD,0x4C,0x37,0x09,0x3E,0x66,0x66,0x26,0xBD,0x60,0xE5
,0xD0,0x3D,0xDB,0x6E,0x02,0xBC,0x2B,0x33,0xE5,0x3D,0x55,0xF9,0x9E,0x3C,0xF0
,0xF7,0xEB,0x3D,0x5E,0x9D,0x23,0x3D,0xDB,0x4F,0xE6,0x3D,0x0A,0xD7,0x63,0x3D
,0x21,0x58,0xD5,0x3D,0x72,0x70,0x89,0x3D,0x7B,0x6A,0xB5,0x3D,0x1B,0xBA,0x99
,0x3D,0x26,0xE1,0x82,0x3D,0xB7,0xEB,0xA5,0x3D,0xFF,0xCE,0xF6,0x3A,0xE7,0xFB
,0xA9,0x3D,0x1B,0x2F,0xDD,0xBD,0xE7,0xFB,0xA9,0x3D,0xDD,0xCE,0x2E,0xBF,0x87
,0x1A,0xA5,0x3D,0xE1,0xCE,0x49,0xBF,0xEA,0x76,0x96,0x3D,0x18,0x5A,0x59,0xBF
,0x75,0x3B,0x7B,0x3D,0x09,0xC2,0x61,0xBF,0x58,0x39,0x34,0x3D,0x19,0x58,0x67
,0xBF,0x7E,0x54,0xC3,0x3C,0xDD,0x79,0x6A,0xBF,0xF4,0xFD,0x54,0x3B,0x1F,0x85
,0x6B,0xBF,0x8E,0x22,0xEB,0xBC,0xEC,0x12,0x69,0xBF,0xE7,0xFB,0x89,0xBD,0x66
,0xBC,0x61,0xBF,0x40,0x31,0xD2,0xBD,0x04,0x01,0x5A,0xBF,0x11,0xA8,0xFE,0xBD
,0x3D,0x60,0x56,0xBF,0x84,0x66,0x17,0xBE,0xA5,0x2F,0x54,0xBF,0x5F,0x0A,0x2F
,0xBE,0xBC,0x74,0x53,0xBF,0xDD,0xD2,0x4A,0xBE,0xF8,0xDD,0x54,0xBF,0xC5,0xC8
,0x62,0xBE,0x9C,0x19,0x59,0xBF,0xDF,0x51,0x73,0xBE,0x98,0x4E,0x5F,0xBF,0xFE
,0xD4,0x78,0xBE,0xDF,0xA3,0x66,0xBF,0xC2,0xF7,0x6E,0xBE,0x11,0x6E,0x6E,0xBF
,0x52,0x60,0x51,0xBE,0xF0,0xFD,0x75,0xBF,0x30,0x62,0x1F,0xBE,0xFF,0x93,0x7B
,0xBF,0xD7,0xA3,0xB0,0xBD,0xA4,0x70,0x7D,0xBF,0xB2,0xD8,0x26,0x3D,0xCE,0x00
,0x77,0xBF,0x20,0xD5,0x10,0x3E,0x29,0xB1,0x63,0xBF,0x62,0x14,0x54,0x3E,0x65
,0x1C,0x43,0xBF,0xFA,0x7E,0x6A,0x3E,0x2F,0xDD,0x14,0xBF,0xFA,0x7E,0x6A,0x3E
,0x68,0x91,0x6D,0x3E,0x05,0x06,0x07,0x05,0x07,0x08,0x05,0x08,0x09,0x05,0x09
,0x0A,0x05,0x0A,0x0B,0x05,0x0B,0x0C,0x05,0x0C,0x0D,0x05,0x0D,0x0E,0x05,0x0E
,0x0F,0x04,0x05,0x0F,0x03,0x04,0x0F,0x02,0x03,0x0F,0x01,0x02,0x0F,0x00,0x01
,0x0F,0x33,0x34,0x10,0x10,0x11,0x12,0x10,0x12,0x13,0x10,0x13,0x14,0x10,0x14
,0x15,0x10,0x15,0x16,0x10,0x16,0x17,0x10,0x17,0x18,0x33,0x10,0x18,0x33,0x18
,0x19,0x33,0x19,0x1A,0x33,0x1A,0x1B,0x32,0x33,0x1B,0x32,0x1B,0x1C,0x31,0x32
,0x1C,0x31,0x1C,0x1D,0x30,0x31,0x1D,0x30,0x1D,0x1E,0x30,0x1E,0x1F,0x30,0x1F
,0x20,0x2F,0x30,0x20,0x2F,0x20,0x21,0x2F,0x21,0x22,0x2F,0x22,0x23,0x2E,0x2F
,0x23,0x2E,0x23,0x24,0x2D,0x2E,0x24,0x2D,0x24,0x25,0x2D,0x25,0x26,0x2C,0x2D
,0x26,0x2C,0x26,0x27,0x2C,0x27,0x28,0x2C,0x28,0x29,0x2C,0x29,0x2A,0x2B,0x2C
,0x2A,0x0F,0x34,0x6B,0x38,0x36,0x01,0x51,0xDD,0x5C,0xBE,0x66,0x66,0x26,0x3F
,0xBD,0xC8,0xEC,0xBE,0x25,0x5A,0x0A,0x3F,0x04,0x56,0xE6,0xBE,0x7B,0x14,0x02
,0x3F,0xA2,0x99,0xCF,0xBE,0x06,0xD7,0x04,0x3F,0x19,0xAC,0xC0,0xBE,0x8F,0xC2
,0x05,0x3F,0x14,0x96,0xB0,0xBE,0x67,0x26,0x04,0x3F,0xB5,0xC6,0xA8,0xBE,0xDF
,0xA3,0xFE,0x3E,0xAB,0x05,0xA6,0xBE,0x02,0x2C,0xEA,0x3E,0xA8,0x1A,0xA5,0xBE
,0x2B,0x87,0xC6,0x3E,0xA8,0x1A,0xA5,0xBE,0xB2,0xF1,0xD0,0xBE,0x08,0x57,0xA8
,0xBE,0x88,0x7F,0xF8,0xBE,0x4A,0x0C,0xB2,0xBE,0xFC,0xAA,0x08,0xBF,0x81,0xEC
,0xC5,0xBE,0x19,0x39,0x0F,0xBF,0x04,0xAA,0xE7,0xBE,0x73,0x68,0x11,0xBF,0x04
,0xAA,0xE7,0xBE,0x9A,0x99,0x19,0xBF,0xF9,0xF3,0x2D,0xBD,0x9A,0x99,0x19,0xBF
,0xF9,0xF3,0x2D,0xBD,0x73,0x68,0x11,0xBF,0x85,0xEF,0xDD,0xBD,0x19,0x39,0x0F
,0xBF,0x73,0xB8,0x16,0xBE,0xFC,0xAA,0x08,0xBF,0xB4,0x22,0x2A,0xBE,0x88,0x7F
,0xF8,0xBE,0xB7,0x9B,0x30,0xBE,0xB2,0xF1,0xD0,0xBE,0x20,0x08,0x30,0xBE,0xC1
,0x1A,0x17,0xBE,0xA3,0x5B,0x2F,0x3D,0xFA,0x26,0xD5,0xBE,0x5B,0xB2,0x8A,0x3D
,0x18,0x42,0xE6,0xBE,0x87,0x6E,0xB6,0x3D,0xA0,0x70,0xF6,0xBE,0x2C,0x81,0xD4
,0x3D,0xFF,0x09,0x02,0xBF,0x09,0x87,0xDE,0x3D,0xAC,0xC6,0x06,0xBF,0x49,0xDB
,0xD8,0x3D,0xE6,0x91,0x0B,0xBF,0xF9,0xD6,0xC7,0x3D,0x37,0xDD,0x0E,0xBF,0xC6
,0xFD,0xA7,0x3D,0xA8,0xC5,0x10,0xBF,0xA0,0xA5,0x6B,0x3D,0x73,0x68,0x11,0xBF
,0xA0,0xA5,0x6B,0x3D,0x9A,0x99,0x19,0xBF,0xBD,0xC8,0xEC,0x3E,0x9A,0x99,0x19
,0xBF,0xBD,0xC8,0xEC,0x3E,0x73,0x68,0x11,0xBF,0x16,0x85,0xCD,0x3E,0xE3,0x1A
,0x0F,0xBF,0x95,0xEF,0xB1,0x3E,0x33,0x32,0x08,0xBF,0xC4,0x21,0x93,0x3E,0xD8
,0x10,0xF4,0xBE,0x5D,0x6A,0x54,0x3E,0xAE,0xEF,0xC3,0xBE,0x62,0x10,0x18,0xBD
,0xFE,0x5F,0x95,0xBD,0x4C,0x37,0xE9,0x3D,0x1F,0x85,0x6B,0x3D,0x87,0xC5,0x48
,0x3E,0x27,0x32,0x03,0x3E,0xE6,0xCE,0x84,0x3E,0xFA,0xD2,0x2B,0x3E,0xBD,0x8A
,0xA4,0x3E,0x6A,0x69,0x3E,0x3E,0x54,0x8F,0xCC,0x3E,0xA6,0x9B,0x44,0x3E,0x54
,0x8F,0xCC,0x3E,0x42,0x60,0x65,0x3E,0xA0,0xA5,0x6B,0x3D,0x42,0x60,0x65,0x3E
,0xA0,0xA5,0x6B,0x3D,0xA6,0x9B,0x44,0x3E,0x1C,0x5C,0xBA,0x3D,0x47,0x3A,0x43
,0x3E,0x41,0xB8,0xE2,0x3D,0x6E,0x16,0x3F,0x3E,0x6E,0x6C,0xF6,0x3D,0x21,0x5B
,0x36,0x3E,0xD2,0xFD,0xFC,0x3D,0x22,0x33,0x27,0x3E,0x20,0x60,0xED,0x3D,0xD6
,0x74,0x0D,0x3E,0x09,0x87,0xBE,0x3D,0x41,0x10,0xE0,0x3D,0xAF,0x5D,0xDA,0x3C
,0x61,0x70,0x4D,0x3D,0xB7,0x9B,0x30,0xBE,0x02,0x2B,0x07,0xBE,0xB7,0x9B,0x30
,0xBE,0x66,0x66,0x26,0x3F,0x36,0x37,0x00,0x2C,0x2D,0x2E,0x2C,0x2E,0x2F,0x2C
,0x2F,0x30,0x2C,0x30,0x31,0x2B,0x2C,0x31,0x2A,0x2B,0x31,0x29,0x2A,0x31,0x28
,0x29,0x31,0x28,0x31,0x32,0x27,0x28,0x32,0x27,0x32,0x33,0x27,0x33,0x34,0x27
,0x34,0x35,0x26,0x27,0x35,0x26,0x35,0x36,0x1F,0x20,0x21,0x1F,0x21,0x22,0x1F
,0x22,0x23,0x1F,0x23,0x24,0x0E,0x0F,0x10,0x0E,0x10,0x11,0x0E,0x11,0x12,0x0D
,0x0E,0x12,0x0C,0x0D,0x12,0x0B,0x0C,0x12,0x0A,0x0B,0x12,0x0A,0x12,0x13,0x0A
,0x13,0x14,0x09,0x0A,0x14,0x08,0x09,0x14,0x08,0x14,0x15,0x00,0x01,0x02,0x00
,0x02,0x03,0x00,0x03,0x04,0x00,0x04,0x05,0x1D,0x1E,0x1F,0x1D,0x1F,0x24,0x1C
,0x1D,0x24,0x1B,0x1C,0x24,0x1A,0x1B,0x24,0x1A,0x24,0x25,0x19,0x1A,0x25,0x18
,0x19,0x25,0x17,0x18,0x25,0x17,0x25,0x26,0x16,0x17,0x26,0x15,0x16,0x26,0x15
,0x26,0x36,0x08,0x15,0x36,0x08,0x36,0x00,0x07,0x08,0x00,0x06,0x07,0x00,0x06
,0x00,0x05,0x37,0x6C,0x1C,0x1A,0x01,0x60,0xE5,0x10,0x3D,0x8B,0x6C,0x27,0x3F
,0x85,0xEB,0x51,0xBE,0x04,0x56,0x0E,0x3F,0x5C,0x8F,0x42,0xBE,0xDD,0x24,0x06
,0x3F,0xC3,0x9F,0x21,0xBE,0x94,0xD9,0x08,0x3F,0x1D,0x02,0x07,0xBE,0x87,0xC0
,0x09,0x3F,0xD9,0x7A,0xE6,0xBD,0x1D,0x04,0x09,0x3F,0xF0,0xA7,0xC6,0xBD,0xDD
,0xCE,0x06,0x3F,0x34,0x85,0xAE,0xBD,0x36,0x94,0x02,0x3F,0x5B,0x0C,0x9E,0xBD
,0x69,0x8F,0xF7,0x3E,0x6D,0x91,0x94,0xBD,0x37,0xC7,0xD9,0x3E,0x73,0x68,0x91
,0xBD,0x9C,0xC4,0xA0,0x3E,0x73,0x68,0x91,0xBD,0x19,0x04,0xD6,0xBE,0x2A,0x54
,0x97,0xBD,0x19,0x04,0xF6,0xBE,0x44,0x16,0xA9,0xBD,0xFE,0xD4,0x04,0xBF,0x01
,0x50,0xC5,0xBD,0x30,0x9D,0x0A,0xBF,0x7B,0x9F,0xEA,0xBD,0xF2,0x7C,0x0E,0xBF
,0x2A,0x58,0x13,0xBE,0x8A,0xAD,0x10,0xBF,0x81,0x95,0x43,0xBE,0x73,0x68,0x11
,0xBF,0x81,0x95,0x43,0xBE,0x9A,0x99,0x19,0xBF,0x85,0xEB,0x51,0x3E,0x9A,0x99
,0x19,0xBF,0x85,0xEB,0x51,0x3E,0x73,0x68,0x11,0xBF,0x1B,0x2C,0x1C,0x3E,0x0B
,0x98,0x10,0xBF,0x21,0x58,0xF5,0x3D,0xE5,0x26,0x0E,0xBF,0x1A,0xA8,0xCC,0x3D
,0x5B,0x23,0x0A,0xBF,0x61,0x18,0xB0,0x3D,0xA6,0x9B,0x04,0xBF,0x4B,0x39,0x9F
,0x3D,0xCD,0x20,0xF6,0xBE,0x9A,0x99,0x99,0x3D,0x19,0x04,0xD6,0xBE,0x9A,0x99
,0x99,0x3D,0x8B,0x6C,0x27,0x3F,0x00,0x01,0x02,0x00,0x02,0x03,0x00,0x03,0x04
,0x00,0x04,0x05,0x00,0x05,0x06,0x1A,0x1B,0x00,0x12,0x13,0x14,0x12,0x14,0x15
,0x12,0x15,0x16,0x12,0x16,0x17,0x12,0x17,0x18,0x10,0x11,0x12,0x10,0x12,0x18
,0x0F,0x10,0x18,0x0E,0x0F,0x18,0x0D,0x0E,0x18,0x0C,0x0D,0x18,0x0C,0x18,0x19
,0x0C,0x19,0x1A,0x0B,0x0C,0x1A,0x0B,0x1A,0x00,0x0A,0x0B,0x00,0x09,0x0A,0x00
,0x08,0x09,0x00,0x07,0x08,0x00,0x07,0x00,0x06,0x1B,0x6D,0x65,0x63,0x01,0x60
,0xE5,0xD0,0xBE,0x68,0x91,0x6D,0x3E,0x1D,0x5A,0xE4,0xBE,0x68,0x91,0x6D,0x3E
,0xD7,0xA3,0x30,0xBF,0x4C,0x37,0x09,0x3E,0x7D,0x3F,0x2D,0xBF,0x60,0xE5,0xD0
,0x3D,0xD0,0x0D,0x25,0xBF,0x2B,0x33,0xE5,0x3D,0xA4,0x1A,0x1E,0xBF,0xF0,0xF7
,0xEB,0x3D,0x99,0xD8,0x18,0xBF,0x0D,0x54,0xE6,0x3D,0xFE,0xD4,0x14,0xBF,0x62
,0x68,0xD5,0x3D,0x60,0xE4,0x11,0xBF,0x60,0x93,0xB5,0x3D,0x2B,0xDB,0x0F,0xBF
,0x76,0x33,0x83,0x3D,0xF8,0x54,0x0E,0xBF,0xD3,0x16,0xD7,0x3A,0xF2,0xD2,0x0D
,0xBF,0x1B,0x2F,0xDD,0xBD,0xF2,0xD2,0x0D,0xBF,0x19,0x04,0xD6,0xBE,0xC6,0x8A
,0x0E,0xBF,0xCF,0xF5,0xF5,0xBE,0x31,0xB2,0x10,0xBF,0x5A,0xB8,0x04,0xBF,0x60
,0x3B,0x14,0xBF,0x32,0x72,0x0A,0xBF,0x8F,0x18,0x19,0xBF,0x3D,0x60,0x0E,0xBF
,0xD0,0xB8,0x20,0xBF,0x65,0xA6,0x10,0xBF,0x44,0x8B,0x2C,0xBF,0x73,0x68,0x11
,0xBF,0x44,0x8B,0x2C,0xBF,0x9A,0x99,0x19,0xBF,0x04,0x56,0x8E,0xBE,0x9A,0x99
,0x19,0xBF,0x04,0x56,0x8E,0xBE,0x73,0x68,0x11,0xBF,0x37,0x33,0xAA,0xBE,0x65
,0xA6,0x10,0xBF,0xC5,0xC8,0xBA,0xBE,0x3D,0x60,0x0E,0xBF,0x49,0x63,0xC4,0xBE
,0x0B,0x96,0x0A,0xBF,0xE8,0x4F,0xCB,0xBE,0xAE,0x47,0x05,0xBF,0xF1,0x7F,0xCF
,0xBE,0x29,0x5C,0xF7,0xBE,0x60,0xE5,0xD0,0xBE,0x19,0x04,0xD6,0xBE,0x60,0xE5
,0xD0,0xBE,0x22,0xC5,0x80,0x3C,0x15,0x58,0xB8,0xBE,0x18,0x21,0x7C,0x3D,0x64
,0x3B,0x97,0xBE,0x07,0x0C,0xD2,0x3D,0xE9,0x7E,0x7E,0xBE,0xEB,0x1E,0xF9,0x3D
,0x7B,0x14,0x4E,0xBE,0x6F,0x12,0x03,0x3E,0xAC,0xCA,0x0E,0xBE,0xC6,0x6C,0xE9
,0x3D,0x09,0x87,0xBE,0xBD,0x74,0x43,0x93,0x3D,0xB7,0xEF,0x91,0xBD,0x89,0x40
,0x75,0x3C,0x6F,0x12,0x83,0xBD,0xE8,0x4B,0x8F,0xBD,0x6F,0x12,0x83,0xBD,0x19
,0x04,0xD6,0xBE,0x01,0x50,0x85,0xBD,0x54,0x8C,0xF3,0xBE,0x31,0x08,0x8C,0xBD
,0x75,0x3D,0x01,0xBF,0xE5,0x7E,0xA7,0xBD,0xA0,0x4F,0x08,0xBF,0x7D,0x97,0xD2
,0xBD,0x2D,0x5D,0x0D,0xBF,0x3E,0x5D,0x0D,0xBE,0xA5,0x65,0x10,0xBF,0xF0,0xA7
,0x46,0xBE,0x73,0x68,0x11,0xBF,0xF0,0xA7,0x46,0xBE,0x9A,0x99,0x19,0xBF,0x87
,0x16,0x59,0x3E,0x9A,0x99,0x19,0xBF,0x87,0x16,0x59,0x3E,0x73,0x68,0x11,0xBF
,0x46,0xB6,0x23,0x3E,0xC2,0x89,0x10,0xBF,0xD7,0xA3,0x00,0x3E,0x8D,0xED,0x0D
,0xBF,0x80,0xB6,0xD5,0x3D,0xEA,0x06,0x0A,0xBF,0xFF,0xAF,0xBA,0x3D,0xBB,0x48
,0x05,0xBF,0xB6,0x9F,0xAC,0x3D,0x28,0x44,0xF8,0xBE,0x9E,0xEF,0xA7,0x3D,0x19
,0x04,0xD6,0xBE,0x9E,0xEF,0xA7,0x3D,0xED,0xF1,0x02,0xBD,0x77,0xBE,0x9F,0x3D
,0xB2,0x9F,0xC5,0x3B,0x39,0x64,0xA3,0x3D,0x22,0xC5,0x80,0x3C,0xAA,0x9D,0x11
,0x3E,0x7F,0xBE,0x8D,0x3D,0xE7,0x53,0x47,0x3E,0x0E,0x2D,0xD2,0x3D,0x0B,0xD3
,0x77,0x3E,0x59,0x14,0xF6,0x3D,0x41,0x0C,0x94,0x3E,0x25,0x06,0x01,0x3E,0xA8
,0xC4,0xB5,0x3E,0x87,0x1A,0xE5,0x3D,0xDD,0xD0,0xCC,0x3E,0xC3,0x45,0x8E,0x3D
,0x25,0x05,0xD6,0x3E,0x4D,0xD9,0x69,0x3C,0x87,0x16,0xD9,0x3E,0x31,0x7D,0x8F
,0xBD,0x87,0x16,0xD9,0x3E,0x19,0x04,0xD6,0xBE,0xD7,0xA3,0xD8,0x3E,0x3B,0xE0
,0xF2,0xBE,0xC7,0x4B,0xD7,0x3E,0xC5,0xCA,0x00,0xBF,0x00,0x54,0xD1,0x3E,0xFB
,0x94,0x07,0xBF,0xB0,0x72,0xC8,0x3E,0x79,0x3D,0x0C,0xBF,0x19,0xAE,0xB6,0x3E
,0xB0,0x1D,0x10,0xBF,0x5E,0x12,0x9F,0x3E,0x73,0x68,0x11,0xBF,0x2B,0x87,0x96
,0x3E,0x73,0x68,0x11,0xBF,0x2B,0x87,0x96,0x3E,0x9A,0x99,0x19,0xBF,0xD7,0xA3
,0x30,0x3F,0x9A,0x99,0x19,0xBF,0xD7,0xA3,0x30,0x3F,0x73,0x68,0x11,0xBF,0x73
,0xF4,0x24,0x3F,0xC2,0x89,0x10,0xBF,0x4A,0x0D,0x1D,0x3F,0x8D,0xED,0x0D,0xBF
,0xA8,0xFC,0x17,0x3F,0x01,0x15,0x0A,0xBF,0xDD,0xD0,0x14,0x3F,0x06,0x81,0x05
,0xBF,0xDB,0xA4,0x12,0x3F,0x7D,0x93,0xF6,0xBE,0x85,0xEB,0x11,0x3F,0x19,0x04
,0xD6,0xBE,0x85,0xEB,0x11,0x3F,0x6A,0x6C,0x8F,0xBD,0x8E,0xAC,0x10,0x3F,0xE4
,0xF6,0xCB,0x3C,0x9A,0xEF,0x0C,0x3F,0xA1,0xBA,0xB9,0x3D,0x14,0x22,0x04,0x3F
,0x0D,0xFE,0x1E,0x3E,0xF8,0x55,0xF1,0x3E,0xE3,0xA5,0x4B,0x3E,0xCA,0xDF,0xD5
,0x3E,0x98,0x16,0x65,0x3E,0x00,0xAC,0xB6,0x3E,0x68,0x91,0x6D,0x3E,0xE1,0x24
,0x95,0x3E,0xA4,0x17,0x65,0x3E,0x67,0x0E,0x69,0x3E,0x15,0xAA,0x4B,0x3E,0xE5
,0x7C,0x21,0x3E,0xA3,0x1D,0x17,0x3E,0xF5,0xD8,0x96,0x3D,0xFB,0x1E,0x75,0x3D
,0xAB,0x5B,0x3D,0x3D,0xEF,0xFE,0x08,0x3E,0x24,0x0E,0x59,0x3A,0x1C,0x5E,0x40
,0x3E,0x6F,0xF1,0x70,0xBD,0xA6,0x44,0x62,0x3E,0x93,0x18,0x04,0xBE,0x68,0x91
,0x6D,0x3E,0xEE,0x24,0x32,0xBE,0xD9,0xD1,0x68,0x3E,0xD7,0xA3,0x60,0xBE,0x6C
,0x93,0x5A,0x3E,0x5C,0xE4,0x86,0xBE,0x72,0xBF,0x43,0x3E,0x5C,0xE3,0x9B,0xBE
,0x7D,0x3F,0x25,0x3E,0x60,0xE5,0xD0,0xBE,0xFB,0x1E,0x75,0x3D,0x01,0x02,0x03
,0x01,0x03,0x04,0x01,0x04,0x05,0x01,0x05,0x06,0x01,0x06,0x07,0x48,0x49,0x4A
,0x48,0x4A,0x4B,0x48,0x4B,0x4C,0x48,0x4C,0x4D,0x48,0x4D,0x4E,0x46,0x47,0x48
,0x2C,0x2D,0x2E,0x2C,0x2E,0x2F,0x2C,0x2F,0x30,0x2C,0x30,0x31,0x2C,0x31,0x32
,0x13,0x14,0x15,0x13,0x15,0x16,0x13,0x16,0x17,0x13,0x17,0x18,0x13,0x18,0x19
,0x64,0x00,0x01,0x64,0x01,0x07,0x64,0x07,0x08,0x64,0x08,0x09,0x64,0x09,0x0A
,0x48,0x4E,0x4F,0x11,0x12,0x13,0x11,0x13,0x19,0x10,0x11,0x19,0x0F,0x10,0x19
,0x0E,0x0F,0x19,0x0E,0x19,0x1A,0x0D,0x0E,0x1A,0x0D,0x1A,0x1B,0x0C,0x0D,0x1B
,0x0B,0x0C,0x1B,0x0B,0x1B,0x1C,0x0A,0x0B,0x1C,0x64,0x0A,0x1C,0x64,0x1C,0x1D
,0x63,0x64,0x1D,0x63,0x1D,0x1E,0x63,0x1E,0x1F,0x62,0x63,0x1F,0x2A,0x2B,0x2C
,0x2A,0x2C,0x32,0x29,0x2A,0x32,0x28,0x29,0x32,0x27,0x28,0x32,0x27,0x32,0x33
,0x26,0x27,0x33,0x26,0x33,0x34,0x25,0x26,0x34,0x24,0x25,0x34,0x24,0x34,0x35
,0x23,0x24,0x35,0x23,0x35,0x36,0x62,0x1F,0x20,0x61,0x62,0x20,0x60,0x61,0x20
,0x60,0x20,0x21,0x5F,0x60,0x21,0x5E,0x5F,0x21,0x5E,0x21,0x22,0x5D,0x5E,0x22
,0x5C,0x5D,0x22,0x5B,0x5C,0x22,0x5B,0x22,0x23,0x5B,0x23,0x36,0x5B,0x36,0x37
,0x5B,0x37,0x38,0x5A,0x5B,0x38,0x5A,0x38,0x39,0x59,0x5A,0x39,0x59,0x39,0x3A
,0x59,0x3A,0x3B,0x58,0x59,0x3B,0x57,0x58,0x3B,0x57,0x3B,0x3C,0x56,0x57,0x3C
,0x55,0x56,0x3C,0x55,0x3C,0x3D,0x54,0x55,0x3D,0x53,0x54,0x3D,0x53,0x3D,0x3E
,0x52,0x53,0x3E,0x51,0x52,0x3E,0x51,0x3E,0x3F,0x50,0x51,0x3F,0x50,0x3F,0x40
,0x50,0x40,0x41,0x4F,0x50,0x41,0x4F,0x41,0x42,0x4F,0x42,0x43,0x45,0x46,0x48
,0x45,0x48,0x4F,0x44,0x45,0x4F,0x44,0x4F,0x43,0x64,0x6E,0x3C,0x3A,0x01,0x2F
,0xDD,0x24,0xBE,0x68,0x91,0x6D,0x3E,0xA8,0xC6,0x4B,0xBE,0x68,0x91,0x6D,0x3E
,0xE5,0xD0,0xE2,0xBE,0x4C,0x37,0x09,0x3E,0x31,0x08,0xDC,0xBE,0x60,0xE5,0xD0
,0x3D,0xD7,0xA4,0xCB,0xBE,0x2B,0x33,0xE5,0x3D,0x7F,0xBE,0xBD,0xBE,0xF0,0xF7
,0xEB,0x3D,0xFE,0x2B,0xB3,0xBE,0x7A,0x1C,0xE6,0x3D,0xCA,0xF9,0xAA,0xBE,0x22
,0x8B,0xD4,0x3D,0x48,0xDF,0xA4,0xBE,0x93,0xC4,0xB2,0x3D,0x75,0x93,0xA0,0xBE
,0xEF,0x92,0x78,0x3D,0x29,0x08,0x9E,0xBE,0xCA,0x17,0x34,0x3A,0x1B,0x2F,0x9D
,0xBE,0x68,0x91,0xED,0xBD,0x1B,0x2F,0x9D,0xBE,0x19,0x04,0xD6,0xBE,0x58,0xE4
,0x9F,0xBE,0x96,0xAF,0xFB,0xBE,0x10,0x04,0xA8,0xBE,0x97,0x6F,0x09,0xBF,0x3B
,0xDF,0xB7,0xBE,0x40,0x6A,0x0F,0xBF,0xB0,0xC6,0xD1,0xBE,0x73,0x68,0x11,0xBF
,0xBE,0x9F,0xDA,0xBE,0x73,0x68,0x11,0xBF,0xBE,0x9F,0xDA,0xBE,0x9A,0x99,0x19
,0xBF,0x68,0x91,0xED,0xBC,0x9A,0x99,0x19,0xBF,0x68,0x91,0xED,0xBC,0x73,0x68
,0x11,0xBF,0x5C,0x8D,0xAC,0xBD,0x65,0xA6,0x10,0xBF,0x9F,0xE7,0xEF,0xBD,0x3D
,0x60,0x0E,0xBF,0xE6,0xCB,0x0B,0xBE,0x57,0x79,0x0A,0xBF,0x48,0xE1,0x1A,0xBE
,0xFE,0xD4,0x04,0xBF,0x14,0x5E,0x22,0xBE,0xF2,0xD0,0xF7,0xBE,0x2F,0xDD,0x24
,0xBE,0x19,0x04,0xD6,0xBE,0x2F,0xDD,0x24,0xBE,0x14,0xAF,0x32,0x3C,0x32,0x93
,0x68,0xBD,0xF3,0x1E,0xC7,0x3D,0x93,0x18,0x44,0x3D,0x25,0x06,0x01,0x3E,0x89
,0xEB,0xD8,0x3D,0x57,0x60,0xE8,0x3D,0xD8,0xD2,0x13,0x3E,0xF7,0x5B,0x9B,0x3D
,0xF6,0xD1,0x29,0x3E,0x14,0x5C,0x2C,0x3C,0xE9,0x26,0x31,0x3E,0x4D,0x87,0xAE
,0xBD,0xE9,0x26,0x31,0x3E,0x19,0x04,0xD6,0xBE,0x89,0x41,0x30,0x3E,0xD7,0x50
,0xF2,0xBE,0x68,0x91,0x2D,0x3E,0xBD,0x1E,0x00,0xBF,0x6B,0x11,0x21,0x3E,0x63
,0x9C,0x07,0xBF,0xBB,0x9D,0x0D,0x3E,0x31,0x07,0x0D,0xBF,0x39,0x0E,0xDC,0x3D
,0x27,0x50,0x10,0xBF,0x5B,0x7A,0x74,0x3D,0x73,0x68,0x11,0xBF,0xC5,0x20,0x30
,0x3D,0x73,0x68,0x11,0xBF,0xC5,0x20,0x30,0x3D,0x9A,0x99,0x19,0xBF,0xE5,0xD0
,0xE2,0x3E,0x9A,0x99,0x19,0xBF,0xE5,0xD0,0xE2,0x3E,0x73,0x68,0x11,0xBF,0x96
,0x96,0xC9,0x3E,0x41,0x9F,0x10,0xBF,0xB0,0xC6,0xB9,0x3E,0x9A,0x43,0x0E,0xBF
,0x40,0x33,0xB0,0x3E,0x7C,0x80,0x0A,0xBF,0x0C,0xAE,0xA9,0x3E,0x06,0x81,0x05
,0xBF,0x8E,0xEA,0xA4,0x3E,0xC4,0xCC,0xF6,0xBE,0xF8,0x53,0xA3,0x3E,0x19,0x04
,0xD6,0xBE,0xF8,0x53,0xA3,0x3E,0x72,0xDD,0x94,0xBD,0x02,0xD8,0xA0,0x3E,0x2F
,0xDC,0xB9,0x3C,0x63,0x64,0x99,0x3E,0xE1,0xEF,0xB7,0x3D,0x2A,0x72,0x88,0x3E
,0xA4,0xC4,0x1E,0x3E,0x66,0x12,0x65,0x3E,0xE3,0xA5,0x4B,0x3E,0x4C,0x8A,0x2F
,0x3E,0x98,0x16,0x65,0x3E,0x25,0x06,0xE1,0x3D,0x68,0x91,0x6D,0x3E,0xA2,0x45
,0xB6,0xBC,0xF3,0x55,0x42,0x3E,0x2F,0xDD,0x24,0xBE,0xAE,0x47,0x81,0x3D,0x01
,0x02,0x03,0x01,0x03,0x04,0x01,0x04,0x05,0x01,0x05,0x06,0x01,0x06,0x07,0x2A
,0x2B,0x2C,0x2A,0x2C,0x2D,0x2A,0x2D,0x2E,0x2A,0x2E,0x2F,0x2A,0x2F,0x30,0x28
,0x29,0x2A,0x12,0x13,0x14,0x12,0x14,0x15,0x12,0x15,0x16,0x12,0x16,0x17,0x12
,0x17,0x18,0x10,0x11,0x12,0x3B,0x00,0x01,0x3B,0x01,0x07,0x3B,0x07,0x08,0x3B
,0x08,0x09,0x3B,0x09,0x0A,0x2A,0x30,0x31,0x12,0x18,0x19,0x3B,0x0A,0x0B,0x27
,0x28,0x2A,0x27,0x2A,0x31,0x26,0x27,0x31,0x25,0x26,0x31,0x24,0x25,0x31,0x23
,0x24,0x31,0x23,0x31,0x32,0x22,0x23,0x32,0x21,0x22,0x32,0x21,0x32,0x33,0x20
,0x21,0x33,0x20,0x33,0x34,0x20,0x34,0x35,0x1F,0x20,0x35,0x1F,0x35,0x36,0x1F
,0x36,0x37,0x1E,0x1F,0x37,0x1E,0x37,0x38,0x1E,0x38,0x39,0x1E,0x39,0x3A,0x1D
,0x1E,0x3A,0x1C,0x1D,0x3A,0x1C,0x3A,0x3B,0x1B,0x1C,0x3B,0x1B,0x3B,0x0B,0x1A
,0x1B,0x0B,0x1A,0x0B,0x0C,0x19,0x1A,0x0C,0x19,0x0C,0x0D,0x19,0x0D,0x0E,0x0F
,0x10,0x12,0x0F,0x12,0x19,0x0F,0x19,0x0E,0x3B,0x6F,0x26,0x26,0x02,0xA3,0xCC
,0xC6,0xBD,0xEA,0x23,0x60,0x3E,0xB3,0x45,0x42,0xBE,0x2B,0xDB,0x37,0x3E,0x09
,0xC1,0x8A,0xBE,0xF5,0xF3,0xE6,0x3D,0xA4,0x72,0xAB,0xBE,0x01,0x15,0x8E,0x3C
,0x41,0x9F,0xC0,0xBE,0x85,0xEF,0xBD,0xBD,0x14,0xAE,0xC7,0xBE,0x4E,0x62,0x50
,0xBE,0x41,0x0C,0xBC,0xBE,0xA4,0x6F,0xB2,0xBE,0xE9,0x26,0x99,0xBE,0x04,0x02
,0xF5,0xBE,0x40,0xDD,0x30,0xBE,0x1E,0x6F,0x16,0xBF,0x35,0x60,0x10,0xBC,0x77
,0xBE,0x1F,0xBF,0x41,0xD5,0xC8,0x3D,0x14,0x22,0x1C,0xBF,0x86,0x3B,0x47,0x3E
,0xCB,0x4C,0x11,0xBF,0x21,0x58,0x8D,0x3E,0x45,0xB7,0xFE,0xBE,0x36,0x06,0xAD
,0x3E,0x06,0xD7,0xCC,0xBE,0x0C,0x04,0xC1,0x3E,0x51,0x4E,0x94,0xBE,0x14,0xAE
,0xC7,0x3E,0x3F,0xE5,0x38,0xBE,0x85,0x98,0xBB,0x3E,0xD8,0xD6,0x0F,0xBD,0x19
,0x58,0x97,0x3E,0x76,0xE3,0xBD,0x3D,0xCA,0x4F,0x2A,0x3E,0x8E,0xE9,0x49,0x3E
,0x42,0x60,0x65,0x3A,0x68,0x91,0x6D,0x3E,0xD4,0x0B,0x7E,0x3D,0x23,0x83,0x1C
,0x3E,0xF1,0x82,0x08,0x3E,0x3D,0x0A,0xB7,0x3D,0x25,0xB2,0x4F,0x3E,0x29,0x3F
,0x69,0xBD,0x8B,0x6C,0x67,0x3E,0x7F,0x12,0x7F,0xBE,0x38,0x30,0x59,0x3E,0xE2
,0x8F,0xCA,0xBE,0xFA,0x7A,0x2E,0x3E,0xCA,0xA3,0xFB,0xBE,0x60,0xB0,0xDB,0x3D
,0xE6,0x5B,0x0B,0xBF,0x8D,0x97,0xEE,0x3C,0x3B,0xDF,0x0F,0xBF,0x11,0x8B,0x98
,0xBD,0x29,0xB1,0x07,0xBF,0x71,0x8D,0x1F,0xBE,0xE4,0x4D,0xDE,0xBE,0xB4,0x74
,0x55,0xBE,0xFC,0xA8,0x96,0xBE,0x8B,0x6C,0x67,0xBE,0x5D,0x37,0x05,0xBE,0xD3
,0xF6,0x5F,0xBE,0xCD,0xAE,0xFB,0xBC,0x68,0x95,0x49,0xBE,0xDE,0xFF,0x47,0x3D
,0xA3,0x1D,0x27,0xBE,0x1B,0x2F,0xDD,0x3D,0xB4,0xC8,0xF6,0xBD,0x5E,0xBE,0x15
,0x3E,0xD4,0x49,0x96,0xBD,0x84,0x11,0x2B,0x3E,0x84,0xD5,0xD8,0xBC,0x0E,0x2D
,0x32,0x3E,0x01,0x21,0x22,0x21,0x01,0x02,0x01,0x22,0x23,0x00,0x01,0x23,0x00
,0x23,0x24,0x13,0x00,0x24,0x13,0x24,0x25,0x13,0x25,0x14,0x12,0x13,0x14,0x12
,0x14,0x15,0x11,0x12,0x15,0x11,0x15,0x16,0x10,0x11,0x16,0x21,0x02,0x03,0x20
,0x21,0x03,0x20,0x03,0x04,0x1F,0x20,0x04,0x1F,0x04,0x05,0x1E,0x1F,0x05,0x1E
,0x05,0x06,0x1E,0x06,0x07,0x1D,0x1E,0x07,0x1D,0x07,0x08,0x1C,0x1D,0x08,0x1C
,0x08,0x09,0x1B,0x1C,0x09,0x1B,0x09,0x0A,0x1A,0x1B,0x0A,0x1A,0x0A,0x0B,0x19
,0x1A,0x0B,0x19,0x0B,0x0C,0x18,0x19,0x0C,0x18,0x0C,0x0D,0x17,0x18,0x0D,0x17
,0x0D,0x0E,0x17,0x0E,0x0F,0x16,0x17,0x0F,0x10,0x16,0x0F,0x13,0x25,0x70,0x41
,0x41,0x02,0xAA,0x9D,0xD1,0xBE,0x60,0xE5,0xD0,0x3D,0x25,0x5A,0xC2,0xBE,0x4B
,0xE7,0xE3,0x3D,0x29,0x08,0xB6,0xBE,0x71,0x3D,0xEA,0x3D,0xC6,0xE0,0xA9,0xBE
,0x07,0xCF,0xE4,0x3D,0x4C,0x8D,0xA0,0xBE,0x45,0x83,0xD4,0x3D,0xD4,0xB6,0x99
,0xBE,0x29,0x5A,0xB9,0x3D,0x36,0x06,0x95,0xBE,0x3B,0x54,0x93,0x3D,0xEC,0x4F
,0x92,0xBE,0x72,0xDD,0x14,0x3D,0x73,0x68,0x91,0xBE,0x42,0x60,0xE5,0xBC,0x73
,0x68,0x91,0xBE,0x96,0x43,0x4B,0xBF,0x9E,0x97,0x92,0xBE,0xC3,0x9F,0x5D,0xBF
,0xDD,0x24,0x96,0xBE,0xAC,0x70,0x67,0xBF,0xFE,0xD5,0x9B,0xBE,0x4C,0x38,0x6C
,0xBF,0xAC,0x70,0xA3,0xBE,0xDD,0x78,0x6F,0xBF,0xFA,0xD3,0xB6,0xBE,0xE5,0x25
,0x73,0xBF,0xF2,0xD2,0xCD,0xBE,0x58,0x39,0x74,0xBF,0x87,0x16,0xD9,0xBE,0x58
,0x39,0x74,0xBF,0x87,0x16,0xD9,0xBE,0x7F,0x6A,0x7C,0xBF,0x6F,0x12,0x03,0xBB
,0x7F,0x6A,0x7C,0xBF,0x6F,0x12,0x03,0xBB,0x58,0x39,0x74,0xBF,0x6A,0xF7,0x6B
,0xBD,0x02,0x61,0x73,0xBF,0xBE,0x14,0xBE,0xBD,0x12,0xD8,0x70,0xBF,0xEC,0xA5
,0xE9,0xBD,0x7A,0xDF,0x6C,0xBF,0xCA,0xF9,0x02,0xBE,0x4E,0xB8,0x67,0xBF,0x7C
,0xBA,0x0A,0xBE,0x2E,0x1E,0x5E,0xBF,0xDF,0x4F,0x0D,0xBE,0xCD,0xCC,0x4C,0xBF
,0xDF,0x4F,0x0D,0xBE,0x04,0xAC,0x0D,0xBF,0xB5,0xA7,0xC4,0xBD,0xBB,0x09,0x16
,0xBF,0x31,0x7D,0x6F,0xBD,0x4E,0x0E,0x1B,0xBF,0x38,0x9E,0xCF,0xBB,0x71,0x92
,0x1E,0xBF,0x77,0xBE,0x5F,0x3D,0x77,0xBE,0x1F,0xBF,0xC6,0xA3,0x44,0x3E,0xFB
,0x93,0x18,0xBF,0x61,0x8D,0x9B,0x3E,0x77,0x14,0x03,0xBF,0x35,0xB4,0xC9,0x3E
,0xB2,0xF3,0xB6,0xBE,0x87,0x16,0xD9,0x3E,0xD9,0x76,0x2A,0xBE,0xD7,0x4E,0xCC
,0x3E,0xE0,0x4D,0x37,0x3B,0xC6,0xF7,0xA5,0x3E,0x52,0x10,0x0C,0x3E,0xC3,0x45
,0x6E,0x3E,0x12,0x31,0x55,0x3E,0x91,0x9D,0xF7,0x3D,0x68,0x91,0x6D,0x3E,0x94
,0x14,0x58,0x3D,0x28,0xB6,0x62,0x3E,0xCE,0xE2,0x45,0xBC,0xEE,0x24,0x42,0x3E
,0x9F,0xE5,0x99,0xBD,0xF2,0x7C,0x06,0x3E,0xDF,0x4F,0x0D,0xBE,0xD5,0x78,0x29
,0x3D,0xDF,0x4F,0x0D,0xBE,0xD5,0x78,0x69,0x3E,0xB2,0x9D,0x2F,0xBE,0xD5,0x78
,0x69,0x3E,0x3D,0x0A,0xD7,0xBE,0xB8,0x1E,0x05,0x3E,0xD0,0x47,0x99,0xBD,0xD8
,0x9B,0x58,0x3D,0xB8,0x1E,0x05,0xBD,0x68,0x91,0xAD,0x3D,0x9A,0x99,0x19,0x3C
,0x83,0x35,0xCE,0x3D,0xBF,0xB5,0x53,0x3D,0x87,0x16,0xD9,0x3D,0xB5,0xC0,0xFE
,0x3D,0x5E,0xBE,0xB5,0x3D,0x57,0xE9,0x3E,0x3E,0xCE,0x6C,0x17,0x3D,0x37,0x8D
,0x7D,0x3E,0x65,0x1A,0xAD,0xBD,0x4C,0x37,0x89,0x3E,0x9A,0xE9,0x7E,0xBE,0x29
,0x06,0x80,0x3E,0x50,0xE4,0xC9,0xBE,0x3F,0xE5,0x48,0x3E,0x94,0xC2,0x00,0xBF
,0xF7,0x01,0x08,0x3E,0x92,0x03,0x0E,0xBF,0xCE,0x6C,0x57,0x3D,0x98,0x6E,0x12
,0xBF,0x62,0x84,0x70,0xBC,0xAA,0xF0,0x0F,0xBF,0xF9,0xF3,0x8D,0xBD,0xD1,0x76
,0x08,0xBF,0xD8,0x82,0xDE,0xBD,0x5E,0x67,0xFB,0xBE,0xD9,0x26,0x05,0xBE,0x31
,0xB4,0xE2,0xBE,0xAE,0x45,0x0B,0xBE,0x64,0x3C,0xCA,0xBE,0xDF,0x4F,0x0D,0xBE
,0x54,0x8B,0xA0,0xBE,0xDF,0x4F,0x0D,0xBE,0x08,0xAB,0x31,0xBC,0x40,0x04,0x05
,0x40,0x05,0x06,0x40,0x06,0x07,0x40,0x07,0x08,0x3F,0x40,0x08,0x3F,0x08,0x09
,0x3E,0x3F,0x09,0x3D,0x3E,0x09,0x3C,0x3D,0x09,0x2A,0x2B,0x2C,0x11,0x12,0x13
,0x11,0x13,0x14,0x11,0x14,0x15,0x11,0x15,0x16,0x11,0x16,0x17,0x0F,0x10,0x11
,0x2C,0x2D,0x00,0x2C,0x00,0x01,0x2C,0x01,0x02,0x2C,0x02,0x03,0x2C,0x03,0x04
,0x2A,0x2C,0x04,0x2A,0x04,0x40,0x2A,0x40,0x2E,0x29,0x2A,0x2E,0x29,0x2E,0x2F
,0x28,0x29,0x2F,0x28,0x2F,0x30,0x11,0x17,0x18,0x28,0x30,0x31,0x27,0x28,0x31
,0x26,0x27,0x31,0x26,0x31,0x32,0x25,0x26,0x32,0x25,0x32,0x33,0x24,0x25,0x33
,0x23,0x24,0x33,0x23,0x33,0x34,0x22,0x23,0x34,0x22,0x34,0x35,0x21,0x22,0x35
,0x21,0x35,0x36,0x20,0x21,0x36,0x20,0x36,0x37,0x1F,0x20,0x37,0x1F,0x37,0x38
,0x1E,0x1F,0x38,0x1E,0x38,0x39,0x1D,0x1E,0x39,0x1D,0x39,0x3A,0x1C,0x1D,0x3A
,0x1C,0x3A,0x3B,0x1B,0x1C,0x3B,0x1A,0x1B,0x3B,0x1A,0x3B,0x3C,0x1A,0x3C,0x09
,0x19,0x1A,0x09,0x19,0x09,0x0A,0x18,0x19,0x0A,0x18,0x0A,0x0B,0x0E,0x0F,0x11
,0x0E,0x11,0x18,0x0D,0x0E,0x18,0x0C,0x0D,0x18,0x0C,0x18,0x0B,0x2D,0x40,0x71
,0x38,0x38,0x02,0xCB,0xA1,0x85,0x3E,0x68,0x91,0x6D,0x3E,0x50,0x35,0x5A,0x3E
,0x84,0xBC,0x4E,0x3E,0x79,0xE9,0x26,0x3E,0x40,0x85,0x33,0x3E,0x05,0x52,0x02
,0x3E,0x6C,0xEA,0x4C,0x3E,0x98,0x6E,0xB2,0x3D,0x4E,0x0E,0x5F,0x3E,0xE8,0x4B
,0x2F,0x3D,0xA2,0xF0,0x69,0x3E,0x58,0x39,0xB4,0xBB,0x68,0x91,0x6D,0x3E,0x7D
,0xEA,0x28,0xBE,0x0A,0x81,0x4C,0x3E,0x65,0x39,0x99,0xBE,0xE0,0x9F,0xD2,0x3D
,0x44,0xF8,0xC7,0xBE,0x23,0xDB,0x39,0xBD,0x50,0x8D,0xD7,0xBE,0xBF,0x47,0x6D
,0xBE,0x98,0xC2,0xCB,0xBE,0x8E,0x01,0xC9,0xBE,0x4E,0x62,0xA8,0xBE,0x9C,0x6F
,0x04,0xBF,0x11,0x00,0x6C,0xBE,0xC0,0xEA,0x18,0xBF,0x6A,0xBC,0xF4,0xBD,0x77
,0xBE,0x1F,0xBF,0x6E,0x51,0x66,0xBD,0x78,0x7E,0x1D,0xBF,0x33,0x34,0x9E,0x3B
,0x7B,0xBE,0x16,0xBF,0x78,0x0C,0x8F,0x3D,0x88,0xD7,0x09,0xBF,0xE1,0x7A,0x14
,0x3E,0xB3,0x45,0xEA,0xBE,0xE1,0x7A,0x14,0x3E,0x04,0x56,0x4E,0xBF,0x9B,0xC7
,0x11,0x3E,0x3F,0xFF,0x5D,0xBF,0x0C,0xAE,0x09,0x3E,0x5C,0x8E,0x67,0xBF,0xD4
,0x2A,0xFA,0x3D,0xA1,0x10,0x6D,0xBF,0xE2,0xCA,0xD9,0x3D,0x75,0x93,0x70,0xBF
,0xC4,0x7C,0x99,0x3D,0xE8,0x4F,0x73,0xBF,0x3A,0xCA,0x01,0x3D,0x58,0x39,0x74
,0xBF,0x8F,0xC2,0x75,0x3C,0x58,0x39,0x74,0xBF,0x8F,0xC2,0x75,0x3C,0x7F,0x6A
,0x7C,0xBF,0x50,0x8D,0xD7,0x3E,0x7F,0x6A,0x7C,0xBF,0x50,0x8D,0xD7,0x3E,0x58
,0x39,0x74,0xBF,0x85,0xE9,0xBB,0x3E,0xCE,0x6C,0x73,0xBF,0x8F,0x1A,0xAB,0x3E
,0x31,0x07,0x71,0xBF,0x58,0x38,0xA1,0x3E,0x7A,0x34,0x6D,0xBF,0x25,0x5A,0x9A
,0x3E,0xC5,0x20,0x68,0xBF,0xF8,0x54,0x96,0x3E,0xFA,0xD3,0x5E,0xBF,0xF4,0xFD
,0x94,0x3E,0x04,0x56,0x4E,0xBF,0xF4,0xFD,0x94,0x3E,0x68,0x91,0x6D,0x3E,0x8A
,0x1E,0xD8,0x3D,0xBC,0x5C,0xE4,0xBE,0xCC,0xD4,0x84,0x3D,0xF6,0x7E,0xF3,0xBE
,0x6D,0xC9,0xAA,0x3C,0x33,0x1B,0xFC,0xBE,0xFD,0xBF,0xEA,0xBC,0xDB,0xF9,0xFE
,0xBE,0x59,0x31,0xFC,0xBD,0x06,0x2D,0xF4,0xBE,0xD3,0xA5,0x4F,0xBE,0xA8,0xC6
,0xD3,0xBE,0x4C,0x8E,0x83,0xBE,0x6D,0x8F,0x9E,0xBE,0xCD,0xCC,0x8C,0xBE,0xBE
,0x9F,0x2A,0xBE,0xD0,0xB9,0x83,0xBE,0x53,0xCE,0x97,0xBC,0xF3,0x01,0x51,0xBE
,0xE9,0x26,0xB1,0x3D,0x0F,0x2A,0x01,0xBE,0x39,0xB4,0x18,0x3E,0x96,0x43,0x0B
,0xBD,0x7B,0x14,0x2E,0x3E,0xB6,0x4B,0x9B,0x3C,0xE7,0x53,0x27,0x3E,0xEF,0x74
,0x87,0x3D,0x6F,0x12,0x13,0x3E,0x47,0x58,0xD4,0x3D,0x80,0x49,0xEA,0x3D,0xC5
,0xC8,0x02,0x3E,0xA5,0xC0,0xA2,0x3D,0x6B,0x0E,0x10,0x3E,0x10,0xEB,0x0D,0x3D
,0xE1,0x7A,0x14,0x3E,0xF0,0xA7,0xC6,0xBC,0xE1,0x7A,0x14,0x3E,0x96,0xEB,0xCD
,0xBE,0x34,0x00,0x01,0x34,0x01,0x02,0x33,0x34,0x02,0x33,0x02,0x03,0x32,0x33
,0x03,0x32,0x03,0x04,0x31,0x32,0x04,0x31,0x04,0x05,0x31,0x05,0x06,0x30,0x31
,0x06,0x30,0x06,0x07,0x2F,0x30,0x07,0x23,0x24,0x00,0x1B,0x1C,0x1D,0x1B,0x1D
,0x1E,0x1B,0x1E,0x1F,0x1B,0x1F,0x20,0x1B,0x20,0x21,0x19,0x1A,0x1B,0x2E,0x2F
,0x07,0x2E,0x07,0x08,0x2D,0x2E,0x08,0x2D,0x08,0x09,0x2C,0x2D,0x09,0x2C,0x09
,0x0A,0x2B,0x2C,0x0A,0x2B,0x0A,0x0B,0x2A,0x2B,0x0B,0x2A,0x0B,0x0C,0x2A,0x0C
,0x0D,0x29,0x2A,0x0D,0x29,0x0D,0x0E,0x28,0x29,0x0E,0x28,0x0E,0x0F,0x28,0x0F
,0x10,0x28,0x10,0x11,0x27,0x28,0x11,0x26,0x27,0x11,0x26,0x11,0x12,0x25,0x26
,0x12,0x37,0x25,0x12,0x1B,0x21,0x22,0x18,0x19,0x1B,0x00,0x34,0x35,0x23,0x00
,0x35,0x23,0x35,0x36,0x23,0x36,0x37,0x23,0x37,0x12,0x23,0x12,0x13,0x22,0x23
,0x13,0x22,0x13,0x14,0x22,0x14,0x15,0x22,0x15,0x16,0x22,0x16,0x17,0x1B,0x22
,0x17,0x18,0x1B,0x17,0x24,0x37,0x72,0x2F,0x2D,0x01,0x54,0xE3,0x65,0xBD,0x68
,0x91,0x6D,0x3E,0xAC,0x1C,0x9A,0xBE,0x4C,0x37,0x09,0x3E,0x79,0x95,0x95,0xBE
,0x60,0xE5,0xD0,0x3D,0x16,0xDB,0x84,0xBE,0x2B,0x33,0xE5,0x3D,0x4C,0xDF,0x6B
,0xBE,0xF0,0xF7,0xEB,0x3D,0xAE,0x46,0x56,0xBE,0x15,0xA8,0xE5,0x3D,0x33,0x87
,0x44,0xBE,0x84,0xB8,0xD2,0x3D,0x0A,0x31,0x37,0xBE,0x5B,0x08,0xB2,0x3D,0xE5
,0xD4,0x2E,0xBE,0xB8,0x76,0x82,0x3D,0xAD,0x6D,0x2A,0xBE,0x0D,0xC6,0x08,0x3C
,0xC3,0xF5,0x28,0xBE,0xF4,0xFD,0xD4,0xBD,0xC3,0xF5,0x28,0xBE,0x19,0x04,0xD6
,0xBE,0x7A,0x18,0x2A,0xBE,0x39,0x0B,0xF3,0xBE,0x28,0x81,0x2D,0xBE,0x89,0xEB
,0x00,0xBF,0x3A,0x08,0x3A,0xBE,0x7F,0xC0,0x07,0xBF,0x63,0x60,0x4D,0xBE,0xA8
,0x70,0x0C,0xBF,0x02,0x83,0x74,0xBE,0x78,0x2A,0x10,0xBF,0x19,0x04,0x96,0xBE
,0x73,0x68,0x11,0xBF,0x19,0x04,0x96,0xBE,0x9A,0x99,0x19,0xBF,0x00,0x00,0x00
,0x3E,0x9A,0x99,0x19,0xBF,0x00,0x00,0x00,0x3E,0x73,0x68,0x11,0xBF,0x90,0x12
,0x9B,0x3D,0x2C,0x66,0x10,0xBF,0xC1,0xCA,0x21,0x3D,0x35,0x5F,0x0D,0xBF,0x85
,0x5F,0x6A,0x3C,0xBB,0xD4,0x08,0xBF,0xBF,0xD2,0xF9,0xBA,0xA6,0x47,0x03,0xBF
,0x84,0x49,0x71,0xBC,0x79,0xE7,0xF0,0xBE,0xE3,0xA5,0x9B,0xBC,0xAA,0xF1,0xD2
,0xBE,0xE3,0xA5,0x9B,0xBC,0x81,0x95,0xC3,0xBC,0x46,0x06,0xB9,0x3C,0xE5,0x28
,0x40,0x3D,0x5B,0xB4,0x80,0x3D,0xF7,0x78,0xC1,0x3D,0xF9,0x2E,0xA5,0x3D,0x4D
,0x66,0xDC,0x3D,0xBA,0xF9,0xC6,0x3D,0x42,0x60,0xE5,0x3D,0x99,0xF1,0xF6,0x3D
,0x19,0x04,0xD6,0x3D,0x89,0x99,0x1D,0x3E,0x9E,0xEF,0xA7,0x3D,0x78,0xED,0x42
,0x3E,0x46,0xB6,0x73,0x3D,0x02,0x83,0x64,0x3E,0xF4,0xFD,0x54,0x3D,0x8F,0x6D
,0x81,0x3E,0x52,0x64,0x6D,0x3D,0x04,0x56,0x8E,0x3E,0xB6,0x4B,0x9B,0x3D,0x02
,0x2B,0x97,0x3E,0x30,0x83,0xD1,0x3D,0xAC,0x1C,0x9A,0x3E,0xD0,0x47,0x09,0x3E
,0xBF,0x0C,0x96,0x3E,0x11,0xA8,0x2E,0x3E,0x3C,0xDD,0x89,0x3E,0xE8,0x4B,0x4F
,0x3E,0xAE,0xF0,0x6E,0x3E,0xE7,0xFF,0x65,0x3E,0x58,0x91,0x41,0x3E,0x68,0x91
,0x6D,0x3E,0x49,0x68,0xAB,0x3D,0x5F,0x0A,0x3F,0x3E,0xE3,0xA5,0x9B,0xBC,0xF2
,0xD2,0x4D,0x3D,0xE3,0xA5,0x9B,0xBC,0x68,0x91,0x6D,0x3E,0x01,0x02,0x03,0x00
,0x01,0x03,0x00,0x03,0x04,0x00,0x04,0x05,0x00,0x05,0x06,0x2D,0x2E,0x00,0x2D
,0x00,0x06,0x2D,0x06,0x07,0x2D,0x07,0x08,0x2D,0x08,0x09,0x12,0x13,0x14,0x12
,0x14,0x15,0x12,0x15,0x16,0x12,0x16,0x17,0x12,0x17,0x18,0x12,0x18,0x19,0x10
,0x11,0x12,0x2D,0x09,0x0A,0x2D,0x0A,0x0B,0x2A,0x2B,0x2C,0x29,0x2A,0x2C,0x28
,0x29,0x2C,0x27,0x28,0x2C,0x26,0x27,0x2C,0x25,0x26,0x2C,0x24,0x25,0x2C,0x23
,0x24,0x2C,0x22,0x23,0x2C,0x21,0x22,0x2C,0x20,0x21,0x2C,0x1F,0x20,0x2C,0x1E
,0x1F,0x2C,0x1E,0x2C,0x2D,0x1D,0x1E,0x2D,0x1C,0x1D,0x2D,0x1B,0x1C,0x2D,0x1B
,0x2D,0x0B,0x1A,0x1B,0x0B,0x19,0x1A,0x0B,0x19,0x0B,0x0C,0x19,0x0C,0x0D,0x19
,0x0D,0x0E,0x19,0x0E,0x0F,0x12,0x19,0x0F,0x10,0x12,0x0F,0x2E,0x73,0x3B,0x39
,0x01,0x7F,0x6A,0x3C,0x3E,0x68,0x91,0x6D,0x3E,0xCB,0x47,0x32,0x3E,0xD9,0x79
,0x5B,0x3E,0xF5,0x80,0x29,0x3E,0x1B,0x2B,0x51,0x3E,0xE9,0xD2,0x1F,0x3E,0x08
,0x5B,0x4C,0x3E,0xCA,0xF9,0x12,0x3E,0x83,0xC0,0x4A,0x3E,0xFF,0x07,0xF8,0x3D
,0x38,0x2F,0x4E,0x3E,0xDE,0x74,0xAB,0x3D,0xD1,0x7A,0x58,0x3E,0x89,0x7B,0xAC
,0x3C,0xE4,0x4B,0x68,0x3E,0x25,0x06,0x01,0xBD,0x68,0x91,0x6D,0x3E,0x97,0xC5
,0x04,0xBE,0x41,0x65,0x5C,0x3E,0xDD,0xD0,0x54,0xBE,0x50,0xE1,0x28,0x3E,0x03
,0x96,0x84,0xBE,0x3F,0xAC,0xB7,0x3D,0xDF,0x4F,0x8D,0xBE,0x8F,0xC2,0xF5,0xBA
,0x6D,0x3C,0x88,0xBE,0x2F,0x89,0x93,0xBD,0x29,0x04,0x72,0xBE,0x73,0x10,0x04
,0xBE,0xA0,0xC5,0x32,0xBE,0x58,0x8F,0x3B,0xBE,0x5F,0x62,0x8C,0xBD,0x4C,0x37
,0x79,0xBE,0x4C,0x33,0x1D,0x3D,0x12,0x30,0x9A,0xBE,0x19,0x8F,0xD2,0x3D,0x1F
,0x31,0xB2,0xBE,0xA8,0x72,0x0A,0x3E,0x50,0xE4,0xC9,0xBE,0x06,0x81,0x15,0x3E
,0x4B,0x8F,0xE6,0xBE,0x6F,0xBC,0x0B,0x3E,0x19,0x70,0xFE,0xBE,0x51,0xDD,0xDC
,0x3D,0xC3,0x4A,0x09,0xBF,0xB7,0x43,0x83,0x3D,0x9E,0x25,0x10,0xBF,0x0C,0x03
,0x16,0x3C,0x98,0x6E,0x12,0xBF,0x7A,0xC8,0x94,0xBD,0xA4,0x6F,0x0E,0xBF,0xCB
,0xA1,0x15,0xBE,0xA8,0x72,0x02,0xBF,0xE9,0x26,0x51,0xBE,0xF2,0x28,0xDD,0xBE
,0x8F,0xC2,0x75,0xBE,0x54,0xE3,0xA5,0xBE,0x96,0x43,0x8B,0xBE,0x54,0xE3,0xA5
,0xBE,0x96,0x43,0x8B,0xBE,0x08,0xAC,0x1C,0xBF,0x8F,0xC2,0x75,0xBE,0x08,0xAC
,0x1C,0xBF,0xE1,0xD4,0x67,0xBE,0xB0,0x72,0x18,0xBF,0x62,0x68,0x55,0xBE,0x3D
,0x0A,0x17,0xBF,0x1B,0xD7,0x3F,0xBE,0x81,0x95,0x17,0xBF,0x4F,0xE5,0x24,0xBE
,0x4C,0x37,0x19,0xBF,0xFB,0x76,0x92,0xBD,0xAC,0x1C,0x1E,0xBF,0x83,0xBE,0xF4
,0x3B,0x77,0xBE,0x1F,0xBF,0xF6,0x7A,0xD7,0x3D,0x98,0x6D,0x1B,0xBF,0x3D,0xB6
,0x45,0x3E,0xEA,0x7A,0x0E,0xBF,0xC0,0xB2,0x82,0x3E,0xD3,0xF5,0xF4,0xBE,0xDF
,0x4F,0x8D,0x3E,0x29,0x04,0xC2,0xBE,0x73,0xBA,0x6C,0x3E,0x0C,0x01,0x80,0xBE
,0x9C,0x14,0xC6,0x3D,0x90,0x6A,0x18,0xBE,0x93,0x18,0x04,0xBD,0xD7,0xA3,0xB0
,0xBD,0x24,0x7F,0xD0,0xBD,0xD3,0xA5,0x3F,0xBD,0x6A,0x14,0x12,0xBE,0xE9,0x27
,0x1C,0xBC,0x77,0x14,0x27,0xBE,0xB2,0x9D,0xEF,0x3C,0x7B,0x14,0x2E,0xBE,0xE2
,0xCA,0x99,0x3D,0xA3,0x75,0x24,0xBE,0x5A,0x68,0xE7,0x3D,0xA2,0x99,0x07,0xBE
,0x79,0xE9,0x16,0x3E,0xB3,0x26,0xB6,0xBD,0xE0,0xF7,0x2F,0x3E,0xE0,0xDA,0x09
,0xBD,0xEC,0x51,0x38,0x3E,0x1B,0x7F,0x22,0x3D,0xF9,0x4E,0x2C,0x3E,0xB1,0x4D
,0xCA,0x3D,0x99,0x45,0x08,0x3E,0x92,0x97,0x15,0x3E,0x2C,0xD5,0x85,0x3D,0x7F
,0x6A,0x3C,0x3E,0x58,0x39,0x34,0xBD,0x1B,0x2F,0x5D,0x3E,0x58,0x39,0x34,0xBD
,0x1B,0x2F,0x5D,0x3E,0x68,0x91,0x6D,0x3E,0x39,0x3A,0x00,0x39,0x00,0x01,0x39
,0x01,0x02,0x39,0x02,0x03,0x38,0x39,0x03,0x37,0x38,0x03,0x36,0x37,0x03,0x36
,0x03,0x04,0x36,0x04,0x05,0x35,0x36,0x05,0x35,0x05,0x06,0x35,0x06,0x07,0x34
,0x35,0x07,0x34,0x07,0x08,0x34,0x08,0x09,0x33,0x34,0x09,0x32,0x33,0x09,0x32
,0x09,0x0A,0x31,0x32,0x0A,0x31,0x0A,0x0B,0x30,0x31,0x0B,0x1D,0x1E,0x1F,0x1D
,0x1F,0x20,0x1C,0x1D,0x20,0x1B,0x1C,0x20,0x1A,0x1B,0x20,0x1A,0x20,0x21,0x1A
,0x21,0x22,0x1A,0x22,0x23,0x19,0x1A,0x23,0x19,0x23,0x24,0x18,0x19,0x24,0x18
,0x24,0x25,0x18,0x25,0x26,0x17,0x18,0x26,0x16,0x17,0x26,0x16,0x26,0x27,0x15
,0x16,0x27,0x15,0x27,0x28,0x14,0x15,0x28,0x14,0x28,0x29,0x13,0x14,0x29,0x13
,0x29,0x2A,0x12,0x13,0x2A,0x12,0x2A,0x2B,0x11,0x12,0x2B,0x10,0x11,0x2B,0x10
,0x2B,0x2C,0x0F,0x10,0x2C,0x0F,0x2C,0x2D,0x0E,0x0F,0x2D,0x0E,0x2D,0x2E,0x0D
,0x0E,0x2E,0x0C,0x0D,0x2E,0x0C,0x2E,0x2F,0x0B,0x0C,0x2F,0x30,0x0B,0x2F,0x3A
,0x74,0x22,0x20,0x01,0x6F,0x12,0x03,0x3B,0x21,0xB0,0xF2,0x3E,0x7B,0xD9,0xF6
,0xBC,0xB0,0x1D,0xCC,0x3E,0xD0,0x97,0x5E,0xBD,0xB8,0xCA,0xB3,0x3E,0x5A,0x2F
,0xC6,0xBD,0x4A,0x79,0x95,0x3E,0x26,0x35,0x14,0xBE,0x19,0x58,0x77,0x3E,0x96
,0x98,0x47,0xBE,0x7A,0x17,0x4F,0x3E,0x23,0xDB,0x79,0xBE,0x02,0x83,0x34,0x3E
,0x23,0xDB,0x79,0xBE,0x50,0x8D,0x17,0x3E,0x68,0x91,0xED,0xBD,0x50,0x8D,0x17
,0x3E,0x68,0x91,0xED,0xBD,0x08,0x58,0xCB,0xBE,0x17,0x0E,0xE4,0xBD,0xC8,0x5F
,0xF2,0xBE,0x2F,0x85,0xC7,0xBD,0x33,0xDE,0x06,0xBF,0x6F,0x0E,0x97,0xBD,0x91
,0x62,0x10,0xBF,0x40,0x85,0x23,0xBD,0x7B,0x68,0x17,0xBF,0xDB,0x89,0x12,0xBA
,0x6A,0xBD,0x1B,0xBF,0xD6,0x8E,0x22,0x3D,0x1B,0x2F,0x1D,0xBF,0xE9,0x9D,0xCA
,0x3D,0xCC,0x62,0x1A,0xBF,0x17,0x2D,0x20,0x3E,0xF0,0xFD,0x11,0xBF,0x75,0x94
,0x53,0x3E,0xE0,0x0E,0x04,0xBF,0x23,0xDB,0x79,0x3E,0xAE,0x47,0xE1,0xBE,0x3D
,0x0A,0x57,0x3E,0xAE,0x47,0xE1,0xBE,0x8F,0x6E,0x44,0x3E,0xAA,0x44,0xF1,0xBE
,0x9B,0x74,0x2B,0x3E,0xDD,0xD0,0xFC,0xBE,0x17,0xD9,0x0E,0x3E,0xEA,0xE7,0x01
,0xBF,0xDE,0xAF,0xE2,0x3D,0x6F,0x12,0x03,0xBF,0x01,0xA2,0xA0,0x3D,0xE7,0x51
,0x01,0xBF,0x89,0x7C,0x57,0x3D,0xC5,0x20,0xF8,0xBE,0xD5,0x07,0x12,0x3D,0x61
,0x8D,0xE3,0xBE,0x8F,0xC2,0xF5,0x3C,0x23,0xDB,0xC1,0xBE,0x8F,0xC2,0xF5,0x3C
,0x06,0x81,0x15,0x3E,0xE0,0xF7,0x5F,0x3E,0x50,0x8D,0x17,0x3E,0xE0,0xF7,0x5F
,0x3E,0xF4,0xFD,0x54,0x3E,0x8F,0xC2,0xF5,0x3C,0xF4,0xFD,0x54,0x3E,0x8F,0xC2
,0xF5,0x3C,0x21,0xB0,0xF2,0x3E,0x12,0x13,0x14,0x12,0x14,0x15,0x12,0x15,0x16
,0x11,0x12,0x16,0x11,0x16,0x17,0x20,0x21,0x00,0x20,0x00,0x01,0x20,0x01,0x02
,0x20,0x02,0x03,0x1E,0x1F,0x20,0x1D,0x1E,0x20,0x06,0x07,0x08,0x05,0x06,0x08
,0x04,0x05,0x08,0x11,0x17,0x18,0x10,0x11,0x18,0x10,0x18,0x19,0x0F,0x10,0x19
,0x20,0x03,0x04,0x20,0x04,0x08,0x1D,0x20,0x08,0x1D,0x08,0x09,0x1C,0x1D,0x09
,0x1B,0x1C,0x09,0x1B,0x09,0x0A,0x1B,0x0A,0x0B,0x1A,0x1B,0x0B,0x1A,0x0B,0x0C
,0x1A,0x0C,0x0D,0x1A,0x0D,0x0E,0x1A,0x0E,0x0F,0x1A,0x0F,0x19,0x21,0x75,0x30
,0x2E,0x01,0x50,0x8D,0x17,0x3D,0xF4,0xFD,0x54,0x3E,0x50,0x8D,0x17,0x3D,0x58
,0x39,0x34,0x3E,0x91,0xEF,0xD2,0x3D,0xEA,0x78,0x2C,0x3E,0x3C,0x33,0x11,0x3E
,0xA2,0xED,0x18,0x3E,0x47,0x8E,0x24,0x3E,0xA2,0x99,0xE7,0x3D,0x0C,0x02,0x2B
,0x3E,0x1B,0x2F,0x5D,0x3D,0x0C,0x02,0x2B,0x3E,0x14,0x5A,0xCE,0xBE,0x7D,0xE7
,0xD7,0x3D,0x50,0x38,0xEB,0xBE,0x75,0x93,0x58,0x3D,0xF4,0xFD,0xFC,0xBE,0xFC
,0xA9,0xF1,0x3B,0x77,0x14,0x03,0xBF,0x05,0x6C,0x07,0xBD,0xA6,0x9B,0x04,0xBF
,0x24,0x0A,0xAD,0xBD,0x0C,0x58,0x02,0xBF,0x06,0xD9,0x02,0xBE,0xA0,0x1A,0xF7
,0xBE,0xBA,0xF7,0x20,0xBE,0x35,0x08,0xDB,0xBE,0x0C,0x02,0x2B,0xBE,0x4E,0x0A
,0xAB,0xBE,0x0C,0x02,0x2B,0xBE,0xF4,0xFD,0x54,0x3E,0x54,0xE3,0xE5,0xBE,0xF4
,0xFD,0x54,0x3E,0x54,0xE3,0xE5,0xBE,0x58,0x39,0x34,0x3E,0x33,0x89,0xCA,0xBE
,0x3F,0xE0,0x31,0x3E,0x2B,0x2F,0xB9,0xBE,0xF4,0xF9,0x28,0x3E,0x08,0x59,0xAE
,0xBE,0x69,0x37,0x1A,0x3E,0x3B,0x8B,0xA6,0xBE,0xD4,0x49,0x06,0x3E,0xED,0xD3
,0xA1,0xBE,0x1E,0x8B,0xCD,0x3D,0x89,0x41,0xA0,0xBE,0xF4,0xFD,0x54,0x3D,0x89
,0x41,0xA0,0xBE,0x1F,0x2D,0x9E,0xBE,0xA0,0xC6,0x9D,0xBE,0xBC,0x76,0xD1,0xBE
,0x04,0x56,0x96,0xBE,0xF4,0xA5,0xF7,0xBE,0x9C,0x86,0x88,0xBE,0xF4,0x32,0x0A
,0xBF,0x22,0xDF,0x65,0xBE,0x21,0xB1,0x15,0xBF,0xBE,0x4A,0x2E,0xBE,0x2A,0x3B
,0x1D,0xBF,0x88,0xF1,0xDA,0xBD,0x77,0xBE,0x1F,0xBF,0xB7,0x43,0x43,0xBD,0x2D
,0xE8,0x1D,0xBF,0x90,0xC1,0x0A,0x3C,0x62,0x65,0x18,0xBF,0x44,0x6A,0x9A,0x3D
,0xCE,0xC1,0x0B,0xBF,0x0C,0x02,0x2B,0x3E,0x77,0x12,0xE9,0xBE,0x0C,0x02,0x2B
,0x3E,0x77,0xBE,0x1F,0xBF,0x66,0xBE,0x53,0x3E,0x77,0xBE,0x1F,0xBF,0x54,0xE3
,0xE5,0x3E,0xF0,0xA7,0x06,0xBF,0x50,0x8D,0xDF,0x3E,0x91,0xED,0xFC,0xBE,0xED
,0xD2,0xCE,0x3E,0x8F,0x56,0x01,0xBF,0x7D,0xE7,0xBF,0x3E,0xCB,0x4B,0x02,0xBF
,0x7D,0x93,0xB6,0x3E,0x88,0x82,0x01,0xBF,0x0B,0x7F,0xAE,0x3E,0xE4,0x4D,0xFE
,0xBE,0x49,0x0E,0xA8,0x3E,0x45,0xA1,0xF5,0xBE,0xE3,0xA5,0xA3,0x3E,0x0E,0x2F
,0xE8,0xBE,0x97,0x1A,0xA1,0x3E,0xE6,0x92,0xCA,0xBE,0x89,0x41,0xA0,0x3E,0x73
,0x68,0x91,0xBE,0x89,0x41,0xA0,0x3E,0xF4,0xFD,0x54,0x3E,0x22,0x23,0x24,0x0F
,0x10,0x11,0x0F,0x11,0x12,0x0F,0x12,0x13,0x0F,0x13,0x14,0x0F,0x14,0x15,0x0E
,0x0F,0x15,0x0E,0x15,0x16,0x0E,0x16,0x17,0x0E,0x17,0x18,0x0E,0x18,0x19,0x0D
,0x0E,0x19,0x0D,0x19,0x1A,0x0D,0x1A,0x1B,0x0C,0x0D,0x1B,0x0C,0x1B,0x1C,0x0C
,0x1C,0x1D,0x0B,0x0C,0x1D,0x0B,0x1D,0x1E,0x0B,0x1E,0x1F,0x0A,0x0B,0x1F,0x0A
,0x1F,0x20,0x09,0x0A,0x20,0x09,0x20,0x21,0x08,0x09,0x21,0x08,0x21,0x22,0x07
,0x08,0x22,0x06,0x07,0x22,0x2F,0x00,0x01,0x2F,0x01,0x02,0x2F,0x02,0x03,0x2F
,0x03,0x04,0x2E,0x2F,0x04,0x2E,0x04,0x05,0x2E,0x05,0x06,0x2D,0x2E,0x06,0x24
,0x25,0x26,0x24,0x26,0x27,0x24,0x27,0x28,0x24,0x28,0x29,0x22,0x24,0x29,0x22
,0x29,0x2A,0x22,0x2A,0x2B,0x22,0x2B,0x2C,0x22,0x2C,0x2D,0x22,0x2D,0x06,0x2F
,0x76,0x24,0x22,0x01,0x7F,0xBE,0xDD,0xBE,0x58,0x39,0x34,0x3E,0x4B,0xE4,0xCA
,0xBE,0x62,0xBE,0x2C,0x3E,0xEE,0x7C,0xBF,0xBE,0x0F,0xD5,0x24,0x3E,0x06,0xBE
,0xB2,0xBE,0xB2,0x9C,0x14,0x3E,0xD9,0x78,0xA8,0xBE,0xF3,0x01,0x01,0x3E,0xCE
,0x36,0x9F,0xBE,0x6F,0x9B,0xC9,0x3D,0x06,0x81,0x95,0xBE,0x8E,0x22,0x6B,0x3D
,0x6C,0xE9,0xD1,0xBB,0x77,0xBE,0x1F,0xBF,0x27,0xBC,0x04,0x3D,0x77,0xBE,0x1F
,0xBF,0xAC,0xC8,0xA0,0x3E,0x96,0x43,0x8B,0x3D,0x58,0x37,0xAE,0x3E,0xA2,0xED
,0xF8,0x3D,0x6D,0x3B,0xBD,0x3E,0x25,0x06,0x21,0x3E,0x33,0xDF,0xC9,0x3E,0x89
,0x99,0x2D,0x3E,0x7F,0xBE,0xDD,0x3E,0x58,0x39,0x34,0x3E,0x7F,0xBE,0xDD,0x3E
,0xF4,0xFD,0x54,0x3E,0x52,0xB8,0x2E,0x3E,0xF4,0xFD,0x54,0x3E,0x52,0xB8,0x2E
,0x3E,0x58,0x39,0x34,0x3E,0xA8,0xC4,0x55,0x3E,0xE3,0xFB,0x32,0x3E,0x36,0x06
,0x6D,0x3E,0x85,0x43,0x2F,0x3E,0x14,0x08,0x7B,0x3E,0x07,0x29,0x28,0x3E,0x13
,0x2B,0x83,0x3E,0xAC,0xC4,0x1C,0x3E,0x37,0xDE,0x85,0x3E,0x27,0x83,0x13,0x3E
,0xA4,0xC4,0x86,0x3E,0x22,0x00,0x08,0x3E,0xFA,0x2B,0x84,0x3E,0x38,0x68,0xCF
,0x3D,0x7A,0xC4,0x78,0x3E,0xD5,0xED,0x6C,0x3D,0x32,0x1E,0x65,0x3D,0x8B,0x6C
,0xCF,0xBE,0x61,0x35,0x06,0xBE,0x2F,0x52,0x28,0x3D,0xB4,0x20,0x14,0xBE,0xD3
,0xF5,0xA4,0x3D,0xBD,0xC4,0x18,0xBE,0x28,0x0C,0xEA,0x3D,0x8A,0x3D,0x14,0xBE
,0xF4,0xFB,0x0E,0x3E,0xF0,0xA7,0x06,0xBE,0x6F,0x12,0x23,0x3E,0x44,0x87,0xE0
,0xBD,0xBF,0xEF,0x2F,0x3E,0x9D,0x9F,0xA2,0xBD,0x58,0x39,0x34,0x3E,0x08,0x21
,0x60,0xBD,0x58,0x39,0x34,0x3E,0x08,0x21,0x60,0xBD,0xF4,0xFD,0x54,0x3E,0x7F
,0xBE,0xDD,0xBE,0xF4,0xFD,0x54,0x3E,0x22,0x23,0x00,0x22,0x00,0x01,0x22,0x01
,0x02,0x22,0x02,0x03,0x22,0x03,0x04,0x22,0x04,0x05,0x22,0x05,0x06,0x20,0x21
,0x22,0x0E,0x0F,0x10,0x0E,0x10,0x11,0x0E,0x11,0x12,0x0E,0x12,0x13,0x0E,0x13
,0x14,0x0D,0x0E,0x14,0x0C,0x0D,0x14,0x0B,0x0C,0x14,0x0A,0x0B,0x14,0x0A,0x14
,0x15,0x0A,0x15,0x16,0x09,0x0A,0x16,0x09,0x16,0x17,0x09,0x17,0x18,0x08,0x09
,0x18,0x08,0x18,0x19,0x07,0x08,0x19,0x06,0x07,0x19,0x06,0x19,0x1A,0x06,0x1A
,0x1B,0x06,0x1B,0x1C,0x06,0x1C,0x1D,0x1F,0x20,0x22,0x06,0x1D,0x1E,0x22,0x06
,0x1E,0x1F,0x22,0x1E,0x23,0x77,0x30,0x2E,0x01,0x81,0x40,0x23,0xBF,0x58,0x39
,0x34,0x3E,0x5A,0x83,0x17,0xBF,0xC3,0x9B,0x25,0x3E,0x52,0x62,0x0F,0xBF,0xCE
,0x4F,0x11,0x3E,0x4B,0x01,0x09,0xBF,0xF6,0xD0,0xDE,0x3D,0x16,0x84,0x02,0xBF
,0xF8,0xDE,0x5F,0x3D,0xE9,0xD2,0x7F,0xBE,0x77,0xBE,0x1F,0xBF,0xBB,0x45,0x60
,0xBE,0x77,0xBE,0x1F,0xBF,0xBE,0x15,0x89,0x3C,0xC9,0x76,0xDE,0xBD,0x7B,0x68
,0x5F,0x3E,0x77,0xBE,0x1F,0xBF,0xFC,0x53,0x82,0x3E,0x77,0xBE,0x1F,0xBF,0x21
,0x5B,0x06,0x3F,0xFC,0xA9,0x91,0x3D,0x3F,0x35,0x12,0x3F,0xBC,0x74,0x13,0x3E
,0x81,0x40,0x23,0x3F,0x58,0x39,0x34,0x3E,0x81,0x40,0x23,0x3F,0xF4,0xFD,0x54
,0x3E,0x0F,0x2B,0xC4,0x3E,0xF4,0xFD,0x54,0x3E,0x0F,0x2B,0xC4,0x3E,0x58,0x39
,0x34,0x3E,0x1E,0xC5,0xD9,0x3E,0xCD,0x74,0x2F,0x3E,0x27,0xDD,0xE6,0x3E,0xAB
,0xCC,0x24,0x3E,0x89,0x7E,0xED,0x3E,0xEC,0x4B,0x16,0x3E,0x3E,0xB4,0xEF,0x3E
,0xF8,0xFB,0x05,0x3E,0x6B,0x66,0xED,0x3E,0x9E,0xEF,0xC7,0x3D,0xF2,0x7C,0xE6
,0x3E,0xF8,0xDE,0x5F,0x3D,0xB9,0x1C,0x8F,0x3E,0x3E,0x5E,0xC8,0xBE,0xD3,0xF5
,0xC4,0x3D,0x0E,0x2D,0x92,0x3D,0xAC,0x1C,0xBA,0x3D,0xBC,0xE9,0xB6,0x3D,0xC8
,0x7E,0xB6,0x3D,0xC6,0xC0,0xDA,0x3D,0x68,0xB2,0xBF,0x3D,0x1A,0x33,0x09,0x3E
,0xB6,0x4B,0xDB,0x3D,0xBA,0x49,0x1C,0x3E,0xA2,0xEB,0x12,0x3E,0x51,0xDD,0x2C
,0x3E,0xED,0x80,0x4B,0x3E,0x58,0x39,0x34,0x3E,0xED,0x80,0x4B,0x3E,0xF4,0xFD
,0x54,0x3E,0x37,0x35,0x40,0xBE,0xF4,0xFD,0x54,0x3E,0x37,0x35,0x40,0xBE,0x58
,0x39,0x34,0x3E,0xAE,0x9B,0x12,0xBE,0xE3,0xFF,0x2E,0x3E,0x1E,0x52,0xEC,0xBD
,0x8B,0xC4,0x24,0x3E,0xE8,0xA3,0xAC,0xBD,0xC5,0x76,0x07,0x3E,0x34,0x0E,0x75
,0xBD,0x21,0xB0,0xB2,0x3D,0x3C,0xF4,0x5D,0xBC,0xD8,0xB8,0xFE,0xBC,0x0D,0xFE
,0x3E,0xBE,0xD5,0x78,0xD1,0xBE,0x70,0x3F,0xB8,0xBE,0xF8,0xDE,0x5F,0x3D,0x00
,0x02,0xBE,0xBE,0x23,0xDB,0xB9,0x3D,0x85,0xED,0xBF,0xBE,0x22,0x00,0xF8,0x3D
,0xF1,0x10,0xBE,0xBE,0xA4,0x6F,0x12,0x3E,0xF2,0x7A,0xB8,0xBE,0xFC,0xFD,0x22
,0x3E,0xFB,0xE8,0xAC,0xBE,0x8F,0x1B,0x2E,0x3E,0xA0,0x18,0x99,0xBE,0x58,0x39
,0x34,0x3E,0xA0,0x18,0x99,0xBE,0xF4,0xFD,0x54,0x3E,0x81,0x40,0x23,0xBF,0xF4
,0xFD,0x54,0x3E,0x2E,0x2F,0x00,0x2E,0x00,0x01,0x2E,0x01,0x02,0x2E,0x02,0x03
,0x2E,0x03,0x04,0x1E,0x1F,0x20,0x1E,0x20,0x21,0x1E,0x21,0x22,0x1E,0x22,0x23
,0x1E,0x23,0x24,0x0D,0x0E,0x0F,0x0D,0x0F,0x10,0x0D,0x10,0x11,0x0D,0x11,0x12
,0x0C,0x0D,0x12,0x0B,0x0C,0x12,0x0A,0x0B,0x12,0x0A,0x12,0x13,0x0A,0x13,0x14
,0x0A,0x14,0x15,0x09,0x0A,0x15,0x09,0x15,0x16,0x08,0x09,0x16,0x07,0x08,0x16
,0x07,0x16,0x17,0x1C,0x1D,0x1E,0x1C,0x1E,0x24,0x1B,0x1C,0x24,0x1A,0x1B,0x24
,0x19,0x1A,0x24,0x19,0x24,0x25,0x18,0x19,0x25,0x17,0x18,0x25,0x07,0x17,0x25
,0x07,0x25,0x26,0x06,0x07,0x26,0x05,0x06,0x26,0x04,0x05,0x26,0x04,0x26,0x27
,0x04,0x27,0x28,0x04,0x28,0x29,0x2C,0x2D,0x2E,0x04,0x29,0x2A,0x04,0x2A,0x2B
,0x2E,0x04,0x2B,0x2C,0x2E,0x2B,0x2F,0x78,0x3D,0x3B,0x01,0xBE,0x9F,0xDA,0xBE
,0x58,0x39,0x34,0x3E,0x52,0xB9,0xC1,0xBE,0xF0,0xFD,0x2D,0x3E,0x52,0x64,0xAD
,0xBE,0xB6,0x4B,0x1B,0x3E,0xA6,0x9C,0x97,0xBE,0xBB,0x45,0xE0,0x3D,0x6A,0xBC
,0x74,0xBE,0x4A,0x0C,0x02,0x3D,0x98,0x6E,0xB2,0xBD,0xAA,0xF1,0x42,0xBE,0x5C
,0x8F,0x82,0xBE,0xB4,0x74,0xD5,0xBE,0xF6,0xD2,0xA4,0xBE,0x0F,0x97,0x00,0xBF
,0x1F,0x85,0xBB,0xBE,0x89,0x41,0x0C,0xBF,0x3B,0x89,0xC8,0xBE,0x9E,0xEF,0x0F
,0xBF,0x50,0x8D,0xD7,0xBE,0x85,0xEB,0x11,0xBF,0x50,0x8D,0xD7,0xBE,0x9A,0x99
,0x19,0xBF,0xE5,0xD0,0x22,0xBE,0x9A,0x99,0x19,0xBF,0xE5,0xD0,0x22,0xBE,0x85
,0xEB,0x11,0xBF,0xF0,0xA7,0x46,0xBE,0x8A,0x77,0x10,0xBF,0x89,0x41,0x60,0xBE
,0x18,0x05,0x0D,0xBF,0xB2,0x9D,0x6F,0xBE,0xA8,0x51,0x08,0xBF,0x6A,0xBC,0x74
,0xBE,0xA0,0x1A,0x03,0xBF,0x56,0xB8,0x65,0xBE,0x19,0x04,0xF6,0xBE,0x19,0xAC
,0x38,0xBE,0xD3,0xF5,0xD4,0xBE,0xE2,0x05,0x51,0xBD,0x59,0x89,0x79,0xBE,0xF2
,0x5D,0x8A,0x3D,0x8D,0x43,0xD5,0xBE,0xC0,0x97,0xE2,0x3D,0x5C,0x8E,0xF7,0xBE
,0x00,0x00,0x00,0x3E,0xE4,0xF9,0x04,0xBF,0xF8,0x4F,0xF7,0x3D,0x25,0xB0,0x09
,0xBF,0x5B,0x3F,0xDD,0x3D,0x42,0x60,0x0D,0xBF,0x98,0x6A,0xA6,0x3D,0xBD,0xC8
,0x10,0xBF,0x45,0xA0,0x3A,0x3D,0x85,0xEB,0x11,0xBF,0x45,0xA0,0x3A,0x3D,0x9A
,0x99,0x19,0xBF,0xBE,0x9F,0xDA,0x3E,0x9A,0x99,0x19,0xBF,0xBE,0x9F,0xDA,0x3E
,0x85,0xEB,0x11,0xBF,0xE6,0x3C,0xC3,0x3E,0xA3,0x05,0x10,0xBF,0x9E,0x99,0xB0
,0x3E,0x1D,0xB0,0x0B,0xBF,0x03,0xE9,0x9A,0x3E,0x08,0xAC,0x00,0xBF,0x6A,0xBC
,0x74,0x3E,0xB4,0x74,0xD5,0xBE,0xD1,0x22,0x1B,0x3D,0xAA,0x49,0x00,0xBE,0x95
,0x9B,0x18,0x3E,0x3B,0xE0,0xBA,0x3C,0x01,0xA6,0x5C,0x3E,0x86,0x3B,0xD7,0x3D
,0xD7,0xA3,0x88,0x3E,0x31,0x08,0x1C,0x3E,0xCF,0xA3,0x9A,0x3E,0xED,0xD4,0x2C
,0x3E,0xB2,0x9D,0xAF,0x3E,0x58,0x39,0x34,0x3E,0xB2,0x9D,0xAF,0x3E,0xF4,0xFD
,0x54,0x3E,0x8D,0x0C,0x92,0x3D,0xF4,0xFD,0x54,0x3E,0x8D,0x0C,0x92,0x3D,0x58
,0x39,0x34,0x3E,0x1F,0x85,0xCB,0x3D,0x27,0xDB,0x30,0x3E,0x19,0x8F,0xF2,0x3D
,0x94,0xC0,0x26,0x3E,0xB0,0x76,0x04,0x3E,0xA0,0xC4,0x17,0x3E,0x27,0x31,0x08
,0x3E,0x8F,0xC2,0x05,0x3E,0xE3,0x6E,0xF0,0x3D,0x3F,0x8D,0xBB,0x3D,0xE8,0x4B
,0x8F,0x3D,0xE9,0x26,0xB1,0x3C,0x67,0x62,0x3A,0x3B,0x99,0x49,0x94,0xBD,0x89
,0x41,0x60,0xBD,0x7A,0xFF,0x9F,0x3C,0x9E,0x3F,0xAD,0xBD,0xDE,0xAF,0x82,0x3D
,0xEA,0x05,0xDF,0xBD,0x12,0xA4,0xD2,0x3D,0xB2,0x9D,0xEF,0xBD,0x21,0x58,0x05
,0x3E,0x51,0x14,0xE8,0xBD,0xA9,0x6C,0x18,0x3E,0xB3,0x78,0xD1,0xBD,0x07,0x5C
,0x27,0x3E,0x18,0x25,0xA8,0xBD,0xF3,0x01,0x31,0x3E,0x60,0xE5,0x50,0xBD,0x58
,0x39,0x34,0x3E,0x60,0xE5,0x50,0xBD,0xF4,0xFD,0x54,0x3E,0xBE,0x9F,0xDA,0xBE
,0xF4,0xFD,0x54,0x3E,0x3B,0x3C,0x00,0x3B,0x00,0x01,0x3B,0x01,0x02,0x3B,0x02
,0x03,0x29,0x2A,0x2B,0x29,0x2B,0x2C,0x29,0x2C,0x2D,0x29,0x2D,0x2E,0x28,0x29
,0x2E,0x27,0x28,0x2E,0x26,0x27,0x2E,0x25,0x26,0x2E,0x25,0x2E,0x2F,0x24,0x25
,0x2F,0x24,0x2F,0x30,0x24,0x30,0x31,0x23,0x24,0x31,0x23,0x31,0x32,0x1C,0x1D
,0x1E,0x1C,0x1E,0x1F,0x1C,0x1F,0x20,0x1C,0x20,0x21,0x0B,0x0C,0x0D,0x0B,0x0D
,0x0E,0x0B,0x0E,0x0F,0x0B,0x0F,0x10,0x0A,0x0B,0x10,0x09,0x0A,0x10,0x08,0x09
,0x10,0x07,0x08,0x10,0x07,0x10,0x11,0x06,0x07,0x11,0x06,0x11,0x12,0x06,0x12
,0x13,0x05,0x06,0x13,0x05,0x13,0x14,0x39,0x3A,0x3B,0x39,0x3B,0x03,0x38,0x39
,0x03,0x37,0x38,0x03,0x37,0x03,0x04,0x36,0x37,0x04,0x35,0x36,0x04,0x35,0x04
,0x05,0x34,0x35,0x05,0x33,0x34,0x05,0x32,0x33,0x05,0x23,0x32,0x05,0x23,0x05
,0x14,0x22,0x23,0x14,0x22,0x14,0x15,0x22,0x15,0x16,0x22,0x16,0x17,0x21,0x22
,0x17,0x1A,0x1B,0x1C,0x1A,0x1C,0x21,0x19,0x1A,0x21,0x18,0x19,0x21,0x18,0x21
,0x17,0x3C,0x79,0x3A,0x38,0x01,0x45,0xB8,0xE1,0xBE,0x58,0x39,0x34,0x3E,0x34
,0xDB,0xCD,0xBE,0x25,0x5C,0x28,0x3E,0xD3,0x4F,0xC0,0xBE,0xDF,0xA7,0x1A,0x3E
,0x6F,0x11,0xB8,0xBE,0xC1,0x74,0x0A,0x3E,0xC9,0xCC,0xAD,0xBE,0xEE,0x7C,0xDF
,0x3D,0x96,0xEC,0xA0,0xBE,0x25,0x5A,0x92,0x3D,0xAC,0x1E,0x98,0xBE,0xD2,0xC2
,0x25,0x3D,0x76,0xC1,0xE0,0xBB,0x37,0x89,0x11,0xBF,0xD4,0x80,0x81,0xBD,0x92
,0x97,0x35,0xBF,0xA3,0x1E,0xC2,0xBD,0xD3,0x17,0x46,0xBF,0xE9,0x26,0x01,0xBE
,0x0A,0x2C,0x50,0xBF,0x4F,0xE6,0x1F,0xBE,0xB1,0x52,0x55,0xBF,0x28,0x2D,0x3C
,0xBE,0x3D,0x0A,0x57,0xBF,0x8D,0xEF,0x4B,0xBE,0x13,0x43,0x56,0xBF,0x7E,0x3B
,0x69,0xBE,0x96,0xED,0x53,0xBF,0xAE,0x9E,0x8B,0xBE,0x84,0x80,0x50,0xBF,0x44
,0x35,0x9D,0xBE,0x29,0x5C,0x4F,0xBF,0x54,0x8B,0xB0,0xBE,0x08,0xCB,0x50,0xBF
,0x8D,0x41,0xBF,0xBE,0x94,0x17,0x55,0xBF,0x58,0x8E,0xC8,0xBE,0x3C,0xC0,0x5B
,0xBF,0xFC,0xA7,0xCB,0xBE,0x91,0x43,0x64,0xBF,0xB3,0x5F,0xC7,0xBE,0x4E,0x97
,0x6D,0xBF,0x3C,0x87,0xBA,0xBE,0xCD,0xCB,0x75,0xBF,0x9E,0x07,0xA7,0xBE,0x6A
,0x87,0x7B,0xBF,0xCD,0xCA,0x8E,0xBE,0xA4,0x70,0x7D,0xBF,0xAD,0x17,0x53,0xBE
,0x77,0x9D,0x79,0xBF,0xEB,0xA9,0x05,0xBE,0xD1,0x23,0x6E,0xBF,0x7B,0x4B,0x79
,0xBD,0x55,0x2F,0x5B,0xBF,0x62,0x11,0xC3,0xBB,0x89,0xEB,0x40,0xBF,0x94,0xC2
,0xA4,0x3E,0xE3,0x1A,0x5F,0x3D,0x79,0x3F,0xAE,0x3E,0x4D,0x12,0xCB,0x3D,0xBD
,0x1E,0xB4,0x3E,0x76,0x6E,0xFA,0x3D,0x7C,0x80,0xBE,0x3E,0x81,0x3D,0x16,0x3E
,0xF0,0xFB,0xC7,0x3E,0xB0,0xCA,0x25,0x3E,0x0B,0xEF,0xD2,0x3E,0xDD,0xCE,0x2E
,0x3E,0x45,0xB8,0xE1,0x3E,0x58,0x39,0x34,0x3E,0x45,0xB8,0xE1,0x3E,0xF4,0xFD
,0x54,0x3E,0x35,0x62,0x36,0x3E,0xF4,0xFD,0x54,0x3E,0x35,0x62,0x36,0x3E,0x58
,0x39,0x34,0x3E,0xF3,0x54,0x57,0x3E,0x98,0xBF,0x32,0x3E,0x44,0x8B,0x6C,0x3E
,0xD2,0x51,0x2E,0x3E,0x2F,0xDC,0x79,0x3E,0x5E,0x63,0x27,0x3E,0x60,0x8F,0x81
,0x3E,0x88,0x66,0x1E,0x3E,0x89,0xEC,0x83,0x3E,0xED,0x2A,0x14,0x3E,0x41,0xB6
,0x84,0x3E,0x34,0xDB,0x05,0x3E,0xC9,0xCB,0x82,0x3E,0x1B,0x7F,0xC2,0x3D,0x7A
,0x18,0x7A,0x3E,0x59,0xC4,0x70,0x3D,0x8B,0xC4,0x84,0x3D,0x39,0x60,0xC7,0xBE
,0xAB,0xED,0x06,0xBE,0x04,0x55,0xA3,0x3C,0xA6,0x45,0x1D,0xBE,0x66,0x87,0x98
,0x3D,0x39,0xB8,0x24,0xBE,0x15,0xFE,0xEC,0x3D,0xBA,0xA3,0x1F,0xBE,0x05,0xA7
,0x0E,0x3E,0x80,0x66,0x10,0xBE,0x0A,0x83,0x22,0x3E,0x3A,0x04,0xEE,0xBD,0xD5
,0xCB,0x2F,0x3E,0xDC,0xF1,0xA6,0xBD,0x58,0x39,0x34,0x3E,0xC1,0xFD,0x80,0xBD
,0x58,0x39,0x34,0x3E,0xC1,0xFD,0x80,0xBD,0xF4,0xFD,0x54,0x3E,0x45,0xB8,0xE1
,0xBE,0xF4,0xFD,0x54,0x3E,0x38,0x39,0x00,0x38,0x00,0x01,0x38,0x01,0x02,0x38
,0x02,0x03,0x38,0x03,0x04,0x38,0x04,0x05,0x36,0x37,0x38,0x24,0x25,0x26,0x24
,0x26,0x27,0x24,0x27,0x28,0x24,0x28,0x29,0x24,0x29,0x2A,0x23,0x24,0x2A,0x22
,0x23,0x2A,0x21,0x22,0x2A,0x20,0x21,0x2A,0x1F,0x20,0x2A,0x1F,0x2A,0x2B,0x1F
,0x2B,0x2C,0x1E,0x1F,0x2C,0x1E,0x2C,0x2D,0x1D,0x1E,0x2D,0x1D,0x2D,0x2E,0x1C
,0x1D,0x2E,0x1C,0x2E,0x2F,0x35,0x36,0x38,0x35,0x38,0x05,0x34,0x35,0x05,0x33
,0x34,0x05,0x32,0x33,0x05,0x32,0x05,0x06,0x31,0x32,0x06,0x31,0x06,0x07,0x30
,0x31,0x07,0x2F,0x30,0x07,0x1C,0x2F,0x07,0x1C,0x07,0x08,0x1C,0x08,0x09,0x1B
,0x1C,0x09,0x1B,0x09,0x0A,0x1A,0x1B,0x0A,0x1A,0x0A,0x0B,0x1A,0x0B,0x0C,0x19
,0x1A,0x0C,0x19,0x0C,0x0D,0x18,0x19,0x0D,0x18,0x0D,0x0E,0x18,0x0E,0x0F,0x17
,0x18,0x0F,0x16,0x17,0x0F,0x16,0x0F,0x10,0x15,0x16,0x10,0x15,0x10,0x11,0x14
,0x15,0x11,0x13,0x14,0x11,0x13,0x11,0x12,0x39,0x7A,0x18,0x16,0x01,0x2F,0xDD
,0xA4,0x3E,0x58,0x39,0xB4,0xBE,0x88,0x69,0x9F,0x3E,0x60,0x92,0xDA,0xBE,0x0E
,0xD9,0x98,0x3E,0xB2,0xF1,0xF0,0xBE,0x7A,0xFE,0x8C,0x3E,0xCA,0xE0,0x00,0xBF
,0xFA,0xD2,0x7B,0x3E,0xD3,0x4D,0x06,0xBF,0x7F,0x67,0x4B,0x3E,0x85,0x41,0x09
,0xBF,0x23,0xDB,0xF9,0x3D,0x71,0x3D,0x0A,0xBF,0x8E,0x3F,0x31,0xBE,0x71,0x3D
,0x0A,0xBF,0x7F,0x6A,0xBC,0x3E,0x02,0x83,0x34,0x3E,0x7F,0x6A,0xBC,0x3E,0xF4
,0xFD,0x54,0x3E,0x52,0xB8,0x9E,0xBE,0xF4,0xFD,0x54,0x3E,0x9C,0xC4,0xA0,0xBE
,0x42,0x60,0x65,0xBC,0x04,0x56,0x8E,0xBE,0x42,0x60,0x65,0xBC,0xEB,0xFE,0x89
,0xBE,0x6D,0xE7,0x3B,0x3D,0x23,0x87,0x80,0xBE,0x67,0xF1,0xC2,0x3D,0x46,0xB4
,0x6D,0xBE,0x7D,0xE7,0xF7,0x3D,0xC5,0x70,0x55,0xBE,0xBA,0x49,0x0C,0x3E,0xC8
,0xCE,0x2B,0xBE,0x6A,0xBC,0x14,0x3E,0x39,0xB4,0xC8,0xBD,0x50,0x8D,0x17,0x3E
,0xE4,0x4D,0x2E,0x3E,0x50,0x8D,0x17,0x3E,0x7F,0x6A,0xBC,0xBE,0x10,0x58,0x11
,0xBF,0x7F,0x6A,0xBC,0xBE,0x9A,0x99,0x19,0xBF,0xC5,0x20,0xB0,0x3E,0x9A,0x99
,0x19,0xBF,0x7D,0x3F,0xB5,0x3E,0x58,0x39,0xB4,0xBE,0x16,0x17,0x00,0x16,0x00
,0x01,0x16,0x01,0x02,0x16,0x02,0x03,0x0A,0x0B,0x0C,0x0A,0x0C,0x0D,0x0A,0x0D
,0x0E,0x0A,0x0E,0x0F,0x16,0x03,0x04,0x14,0x15,0x16,0x08,0x09,0x0A,0x0A,0x0F
,0x10,0x16,0x04,0x05,0x14,0x16,0x05,0x14,0x05,0x06,0x14,0x06,0x07,0x13,0x14
,0x07,0x13,0x07,0x08,0x13,0x08,0x0A,0x12,0x13,0x0A,0x11,0x12,0x0A,0x11,0x0A
,0x10,0x17,0x7B,0x33,0x31,0x01,0xF1,0x2C,0x21,0x3E,0x63,0x45,0x6D,0xBF,0x09
,0xFC,0xC1,0x3D,0x54,0xE4,0x60,0xBF,0x27,0x4E,0x6E,0x3D,0xE8,0xDC,0x51,0xBF
,0x7F,0x6A,0x3C,0x3D,0x94,0x6C,0x41,0xBF,0x60,0xE5,0x50,0x3D,0x82,0xFF,0x31
,0xBF,0x02,0x2B,0x87,0x3D,0x0C,0x58,0x1E,0xBF,0x54,0xE3,0xA5,0x3D,0x16,0x2F
,0x0A,0xBF,0xC5,0x20,0xB0,0x3D,0xEA,0x7A,0xF2,0xBE,0xA7,0xCC,0x8D,0x3D,0xB8
,0x5C,0xC5,0xBE,0x52,0x43,0x9B,0x3C,0xD1,0x24,0x99,0xBE,0x26,0xA8,0x81,0xBD
,0x0C,0xAF,0x64,0xBE,0x58,0x39,0x34,0xBE,0xCC,0xF1,0x2A,0xBE,0x9D,0x82,0x7C
,0xBD,0x51,0x33,0xE4,0xBD,0x57,0x24,0xA6,0x3C,0x04,0x1B,0x17,0xBD,0xB7,0x7A
,0x8E,0x3D,0x5A,0x9B,0x46,0x3D,0xC5,0x20,0xB0,0x3D,0xC8,0xCE,0x0B,0x3E,0x54
,0xE3,0xA5,0x3D,0x23,0xDD,0x4F,0x3E,0x02,0x2B,0x87,0x3D,0x90,0x6A,0x90,0x3E
,0x60,0xE5,0x50,0x3D,0x54,0xE1,0xB7,0x3E,0x7F,0x6A,0x3C,0x3D,0x17,0xD9,0xD6
,0x3E,0x27,0x4E,0x6E,0x3D,0x9C,0xDC,0xF7,0x3E,0x09,0xFC,0xC1,0x3D,0x21,0x05
,0x0B,0x3F,0xF1,0x2C,0x21,0x3E,0x17,0x64,0x17,0x3F,0xB6,0xF3,0x7D,0x3E,0x77
,0xBE,0x1F,0x3F,0xB6,0xF3,0x7D,0x3E,0x8B,0x6C,0x27,0x3F,0x2E,0xE3,0xE6,0x3D
,0xB6,0x49,0x1D,0x3F,0xFC,0xA6,0xB0,0x3B,0x1F,0xD9,0x08,0x3F,0x45,0x68,0x84
,0xBD,0x6D,0xE7,0xDB,0x3E,0x58,0x39,0xB4,0xBD,0x60,0xE5,0xA0,0x3E,0xD5,0x78
,0xA9,0xBD,0x42,0xB0,0x7A,0x3E,0x4C,0x37,0x89,0xBD,0x9A,0x99,0x29,0x3E,0x85
,0xEB,0x51,0xBD,0x49,0x14,0xBA,0x3D,0x7F,0x6A,0x3C,0xBD,0x01,0x15,0x0E,0x3D
,0x51,0x4B,0x73,0xBD,0x0F,0x0F,0xE1,0xBC,0xF0,0xF7,0xCB,0xBD,0xCC,0x99,0xAD
,0xBD,0x72,0x6A,0x27,0xBE,0xA0,0x19,0x04,0xBE,0xB6,0xF3,0x7D,0xBE,0xE3,0xA5
,0x1B,0xBE,0xB6,0xF3,0x7D,0xBE,0xC9,0x76,0x3E,0xBE,0x72,0x6A,0x27,0xBE,0x39
,0x0C,0x56,0xBE,0xF0,0xF7,0xCB,0xBD,0xA0,0xC2,0x81,0xBE,0x51,0x4B,0x73,0xBD
,0x3D,0x0A,0x9F,0xBE,0x7F,0x6A,0x3C,0xBD,0xC9,0x76,0xBE,0xBE,0x85,0xEB,0x51
,0xBD,0x1A,0x31,0xDB,0xBE,0x4C,0x37,0x89,0xBD,0x7B,0xC0,0x00,0xBF,0xD5,0x78
,0xA9,0xBD,0x59,0xFA,0x14,0xBF,0x58,0x39,0xB4,0xBD,0x54,0x8D,0x26,0xBF,0x45
,0x68,0x84,0xBD,0xCE,0x16,0x44,0xBF,0xFC,0xA6,0xB0,0x3B,0x2B,0xDC,0x5E,0xBF
,0x2E,0xE3,0xE6,0x3D,0xF0,0x32,0x73,0xBF,0xB6,0xF3,0x7D,0x3E,0xA4,0x70,0x7D
,0xBF,0xB6,0xF3,0x7D,0x3E,0x8F,0xC2,0x75,0xBF,0x30,0x31,0x32,0x30,0x32,0x00
,0x30,0x00,0x01,0x2F,0x30,0x01,0x2F,0x01,0x02,0x17,0x18,0x19,0x16,0x17,0x19
,0x15,0x16,0x19,0x15,0x19,0x1A,0x14,0x15,0x1A,0x2E,0x2F,0x02,0x2E,0x02,0x03
,0x14,0x1A,0x1B,0x13,0x14,0x1B,0x2D,0x2E,0x03,0x2D,0x03,0x04,0x2D,0x04,0x05
,0x2C,0x2D,0x05,0x2C,0x05,0x06,0x2B,0x2C,0x06,0x2B,0x06,0x07,0x2A,0x2B,0x07
,0x2A,0x07,0x08,0x29,0x2A,0x08,0x29,0x08,0x09,0x28,0x29,0x09,0x28,0x09,0x0A
,0x27,0x28,0x0A,0x27,0x0A,0x0B,0x26,0x27,0x0B,0x25,0x26,0x0B,0x24,0x25,0x0B
,0x23,0x24,0x0B,0x23,0x0B,0x0C,0x22,0x23,0x0C,0x21,0x22,0x0C,0x21,0x0C,0x0D
,0x20,0x21,0x0D,0x20,0x0D,0x0E,0x1F,0x20,0x0E,0x1F,0x0E,0x0F,0x1E,0x1F,0x0F
,0x1E,0x0F,0x10,0x1D,0x1E,0x10,0x1D,0x10,0x11,0x1C,0x1D,0x11,0x1C,0x11,0x12
,0x1B,0x1C,0x12,0x13,0x1B,0x12,0x32,0x7C,0x04,0x02,0x01,0xBC,0x74,0x13,0xBD
,0x8B,0x6C,0x27,0x3F,0xBC,0x74,0x13,0xBD,0xA4,0x70,0x7D,0xBF,0xBC,0x74,0x13
,0x3D,0xA4,0x70,0x7D,0xBF,0xBC,0x74,0x13,0x3D,0x8B,0x6C,0x27,0x3F,0x02,0x03
,0x00,0x02,0x00,0x01,0x03,0x7D,0x33,0x31,0x01,0xF1,0x2C,0x21,0xBE,0xAA,0x65
,0x17,0x3F,0x09,0xFC,0xC1,0xBD,0x29,0x07,0x0B,0x3F,0x27,0x4E,0x6E,0xBD,0xC5
,0xE2,0xF7,0x3E,0x7F,0x6A,0x3C,0xBD,0x69,0xE5,0xD6,0x3E,0x60,0xE5,0x50,0xBD
,0x91,0xEE,0xB7,0x3E,0x02,0x2B,0x87,0xBD,0xF2,0x7A,0x90,0x3E,0x54,0xE3,0xA5
,0xBD,0x08,0x06,0x50,0x3E,0xC5,0x20,0xB0,0xBD,0x11,0x00,0x0C,0x3E,0x57,0xB0
,0x8D,0xBD,0x92,0xB2,0x45,0x3D,0x0C,0x78,0x99,0xBC,0x9C,0x4F,0x1D,0xBD,0x48
,0x37,0x82,0x3D,0x87,0x6A,0xEA,0xBD,0x58,0x39,0x34,0x3E,0xE0,0x2A,0x2F,0xBE
,0xEC,0xA1,0x7D,0x3D,0xE6,0xCF,0x67,0xBE,0x29,0x5B,0xA4,0xBC,0x9E,0xED,0x99
,0xBE,0x68,0x5E,0x8E,0xBD,0x0B,0x97,0xC5,0xBE,0xC5,0x20,0xB0,0xBD,0xF6,0xD4
,0xF2,0xBE,0x54,0xE3,0xA5,0xBD,0x7B,0x33,0x0A,0xBF,0x02,0x2B,0x87,0xBD,0xE7
,0x50,0x1E,0xBF,0x60,0xE5,0x50,0xBD,0x98,0xFA,0x31,0xBF,0x7F,0x6A,0x3C,0xBD
,0x73,0x68,0x41,0xBF,0x27,0x4E,0x6E,0xBD,0x76,0xFE,0x51,0xBF,0x09,0xFC,0xC1
,0xBD,0xFC,0xFF,0x60,0xBF,0xF1,0x2C,0x21,0xBE,0x44,0x4C,0x6D,0xBF,0xB6,0xF3
,0x7D,0xBE,0x8F,0xC2,0x75,0xBF,0xB6,0xF3,0x7D,0xBE,0xA4,0x70,0x7D,0xBF,0x2E
,0xE3,0xE6,0xBD,0xCB,0x2B,0x73,0xBF,0xFC,0xA6,0xB0,0xBB,0x77,0xBF,0x5E,0xBF
,0x45,0x68,0x84,0x3D,0xF5,0xF2,0x43,0xBF,0x58,0x39,0xB4,0x3D,0x54,0x8D,0x26
,0xBF,0xE7,0xFB,0xA9,0x3D,0x59,0xFA,0x14,0xBF,0x96,0x43,0x8B,0x3D,0x7B,0xC0
,0x00,0xBF,0x87,0x16,0x59,0x3D,0x1A,0x31,0xDB,0xBE,0xA6,0x9B,0x44,0x3D,0xC9
,0x76,0xBE,0xBE,0x0F,0xF2,0x7A,0x3D,0x3D,0x0A,0x9F,0xBE,0x1E,0xFA,0xCE,0x3D
,0xA0,0xC2,0x81,0xBE,0x99,0x45,0x28,0x3E,0x39,0x0C,0x56,0xBE,0xB6,0xF3,0x7D
,0x3E,0xC9,0x76,0x3E,0xBE,0xB6,0xF3,0x7D,0x3E,0xE3,0xA5,0x1B,0xBE,0x99,0x45
,0x28,0x3E,0x0D,0xFD,0x03,0xBE,0x1E,0xFA,0xCE,0x3D,0x29,0xB4,0xAC,0xBD,0x0F
,0xF2,0x7A,0x3D,0xEA,0x93,0xDC,0xBC,0xA6,0x9B,0x44,0x3D,0x01,0x15,0x0E,0x3D
,0x87,0x16,0x59,0x3D,0x49,0x14,0xBA,0x3D,0x96,0x43,0x8B,0x3D,0x9A,0x99,0x29
,0x3E,0xE7,0xFB,0xA9,0x3D,0xD9,0x76,0x7A,0x3E,0x58,0x39,0xB4,0x3D,0xB0,0x72
,0xA0,0x3E,0x45,0x68,0x84,0x3D,0x6A,0x12,0xDC,0x3E,0xFC,0xA6,0xB0,0xBB,0xC3
,0xF5,0x08,0x3F,0x2E,0xE3,0xE6,0xBD,0xEC,0x50,0x1D,0x3F,0xB6,0xF3,0x7D,0xBE
,0x8B,0x6C,0x27,0x3F,0xB6,0xF3,0x7D,0xBE,0x77,0xBE,0x1F,0x3F,0x17,0x18,0x19
,0x16,0x17,0x19,0x15,0x16,0x19,0x15,0x19,0x1A,0x14,0x15,0x1A,0x30,0x31,0x32
,0x30,0x32,0x00,0x30,0x00,0x01,0x2F,0x30,0x01,0x2F,0x01,0x02,0x2E,0x2F,0x02
,0x2E,0x02,0x03,0x14,0x1A,0x1B,0x13,0x14,0x1B,0x2D,0x2E,0x03,0x2D,0x03,0x04
,0x2D,0x04,0x05,0x2C,0x2D,0x05,0x2C,0x05,0x06,0x2B,0x2C,0x06,0x2B,0x06,0x07
,0x2A,0x2B,0x07,0x2A,0x07,0x08,0x29,0x2A,0x08,0x29,0x08,0x09,0x28,0x29,0x09
,0x28,0x09,0x0A,0x27,0x28,0x0A,0x27,0x0A,0x0B,0x26,0x27,0x0B,0x25,0x26,0x0B
,0x24,0x25,0x0B,0x23,0x24,0x0B,0x23,0x0B,0x0C,0x22,0x23,0x0C,0x21,0x22,0x0C
,0x21,0x0C,0x0D,0x20,0x21,0x0D,0x20,0x0D,0x0E,0x1F,0x20,0x0E,0x1F,0x0E,0x0F
,0x1E,0x1F,0x0F,0x1E,0x0F,0x10,0x1D,0x1E,0x10,0x1D,0x10,0x11,0x1C,0x1D,0x11
,0x1C,0x11,0x12,0x1B,0x1C,0x12,0x13,0x1B,0x12,0x32,0x7E,0x1E,0x1C,0x01,0x64
,0xE6,0xD2,0x3E,0xC4,0x7C,0x99,0xBD,0x4A,0xB8,0xC0,0x3E,0x2F,0x35,0x02,0xBE
,0xD5,0x21,0xA7,0x3E,0xA8,0x1C,0x23,0xBE,0x07,0x29,0x88,0x3E,0x7B,0x14,0x2E
,0xBE,0x6C,0x96,0x3B,0x3E,0x9A,0x95,0x1D,0xBE,0x2D,0x77,0x66,0x3D,0x6A,0x31
,0xD8,0xBD,0xEB,0x74,0xA0,0xBD,0x26,0x1B,0x4F,0xBD,0x64,0x93,0x3C,0xBE,0x2B
,0x89,0x6C,0xBC,0x27,0x89,0x65,0xBE,0x66,0x69,0xA7,0xBB,0xC4,0xCC,0x86,0xBE
,0x6F,0x12,0x03,0xBB,0x13,0x82,0xAD,0xBE,0xDC,0xF2,0x91,0xBC,0xF6,0xD0,0xCE
,0xBE,0x2E,0xAA,0x85,0xBD,0xB9,0x72,0xE6,0xBE,0x04,0x01,0x12,0xBE,0xC5,0x20
,0xF0,0xBE,0xB6,0xF3,0x7D,0xBE,0x08,0xAC,0xDC,0xBE,0xB6,0xF3,0x7D,0xBE,0xCD
,0x77,0xD0,0xBE,0x81,0x41,0x32,0xBE,0xA6,0xEF,0xBD,0xBE,0xBE,0x14,0xFE,0xBD
,0x83,0xC1,0xA5,0xBE,0xE5,0x9D,0xC3,0xBD,0xB7,0x9B,0x88,0xBE,0xC5,0x20,0xB0
,0xBD,0x19,0xAA,0x42,0xBE,0x5E,0x2F,0xCD,0xBD,0xA3,0x20,0x98,0xBD,0x0E,0x2D
,0x12,0xBE,0xCC,0x24,0xAA,0x3D,0xE8,0xF6,0x52,0xBE,0x9E,0x97,0x2A,0x3E,0xB7
,0xEF,0x71,0xBE,0x41,0xBA,0x58,0x3E,0xCC,0x7B,0x7C,0xBE,0x23,0xDB,0x81,0x3E
,0x00,0x00,0x80,0xBE,0xFA,0xD2,0xAB,0x3E,0xCF,0x49,0x6F,0xBE,0xF6,0xD0,0xCE
,0x3E,0xFA,0x26,0x3D,0xBE,0x69,0xE5,0xE6,0x3E,0x57,0xED,0xDA,0xBD,0xC5,0x20
,0xF0,0x3E,0x6F,0x12,0x83,0xBB,0xE3,0xA5,0xDB,0x3E,0x6F,0x12,0x83,0xBB,0x0C
,0x0D,0x0E,0x0C,0x0E,0x0F,0x1B,0x1C,0x1D,0x1B,0x1D,0x00,0x1B,0x00,0x01,0x1A
,0x1B,0x01,0x1A,0x01,0x02,0x19,0x1A,0x02,0x19,0x02,0x03,0x18,0x19,0x03,0x0C
,0x0F,0x10,0x0B,0x0C,0x10,0x0B,0x10,0x11,0x0A,0x0B,0x11,0x09,0x0A,0x11,0x09
,0x11,0x12,0x08,0x09,0x12,0x08,0x12,0x13,0x07,0x08,0x13,0x06,0x07,0x13,0x06
,0x13,0x14,0x05,0x06,0x14,0x05,0x14,0x15,0x04,0x05,0x15,0x04,0x15,0x16,0x03
,0x04,0x16,0x03,0x16,0x17,0x18,0x03,0x17,0x1D,0x7F,0x08,0x04,0x02,0xBC,0x74
,0x13,0xBD,0x8B,0x6C,0x27,0x3F,0xBC,0x74,0x13,0xBD,0x00,0x00,0x00,0x00,0xBC
,0x74,0x13,0x3D,0x00,0x00,0x00,0x00,0xBC,0x74,0x13,0x3D,0x8B,0x6C,0x27,0x3F
,0xBC,0x74,0x13,0xBD,0x31,0x08,0xAC,0xBE,0xBC,0x74,0x13,0xBD,0xA4,0x70,0x7D
,0xBF,0xBC,0x74,0x13,0x3D,0xA4,0x70,0x7D,0xBF,0xBC,0x74,0x13,0x3D,0x31,0x08
,0xAC,0xBE,0x02,0x03,0x00,0x02,0x00,0x01,0x06,0x07,0x04,0x06,0x04,0x05,0x03
,0x07};
/* end binary data. size = 45361 bytes */

    fwrite(&bin_data,  c_size, 45361, font_file);

    fclose(font_file);

}





