/*  common.h
 *
 *  Header file for xfhell
 */

/*
 *  xfhell: An application to transmit and receive
 *  Feld Hell signals using a computer's sound card
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef COMMON_H
#define COMMON_H	 1

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <alsa/asoundlib.h>

/* General definitions for signal processing */
#define KEY_BUFF_SIZE		 256 /* Keyboard buffer size */

/* Number and size of labels for commands  */
#define NUM_OF_LABELS		10
#define LABEL_LENGTH		25

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE	128

/* Signal Detector parameter definitions */
#define ELEMENT_MAX			100	/* Defined max val of dot elements */

/* DFT parameter definitions */
#define DFT_INPUT_SIZE	1020 /* Size of dft input samples array */

/* Flag definitions */
#define NO_CALLBACKS		0x00000001U	/* Don't respond to callback if needed */
#define RECORD_QSO			0x00000002	/* Enable recording of QSO's */
#define CAPITALIZE			0x00000004	/* Capitalize letters in Transmit */
#define CAPTURE_SETUP		0x00000008	/* Sound Capture has been set-up  */
#define PLAYBACK_SETUP		0x00000010	/* Sound Playback has been set-up */
#define MIXER_SETUP			0x00000020	/* Sound Mixer has been set-up */
#define TRANSMIT_MODE		0x00000040	/* xfhell is in Transmit mode  */
#define RECEIVE_MODE		0x00000080	/* xfhell is in Receive mode   */
#define TRANSMIT_MACRO		0x00000100	/* Start transmission of Macro */
#define TRANSMIT_TAG		0x00000200	/* Transmit text of tag in macro */
#define TRANSMIT_KEYBD		0x00000400	/* Transmission from Keyboard */
#define SAVE_RECORD			0x00000800	/* Record was saved to file */
#define ENABLE_SCOPE		0x00001000	/* Display signal strength 'scope */
#define CLEAR_TX_WINDOW		0x00002000	/* Clear the Transmit window */
#define CLEAR_RX_WINDOW		0x00004000	/* Clear the Receive window */
#define AUTO_DESKEW			0x00008000	/* Enable automatic deskewing */
#define TX2RX_LOOPBACK		0x00010000	/* Enable loopback from Tx to Rx */
#define KEYBD_BUSY			0x00020000	/* Transmit from Keyboard */
#define SAVE_PIXBUF			0x00040000	/* Save the Receive window pixbuf */
#define ENABLE_CAT			0x00080000	/* Enable transceiver CAT */
#define CAT_SETUP			0x00100000	/* Trcvr CAT setup OK  */
#define ENABLE_RTS_DTR		0x01000000	/* Enable RTS/DTR control of PTT */
#define ENABLE_DTR			0x02000000	/* Enable DTR control of PTT */
#define RTS_DTR_SETUP		0x04000000	/* RTS control setup OK */
#define MODE_FELDHELL		0x08000000	/* Select FeldHell mode */
#define INITIALIZE_CAIRO	0x10000000	/* Initialize cairo lib */
#define TCVR_SERIAL_TEST	0x20000000	/* Serial port is under test */
#define NEW_BAUD_RATE		0x40000000	/* New baud rate selected by user */

/* Some character definitions */
#define HT			0x09 /* Horizontal Tab  */
#define CR			0x0d /* Carriage Return */
#define LF			0x0a /* Line Feed       */
#define BS			0x08 /* Back Space      */
#define IDLE_CHAR	0x1f /* Idle character  */

/* Return values */
#define SUCCESS		0
#define ERROR		1

/* For Error Dialog */
#define QUIT	TRUE
#define OK		FALSE

/*-------------------------------------------------------------------*/

/* Glyph data structure */
typedef struct
{
  int
	encoding,
	dwidth_x,
	dwidth_y,
	bbx_w,
	bbx_h,
	bbx_x,
	bbx_y;

  unsigned int *bitmap;

} glyph_data_t;

/* Font data structure */
typedef struct
{
  /* Name of current font */
  char font_name[81];

  /* Font bounding box */
  int fbb_w, fbb_h, fbb_x, fbb_y;

  int
	first_glyph, /* First character in font file */
	last_glyph,  /* Last character in font file  */
	num_glyphs;  /* Number of chars in font file */

  glyph_data_t *glyph_data;

} font_data_t;

/* Configuration parameters structure */
typedef struct
{
  char
	snd_card[32], /* Sound card name */
	pcm_dev[32],  /* PCM device name */
	cap_src[32],  /* Capture source  */
	cap_vol[32],  /* Capture volume control  */
	mst_vol[32],  /* Master volume control   */
	pcm_vol[32];  /* Playback volume control */

  int
	channel,	/* ALSA "channel" for use in mixer setup */
	num_chn,	/* Number of audio channels (2=stereo, 1=mono) */
	use_chn,	/* Channel in use (frontleft=0, frontright=1 etc) */
	cap_lev,	/* Recording/Capture level */
	pbk_lev,	/* Playback (volume) level */
	tx_output,  /* Tx power output level */
	dsp_rate;	/* DSP rate (speed) samples/sec */

  /* Transceiver type 1=FT847 2=FT857 3=K2 4=K3 0=NONE */
  int tcvr_type;

  char
	cat_serial[32], /* CAT serial port */
	rts_serial[32]; /* RTS/DTR serial port */

  gchar font_file[81]; /* Font file name */

  /* Hellschreiber transmit baud rate */
  double baud_rate;

  int
	wpm,				/* Morse encoder speed wpm */
	wwrap,				/* Word wrap column number */
	dot_size,			/* Receive dot height in pixels */
	bitmap_height,		/* Receive bitmap height in pixels */
	bitmap_width,		/* Receive bitmap width in pixels  */
	num_rows,			/* Number of character rows in Rx window */
	samples_per_cycle,	/* Samples per cycle of audio freq. */
	samples_per_dot,	/* Samples per Hell character half dot */
	tone_freq,			/* Input tone (Rx BFO) frequency */
	dft_stride,			/* DFT's stride over input data  */
	center_line,		/* Position of tuning reference in dft window */
	txbuff_len,			/* Tx signal samples buffer size */
	rxbuff_len;			/* Rx signal samples buffer size */

  gboolean rev_video; /* Reverse "video" on transmit */

  FILE
	*qso_record_fp,  /* File for recording QSO's */
	*log_adif_fp,    /* File for ADIF QSO record */
	*station_log_fp; /* File for station's log   */

  /* Local operator data */
  char
	call[15],
	name[13],
	zone[15],
	qth[13],
	loc[7];

  /* Tx/Rx power, N.Fig, antennas */
  char
	tx[12],
	tx_power[10],
	tx_ant[16],
	rx[12],
	rx_nfig[8],
	rx_ant[16];

  /* CW message to send*/
  char cw_mesg[25];

  /* Font data buffer */
  font_data_t font_data;

} rc_data_t;

/* QSO record type definition */
typedef struct
{
  /* QSO data */
  char
	dx_call[15],  /* Remote (dx) station callsign */
	dx_rst[4],    /* My RST report to Dx station  */
	my_rst[4],    /* My RST report to Dx station  */
	dx_name[13],  /* Remote station operator name */
	dx_qth[13],   /* Remote station QTH loc. name */
	dx_loc[7],    /* Remote station QTH locator   */
	date[12],     /* Normal format QSO date  */
	date_adif[9], /* QSO date in ADIF format */
	time[6],      /* Normal format QSO time  */
	time_adif[5], /* QSO time in ADIF format */
	freq[14],     /* QSO operating frequency MHz */
	mode[12],     /* Mode of operation */
	via[13];      /* Via which path or relay Sat */

  /* Remarks on QSO */
  char remarks[46];

} qso_record_t;

/* Data needed to handle pixbufs related to
 * the Receive and Waterfal drawing areas */
typedef struct
{
  GdkPixbuf *pixbuf; /* The GDK pixbuf */

  guchar *pixels;	 /* Pointer to the pixels in above */

  int
	rowstride,		 /* Row stride of pixbuf */
	n_channels,		 /* Number of channels of pixbuf */
	width,			 /* Width of pixbuf in pixels  */
	height;			 /* Height of pixbuf in pixels */

} pixbuf_data_t;

/*-------------------------------------------------------------------*/

/* Transceiver status data */
typedef struct
{
  int
	comp_lev,	/* Compressor Level */
	tx_power,	/* Tx Power Output  */
	vfo_freq,	/* VFO A frequency  */
	rig_mode;	/* Transceiver mode (usb|cw etc) */

} tcvr_status_t;

/*-------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* callbacks.c */
void Error_Dialog(char *message, gboolean hide);
void Quit_Dialog(char *message);
void Save_Dialog(char *message);
void Bad_Entry_Dialog(char *message);
void on_main_window_destroy(GtkObject *object, gpointer user_data);
gboolean on_main_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_tx_textview_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean on_tx_textview_key_press_event(GtkWidget *widget, GdkEventKey *event, gpointer user_data);
void on_xmit_clicked(GtkButton *button, gpointer user_data);
void on_txvol_spinbutton_value_changed(GtkSpinButton *spinbutton, gpointer user_data);
gboolean on_drawingarea_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_drawingarea_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_drawingarea_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
void on_rcve_clicked(GtkButton *button, gpointer user_data);
void on_deskew_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_height_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_rxvol_spinbutton_value_changed(GtkSpinButton *spinbutton, gpointer user_data);
void on_contrast_value_changed(GtkRange *range, gpointer user_data);
gboolean on_scope_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_scope_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_scope_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
void on_rst_out_changed(GtkEditable *editable, gpointer user_data);
void on_rst_in_changed(GtkEditable *editable, gpointer user_data);
void on_op_name_changed(GtkEditable *editable, gpointer user_data);
void on_qth_name_changed(GtkEditable *editable, gpointer user_data);
void on_locator_changed(GtkEditable *editable, gpointer user_data);
void on_band_changed(GtkEditable *editable, gpointer user_data);
void on_callsign_changed(GtkEditable *editable, gpointer user_data);
void on_new_record_clicked(GtkButton *button, gpointer user_data);
void on_save_record_clicked(GtkButton *button, gpointer user_data);
void on_clear_fields_clicked(GtkButton *button, gpointer user_data);
void on_clear_windows_clicked(GtkButton *button, gpointer user_data);
void on_macro1_clicked(GtkButton *button, gpointer user_data);
void on_macro2_clicked(GtkButton *button, gpointer user_data);
void on_macro3_clicked(GtkButton *button, gpointer user_data);
void on_macro4_clicked(GtkButton *button, gpointer user_data);
void on_macro5_clicked(GtkButton *button, gpointer user_data);
void on_macro6_clicked(GtkButton *button, gpointer user_data);
void on_macro7_clicked(GtkButton *button, gpointer user_data);
void on_macro8_clicked(GtkButton *button, gpointer user_data);
void on_macro9_clicked(GtkButton *button, gpointer user_data);
void on_macro10_clicked(GtkButton *button, gpointer user_data);
void on_error_dialog_destroy(GtkObject *object, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_error_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_error_ok_button_clicked(GtkButton *button, gpointer user_data);
void on_cancel_save_clicked(GtkButton *button, gpointer user_data);
void on_save_ok_clicked(GtkButton *button, gpointer user_data);
void on_feldhell_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fmhell_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_bdrate_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_font_selector_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_show_font_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_reverse_video_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_k2_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_k3_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_ft847_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_ft857_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_rts_ctl_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_none_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_md2x4_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_md1x2_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_record_qsos_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_capitalize_letters_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_loopback_tx_to_rx_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_identify_in_cw_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_quit_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_bad_entry_ok_clicked(GtkButton *button, gpointer user_data);
void on_cancel_quit_clicked(GtkButton *button, gpointer user_data);
void on_quit_ok_clicked(GtkButton *button, gpointer user_data);
void on_fileselection_response(GtkDialog *dialog, gint response_id, gpointer user_data);
/* cat.c */
gboolean Open_Tcvr_Serial(void);
void Close_Tcvr_Serial(void);
gboolean Open_RTS_DTR_Serial(void);
void Cancel_CAT(void);
gboolean Read_Tcvr_Serial(char *data, size_t len);
gboolean Write_Tcvr_Serial(char *data, size_t len);
gboolean Read_Rx_Status(char *freq, char *rst);
gboolean Set_Tcvr_Mode(void);
gboolean Tune_Tcvr(double x);
gboolean Read_Rx_Freq(int *freq);
gboolean Write_Rx_Freq(int freq);
gboolean RS232_RTS_DTR_Set(gboolean set);
gboolean Tcvr_Status(int action, tcvr_status_t *status);
gboolean Set_Tx_Power(int tx_power, int comp_lev);
/* codec.c */
gboolean Assemble_Column(unsigned char *column);
gboolean Transmit_Character(font_data_t *font_data, int glx);
gboolean Tx_FMHell_Character(font_data_t *font_data, int glx);
gboolean Tx_Feld_Character(font_data_t *font_data, int glx);
gboolean Morse_Transmit(char *mesg);
gboolean Make_Sin_Wavetable(void);
gboolean Make_Cos_Wavetable(void);
/* detect.c */
gboolean Get_Element(int *elem_lev);
gboolean Feld_Level(int *elem_lev);
gboolean FMHell_Level(int *elem_lev);
/* display.c */
gboolean Draw_Column(unsigned char *column);
void Display_Waterfall(void);
gboolean Create_Pixbuf(GtkWidget *drawingarea);
void Clear_Pixbuf(pixbuf_data_t *pixbuf, guint32 fg);
void Print_Character(int printchr, GtkTextBuffer *text_buffer, GtkWidget *scrolledwindow, GtkTextIter *iter);
void Display_Signal(int plot);
void Set_TxRx_Labels(void);
/* dft.c */
void Idft_Init(int dft_input_size, int dft_bin_size);
void Idft(int dft_input_size, int dft_bin_size);
/* interface.c */
GtkWidget *create_main_window(void);
GtkWidget *create_error_dialog(void);
GtkWidget *create_save_dialog(void);
GtkWidget *create_main_menu(void);
GtkWidget *create_bad_entry_dialog(void);
GtkWidget *create_quit_dialog(void);
GtkWidget *create_fileselection(void);
/* main.c */
int main(int argc, char *argv[]);
gboolean Load_Config(gpointer data);
/* operation.c */
gboolean Transmit_Keybd(gpointer data);
gboolean Transmit_Macro(gpointer data);
gboolean Receive_Mode(gpointer data);
gboolean Initialize_Transmit(gpointer data);
/* shared.c */
/* sound.c */
gboolean Open_PCM(snd_pcm_t **handle, snd_pcm_stream_t stream, char *mesg, int *error);
gboolean Open_Capture(char *mesg, int *error);
gboolean Open_Playback(char *mesg, int *error);
gboolean Open_Mixer(char *mesg, int *error);
gboolean Set_Playback_Level(int level, char *mesg, int *error);
gboolean Set_Capture_Level(int level, char *mesg, int *error);
void Close_Playback(void);
void Close_Capture(void);
void Close_Mixer(void);
gboolean Signal_Sample(short *sample_val);
gboolean DSP_Write(short *buffer, int buff_size);
gboolean Xrun_Recovery(snd_pcm_t *handle, int error);
/* support.c */
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name);
void add_pixmap_directory(const gchar *directory);
GtkWidget *create_pixmap(GtkWidget *widget, const gchar *filename);
GdkPixbuf *create_pixbuf(const gchar *filename);
void glade_set_atk_action_description(AtkAction *action, const gchar *action_name, const gchar *description);
/* utils.c */
gboolean Save_QSO_Record(void);
gboolean Read_QSO_Record(void);
void Clear_Record_Fields(gboolean all);
gboolean Open_Record_File(void);
gboolean Load_Fonts(char *font_file);
gboolean Find_Line(char *line, const char *str, FILE *fp);
void Find_String(char *line, int *idx);
unsigned int Hex2Bitmap(char *hex_str);
gboolean Render_Font(font_data_t *font, int glx);
gboolean Save_Pixbuf(void);
gboolean Alloc_Tx_Buffer(void);
void Process_RST(GtkEditable *editable);
void Select_Macro(int macro_num);
int Open_File(FILE **fp, char *fpath, const char *mode);
void Close_File(FILE **fp);
gboolean File_Print(FILE **fp, char *string);
gboolean Locate_Line(char **line_idx, char *line, const char *string, FILE *fp);
gboolean Locate_String(char **line_idx, char *line, const char *string);
void New_Parameters(void);
gboolean mem_alloc(void **ptr, size_t req);
gboolean mem_realloc(void **ptr, size_t req);
void free_ptr(void **ptr);
int isFlagSet(long flag);
int isFlagClear(long flag);
void SetFlag(long flag);
void ClearFlag(long flag);
void ToggleFlag(long flag);
void Cleanup(void);
int Load_Line(char *buff, FILE *pfile, char *messg);
void Usage(void);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);
double Atof(const char *nptr);

#endif /* End of xfhell.h */

