/*  detect.c
 *
 *  Signal detection functions of xfhell application
 */

/*
 *  xfhell: An application to transmit and receive
 *  Feld Hell signals using a computer's sound card
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Get_Element()
 *
 *  Gets the signal level over a part of a character's dot.
 */

  gboolean
Get_Element( int *elem_lev )
{
  if( isFlagSet(MODE_FELDHELL) )
  {
	if( !Feld_Level(elem_lev) )
	  return( FALSE );
  }
  else if( !FMHell_Level(elem_lev) )
	return( FALSE );

  /* Display (scaled) signal level graph */
  if( isFlagSet(ENABLE_SCOPE) )
	Display_Signal( *elem_lev );

  return( TRUE );
} /* End of Get_Element() */

/*------------------------------------------------------------------------*/

/*  Feld_Level()
 *
 *  Detects level of a FeldHell signal using Goertzel's algorithm.
 */

  gboolean
Feld_Level( int *elem_lev )
{
  static int
	fbb_h = 0,	/* Font height */
	first_call = TRUE,
	det_period,	/* Integration period of Goertzel detector */
	buff_idx;	/* Sample buffer index */

  int idx;

  /* Circular signal samples buffer for Goertzel detector */
  static short *samples_buff = NULL;

  /* Variables for the Goertzel algorithm */
  static double coeff, scale;
  double q0, q1, q2;


  /* Initialize on change of parameters */
  if( first_call ||
	  isFlagSet(NEW_BAUD_RATE) ||
	  (fbb_h != rc_data.font_data.fbb_h) )
  {
	double w;

	first_call = FALSE;
	ClearFlag( NEW_BAUD_RATE );
	fbb_h = rc_data.font_data.fbb_h;

	/* Calculate omega for Goertzel detector */
	w = 2.0 * M_PI / (double)rc_data.samples_per_cycle;
	coeff = 2.0 * cos( w );

	/* Allocate samples buffer and clear */
	det_period = rc_data.samples_per_dot + fbb_h;
	if( !mem_realloc((void **)&samples_buff,
		  sizeof(short) * (size_t)det_period) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );

	  return( FALSE );
	}
	bzero( samples_buff, sizeof(short) * (size_t)det_period );
	buff_idx = 0;

  } /* if( first_call || isFlagSet(NEW_BAUD_RATE) ) */

  /* Save samples for a font dot pixel, include deskew if selected */
  det_period = rc_data.samples_per_dot + deskew;
  for( idx = 0; idx < det_period; idx++ )
  {
	/* Get next signal sample from buffer, abort on error */
	if( !Signal_Sample(&samples_buff[buff_idx]) )
	  return( FALSE );

	/* Icrement/reset circular buffer's index */
	buff_idx++;
	if( buff_idx >= det_period ) buff_idx = 0;

  } /* for( idx = 0; idx < rc_data.samples_per_dot; idx++ ) */

  /*** Calculate signal element level ***/
  /* Calculate element level using Goertzel algorithm */
  q1 = q2 = 0.0;
  for( idx = 0; idx < det_period; idx++ )
  {
	q0 = coeff * q1 - q2 + (double)samples_buff[buff_idx];
	q2 = q1;
	q1 = q0;

	/* Icrement/reset circular buffers' index */
	buff_idx++;
	if( buff_idx >= det_period ) buff_idx = 0;

  } /* for( idx = 0; idx < det_period; idx++ ) */

  /* To reduce signal level to usable value */
  scale = det_period * FELDHL_SCALE_FACTOR;

  /* Level of input signal scaled down and clamped */
  q1 /= scale;
  q2 /= scale;
  *elem_lev = (int)(q1*q1 + q2*q2 - q1*q2*coeff);

  return( TRUE );
} /* Feld_Level() */

/*------------------------------------------------------------------------*/

/* FMHell_Level()
 *
 * Detects frequency shifts of FMHell carrier
 */

  gboolean
FMHell_Level( int *elem_lev )
{
  static int
	first_call = TRUE,
	det_period,	/* Integration period of Goertzel detector */
	buff_idx;	/* Sample buffer index */

  int
	idx,
	blk_lev,    /* Level of the black FeldHell signal 'element' */
	wht_lev;    /* Level of the white FeldHell signal 'element' */

  /* Circular signal samples buffer for Phase detector */
  static short *samples_buff = NULL;

  /* Variables for the Goertzel algorithm */
  static double blk_cosw, blk_coeff, scale;
  static double wht_cosw, wht_coeff;
  double blk_q0, blk_q1, blk_q2;
  double wht_q0, wht_q1, wht_q2;


  /* Initialize on change of parameters */
  if( first_call || isFlagSet(NEW_BAUD_RATE) )
  {
	double w, twopi = 2.0 * M_PI;

	/* Omega for the white frequency */
	w = twopi / (double)rc_data.dsp_rate *
	  ( (double)rc_data.tone_freq + rc_data.baud_rate );
	wht_cosw  = cos( w );
	wht_coeff = 2.0 * wht_cosw;

	/* Omega for the black frequency */
	w = twopi / (double)rc_data.dsp_rate *
	  ( (double)rc_data.tone_freq - rc_data.baud_rate );
	blk_cosw  = cos( w );
	blk_coeff = 2.0 * blk_cosw;

	det_period = rc_data.samples_per_dot;

	/* To keep values of detected signal in reasonable limits */
	scale = (double)det_period * FMHELL_SCALE_FACTOR;

	/* Allocate samples buffer and clear */
	if( !mem_realloc((void **)&samples_buff,
		sizeof(short) * (size_t)det_period) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );

	  return( FALSE );
	}
	bzero( samples_buff, sizeof(short) * (size_t)det_period );
	buff_idx = 0;

	first_call = FALSE;
	ClearFlag( NEW_BAUD_RATE );

  } /* if( first_call... ) */

  /* Save samples for a font dot */
  for( idx = 0; idx < rc_data.samples_per_dot; idx++ )
  {
	/* Get signal sample from buffer, abort on error */
	if( !Signal_Sample(&samples_buff[buff_idx]) )
	  return( FALSE );

	/* Increment/reset circular buffer's index */
	buff_idx++;
	if( buff_idx >= det_period ) buff_idx = 0;

  } /* for( idx = 0; idx < rc_data.samples_per_pixel; idx++ ) */

  /* Calculate signal level of black and white
   * tone frequencies using Goertzel algorithm */
  blk_q1 = blk_q2 = 0.0;
  wht_q1 = wht_q2 = 0.0;
  for( idx = 0; idx < det_period; idx++ )
  {
	blk_q0 =
	  blk_coeff * blk_q1 - blk_q2 +
	  (double)samples_buff[buff_idx];
	blk_q2 = blk_q1;
	blk_q1 = blk_q0;

	wht_q0 =
	  wht_coeff * wht_q1 - wht_q2 +
	  (double)samples_buff[buff_idx];
	wht_q2 = wht_q1;
	wht_q1 = wht_q0;

	/* Increment/reset circular buffers' index */
	buff_idx++;
	if( buff_idx >= det_period ) buff_idx = 0;

  } /* for( idx = 0; idx < det_period; idx++ ) */

  /* Magnitude of black tone scaled by dot size and tone freq */
  blk_q1 /= scale;
  blk_q2 /= scale;
  blk_lev = (int)
	((blk_q1*blk_q1 + blk_q2*blk_q2 - blk_q1*blk_q2*blk_coeff));

  /* Magnitude of white tone scaled by dot size and tone freq */
  wht_q1 /= scale;
  wht_q2 /= scale;
  wht_lev = (int)
	((wht_q1*wht_q1 + wht_q2*wht_q2 - wht_q1*wht_q2*wht_coeff));

  /* The detector o/p is the level diff
   * between white and black signals */
  *elem_lev = 127 + blk_lev - wht_lev;

  return( TRUE );
} /* End of FMHell_Level() */

/*------------------------------------------------------------------------*/

