/* utils.c
 *
 *  Utility functions for xfhell
 */

/*
 *  xfhell: An application to transmit and receive
 *  Feld Hell signals using a computer's sound card
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"

/*------------------------------------------------------------------------*/

/*  Save_QSO_Record()
 *
 *  Saves QSO record to .adif and .txt files
 */

  gboolean
Save_QSO_Record( void )
{
  /* For opening files */
  char file_name[64];

  /* Buffer for reading and printing to files */
  char buff[REC_BUFF_SIZE];
  char *buff_idx;

  int
	num_records, /* Number of QSO records    */
	num_logged,  /* Number of QSL's printed  */
	fopen_ret;   /* Open_File() return value */

  if( isFlagClear(SAVE_RECORD) ) return( TRUE );
  ClearFlag( SAVE_RECORD );

  /* Create a file for QSO ADIF log if not open */
  Strlcpy( file_name, getenv("HOME"), sizeof(file_name) );
  Strlcat( file_name, "/xfhell/log.adif", sizeof(file_name) );
  fopen_ret = Open_File( &rc_data.log_adif_fp, file_name, "a" );
  if( fopen_ret == FILE_NEW )
  {
	snprintf( buff, sizeof(buff),
		"QSO Log file created in ADIF v1.0 format by %s <EOH>\n\n",
		PACKAGE_STRING );
	if( !File_Print(&rc_data.log_adif_fp, buff) )
	  return( FALSE );
  }
  else if( fopen_ret == FILE_FAIL )
	return( FALSE );

  /* Print QSO record to adif format file */
  snprintf( buff, sizeof(buff),
	  "<CALL:%d>%s<QSO_DATE:8>%s<TIME_ON:4>%s\n"
	  "<FREQ:%d>%s<MODE:4>Hell<RST_SENT:3>%3s<EOR>\n\n",
	  (int)strlen(qso_record.dx_call), qso_record.dx_call,
	  qso_record.date_adif, qso_record.time_adif,
	  (int)strlen(qso_record.freq), qso_record.freq,
	  qso_record.dx_rst );
  if( !File_Print(&rc_data.log_adif_fp, buff) )
	return( FALSE );

  /* Create a file for station log if not already open */
  Strlcpy( file_name, getenv("HOME"), sizeof(file_name) );
  Strlcat( file_name, "/xfhell/log.txt", sizeof(file_name) );
  fopen_ret = Open_File( &rc_data.station_log_fp, file_name, "r+" );
  if( fopen_ret == FILE_NEW )
  {
	/* Print a header to log file */
	snprintf( buff, sizeof(buff),
		_("|------------------------------------"\
		"--------------------------------------|\n"\
		"|         Station Log File - Created in"\
		" Text format by %10s          |\n"\
		"|------------------------------------"\
		"--------------------------------------|\n"),
		PACKAGE_STRING );
	if( !File_Print(&rc_data.station_log_fp, buff) )
	  return( FALSE );

	/* Print an initial (0) count of records and qsl's */
	snprintf( buff, sizeof(buff),
		_("|  Num of QSO Records:%-6d  Records Logged:%-6d"\
		"  Size of Records:%-4d  |\n"\
		"|------------------------------------"\
		"--------------------------------------|\n"),
		0, 0, RECORD_SIZE );
	if( !File_Print(&rc_data.station_log_fp, buff) )
	  return( FALSE );
  }
  else if( fopen_ret == FILE_FAIL )
	return( FALSE );

  /*** Increment and print Record and QSL count ***/
  /* Go to beginning of log file */
  rewind( rc_data.station_log_fp );

  /* Find line with "Number of QSO Records:" */
  if( !Locate_Line(&buff_idx, buff, "QSO Records:", rc_data.station_log_fp) )
	return( FALSE );
  num_records = atoi(buff_idx);

  /* Find "Number of Records Logged:" string */
  if( !Locate_String(&buff_idx, buff, "Records Logged:") )
	return( FALSE );
  num_logged = atoi(buff_idx);

  /* Go back to beginning of line and print new record */
  if( fseek(rc_data.station_log_fp, -(long)(strlen(buff)+1), SEEK_CUR) != 0 )
	return( FALSE );
  snprintf( buff, sizeof(buff),
	  _("|  Num of QSO Records:%-6d  Records Logged:%-6d"\
	  "  Size of Records:%-4d  |\n"\
	  "|------------------------------------"\
	  "--------------------------------------|\n"),
	  ++num_records, num_logged, RECORD_SIZE );
  if( !File_Print(&rc_data.station_log_fp, buff) )
	return( FALSE );

  /*** Print record in text log file ***/
  if( fseek(rc_data.station_log_fp, 0, SEEK_END) != 0 )
	return( FALSE );
  snprintf( buff, sizeof(buff),
	  _("|DX/ CALL: %-14s NAME: %-12s QTH: %-12s  LOC: %-6s|\n"\
	  "|MY/ CALL: %-14s ZONE: %-12s QTH: %-12s  LOC: %-6s|\n"\
	  "|QSO DATE: %-11s    TIME: %-5s UTC   FREQ: %-13s QSL: NO    |\n"\
	  "|QSO MODE: %-11s DX-REPT: %-3s      MY-REPT: %-3s     VIA: %-12s|\n"\
	  "|MY TRANS: %-11s   POWER: %-9s    ANT: %-15s          |\n"\
	  "|MY RECVR: %-11s   N.FIG: %-7s      ANT: %-15s          |\n"\
	  "| REMARKS: %-45s                   |\n"\
	  "|------------------------------------"\
	  "--------------------------------------|\n"),
	  qso_record.dx_call, qso_record.dx_name, qso_record.dx_qth, qso_record.dx_loc,
	  rc_data.call, rc_data.zone, rc_data.qth, rc_data.loc,
	  qso_record.date, qso_record.time, qso_record.freq,
	  qso_record.mode, qso_record.dx_rst, qso_record.my_rst, qso_record.via,
	  rc_data.tx, rc_data.tx_power, rc_data.tx_ant,
	  rc_data.rx, rc_data.rx_nfig, rc_data.rx_ant, qso_record.remarks );
  if( !File_Print(&rc_data.station_log_fp, buff) )
	return( FALSE );

  return( TRUE );
} /* Save_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Read_QSO_Record()
 *
 *  Reads and validates QSO record entries
 */

  gboolean
Read_QSO_Record( void )
{
  GtkWidget *entry;

  /* Enter field values to QSO record structure */
  entry = lookup_widget( main_window, "callsign" );
  Strlcpy( qso_record.dx_call,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.dx_call) );
  entry = lookup_widget( main_window, "rst_in" );
  Strlcpy( qso_record.my_rst,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.my_rst) );
  entry = lookup_widget( main_window, "rst_out" );
  Strlcpy( qso_record.dx_rst,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.dx_rst) );
  entry = lookup_widget( main_window, "op_name" );
  Strlcpy( qso_record.dx_name,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.dx_name) );
  entry = lookup_widget( main_window, "qth_name" );
  Strlcpy( qso_record.dx_qth,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.dx_qth) );
  entry = lookup_widget( main_window, "locator" );
  Strlcpy( qso_record.dx_loc,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.dx_loc) );
  entry = lookup_widget( main_window, "band" );
  Strlcpy( qso_record.freq,
	  gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_record.freq) );

  /* Enter operating mode */
  Strlcpy( qso_record.mode, "Hell", sizeof(qso_record.mode) );

  /* Validate QSO Record */
  if( (strlen(qso_record.dx_call) > 2) &&
	  (strlen(qso_record.my_rst) == 3) &&
	  (strlen(qso_record.freq)    > 0) )
	return( TRUE );
  else return( FALSE );

} /* Read_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Clear_Record_Fields()
 *
 *  Clears all QSO Record fields
 */

  void
Clear_Record_Fields( gboolean all )
{
  /* Save outstanding records */
  if( isFlagSet(SAVE_RECORD) )
	Save_QSO_Record();
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "callsign")), "" );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "rst_in")), "" );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "rst_out")), "" );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "op_name")), "" );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "qth_name")), "" );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "locator")), "" );
  if( all )
	gtk_entry_set_text( GTK_ENTRY(lookup_widget(main_window, "band")), "" );

  qso_record.dx_call[0] = '\0';
  qso_record.my_rst[0]  = '\0';
  qso_record.dx_rst[0]  = '\0';
  Strlcpy( qso_record.dx_name, "OP", sizeof(qso_record.dx_name) );
  qso_record.dx_qth[0]  = '\0';
  qso_record.dx_loc[0]  = '\0';
  if( all )
	qso_record.freq[0]  = '\0';

} /* Clear_Record_Fields() */

/*------------------------------------------------------------------------*/

/*  Open_Record_File()
 *
 *  Opens a file for raw QSO recording
 */

  gboolean
Open_Record_File(void)
{
  /* File path for QSO record  */
  char qso_record_fpath[64];


  /* Open file for QSO Record, abort on error */
  if( isFlagSet(RECORD_QSO) )
  {
	/* Create a file for recording QSO's */
	if( rc_data.qso_record_fp == NULL )
	{
	  /* Make file path for record file */
	  snprintf( qso_record_fpath, sizeof(qso_record_fpath),
		  "%s/xfhell/record.txt", getenv("HOME") );

	  /* Open file, abort on error */
	  rc_data.qso_record_fp = fopen( qso_record_fpath, "a" );
	  if( rc_data.qso_record_fp == NULL )
	  {
		perror( qso_record_fpath );
		Error_Dialog(
			_("Failed to open QSO Record file\n"\
			  "Quit xfhell and correct"), QUIT );
		return( FALSE );
	  }

	} /* if( QSO_record_fp == NULL ) */

  } /* if( isFlagSet(RECORD_QSO) ) */

  return( TRUE );
} /* Open_Record_File(void) */

/*------------------------------------------------------------------------*/

/*  Load_Fonts()
 *
 *  Loads a .bdf font file into a font buffer
 */

  gboolean
Load_Fonts( char *font_file )
{
  FILE *font_fp = NULL;

  int
	bmx,
	bmy,
	dot_cnt,
	glyph_idx,
	bmap_idx,
	old_fbb_h,
	idx;

  /* Buffer for Load_Line  */
  char line[81];

  unsigned int
	new_dot,
	last_dot;

  /* Open font file, abort on error */
  font_fp = fopen( font_file, "r" );
  if( font_fp == NULL )
  {
	perror( font_file );
	Error_Dialog( _("Failed to open Font file"), OK );
	return( FALSE );
  }

  /* Find font bounding box */
  if( !Find_Line(line, "FONTBOUNDINGBOX ", font_fp) )
  {
	Error_Dialog(
		_("Error reading Font file\n"\
		  "FONTBOUNDINGBOX entry not found"), OK );
	return( FALSE );
  }

  idx = 14;
  Find_String( line, &idx );
  rc_data.font_data.fbb_w = atoi( &line[idx] );
  Find_String( line, &idx );
  old_fbb_h = rc_data.font_data.fbb_h;
  rc_data.font_data.fbb_h = atoi( &line[idx] );
  Find_String( line, &idx );
  rc_data.font_data.fbb_x = atoi( &line[idx] );
  Find_String( line, &idx );
  rc_data.font_data.fbb_y = atoi( &line[idx] );

  /* If fbbx height changes, clear Rx window */
  if( old_fbb_h != rc_data.font_data.fbb_h )
	SetFlag( CLEAR_RX_WINDOW );

  /* Check fbbx size and position */
  if( (rc_data.font_data.fbb_w > 30) ||
	  (rc_data.font_data.fbb_w <  5) ||
	  (rc_data.font_data.fbb_x != 0) ||
	  (rc_data.font_data.fbb_y < -4) )
  {
	Error_Dialog(
		_("Error reading Font file\n"\
		  "FONTBOUNDINGBOX data incorrect"), OK );
	return( FALSE );
  }

  /* bitmap_height & bitmap_width are the height and width in pixels
   * of bitmap of Hell characters. This is the product of dot size in
   * pixels * size in dots of Hell characters. Basic Hellschreiber
   * characters are 7x7 dots but vertically dots are two pixels
   * high so the Tx font height is 2x the character size in dots.
   */
  rc_data.bitmap_height = rc_data.dot_size * rc_data.font_data.fbb_h;

  /* Read font name */
  if( !Find_Line(line, "FONT_NAME ", font_fp) )
  {
	Error_Dialog(
		_("Error reading Font file\n"\
		  "FONT_NAME entry not found"), OK );
	return( FALSE );
  }

  /* Remove spaces */
  idx = 8;
  Find_String( line, &idx );
  Strlcpy( rc_data.font_data.font_name, &line[idx],
	  sizeof( rc_data.font_data.font_name ) );

  /* Read number of characters */
  if( !Find_Line(line, "CHARS ", font_fp) )
  {
	Error_Dialog(
		_("Error reading Font file\n"\
		  "CHARS entry not found"), OK );
	return( FALSE );
  }
  rc_data.font_data.num_glyphs = atoi( &line[6] );

  /* Allocate memory to glyph buffer */
  rc_data.font_data.glyph_data = NULL;
  if( !mem_alloc( (void **)&rc_data.font_data.glyph_data,
	  (size_t)rc_data.font_data.num_glyphs * sizeof(glyph_data_t)) )
	return( FALSE );

  /* Read all glyph data */
  for( glyph_idx = 0; glyph_idx < rc_data.font_data.num_glyphs; glyph_idx++ )
  {
	/* Read ENCODING */
	if( !Find_Line(line, "STARTCHAR ", font_fp) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"STARTCHAR entry not found"), OK );
	  return( FALSE );
	}

	if( !Find_Line(line, "ENCODING ", font_fp) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"ENCODING entry not found"), OK );
	  return( FALSE );
	}
	rc_data.font_data.glyph_data[glyph_idx].encoding = atoi( &line[9] );

	/* Read DWIDTH */
	if( !Find_Line(line, "DWIDTH ", font_fp) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"DWIDTH entry not found"), OK );
	  return( FALSE );
	}

	idx = 5;
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].dwidth_x = atoi( &line[idx] )+1;
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].dwidth_y = atoi( &line[idx] );

	/* Read BBX */
	if( !Find_Line(line, "BBX ", font_fp) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"BBX entry not found"), OK );
	  return( FALSE );
	}

	idx = 2;
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].bbx_w = atoi( &line[idx] );
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].bbx_h = atoi( &line[idx] );
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].bbx_x = atoi( &line[idx] );
	Find_String( line, &idx );
	rc_data.font_data.glyph_data[glyph_idx].bbx_y = atoi( &line[idx] );

	/* Check DWIDTH and BBX w are same. DWIDTH
	 * was incremented by 1 to leave char space */
	if( (rc_data.font_data.glyph_data[glyph_idx].bbx_w !=
		  rc_data.font_data.glyph_data[glyph_idx].dwidth_x-1) &&
		(rc_data.font_data.glyph_data[glyph_idx].encoding != 0x20) )
	  fprintf( stderr, _("Encoding %2x: DWIDTH != BBX width\n"),
		  rc_data.font_data.glyph_data[glyph_idx].encoding );

	/* Make sure BBX is at the x origin */
	if( rc_data.font_data.glyph_data[glyph_idx].bbx_x != 0 )
	  fprintf( stderr, _("Encoding %2x: BBX x != 0\n"),
		  rc_data.font_data.glyph_data[glyph_idx].encoding );

	/* Read and convert bitmap entries */
	bmap_idx =
	  rc_data.font_data.fbb_h -
	  rc_data.font_data.glyph_data[glyph_idx].bbx_h -
	  rc_data.font_data.glyph_data[glyph_idx].bbx_y +
	  rc_data.font_data.fbb_y;

	/* Allocate bitmap buffer */
	rc_data.font_data.glyph_data[glyph_idx].bitmap = NULL;
	if( !mem_alloc((void **)&rc_data.font_data.glyph_data[glyph_idx].bitmap,
		sizeof(unsigned int) * (size_t)rc_data.font_data.fbb_h) )
	  return( FALSE );

	/* Clear unused part of bitmap */
	for( idx = 0; idx < bmap_idx; idx++ )
	  rc_data.font_data.glyph_data[glyph_idx].bitmap[idx] = 0;

	if( !Find_Line(line, "BITMAP", font_fp) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"BITMAP entry not found"), OK );
	  return( FALSE );
	}

	/* Read hex bitmap and convert to binary bitmap */
	for( idx = 0; idx < rc_data.font_data.glyph_data[glyph_idx].bbx_h; idx++ )
	{
	  if( Load_Line(line, font_fp, "Font hex bitmap" ) != SUCCESS )
		return( FALSE );
	  rc_data.font_data.glyph_data[glyph_idx].bitmap[bmap_idx++] = Hex2Bitmap(line);
	}

	/* Clear unused part of bitmap */
	for( idx = bmap_idx; idx < rc_data.font_data.fbb_h; idx++ )
	  rc_data.font_data.glyph_data[glyph_idx].bitmap[idx] = 0;

	/* Check that there are no lone 1 or 0 bits in glyph columns */
	dot_cnt = 1; last_dot = 0;
	for( bmx = 0; bmx < rc_data.font_data.glyph_data[glyph_idx].dwidth_x; bmx++ )
	{
	  /* Go up in bitmap buffer row by row */
	  for( bmy = rc_data.font_data.fbb_h-1; bmy >= 0; bmy-- )
	  {
		dot_cnt++;
		new_dot = rc_data.font_data.glyph_data[glyph_idx].bitmap[bmy] & (1 << (31-bmx));
		if( new_dot != last_dot )
		{
		  /* Check for lone 1|0 bits */
		  if( dot_cnt < 2 )
			fprintf( stderr,
				_("Encoding %2x: Single half-pixel in column %d\n"),
				rc_data.font_data.glyph_data[glyph_idx].encoding, bmx );

		  /* Check for odd num. of half-pixels in Lo-res fonts */
		  if( (dot_cnt & 1) && (strstr(rc_data.font_data.font_name, "Lo") != NULL) )
		  {
			fprintf( stderr,
				_("Encoding %2x: Odd number of half-pixels in column %d\n"),
				rc_data.font_data.glyph_data[glyph_idx].encoding, bmx );
		  }

		  dot_cnt = 0;
		}
		last_dot = new_dot;

	  } /* for( bmy = rc_data.font_data.fbb_h-1; bmy >= 0; bmy-- ) */

	} /* for( bmx = 0; bmx < ..... ) */

  } /* for( glyph_idx = 0; glyph_idx < ..... ) */

  /* Encoding num of first & last char in file */
  rc_data.font_data.first_glyph = rc_data.font_data.glyph_data[0].encoding;
  rc_data.font_data.last_glyph  =
	rc_data.font_data.first_glyph + rc_data.font_data.num_glyphs;

  fclose( font_fp );

  return( Alloc_Tx_Buffer() );
} /* Load_Fonts() */

/*------------------------------------------------------------------------*/

/*  Find_Line()
 *
 *  Finds a line starting with a given string
 */

  gboolean
Find_Line( char *line, const char *str, FILE *fp )
{
  do
  {
	if( Load_Line(line, fp, "Text file line") != SUCCESS )
	  return( FALSE );
  }
  while( strncmp( line, str, strlen(str) ) != 0 );

  return( TRUE );
} /* Find_Line() */

/*------------------------------------------------------------------------*/

/*  Find_String()
 *
 *  Finds a string following spaces
 */

  void
Find_String( char *line, int *idx )
{
  /* Look for a space */
  while( line[++*idx] != ' ' )
	if( (line[*idx] == CR) || (line[*idx] == LF) )
	{
	  Error_Dialog(
		  _("Error reading Font file\n"\
			"Unexpected End-Of-Line"), OK );
	  return;
	}

  /* Look for next non-space char */
  while( line[++*idx] == ' ' );
  if( (line[*idx] == CR) || (line[*idx] == LF) )
  {
	Error_Dialog(
		_("Error reading Font file\n"\
		  "Unexpected End-Of-Line"), OK );
	return;
  }

} /*  Find_String() */

/*------------------------------------------------------------------------*/

/*  Hex2Bitmap()
 *
 *  Convert a hex number to bitmap
 */

  unsigned int
Hex2Bitmap( char *hex_str )
{
  unsigned int bitmap;
  char hex[] = {
	'0','1','2','3','4','5','6','7',
	'8','9','A','B','C','D','E','F' };

  int
	str_idx,
	hex_idx;


  /* Convert characters of hex string to binary bitmap */
  str_idx = bitmap = 0;
  while( (str_idx < 8) && (hex_str[str_idx] != '\0') )
  {
	bitmap <<= 4;
	for( hex_idx = 0; hex_idx < 16; hex_idx++ )
	  if( hex[hex_idx] == hex_str[str_idx] ) break;
	str_idx++;

	bitmap |= (unsigned int)hex_idx;
  }

  /* Shift bitmap to uppermost bit */
  while( str_idx++ < 8 ) bitmap <<= 4;

  return( bitmap );
} /* Hex2Bitmap( char *hex_str ) */

/*------------------------------------------------------------------------*/

/*  Render_Font()
 *
 *  Renders a font buffer on the Rx screen
 */

  gboolean
Render_Font( font_data_t *font, int glx )
{
  unsigned char	dot_val; /* Dot value (black|white) */
  static unsigned char *column = NULL; /* Column of dots */
  static int fbb_h = 0, dot_size = 0;

  int
	idx,
	bmx,   /* Bitmap x index */
	bmy,   /* Bitmap y index */
	fpix;  /* Pixels in font */


  /* Allocate memory to column */
  if( (fbb_h != rc_data.dot_size * rc_data.font_data.fbb_h) ||
	  (dot_size != rc_data.dot_size) )
  {
	fbb_h = rc_data.dot_size * rc_data.font_data.fbb_h;
	dot_size = rc_data.dot_size;
	if( !mem_realloc((void **)&column,
		  sizeof(unsigned char) * (size_t)fbb_h) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  }
 
  /* Use columns according to DWIDTH y of font */
  for( bmx = 0; bmx < font->glyph_data[glx].dwidth_x; bmx++ )
  {
	/* Pointer to current pixel in column */
	fpix = rc_data.dot_size * rc_data.font_data.fbb_h - 1;

	/* Go down in bitmap buffer row by row */
	for( bmy = 0; bmy < rc_data.font_data.fbb_h; bmy++ )
	{
	  /* Set black or white according to bit value */
	  if( font->glyph_data[glx].bitmap[bmy] & (1 << (31-bmx)) )
		dot_val = 0xff; /* Black dot (intensity is inverted) */
	  else dot_val = 0x00; /* White dot */

	  /* Fill in all pixels of each dot */
	  for( idx = 0; idx < rc_data.dot_size; idx++ )
		column[fpix--] = dot_val;
	}

	if( !Draw_Column(column) ) return( FALSE );

  } /* for( bmx = font->glyph_data[glx].dwidth_y; bmx >= 0; bmx-- ) */

  return( TRUE );
} /* Render_Font() */

/*------------------------------------------------------------------------*/

/*  Save_Pixbuf()
 *
 *  Saves the Receive pixbuf as a jpeg image
 */

  gboolean
Save_Pixbuf( void )
{
  /* For saving pixbuf */
  char pixbuf_fpath[80];
  GError *error = NULL;
  gboolean err;
  int len;

  /* Variables for reading time (UTC) */
  time_t tp;
  struct tm utc;

  if( isFlagClear(SAVE_PIXBUF) ) return(TRUE);

  /* Make a file path for pixbuf file */
  snprintf( pixbuf_fpath, sizeof(pixbuf_fpath),
	  "%s/xfhell/pixbufs/", getenv("HOME") );

  /* Prepare a file name as UTC date-time. */
  /* Default paths are images/ and record/ */
  time( &tp );
  utc = *gmtime( &tp );
  len = (int)strlen( pixbuf_fpath );
  strftime( &pixbuf_fpath[len], 21, "%d%b%Y-%H%M%S.jpg", &utc );

  /* Save Receive pixbuf */
  err = gdk_pixbuf_save( receive_pixbuf.pixbuf, pixbuf_fpath,
	  "jpeg", &error, "quality", "75", NULL );
  if( error != NULL )
  {
	fprintf( stderr, "%s\n", error->message );
	Error_Dialog( _("Failed to save Receive pixbuf\n"), OK );
  }

  return( err );
} /* Save_Pixbuf() */

/*------------------------------------------------------------------------*/

/* Alloc_Tx_Buff()
 *
 * Allocates memory to transmit samples buffer
 */

  gboolean
Alloc_Tx_Buffer(void)
{
  /* Allocate memory to samples buffer */
  rc_data.txbuff_len =
	rc_data.font_data.fbb_h *
	rc_data.samples_per_dot;

  if( !mem_realloc((void **)&xmit_buffer,
	  (size_t)(rc_data.txbuff_len * rc_data.num_chn) * sizeof(short)) )
	return( FALSE );
  memset( xmit_buffer, 0,
	  (size_t)(rc_data.txbuff_len * rc_data.num_chn) * sizeof(short));

  return( TRUE );
} /* Alloc_Tx_Buff() */

/*------------------------------------------------------------------------*/

/*  Process_RST()
 *
 *  Processes RST-out and RST-in fields
 */

  void
Process_RST( GtkEditable *editable )
{
  G_CONST_RETURN gchar *rst;
  int len, idx;

  /* Validate RST: numerics only, */
  /* min 337 max 559, ignore blank field */
  rst = gtk_entry_get_text( GTK_ENTRY(editable) );
  len = (int)strlen(rst);

  /* Ignore blank fields */
  if( len == 0 ) return;

  for( idx = 0; idx < len; idx++ )
  {
	/* Reject non-numbers */
	if( (rst[idx] < '0') ||
		(rst[idx] > '9') )
	{
	  Bad_Entry_Dialog(
		  _("Invalid character entered\n"\
			"Numbers only allowed") );
	  return;
	}

	/* Reject RST < 337 or > 599 */
	switch( idx )
	{
	  case 0:
		if( (rst[idx] < '3') ||
			(rst[idx] > '5') )
		{
		  Bad_Entry_Dialog(
			  _("Number out of range\n"\
				"Min = 3 and Max = 5") );
		  return;
		}
		break;

	  case 1:
		if( rst[idx] < '3' )
		{
		  Bad_Entry_Dialog(
			  _("Number out of range\n"\
				"Min = 3 and Max = 9") );
		  return;
		}
		break;

	  case 2:
		if( rst[idx] < '7' )
		{
		  Bad_Entry_Dialog(
			  _("Number out of range\n"\
				"Min = 7 and Max = 9") );
		  return;
		}
	} /* switch( idx ) */

  } /* for( idx = 0; idx < len; idx++ ) */

} /* Process_RST() */

/*------------------------------------------------------------------------*/

/*  Select_Macro()
 *
 *  Selects a macro to be transmitted
 */

  void
Select_Macro( int macro_num )
{
  /* Transmit a macro */
  if( isFlagClear(TRANSMIT_MACRO) )
  {
	SetFlag( TRANSMIT_MACRO );
	ClearFlag( RECEIVE_MODE );
	gbl_macro_idx = macro_num - 1;
	Set_TxRx_Labels();
	gtk_idle_add( Transmit_Macro, NULL );
  }
  else ClearFlag( TRANSMIT_MACRO );

} /* Select_Macro() */

/*------------------------------------------------------------------------*/

/*  Open_File()
 *
 *  Opens a file
 */

  int
Open_File( FILE **fp, char *fpath, const char *mode )
{
  struct stat buf;

  /* Check if file exists */
  if( *fp == NULL )
  {
	/* If file does not exist, open new */
	if( stat(fpath, &buf) == -1 )
	{
	  /* Create new file */
	  if( (*fp = fopen(fpath, "w+")) == NULL )
	  {
		char mesg[MESG_SIZE];
		perror( fpath );
		snprintf( mesg, sizeof(mesg),
			_("Failed to create\n%s\n"\
			  "Quit xfhell and correct"), fpath );
		Error_Dialog( mesg, QUIT );
		return( FILE_FAIL );
	  }
	  else
	  {
		/* Open in requested mode */
		if( (*fp = fopen(fpath, mode)) == NULL )
		{
		  char mesg[MESG_SIZE];
		  perror( fpath );
		  snprintf( mesg, sizeof(mesg),
			  _("Failed to open\n%s\n"\
				"Quit xfhell and correct"), fpath );
		  Error_Dialog( mesg, QUIT );
		  return( -1 );
		}
		else return( FILE_NEW );
	  }
	} /* if( stat(fpath, buf) == -1 ) */
	else /* If file exists, re-open in required mode */
	{
	  if( (*fp = fopen(fpath, mode)) == NULL )
	  {
		char mesg[MESG_SIZE];
		perror( fpath );
		snprintf( mesg, sizeof(mesg),
			_("Failed to open\n%s\n"\
			  "Quit xfhell and correct"), fpath );
		Error_Dialog( mesg, QUIT );
		return( -1 );
	  }
	  else return( FILE_OPEN );

	} /* else */
  } /* if( *fp == NULL ) */

  return( FILE_OPEN );
} /* Open_File() */

/*------------------------------------------------------------------------*/

/*  Close_File()
 *
 *  Closes and NULL's an open file pointer
 */

  void
Close_File( FILE **fp )
{
  if( *fp != NULL )
  {
	fclose( *fp );
	*fp = NULL;
  }
}

/*------------------------------------------------------------------------*/

/*  File_Print()
 *
 *  Prints a string to a file
 */

  gboolean
File_Print( FILE **fp, char *string )
{
  if( fprintf(*fp, "%s", string) != (int)strlen(string) )
  {
	perror( "xfhell: fprintf()" );
	Error_Dialog(
		_("Error printing to file\n"\
		  "Quit xfhell and correct"), QUIT );
	Close_File( fp );
	return( FALSE );
  }

  /* Flush file */
  if( fflush(*fp) != 0 )
  {
	perror( "xfhell: fflush()" );
	Error_Dialog(
		_("Failed to fflush file\n"\
		"Quit xfhell and correct"), QUIT );
	return( FALSE );
  }

  return( TRUE );
} /* File_Print() */

/*------------------------------------------------------------------------*/

/*  Find_Line()
 *
 *  Finds a line in a text file containing a given string
 */

  gboolean
Locate_Line( char **line_idx, char *line, const char *string, FILE *fp )
{
  int err;

  /* Load lines and look for string */
  do
  {
	err = Load_Line( line, fp, "Text file line" );
	*line_idx = strstr(line, string);
  }
  while( (err == SUCCESS) && (*line_idx == NULL) );

  /* Usually EOF is the cause of error */
  if( err )
  {
	Error_Dialog(
		_("Error searching text file\n"\
		  "May be empty or corrupted"), OK );
	return( FALSE );
  }

  *line_idx += strlen(string);

  return( TRUE );
} /* Find_Line() */

/*------------------------------------------------------------------------*/

/*  Locate_String()
 *
 *  Finds a string in a buffer
 */

  gboolean
Locate_String( char **line_idx, char *line, const char *string )
{
  if( (*line_idx = strstr(line, string)) == NULL )
  {
	Error_Dialog(
		_("Error searching station_log.txt\n"\
		  "May be empty or corrupted"), OK );
	return( FALSE );
  }

  *line_idx += strlen(string);

  return( TRUE );
} /* Locate_String() */

/*------------------------------------------------------------------------*/

/* New_Parameters()
 *
 * Sets up parameters in the program after one changes
 */

  void
New_Parameters( void )
{
  int idx;
  char btn[7], font_file[81];
  double baud_rate[BAUD_RATES] =
  { BAUD122, BAUD105, BAUD61, BAUD30, BAUD15,\
	BAUD7, BAUD14, BAUD2, BAUD1, BAUD06 };

  if( isFlagSet(NO_CALLBACKS) ) return;

  /* Find active baud rate menu item */
  for( idx = 0; idx < BAUD_RATES; idx++ )
  {
	snprintf( btn, sizeof(btn), "br%02d",
		(int)(baud_rate[idx] * 10.0 + 0.001) );
	btn[6] = '\0';
	if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
			lookup_widget(gbl_popup_menu, btn))) )
	  break;
  }
  if( idx == BAUD_RATES ) idx = 2; /* BAUD122 */
  rc_data.baud_rate = baud_rate[ idx ];

  /* Prepare default Font file path if needed */
  if( strlen(rc_data.font_file) )
	Strlcpy( font_file, rc_data.font_file, sizeof(font_file) );
  else
  {
	/* Prepare font path */
	Strlcpy( font_file, getenv("HOME"), sizeof(font_file) );
	Strlcat( font_file, "/xfhell/fonts/", sizeof(font_file) );
	if( rc_data.baud_rate == BAUD105 )
	{
	  Strlcat( font_file, "12pt/", sizeof(font_file) );
	  if( isFlagClear(MODE_FELDHELL) && (baud_rate[idx] == BAUD105) )
		Strlcat( font_file, FONT_LOWRES, sizeof(font_file) );
	  else
		Strlcat( font_file, FONT_NRMRES, sizeof(font_file) );
	}
	else
	{
	  Strlcat( font_file, "14pt/", sizeof(font_file) );
	  Strlcat( font_file, FONT_NRMRES, sizeof(font_file) );
	}
  }
  font_file[80] = '\0';

  /* Free font buffers before loading new fonts */
  for( idx = 0; idx < rc_data.font_data.num_glyphs; idx++ )
	free( rc_data.font_data.glyph_data[idx].bitmap );
  free( rc_data.font_data.glyph_data );
  Load_Fonts( font_file );

  /* Duration of various elements in DSP samples */
  rc_data.samples_per_dot =
	(int)((double)rc_data.dsp_rate/rc_data.baud_rate/2.0 + 0.5);
  rc_data.samples_per_cycle = rc_data.dsp_rate / rc_data.tone_freq;

  /* Puts the tone freq in middle of waterfall */
  idx = (2 * DFT_INPUT_SIZE) / wfall_pixbuf.width;
  rc_data.dft_stride = rc_data.dsp_rate / rc_data.tone_freq / idx;

  /* Height and width in pixels of bitmap of Hell characters */
  rc_data.bitmap_height = rc_data.dot_size * rc_data.font_data.fbb_h;
  rc_data.bitmap_width  = rc_data.bitmap_height / 2;

  /* Make wavetables for FeldHell and CW id's */
  Make_Cos_Wavetable();

  SetFlag( NEW_BAUD_RATE );
  SetFlag( CLEAR_RX_WINDOW );
  Set_TxRx_Labels();
  Alloc_Tx_Buffer();

  return;
} /* New_Parameters() */

/*------------------------------------------------------------------------*/

/***  Memory allocation/freeing utils ***/
gboolean mem_alloc( void **ptr, size_t req )
{
  free_ptr( ptr );
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	perror( "xfhell: alloc():" );
	Error_Dialog(
		_("A memory allocation failed\n"\
		  "Please quit xfhell and correct"), QUIT  );
	return( FALSE );
  }
  return( TRUE );
} /* End of void mem_alloc() */

/*------------------------------------------------------------------------*/

gboolean mem_realloc( void **ptr, size_t req )
{
  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
	perror( "xfhell: realloc():" );
	Error_Dialog( _("A memory re-allocation failed\n"\
		  "Please quit xfhell and correct"), QUIT  );
	return( FALSE );
  }
  return( TRUE );
} /* End of void mem_realloc() */

/*------------------------------------------------------------------------*/

void free_ptr( void **ptr )
{
  if( *ptr != NULL )
	free( *ptr );
  *ptr = NULL;

} /* End of void free_ptr() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flow control flags
 *
 *  See xfhell.h for definition of flow control flags
 */

/* An int variable holding the single-bit flags */
static long Flags = 0;

  int
isFlagSet( long flag )
{
  return( (Flags & flag) == flag );
}

  int
isFlagClear( long flag )
{
  return( (~Flags & flag) == flag );
}

  void
SetFlag( long flag )
{
  Flags |= flag;
}

  void
ClearFlag( long flag )
{
  Flags &= ~flag;
}

  void
ToggleFlag( long flag )
{
  Flags ^= flag;
}

/*------------------------------------------------------------------------*/

/*  Cleanup()
 *
 *  Cleans up before quitting
 */

  void
Cleanup( void )
{
  /* Save outstanding records */
  if( isFlagSet(SAVE_RECORD) )
  {
	Save_QSO_Record();
	Save_Pixbuf();
  }

  /* Close open files */
  if( rc_data.qso_record_fp != NULL )
  {
	ClearFlag(RECORD_QSO);
	fclose( rc_data.qso_record_fp );
	rc_data.qso_record_fp = NULL;
  }

  if( rc_data.log_adif_fp != NULL )
  {
	fclose( rc_data.log_adif_fp );
	rc_data.log_adif_fp = NULL;
  }

  if( rc_data.station_log_fp != NULL )
  {
	fclose( rc_data.station_log_fp );
	rc_data.station_log_fp = NULL;
  }

  /* Release sound card and Tcvr */
  ClearFlag (
	  TRANSMIT_MODE  |
	  TRANSMIT_MACRO |
	  TRANSMIT_KEYBD |
	  RECEIVE_MODE );

  Close_Tcvr_Serial();
  Close_Playback();
  Close_Capture();
  Close_Mixer();

} /* Cleanup( void ) */

/*------------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, char *mesg )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, sizeof(error_mesg),
	  _("Error reading %s\n"\
		"Premature EOF (End Of File)"), mesg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  if( (chr = fgetc(pfile)) == EOF )
  {
	fprintf( stderr, "xfhell: %s\n", error_mesg );
	fclose( pfile );
	Error_Dialog( error_mesg, QUIT );
	return( EOF );
  }

  /* Ignore commented lines, white spaces and eol/cr */
  while(
	  (chr == '#') ||
	  (chr == HT ) ||
	  (chr == ' ') ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xfhell: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, QUIT );
		return( EOF );
	  }

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xfhell: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, QUIT );
		return( EOF );
	  }

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < 80 )
  {
	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) ) break;

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	if( (chr = fgetc(pfile)) == EOF )
	{
	  /* Terminate buffer as a string if chr = EOF */
	  buff[num_chr] = '\0';
	  return( SUCCESS );
	}

	/* Abort if end of line not reached at 80 char. */
	if( (num_chr == 80) && (chr != LF) && (chr != CR) )
	{
	  /* Terminate buffer as a string */
	  buff[num_chr] = '\0';
	  snprintf( error_mesg, sizeof(error_mesg),
		  _("Error reading %s\n"\
			"Line longer than 80 characters"), mesg );
	  fprintf( stderr, "xfhell: %s\n%s\n", error_mesg, buff );
	  fclose( pfile );
	  Error_Dialog( error_mesg, QUIT );
	  return( ERROR );
	}

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n",
	  _("Usage: xfhell [-hv]") );

  fprintf( stderr, "%s\n",
	  _("       -h: Print this usage information and exit"));

  fprintf( stderr, "%s\n",
	  _("       -v: Print version number and exit"));

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;
 
  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src
 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  int idd = 0; /* dest index */
  int ids = 0; /* src  index */

  /* Find terminating null of dest */
  while( (n > 0) && (ch != '\0') )
  {
	idd++;
	ch = dest[idd];
	n--; /* Count remaining char's in dest */
  }

  /* Copy n-1 chars to leave room for terminating null */
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
	dest[idd] = src[ids];
	ids++;
	ch = src[ids];
	idd++;
	n--;
  }

  /* Terminate dest string */
  dest[idd] = '\0';

} /* Strlcat() */

/*------------------------------------------------------------------*/

/* Atof()
 *
 * Replaces atof() to take into account the
 * locale-dependent decimal point character
 */
double Atof( const char *nptr )
{
  int idx;
  size_t len;
  double d = 0.0;
  char *s = NULL;
  static gboolean first_call = TRUE;
  static char dp = '.';

  /* Find locale-dependent decimal point character */
  if( first_call )
  {
	struct lconv *lcnv;
	lcnv = localeconv();
	dp = *lcnv->decimal_point;
	first_call = FALSE;
  }

  /* Look for a . or , decimal point character
   * in the supplied number buffer (string) */
  len = strlen( nptr );
  for( idx = 0; idx < (int)len; idx++ )
	if( (nptr[idx] == ',') || (nptr[idx] == '.') )
	  break;

  /* Create temporary string to modify decimal point */
  if( !mem_alloc((void **)&s, len + 1) ) return( d );
  strncpy( s, nptr, len );
  s[ len ] = '\0';

  /* If a decimal point character is found, replace */
  if( idx < (int)len ) s[idx] = dp;
  d = atof( s );
  free_ptr( (void **)&s );
  
  return( d );
} /* End of Atof() */

/*------------------------------------------------------------------*/
