/*******************************************************************************

Copyright (c) 2007-2008, Perforce Software, Inc.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1.  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

2.  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTR
IBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL PERFORCE SOFTWARE, INC. BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: //depot/r10.1/p4-python/P4Result.cpp#1 $
*******************************************************************************/

#include <Python.h>
#include <bytesobject.h>
#include "undefdups.h"
#include "python2to3.h"
#include <clientapi.h>
#include "P4Result.h"
#include "P4PythonDebug.h"
#include <iostream>

using namespace std;

P4Result::P4Result()
    : output(NULL), warnings(NULL), errors(NULL)
{
    Reset();
}

P4Result::~P4Result()
{
    // reduce references to Python objects to avoid memory leaks
    
    if (output) 
        Py_DECREF(output);
    
    if (warnings) 
        Py_DECREF(warnings);
    
    if (errors)
        Py_DECREF(errors);
}

PyObject * P4Result::GetOutput()
{   
    PyObject * temp = output;
    output = NULL;  // last reference is removed by caller
    return temp;
}

void
P4Result::Reset()
{
    output = PyList_New(0);
    
    if (warnings) 
        Py_DECREF(warnings);
    warnings = PyList_New(0);
    
    if (errors)
        Py_DECREF(errors);
    errors = PyList_New(0);
    
    if (output == NULL || warnings == NULL || errors == NULL) {
    	cerr << "[P4Result::P4Result] Error creating lists" << endl;
    }    
}

void P4Result::AppendString(PyObject * list, const char * str)
{
    PyObject *s = CreatePythonString(str);
    if (!s) {
	// TODO: Throw exception
    }
    if (PyList_Append(list, s) == -1) {
    	// TODO: Throw exception
    }
    Py_DECREF(s);
}

void P4Result::AddOutput( const char *msg )
{
    AppendString(output, msg);
}

void P4Result::AddOutput( PyObject * out )
{
    if (PyList_Append(output, out) == -1) {
    	// TODO: Throw exception
    }
    Py_DECREF(out);
}

void
P4Result::AddError( Error *e )
{
    StrBuf	m;
    e->Fmt( &m, EF_PLAIN );

    int s;
    s = e->GetSeverity();

    // 
    // Empty and informational messages are pushed out as output as nothing
    // worthy of error handling has occurred. Warnings go into the warnings
    // list and the rest are lumped together as errors.
    //

    if ( s == E_EMPTY || s == E_INFO )
    {
	AddOutput( m.Text() );
	return;
    }

    if ( s == E_WARN )
	AppendString(warnings, m.Text());
    else
	AppendString(errors, m.Text());
}

int
P4Result::ErrorCount()
{
    return PyList_Size( errors );
}

int
P4Result::WarningCount()
{
    return PyList_Size( warnings );
}

void
P4Result::FmtErrors( StrBuf &buf )
{
    Fmt( "[Error]: ", errors, buf );
}

void
P4Result::FmtWarnings( StrBuf &buf )
{
    Fmt( "[Warning]: ", warnings, buf );
}


int
P4Result::Length( PyObject * list )
{
    return PyList_Size( list );
}

void
P4Result::Fmt( const char *label, PyObject * list, StrBuf &buf )
{
    buf.Clear();
    // If the array is empty, then we just return
    if( PyList_Size(list) == 0 ) return;

    // This is the string we'll use to prefix each entry in the array
    StrBuf csep;
    
    csep << "\n\t" << label;
    
    PyObject * jsep = CreatePythonString( csep.Text() );
    if (jsep == NULL) {
    	// TODO throw exception
    }
    
    // Since Array#join() won't prefix the first element with the separator
    // we'll have to do it manually.
    buf << csep;

    // Join the array elements together, and append the result to the buffer
    PyObject * str = PyObject_CallMethod(jsep, "join", "O", list);
    buf << GetPythonString(str);

    return;
}

